// TEPipePanelController.m
// TextExtras - Yellow Box
//
// Copyright 1996-1999, Mike Ferris.
// All rights reserved.

#import "TEPipePanelController.h"
#import "TEPipeCommand.h"

@implementation TEPipePanelController

+ (id)sharedPipePanelController {
    static TEPipePanelController *sharedInstance = nil;
    if (!sharedInstance) {
        sharedInstance = [[TEPipePanelController alloc] init];
    }
    return sharedInstance;
}

#define USER_PIPE_FILE @".TEUserPipes.plist"

static NSMutableArray *userPipes = nil;
static NSMenu *userPipesMenu = nil;

+ (NSArray *)userPipes {
    if (!userPipes) {
        // Read the user's pipe-dict file.
        NSString *filePath = [NSHomeDirectory() stringByAppendingPathComponent:USER_PIPE_FILE];

        userPipes = [[NSMutableArray allocWithZone:NULL] init];

        if ([[NSFileManager defaultManager] isReadableFileAtPath:filePath]) {
            NSString *fileString = [NSString stringWithContentsOfFile:filePath];
            NSArray *plist;
            unsigned i, c;
            NSDictionary *curDict;
            TEPipeCommand *newCommand;

            NS_DURING
                plist = [fileString propertyList];
            NS_HANDLER
                NSLog(@"TextExtras: The user pipe file is not a valid property list.  It should be an array of dictionaries.");
                return userPipes;
            NS_ENDHANDLER
            
            if (!plist) {
                NSLog(@"TextExtras: The user pipe file is not a valid property list.  It should be an array of dictionaries.");
                return userPipes;
            }
            if (![plist isKindOfClass:[NSArray class]]) {
                NSLog(@"TextExtras: The user pipe file must contain a top-level array (of dictionaries).");
                return userPipes;
            }
            c = [plist count];
            for (i=0; i<c; i++) {
                curDict = [plist objectAtIndex:i];
                if (![curDict isKindOfClass:[NSDictionary class]]) {
                    NSLog(@"TextExtras: Element number %u in user pipes array is not a dictionary.");
                } else {
                    newCommand = [[TEPipeCommand allocWithZone:NULL] initWithDictionary:curDict];
                    if (!newCommand) {
                        NSLog(@"TextExtras: Element number %u in user pipes array is not a valid pipe command.  Skipping.", i);
                    } else {
                        [userPipes addObject:newCommand];
                        [newCommand release];
                    }
                }
            }
        }
    }
    return userPipes;
}

+ (void)refreshUserPipesMenu {
    if (userPipesMenu) {
        NSArray *pipes = [self userPipes];
        unsigned i, c = [pipes count];

        // Empty the menu
        while ([[userPipesMenu itemArray] count] > 0) {
            [userPipesMenu removeItem:[[userPipesMenu itemArray] objectAtIndex:0]];
        }
        // Fill  the menu
        if (c>0) {
            NSMenuItem *item;
            TEPipeCommand *command;

            for (i=0; i<c; i++) {
                command = [pipes objectAtIndex:i];
                item = [userPipesMenu addItemWithTitle:[command description] action:@selector(TE_executeUserPipe:) keyEquivalent:command->keyEquivalent];
                [item setRepresentedObject:command];
            }
        }
    }
}

+ (void)setUserPipes:(NSArray *)newUserPipes {
    NSString *filePath = [NSHomeDirectory() stringByAppendingPathComponent:USER_PIPE_FILE];
    NSFileManager *fm = [NSFileManager defaultManager];
    BOOL dummy;
    
    [userPipes release];
    userPipes = (newUserPipes ? [newUserPipes retain] : [[NSMutableArray allocWithZone:NULL] init]);
    if (![fm fileExistsAtPath:filePath isDirectory:&dummy] || [fm isWritableFileAtPath:filePath]) {
        NSMutableArray *plist = [[NSMutableArray allocWithZone:NULL] initWithCapacity:[userPipes count]];
        unsigned i, c = [userPipes count];

        for (i=0; i<c; i++) {
            [plist addObject:[[userPipes objectAtIndex:i] dictionaryRepresentation]];
        }
        [[plist description] writeToFile:filePath atomically:YES];
        [plist release];
    }
    [self refreshUserPipesMenu];
}

+ (NSMenu *)userPipesMenu {
    if (!userPipesMenu) {
        userPipesMenu = [[NSMenu allocWithZone:[NSMenu menuZone]] initWithTitle:NSLocalizedStringFromTableInBundle(@"User Pipes", @"TextExtras", [NSBundle bundleForClass:[self class]], @"Title of User Pipes submenu item")];
        [self refreshUserPipesMenu];
    }

    return userPipesMenu;
}

- (id)init {
    return [super initWithWindowNibName:@"TEPipePanel" owner:self];
}

- (void)dealloc {
    [_previousCommands release];
    [super dealloc];
}

- (void)windowDidLoad {
    BOOL win95Style = ((NSInterfaceStyleForKey(nil, [self window]) == NSWindows95InterfaceStyle) ? YES : NO);
    NSButtonCell *okButtonCell, *cancelButtonCell;

    [super windowDidLoad];
    
    // Buttons are "backwards" on windows.
    if (win95Style) {
        okButtonCell = [buttonMatrix cellAtRow:0 column:0];
        cancelButtonCell = [buttonMatrix cellAtRow:0 column:1];
    } else {
        okButtonCell = [buttonMatrix cellAtRow:0 column:1];
        cancelButtonCell = [buttonMatrix cellAtRow:0 column:0];
    }
    [okButtonCell setTitle:NSLocalizedStringFromTableInBundle(@"OK", @"TextExtras", [NSBundle bundleForClass:[self class]], @"OK button title.")];
    [okButtonCell setTag:NSOKButton];
    [okButtonCell setKeyEquivalent:@"\r"];
    [commandComboBox setTarget:okButtonCell];
    [commandComboBox setAction:@selector(performClick:)];
    [cancelButtonCell setTitle:NSLocalizedStringFromTableInBundle(@"Cancel", @"TextExtras", [NSBundle bundleForClass:[self class]], @"Cancel button title.")];
    [cancelButtonCell setTag:NSCancelButton];

    _previousCommands = [[NSMutableArray allocWithZone:[self zone]] initWithArray:[[self class] userPipes]];
}

- (void)buttonMatrixAction:(id)sender {
    [NSApp stopModalWithCode:[[sender selectedCell] tag]];
}

- (void)runPipePanelWithTextView:(NSTextView *)textView {
    // Prompts the user for a command to run, then calls executeCommand:withInputString: to execute it.
    NSWindow *w = [self window];  // Force nib loading.

    [w makeFirstResponder:w];
    
    if ([NSApp runModalForWindow:w] == NSOKButton) {
        NSString *command;
        TEStandardInputSource inSrc;
        TEStandardOutputDestination outDest;
        TEPipeCommand *pipe;
        
        if (![w makeFirstResponder:w]) {
            [w endEditingFor:nil];
        }
        [w orderOut:self];

        command = [commandComboBox stringValue];
        inSrc = [inputRadioMatrix selectedRow];
        outDest = [outputRadioMatrix selectedRow];

        pipe = [[TEPipeCommand allocWithZone:[self zone]] initWithCommandString:command standardInputSource:inSrc standardOutputDestination:outDest keyEquivalent:@""];

        if (pipe) {
            if ([pipe runWithTextView:textView]) {
                // If the command executed successfully, add it to the command history.  To avoid duplicates we remove any previous occurence of the same command.
                unsigned existingIndex = [_previousCommands indexOfObject:pipe];
                if (existingIndex != NSNotFound) {
                    [_previousCommands removeObjectAtIndex:existingIndex];
                }
                [_previousCommands insertObject:pipe atIndex:0];
                [commandComboBox reloadData];
            }
            [pipe release];
        }
    } else {
        [w orderOut:self];
    }
}

- (void)runUserPipe:(id)pipeIdentifier withTextView:(NSTextView *)textView {
    if ([pipeIdentifier isKindOfClass:[TEPipeCommand class]]) {
        [pipeIdentifier runWithTextView:textView];
    }
}

- (int)numberOfItemsInComboBox:(NSComboBox *)aComboBox {
    return [_previousCommands count];
}

- (id)comboBox:(NSComboBox *)aComboBox objectValueForItemAtIndex:(int)index {
    return [[_previousCommands objectAtIndex:index] description];
}

- (unsigned int)comboBox:(NSComboBox *)aComboBox indexOfItemWithStringValue:(NSString *)string {
    return NSNotFound;
}

- (void)comboBoxSelectionDidChange:(NSNotification *)notification {
    int selectedIndex = [commandComboBox indexOfSelectedItem];
    if ((selectedIndex >= 0) && (selectedIndex < [_previousCommands count])) {
        TEPipeCommand *selectedCommand = [_previousCommands objectAtIndex:selectedIndex];
        [inputRadioMatrix selectCellAtRow:selectedCommand->inputSource column:0];
        [outputRadioMatrix selectCellAtRow:selectedCommand->outputDestination column:0];
        [commandComboBox setStringValue:selectedCommand->commandString];
    }
}

- (void)windowWillClose:(NSNotification *)notification {
    [NSApp stopModalWithCode:NSCancelButton];
}

- (BOOL)windowShouldClose:(NSWindow *)window {
    // MF: To prevent AppKit from doing a misguided stopModalWithCode: when the window is closed.  Actually, on Windows we could leave out this method and remove the windowWillClose: handler above, but on Mach, the windowWillClose: is necessary.  Therefore, to avoid stopModalWithCode: from being invoked twice (which actually would be harmless...), we implement this method.  It is super-nasty that on Windows closing a modal window behaves differently than on Mach, but it does.
    return YES;
}

@end
