// TEPreferencesController.m
// TextExtras - Yellow Box
//
// Copyright 1996-1999, Mike Ferris.
// All rights reserved.

#import "TEPreferencesController.h"

#define TabWidthDefault @"TabWidth"
#define IndentWidthDefault @"IndentWidth"
#define IndentWrappedLinesDefault @"IndentWrappedLines"
#define WrappedLineIndentWidthDefault @"WrappedLineIndentWidth"
#define SelectToMatchingBraceDefault @"SelectToMatchingBrace"
#define ShowMatchingBraceDefault @"ShowMatchingBrace"
#define FancyEscapeCompletionDefault @"FancyEscapeCompletion"
#define EnforceTabStopsDefault @"EnforceTabStops"
#define TryToInstallExtrasMenuDefault @"InstallTextExtrasMenu"
#define TryHardToInstallExtrasMenuDefault @"InstallTextExtrasMenuWithNoFormatMenu"
#define TryToInstallDebugMenuDefault @"InstallTextExtrasDebugMenu"
#define OpenQuicklyPathsDefault @"OpenQuicklyPaths"
#define SavesOpenQuicklyPathsGloballyDefault @"SavesOpenQuicklyPathsGlobally"

enum {
    IndentWrappedLinesCheckBoxRow = 0,
    SelectToMatchingBraceCheckBoxRow = 1,
    ShowMatchingBraceCheckBoxRow = 2,
    FancyEscapeCompletionCheckBoxRow = 3,
    EnforceTabStopsCheckBoxRow = 4,
};

enum {
    TabWidthFormRow = 0,
    IndentWidthFormRow = 1,
    WrappedLineIndentWidthFormRow = 2,
};

enum {
    GlobalInstallAlwaysRow = 0,
    GlobalInstallInFormatRow = 1,
    GlobalInstallNeverRow = 2,
};

enum {
    LocalUseGlobalSettingsRow = 0,
    LocalInstallAlwaysRow = 1,
    LocalInstallNeverRow = 2,
};

enum {
    ModifierKeyCommand = 0,
    ModifierKeyControl = 1,
    ModifierKeyAlternate = 2,
};

NSString *TEPreferencesDidChangeNotification = @"TEPreferencesDidChange";

@implementation TEPreferencesController

+ (BOOL)tryToInstallExtrasMenu {
    static BOOL TryToInstallExtrasMenu = YES;
    static BOOL ReadDefault = NO;
    if (!ReadDefault) {
        id val = [[NSUserDefaults standardUserDefaults] objectForKey:TryToInstallExtrasMenuDefault];
        if (!val || ([val hasPrefix:@"y"] || [val hasPrefix:@"Y"])) {
            TryToInstallExtrasMenu = YES;
        } else {
            TryToInstallExtrasMenu = NO;
        }
        ReadDefault = YES;
    }
    return TryToInstallExtrasMenu;
}

+ (BOOL)tryHardToInstallExtrasMenu {
    static BOOL TryHardToInstallExtrasMenu = YES;
    static BOOL ReadDefault = NO;
    if (!ReadDefault) {
        id val = [[NSUserDefaults standardUserDefaults] objectForKey:TryHardToInstallExtrasMenuDefault];
        if (!val || ([val hasPrefix:@"y"] || [val hasPrefix:@"Y"])) {
            TryHardToInstallExtrasMenu = YES;
        } else {
            TryHardToInstallExtrasMenu = NO;
        }
        ReadDefault = YES;
    }
    return TryHardToInstallExtrasMenu;
}

+ (BOOL)tryToInstallDebugMenu {
    static BOOL TryToInstallDebugMenu = NO;
    static BOOL ReadDefault = NO;
    if (!ReadDefault) {
        id val = [[NSUserDefaults standardUserDefaults] objectForKey:TryToInstallDebugMenuDefault];
        if (([val hasPrefix:@"y"] || [val hasPrefix:@"Y"])) {
            TryToInstallDebugMenu = YES;
        } else {
            TryToInstallDebugMenu = NO;
        }
        ReadDefault = YES;
    }
    return TryToInstallDebugMenu;
}

+ (id)sharedPreferencesController {
    static TEPreferencesController *sharedInstance = nil;
    if (!sharedInstance) {
        sharedInstance = [[TEPreferencesController alloc] init];
    }
    return sharedInstance;
}

- (id)init {
    self = [super initWithWindowNibName:@"TEPreferences" owner:self];
    if (self) {
        [self setWindowFrameAutosaveName:@"TextExtrasPreferences"];
        [self readDefaults];
    }
    return self;
}

- (void)dealloc {
    [[NSNotificationCenter defaultCenter] removeObserver:self];
    [generalTab release];
    [generalBox release];
    [openQuicklyTab release];
    [openQuicklyBox release];
    [completionTab release];
    [completionBox release];
    [menuSettingsTab release];
    [menuSettingsBox release];
#ifdef WIN32
    [modifierFlagsTab release];
    [modifierFlagsBox release];
#endif
    [openQuicklyPaths release];
    [completionDict release];
    [sortedCompletionKeys release];
    [lastReadCompletionDictDate release];
    [super dealloc];
}

- (void)readDefaults {
    NSUserDefaults *defaults = [NSUserDefaults standardUserDefaults];
    NSDictionary *globalDomain, *localDomain;
    id val;

    // General settings
    tabWidth = NSNotFound;
    val = [defaults objectForKey:TabWidthDefault];
    if (val) {
        tabWidth = [val intValue];
    }
    if ((tabWidth <= 0) || (tabWidth >= 100)) {
        tabWidth = 8;
    }

    indentWidth = NSNotFound;
    val = [defaults objectForKey:IndentWidthDefault];
    if (val) {
        indentWidth = [val intValue];
    }
    if ((indentWidth <= 0) || (indentWidth >= 100)) {
        indentWidth = 4;
    }

    wrappedLineIndentWidth = NSNotFound;
    val = [defaults objectForKey:WrappedLineIndentWidthDefault];
    if (val) {
        wrappedLineIndentWidth = [val intValue];
    }
    if ((wrappedLineIndentWidth <= 0) || (wrappedLineIndentWidth >= 100)) {
        wrappedLineIndentWidth = 4;
    }

    val = [defaults objectForKey:IndentWrappedLinesDefault];
    if (val && ([val hasPrefix:@"y"] || [val hasPrefix:@"Y"])) {
        indentWrappedLines = YES;
    } else {
        indentWrappedLines = NO;
    }

    val = [defaults objectForKey:SelectToMatchingBraceDefault];
    if (val && ([val hasPrefix:@"y"] || [val hasPrefix:@"Y"])) {
        selectToMatchingBrace = YES;
    } else {
        selectToMatchingBrace = NO;
    }

    val = [defaults objectForKey:ShowMatchingBraceDefault];
    if (val && ([val hasPrefix:@"y"] || [val hasPrefix:@"Y"])) {
        showMatchingBrace = YES;
    } else {
        showMatchingBrace = NO;
    }

    val = [defaults objectForKey:FancyEscapeCompletionDefault];
    if (val && ([val hasPrefix:@"y"] || [val hasPrefix:@"Y"])) {
        fancyEscapeCompletion = YES;
    } else {
        fancyEscapeCompletion = NO;
    }

    val = [defaults objectForKey:EnforceTabStopsDefault];
    if (val && ([val hasPrefix:@"y"] || [val hasPrefix:@"Y"])) {
        enforceTabStops = YES;
    } else {
        enforceTabStops = NO;
    }

    // Open quickly paths settings
    [openQuicklyPaths release];
    openQuicklyPaths = nil;
    val = [defaults objectForKey:OpenQuicklyPathsDefault];
    if (val) {
        if ([val isKindOfClass:[NSArray class]]) {
            openQuicklyPaths = [val mutableCopyWithZone:[self zone]];
        } else if ([val isKindOfClass:[NSString class]]) {
            NSArray *temp = [val propertyList];
            if (temp && [temp isKindOfClass:[NSArray class]]) {
                openQuicklyPaths = [temp mutableCopyWithZone:[self zone]];
            } else {
                NSLog(@"TextExtras: String value for default OpenQuicklyPaths did not parse into an array.");
            }
        } else {
            NSLog(@"TextExtras: Unexpected value of class %@ for default OpenQuicklyPaths.", [val class]);
        }
    }
    if (!openQuicklyPaths) {
        openQuicklyPaths = [[NSMutableArray allocWithZone:[self zone]] init];
    }

    val = [defaults objectForKey:SavesOpenQuicklyPathsGloballyDefault];
    if (!val || ([val hasPrefix:@"y"] || [val hasPrefix:@"Y"])) {
        // Yes is default
        savesOpenQuicklyPathsGlobally = YES;
    } else {
        savesOpenQuicklyPathsGlobally = NO;
    }

    // Completion dict settings (these are stored in a file, not NSUserDefaults)
    [completionDict release];
    completionDict = nil;

    // Menu settings
    globalDomain = [defaults persistentDomainForName:NSGlobalDomain];

    val = [globalDomain objectForKey:TryToInstallExtrasMenuDefault];
    if (!val || ([val hasPrefix:@"y"] || [val hasPrefix:@"Y"])) {
        // Yes is default
        globalTryMenu = YES;
    } else {
        globalTryMenu = NO;
    }

    val = [globalDomain objectForKey:TryHardToInstallExtrasMenuDefault];
    if (!val || ([val hasPrefix:@"y"] || [val hasPrefix:@"Y"])) {
        // Yes is default
        globalTryMenuHard = YES;
    } else {
        globalTryMenuHard = NO;
    }

    val = [globalDomain objectForKey:TryToInstallDebugMenuDefault];
    if (val && ([val hasPrefix:@"y"] || [val hasPrefix:@"Y"])) {
        globalDebugMenu = YES;
    } else {
        globalDebugMenu = NO;
    }

    localDomain = [defaults persistentDomainForName:[[NSProcessInfo processInfo] processName]];

    val = [localDomain objectForKey:TryToInstallExtrasMenuDefault];
    if (val) {
        localStoresMenuSettings = YES;
        
        if (([val hasPrefix:@"y"] || [val hasPrefix:@"Y"])) {
            // Yes is default
            localTryMenu = YES;
        } else {
            localTryMenu = NO;
        }
        localTryMenuHard = localTryMenu;
    } else {
        localStoresMenuSettings = NO;
        localTryMenu = NO;
        localTryMenuHard = NO;
    }

#ifdef WIN32
    // Modifier mappings (only for Windows)
    leftControlModifier = ModifierKeyCommand;
    rightControlModifier = ModifierKeyCommand;
    leftAlternateModifier = ModifierKeyAlternate;
    rightAlternateModifier = ModifierKeyAlternate;

    val = [defaults objectForKey:@"NSModifierFlagMapping"];
    if (val && [val isKindOfClass:[NSDictionary class]]) {
        NSString *temp;
	
        temp = [val objectForKey:@"LeftControl"];
        if (temp && [temp isEqualToString:@"Control"]) {
            // The only acceptable values are Control and Command.  If it isn't Control, assume Command.
            leftControlModifier = ModifierKeyControl;
        }
        temp = [val objectForKey:@"RightControl"];
        if (temp && [temp isEqualToString:@"Control"]) {
            // The only acceptable values are Control and Command.  If it isn't Control, assume Command.
            rightControlModifier = ModifierKeyControl;
        }
        temp = [val objectForKey:@"LeftAlt"];
        if (temp && [temp isEqualToString:@"Control"]) {
            // The only acceptable values are Control, Command, and Alternate.  If it isn't Control or Command, assume Alternate.
            leftAlternateModifier = ModifierKeyControl;
        } else if (temp && [temp isEqualToString:@"Command"]) {
            leftAlternateModifier = ModifierKeyCommand;
        }
        temp = [val objectForKey:@"RightAlt"];
        if (temp && [temp isEqualToString:@"Control"]) {
            // The only acceptable values are Control, Command, and Alternate.  If it isn't Control or Command, assume Alternate.
            rightAlternateModifier = ModifierKeyControl;
        } else if (temp && [temp isEqualToString:@"Command"]) {
            rightAlternateModifier = ModifierKeyCommand;
        }
    }

    val = [defaults objectForKey:@"NSMnemonicsWorkInText"];
    // Default is YES.
    if (val && [val isEqualToString:@"NO"]) {
        mnemonicsWorkInText = NO;
    } else {
        mnemonicsWorkInText = YES;
    }
#endif
}

- (void)revertPanel {    
    [[checkBoxMatrix cellAtRow:IndentWrappedLinesCheckBoxRow column:0] setState:[self indentWrappedLines]];
    [[checkBoxMatrix cellAtRow:SelectToMatchingBraceCheckBoxRow column:0] setState:[self selectToMatchingBrace]];
    [[checkBoxMatrix cellAtRow:ShowMatchingBraceCheckBoxRow column:0] setState:[self showMatchingBrace]];
    [[checkBoxMatrix cellAtRow:FancyEscapeCompletionCheckBoxRow column:0] setState:[self fancyEscapeCompletion]];
    [[checkBoxMatrix cellAtRow:EnforceTabStopsCheckBoxRow column:0] setState:[self enforceTabStops]];

    [[widthForm cellAtIndex:TabWidthFormRow] setIntValue:[self tabWidth]];
    [[widthForm cellAtIndex:IndentWidthFormRow] setIntValue:[self indentWidth]];
    [[widthForm cellAtIndex:WrappedLineIndentWidthFormRow] setIntValue:[self wrappedLineIndentWidth]];

    [self updateOpenQuicklyTextView];
    [openQuicklyGlobalButton setState:([self savesOpenQuicklyPathsGlobally] ? NSOffState : NSOnState)];

    [self readCompletionDictIfNeeded];
    [completionTableView reloadData];
    [self updateCompletionTextView];
    [removeCompletionButton setEnabled:(([completionDict count] > 0) ? YES : NO)];

    if (globalTryMenu) {
        if (globalTryMenuHard) {
            [globalMenuRadioButtons selectCellAtRow:GlobalInstallAlwaysRow column:0];
        } else {
            [globalMenuRadioButtons selectCellAtRow:GlobalInstallInFormatRow column:0];
        }
    } else {
        [globalMenuRadioButtons selectCellAtRow:GlobalInstallNeverRow column:0];
    }
    [globalDebugMenuCheckbox setState:globalDebugMenu];

    if (localStoresMenuSettings) {
        if (localTryMenu) {
            [localMenuRadioButtons selectCellAtRow:LocalInstallAlwaysRow column:0];
        } else {
            [localMenuRadioButtons selectCellAtRow:LocalInstallNeverRow column:0];
        }
    } else {
        [localMenuRadioButtons selectCellAtRow:LocalUseGlobalSettingsRow column:0];
    }

#ifdef WIN32
    [leftControlPopUp selectItemAtIndex:leftControlModifier];
    [rightControlPopUp selectItemAtIndex:rightControlModifier];
    [leftAlternatePopUp selectItemAtIndex:leftAlternateModifier];
    [leftAlternatePopUp selectItemAtIndex:leftAlternateModifier];
    [mnemonicsWorkInTextCheckbox setState:mnemonicsWorkInText];
#endif
}

- (void)windowDidLoad {
    NSBundle *bundle = [NSBundle bundleForClass:[TEPreferencesController class]];
    NSMutableArray *throwAwayWindows = [[NSMutableArray allocWithZone:[self zone]] init];
    NSWindow *curWindow;
    
    [super windowDidLoad];

    // Grab hold of the content views and dispose of the temporary panel
    [generalBox retain];
    curWindow = [generalBox window];
    [throwAwayWindows addObject:curWindow];
    [curWindow release];
    [generalBox removeFromSuperview];
    [completionBox retain];
    curWindow = [completionBox window];
    [throwAwayWindows addObject:curWindow];
    [curWindow release];
    [completionBox removeFromSuperview];
    [openQuicklyBox retain];
    curWindow = [openQuicklyBox window];
    [throwAwayWindows addObject:curWindow];
    [curWindow release];
    [openQuicklyBox removeFromSuperview];
    [menuSettingsBox retain];
    curWindow = [menuSettingsBox window];
    [throwAwayWindows addObject:curWindow];
    [curWindow release];
    [menuSettingsBox removeFromSuperview];
#ifdef WIN32
    [modifierFlagsBox retain];
    curWindow = [modifierFlagsBox window];
    [throwAwayWindows addObject:curWindow];
    [curWindow release];
    [modifierFlagsBox removeFromSuperview];
#else
    // If we're not on windows, just throw away this tab content view.
    [[modifierFlagsBox window] release];
    modifierFlagsBox = nil;
    leftControlPopUp = nil;
    rightControlPopUp = nil;
    leftAlternatePopUp = nil;
    rightAlternatePopUp = nil;
    mnemonicsWorkInTextCheckbox = nil;
#endif

    [throwAwayWindows release];
    
    // Set up the tab view.
    generalTab = [[NSTabViewItem allocWithZone:[self zone]] initWithIdentifier:@"General"];
    [generalTab setView:generalBox];
    [generalTab setLabel:NSLocalizedStringFromTableInBundle(@"General", @"TextExtras", bundle, @"Tab item name")];
    [generalTab setInitialFirstResponder:checkBoxMatrix];
    [tabView addTabViewItem:generalTab];

    completionTab = [[NSTabViewItem allocWithZone:[self zone]] initWithIdentifier:@"Completion"];
    [completionTab setView:completionBox];
    [completionTab setLabel:NSLocalizedStringFromTableInBundle(@"Completion", @"TextExtras", bundle, @"Tab item name")];
    [completionTab setInitialFirstResponder:completionTableView];
    [tabView addTabViewItem:completionTab];

    openQuicklyTab = [[NSTabViewItem allocWithZone:[self zone]] initWithIdentifier:@"Open Quickly"];
    [openQuicklyTab setView:openQuicklyBox];
    [openQuicklyTab setLabel:NSLocalizedStringFromTableInBundle(@"Open Quickly", @"TextExtras", bundle, @"Tab item name")];
    [openQuicklyTab setInitialFirstResponder:openQuicklyTextView];
    [tabView addTabViewItem:openQuicklyTab];

    menuSettingsTab = [[NSTabViewItem allocWithZone:[self zone]] initWithIdentifier:@"Menu Settings"];
    [menuSettingsTab setView:menuSettingsBox];
    [menuSettingsTab setLabel:NSLocalizedStringFromTableInBundle(@"Menu Settings", @"TextExtras", bundle, @"Tab item name")];
    [menuSettingsTab setInitialFirstResponder:globalMenuRadioButtons];
    [tabView addTabViewItem:menuSettingsTab];

#ifdef WIN32
    modifierFlagsTab = [[NSTabViewItem allocWithZone:[self zone]] initWithIdentifier:@"Modifier Flags"];
    [modifierFlagsTab setView:modifierFlagsBox];
    [modifierFlagsTab setLabel:NSLocalizedStringFromTableInBundle(@"Modifier Flags", @"TextExtras", bundle, @"Tab item name")];
    [modifierFlagsTab setInitialFirstResponder:leftControlPopUp];
    [tabView addTabViewItem:modifierFlagsTab];
#endif

    // Make sure we start at General
    [tabView selectTabViewItem:generalTab];
    
    [self revertPanel];
}

- (unsigned)tabWidth {
    return tabWidth;
}

- (unsigned)indentWidth {
    return indentWidth;
}

- (unsigned)wrappedLineIndentWidth {
    return wrappedLineIndentWidth;
}

- (BOOL)indentWrappedLines {
    return indentWrappedLines;
}

- (BOOL)selectToMatchingBrace {
    return selectToMatchingBrace;
}

- (BOOL)showMatchingBrace {
    return showMatchingBrace;
}

- (BOOL)fancyEscapeCompletion {
    return fancyEscapeCompletion;
}

- (BOOL)enforceTabStops {
    return enforceTabStops;
}

- (NSArray *)openQuicklyPaths {
    return openQuicklyPaths;
}

- (BOOL)savesOpenQuicklyPathsGlobally {
    return savesOpenQuicklyPathsGlobally;
}

- (NSDictionary *)completionDictionary {
    [self readCompletionDictIfNeeded];
    return completionDict;
}

- (void)didChange {
    [[NSNotificationCenter defaultCenter] postNotificationName:TEPreferencesDidChangeNotification object:self];
}

- (IBAction)checkBoxMatrixAction:(id)sender {
    // Called when a checkbox changes state.
    switch ([sender selectedRow]) {
        case IndentWrappedLinesCheckBoxRow:
            indentWrappedLines = [[sender selectedCell] state];
            [[NSUserDefaults standardUserDefaults] setObject:(indentWrappedLines ? @"YES" : @"NO") forKey:IndentWrappedLinesDefault];
            break;
        case SelectToMatchingBraceCheckBoxRow:
            selectToMatchingBrace = [[sender selectedCell] state];
            [[NSUserDefaults standardUserDefaults] setObject:(selectToMatchingBrace ? @"YES" : @"NO") forKey:SelectToMatchingBraceDefault];
            break;
        case ShowMatchingBraceCheckBoxRow:
            showMatchingBrace = [[sender selectedCell] state];
            [[NSUserDefaults standardUserDefaults] setObject:(showMatchingBrace ? @"YES" : @"NO") forKey:ShowMatchingBraceDefault];
            break;
        case FancyEscapeCompletionCheckBoxRow:
            fancyEscapeCompletion = [[sender selectedCell] state];
            [[NSUserDefaults standardUserDefaults] setObject:(fancyEscapeCompletion ? @"YES" : @"NO") forKey:FancyEscapeCompletionDefault];
            break;
        case EnforceTabStopsCheckBoxRow:
            enforceTabStops = [[sender selectedCell] state];
            [[NSUserDefaults standardUserDefaults] setObject:(enforceTabStops ? @"YES" : @"NO") forKey:EnforceTabStopsDefault];
            break;
    }
    [self didChange];
}

- (IBAction)makeGlobalAction:(id)sender {
    // Copy the settings for all the General settings tab stuff into the global domain.
    NSUserDefaults *defaults = [NSUserDefaults standardUserDefaults];
    NSMutableDictionary *globalDomain;

    globalDomain = [[defaults persistentDomainForName:NSGlobalDomain] mutableCopyWithZone:[self zone]];
    [globalDomain setObject:(indentWrappedLines ? @"YES" : @"NO") forKey:IndentWrappedLinesDefault];
    [globalDomain setObject:(selectToMatchingBrace ? @"YES" : @"NO") forKey:SelectToMatchingBraceDefault];
    [globalDomain setObject:(showMatchingBrace ? @"YES" : @"NO") forKey:ShowMatchingBraceDefault];
    [globalDomain setObject:(fancyEscapeCompletion ? @"YES" : @"NO") forKey:FancyEscapeCompletionDefault];
    [globalDomain setObject:(enforceTabStops ? @"YES" : @"NO") forKey:EnforceTabStopsDefault];
    [globalDomain setObject:[NSString stringWithFormat:@"%u", tabWidth] forKey:TabWidthDefault];
    [globalDomain setObject:[NSString stringWithFormat:@"%u", indentWidth] forKey:IndentWidthDefault];
    [globalDomain setObject:[NSString stringWithFormat:@"%u", wrappedLineIndentWidth] forKey:WrappedLineIndentWidthDefault];
    [defaults setPersistentDomain:globalDomain forName:NSGlobalDomain];
    [globalDomain release];
}

- (IBAction)menuSettingsAction:(id)sender {
    NSUserDefaults *defaults = [NSUserDefaults standardUserDefaults];
    NSMutableDictionary *globalDomain;

    switch ([globalMenuRadioButtons selectedRow]) {
        case GlobalInstallNeverRow:
            globalTryMenu = NO;
            globalTryMenuHard = NO;
            break;
        case GlobalInstallInFormatRow:
            globalTryMenu = YES;
            globalTryMenuHard = NO;
            break;
        case GlobalInstallAlwaysRow:
        default:
            globalTryMenu = YES;
            globalTryMenuHard = YES;
            break;
    }
    globalDebugMenu = ([globalDebugMenuCheckbox state] ? YES : NO);

    switch ([localMenuRadioButtons selectedRow]) {
        case LocalInstallAlwaysRow:
            localStoresMenuSettings = YES;
            localTryMenu = YES;
            localTryMenuHard = YES;
            break;
        case LocalInstallNeverRow:
            localStoresMenuSettings = YES;
            localTryMenu = NO;
            localTryMenuHard = NO;
            break;
        case LocalUseGlobalSettingsRow:
        default:
            localStoresMenuSettings = NO;
            localTryMenu = NO;
            localTryMenuHard = NO;
            break;
    }

    // Do the global domain settings
    globalDomain = [[defaults persistentDomainForName:NSGlobalDomain] mutableCopyWithZone:[self zone]];
    [globalDomain setObject:(globalTryMenu ? @"YES" : @"NO") forKey:TryToInstallExtrasMenuDefault];
    [globalDomain setObject:(globalTryMenuHard ? @"YES" : @"NO") forKey:TryHardToInstallExtrasMenuDefault];
    [globalDomain setObject:(globalDebugMenu ? @"YES" : @"NO") forKey:TryToInstallDebugMenuDefault];
    [defaults setPersistentDomain:globalDomain forName:NSGlobalDomain];
    [globalDomain release];

    // Do the app domain settings
    if (localStoresMenuSettings) {
        [defaults setObject:(localTryMenu ? @"YES" : @"NO") forKey:TryToInstallExtrasMenuDefault];
        [defaults setObject:(localTryMenuHard ? @"YES" : @"NO") forKey:TryHardToInstallExtrasMenuDefault];
    } else {
        [defaults removeObjectForKey:TryToInstallExtrasMenuDefault];
        [defaults removeObjectForKey:TryHardToInstallExtrasMenuDefault];
    }
    [self didChange];
}

- (IBAction)addCompletionAction:(id)sender {
    NSBundle *bundle = [NSBundle bundleForClass:[TEPreferencesController class]];
    NSString *baseStr = NSLocalizedStringFromTableInBundle(@"new", @"TextExtras", bundle, @"Base name for new completion entry.  May have a number added to it.");
    NSString *key = baseStr;
    unsigned counter = 1;
    unsigned row;
    
    while ([completionDict objectForKey:key]) {
        key = [NSString stringWithFormat:@"%@-%u", baseStr, counter++];
    }

    [[self window] makeFirstResponder:completionTableView];
    [completionDict setObject:@"" forKey:key];
    [sortedCompletionKeys release];
    sortedCompletionKeys = [[[completionDict allKeys] sortedArrayUsingSelector:@selector(caseInsensitiveCompare:)] retain];
    [self writeCompletionDict];
    [completionTableView reloadData];
    row = [sortedCompletionKeys indexOfObject:key];
    [completionTableView selectRow:row byExtendingSelection:NO];
    [completionTableView editColumn:0 row:row withEvent:nil select:YES];
    [removeCompletionButton setEnabled:(([completionDict count] > 0) ? YES : NO)];
    [self didChange];
}

- (IBAction)removeCompletionAction:(id)sender {
    int row = [completionTableView selectedRow];
    if (row >= 0) {
        [completionDict removeObjectForKey:[sortedCompletionKeys objectAtIndex:row]];
        [sortedCompletionKeys release];
        sortedCompletionKeys = [[[completionDict allKeys] sortedArrayUsingSelector:@selector(caseInsensitiveCompare:)] retain];
        [self writeCompletionDict];
        [completionTableView reloadData];
        if ([completionDict count] > 0) {
            if (row == [completionDict count]) {
                row--;
            }
            [completionTableView selectRow:row byExtendingSelection:NO];
        }
        [removeCompletionButton setEnabled:(([completionDict count] > 0) ? YES : NO)];
        [self didChange];
    }
}

- (IBAction)addOpenQuicklyAction:(id)sender {
    NSBundle *bundle = [NSBundle bundleForClass:[TEPreferencesController class]];
    NSOpenPanel *op = [NSOpenPanel openPanel];

    [op setPrompt:NSLocalizedStringFromTableInBundle(@"Folder", @"TextExtras", bundle, @"Prompt for add open quickly path open panel.")];
    [op setTitle:NSLocalizedStringFromTableInBundle(@"Add Folder Path", @"TextExtras", bundle, @"Title for add open quickly path open panel.")];
    [op setAllowsMultipleSelection:YES];
    [op setCanChooseDirectories:YES];
    [op setCanChooseFiles:NO];
    if ([op runModalForTypes:nil] == NSOKButton) {
        NSArray *paths = [op filenames];
        unsigned i, c = [paths count];
        
        [self saveOpenQuicklyPaths];
        for (i=0; i<c; i++) {
            [openQuicklyPaths addObject:[[paths objectAtIndex:i] stringByAbbreviatingWithTildeInPath]];
        }
        [self updateOpenQuicklyTextView];
        // Not exactly the most efficient way to do this, but it'll do.
        openQuicklyTextViewChanged = YES;
        [self saveOpenQuicklyPaths];
        [self didChange];
    }
}

- (IBAction)openQuicklyGlobalAction:(id)sender {
    savesOpenQuicklyPathsGlobally = ([[sender selectedCell] state] ? NO : YES);
    if (savesOpenQuicklyPathsGlobally) {
        // This is the default and "normal" state.  Just remove the default.
        [[NSUserDefaults standardUserDefaults] removeObjectForKey:SavesOpenQuicklyPathsGloballyDefault];
        [[NSUserDefaults standardUserDefaults] removeObjectForKey:OpenQuicklyPathsDefault];
    } else {
        [[NSUserDefaults standardUserDefaults] setObject:(savesOpenQuicklyPathsGlobally ? @"YES" : @"NO") forKey:SavesOpenQuicklyPathsGloballyDefault];
    }
    [self saveOpenQuicklyPaths];
}

#ifdef WIN32
- (void)_addSetting:(int)popupRow forKey:(NSString *)dictKey inModifierFlagsDict:(NSMutableDictionary *)dict {
    switch (popupRow) {
        case ModifierKeyCommand:
            [dict setObject:@"Command" forKey:dictKey];
            break;
        case ModifierKeyControl:
            [dict setObject:@"Control" forKey:dictKey];
            break;
        case ModifierKeyAlternate:
            [dict setObject:@"Alt" forKey:dictKey];
            break;
        default:
            break;
    }
}
#endif

- (IBAction)modifierFlagPopUpAction:(id)sender {
#ifdef WIN32
    NSUserDefaults *defaults = [NSUserDefaults standardUserDefaults];
    NSMutableDictionary *globalDomain;
    NSMutableDictionary *newSettings = [[NSMutableDictionary allocWithZone:[self zone]] init];
        
    leftControlModifier = [leftControlPopUp indexOfSelectedItem];
    rightControlModifier = [rightControlPopUp indexOfSelectedItem];
    leftAlternateModifier = [leftAlternatePopUp indexOfSelectedItem];
    rightAlternateModifier = [rightAlternatePopUp indexOfSelectedItem];

    [self _addSetting:leftControlModifier forKey:@"LeftControl" inModifierFlagsDict:newSettings];
    [self _addSetting:rightControlModifier forKey:@"RightControl" inModifierFlagsDict:newSettings];
    [self _addSetting:leftAlternateModifier forKey:@"LeftAlt" inModifierFlagsDict:newSettings];
    [self _addSetting:rightAlternateModifier forKey:@"RightAlt" inModifierFlagsDict:newSettings];

    // Do the global domain settings
    globalDomain = [[defaults persistentDomainForName:NSGlobalDomain] mutableCopyWithZone:[self zone]];
    [globalDomain setObject:newSettings forKey:@"NSModifierFlagMapping"];
    [defaults setPersistentDomain:globalDomain forName:NSGlobalDomain];
    [globalDomain release];

    [self didChange];
#endif
}

- (IBAction)mnemonicsWorkInTextCheckboxAction:(id)sender {
#ifdef WIN32
    NSUserDefaults *defaults = [NSUserDefaults standardUserDefaults];
    NSMutableDictionary *globalDomain;

    mnemonicsWorkInText = [mnemonicsWorkInTextCheckbox state];
    // Do the global domain settings
    globalDomain = [[defaults persistentDomainForName:NSGlobalDomain] mutableCopyWithZone:[self zone]];
    [globalDomain setObject:((mnemonicsWorkInText == NSOnState) ? @"YES" : @"NO") forKey:@"NSMnemonicsWorkInText"];
    [defaults setPersistentDomain:globalDomain forName:NSGlobalDomain];
    [globalDomain release];

    [self didChange];
#endif
}

- (void)controlTextDidEndEditing:(NSNotification *)notification {
    // Called when a text entry form ends editing.
    switch ([widthForm selectedRow]) {
        case TabWidthFormRow:
            tabWidth = [[widthForm selectedCell] intValue];
            if ((tabWidth <= 0) || (tabWidth >= 100)) {
                tabWidth = 8;
                [[widthForm cellAtIndex:TabWidthFormRow] setIntValue:tabWidth];
            }
            [[NSUserDefaults standardUserDefaults] setObject:[NSString stringWithFormat:@"%u", tabWidth] forKey:TabWidthDefault];
            [self didChange];
            break;
        case IndentWidthFormRow:
            indentWidth = [[widthForm selectedCell] intValue];
            if ((indentWidth <= 0) || (indentWidth >= 100)) {
                indentWidth = 8;
                [[widthForm cellAtIndex:IndentWidthFormRow] setIntValue:indentWidth];
            }
            [[NSUserDefaults standardUserDefaults] setObject:[NSString stringWithFormat:@"%u", indentWidth] forKey:IndentWidthDefault];
            [self didChange];
            break;
        case WrappedLineIndentWidthFormRow:
            wrappedLineIndentWidth = [[widthForm selectedCell] intValue];
            if ((wrappedLineIndentWidth <= 0) || (wrappedLineIndentWidth >= 100)) {
                wrappedLineIndentWidth = 8;
                [[widthForm cellAtIndex:WrappedLineIndentWidthFormRow] setIntValue:wrappedLineIndentWidth];
            }
            [[NSUserDefaults standardUserDefaults] setObject:[NSString stringWithFormat:@"%u", wrappedLineIndentWidth] forKey:WrappedLineIndentWidthDefault];
            [self didChange];
            break;
        default:
            break;
    }
}

- (int)numberOfRowsInTableView:(NSTableView *)tableView {
    if (tableView == completionTableView) {
        return [completionDict count];
    } else {
        return 0;
    }
}

- (id)tableView:(NSTableView *)tableView objectValueForTableColumn:(NSTableColumn *)tableColumn row:(int)row {
    if (tableView == completionTableView) {
        return [sortedCompletionKeys objectAtIndex:row];
    } else {
        return nil;
    }
}

- (void)tableView:(NSTableView *)tableView setObjectValue:(id)object forTableColumn:(NSTableColumn *)tableColumn row:(int)row {
    if (tableView == completionTableView) {
        NSString *oldKey = [sortedCompletionKeys objectAtIndex:row];
        if (![object isEqualToString:oldKey]) {
            NSString *existingValue = [completionDict objectForKey:object];
            BOOL allowEdit = YES;
            if (existingValue) {
                // Confirm redefining an existing key.
                NSBundle *bundle = [NSBundle bundleForClass:[TEPreferencesController class]];
                if (NSRunAlertPanel(NSLocalizedStringFromTableInBundle(@"Warning", @"TextExtras", bundle, @"Title of warning alert."), NSLocalizedStringFromTableInBundle(@"There is already a completion entry with the string '%@'.  Do you wish to replace it or cancel?", @"TextExtras", bundle, @"Text of warning alert for replacing an already defined completion string."), NSLocalizedStringFromTableInBundle(@"Replace", @"TextExtras", bundle, @"Text of replace button for duplicate completionstring alert."), NSLocalizedStringFromTableInBundle(@"Cancel", @"TextExtras", bundle, @"Text of cancel button for duplicate completionstring alert."), nil, object) != NSAlertDefaultReturn) {
                    allowEdit = NO;
                }
            }
            if (allowEdit) {
                NSString *value = [completionDict objectForKey:oldKey];
                [completionDict setObject:value forKey:object];
                [completionDict removeObjectForKey:oldKey];
                [sortedCompletionKeys release];
                sortedCompletionKeys = [[[completionDict allKeys] sortedArrayUsingSelector:@selector(caseInsensitiveCompare:)] retain];
                [self writeCompletionDict];
                [completionTableView reloadData];
                [self didChange];
            }
        }
    }
}

- (void)tableViewSelectionDidChange:(NSNotification *)notification {
    NSTableView *tableView = [notification object];
    if (tableView == completionTableView) {
        [self updateCompletionTextView];
    }
}

- (void)textDidBeginEditing:(NSNotification *)notification {
    NSTextView *textView = [notification object];
    if (textView == completionTextView) {
        completionTextViewChanged = YES;
    } else if (textView == openQuicklyTextView) {
        openQuicklyTextViewChanged = YES;
    }
}

- (void)textDidEndEditing:(NSNotification *)notification {
    NSTextView *textView = [notification object];
    if (textView == completionTextView) {
        [self saveSelectedCompletionText];
    } else if (textView == openQuicklyTextView) {
        [self saveOpenQuicklyPaths];
    }
}

- (void)windowWillClose:(NSNotification *)aNotification {
    // Try and make sure to catch edits when the user closes the window.
    [[self window] makeFirstResponder:[self window]];
}

- (void)windowDidResignKey:(NSNotification *)aNotification {
    // Try and make sure to catch edits when the user activates another window.
    [[self window] makeFirstResponder:[self window]];
}

- (void)updateCompletionTextView {
    int row = [completionTableView selectedRow];
    if (row < 0) {
        [completionTextView setString:@""];
        [completionTextView setEditable:NO];
    } else {
        [completionTextView setString:[completionDict objectForKey:[sortedCompletionKeys objectAtIndex:row]]];
        [completionTextView setEditable:YES];
    }
}

- (void)saveSelectedCompletionText {
    if (completionTextViewChanged) {
        int row = [completionTableView selectedRow];
        if (row >= 0) {
            [completionDict setObject:[[completionTextView string] copyWithZone:[self zone]] forKey:[sortedCompletionKeys objectAtIndex:row]];
            [self writeCompletionDict];
        }
        completionTextViewChanged = NO;
        [self didChange];
    }
}

- (void)writeCompletionDict {
    [completionDict writeToFile:[NSHomeDirectory() stringByAppendingPathComponent:@".TECompletionDictionary.plist"] atomically:YES];
}

- (void)updateOpenQuicklyTextView {
    NSArray *paths = [self openQuicklyPaths];
    NSMutableString *str = [NSMutableString string];
    unsigned i, c = [paths count];
    for (i=0; i<c; i++) {
        [str appendFormat:@"%@\n", [paths objectAtIndex:i]];
    }
    [openQuicklyTextView setString:str];
}

- (void)saveOpenQuicklyPaths {
    if (openQuicklyTextViewChanged) {
        NSString *str = [openQuicklyTextView string];
        unsigned length = [str length];
        unsigned start = 0, end = 0, contentsEnd = 0;
        NSString *newPath;
        
        [openQuicklyPaths removeAllObjects];

        while (end < length) {
            [str getLineStart:&start end:&end contentsEnd:&contentsEnd forRange:NSMakeRange(end, 1)];
            newPath = [[str substringWithRange:NSMakeRange(start, contentsEnd - start)] copyWithZone:[self zone]];
            [openQuicklyPaths addObject:newPath];
            [newPath release];
        }

        if ([self savesOpenQuicklyPathsGlobally]) {
            // Write the path to the global domain
            NSUserDefaults *defaults = [NSUserDefaults standardUserDefaults];
            NSMutableDictionary *globalDomainCopy;

            // Make sure the global domain is current before we write to it.
            [defaults synchronize];
            globalDomainCopy = [[defaults persistentDomainForName:NSGlobalDomain] mutableCopyWithZone:[self zone]];
            [globalDomainCopy setObject:openQuicklyPaths forKey:OpenQuicklyPathsDefault];
            [defaults setPersistentDomain:globalDomainCopy forName:NSGlobalDomain];
            // Make sure we write it back out immediately.
            [defaults synchronize];
        } else {
            [[NSUserDefaults standardUserDefaults] setObject:openQuicklyPaths forKey:SavesOpenQuicklyPathsGloballyDefault];
        }
        
        openQuicklyTextViewChanged = NO;
        [self didChange];
    }
}

- (void)readCompletionDictIfNeeded {
    static BOOL beenHere = NO;
    NSString *dictPath = [NSHomeDirectory() stringByAppendingPathComponent:@".TECompletionDictionary.plist"];
    BOOL gotNewDict = NO;

    if (!beenHere) {
        // We start updating on app activation starting after the first time we load for other reasons.
        beenHere = YES;
        [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(applicationDidBecomeActive:) name:NSApplicationDidBecomeActiveNotification object:NSApp];
    }
    if ([[NSFileManager defaultManager] isReadableFileAtPath:dictPath]) {
        BOOL readIt = NO;
        NSDictionary *dict = [[NSFileManager defaultManager] fileAttributesAtPath:dictPath traverseLink:YES];
        NSDate *modDate = [dict fileModificationDate];

        if (!completionDict || !lastReadCompletionDictDate) {
            readIt = YES;
        } else {
            if ([modDate compare:lastReadCompletionDictDate] == NSOrderedDescending) {
                readIt = YES;
            }
        }
        if (readIt) {
            NSDictionary *dict = [NSDictionary dictionaryWithContentsOfFile:dictPath];

            [completionDict release];
            completionDict = nil;
            [lastReadCompletionDictDate release];
            lastReadCompletionDictDate = [modDate retain];

            if (dict) {
                completionDict = [dict mutableCopyWithZone:[self zone]];
                gotNewDict = YES;
            } else {
                NSLog(@"TextExtras: Could not read '.TECompletionDictionary.plist' file as a dictionary.");
            }
        }
    } else {
        [lastReadCompletionDictDate release];
        lastReadCompletionDictDate = [[NSDate date] retain];
    }
    if (!completionDict) {
        completionDict = [[NSMutableDictionary allocWithZone:[self zone]] init];
        gotNewDict = YES;
    }
    if (gotNewDict) {
        [sortedCompletionKeys release];
        sortedCompletionKeys = [[[completionDict allKeys] sortedArrayUsingSelector:@selector(caseInsensitiveCompare:)] retain];
        if ([self isWindowLoaded]) {
            [completionTableView reloadData];
            [self updateCompletionTextView];
        }
        [self didChange];
    }
}

- (void)applicationDidBecomeActive:(NSNotification *)notification {
    [self readCompletionDictIfNeeded];
}

@end
