// TESpecialCharactersController.m
// TextExtras - Yellow Box
//
// Copyright 1996-1999, Mike Ferris.
// All rights reserved.

#import "TESpecialCharactersController.h"

@implementation TESpecialCharactersController

+ (id)sharedSpecialCharactersController {
    static TESpecialCharactersController *sharedInstance = nil;
    if (!sharedInstance) {
        sharedInstance = [[TESpecialCharactersController alloc] init];
    }
    return sharedInstance;
}

- (id)init {
    self = [super initWithWindowNibName:@"TESpecialCharacters" owner:self];
    if (self) {
        [self setWindowFrameAutosaveName:@"TextExtrasSpeicalCharacters"];
    }
    return self;
}

- (void)dealloc {
    [buttonSetNames release];
    [buttonSetStrings release];
    [super dealloc];
}

- (NSRange)characterRangeFromString:(NSString *)string {
    NSScanner *scanner = [[NSScanner allocWithZone:[self zone]] initWithString:string];
    unsigned start, end;
    NSRange result;
    
    if (![scanner scanHexInt:&start]) {
        // Invalid string.
        result = NSMakeRange(NSNotFound, 0);
    } else if ([scanner scanString:@"-" intoString:NULL]) {
        // It looks like a range
        if (![scanner scanHexInt:&end]) {
            // Invalid string
            result = NSMakeRange(NSNotFound, 0);
        } else {
            if (start > end) {
                // Invalid string
                result =  NSMakeRange(NSNotFound, 0);
            } else {
                // It is a range
                result =  NSMakeRange(start, ((end + 1) - start));
            }
        }
    } else {
        // It is just a single character
        result =  NSMakeRange(start, 1);
    }
    [scanner release];
    if ((result.location != NSNotFound) && ((result.location > 0xFFFF) || (NSMaxRange(result) > 0x10000))) {
        // Out of range range.
        result = NSMakeRange(NSNotFound, 0);
    }
    return result;
}

- (NSString *)buttonSetStringFromArray:(NSArray *)array {
    unsigned arrayIndex, arrayCount;
    unichar *buff = NULL;
    unsigned buffSize, strLen;
    NSString *curElement;
    NSRange curRange;
    unsigned rangeIndex;
    NSString *result = nil;

    if (![array isKindOfClass:[NSArray class]]) {
        return nil;
    }

    buffSize = 512;
    buff = NSZoneMalloc([self zone], sizeof(unichar) * buffSize);
    if (!buff) {
        return nil;
    }
    strLen = 0;
    
    arrayCount = [array count];
    for (arrayIndex=0; arrayIndex<arrayCount; arrayIndex++) {
        curElement = [array objectAtIndex:arrayIndex];
        curRange = [self characterRangeFromString:curElement];
        if (curRange.location != NSNotFound) {
            if ((strLen + curRange.length) > buffSize) {
                buffSize *= 2;
                if ((strLen + curRange.length) > buffSize) {
                    buffSize = (strLen + curRange.length);
                }
                buff = NSZoneRealloc([self zone], buff, sizeof(unichar) * buffSize);
                if (!buff) {
                    return nil;
                }
            }
            for (rangeIndex=curRange.location; rangeIndex<NSMaxRange(curRange); rangeIndex++) {
                buff[strLen++] = rangeIndex;
            }
        } else {
            NSLog(@"TextExtras: Invalid Unicode range string '%@' was skipped.", curElement);
        }
    }
    if (strLen > 0) {
        result = [NSString stringWithCharacters:buff length:strLen];
    }
    NSZoneFree([self zone], buff);

    return result;
}

#define BUTTON_SET_FILE @".TESpecialCharacterSets.plist"
#define DEFAULT_BUTTON_SET_FILE @"TEDefaultSpecialCharacterSets.plist"

- (void)readButtonSets {
    if (!buttonSetStrings) {
        // Read the user's button sets file.
        NSString *filePath = [NSHomeDirectory() stringByAppendingPathComponent:BUTTON_SET_FILE];
        BOOL foundSets;
        
        buttonSetNames = [[NSMutableArray allocWithZone:NULL] init];
        buttonSetStrings = [[NSMutableArray allocWithZone:NULL] init];

        foundSets = [[NSFileManager defaultManager] isReadableFileAtPath:filePath];
        if (!foundSets) {
            filePath = [[NSBundle bundleForClass:[TESpecialCharactersController class]] pathForResource:DEFAULT_BUTTON_SET_FILE ofType:nil];
            foundSets = [[NSFileManager defaultManager] isReadableFileAtPath:filePath];
        }

        if (foundSets) {
            NSString *fileString = [NSString stringWithContentsOfFile:filePath];
            NSDictionary *plist = nil;

            NS_DURING
                plist = [fileString propertyList];
            NS_HANDLER
                plist = nil;
            NS_ENDHANDLER

            if (!plist) {
                NSLog(@"TextExtras: The special character sets file is not a valid property list.  It should be a dictionary of arrays.");
            } else if (![plist isKindOfClass:[NSDictionary class]]) {
                NSLog(@"TextExtras: The special character sets file must contain a top-level dictionary.  It should be a dictionary of arrays.");
            } else {
                NSEnumerator *keyEnum = [plist keyEnumerator];
                NSString *curName;
                NSArray *curElementArray;
                NSString *curButtonSetString;
                
                while ((curName = [keyEnum nextObject]) != nil) {
                    curElementArray = [plist objectForKey:curName];
                    curButtonSetString = [self buttonSetStringFromArray:curElementArray];
                    if (!curButtonSetString) {
                        NSLog(@"TextExtras: Empty or invalid special character set named '%@' was skipped.", curName);
                    } else {
                        [buttonSetNames addObject:curName];
                        [buttonSetStrings addObject:curButtonSetString];
                    }
                }
            }
        }
    }
}

- (void)windowDidLoad {
    unsigned i, c;

    [(NSPanel *)[self window] setBecomesKeyOnlyIfNeeded:YES];
    [popup removeAllItems];
    [self readButtonSets];
    c = [buttonSetNames count];
    if (c==0) {
        [popup addItemWithTitle:NSLocalizedStringFromTableInBundle(@"<None>", @"TextExtras", [NSBundle bundleForClass:[TESpecialCharactersController class]], @"String to indicate there are no special character sets available.")];
    } else {
        for (i=0; i<c; i++) {
            [popup addItemWithTitle:[buttonSetNames objectAtIndex:i]];
        }
    }
    [popup selectItemAtIndex:0];
    [self popupAction:self];
}

- (void)fillMatrixWithCharactersInString:(NSString *)charString {
    unsigned strLen = [charString length];
    NSButtonCell *curCell;
    
    if (strLen == 0) {
        unsigned colIndex;
        [matrix renewRows:1 columns:8];
        [matrix sizeToCells];
        for (colIndex=0; colIndex<8; colIndex++) {
            curCell = [matrix cellAtRow:0 column:colIndex];
            [curCell setTitle:@""];
            [curCell setEnabled:NO];
        }
    } else {
        unsigned rowIndex, rowCount = ((strLen + 7) / 8), colIndex, colCount = 8;
        unsigned curCharIndex = 0;
        unichar curChar;
        NSString *curTitle;

        [matrix renewRows:rowCount columns:colCount];
        [matrix sizeToCells];
        for (rowIndex = 0 ; rowIndex < rowCount; rowIndex++) {
            for (colIndex = 0 ; colIndex < colCount; colIndex++, curCharIndex++) {
                curCell = [matrix cellAtRow:rowIndex column:colIndex];
                if (curCharIndex < strLen) {
                    curChar = [charString characterAtIndex:curCharIndex];
                    curTitle = [[NSString allocWithZone:[self zone]] initWithCharacters:&curChar length:1];
                    [curCell setTitle:curTitle];
                    [curTitle release];
                    [curCell setEnabled:YES];
                } else {
                    [curCell setTitle:@""];
                    [curCell setEnabled:NO];
                }
            }
        }
    }
    //[matrix setNeedsDisplay:YES];  // Should not have to do this.  Bad NSMatrix, bad!
    [matrix scrollPoint:NSMakePoint(0.0, 0.0)];
}

- (IBAction)popupAction:(id)sender {
    int selectedIndex = [popup indexOfSelectedItem];

    if ((selectedIndex >= 0) && (selectedIndex < [buttonSetNames count])) {
        [self fillMatrixWithCharactersInString:[buttonSetStrings objectAtIndex:selectedIndex]];
    } else {
        [self fillMatrixWithCharactersInString:@""];
    }
}

- (IBAction)matrixAction:(id)sender {
    NSWindow *keyWindow = [NSApp keyWindow];
    id firstResponder;
    
    if (keyWindow == [self window]) {
        // Why are we key?
        return;
    }
    firstResponder = [keyWindow firstResponder];
    if (firstResponder && [firstResponder isKindOfClass:[NSTextView class]]) {
        NSButtonCell *cell = [matrix selectedCell];
        NSString *str = [cell title];
        if (str && ![str isEqualToString:@""]) {
            [firstResponder insertText:[cell title]];
        }
    }
}

@end
