// TETextWatcher.m
// TextExtras - Yellow Box
//
// Copyright 1996-1999, Mike Ferris.
// All rights reserved.

#import "TETextWatcher.h"
#import "TETextUtils.h"
#import "TEInfoPanelController.h"
#import "TEPreferencesController.h"
#import "TEOpenQuicklyController.h"
#import "TEPipePanelController.h"
#import "TESpecialCharactersController.h"

@implementation TETextWatcher

// ********************** Handling preference changes **********************
static BOOL TE_observingTextStorageDidProcessEditing = NO;
static BOOL TE_observingTextViewDidChangeSelection = NO;
static BOOL TE_observingApplicationWillUpdate = NO;
static BOOL TE_fancyEscapeCompletion = NO;

+ (void)updateObserversFromPreferences:(NSNotification *)notification {
    TEPreferencesController *prefs = [TEPreferencesController sharedPreferencesController];

    // Set up the observer for the Scott-wrap and enforce-tabs features.
    if ([prefs indentWrappedLines] || [prefs enforceTabStops]) {
        if (!TE_observingTextStorageDidProcessEditing) {
            [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(textStorageDidProcessEditing:) name:NSTextStorageDidProcessEditingNotification object:nil];
            TE_observingTextStorageDidProcessEditing = YES;
        }
    } else {
        if (TE_observingTextStorageDidProcessEditing) {
            [[NSNotificationCenter defaultCenter] removeObserver:self name:NSTextStorageDidProcessEditingNotification object:nil];
            TE_observingTextStorageDidProcessEditing = NO;
        }
    }

    // Set up the observer for the matching delimiter and/or fancy esc-completion features.
    if ([prefs fancyEscapeCompletion] || [prefs selectToMatchingBrace] || [prefs showMatchingBrace]) {
        if (!TE_observingTextViewDidChangeSelection) {
            [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(textViewDidChangeSelection:) name:NSTextViewDidChangeSelectionNotification object:nil];
            TE_observingTextViewDidChangeSelection = YES;
        }
    } else {
        if (TE_observingTextViewDidChangeSelection) {
            [[NSNotificationCenter defaultCenter] removeObserver:self name:NSTextViewDidChangeSelectionNotification object:nil];
            TE_observingTextViewDidChangeSelection = NO;
        }
    }

    // Set up the other observer for the fancy esc-completion feature.
    if ([prefs fancyEscapeCompletion]) {
        if (!TE_observingApplicationWillUpdate) {
            [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(applicationWillUpdate:) name:NSApplicationWillUpdateNotification object:NSApp];
            TE_observingApplicationWillUpdate = YES;
        }
        TE_fancyEscapeCompletion = YES;
    } else {
        if (TE_observingApplicationWillUpdate) {
            [[NSNotificationCenter defaultCenter] removeObserver:self name:NSApplicationWillUpdateNotification object:NSApp];
            TE_observingApplicationWillUpdate = NO;
        }
        TE_fancyEscapeCompletion = NO;
    }

}

// ************************* Start watching *************************

+ (void)load {
    TEPreferencesController *prefs = [TEPreferencesController sharedPreferencesController];
    
    // Set up ourself as the observer for any notifications we need to listen to according to the preferences.
    [self updateObserversFromPreferences:nil];
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(updateObserversFromPreferences:) name:TEPreferencesDidChangeNotification object:prefs];

    // Install the menu.  We will try at least three different ways of getting notified.
    if ([TEPreferencesController tryToInstallExtrasMenu]) {
        // We try to install the menu at willFinishLaunching, but we will try again at didFinishLaunching just in case willFinishLaunching passed us by.
        [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(tryToInstallExtrasMenu:) name:NSApplicationWillFinishLaunchingNotification object:NSApp];
        [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(tryToInstallExtrasMenu:) name:NSApplicationDidFinishLaunchingNotification object:NSApp];
        // Finally, there are cases (on windows) where the didFinishLaunching has already happened by the time we get loaded.  So we'll also try it with a timer too.
        [self performSelector:@selector(tryToInstallExtrasMenu:) withObject:nil afterDelay:0.0];
    }
}

// ********************** Install extras menu **********************

+ (void)installDebugMenuIfNecessary:(NSMenu *)extrasMenu {
    if ([TEPreferencesController tryToInstallDebugMenu]) {
        NSBundle *bundle = [NSBundle bundleForClass:self];
        NSMenu *debugMenu = [[NSMenu allocWithZone:[NSMenu menuZone]] initWithTitle:NSLocalizedStringFromTableInBundle(@"Debug", @"TextExtras", bundle, @"Title of Debug submenu")];
        NSMenuItem *item;

        [extrasMenu addItem:[NSMenuItem separatorItem]];
        item = [extrasMenu addItemWithTitle:NSLocalizedStringFromTableInBundle(@"Debug", @"TextExtras", bundle, @"Title of Debug submenu") action:NULL keyEquivalent:@""];
        [extrasMenu setSubmenu:debugMenu forItem:item];

        // Add the items for the commands.
        [debugMenu addItemWithTitle:NSLocalizedStringFromTableInBundle(@"Log TextView Descriptions", @"TextExtras", bundle, @"Menu item name") action:@selector(TE_logTextViewDescriptions:) keyEquivalent:@""];
        [debugMenu addItemWithTitle:NSLocalizedStringFromTableInBundle(@"Log TextContainer Descriptions", @"TextExtras", bundle, @"Menu item name") action:@selector(TE_logTextContainerDescriptions:) keyEquivalent:@""];
        [debugMenu addItemWithTitle:NSLocalizedStringFromTableInBundle(@"Log LayoutManager Description", @"TextExtras", bundle, @"Menu item name") action:@selector(TE_logLayoutManagerDescription:) keyEquivalent:@""];
        [debugMenu addItemWithTitle:NSLocalizedStringFromTableInBundle(@"Log LayoutManager Container Description", @"TextExtras", bundle, @"Menu item name") action:@selector(TE_logLayoutManagerContainerDescription:) keyEquivalent:@""];
        [debugMenu addItemWithTitle:NSLocalizedStringFromTableInBundle(@"Log LayoutManager Line Fragment Description", @"TextExtras", bundle, @"Menu item name") action:@selector(TE_logLayoutManagerLineFragmentDescription:) keyEquivalent:@""];
        [debugMenu addItemWithTitle:NSLocalizedStringFromTableInBundle(@"Log LayoutManager Verbose Line Fragment Description", @"TextExtras", bundle, @"Menu item name") action:@selector(TE_logLayoutManagerVerboseLineFragmentDescription:) keyEquivalent:@""];
        [debugMenu addItemWithTitle:NSLocalizedStringFromTableInBundle(@"Log LayoutManager Glyph Description", @"TextExtras", bundle, @"Menu item name") action:@selector(TE_logLayoutManagerGlyphDescription:) keyEquivalent:@""];
        [debugMenu addItemWithTitle:NSLocalizedStringFromTableInBundle(@"Log TextStorage Description", @"TextExtras", bundle, @"Menu item name") action:@selector(TE_logTextStorageDescription:) keyEquivalent:@""];
    }
}

+ (void)tryToInstallExtrasMenu:(NSNotification *)notification {
    static BOOL alreadyInstalled = NO;
    NSMenu *mainMenu = nil;

    // We only try once, but that one time needs to be after there is a main menu.  If there is none now, we will pass and hope this method gets called again later.
    if (!alreadyInstalled && ((mainMenu = [NSApp mainMenu]) != nil)) {
        NSMenu *insertIntoMenu = nil;
        NSMenuItem *item;
        unsigned insertLoc = NSNotFound;
        NSBundle *bundle = [NSBundle bundleForClass:self];
        // Succeed or fail, we do not try again.
        alreadyInstalled = YES;

        item = [mainMenu itemWithTitle:NSLocalizedStringFromTableInBundle(@"Format", @"TextExtras", bundle, @"Title of Format menu item")];
        if (!item) {
            // MF: Try again with unlocalized string "Format".  This is a backstop that assumes that any app not localized into the user's chosen language was probably developed in English...
            item = [mainMenu itemWithTitle:@"Format"];
        }
        if (item && [item hasSubmenu]) {
            insertIntoMenu = [item target];
            item = [insertIntoMenu itemWithTitle:NSLocalizedStringFromTableInBundle(@"Text", @"TextExtras", bundle, @"Title of Format->Text menu item")];
            if (!item) {
                // MF: Try again with unlocalized string "Text".  This is a backstop that assumes that any app not localized into the user's chosen language was probably developed in English...
                item = [insertIntoMenu itemWithTitle:@"Text"];
            }
            if (item) {
                insertLoc = [[insertIntoMenu itemArray] indexOfObjectIdenticalTo:item] + 1;
            } else {
                insertLoc = [[insertIntoMenu itemArray] count];
            }
        } else if (([mainMenu numberOfItems] > 0) && [TEPreferencesController tryHardToInstallExtrasMenu]) {
            // As a last resort, add it to the main menu.  We try to put it right before the Windows menu if there is one, or right before the Services menu if there is one, and if there's neither we put it right before the the last submenu item (ie above Quit and Hide on Mach, at the end on Windows.)
            NSMenu * beforeSubmenu = [NSApp windowsMenu];

            if (!beforeSubmenu) {
                beforeSubmenu = [NSApp servicesMenu];
            }

            insertIntoMenu = mainMenu;

            if (beforeSubmenu) {
                NSArray *itemArray = [insertIntoMenu itemArray];
                unsigned i, c = [itemArray count];

                // Default to end of menu
                insertLoc = c;

                for (i=0; i<c; i++) {
                    if ([[itemArray objectAtIndex:i] target] == beforeSubmenu) {
                        insertLoc = i;
                        break;
                    }
                }
            } else {
                NSArray *itemArray = [insertIntoMenu itemArray];
                unsigned i = [itemArray count];

                // Default to end of menu
                insertLoc = i;
                
                while (i-- > 0) {
                    if ([[itemArray objectAtIndex:i] hasSubmenu]) {
                        insertLoc = i+1;
                        break;
                    }
                }
            }
        }

        if (insertIntoMenu) {
            NSMenu *extrasMenu = [[NSMenu allocWithZone:[NSMenu menuZone]] initWithTitle:NSLocalizedStringFromTableInBundle(@"TextExtras", @"TextExtras", bundle, @"Title of TextExtras menu")];

            item = [insertIntoMenu insertItemWithTitle:NSLocalizedStringFromTableInBundle(@"TextExtras", @"TextExtras", bundle, @"Title of TextExtras menu") action:NULL keyEquivalent:@"" atIndex:insertLoc];
            [insertIntoMenu setSubmenu:extrasMenu forItem:item];

            // Add the items for the commands.
            item = [extrasMenu addItemWithTitle:NSLocalizedStringFromTableInBundle(@"About TextExtras", @"TextExtras", bundle, @"Title of Info Panel menu item") action:@selector(showWindow:) keyEquivalent:@""];
            [item setTarget:[TEInfoPanelController sharedInfoPanelController]];
            item = [extrasMenu addItemWithTitle:NSLocalizedStringFromTableInBundle(@"Preferences...", @"TextExtras", bundle, @"Title of Preferences Panel menu item") action:@selector(showWindow:) keyEquivalent:@""];
            [item setTarget:[TEPreferencesController sharedPreferencesController]];
            [extrasMenu addItem:[NSMenuItem separatorItem]];
            [extrasMenu addItemWithTitle:NSLocalizedStringFromTableInBundle(@"Nest", @"TextExtras", bundle, @"Title of Nest menu item") action:@selector(TE_indentRight:) keyEquivalent:@"]"];
            [extrasMenu addItemWithTitle:NSLocalizedStringFromTableInBundle(@"Unnest", @"TextExtras", bundle, @"Title of Unnest menu item") action:@selector(TE_indentLeft:) keyEquivalent:@"["];
            [extrasMenu addItem:[NSMenuItem separatorItem]];
            [extrasMenu addItemWithTitle:NSLocalizedStringFromTableInBundle(@"Toggle Control Characters", @"TextExtras", bundle, @"Title of Toggle control characters menu item") action:@selector(TE_toggleShowsControlCharacters:) keyEquivalent:@"^"];
            [extrasMenu addItemWithTitle:NSLocalizedStringFromTableInBundle(@"Parse as Property List", @"TextExtras", bundle, @"Title of Parse as Property List menu item") action:@selector(TE_parseAsPropertyList:) keyEquivalent:@"^"];
            [extrasMenu addItem:[NSMenuItem separatorItem]];
            [extrasMenu addItemWithTitle:NSLocalizedStringFromTableInBundle(@"Standardize EOL to LF", @"TextExtras", bundle, @"Title of Standardize EOL to LF menu item") action:@selector(TE_standardizeEndOfLineToLF:) keyEquivalent:@""];
            [extrasMenu addItemWithTitle:NSLocalizedStringFromTableInBundle(@"Standardize EOL to CRLF", @"TextExtras", bundle, @"Title of Standardize EOL to CRLF menu item") action:@selector(TE_standardizeEndOfLineToCRLF:) keyEquivalent:@""];
            [extrasMenu addItemWithTitle:NSLocalizedStringFromTableInBundle(@"Standardize EOL to CR", @"TextExtras", bundle, @"Title of Standardize EOL to CR menu item") action:@selector(TE_standardizeEndOfLineToCR:) keyEquivalent:@""];
            [extrasMenu addItemWithTitle:NSLocalizedStringFromTableInBundle(@"Standardize EOL to PS", @"TextExtras", bundle, @"Title of Standardize EOL to PS (paragraph separator) menu item") action:@selector(TE_standardizeEndOfLineToParagraphSeparator:) keyEquivalent:@""];
            [extrasMenu addItemWithTitle:NSLocalizedStringFromTableInBundle(@"Standardize EOL to LS", @"TextExtras", bundle, @"Title of Standardize EOL to LS (line separator) menu item") action:@selector(TE_standardizeEndOfLineToLineSeparator:) keyEquivalent:@""];
            [extrasMenu addItem:[NSMenuItem separatorItem]];
            [extrasMenu addItemWithTitle:NSLocalizedStringFromTableInBundle(@"Show Goto Panel", @"TextExtras", bundle, @"Title of Goto Panel menu item") action:@selector(TE_gotoPanel:) keyEquivalent:@"l"];
            item = [extrasMenu addItemWithTitle:NSLocalizedStringFromTableInBundle(@"Show Special Characters", @"TextExtras", bundle, @"Title of Show Special Characters menu item") action:@selector(showWindow:) keyEquivalent:@""];
            [item setTarget:[TESpecialCharactersController sharedSpecialCharactersController]];
            item = [extrasMenu addItemWithTitle:NSLocalizedStringFromTableInBundle(@"Open Quickly...", @"TextExtras", bundle, @"Title of Open Quickly menu item") action:@selector(runOpenQuicklyPanel:) keyEquivalent:@"D"];
            [item setTarget:[TEOpenQuicklyController sharedOpenQuicklyController]];
            [extrasMenu addItemWithTitle:NSLocalizedStringFromTableInBundle(@"Execute Pipe...", @"TextExtras", bundle, @"Title of Pipe Panel menu item") action:@selector(TE_executePipe:) keyEquivalent:@"|"];
            item = [extrasMenu addItemWithTitle:NSLocalizedStringFromTableInBundle(@"User Pipes", @"TextExtras", bundle, @"Title of User Pipes submenu item") action:NULL keyEquivalent:@""];
            [extrasMenu setSubmenu:[TEPipePanelController userPipesMenu] forItem:item];

            [self installDebugMenuIfNecessary:extrasMenu];
        }

    }
}

// ******************* Scott wrap feature and enforcing tab stops *******************

+ (void)fixWrappedLineIndentsInEditedTextStorage:(NSTextStorage *)textStorage {
    int wrapIndent = [[TEPreferencesController sharedPreferencesController] wrappedLineIndentWidth];
    NSString *string = [textStorage string];
    NSRange editedRange = [textStorage editedRange];
    NSDictionary *attrs;
    NSRange eRange, paraRange, nextParaRange, tempRange;
    unsigned numSpaces;
    float indentWidth;
    NSFont *font;
    NSParagraphStyle *paraStyle;
    NSMutableParagraphStyle *newStyle;
    unsigned tabWidth = [[TEPreferencesController sharedPreferencesController] tabWidth];
    
    editedRange = [string lineRangeForRange:editedRange];

    // We will traverse the edited range by paragraphs fixing the paragraph styles
    while (editedRange.length > 0) {
        attrs = [textStorage attributesAtIndex:editedRange.location effectiveRange:&eRange];
        font = [attrs objectForKey:NSFontAttributeName];
        if (!font) {
            font = [NSFont fontWithName:@"Helvetica" size:12.0];
        }

        // We can only proceed if this font has a "space" glyph.
        if ([font glyphIsEncoded:' ']) {
            paraStyle = [attrs objectForKey:NSParagraphStyleAttributeName];
            if (!paraStyle) {
                paraStyle = [NSParagraphStyle defaultParagraphStyle];
            }

            // Loop over inidividual paragraphs and change them.
            while (eRange.length > 0) {
                paraRange = [string lineRangeForRange:NSMakeRange(eRange.location, 1)];
                tempRange = paraRange;
                numSpaces = TE_numberOfLeadingSpacesFromRangeInString(string, &tempRange, tabWidth);
                // Coalesce any contiguous paragraphs with the same leading whitespace.
                while (NSMaxRange(paraRange) < NSMaxRange(eRange)) {
                    nextParaRange = [string lineRangeForRange:NSMakeRange(NSMaxRange(paraRange), 1)];
                    tempRange = nextParaRange;
                    if (TE_numberOfLeadingSpacesFromRangeInString(string, &tempRange, tabWidth) == numSpaces) {
                        paraRange = NSUnionRange(paraRange, nextParaRange);
                    } else {
                        break;
                    }
                }
                numSpaces += wrapIndent;
                indentWidth = numSpaces * [font advancementForGlyph:' '].width;

                if (([paraStyle firstLineHeadIndent] != 0.0) || ([paraStyle headIndent] != indentWidth)) {
                    // We need to alter the style.
                    newStyle = [paraStyle mutableCopy];
                    [newStyle setFirstLineHeadIndent:0.0];
                    [newStyle setHeadIndent:indentWidth];
                    [textStorage addAttribute:NSParagraphStyleAttributeName value:newStyle range:paraRange];
                    [newStyle release];
                }
                if (paraRange.length < eRange.length) {
                    eRange.length -= paraRange.length;
                    eRange.location += paraRange.length;
                } else {
                    eRange = NSMakeRange(NSMaxRange(eRange), 0);
                }
            }
        }

        if (NSMaxRange(eRange) < NSMaxRange(editedRange)) {
            editedRange.length = NSMaxRange(editedRange) - NSMaxRange(eRange);
            editedRange.location = NSMaxRange(eRange);
        } else {
            editedRange = NSMakeRange(NSMaxRange(editedRange), 0);
        }
    }
}

+ (void)fixTabStopsInEditedTextStorage:(NSTextStorage *)textStorage {
    NSString *string = [textStorage string];
    
    if ([string length] > 0) {
        // Generally NSTextStorage's attached to plain text NSTextViews only have one font.  But this is not generally true.  To ensure the tabstops are uniform throughout the document we always base them on the font of the first character in the NSTextStorage.
        NSFont *font = [textStorage attribute:NSFontAttributeName atIndex:0 effectiveRange:NULL];

        // Substitute a screen font is the layout manager will do so for display.
        // MF: printing will prbably be an issue here...
        font = [[[textStorage layoutManagers] objectAtIndex:0] substituteFontForFont:font];
        
        if ([font isFixedPitch]) {
            unsigned tabWidth = [[TEPreferencesController sharedPreferencesController] tabWidth];
            NSArray *desiredTabStops = TE_tabStopArrayForFontAndTabWidth(font, tabWidth);
            NSRange editedRange = [textStorage editedRange];
            NSRange eRange;
            NSParagraphStyle *paraStyle;
            NSMutableParagraphStyle *newStyle;

            editedRange = [string lineRangeForRange:editedRange];

            // We will traverse the edited range by paragraphs fixing the paragraph styles
            while (editedRange.length > 0) {
                paraStyle = [textStorage attribute:NSParagraphStyleAttributeName atIndex:editedRange.location longestEffectiveRange:&eRange inRange:editedRange];
                if (!paraStyle) {
                    paraStyle = [NSParagraphStyle defaultParagraphStyle];
                }
		eRange = NSIntersectionRange(editedRange, eRange);
                if (![[paraStyle tabStops] isEqual:desiredTabStops]) {
                    // Make sure we don't change stuff outside editedRange.
                    newStyle = [paraStyle mutableCopyWithZone:[textStorage zone]];
                    [newStyle setTabStops:desiredTabStops];
                    [textStorage addAttribute:NSParagraphStyleAttributeName value:newStyle range:eRange];
                }
                if (NSMaxRange(eRange) < NSMaxRange(editedRange)) {
                    editedRange.length = NSMaxRange(editedRange) - NSMaxRange(eRange);
                    editedRange.location = NSMaxRange(eRange);
                } else {
                    editedRange = NSMakeRange(NSMaxRange(editedRange), 0);
                }
            }
        }
    }
}

+ (void)textStorageDidProcessEditing:(NSNotification *)notification {
    // Some text storage somewhere changed.  We want to do the scott-wrap feature or the enforce tab stops feature only for a very specific kind of text web...
    TEPreferencesController *prefs = [TEPreferencesController sharedPreferencesController];
    NSTextStorage *textStorage = [notification object];
    id scratch = [textStorage layoutManagers];

    if ((scratch == nil) || ([scratch count] != 1)) {
        // Don't do either for NSTextStorage objects with more than one layout manager
        return;
    }
    scratch = [[scratch objectAtIndex:0] textContainers];
    if ((scratch == nil) || ([scratch count] != 1)) {
        // Don't do either for NSLayoutManager objects with more than one text container
        return;
    }
    scratch = [[scratch objectAtIndex:0] textView];
    if ((scratch == nil) || [scratch isRichText] || [scratch isFieldEditor] /* || ![scratch isEditable] */) {
        // We must have no text view or a non-field editor, plain text, or non-editable text view.
        return;
    }

    if ([prefs indentWrappedLines]) {
        [self fixWrappedLineIndentsInEditedTextStorage:textStorage];
    }

    if ([prefs enforceTabStops]) {
        [self fixTabStopsInEditedTextStorage:textStorage];
    }
}

// ************************ Fancy escape completion stuff ************************
static NSTextView *TE_keyTextView;
static unsigned TE_completionStartLocation = NSNotFound;
static NSMutableArray *TE_foundCompletions = nil;
static BOOL TE_isCyclingCompletions = NO;

+ (void)clearFoundEscapeCompletions {
    if (TE_fancyEscapeCompletion) {
        // We don't empty the list immediately because someone may reset the completion start index, and if they do we will restore the found completions.
        TE_completionStartLocation = NSNotFound;
    }
}

+ (void)setEscapeCompletionStartLocation:(unsigned)location {
    if (TE_fancyEscapeCompletion) {
        // The complete: binding should do its thing (probably using alreadyFoundEscapeCompletion:) and then, when it is done it should set the start location (which as a side effect rescues the previous contents of the found completion list) and then add the new completion to the found list.
        TE_completionStartLocation = location;
    }
}

+ (unsigned)escapeCompletionStartLocation {
    if (TE_fancyEscapeCompletion) {
        return TE_completionStartLocation;
    } else {
        return NSNotFound;
    }
}

+ (void)addFoundEscapeCompletion:(NSString *)foundCompletion {
    if (TE_fancyEscapeCompletion) {
        if (TE_foundCompletions == nil) {
            TE_foundCompletions = [[NSMutableArray allocWithZone:NULL] initWithCapacity:1];
        }
        if (TE_completionStartLocation == NSNotFound) {
            // This means we shouldn't be storing completions.  Technically this shouldn't happen.
            [TE_foundCompletions removeAllObjects];
            TE_isCyclingCompletions = NO;
        }
        [TE_foundCompletions addObject:foundCompletion];
    }
}

+ (BOOL)alreadyFoundEscapeCompletion:(NSString *)completion {
    if (TE_fancyEscapeCompletion) {
        if (TE_completionStartLocation == NSNotFound) {
            // This means we shouldn't be storing completions so we empty the list.
            if (TE_foundCompletions != nil) {
                [TE_foundCompletions removeAllObjects];
                TE_isCyclingCompletions = NO;
            }
        }
        return (((TE_foundCompletions != nil) && ([TE_foundCompletions containsObject:completion])) ? YES : NO);
    } else {
        return NO;
    }
}

+ (NSString *)escapeCompletionAfterOldCompletion:(NSString *)completion {
    if (TE_fancyEscapeCompletion) {
        if (TE_completionStartLocation == NSNotFound) {
            // This means we shouldn't be storing completions so we empty the list.
            if (TE_foundCompletions != nil) {
                [TE_foundCompletions removeAllObjects];
                TE_isCyclingCompletions = NO;
            }
        }
        if (TE_foundCompletions) {
            unsigned completionIndex = [TE_foundCompletions indexOfObject:completion];
            if (completionIndex != NSNotFound) {
                completionIndex++;
                if (completionIndex == [TE_foundCompletions count]) {
                    completionIndex = 0;
                }
                TE_isCyclingCompletions = YES;
                return [TE_foundCompletions objectAtIndex:completionIndex];
            }
        }
    }
    return nil;
}

+ (BOOL)isCyclingEscapeCompletions {
    if (TE_fancyEscapeCompletion && (TE_completionStartLocation != NSNotFound)) {
        return TE_isCyclingCompletions;
    } else {
        return NO;
    }
}

+ (void)applicationWillUpdate:(NSNotification *)notification {
    NSWindow *targetWindow = [NSApp keyWindow];
    id target = [targetWindow firstResponder];
    if (target != TE_keyTextView) {
        if ([target isKindOfClass:[NSTextView class]]) {
            target = [[target layoutManager] firstTextView];
            if (target != TE_keyTextView) {
                TE_keyTextView = target;
                [self clearFoundEscapeCompletions];
            }
        } else if (TE_keyTextView != nil) {
            TE_keyTextView = nil;
            [self clearFoundEscapeCompletions];
        }
    }
}

// ********************** Select to/Show matching brace features **********************
// ************************ Plus fancy escape completion stuff ************************

+ (void)textViewDidChangeSelection:(NSNotification *)notification {
    NSTextView *textView = [notification object];
    NSRange selRange = [textView selectedRange];
    TEPreferencesController *prefs = [TEPreferencesController sharedPreferencesController];

    if ([prefs fancyEscapeCompletion]) {
        if ([notification object] == TE_keyTextView) {
            [self clearFoundEscapeCompletions];
        }
    }
    if ([prefs selectToMatchingBrace]) {
        // The NSTextViewDidChangeSelectionNotification is sent before the selection granularity is set.  Therefore we can't tell a double-click by examining the granularity.  Fortunately there's another way.  The mouse-up event that ended the selection is still the current event for the app.  We'll check that instead.  Perhaps, in an ideal world, after checking the length we'd do this instead: ([textView selectionGranularity] == NSSelectByWord).
        if ((selRange.length == 1) && ([[NSApp currentEvent] type] == NSLeftMouseUp) && ([[NSApp currentEvent] clickCount] == 2)) {
            NSRange matchRange = TE_findMatchingBraceForRangeInString(selRange, [textView string]);

            if (matchRange.location != NSNotFound) {
                selRange = NSUnionRange(selRange, matchRange);
                [textView setSelectedRange:selRange];
                [textView scrollRangeToVisible:matchRange];
            }
        }
    }
    if ([prefs showMatchingBrace]) {
        NSRange oldSelRangePtr;
        
        [[[notification userInfo] objectForKey:@"NSOldSelectedCharacterRange"] getValue:&oldSelRangePtr];

        // This test will catch typing sel changes, also it will catch right arrow sel changes, which I guess we can live with.  MF:??? Maybe we should catch left arrow changes too for consistency...
        if ((selRange.length == 0) && (selRange.location > 0) && ([[NSApp currentEvent] type] == NSKeyDown) && (oldSelRangePtr.location == selRange.location - 1)) {
            NSRange origRange = NSMakeRange(selRange.location - 1, 1);
            unichar origChar = [[textView string] characterAtIndex:origRange.location];

            if (TE_isClosingBrace(origChar)) {
                NSRange matchRange = TE_findMatchingBraceForRangeInString(origRange, [textView string]);
                if (matchRange.location != NSNotFound) {
                    NSLayoutManager *layout = [textView layoutManager];

                    // Force layout
                    (void)[layout textContainerForGlyphAtIndex:[layout glyphRangeForCharacterRange:matchRange actualCharacterRange:NULL].location effectiveRange:NULL];
                    // Set selection
                    [textView setSelectedRange:matchRange];
                    // Force display
                    [textView displayIfNeeded];
                    // Force flush
                    [[textView window] flushWindow];
                    // Ping
                    PSWait();
                    // Pause
                    [NSThread sleepUntilDate:[NSDate dateWithTimeIntervalSinceNow:0.125]];

                    // Reset the selection
                    [textView setSelectedRange:selRange];
                }
            }
        }
    }
}

@end
