/*
    File:       QResourceFile.m

    Contains:   A class that encapsulates a Mac OS resources files and gives access
                to individual resources.

    Written by: Quinn "The Eskimo!"

    Created:    Mon 19-May-1997

    Copyright:  (c)1997 by Apple Computer, Inc., all rights reserved.

    Change History (most recent first):

    You may incorporate this sample code into your applications without
    restriction, though the sample code has been provided "AS IS" and the
    responsibility for its operation is 100% yours.  However, what you are
    not permitted to do is to redistribute the source as "DSC Sample Code"
    after having made changes. If you're going to re-distribute the source,
    we require that you make it clear in the source that the code was
    descended from Apple Sample Code, but that you've made changes.
*/

#import "QResourceFile.h"

#import "QResourceTypesEnumerator.h"
#import "QResourceEnumerator.h"

// Code Notes
//
// There are a *lot* of hard coded numbers in this sample.  The reson for this
// is simple: the format of a resource file is set in stone, never to change.
// So there's no point declaring constants for each of these magic numbers.
// Remember, the point of this class is to wrap up the ugliness so you don't
// need to worry about it.
//
// Also, on a Mac it's traditional to declare a structure that maps directly
// on to the various resource data structures, eg the resource file header and
// the resource map header.  I have not done that here because:
//
// a) it requires 68K alignment, and although the compiler is being modified
//    to support that, it's still bleeding edge stuff.
// b) the structures would still require byte swapping on little endian machines.
//
// Instead, I access all fields in data structures by their offset and use
// the QMacDataAccess additions to NSData to ensure that byte ordering.  It isn't
// particularly fast, but it works, and that's what I really care about.

@implementation QResourceFile

///////////////////////////////////////////////////////////////////////////
// Boring internal routines.

- (SInt32)longAtOffset:(SInt32)offset
    // Returns the long at the given offset (in bytes) in the resource
    // file.  Exported for use by the enumerator objects, but also used
    // extensively internally.
{
    return [resourceData MacSInt32AtOffset:offset];
}

- (SInt16)shortAtOffset:(SInt32)offset
    // Returns the short at the given offset (in bytes) in the resource
    // file.  Exported for use by the enumerator objects, but also used
    // extensively internally.
{
    return [resourceData MacSInt16AtOffset:offset];
}

- (void)checkBoundsOf:(SInt32)value lower:(SInt32)lowerBound upper:(SInt32)upperBound
    // Checks that value is within lowerBound and upperBound inclusively.
{
    NSAssert( value >= lowerBound && value <= upperBound, @"Bounds check failed");
}

- (long)resourceTypeListOffset
    // Returns the offset of the resource type list in the resource file.
{
    return (resourceTypeListOffset);
}

///////////////////////////////////////////////////////////////////////////
// Critical internal routines.

- (BOOL)findForType:(QResourceType)aResType numResources:(long *)numResourcesOfThisType
            refList:(long *) referenceListOffset
    // For the specified resource type, find the number of resources of that type and
    // the resource reference list for that type.  Returns true if it finds an entry
    // for that type, false otherwise.
{
    BOOL result = NO;
    long numResourceTypes;
    long typeIndex;
    QResourceType thisResType;
    
    numResourceTypes = [self shortAtOffset:[self resourceTypeListOffset] ] + 1;

    // Search through the resource type list looking for the entry that matches aResType.
    
    typeIndex = 0;
    while ( typeIndex < numResourceTypes && ! result ) {

        thisResType = [self longAtOffset:resourceTypeListOffset + 2 + 8 * typeIndex];

        if (thisResType == aResType) {

            // Found it, let's return the the info from that entry.
            
            *numResourcesOfThisType = [self shortAtOffset:resourceTypeListOffset + 2 + 8 * typeIndex + 4] + 1;
            *referenceListOffset = [self shortAtOffset:resourceTypeListOffset + 2 + 8 * typeIndex + 6] +
                                        resourceTypeListOffset;
            result = YES;
        } else {
            typeIndex += 1;
        }
    }
    
    return (result);
}

- (QResourceObject *)resourceGivenType:(QResourceType)resType andReferenceOffset:(long)offset
    // Given an offset to the resource reference in a resource file, create a
    // QResourceObject suitable for returning to our client.  The format of a resource
    // reference is:
    //
    // Offset Type   Desc
    //  0     UInt16 Resource ID
    //  2     UInt16 Offset from beginning of resource name list to start of
    //               resource name, or -1 if no name
    //  4     UInt32 Resource attributes (top byte) and offset from beginning of
    //               resource data to length of data for this resource (bottom three bytes)
    //  8     UInt32 Reserved for resource handle
    //
    // The resource data itself is stored as:
    //
    // Offset Type   Desc
    //  0     UInt32 Length of resource data
    //  4     ... resource data
{
    QResourceObject *result = nil;
    short resID;
    long resNameOffset;
    long resDataOffset;
    NSData *resData;
    long resLength;
    NSString *resName;
    long resAttributes;

    // Get the resource ID.
    
    resID = [self shortAtOffset:offset];

    // Get the resource name and create an NString to hold it.
    
    resNameOffset = [self shortAtOffset:offset + 2];
    if (resNameOffset == -1) {
        // -1 implies that there is no name for this resource.
        resName = nil;
    } else {
        resNameOffset = resourceNameListOffset + resNameOffset;
        [self checkBoundsOf:resNameOffset lower:resourceNameListOffset
                              upper:resourceFileLength ];
        resName = [resourceData MacPStringAtOffset:resNameOffset];
    }

    // Find the offset to the resource data, and the attributes, which are both
    // packed into the same long.
    
    resDataOffset = [self longAtOffset:offset + 4];

    resAttributes = (resDataOffset >> 24) & 0x0ff;
    
    resDataOffset &= 0x00ffffff;
    resDataOffset = resourceDataOffset + resDataOffset;

    [self checkBoundsOf:resDataOffset lower:resourceDataOffset upper:resourceDataOffset + resourceDataLength ];

    // Get the resource length.
    
    resLength = [self longAtOffset:resDataOffset];
    NSAssert(resLength >= 0, @"Bad resource length");

    // Create a NSData to hold the resource data.
    
    resData = [resourceData subdataWithRange:NSMakeRange(resDataOffset + 4, resLength)];

    // Finally, create a resource object to hold the type, ID, name and data.
    
    if (resData != nil) {
        result = [ [ [QResourceObject alloc] initWithResourceType:resType resourceID:resID
                            resourceName:resName resourceData:resData resourceAttributes:resAttributes] autorelease ];
    }

    return (result);
}

///////////////////////////////////////////////////////////////////////////
// External routines.

- (id)initWithContentsOfFile:(NSString*)path
    // See comments in interface part.
{
    self = [super init];

    if (self != nil) {

        NSAssert( path != nil, @"No path to initialise QResourceFile");
        
        // Create the NSData object used to hold the entire resource fork.
        
        resourceData = [ [NSData alloc] initWithContentsOfMappedFile:path ];
        if (resourceData == nil) {
            [self release];
            self = nil;
        } else {

            // Extract some basic information from the resource file header.
            
            resourceFileLength = [resourceData length];
            resourceDataOffset = [self longAtOffset:0];
            resourceMapOffset  = [self longAtOffset:4];
            resourceDataLength = [self longAtOffset:8];
            resourceMapLength  = [self longAtOffset:12];

            // Sanity check that information.
            
            [self checkBoundsOf:resourceFileLength lower:256 upper:16 * 1024 * 1024];
            [self checkBoundsOf:resourceDataOffset lower:256 upper:resourceFileLength];
            [self checkBoundsOf:resourceMapOffset lower:256 upper:resourceFileLength];
            [self checkBoundsOf:resourceDataLength lower:0 upper:resourceMapOffset-resourceDataOffset];
            [self checkBoundsOf:resourceMapLength lower:0 upper:resourceFileLength - resourceMapOffset];

            // Find the attributes, and the offset to the resource type and name lists.
            
            attributes = [self shortAtOffset:resourceMapOffset + 22 ] & 0x0ffff;

            resourceTypeListOffset = resourceMapOffset + [self shortAtOffset:resourceMapOffset + 24 ];
            resourceNameListOffset = resourceMapOffset + [self shortAtOffset:resourceMapOffset + 26 ];

            // Sanity check those calculations.
            
            [self checkBoundsOf:resourceTypeListOffset lower:resourceMapOffset
                          upper:resourceMapOffset + resourceMapLength];
            [self checkBoundsOf:resourceNameListOffset lower:resourceMapOffset
                          upper:resourceMapOffset + resourceMapLength];
        }
    }
    
    return (self);
}

- (id)init
    // Call through to the designated initialiser.
    // In the non-subclassed case, we violate the pre-condition of the
    // designated initialiser by passing in a nil path pointer.
    // We do this because a) it's the right thing to do (see the book
    // "Object-Oriented Programming and the Objective-C Language" for
    // the reasoning), and b) a subclasser may want to subclass us in a
    // weird way.
{
    return ( [self initWithContentsOfFile:nil] );
}

- (QResourceObject *)resourceOfType:(QResourceType)resType withID:(long)aResID
    // See comments in interface part.
{
    QResourceObject *result = nil;
    long numResourcesOfThisType;
    long referenceListOffset;
    long resIndex;
    long resID;

    // Find the start of the reference list for resource of type resType.
    
    if ( [self findForType:resType numResources:&numResourcesOfThisType refList:&referenceListOffset] ) {

        // Loop through each resource reference, looking for a match with aResID.
        
        for (resIndex = 0; resIndex < numResourcesOfThisType; resIndex++) {
            resID = [self shortAtOffset:referenceListOffset + 12 * resIndex];
            if (resID == aResID) {

                // We have a match, create a resource object to return to the client.
                
                result = [self resourceGivenType:resType andReferenceOffset:referenceListOffset + 12 * resIndex];
                
                break;
            }
        }
    }

    return (result);
}

- (QResourceTypesEnumerator*)enumerateTypes
    // See comments in interface part.
{
    return [ [ [QResourceTypesEnumerator alloc] initWithResourceFile:self ] autorelease ];
}

- (QResourceEnumerator*)enumerateResourcesOfType:(QResourceType)resType
    // See comments in interface part.
{
    return [ [ [QResourceEnumerator alloc] initWithResourceFile:self andType:resType ] autorelease ];
}

- (long)attributes
    // See comments in interface part.
{
    return (attributes);
}

- (void)dealloc
    // See comments in interface part.
{
    if (resourceData != nil) {
        [resourceData release];
    }
    [super dealloc];
}


#if 0

    // Obsolete stuff.

    - (long)resourceDataLength
    {
        return (resourceDataLength);
    }

    - (long)resourceDataOffset
    {
        return (resourceDataOffset);
    }

    - (long)resourceNameListOffset
    {
        return (resourceNameListOffset);
    }

    - (long)resourceFileLength
    {
        return (resourceFileLength);
    }

    - (NSData *)resourceData
    {
        return (resourceData);
    }

#endif

@end
