/*$Id: Shikenjo.m,v 1.2 1998/12/20 16:54:50 marco Exp $*/

// Copyright (c) 1997, 1998, 1999, Sen:te Ltd.  All rights reserved.
//
// Use of this source code is governed by the license in OpenSourceLicense.html
// found in this distribution and at http://www.sente.ch/software/ ,  where the
// original modified of this source code can also be found.
// This notice may not be removed from this file.

#import "Shikenjo.h"
#import <SenTestingKit/SenTestingKit.h>
#import <SenFoundation/SenFoundation.h>

static NSString *SelfTestDefault = @"SelfTest";


static NSString *notificationIdentifier = nil;
static NSString *LoadableExtensions = @"LoadableExtensions";
static NSString *ExecutableExtensions = @"ExecutableExtensions";
static NSString *TestedUnitPath = @"TestedUnitPath";

static NSString *WindowTitle = @"Unit Test";

static NSString *TestRunnerName = @"otest";
#ifdef WIN32
static NSString *TestRunnerExtension = @"exe";
#else
static NSString *TestRunnerExtension = nil;
#endif

@interface NSNotification (Shikenjo)
- (id) unarchivedRun;
@end

@implementation NSNotification (Shikenjo)
- (id) unarchivedRun
{
    return [NSUnarchiver unarchiveObjectWithData:[[self userInfo] objectForKey:@"object"]];
}
@end


@implementation Shikenjo
+ (void) initialize
{
    [super initialize];
    [NSUserDefaults registerDefaultsFromBundle:[NSBundle mainBundle]];
    notificationIdentifier = [[[NSProcessInfo processInfo] globallyUniqueString] copy];
}


+ (NSSet *) executableExtensions
{
    return [NSSet setWithArray:[[NSUserDefaults standardUserDefaults] objectForKey:ExecutableExtensions]];
}


+ (NSSet *) loadableExtensions
{
    return [NSSet setWithArray:[[NSUserDefaults standardUserDefaults] objectForKey:LoadableExtensions]];
}


+ (NSSet *) testableExtensions
{
    NSMutableSet *extensions = [NSMutableSet set];
    [extensions unionSet:[self executableExtensions]];
    [extensions unionSet:[self loadableExtensions]];
    return extensions;
}


- init
{
    self = [super init];
    failures = [[NSMutableArray alloc] init];
    testedUnitPath = [[[NSUserDefaults standardUserDefaults] stringForKey:TestedUnitPath] retain];
    return self;
}


- (void) setupWindow
{
    if (testedUnitPath != nil) {
        [window setTitle:[NSString stringWithFormat:@"%@ - '%@'", WindowTitle, [testedUnitPath lastPathComponent]]];
    }
    else {
        [window setTitle:WindowTitle];
    }
}


- (void) setupFailureTable:(NSTableView *) aTable
{
    [aTable setDataSource:self];
    [aTable setDelegate:self];
}


- (void) resetProgressView
{
    [progressView setDoubleValue:0.0];
    [progressView setMaxValue:1.0];
}


- (void) resetFailureTableView
{
    [failures removeAllObjects];
    [failureTableView reloadData];
}


- (void) resetFields
{
    [casesField setIntValue:0];
    [failuresField setIntValue:0];
    [errorsField setIntValue:0];
    [messageField setStringValue:@""];
    [self resetProgressView];
    [self resetFailureTableView];
}


- (void) registerSelector:(SEL) aSelector withDistributedNotificationName:(NSString *) aNotificationName
{
    [[NSDistributedNotificationCenter defaultCenter] addObserver:self
                                                        selector:aSelector
                                                            name:aNotificationName
                                                          object:notificationIdentifier
                                              suspensionBehavior:NSNotificationSuspensionBehaviorDeliverImmediately];
}


- (void) registerToTestNotifications
{
     [self registerSelector:@selector(testSuiteDidStart:) withDistributedNotificationName:SenTestSuiteDidStartNotification];
     [self registerSelector:@selector(testSuiteDidStop:) withDistributedNotificationName:SenTestSuiteDidStopNotification];

     [self registerSelector:@selector(testCaseDidStart:) withDistributedNotificationName:SenTestCaseDidStartNotification];
     [self registerSelector:@selector(testCaseDidStop:) withDistributedNotificationName:SenTestCaseDidStopNotification];
     [self registerSelector:@selector(testCaseDidFail:) withDistributedNotificationName:SenTestCaseDidFailNotification];
}


- (void) unregisterFromTestNotifications
{
    [[NSDistributedNotificationCenter defaultCenter] removeObserver:self];
}


- (void) setTestingTask:(NSTask *) aTask
{
    if (testingTask != nil) {
        [[NSNotificationCenter defaultCenter] removeObserver:self
                                                        name:NSTaskDidTerminateNotification
                                                      object:testingTask];
    }
    RETAIN (testingTask, aTask);
    if (testingTask != nil) {
        [[NSNotificationCenter defaultCenter] addObserver:self
                                                 selector:@selector(taskDidTerminate:)
                                                     name:NSTaskDidTerminateNotification
                                                   object:testingTask];
    }
}


- (void) setTestedUnitPath:(NSString *) aPath
{
    RETAIN (testedUnitPath, aPath);
    [[NSUserDefaults standardUserDefaults] setObject:testedUnitPath forKey:TestedUnitPath];
    [self setupWindow];
}


- (BOOL) isExecutablePath:(NSString *) path
{
    return [[[self class] executableExtensions] containsObject:[path pathExtension]];
}


- (NSString *) testedUnitPath
{
    return testedUnitPath;
}


- (void) launchTestingTask
{
    NSMutableArray *arguments = [NSMutableArray array];
    NSTask *task = [[[NSTask alloc] init] autorelease];

    [arguments setArgumentDefaultValue:@"YES" forKey:SelfTestDefault];
    [arguments setArgumentDefaultValue:@"SenTestDistributedNotifier" forKey:@"SenTestObserverClass"];
    [arguments setArgumentDefaultValue:notificationIdentifier forKey:@"SenTestNotificationIdentifier"];

    if ([self isExecutablePath:testedUnitPath]) {
        [task setLaunchPath: [[NSBundle bundleWithPath:testedUnitPath] executablePath]];
    }
    
    else  {
        [task setLaunchPath:[[NSBundle mainBundle] pathForResource:TestRunnerName ofType:TestRunnerExtension]];
        [arguments addObject:[testedUnitPath asUnixPath]];
    }
    
    [task setArguments:arguments];
    [self setTestingTask:task];
    [self registerToTestNotifications];
    [testingTask launch];
}


- (void) terminateTestingTask
{
    if ([testingTask isRunning]) {
        [testingTask terminate];
    }
    [self unregisterFromTestNotifications];
}


- (void) start
{
    startedSuiteCount = 0;
    [self resetFields];
    [self launchTestingTask];
    [runButton setImage:[NSImage imageNamed:@"Stop"]];
    [runButton setAction:@selector (stop:)];
}


- (void) stop
{
    [self terminateTestingTask];
    [runButton setImage:[NSImage imageNamed:@"Run"]];
    [runButton setAction:@selector (run:)];
    [self resetProgressView];
}


- (IBAction) run:(id)sender
{
    [self start];
}


- (IBAction) stop:(id)sender
{
    [self stop];
    [messageField setStringValue:@"interrupted."];
}


- (IBAction) chooseUnit:(id)sender
{
    int result;
    NSArray *fileTypes = [[[self class] testableExtensions] allObjects];
    NSOpenPanel *panel = [NSOpenPanel openPanel];

    NSString *openDirectory = (testedUnitPath != nil) ? [testedUnitPath stringByDeletingLastPathComponent] : NSHomeDirectory();
    NSString *openFile = (testedUnitPath != nil) ? [testedUnitPath lastPathComponent] : nil;

    [panel setAllowsMultipleSelection:NO];
    result = [panel runModalForDirectory:openDirectory file:openFile types:fileTypes];
    if (result == NSOKButton) {
        [self setTestedUnitPath:[[panel filenames] firstObject]];
    }
}


- (void) applicationDidFinishLaunching:(NSNotification *) aNotification
{
    [self stop];
    [self setupFailureTable:failureTableView];
    [self setupWindow];
}


- (BOOL) application:(NSApplication *)theApplication openFile:(NSString *)filename
{
    if ([[[self class] testableExtensions] containsObject:[filename pathExtension]]){
        [self setTestedUnitPath:filename];
        [self performSelector:@selector(run:) withObject:nil afterDelay:0.0];
        return YES;
    }
    return NO;
}


- (void) testSuiteDidStart:(NSNotification *) notification
{
    SenTestRun *testRun = [notification unarchivedRun];
    if (startedSuiteCount == 0) {
        RETAIN (startTime, [testRun startDate]);
        [progressView setMaxValue:(double) [[testRun test] testCaseCount]];
    }
    startedSuiteCount++;
}


- (void) testSuiteDidStop:(NSNotification *) notification
{
    startedSuiteCount--;
    if (startedSuiteCount == 0) {
        SenTestRun *testRun = [notification unarchivedRun];
        RETAIN (stopTime, [testRun stopDate]);
        [self stop];
        [messageField setStringValue:[NSString stringWithFormat:@"started %@, terminated %@.",
            [startTime dateWithCalendarFormat:@"%H:%M:%S" timeZone:nil],
            [stopTime dateWithCalendarFormat:@"%H:%M:%S" timeZone:nil]]];
    }
}


- (void) testCaseDidStart:(NSNotification *) notification
{
    SenTestRun *testRun = [notification unarchivedRun];
    [messageField setStringValue:[[testRun test] description]];
}


- (void) testCaseDidStop:(NSNotification *) notification
{
    [casesField setIntValue:[casesField intValue] + 1];
    [progressView setDoubleValue:[casesField doubleValue]];
}


- (void) taskDidTerminate:(NSNotification *) notification
{
}

- (void) testCaseDidFail:(NSNotification *) notification
{
    SenTestRun *testRun = [notification unarchivedRun];
    NSException *exception = [notification exception];
    id test = [testRun test];
    NSMutableDictionary *exceptionDescription = [NSMutableDictionary dictionary];

    if ([exception isOfType:SenTestFailureException]) {
        [failuresField setIntValue:[failuresField intValue] + 1];
        [exceptionDescription setObject:NSLocalizedString (SenTestFailureException, @"") forKey:@"Type"];
    }
    else {
        [errorsField setIntValue:[errorsField intValue] + 1];
        [exceptionDescription setObject:NSLocalizedString (@"OtherException", @"") forKey:@"Type"];
    }

    [exceptionDescription addEntriesFromDictionary:[exception userInfo]];
    [exceptionDescription setObject:[exception reason] forKey:@"Reason"];
    [exceptionDescription setObject:[test description] forKey:@"Case"];
    [failures addObject:exceptionDescription];
    [failureTableView reloadData];
}


- (int) numberOfRowsInTableView:(NSTableView *)aTableView
{
    return [failures count];
}


- (id) tableView:(NSTableView *)aTableView objectValueForTableColumn:(NSTableColumn *)aTableColumn row:(int)rowIndex
{
    return [[failures objectAtIndex:rowIndex] objectForKey:[aTableColumn identifier]];
}


- (void) tableView:(NSTableView *)tv willDisplayCell:(id)cell forTableColumn:(NSTableColumn *)tableColumn row:(int)row
{
    if ([[tableColumn identifier] isEqualToString:@"Icon"]){
        NSString *exceptionName = [[failures objectAtIndex:row] objectForKey:@"Type"];
        [cell setBackgroundColor:[exceptionName isEqualToString:NSLocalizedString (SenTestFailureException, @"")] ? [NSColor redColor] : [NSColor blackColor]];
        [cell setDrawsBackground:YES];
    }
}
@end
