/*+++
 *  title:      MailProxy.m
 *  abstract:   Implementation of MailProxy class for mailapp-utilities.
 *  author:     Tom Hageman <tom@basil.icce.rug.nl>
 *  created:    February 1998
 *  modified:   October 1998 -- complete overhaul into class cluster.
 *  copyleft:
 *
 *	Copyright (C) 1998  Tom R. Hageman, but otherwise perfect freeware.
 *
 *  description:
 *      (see corresponding *.h file)
 *---*/

static const char * const RCSid = ((void)&RCSid,
	"@(#)MailProxy.m,v 1.20 1999/01/18 21:38:47 tom Exp");
#define RCS_MailProxy_ID

#import "MailProxy.h"
#import "mailutil.h"	// for MAIL_APP
#if NEXTSTEP
# import <libc.h>
# import <appkit/Application.h>	 // for -launchApplication:
#else
# import <AppKit/NSWorkspace.h>
# import <AppKit/NSAttributedString.h>
#endif
#if NEXTSTEP || OPENSTEP
# import "MailSpeaker.h"	// for NXPortFromName() etc. 
# import <mach/mach.h>
#endif
#import <objc/objc.h>
#import	<objc/objc-runtime.h>
#import <stdlib.h>
#import <string.h>


/* This defines the mailer-name -> proxy class mapping, used by +classForMailer:
   Mailer name = "*" means match all. */

#ifndef MAILER_TO_PROXY_CLASS_MAPPING
# if NEXTSTEP
#  define MAILER_TO_PROXY_CLASS_MAPPING \
      { "*",		"MailAppProxy"		   }
# elif OPENSTEP
#  define MAILER_TO_PROXY_CLASS_MAPPING \
      { "PopOver",	"MailPopOverProxy"	   }, \
      { "MailViewer",	"MailViewerProxy"	   }, \
      { "*",		"MailAppProxy"		   }
# elif RHAPSODY
#  define MAILER_TO_PROXY_CLASS_MAPPING \
      { "PopOver",	"MailPopOverProxy"	   }, \
      { "*",		"MailViewerProxy"	   }, \
      { "*",		"MailMessageDeliveryProxy" }, \
/*    { "Mail",		"MailAppProxy"		   } */
# endif
#endif


/* Max. Nr. of seconds to retry to establish connection after launch. */
#define MAX_RETRIES  5


#if NEXTSTEP
# define ABSTRACT_METHOD  [self subclassResponsibility:_cmd]
#else
# define ABSTRACT_METHOD  NSAssert(0, @"Abstract method -- should be overridden by subclass!")
#endif

#define _RELEASE_IVARS \
   free(hostname); \
   free(mailer);


// System-specific support functions.
static const char *localhost(void) __ATTRIBUTE_CONST__;
static BOOL islocalhost(const char *aHostname);
static void sleepFor(int seconds);
static BOOL workspaceLaunchMailer(MailProxy *self);


@implementation MailProxy

// Implementation feature tests.

+ (BOOL)canEditMessage		{ return NO; }
+ (BOOL)canDeliverMessage	{ return NO; }
+ (BOOL)canIncorporateNewMail	{ return NO; }
+ (BOOL)canSendBcc		{ return NO; }
+ (BOOL)canSendArbitraryHeaders	{ return NO; }


+ (const char *)defaultMailer
{
   const char *result = NXGetDefaultValue("GLOBAL", "Mailer");

   if (!result || !*result) result = MAIL_APP;
   return result;
}


- initForMailer:(const char *)aMailer
   host:(const char *)aHostName
   forceLaunch:(BOOL)forceLaunch
   directDelivery:(BOOL)direct
{
   if ((self = [super init]))
   {
      if (!aHostName || !*aHostName || strcmp(aHostName, "localhost") == 0)
      {
	 aHostName = localhost();
	 isLocalHost = YES;
      }
      else
      {
	 isLocalHost = islocalhost(aHostName);
      }
      hostname = NXCopyStringBufferFromZone(aHostName, [self zone]);

      if (!aMailer || !*aMailer) aMailer = [[self class] defaultMailer];
      mailer = NXCopyStringBufferFromZone(aMailer, [self zone]);

      deliver = direct;

      isConnected = [self connect:forceLaunch];
   }
   return self;
}

- init
{
   return [self initForMailer:NULL host:NULL forceLaunch:YES directDelivery:NO];
}

#if NEXTSTEP

- free
{
   _RELEASE_IVARS
   return [super free];
}

- (void)dealloc { [self free]; }

- (void)release { [self free]; }

#else // ! NEXTSTEP

- (void)dealloc
{
   _RELEASE_IVARS
   [super dealloc];
}

- free
{
   [self release];
   return nil;
}

#endif // NEXTSTEP


// Access methods.

- (const char *)host	{ return hostname; }
- (const char *)mailer	{ return mailer; }
- (BOOL)directDelivery	{ return deliver; }
- (BOOL)isConnected	{ return isConnected; }
- (BOOL)isLocalHost	{ return isLocalHost; }


// Other methods.

- (BOOL)connect:(BOOL)forceLaunch
{
   if (!isConnected)
   {
      isConnected = [self connectToMailer];

      if (!isConnected && forceLaunch)
      {
	 if ([self launchMailer])
	 {
	    int retries = MAX_RETRIES;

	    while (!(isConnected = [self connectToMailer]) && --retries >= 0)
	    {
	       /* ``sleep'' and retry a few times if connection is
		  not established immediately.
		  -- necessary at least for Rhapsody's MailViewer. */
	       sleepFor(1);
	    }
	    // DEBUG:
            if (retries != MAX_RETRIES) fprintf(stderr, "-[%s %s]: retries = %d\n", isa->name, sel_getName(_cmd), MAX_RETRIES - retries);
	 }
	 if (!isConnected)
	 {
	    isConnected = [self retryConnectToMailer];
	 }
      }
   }
   return isConnected;
}

- (BOOL)connectToMailer
{
   ABSTRACT_METHOD;
   return NO;
}

- (BOOL)retryConnectToMailer
{
   return NO;
}

- (BOOL)sendMailTo:(const char *)to
   subject:(const char *)subject
   body:(const char *)body
   cc:(const char *)cc
   bcc:(const char *)bcc
{
#if NEXTSTEP
   ABSTRACT_METHOD;
   return NO;
#else
   id bodyString = nil;

   if (body)
   {
      // Detect RTF body, and create attributed string from it.
      static const char rtfPrefix[] = "{\\rtf0";

      if (strncmp(rtfPrefix, body, sizeof(rtfPrefix)-1) == 0)
      {
	 NSData *bodyData = [[NSData alloc] initWithBytes:body length:strlen(body)];

	 bodyString = [[[NSAttributedString alloc] initWithRTF:bodyData
			   documentAttributes:NULL] autorelease];
	 [bodyData release];
      }
      if (bodyString == nil)
      {
	 bodyString = [NSString stringWithCString:body];
      }
   }
   return [self sendMailMessageTo:(to ? [NSString stringWithCString:to] : nil)
		subject:(subject ? [NSString stringWithCString:subject] : nil)
		body:bodyString
		cc:(cc ? [NSString stringWithCString:cc] : nil)
		bcc:(bcc ? [NSString stringWithCString:bcc] : nil)];
#endif
}

- (BOOL)incorporateNewMail
{
   return NO;
}

- (BOOL)launchMailer
{
#if NEXTSTEP || OPENSTEP
   port_t mailPort = NXPortFromName([self mailer], [self host]);

   if (mailPort != PORT_NULL)
   {
      port_deallocate(task_self(), mailPort);
      return YES;
   }
#endif

   // Try to launch Mailer by using Workspace manager, if we are local.
   if (!isLocalHost) return NO;

   return workspaceLaunchMailer(self);
}

- (BOOL)makeMailerActive
{
   if (!isLocalHost) return NO; /* Workspace can't activate remote app, sorry... */

#if NEXTSTEP || OPENSTEP
   // Avoid launch if app is not running.
   {
      port_t mailPort = NXPortNameLookup([self mailer], [self host]);

      if (mailPort == PORT_NULL) return NO;

      port_deallocate(task_self(), mailPort);
   }
#endif
   // This assumes that Workspace launch also activates.
   return workspaceLaunchMailer(self);
}

@end // MailProxy


@implementation	MailProxy (Factory)

+ (Class)classForMailer:(const char *)aMailer
   directDelivery:(BOOL)direct
   incorporation:(BOOL)incorporate
{
   static const struct {
      const char *mailer;
      const char *className;
   }
   table[] = {
      MAILER_TO_PROXY_CLASS_MAPPING
   };
   const int tableCount = sizeof(table) / sizeof(table[0]);
   int i;

   if (!aMailer || !*aMailer) aMailer = [self defaultMailer];

   for (i = 0;  i < tableCount; i++)
   {
      if (strcmp(table[i].mailer, "*") == 0 ||
	  strcmp(table[i].mailer, aMailer) == 0)
      {
	 Class class = objc_lookUpClass(table[i].className);

	 if (class == Nil) continue;   // Class not linked in.

	 if (incorporate)
	 {
	    if ([class canIncorporateNewMail]) return class;
	 }
	 else if (direct)
	 {
	    if ([class canDeliverMessage]) return class;
	 }
	 else
	 {
	    if ([class canEditMessage]) return class;
	 }
      }
   }
   return Nil;
}

+ newForMailer:(const char *)aMailer
   host:(const char *)aHostName
   forceLaunch:(BOOL)force
   directDelivery:(BOOL)direct
   incorporation:(BOOL)incorporate
{
   Class class;
   id instance;

   if (!aMailer || !*aMailer) aMailer = [self defaultMailer];

   class = [self classForMailer:aMailer directDelivery:direct incorporation:incorporate];

   if (class == Nil) return nil;
   if (incorporate && ![class canIncorporateNewMail]) return nil;
   if (!direct && ![class canEditMessage]) return nil;

   instance = [[class alloc] initForMailer:aMailer host:aHostName
		     forceLaunch:force directDelivery:direct];
   return instance;
}

+ new
{
   return [self newForMailer:NULL host:NULL forceLaunch:YES directDelivery:NO incorporation:NO];
}

@end // MailProxy (Factory)


#if !NEXTSTEP

/* This category defines a more OPENSTEP-conformant interface.
   These are cover methods for the underlying NEXTSTEP-compatible
   base methods. */

@implementation MailProxy (OpenStep)

+ classWithMailer:(NSString *)aMailer
   directDelivery:(BOOL)direct
   incorporation:(BOOL)incorporate
{
   return [self classForMailer:[aMailer cString]
		directDelivery:direct
		incorporation:incorporate];
}

+ mailProxyWithMailer:(NSString *)aMailer
   host:(NSString *)aHostName
   forceLaunch:(BOOL)force
   directDelivery:(BOOL)direct
   incorporation:(BOOL)incorporate
{
   return [[self newForMailer:[aMailer cString]
		 host:[aHostName cString]
		 forceLaunch:force
		 directDelivery:direct
		 incorporation:incorporate] autorelease];
}

+ mailProxy
{
   return [self mailProxyWithMailer:nil host:nil forceLaunch:YES directDelivery:NO incorporation:NO];
}

- initWithMailer:(NSString *)aMailer
   host:(NSString *)aHostName
   forceLaunch:(BOOL)force
   directDelivery:(BOOL)direct
{
   return [self initForMailer:[aMailer cString]
		host:[aHostName cString]
		forceLaunch:force
		directDelivery:direct];
}

- (NSString *)hostName	 { return [NSString stringWithCString:[self host]]; }
- (NSString *)mailerName { return [NSString stringWithCString:[self mailer]]; }


- (BOOL)sendMailMessageTo:(NSString *)to
   subject:(NSString *)subject
   body:(id)body
   cc:(NSString *)cc
   bcc:(NSString *)bcc
{
   if ([[self class] canSendArbitraryHeaders])
   {
      NSMutableDictionary *headers = [NSMutableDictionary dictionary];

      if (to)      [headers setObject:to      forKey:MAILPROXY_TO];
      if (subject) [headers setObject:subject forKey:MAILPROXY_SUBJECT];
      if (cc)	   [headers setObject:cc      forKey:MAILPROXY_CC];
      if (bcc)	   [headers setObject:bcc     forKey:MAILPROXY_BCC];

      return [self sendMailBody:body headers:headers];
   }
   ABSTRACT_METHOD;
   return NO;
}

- (BOOL)sendMailBody:(id)body headers:(NSDictionary *)headers
{
   if ([[self class] canSendArbitraryHeaders])
   {
      ABSTRACT_METHOD;
      return NO;
   }
   return [self sendMailMessageTo:[headers objectForKey:MAILPROXY_TO]
		subject:[headers objectForKey:MAILPROXY_SUBJECT]
		body:body
		cc:[headers objectForKey:MAILPROXY_CC]
		bcc:[headers objectForKey:MAILPROXY_BCC]];
}


- (id)replacementObjectForBody:(id)body
{
   if (body)
   {
      if (![body isKindOfClass:[NSString class]])
      {
	 // handle NSAttributedString, maybe others.
	 if ([body respondsToSelector:@selector(string)])
	 {
	    body = [body string];
	 }
	 else
	 {
	    // The best we can do.  XXX should we raise instead?
	    body = [body description];
	 }
      }

   }
   return body;
}

@end // MailProxy (OpenStep)

#endif // !NEXTSTEP


// System-specific support functions.

#if NEXTSTEP

# include <sys/param.h>
# include <netdb.h>

static const char *localhost()
{
   static const char *_localhost;

   if (!_localhost)
   {
      char buf[MAXHOSTNAMELEN];

      if (gethostname(buf, sizeof(buf)) != 0)
      {
	 _localhost = "localhost";
      }
      else
      {
	 _localhost = NXCopyStringBuffer(buf);
      }
   }
   return _localhost;
}

static BOOL islocalhost(const char *aHostName)
{
   if (aHostName && *aHostName &&
       strcmp("localhost", aHostName) != 0 &&
       strcmp(localhost(), aHostName) != 0)
   {
      struct hostent *h = gethostbyname((char *)localhost());

      if (h && strcmp(h->h_name, aHostName) != 0)
      {
	 char **aliases = h->h_aliases;

	 do
	 {
	    if (*aliases == NULL) return NO;
	 }
	 while (strcmp(*aliases++, aHostName) != 0);
      }
   }
   return YES;
}

/* ``Sleep'' for given number of seconds.  Ideally, still handles events
   (if we're an application, that is.) */

static void sleepFor(int seconds)
{
   if (NXApp)
   {
      // XXX TODO (but how???).
   }
   sleep(seconds);   // Um... anyone has a better idea?
}

static BOOL workspaceLaunchMailer(MailProxy *self)
{
   return [[Application workspace] launchApplication:[self mailer]];
}

#else // OPENSTEP / RHAPSODY

static const char *localhost()
{
   return [[[NSHost currentHost] name] cString];
}

static BOOL islocalhost(const char *aHostName)
{
   NSHost *theHost;

   if (!(aHostName && *aHostName) || strcmp("localhost", aHostName) == 0) return YES;

   theHost = [NSHost hostWithName:[NSString stringWithCString:aHostName]];

   return [[NSHost currentHost] isEqualToHost:theHost];
}

/* ``Sleep'' for given number of seconds.  Ideally, still handles events
   (if we're an application, that is.) */

static void sleepFor(int seconds)
{
   NSDate *untilDate = [NSDate dateWithTimeIntervalSinceNow:seconds];

   /* Run the runloop until given date, or until its input sources are
      exhausted, whichever comes first. */
   if (![[NSRunLoop currentRunLoop] runMode:NSDefaultRunLoopMode
	    beforeDate:untilDate])
   {
      /* Didn't run until the desired time; sleep the rest of the way. */
      [NSThread sleepUntilDate:untilDate];
   }
}

static BOOL workspaceLaunchMailer(MailProxy *self)
{
   return [[NSWorkspace sharedWorkspace] launchApplication:[self mailerName]];
}

#endif // OPENSTEP / RHAPSODY
