/*
 * TestController.m -- for demonstrating and exercising the WriteUp API.
 * Written originally by Kris Younger, updated for OPENSTEP/Yellow Box
 * by Greg Anderson (September 1997).
 *
 * This module provides a template and examples of how to connect to 
 * WriteUp through its Distributed Object API. The first three methods 
 * below (_connectToWriteUpOnHost:, applicationDidFinishLaunching:, and 
 * applicationWillTerminate:) are the minimum requirement for connecting 
 * to WriteUp and managing its resources. These methods must be included 
 * in whichever class serves as the [NSApp delegate].
 *
 * The remaining methods (runXxxTest:) demonstrate how to send messages 
 * to WriteUp once a connection has been established. In this sample app, 
 * each option is triggered by one of the buttons on the Test Panel.
 *
 * Copyright (c) 1997 by Anderson Financial Systems Inc. Portions may be 
 * used in non-AFS applications that access the WriteUp API.
 */

#import "TestController.h"
#import <WriteUpAPI/WriteUp_Protocol.h>

@implementation TestController

/*
 * This method tries to establish a Distributed Object connection to 
 * WriteUp. By default, NSConnection uses the following rules for aHost:
 *		nil or "" - connect on local host only
 *		"*" - connect on first responder on local subnet
 *		"hostname" - connect on specified hostname only
 */
- _connectToWriteUpOnHost:(NSString *)aHost
	{
	/* 
	 * This little opening block is here only because the demo allows 
	 * the user to change the hostname on each test. If the desired 
	 * hostname different from the established one, we close the existing
	 * connection and let the following code create a new one. For most
	 * routine uses, you won't need this block.
	 */
	if (writeup && aHost && ![aHost isEqualToString:[writeup hostname]])
		{
		[writeup freeResources];
		writeup = nil;
		}
	if (writeup == nil)
		{
		writeup = [[NSConnection rootProxyForConnectionWithRegisteredName:
			WU_DOSERVERNAME host:aHost] retain];
		if (writeup == nil)
			{
			/* friendly warning, most likely diagnostic problem */
			NSRunAlertPanel(@"WriteUp API Tester",
				@"Can't get connection to WriteUp.\nPerhaps you forgot to enable the API Preference?", nil, nil, nil);
			}
		else
			{
			/* set some reasonable defaults for the connection */
			NSConnection *myConnection = [writeup connectionForProxy];
			[myConnection setReplyTimeout:180];
			[myConnection setRequestTimeout:180];
			[myConnection setDelegate:self];
			/* For demo, show which host actually connected */
			[hostnameField setStringValue:[writeup hostname]];
			}

		}
	return writeup;
	}

/*
 * This method goes in your [NSApp delegate] class, which is registered
 * to receive notifications about majors app-level events. Actually, you 
 * can try to connect to WriteUp whenver you want to, but usually it's 
 * most convenient to do it once, after your application has been launched 
 * and initialized successfully.
 */
- (void)applicationDidFinishLaunching:(NSNotification *)aNotification
	{
	/* 
	 * In the demo app, the user can specify where to run the API.
	 * you might want to hardcode one of the values suggested above.
	 */
	[self _connectToWriteUpOnHost:[hostnameField stringValue]];
	/*
	 * In this application, there's no point continuing if WriteUp is 
	 * unavailable. For applications that do more, the next statement 
	 * is optional. And you can always provide an option to try again.
	 */
	if (writeup == nil)
		[NSApp terminate:nil];
	}

/*
 * API Connections force WriteUp to create certain resources and proxies
 * that only benefit the calling application. It's good manners to let 
 * WriteUp know when you're done, so it can free those resources.
 */
- (void)applicationWillTerminate:(NSNotification *)aNotification
	{
	if (writeup)
		[writeup freeResources];
	}

/*
 * Utility method: OPENSTEP doesn't have a pure sleep() function, so this 
 * substitutes. This method is called by many of the demos below, otherwise 
 * the special effects would happen too quickly to be noticed.
 */
- (void)_sleep:(float)seconds
	{
	[NSThread sleepUntilDate:[NSDate dateWithTimeIntervalSinceNow:seconds]];
	}

/*
 * This test block exercises the cut/copy/paste/selectAll methods, then 
 * creates a named paragraph style and applies it to every other paragraph.
 */
- (void)runCutPasteStylesTest:sender
	{
	id newDoc, mainFlow, para, bodyStyle;
	int i, c;
	
	/* Reset connection, in case desired host has change */

	[self _connectToWriteUpOnHost:[hostnameField stringValue]];

	/* Create a new empty document and focus on the main flow */

	newDoc = [writeup newEmptyDocument];
	mainFlow = [newDoc focusOnFlow:WU_MAINFLOW];
	[newDoc beginningOfCurrentFlow];
	
	/* Create a paragraph, select it, cut, paste twice, selectAll again, */
	/* then paste twice more. The ending effect is four identical paras. */
	/* Notice how nicely the method calls self-document what's going on. */

	[mainFlow appendString:[[textPasteField textStorage] string]];
	[mainFlow appendString:@"\n"];
	[newDoc selectAll];
	[newDoc cutPB];
	[newDoc pastePB];
	[newDoc pastePB];
	[newDoc selectAll];
	[newDoc copyPB];
	[newDoc pastePB];
	[newDoc pastePB];
	
	/* Create a named paragraph style. */

	bodyStyle = [newDoc createStyleNamed:@"Body"];
	[bodyStyle setAlignment:NSJustifiedTextAlignment];
	[bodyStyle setIndentLeft:40.0];
	[bodyStyle setIndentDeltaFirst:60.0];
	[bodyStyle setSpaceAfter:60.0];
	[bodyStyle setLineSpacing:1.2];

	/* Go back to the beginning of the document, */
	/* apply the style to every other paragraph  */

	[newDoc beginningOfCurrentFlow];
	c = [mainFlow paragraphCount];
	for (i = 0; i < c; i++)
		{
		para = [mainFlow paragraphAtIndex:i];
		if (i % 2 == 0)
			[para setStylePara:bodyStyle];
		}
	[newDoc recompute];
	}

/*
 * This test block creates a header and footer, changes the document margins, 
 * populates the header and footer with some text and tokens, then populates 
 * the main flow and demonstrates a few column modes.
 */
- (void)runHeadersFootersColumnsTest:sender
	{
	id newDoc, mainFlow;
	float b,t,l,r;
	
	/* Reset connection, in case desired host has changed */

	[self _connectToWriteUpOnHost:[hostnameField stringValue]];

	/* Create a new empty document and focus on the main flow */

	newDoc = [writeup newEmptyDocument];
	mainFlow = [newDoc focusOnFlow:WU_MAINFLOW];
	[newDoc beginningOfCurrentFlow];

	/* Margin operations: get the current document margins, */
	/* then move the left & right 1/2" inward, bottom 1" up */

	[newDoc getMarginsLeft:&l right:&r top:&t bottom:&b];
	[newDoc setMarginsLeft:(l+36.0) right:(r+36.0) 
		top:t bottom:(b+72.)];

	/* 
	 * By default, a WriteUp document only has a main flow. These methods 
	 * create a header and footer that will appear on all pages. There are 
	 * also variants to create odd and even headers/footers. The new header 
	 * or footer takes effect from the current text point forward (which in 
	 * this case is the entire document)
	 */
	[newDoc insertNormalHeaderNamed:nil];
	[newDoc insertNormalFooterNamed:nil];

	/* Shift focus to header, insert text and page number token */
	
	mainFlow = [newDoc focusOnFlow:WU_HEADER];
	[newDoc endOfCurrentFlow];
	[mainFlow appendString:@"Header Text - inserted via API. Page "];
	[newDoc insertPageNumber];

	/* Shift focus to footer, insert text */

	mainFlow = [newDoc focusOnFlow:WU_FOOTER];
	[newDoc endOfCurrentFlow];
	[mainFlow appendString:@"Footer Text - inserted via API."];

	/* Create two columns for the main flow */
	
	[newDoc setNumberOfColumns:2 withGap:2.60];
	[newDoc recompute];

	/* Focus on main flow, paste text with page/column break */

	mainFlow = [newDoc focusOnFlow:WU_MAINFLOW];
	[newDoc endOfCurrentFlow];
	[mainFlow appendString:[[textPasteField textStorage] string]];
	[newDoc insertPageBreak];
	[mainFlow appendString:[[textPasteField textStorage] string]];
	[newDoc recompute];
	[self _sleep:1.0];

	/* Just to show off, switch to four-column mode */
	
	[newDoc setNumberOfColumns:4 withGap:10.0];
	[newDoc recompute];
	}

/*
 * This test block creates two named paragraph styles with different 
 * kinds of tabs, then lays out a simple table.
 */
- (void)runSimpleTableTest:sender
	{
	id newDoc, mainFlow, bodyStyle, headStyle;
	WUTab tab;

	/* Reset connection, in case desired host has changed */

	[self _connectToWriteUpOnHost:[hostnameField stringValue]];

	/* Create a new empty document and focus on the main flow */

	newDoc = [writeup newEmptyDocument];
	mainFlow = [newDoc focusOnFlow:WU_MAINFLOW];
	[newDoc endOfCurrentFlow];

	/* Create a named paragraph style, set attributes. */

	headStyle = [newDoc createStyleNamed:@"TableHead"];
	[headStyle setTabs:NULL :0];
	[headStyle setSpaceAfter:18.0];
	[headStyle setSpaceBefore:12.0];
	[headStyle setIndentLeft:10.0];
	[headStyle setIndentRight:15.0];
	[headStyle setFont:@"Helvetica-Bold" size:12.0];

	/* Create tabs. Values are copied; it's safe to reuse variable */

	tab.type = WU_TAB_LEFT;
	tab.value = 36.0;
	tab.leader = 0;
	[headStyle addTab:&tab];
	tab.type = WU_TAB_CENTER;
	tab.value = 200.0;
	[headStyle addTab:&tab];
	tab.type = WU_TAB_CENTER;
	tab.value = 300.0;
	[headStyle addTab:&tab];
	tab.type = WU_TAB_RIGHT;
	tab.value = 400.0;
	[headStyle addTab:&tab];

	/* Now repeat the process for a second named style. styleWithName: */
	/* is entirely gratuitous (because createStyleNamed: would return  */
	/* it directly), but this demonstrates how to get it back later.   */

	[newDoc createStyleNamed:@"TableBody"];
	bodyStyle = [newDoc styleForName:@"TableBody"];
	[bodyStyle setTabs:NULL :0];
	[bodyStyle setIndentRight:15.0];
	[bodyStyle setFont:@"Times-Roman" size:12.0];

	tab.type = WU_TAB_LEFT;
	tab.value = 36.0;
	tab.leader = 0;
	[bodyStyle addTab:&tab];
	tab.type = WU_TAB_DECIMAL;
	tab.value = 200.0;
	[bodyStyle addTab:&tab];
	tab.type = WU_TAB_DECIMAL;
	tab.value = 300.0;
	[bodyStyle addTab:&tab];
	tab.type = WU_TAB_RIGHT;
	tab.value = 400.0;
	[bodyStyle addTab:&tab];

	/* Apply the headline style, append text for table headings */
	
	[[mainFlow currentParagraph] setStylePara:headStyle];
	[mainFlow appendString:@"\tSales Figures\tVolume\tUnits\tRight Tab\n"];

	/* Apply the body style, append text for 3 lines of table body */
	
	[[mainFlow currentParagraph] setStylePara:bodyStyle];
	[mainFlow appendString:@"\tEast \t123.45\t45.67\tRight One\n"];
	[mainFlow appendString:@"\tWest \t345.67\t45.67\tRight Two \n"];
	[mainFlow appendString:@"\tCentral \t555.67\t45.67\tRight Three\n"];
	
	/* Finally, apply headline style again, append totals */

	[[mainFlow currentParagraph] setStylePara:headStyle];
	[mainFlow appendString:@"\tTotal\t999.99\n"];

	/* This fixes a current font application bug. Force all */
	/* text to conform to the underlying paragraph font.    */

	[newDoc selectAll];
	[newDoc setParagraphDefaultFont];
	[newDoc endOfCurrentFlow];
	[newDoc recompute];
	}

/*
 * This test block creates three paragraphs, each with a different font 
 * attribute. Then it highlights the paragraphs individually and performs 
 * a variety of transformations on the final one.
 */
- (void)runSimpleSelectionTest:sender
	{
	id newDoc, mainFlow;
	int i, c;
	
	/* Reset connection, in case desired host has changed */

	[self _connectToWriteUpOnHost:[hostnameField stringValue]];

	/* Create a new empty document and focus on the main flow */

	newDoc = [writeup newEmptyDocument];
	mainFlow = [newDoc focusOnFlow:WU_MAINFLOW];
	[newDoc beginningOfCurrentFlow];

	/* Write sample text into 3 paras, each with a different attribute */

	[newDoc setUnderline:YES];
	[mainFlow appendString:[[textPasteField textStorage] string]];
	[newDoc setUnderline:NO];
	[mainFlow appendString:@"\n"];

	[newDoc setItalic:YES];
	[mainFlow appendString:[[textPasteField textStorage] string]];
	[newDoc setItalic:NO];
	[mainFlow appendString:@"\n"];
	
	[newDoc setBold:YES];
	[mainFlow appendString:[[textPasteField textStorage] string]];
	[newDoc setBold:NO];

	/* Highlight the paras individually, just because we can. 8^) */
	/* sleep method gives the user time to see what's happening. */

	c = [mainFlow paragraphCount];
	for (i = 0; i < c; i++)
		{
		[mainFlow selectParagraph:[mainFlow paragraphAtIndex:i]];
		[self _sleep:1.0];
		}
	
	/* Now run the last paragraph through some hoops. */
	/* It is still selected from the previous for() loop. */

	[newDoc alignCenter:self];
	[self _sleep:1.0];

	[newDoc alignSelJustify:self];
	[self _sleep:1.0];

	[newDoc alignLeft:self];
	[self _sleep:1.0];

	[newDoc alignRight:self];
	[self _sleep:1.0];

	[newDoc convertSelUpperCase:self];
	[self _sleep:1.0];

	[newDoc convertSelMixedCase:self];
	[self _sleep:1.0];

	[newDoc convertSelLowerCase:self];
	}

/*
 * This test block creates two paragraphs, then selects the second one 
 * and returns the beginning and ending selection points.
 */
- (void)runExtendedSelectionTest:sender
	{
	id newDoc, mainFlow, sel;

	/* Reset connection, in case desired host has changed */

	[self _connectToWriteUpOnHost:[hostnameField stringValue]];

	/* Create a new empty document and focus on the main flow */

	newDoc = [writeup newEmptyDocument];
	mainFlow = [newDoc focusOnFlow:WU_MAINFLOW];
	[newDoc beginningOfCurrentFlow];

	/* Write two paragraphs of sample text, recompute */
	
	[mainFlow appendString:[[textPasteField textStorage] string]];
	[mainFlow appendString:@"\n"];
	[mainFlow appendString:[[textPasteField textStorage] string]];
	[newDoc recompute];

	/* Select the current paragraph, report selection points */

	[mainFlow selectCurrentParagraph];
	sel = [newDoc getSelection];
	if (sel != NULL)
		{
		int bP, bW, bC, eP, eW, eC;
		[sel getSelectionFrom:&bP :&bW :&bC to:&eP :&eW :&eC];
		NSRunAlertPanel(@"WriteUp API Tester",
				@"Text selection = (%d,%d,%d) to (%d,%d,%d)", 
				nil, nil, nil, bP, bW, bC, eP, eW, eC);
		}
	else
		NSRunAlertPanel(@"WriteUp API Tester",
				@"No text selection.", nil, nil, nil);
	}

/*
 * This test block creates two paragraphs, the first of which contains 
 * an embedded graphic, then changes some of the information dynamically.
 */
- (void)runGraphicsTest:sender
	{
	id newDoc, mainFlow;
	NSImage *anImage;

	/* Reset connection, in case desired host has changed */

	[self _connectToWriteUpOnHost:[hostnameField stringValue]];

	/* Create a new empty document and focus on the main flow */

	newDoc = [writeup newEmptyDocument];
	mainFlow = [newDoc focusOnFlow:WU_MAINFLOW];
	[newDoc beginningOfCurrentFlow];

	/* Write two paragraphs of sample text with graphic, recompute */
	
	[mainFlow appendString:@"This is the WriteUp document logo:"];
	[mainFlow insertInlineGraphic:@"/LocalConvert/WriteUp/wup.tiff" 
		named:@"ReplaceMe"];
	[mainFlow appendString:@"\n"];
	[mainFlow appendString:[[textPasteField textStorage] string]];
	[newDoc recompute];
	[self _sleep:3.0];

	/* Now replace the product name and image contents a few times */

	[newDoc findString:@"WriteUp" replaceWith:@"WriteNow" 
		wholeWord:NO select:NO];
	[newDoc replaceFilename:@"/LocalConvert/WriteUp/wn.tiff" 
		forGraphicNamed:@"ReplaceMe"];
	[self _sleep:3.0];

	/* an alternate way to do it, using NSImage instead of a filename */

	[newDoc findString:@"WriteNow" replaceWith:@"WordPerfect" 
		wholeWord:NO select:NO];
	anImage = [[NSImage alloc] 
		initWithContentsOfFile:@"/LocalConvert/WriteUp/wp.tiff"];
	[newDoc replaceImage:anImage forGraphicNamed:@"ReplaceMe"];
	}

@end
