// Copyright 1997-1998 Omni Development, Inc.  All rights reserved.
//
// This software may only be used and reproduced according to the
// terms in the file OmniSourceLicense.html, which should be
// distributed with this project and can also be found at
// http://www.omnigroup.com/DeveloperResources/OmniSourceLicense.html.

#import "OWObjectStream.h"

#import <Foundation/Foundation.h>
#import <OmniBase/OmniBase.h>
#import <OmniFoundation/OmniFoundation.h>

#import "OWObjectStreamCursor.h"

RCS_ID("$Header: /Network/Developer/Source/CVS/OmniGroup/OWF/Content.subproj/OWObjectStream.m,v 1.5 1998/12/08 04:05:43 kc Exp $")

@interface OWObjectStream (Private)
- (void)waitForMoreObjects;
- (void)_noMoreData;
@end

@implementation OWObjectStream

enum {
    WRITEOBJECTS_EMPTY, WRITEOBJECTS_AVAILABLE,
};

// Init and dealloc

- init;
{
    if (![super init])
	return nil;
    readObjects = [[NSMutableArray alloc] init];
    readObjectsLock = [[NSRecursiveLock alloc] init];
    writeObjects = [[NSMutableArray alloc] init];
    writeObjectsLock = [[NSConditionLock alloc]
			initWithCondition:WRITEOBJECTS_EMPTY];
    endOfDataLock = [[NSLock alloc] init];
    [endOfDataLock lock];
    return self;
}

- (void)dealloc;
{
    [readObjects release];
    [readObjectsLock release];
    [writeObjects release];
    [writeObjectsLock release];
    [endOfDataLock release];
    [super dealloc];
}

//

- (void)writeObject:(id)anObject;
{
    if (!anObject)
	return;
    [writeObjectsLock lock];
    [writeObjects addObject:anObject];
    [writeObjectsLock unlockWithCondition:WRITEOBJECTS_AVAILABLE];
}

//

- (id)objectAtIndex:(unsigned int)index;
{
    while (index >= [readObjects count]) {
	if (endOfReadObjects)
	    return nil;
	[self waitForMoreObjects];
    }
    return [readObjects objectAtIndex:index];
}

- (unsigned int)objectCount;
{
    [self waitForDataEnd];
    return [readObjects count] + [writeObjects count];
}

// OWObjectStream subclass

- (void)dataEnd;
{
    [self _noMoreData];
}

- (void)dataAbort;
{
    [self _noMoreData];
}

- (void)waitForDataEnd;
{
    [endOfDataLock lock];
    [endOfDataLock unlock];
}

- (BOOL)endOfData;
{
    return endOfReadObjects;
}

// Debugging

- (NSMutableDictionary *)debugDictionary;
{
    NSMutableDictionary *debugDictionary;

    debugDictionary = [super debugDictionary];
    if (readObjects)
	[debugDictionary setObject:readObjects forKey:@"readObjects"];
    if (writeObjectsLock)
	[debugDictionary setObject:writeObjectsLock forKey:@"writeObjectsLock"];
    [debugDictionary setObject:endOfReadObjects ? @"YES" : @"NO" forKey:@"endOfReadObjects"];
    if (writeObjects)
	[debugDictionary setObject:writeObjects forKey:@"writeObjects"];
    if (writeObjectsLock)
	[debugDictionary setObject:writeObjectsLock forKey:@"writeObjectsLock"];
    [debugDictionary setObject:endOfWriteObjects ? @"YES" : @"NO" forKey:@"endOfWriteObjects"];

    return debugDictionary;
}

@end

@implementation OWObjectStream (Private)

- (void)waitForMoreObjects;
{
    unsigned int oldReadObjectsCount;

    if (endOfReadObjects)
	return;
    oldReadObjectsCount = [readObjects count];
    [readObjectsLock lock];
    if (endOfReadObjects || [readObjects count] > oldReadObjectsCount) {
	[readObjectsLock unlock];
	return;
    }
    [writeObjectsLock lockWhenCondition:WRITEOBJECTS_AVAILABLE];
    [readObjects addObjectsFromArray:writeObjects];
    if (endOfWriteObjects)
	endOfReadObjects = YES;
    [writeObjects removeAllObjects];
    [writeObjectsLock unlockWithCondition:WRITEOBJECTS_EMPTY];
    [readObjectsLock unlock];
}

- (void)_noMoreData;
{
    [writeObjectsLock lock];
    endOfWriteObjects = YES;
    [writeObjectsLock unlockWithCondition:WRITEOBJECTS_AVAILABLE];
    [endOfDataLock unlock];
}

@end
