/* -*-ObjC-*-
*******************************************************************************
*
* File:         compactmail.m
* RCS:          /usr/local/sources/CVS/mailapp-utilities/compactmail.m,v 1.16 1998/12/04 19:49:29 tom Exp
* Description:  Compact Mail.app mailboxes (expunge deleted messages)
* Author:       Carl Edman
* Created:      Fri Mar 12 18:21:23 1993
* Modified:     Fri Aug  7 19:27:24 1998 Tom Hageman <tom@basil.icce.rug.nl>
* Language:     Objective C
* Package:      mailapp-utilities
* Status:       Exp
*
* (C) Copyright 1993, but otherwise this file is perfect freeware.
*
*******************************************************************************
*/

#import <libc.h>
#import <errno.h>
//#import <stdlib.h>
//#import <stdio.h>
//#import <string.h>
//#import <time.h>
//#import <sys/file.h>
//#import <sys/param.h>
//#import <sys/types.h>
//#import <sys/stat.h>
#import "compat.h"
#import "mailutil.h"
#define _MAILTOC_DEFINES 1
#import "mailtoc.h"
#import "optutil.h"
#import "re.h"

#define USAGE "\
Usage: %s [-nrutv] [-d days] [-f from] [-s subject] [-V|-H] mbox...\n"

#define HELP "\
 -n            skip mailbox if locked\n\
 -r            expunge read messages too (default deleted messages only)\n\
 -u            expunge all messages (even unread, new and flagged)\n\
 -t            test mode -- do nothing, just report what would be expunged.\n\
 -v            talkative mode\n\
 -d days       restrict to messages of at least `days' days old.\n\
 -f from       restrict to messages whose From: header matches (regexp)\n\
 -s subject    restrict to messages whose Subject: header matches (regexp)\n\
 -T sec        timeout: give up after `sec' seconds if mailbox is locked\n\
 -V,--version  show version\n\
 -H,--help     this help\n\
"

int main(int ac,char *av[])
{
   char *buf=0;
   int c,pos;
   int nowaitflg=0,verboseflg=0,deletereadflg=0,deleteunreadflg=0,testflg=0;
   int days=-1;
   struct regex *subjectregexp=0,*fromregexp=0;
   time_t mboxtime;
   int mboxdis,tocdis,msgdis,changed;
   int mboxfd=-1;
   FILE *tocf=NULL;
   struct table_of_contents_header *toch=0;
   struct message_index *mi=0;
   int status=EXIT_SUCCESS;
   char *attachdir;
   POOL_INIT

   set_progname(av[0]);

   for (;;)
   {
      switch ((c=getopt(ac,av,"d:s:f:nrutvT:VH")))
      {
       case 'd':
	 if (days!=-1 || (days=atoi(optarg))<=0)
	 {
	    status=EXIT_USAGE;
	    break;
	 }
	 continue;
       case 's':
	 if (subjectregexp || !(subjectregexp=re_compile(optarg,0)))
	 {
	    status=EXIT_USAGE;
	    break;
	 }
	 continue;
       case 'f':
	 if (fromregexp || !(fromregexp=re_compile(optarg,0)))
	 {
	    status=EXIT_USAGE;
	    break;
	 }
	 continue;
       case 'n':
	 nowaitflg++;
	 continue;
       case 'r':
	 deletereadflg++;
	 continue;
       case 'u':
	 deleteunreadflg++;
	 continue;
       case 't':
	 testflg++;
	 continue;
       case 'v':
	 verboseflg++;
	 continue;
       case 'T':
	 nowaitflg = -atoi(optarg);
	 if (nowaitflg >= 0) nowaitflg = !nowaitflg;
	 continue;
       case 'V':
	 status=EXIT_VERSION;
	 break;
       case 'H':
	 status=EXIT_HELP;
	 break;
       case EOF:
	 status=check_arg_for_long_help_version(av[optind-1]);
	 break;
       case '?':
       default:
	 status=EXIT_USAGE;
      }
      break;
   }
   if (status==EXIT_SUCCESS && optind>=ac) status=EXIT_USAGE;
   handle_usage_help_version(status, USAGE, HELP);

   for(;optind<ac;optind++)
   {
      if (verboseflg) logprintf("Entering %s...",av[optind]);
      if (cd_mbox(av[optind], 0)) goto next;
      if (lock_mbox(nowaitflg)) goto next;

      mboxtime=mtime(MBOX_CONTENTS_FILE);

      if (((mboxfd=open(MBOX_CONTENTS_FILE,O_RDWR))==-1) ||
          ((tocf=fopen(MBOX_TOC_FILE,"r+b"))==NULL))
      {
         logprintf("opening %s: %s\n",av[optind],strerror(errno));
         status=EXIT_FAILURE;
         goto unlock;
      }

      toch=get_table_of_contents_header(tocf,0);
      if (!toch)
      {
         logprintf("%s: invalid old table_of_contents",av[optind]);
         status=EXIT_FAILURE;
         goto unlock;
      }

      if (toch->mbox_time!=mboxtime)
      {
         logprintf("%s: table of contents out of sync",av[optind]);
         status=EXIT_FAILURE;
         goto unlock;
      }

      for (changed = FALSE, msgdis = mboxdis = tocdis = c = 0;
	   c < toch->num_msgs;
	   c++, free(mi), mi = 0)
      {
         if ((mi=get_message_index(tocf))==0)
	 {
            logprintf("%s: reading old mbox entries: %s",av[optind],(errno?strerror(errno):"invalid"));
            status=EXIT_FAILURE;
            goto unlock;
	 }
         if ((mi->status == MT_STATUS_DELETED_1 ||
	      mi->status == MT_STATUS_DELETED_2 ||
	      deleteunreadflg ||
              (mi->status != MT_STATUS_NEW &&
	       mi->status != MT_STATUS_UNREAD_1 &&
	       mi->status != MT_STATUS_UNREAD_2 && deletereadflg)) &&
             (days==-1 || message_age(mi)>=days) &&
             (!subjectregexp || re_match(message_subject(mi),subjectregexp)) &&
             (!fromregexp || re_match(message_from(mi),fromregexp)))
	 {
            msgdis++;
            if (verboseflg>1 || testflg)
	    {
               logprintf("%s \"%s\" from %s (%d days old).",(testflg ? "Would expunge" : "Expunging"), message_subject(mi),message_from(mi),message_age(mi));
	       if (testflg) continue;
	    }
	    attachdir = message_reference(mi);
	    if (attachdir && *attachdir)
	    {
	       int pid,childpid;
	       int status;

	       switch (childpid=fork())
	       {
	       case -1:
		  break;
	       case 0:
		  execl("/bin/rm", "rm", "-rf", attachdir, NULL);
		  _exit(255);
	       default:
		  if (verboseflg>2)
		  {
		     logprintf("  removing attachment \"%s\".", attachdir);
		  }
		  while ((pid=wait((void *)&status))>=0 && pid!=childpid) ;
	       }
	    }
            changed=TRUE;
            tocdis +=mi->record_length;
            mboxdis+=mi->mes_length;
	 }
         else
	 {
            if (verboseflg>2)
	    {
               logprintf("Keeping \"%s\" from %s (%d days old).",message_subject(mi),message_from(mi),message_age(mi));
	    }
            if (mboxdis!=0)
	    {
               lseek(mboxfd,mi->mes_offset,SEEK_SET);
               buf=malloc(mi->mes_length);
               read(mboxfd,buf,mi->mes_length);
               mi->mes_offset -= mboxdis;
               lseek(mboxfd,mi->mes_offset,SEEK_SET);
               write(mboxfd,buf,mi->mes_length);
               free(buf); buf=0;
	    }

            if (tocdis!=0 || mboxdis!=0)
	    {
               pos=ftell(tocf);
               fseek(tocf,-mi->record_length-tocdis,SEEK_CUR);
               put_message_index(tocf,mi);
               fseek(tocf,pos,SEEK_SET);
	    }
	 }
      }
      
      if (tocdis!=0)
      {
	 fflush(tocf);
	 fseek(tocf,0,SEEK_END);
         pos=ftell(tocf);
	 fseek(tocf,0,SEEK_SET); // trying to defeat FILE <-> fd interference...
         ftruncate(fileno(tocf),pos-tocdis);
      }
      if (mboxdis!=0)
      {
         pos=lseek(mboxfd,0,SEEK_END);
         ftruncate(mboxfd,pos-mboxdis);
      }
      if (tocdis!=0 || mboxdis!=0)
      {
	 unlink(MBOX_INDEX_FILE); // XXX Should at least preserve existence of index.
      }
      close(mboxfd), mboxfd=-1;
      
      if (verboseflg || testflg)
      {
	 logprintf((testflg ? "Would expunge %d message%s." : "%d message%s expunged."), msgdis, (msgdis == 1 ? "" : "s"));
      }
      toch->num_msgs-=msgdis;
      mboxtime=mtime(MBOX_CONTENTS_FILE);
      if ((toch->mbox_time!=mboxtime) || changed)
      {
         toch->mbox_time=mboxtime;
         fseek(tocf,0,SEEK_SET);
         if (put_table_of_contents_header(tocf,toch))
	 {
            logprintf("%s: writing updated table_of_contents: %s",av[optind],strerror(errno));
            status=EXIT_FAILURE;
            goto unlock;
	 }
      }
    unlock:
      if(mboxfd>=0) close(mboxfd), mboxfd=-1;
      if(tocf!=NULL) fclose(tocf), tocf=NULL;
      unlock_mbox();
    next:;
      if (toch) { free(toch); toch=0; }
      if (mi) { free(mi); mi=0; }
      if (buf) { free(buf); buf=0; }
      uncd_mbox();
   }
   POOL_RELEASE
   return status;
}
