// Copyright 1998 Omni Development, Inc.  All rights reserved.

#import "OLIndexedFolder.h"

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>
#import <OmniBase/OmniBase.h>
#import <OmniAIAT/OAIIndex.h>

#import "OLIndexFolderDragging.h"

RCS_ID("$Header: /Network/Developer/Source/CVS/OmniGroup/OmniLibrarian/OLIndexedFolder.m,v 1.7 1998/03/24 05:50:42 andrew Exp $")

@interface NSMutableString (OLExtensions)
- (void)replaceInstancesOfString:(NSString *)stringA withString:(NSString *)stringB;
@end

@implementation NSMutableString (OLExtensions)

- (void)replaceInstancesOfString:(NSString *)stringA withString:(NSString *)stringB;
{
    BOOL done  = NO;

    ASSERT (![stringA isEqualToString:stringB]);
    
    while (!done) {
        NSRange range;

        range = [self rangeOfString:stringA];
        if (range.length == 0) {
            done = YES;
            break;
        }

        [self replaceCharactersInRange:range withString:stringB];
    }
}

@end


@interface OLIndexedFolder (Private)
- (NSString *)indexFilenameForTarget:(NSString *)targetPath;
- (NSString *)libraryDirectory;
@end


@implementation OLIndexedFolder

// Init and dealloc;

- initWithIndexFilename:(NSString *)filename forFolder:(NSString *)folderName label:(NSString *)aLabel lastUpdate:(NSCalendarDate *)last;
{
    if (![super init])
        return nil;

    if (filename)
        indexFilename = [filename retain];
    else {
    }
    folder = [folderName retain];
    label = [aLabel copy];
    lastUpdated = [last retain];

    return self;
}

- initWithInfoDictionary:(NSDictionary *)infoDictionary;
{
    NSString *filename, *folderName, *aLabel;
    NSCalendarDate *date;
   
    filename = [infoDictionary objectForKey:@"filename"];
    folderName = [infoDictionary objectForKey:@"folder"];
    aLabel = [infoDictionary objectForKey:@"label"];
    date = [NSCalendarDate dateWithString:[infoDictionary objectForKey:@"lastUpdated"] calendarFormat:@"%m/%d/%Y %H:%M:%S"];
    return [self initWithIndexFilename:filename forFolder:folderName label:aLabel lastUpdate:date];
}

- (void)dealloc;
{
    [index release];
    [indexFilename release];
    [folder release];
    [lastUpdated release];
    [super dealloc];
}

//

- (BOOL)isIndexed;
{
    return [self indexFilename] != nil;
}

- (NSString *)indexFilenameForTarget:(NSString *)targetPath;
{
    NSMutableString *resultFilename;

    resultFilename = [[targetPath stringByStandardizingPath] mutableCopy];
//    [resultFilename replaceInstancesOfString:@"_" withString:@"__"];
    [resultFilename replaceInstancesOfString:@"/" withString:@"_"];

    return [resultFilename stringByAppendingPathExtension:@"OmniLibrarianIndex"];
}

- (NSString *)libraryDirectory;
{
    NSFileManager *fileManager;
    NSString *libraryDirectory;

    libraryDirectory = [@"~/Library/OmniLibrarian" stringByExpandingTildeInPath];
    fileManager = [NSFileManager defaultManager];
    if (![fileManager fileExistsAtPath:libraryDirectory]) {
        if (![fileManager createDirectoryAtPath:libraryDirectory attributes:nil])
            [NSException raise:@"CreateLibraryDirectoryFailed" format:@"Unable to create library directory at %@", libraryDirectory];
    }

    return libraryDirectory;
}

- (OAIIndex *)index;
{
    NSFileManager *fileManager;
    NSString *filename;

    // if the target folder isn't yet indexed, index it now
    if (!index) {
        filename = [self indexFilename];
        if (filename) {
            index = [[OAIIndex alloc] initWithIndex:filename forDirectory:folder];
        } else {
            fileManager = [NSFileManager defaultManager];
            
            // if we can write to target folder, place index there
            if ([fileManager isWritableFileAtPath:folder]) {
                filename = [folder stringByAppendingPathComponent:@".OmniLibrarianIndex"];
                index = [[OAIIndex alloc] initNewIndex:filename forDirectory:folder];
                [index update];
                indexFilename = [filename retain];
                
            } else {	// otherwise, put it in ~/Library/OmniLibrarian
                NSString *libraryDirectory;
                unsigned int digit;

                libraryDirectory = [self libraryDirectory];
                indexFilename = [[libraryDirectory stringByAppendingPathComponent:[self indexFilenameForTarget:folder]] retain];
                index = [[OAIIndex alloc] initNewIndex:indexFilename forDirectory:folder];
                [index update];
            }
        }
    }
    
    return index;
}

- (NSString *)indexFilename;
{
    // if we don't have an index filename, try to find one
    if (!indexFilename) {
        NSFileManager *fileManager;
        NSString *tempIndexFilename;

        fileManager = [NSFileManager defaultManager];
        
        // first, check for OmniLibrarian index file in the target folder
        tempIndexFilename = [folder stringByAppendingPathComponent:@".OmniLibrarianIndex"];
        
        // if not one, check OmniLibrarian library folder for an index file for the target folder
        if (![fileManager isReadableFileAtPath:tempIndexFilename])
            tempIndexFilename = [[self libraryDirectory] stringByAppendingPathComponent:[self indexFilenameForTarget:folder]];
        
        // if we found an index filename, remember it, and read in the index
        if ([fileManager isReadableFileAtPath:tempIndexFilename]) {
            indexFilename = [tempIndexFilename retain];
            index = [[OAIIndex alloc] initWithIndex:indexFilename forDirectory:folder];
        }
    }

    return indexFilename;
}

- (NSString *)folder;
{
    return folder;
}

- (NSString *)label; { return label ? label : [folder lastPathComponent]; }
- (void)setLabel:(NSString *)newLabel;
{
    [label autorelease];
    label = [newLabel copy];
}

- (NSCalendarDate *)lastUpdated;
{
    return lastUpdated;
}

- (void)updateIndex;
{
    [[self index] update];
    [lastUpdated release];
    lastUpdated = [[NSCalendarDate calendarDate] retain];
}

- (NSMutableDictionary *)infoDictionary;
{
    NSMutableDictionary *info;

    info = [NSMutableDictionary dictionary];
    if (indexFilename)
        [info setObject:indexFilename forKey:@"filename"];
    [info setObject:folder forKey:@"folder"];
    if (label)
        [info setObject:label forKey:@"label"];
    if (lastUpdated)
        [info setObject:[lastUpdated descriptionWithCalendarFormat:@"%m/%d/%Y %H:%M:%S"] forKey:@"lastUpdated"];
    return info;
}

- (void)pasteboard:(NSPasteboard *)pasteboard provideDataForType:(NSString *)type;
{
    if ([type isEqualToString:[OLIndexFolderDragging indexFolderType]]) {
        [pasteboard setPropertyList:[self infoDictionary] forType:type];
    } else if ([type isEqualToString:NSFilenamesPboardType]) {
        [pasteboard setPropertyList:[NSArray arrayWithObject:folder] forType:type];
    }
}

@end
