#ifndef __MiscTableScroll_h
#define __MiscTableScroll_h
//=============================================================================
//
//  Copyright (C) 1995,1996,1997,1998 by Paul S. McCarthy and Eric Sunshine.
//		Written by Paul S. McCarthy and Eric Sunshine.
//			    All Rights Reserved.
//
//	This notice may not be removed from this source code.
//
//	This object is included in the MiscKit by permission from the authors
//	and its use is governed by the MiscKit license, found in the file
//	"License.rtf" in the MiscKit distribution.  Please refer to that file
//	for a list of all applicable permissions and restrictions.
//	
//=============================================================================
//-----------------------------------------------------------------------------
// <MiscTableScroll.h>
//
//	ScrollView class that displays a 2-D table of cells.
//
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// $Id: MiscTableScroll.h,v 1.31 98/03/29 23:54:21 sunshine Exp $
// $Log:	MiscTableScroll.h,v $
// Revision 1.31  98/03/29  23:54:21  sunshine
// v138.1: Added -tableScroll:shouldDelayWindowOrderingForEvent:.
// Worked around OPENSTEP 4.2 for NT bug where compiler crashes when
// sending a message to 'super' from within a category.
// 
// Revision 1.30  98/03/23  07:47:32  sunshine
// v134.1: Added -suspendEditing, -resumeEditing.
// 
// Revision 1.29  98/03/22  13:15:25  sunshine
// v133.1: Scrollers are no longer forced on.  nullView -> cornerView.
// Eliminated constrain-max functionality.  Constrain-min functionality
// is now implicit based on presence/absence of auto-size slots.
// Eliminated {get|set}{Min|Max}Total{Size|Height|Width} methods.
// Eliminated data-sizing.  Added sizeToFit.  Added {get|set}cornerTitle.
// Added {get|set}{min|max}Uniform{Size|Height|Width} methods.
// Added {get|set}drawClippedText methods.
// Broke off MiscTableScrollIO.M
//-----------------------------------------------------------------------------
#import <MiscTableScroll/MiscTableTypes.h>

MISC_TS_EXTERN_BEGIN( "Objective-C" )
#import <AppKit/NSScrollView.h>
MISC_TS_EXTERN_END

MISC_TS_CLASS_DEF( MiscTableBorder );
MISC_TS_CLASS_DEF( MiscDelegateFlags );
@class MiscTableScroll, MiscTableView, MiscBorderView, MiscCornerView;
@class NSCell, NSClipView, NSFont, NSText;


typedef struct MiscCellEditInfo
	{
	BOOL			editing;	// Is editing in progress?
	MiscCoord_P		row;		// Coords of edited cell.
	MiscCoord_P		col;
	NSText*			editor;		// Field editor.
	id			cell;		// Temporary editing cell.
	int			suspended;	// Suspend/resume counter.
	} MiscCellEditInfo;


typedef struct MiscBorderInfo
	{
	MiscTableBorder*	border;
	MiscBorderView*		view;
	NSClipView*		clip;
	NSArray*		sort_vector;
	BOOL			isOn;
	BOOL			autoSort;
	} MiscBorderInfo;


@interface MiscTableScroll : NSScrollView
    {
@private
    MiscTableView*	tableView;
    MiscBorderInfo*	info[2];	// { &colInfo, &rowInfo }
    MiscBorderInfo	colInfo;
    MiscBorderInfo	rowInfo;
    MiscCornerView*	cornerView;
    NSFont*		font;
    NSColor*		textColor;
    NSColor*		backgroundColor;
    NSColor*		selectedTextColor;
    NSColor*		selectedBackgroundColor;
    id			delegate;
    id			dataDelegate;
    MiscDelegateFlags*	delegateFlags;
    MiscDelegateFlags*	dataDelegateFlags;
    id			target;
    id			doubleTarget;
    SEL			action;
    SEL			doubleAction;
    int			tag;
    MiscCompareEntryFunc sort_entry_func;
    MiscCompareSlotFunc	sort_slot_func;
    MiscSelectionMode	mode;
    int			num_cols;	// Currently active number of
    int			num_rows;	// columns and rows.
    int			max_rows;	// Highwater mark for Cell allocations.
    int			max_cells;
    id*			cells;
    id			pageHeader;
    id			pageFooter;
    MiscCellEditInfo	editInfo;
    BOOL		tracking;
    BOOL		enabled;
    BOOL		lazy;
    BOOL		drawClippedText;
    }

- (id)initWithFrame:(NSRect)frameRect;
- (void)dealloc;


// DELEGATE / TAG -------------------------------------------------------------
- (id)delegate;
- (void)setDelegate:(id)obj;
- (id)dataDelegate;
- (void)setDataDelegate:(id)obj;

- (int)tag;
- (void)setTag:(int)x;

// TARGET / ACTION ------------------------------------------------------------
- (id)target;
- (void)setTarget:(id)obj;
- (id)doubleTarget;
- (void)setDoubleTarget:(id)obj;
- (SEL)action;
- (void)setAction:(SEL)new_sel;
- (SEL)doubleAction;
- (void)setDoubleAction:(SEL)new_sel;

- (BOOL)sendAction:(SEL)theAction to:(id)theTarget;
- (BOOL)sendAction;
- (BOOL)sendActionIfEnabled;
- (BOOL)sendDoubleAction;
- (BOOL)sendDoubleActionIfEnabled;


// FONT -----------------------------------------------------------------------
- (NSFont*)font;
- (void)setFont:(NSFont*)newFont;
- (void)changeFont:(id)sender;
+ (NSFont*)defaultFont;


// COLOR ----------------------------------------------------------------------
- (NSColor*)backgroundColor;
- (void)setBackgroundColor:(NSColor*)value;		// Sent to all cells.
- (NSColor*)textColor;
- (void)setTextColor:(NSColor*)value;			// Sent to all cells.
- (NSColor*)selectedBackgroundColor;
- (void)setSelectedBackgroundColor:(NSColor*)value;	// Sent to all cells.
- (NSColor*)selectedTextColor;
- (void)setSelectedTextColor:(NSColor*)value;		// Sent to all cells.

- (void)setColor:(NSColor*)value;	// Alias for -setBackgroundColor:
- (NSColor*)color;			// Alias for -backgroundColor

+ (NSColor*)defaultBackgroundColor;
+ (NSColor*)defaultTextColor;
+ (NSColor*)defaultSelectedBackgroundColor;
+ (NSColor*)defaultSelectedTextColor;


// SLOT ORDER -----------------------------------------------------------------
- (NSArray*)slotOrder:(MiscBorderType)b;
- (NSArray*)columnOrder;
- (NSArray*)rowOrder;

- (BOOL)border:(MiscBorderType)b setSlotOrder:(NSArray*)list;
- (BOOL)setColumnOrder:(NSArray*)list;
- (BOOL)setRowOrder:(NSArray*)list;

- (NSString*)slotOrderAsString:(MiscBorderType)b;
- (NSString*)columnOrderAsString;
- (NSString*)rowOrderAsString;

- (BOOL)border:(MiscBorderType)b setSlotOrderFromString:(NSString*)s;
- (BOOL)setColumnOrderFromString:(NSString*)s;
- (BOOL)setRowOrderFromString:(NSString*)s;


// SLOT SIZES -----------------------------------------------------------------
- (NSArray*)slotSizes:(MiscBorderType)b;
- (NSArray*)columnSizes;
- (NSArray*)rowSizes;

- (BOOL)border:(MiscBorderType)b setSlotSizes:(NSArray*)list;
- (BOOL)setColumnSizes:(NSArray*)list;
- (BOOL)setRowSizes:(NSArray*)list;

- (NSString*)slotSizesAsString:(MiscBorderType)b;
- (NSString*)columnSizesAsString;
- (NSString*)rowSizesAsString;

- (BOOL)border:(MiscBorderType)b setSlotSizesFromString:(NSString*)s;
- (BOOL)setColumnSizesFromString:(NSString*)s;
- (BOOL)setRowSizesFromString:(NSString*)s;


// MULTICAST MESSAGES ---------------------------------------------------------
- (void)sendAction:(SEL)aSelector to:(id)anObject forAllCells:(BOOL)flag;

- (int)makeCellsPerformSelector:(SEL)aSel selectedOnly:(BOOL)flag;
- (int)makeCellsPerformSelector:(SEL)aSel with:(id)arg1 selectedOnly:(BOOL)f;
- (int)makeCellsPerformSelector:(SEL)aSel with:(id)arg1 with:(id)arg2
		selectedOnly:(BOOL)flag;

- (int)makeCellsPerformSelector:(SEL)aSel;
- (int)makeCellsPerformSelector:(SEL)aSel with:(id)arg1;
- (int)makeCellsPerformSelector:(SEL)aSel with:(id)arg1 with:(id)arg2;


// FINDING CELLS --------------------------------------------------------------
- (int)border:(MiscBorderType)b slotWithTag:(int)x;
- (int)columnWithTag:(int)x;
- (int)rowWithTag:(int)x;
- (id)cellWithTag:(int)x;
- (BOOL)getRow:(int*)row column:(int*)col ofCellWithTag:(int)x;
- (BOOL)getRow:(int*)row column:(int*)col ofCell:(NSCell*)cell;


// TOTAL SIZE -----------------------------------------------------------------
- (float)totalSize:(MiscBorderType)b;
- (float)totalWidth;
- (float)totalHeight;

- (void)constrainSize;		// Applies constraints, updates views.


// CORNER VIEW ----------------------------------------------------------------
- (NSString*)cornerTitle;
- (void)setCornerTitle:(NSString*)s;


// SLOTS ----------------------------------------------------------------------
- (MiscTableBorder*)border:(MiscBorderType)b;
- (BOOL)slotTitlesOn:(MiscBorderType)b;
- (BOOL)border:(MiscBorderType)b setSlotTitlesOn:(BOOL)on_off;
	// -border:setTitlesOn: Returns YES if changed.
- (MiscTableTitleMode)slotTitleMode:(MiscBorderType)b;
- (void)border:(MiscBorderType)b setSlotTitleMode:(MiscTableTitleMode)x;
- (float)slotTitlesSize:(MiscBorderType)b;
- (void)border:(MiscBorderType)b setSlotTitlesSize:(float)x;

- (void)border:(MiscBorderType)b moveSlotFrom:(int)pos to:(int)pos;
- (int)border:(MiscBorderType)b slotPosition:(int)slot;
- (int)border:(MiscBorderType)b slotAtPosition:(int)pos;
- (NSArray*)border:(MiscBorderType)b physicalToVisual:(NSArray*)list;
- (NSArray*)border:(MiscBorderType)b visualToPhysical:(NSArray*)list;

- (BOOL)sizeableSlots:(MiscBorderType)b;
- (BOOL)draggableSlots:(MiscBorderType)b;
- (BOOL)modifierDragSlots:(MiscBorderType)b;
- (float)uniformSizeSlots:(MiscBorderType)b;
- (float)minUniformSizeSlots:(MiscBorderType)b;
- (float)maxUniformSizeSlots:(MiscBorderType)b;

- (float)border:(MiscBorderType)b slotAdjustedSize:(int)slot;
- (float)border:(MiscBorderType)b slotSize:(int)slot;
- (float)border:(MiscBorderType)b slotMinSize:(int)slot;
- (float)border:(MiscBorderType)b slotMaxSize:(int)slot;
- (BOOL)border:(MiscBorderType)b slotIsSizeable:(int)slot;
- (BOOL)border:(MiscBorderType)b slotIsAutosize:(int)slot;
- (NSString*)border:(MiscBorderType)b slotTitle:(int)slot;
- (int)border:(MiscBorderType)b slotTag:(int)slot;
- (MiscTableCellStyle)border:(MiscBorderType)b slotCellType:(int)slot;
- (id)border:(MiscBorderType)b slotCellPrototype:(int)slot;

- (void)border:(MiscBorderType)b setSizeableSlots:(BOOL)flag;
- (void)border:(MiscBorderType)b setDraggableSlots:(BOOL)flag;
- (void)border:(MiscBorderType)b setModifierDragSlots:(BOOL)flag;
- (void)border:(MiscBorderType)b setUniformSizeSlots:(float)uniform_size;
- (void)border:(MiscBorderType)b setMinUniformSizeSlots:(float)size;
- (void)border:(MiscBorderType)b setMaxUniformSizeSlots:(float)size;

- (void)border:(MiscBorderType)b setSlot:(int)n size:(float)size;
- (void)border:(MiscBorderType)b setSlot:(int)n minSize:(float)size;
- (void)border:(MiscBorderType)b setSlot:(int)n maxSize:(float)size;
- (void)border:(MiscBorderType)b setSlot:(int)n sizeable:(BOOL)flag;
- (void)border:(MiscBorderType)b setSlot:(int)n autosize:(BOOL)flag;
- (void)border:(MiscBorderType)b setSlot:(int)n title:(NSString*)title;
- (void)border:(MiscBorderType)b setSlot:(int)n tag:(int)tag;
- (void)border:(MiscBorderType)b setSlot:(int)n
		cellType:(MiscTableCellStyle)t;
- (void)border:(MiscBorderType)b setSlot:(int)n cellPrototype:(id)cell;


// COLS -----------------------------------------------------------------------
- (MiscTableBorder*)columnBorder;
- (BOOL)columnTitlesOn;
- (BOOL)setColumnTitlesOn:(BOOL)on_off;	// Returns YES if changed.
- (MiscTableTitleMode)columnTitleMode;
- (void)setColumnTitleMode:(MiscTableTitleMode)x;
- (float)columnTitlesHeight;
- (void)setColumnTitlesHeight:(float)x;

- (void)moveColumnFrom:(int)pos to:(int)pos;
- (int)columnPosition:(int)col;
- (int)columnAtPosition:(int)pos;

- (float)uniformSizeColumns;
- (float)minUniformSizeColumns;
- (float)maxUniformSizeColumns;
- (BOOL)sizeableColumns;
- (BOOL)draggableColumns;
- (BOOL)modifierDragColumns;
- (float)columnAdjustedSize:(int)col;
- (float)columnSize:(int)col;
- (float)columnMinSize:(int)col;
- (float)columnMaxSize:(int)col;
- (BOOL)columnIsSizeable:(int)col;
- (BOOL)columnIsAutosize:(int)col;
- (NSString*)columnTitle:(int)col;
- (int)columnTag:(int)col;
- (MiscTableCellStyle)columnCellType:(int)col;
- (id)columnCellPrototype:(int)col;

- (void)setSizeableColumns:(BOOL)flag;
- (void)setDraggableColumns:(BOOL)flag;
- (void)setModifierDragColumns:(BOOL)flag;
- (void)setUniformSizeColumns:(float)uniform_size;
- (void)setMinUniformSizeColumns:(float)size;
- (void)setMaxUniformSizeColumns:(float)size;

- (void)setColumn:(int)col size:(float)size;
- (void)setColumn:(int)col minSize:(float)size;
- (void)setColumn:(int)col maxSize:(float)size;
- (void)setColumn:(int)col sizeable:(BOOL)flag;
- (void)setColumn:(int)col autosize:(BOOL)flag;
- (void)setColumn:(int)col title:(NSString*)title;
- (void)setColumn:(int)col tag:(int)tag;
- (void)setColumn:(int)col cellType:(MiscTableCellStyle)type;
- (void)setColumn:(int)col cellPrototype:(id)cell;

- (int)numberOfVisibleColumns;
- (int)firstVisibleColumn;
- (int)lastVisibleColumn;
- (BOOL)columnIsVisible:(int)n;
- (void)setFirstVisibleColumn:(int)n;
- (void)setLastVisibleColumn:(int)n;


// ROWS -----------------------------------------------------------------------
- (MiscTableBorder*)rowBorder;
- (BOOL)rowTitlesOn;
- (BOOL)setRowTitlesOn:(BOOL)on_off;
- (MiscTableTitleMode)rowTitleMode;
- (void)setRowTitleMode:(MiscTableTitleMode)x;
- (float)rowTitlesWidth;
- (void)setRowTitlesWidth:(float)x;

- (void)moveRowFrom:(int)fromPos to:(int)toPos;
- (int)rowPosition:(int)row;
- (int)rowAtPosition:(int)pos;

- (BOOL)sizeableRows;
- (BOOL)draggableRows;
- (BOOL)modifierDragRows;
- (float)uniformSizeRows;
- (float)minUniformSizeRows;
- (float)maxUniformSizeRows;
- (float)rowAdjustedSize:(int)row;
- (float)rowSize:(int)row;
- (float)rowMinSize:(int)row;
- (float)rowMaxSize:(int)row;
- (BOOL)rowIsSizeable:(int)row;
- (BOOL)rowIsAutosize:(int)row;
- (NSString*)rowTitle:(int)row;
- (int)rowTag:(int)row;
- (MiscTableCellStyle)rowCellType:(int)row;
- (id)rowCellPrototype:(int)row;

- (void)setSizeableRows:(BOOL)flag;
- (void)setDraggableRows:(BOOL)flag;
- (void)setModifierDragRows:(BOOL)flag;
- (void)setUniformSizeRows:(float)uniform_size;
- (void)setMinUniformSizeRows:(float)size;
- (void)setMaxUniformSizeRows:(float)size;

- (void)setRow:(int)row size:(float)size;
- (void)setRow:(int)row minSize:(float)size;
- (void)setRow:(int)row maxSize:(float)size;
- (void)setRow:(int)row sizeable:(BOOL)flag;
- (void)setRow:(int)row autosize:(BOOL)flag;
- (void)setRow:(int)row title:(NSString*)title;
- (void)setRow:(int)row tag:(int)tag;
- (void)setRow:(int)row cellType:(MiscTableCellStyle)type;
- (void)setRow:(int)row cellPrototype:(id)cell;

- (int)numberOfVisibleRows;
- (int)firstVisibleRow;
- (int)lastVisibleRow;
- (BOOL)rowIsVisible:(int)n;
- (void)setFirstVisibleRow:(int)n;
- (void)setLastVisibleRow:(int)n;


// DRAWING --------------------------------------------------------------------
- (NSRect)documentClipRect;

- (void)drawCellAtRow:(int)row column:(int)col;	// Physical coords

- (void)drawRow:(int)row;				// Physical coords
- (void)drawColumn:(int)col;				// Physical coords
- (void)border:(MiscBorderType)b drawSlot:(int)n;

- (int)numberOfVisibleSlots:(MiscBorderType)b;
- (int)firstVisibleSlot:(MiscBorderType)b;
- (int)lastVisibleSlot:(MiscBorderType)b;
- (BOOL)border:(MiscBorderType)b slotIsVisible:(int)n;
- (void)border:(MiscBorderType)b setFirstVisibleSlot:(int)n;
- (void)border:(MiscBorderType)b setLastVisibleSlot:(int)n;

- (void)scrollCellToVisibleAtRow:(int)row column:(int)col; // Physical coords
- (void)scrollRowToVisible:(int)row;			// Physical coord
- (void)scrollColumnToVisible:(int)col;			// Physical coord
- (void)scrollSelectionToVisible;

- (void)border:(MiscBorderType)b drawSlotTitle:(int)n;
- (void)drawRowTitle:(int)n;
- (void)drawColumnTitle:(int)n;

- (void)sizeToCells;
- (void)sizeToFit;

- (BOOL)drawClippedText;
- (void)setDrawClippedText:(BOOL)x;


// INTERNAL COMMUNICATIONS ON USER-ACTIONS FOR SUBCLASSES ONLY ----------------
// Private: BorderView -> TableScroll
- (void)border:(MiscBorderType)b slotDraggedFrom:(int)fromPos to:(int)toPos;
- (void)border:(MiscBorderType)b slotSortReversed:(int)n;
- (void)border:(MiscBorderType)b slotResized:(int)n;

// OPENSTEP 4.2/NT COMPILER BUG WORK-AROUND; DO NOT USE THESE METHODS ---------
// See implementation for an explanation.
- (id)superInitWithCoder:(NSCoder*)coder;
- (void)superEncodeWithCoder:(NSCoder*)coder;
- (void)superKeyDown:(NSEvent*)p;
- (id)superValidRequestorForSendType:(NSString*)s returnType:(NSString*)r;

@end


// SELECTION ------------------------------------------------------------------
@interface MiscTableScroll(Selection)
- (MiscSelectionMode)selectionMode;
- (void)setSelectionMode:(MiscSelectionMode)x;

- (BOOL)hasSlotSelection:(MiscBorderType)b;
- (BOOL)hasRowSelection;
- (BOOL)hasColumnSelection;
- (BOOL)hasMultipleSlotSelection:(MiscBorderType)b;
- (BOOL)hasMultipleRowSelection;
- (BOOL)hasMultipleColumnSelection;
- (unsigned int)numberOfSelectedSlots:(MiscBorderType)b;
- (unsigned int)numberOfSelectedRows;
- (unsigned int)numberOfSelectedColumns;
- (BOOL)border:(MiscBorderType)b slotIsSelected:(MiscCoord_P)slot;
- (BOOL)rowIsSelected:(MiscCoord_P)row;
- (BOOL)columnIsSelected:(MiscCoord_P)col;
- (BOOL)cellIsSelectedAtRow:(MiscCoord_P)row column:(MiscCoord_P)col;

- (MiscCoord_P)selectedSlot:(MiscBorderType)b;
- (MiscCoord_P)selectedRow;
- (MiscCoord_P)selectedColumn;
- (id)selectedCell;
- (NSArray*)selectedSlotTags:(MiscBorderType)b;
- (NSArray*)selectedRowTags;
- (NSArray*)selectedColumnTags;
- (NSArray*)selectedSlots:(MiscBorderType)b;
- (NSArray*)selectedRows;
- (NSArray*)selectedColumns;

- (void)border:(MiscBorderType)b selectSlot:(MiscCoord_P)slot
	byExtension:(BOOL)flag;
- (void)border:(MiscBorderType)b selectSlot:(MiscCoord_P)slot;
- (void)selectRow:(MiscCoord_P)row byExtension:(BOOL)flag;
- (void)selectRow:(MiscCoord_P)row;
- (void)selectColumn:(MiscCoord_P)col byExtension:(BOOL)flag;
- (void)selectColumn:(MiscCoord_P)col;
- (void)border:(MiscBorderType)b selectSlotTags:(NSArray*)tags
	 byExtension:(BOOL)flag;
- (void)border:(MiscBorderType)b selectSlotTags:(NSArray*)tags;
- (void)selectRowTags:(NSArray*)tags byExtension:(BOOL)flag;
- (void)selectRowTags:(NSArray*)tags;
- (void)selectColumnTags:(NSArray*)tags byExtension:(BOOL)flag;
- (void)selectColumnTags:(NSArray*)tags;
- (void)border:(MiscBorderType)b selectSlots:(NSArray*)slots
	 byExtension:(BOOL)flag;
- (void)border:(MiscBorderType)b selectSlots:(NSArray*)slots;
- (void)selectRows:(NSArray*)rows byExtension:(BOOL)flag;
- (void)selectRows:(NSArray*)rows;
- (void)selectColumns:(NSArray*)cols byExtension:(BOOL)flag;
- (void)selectColumns:(NSArray*)cols;
- (void)selectAllSlots:(MiscBorderType)b;
- (void)selectAllRows;
- (void)selectAllColumns;
- (void)selectAll:(id)sender;	// -selectAllRows and sends action to target.

- (void)border:(MiscBorderType)b deselectSlot:(MiscCoord_P)slot;
- (void)deselectRow:(MiscCoord_P)row;
- (void)deselectColumn:(MiscCoord_P)col;
- (void)border:(MiscBorderType)b deselectSlotTags:(NSArray*)tags;
- (void)deselectRowTags:(NSArray*)tags;
- (void)deselectColumnTags:(NSArray*)tags;
- (void)border:(MiscBorderType)b deselectSlots:(NSArray*)slots;
- (void)deselectRows:(NSArray*)rows;
- (void)deselectColumns:(NSArray*)cols;
- (void)clearSlotSelection:(MiscBorderType)b;
- (void)clearRowSelection;
- (void)clearColumnSelection;
- (void)clearSelection;
- (void)deselectAll:(id)sender;	// -clearSelection and sends action to target.

- (void)selectionChanged;	// Subclasses may want to override.


// MOUSE & KEYBOARD TRACKING (SELECTION ORIENTATION) --------------------------
- (void)trackBy:(MiscBorderType)b;
- (MiscBorderType)trackingBy;

- (BOOL)tracking;
- (MiscCoord_P)clickedSlot:(MiscBorderType)b;
- (MiscCoord_P)clickedRow;
- (MiscCoord_P)clickedColumn;
- (id)clickedCell;


// KEYBOARD CURSOR ------------------------------------------------------------
- (MiscCoord_P)cursorSlot:(MiscBorderType)b;
- (MiscCoord_P)cursorRow;
- (MiscCoord_P)cursorColumn;
- (void)border:(MiscBorderType)b setCursorSlot:(MiscCoord_P)slot;
- (void)setCursorRow:(MiscCoord_P)row;
- (void)setCursorColumn:(MiscCoord_P)col;
- (void)clearCursorSlot:(MiscBorderType)b;
- (void)clearCursorRow;
- (void)clearCursorColumn;
- (void)clearCursor;
- (BOOL)hasValidCursorSlot:(MiscBorderType)b;
- (BOOL)hasValidCursorRow;
- (BOOL)hasValidCursorColumn;
- (void)disableCursor;		// Can nest.
- (void)enableCursor;
- (BOOL)isCursorEnabled;


// ENABLED --------------------------------------------------------------------
- (BOOL)isEnabled;
- (void)setEnabled:(BOOL)flag;
@end


// SORTING --------------------------------------------------------------------
@interface MiscTableScroll(Sort)

- (MiscCompareSlotFunc)compareSlotFunction;
- (void)setCompareSlotFunction:(MiscCompareSlotFunc)f;

- (void)sortInfoInit:(MiscSlotSortInfo*)ip border:(MiscBorderType)b;
- (void)sortInfoDone:(MiscSlotSortInfo*)ip;

// Slots
- (BOOL)autoSortSlots:(MiscBorderType)b;
- (void)border:(MiscBorderType)b setAutoSortSlots:(BOOL)flag;
- (NSArray*)slotSortVector:(MiscBorderType)b;
- (void)border:(MiscBorderType)b setSlotSortVector:(NSArray*)v;

- (void)sortSlots:(MiscBorderType)b;
- (BOOL)slotsAreSorted:(MiscBorderType)b;
- (BOOL)border:(MiscBorderType)b sortSlot:(int)n;
- (BOOL)border:(MiscBorderType)b slotIsSorted:(int)n;

- (int)border:(MiscBorderType)b compareSlots:(int)slot1 :(int)slot2
	info:(MiscSlotSortInfo*)ip;
- (int)border:(MiscBorderType)b compareSlots:(int)slot1 :(int)slot2;


- (MiscCompareEntryFunc)border:(MiscBorderType)b slotSortFunction:(int)n;
- (MiscSortDirection)border:(MiscBorderType)b slotSortDirection:(int)n;
- (MiscSortType)border:(MiscBorderType)b slotSortType:(int)n;
- (void)border:(MiscBorderType)b setSlot:(int)n
		sortFunction:(MiscCompareEntryFunc)x;
- (void)border:(MiscBorderType)b setSlot:(int)n
		sortDirection:(MiscSortDirection)x;
- (void)border:(MiscBorderType)b setSlot:(int)n
		sortType:(MiscSortType)x;

// Cols
- (BOOL)autoSortColumns;
- (void)setAutoSortColumns:(BOOL)flag;
- (NSArray*)columnSortVector;
- (void)setColumnSortVector:(NSArray*)v;

- (void)sortColumns;
- (BOOL)columnsAreSorted;
- (BOOL)sortColumn:(int)n;
- (BOOL)columnIsSorted:(int)n;
- (int)compareColumns:(int)c1 :(int)c2 info:(MiscSlotSortInfo*)ip;
- (int)compareColumns:(int)c1 :(int)c2;

- (MiscCompareEntryFunc)columnSortFunction:(int)n;
- (MiscSortDirection)columnSortDirection:(int)n;
- (MiscSortType)columnSortType:(int)n;
- (void)setColumn:(int)n sortFunction:(MiscCompareEntryFunc)x;
- (void)setColumn:(int)n sortDirection:(MiscSortDirection)x;
- (void)setColumn:(int)n sortType:(MiscSortType)x;

// Rows
- (BOOL)autoSortRows;
- (void)setAutoSortRows:(BOOL)flag;
- (NSArray*)rowSortVector;
- (void)setRowSortVector:(NSArray*)v;

- (void)sortRows;
- (BOOL)rowsAreSorted;
- (BOOL)sortRow:(int)n;
- (BOOL)rowIsSorted:(int)n;
- (int)compareRows:(int)r1 :(int)r2 info:(MiscSlotSortInfo*)ip;
- (int)compareRows:(int)r1 :(int)r2;

- (MiscCompareEntryFunc)rowSortFunction:(int)n;
- (MiscSortDirection)rowSortDirection:(int)n;
- (MiscSortType)rowSortType:(int)n;
- (void)setRow:(int)n sortFunction:(MiscCompareEntryFunc)x;
- (void)setRow:(int)n sortDirection:(MiscSortDirection)x;
- (void)setRow:(int)n sortType:(MiscSortType)x;
@end


// DATA MANIPULATION ----------------------------------------------------------
@interface MiscTableScroll(Data)
- (BOOL)isLazy;
- (void)setLazy:(BOOL)flag;

- (id)cellAtRow:(int)row column:(int)col;

// Lazy mode tables that perform multiple-buffering:
- (int)bufferCount;

// Lazy mode asks delegate/dataDelegate first then uses cellAtRow:column:
// Eager mode goes straight to cellAtRow:column:
- (int)tagAtRow:(int)row column:(int)col;
- (int)intValueAtRow:(int)row column:(int)col;
- (float)floatValueAtRow:(int)row column:(int)col;
- (double)doubleValueAtRow:(int)row column:(int)col;
- (NSString*)stringValueAtRow:(int)row column:(int)col;
// For ButtonCells.
- (NSString*)titleAtRow:(int)row column:(int)col;
- (int)stateAtRow:(int)row column:(int)col;

// Invoke delegate methods if possible.
- (id)reviveCell:(id)cell atRow:(int)row column:(int)col;
- (id)retireCell:(id)cell atRow:(int)row column:(int)col;

// Builtin default implementation.
- (id)doReviveCell:(id)cell atRow:(int)row column:(int)col;
- (id)doRetireCell:(id)cell atRow:(int)row column:(int)col;

- (void)addSlot:(MiscBorderType)b;
- (void)border:(MiscBorderType)b insertSlot:(int)pos;
- (void)border:(MiscBorderType)b removeSlot:(int)pos;
- (int)numberOfSlots:(MiscBorderType)b;

- (void)addColumn;
- (void)insertColumn:(int)pos;
- (void)removeColumn:(int)pos;
- (int)numberOfColumns;

- (void)addRow;
- (void)insertRow:(int)pos;
- (void)removeRow:(int)pos;
- (int)numberOfRows;

- (void)renewRows:(int)count;
- (void)empty;					// [self renewRows:0];
- (void)emptyAndReleaseCells;
@end


// READ / WRITE ---------------------------------------------------------------
@interface MiscTableScroll(IO) <NSCoding>
- (void)awakeFromNib;
- (id)initWithCoder:(NSCoder*)coder;
- (void)encodeWithCoder:(NSCoder*)coder;
@end


// PASTEBOARD SUPPORT ---------------------------------------------------------
@interface MiscTableScroll(Pasteboard)
- (void)copy:(id)sender;
- (void)cut:(id)sender;
- (id)validRequestorForSendType:(NSString*)t_send
	returnType:(NSString*)t_return;
- (id)builtinValidRequestorForSendType:(NSString*)t_send
	returnType:(NSString*)t_return;
- (BOOL)canWritePboardType:(NSString*)type;
- (BOOL)builtinCanWritePboardType:(NSString*)type;
- (BOOL)writeSelectionToPasteboard:(NSPasteboard*)pboard types:(NSArray*)types;
- (BOOL)builtinWriteSelectionToPasteboard:(NSPasteboard*)pboard
	types:(NSArray*)types;
- (NSString*)stringForPboardType:(NSString*)type;
- (NSString*)builtinStringForPboardType:(NSString*)type;
- (NSString*)stringForNSStringPboardType;
- (NSString*)stringForNSTabularTextPBoardType;
- (BOOL)readSelectionFromPasteboard:(NSPasteboard*)pboard;
- (BOOL)builtinReadSelectionFromPasteboard:(NSPasteboard*)pboard;
- (void)builtinRegisterServicesTypes;
- (void)registerServicesTypes;
@end


// INCREMENTAL SEARCH ---------------------------------------------------------
@interface MiscTableScroll(IncrementalSearch)
- (BOOL)incrementalSearch:(NSEvent*)p;
- (BOOL)doIncrementalSearch:(NSEvent*)p column:(int)col;
- (BOOL)getISearchColumn:(int*)col;
- (BOOL)doGetISearchColumn:(int*)col;
@end


// KEYBOARD EVENTS ------------------------------------------------------------
@interface MiscTableScroll(Keyboard)
- (void)keyDown:(NSEvent*)event;
@end


// EDITING --------------------------------------------------------------------
@interface MiscTableScroll(Edit)
- (NSRect)cellFrameAtRow:(int)row column:(int)col;	// Physical coords
- (BOOL)getRow:(int*)row column:(int*)col forPoint:(NSPoint)point; // Physical.
- (BOOL)getPreviousEditRow:(MiscCoord_P*)p_row column:(MiscCoord_P*)p_col;
- (BOOL)getNextEditRow:(MiscCoord_P*)p_row column:(MiscCoord_P*)p_col;
- (BOOL)getNext:(BOOL)foreward
	editRow:(MiscCoord_P*)p_row column:(MiscCoord_P*)p_col;
- (BOOL)isEditing;
- (BOOL)finishEditing;		// Normal, conditional termination.
- (BOOL)abortEditing;		// Forcibly abort the editing session.
- (void)suspendEditing;		// Temporary suspension while resizing.
- (void)resumeEditing;		// Resume editing after resizing.
- (void)textDidEndEditing:(NSNotification*)notification;
- (void)textDidBeginEditing:(NSNotification*)notification;
- (void)textDidChange:(NSNotification*)notification;
- (BOOL)textShouldBeginEditing:(NSText*)sender;
- (BOOL)textShouldEndEditing:(NSText*)sender;
- (void)edit:(NSEvent*)ev atRow:(MiscCoord_P)row column:(MiscCoord_P)col;
- (BOOL)canEdit:(NSEvent*)ev atRow:(MiscCoord_P)row column:(MiscCoord_P)col;
- (BOOL)editIfAble:(NSEvent*)ev atRow:(MiscCoord_P)row column:(MiscCoord_P)col;
- (void)editCellAtRow:(MiscCoord_P)row column:(MiscCoord_P)col;
@end


// PRINTING -------------------------------------------------------------------
@interface MiscTableScroll(Print)
- (void)print:(id)sender;
- (MiscTablePrintInfo const*)getPrintInfo;
- (id)getPageHeader;
- (id)getPageFooter;
- (void)setPageHeader:(id)obj;
- (void)setPageFooter:(id)obj;
@end


// DELEGATE PROTOCOL ----------------------------------------------------------
@interface MiscTableScroll(Delegate)
- (void)tableScroll:(MiscTableScroll*)scroll border:(MiscBorderType)b
	slotDraggedFrom:(int)fromPos to:(int)toPos;
- (void)tableScroll:(MiscTableScroll*)scroll border:(MiscBorderType)b
	slotSortReversed:(int)n;
- (void)tableScroll:(MiscTableScroll*)scroll border:(MiscBorderType)b
	slotResized:(int)n;

- (void)tableScroll:(MiscTableScroll*)scroll
	changeFont:(NSFont*)oldFont to:(NSFont*)newFont;
- (void)tableScroll:(MiscTableScroll*)scroll
	fontChangedFrom:(NSFont*)oldFont to:(NSFont*)newFont;

- (void)tableScroll:(MiscTableScroll*)scroll
	backgroundColorChangedTo:(NSColor*)x;
- (void)tableScroll:(MiscTableScroll*)scroll
	selectedBackgroundColorChangedTo:(NSColor*)x;
- (void)tableScroll:(MiscTableScroll*)scroll
	selectedTextColorChangedTo:(NSColor*)x;
- (void)tableScroll:(MiscTableScroll*)scroll
	textColorChangedTo:(NSColor*)x;

- (BOOL)tableScroll:(MiscTableScroll*)scroll getISearchColumn:(int*)col;
- (int)tableScrollBufferCount:(MiscTableScroll*)scroll;

- (id)tableScroll:(MiscTableScroll*)scroll
	border:(MiscBorderType)b slotPrototype:(int)slot;
- (NSString*)tableScroll:(MiscTableScroll*)scroll
	border:(MiscBorderType)b slotTitle:(int)slot;

- (id)tableScroll:(MiscTableScroll*)scroll cellAtRow:(int)row column:(int)col;
- (id)tableScroll:(MiscTableScroll*)scroll reviveCell:(id)cell
	atRow:(int)row column:(int)col;
- (id)tableScroll:(MiscTableScroll*)scroll retireCell:(id)cell
	atRow:(int)row column:(int)col;

- (int)tableScroll:(MiscTableScroll*)scroll
	tagAtRow:(int)row column:(int)col;
- (int)tableScroll:(MiscTableScroll*)scroll
	intValueAtRow:(int)row column:(int)col;
- (float)tableScroll:(MiscTableScroll*)scroll
	floatValueAtRow:(int)row column:(int)col;
- (double)tableScroll:(MiscTableScroll*)scroll
	doubleValueAtRow:(int)row column:(int)col;
- (NSString*)tableScroll:(MiscTableScroll*)scroll
	stringValueAtRow:(int)row column:(int)col;
- (NSString*)tableScroll:(MiscTableScroll*)scroll
	titleAtRow:(int)row column:(int)col;
- (int)tableScroll:(MiscTableScroll*)scroll
	stateAtRow:(int)row column:(int)col;

- (BOOL)tableScroll:(MiscTableScroll*)scroll
	allowDragOperationAtRow:(int)row column:(int)col;
- (void)tableScroll:(MiscTableScroll*)scroll
	preparePasteboard:(NSPasteboard*)pb
	forDragOperationAtRow:(int)row column:(int)col;
- (NSImage*)tableScroll:(MiscTableScroll*)scroll
	imageForDragOperationAtRow:(int)row column:(int)col;
- (unsigned int)tableScroll:(MiscTableScroll*)scroll
	draggingSourceOperationMaskForLocal:(BOOL)isLocal;
- (BOOL)tableScrollIgnoreModifierKeysWhileDragging:(MiscTableScroll*)scroll;
- (BOOL)tableScroll:(MiscTableScroll*)scroll
	shouldDelayWindowOrderingForEvent:(NSEvent*)event;

- (void)tableScrollRegisterServicesTypes:(MiscTableScroll*)scroll;
- (id)tableScroll:(MiscTableScroll*)scroll
	validRequestorForSendType:(NSString*)t_write
	returnType:(NSString*)t_read;
- (BOOL)tableScroll:(MiscTableScroll*)scroll 
	canWritePboardType:(NSString*)type;
- (NSString*)tableScroll:(MiscTableScroll*)scroll
	stringForPboardType:(NSString*)type;
- (BOOL)tableScroll:(MiscTableScroll*)scroll
	writeSelectionToPasteboard:(NSPasteboard*)pboard types:(NSArray*)types;
- (BOOL)tableScroll:(MiscTableScroll*)scroll
	readSelectionFromPasteboard:(NSPasteboard*)pboard;

- (void)tableScrollWillPrint:(MiscTableScroll*)scroll;
- (void)tableScrollDidPrint:(MiscTableScroll*)scroll;
- (void)tableScroll:(MiscTableScroll*)scroll
	willPrintPageHeader:(id)pageHeader
	info:(MiscTablePrintInfo const*)info;
- (void)tableScroll:(MiscTableScroll*)scroll
	willPrintPageFooter:(id)pageFooter
	info:(MiscTablePrintInfo const*)info;

- (BOOL)tableScroll:(MiscTableScroll*)scroll
	canEdit:(NSEvent*)ev atRow:(int)row column:(int)col;
- (void)tableScroll:(MiscTableScroll*)scroll
	edit:(NSEvent*)ev atRow:(int)row column:(int)col;
- (BOOL)tableScroll:(MiscTableScroll*)scroll
	setStringValue:(NSString*)s atRow:(int)row column:(int)col;
- (void)tableScroll:(MiscTableScroll*)scroll
	abortEditAtRow:(int)row column:(int)col;
- (void)tableScroll:(MiscTableScroll*)scroll
	willEditAtRow:(int)row column:(int)col;
- (void)tableScroll:(MiscTableScroll*)scroll
	didEdit:(BOOL)changed atRow:(int)row column:(int)col;
@end


// DATA CELL PROTOCOL ---------------------------------------------------------
@interface MiscTableScroll(DataCell)
- (id)tableScroll:(MiscTableScroll*)scroll
	reviveAtRow:(int)row column:(int)col;
- (id)tableScroll:(MiscTableScroll*)scroll
	retireAtRow:(int)row column:(int)col;
@end

#endif // __MiscTableScroll_h
