/* InterfaceStyle.m created by stephane on Wed 18-Feb-1998 */

/*	InterfaceStyle.preference is a bundle for Preferences.app, on RDR2.
 *	It adds the possibility to set the default look of the applications
 *	on RDR2 by using the NSInterfaceStyle API
 *
 *	Copyright (C) 1998  Stephane Corthesy
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version 2
 *	of the License, or (at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#import "InterfaceStyle.h"

#define WINDOWS_STYLE	@"windows"
#define NEXTSTEP_STYLE	@"nextstep"
#define MACINTOSH_STYLE	@"macintosh"

#define CURRENT_STYLE_KEY	@"CurrentStyle"
#define INTERFACE_STYLE_KEY	@"NSInterfaceStyleKey"

@interface InterfaceStyle(Private)
- (int) _tagForStyle:(NSString *)aStyle;
- (NSString *) _styleForTag:(int)aTag;
- (void) _userDefaultsDidChange:(NSNotification *)aNotif;
- (void) _showChangeStatus:(BOOL)hasChanges;
@end

@implementation InterfaceStyle

- (id) init
{
    self = [super init];

    if(!self || ![NSBundle loadNibNamed:@"InterfaceStyle" owner:self]){
        NSLog(@"### %s:%d: Unable to load nib named <InterfaceStyle>", __FILE__, __LINE__);
        [self release];
        return nil;
    }
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(_userDefaultsDidChange:) name:NSUserDefaultsDidChangeNotification object:nil];
    return self;
}

- (void) dealloc
{
    [[NSNotificationCenter defaultCenter] removeObserver:self name:NSUserDefaultsDidChangeNotification object:nil];
    [styles release];
    [warningTextField release];
    [warningImageView release];
    [panel release];
    [super dealloc];
}

- (void) awakeFromNib
{
    NSString	*aPath = [[NSBundle bundleForClass:[self class]] pathForResource:@"InterfaceStyles" ofType:@"plist"];

    view = [panel contentView];
    [elementListPopup removeAllItems];
    if(aPath){
        NSArray	*readStyles = [NSArray arrayWithContentsOfFile:aPath];

        if(readStyles){
            NSEnumerator		*styleEnum = [readStyles objectEnumerator];
            NSMutableDictionary	*aStyleDictionary;

            styles = [[NSMutableArray allocWithZone:[self zone]] initWithCapacity:[readStyles count]];
            while(aStyleDictionary = [styleEnum nextObject]){
                [elementListPopup addItemWithTitle:NSLocalizedStringFromTableInBundle([aStyleDictionary objectForKey:@"Title"], nil, [NSBundle bundleForClass:[self class]], nil)];
                [styles addObject:[NSMutableDictionary dictionaryWithDictionary:aStyleDictionary]];
            }
            [elementListPopup selectItemAtIndex:0];
    	}
        else
            NSLog(@"### %s:%d: Unable to create an NSArray from file named <InterfaceStyles.plist>", __FILE__, __LINE__);
    }
    else
        NSLog(@"### %s:%d: Unable to load file named <InterfaceStyles.plist>", __FILE__, __LINE__);
    [[descriptionTextField cell] setWraps:YES];
    // We assume that the first item is the Global Interface Style
    if(NSInterfaceStyleForKey(@"NSInterfaceStyle", styleMatrix) == NSNextStepInterfaceStyle || NSInterfaceStyleForKey(@"NSInterfaceStyle", styleMatrix) == NSWindows95InterfaceStyle){
        NSCell			*aCell;
        NSEnumerator	*cellEnum = [[styleMatrix cells] objectEnumerator];

        while(aCell = [cellEnum nextObject])
            [aCell setShowsFirstResponder:NO];
		// It seems it has no effect; the dotted line is still displayed
    }
    // The following call must be delayed, because it seems there is a bug with
    // the display of the matrix: the matrix displays two selected radio buttons,
    // if the one selected in the nib is not the one selected during run-time!!!
    [self performSelector:@selector(revertStyles:) withObject:nil afterDelay:0.0];
    [[warningTextField cell] setWraps:YES];// Unable to set it in IB...
	[warningTextField retain];
	[warningImageView retain];
}

- (void) didSelect:sender
{
    [super didSelect:sender];
    [[view window] makeKeyAndOrderFront:nil];
}

- (void) didHide:sender
{
    [super didHide:sender];
    [[view window] orderOut:nil];
}

/*
- (void) didUnhide:sender
{
}
*/

- (IBAction) selectElementFromList:(id)sender
{
    int	index = [elementListPopup indexOfSelectedItem];

    if(index >= 0){
        NSDictionary	*styleDict = [styles objectAtIndex:index];
        NSString		*selectedStyle = [styleDict objectForKey:CURRENT_STYLE_KEY];
        
        [styleMatrix setEnabled:YES];
        [styleMatrix selectCellWithTag:[self _tagForStyle:selectedStyle]];
        [descriptionTextField setStringValue:NSLocalizedStringFromTableInBundle([styleDict objectForKey:@"Description"], nil, [NSBundle bundleForClass:[self class]], nil)];
    }
    else{
        [styleMatrix setEnabled:NO];
        [styleMatrix selectCellWithTag:0];
        [descriptionTextField setStringValue:@"Something is rotten in the State of Denmark."];
    }
}

- (IBAction) selectStyle:(id)sender
{
    NSMutableDictionary	*aStyleDictionary = [styles objectAtIndex:[elementListPopup indexOfSelectedItem]];
    NSString			*styleName = [self _styleForTag:[[styleMatrix selectedCell] tag]];

    if(!styleName)
        [aStyleDictionary removeObjectForKey:CURRENT_STYLE_KEY];
    else
        [aStyleDictionary setObject:styleName forKey:CURRENT_STYLE_KEY];
    [self _showChangeStatus:YES];
}

- (IBAction) resetAll:(id)sender
{
    NSEnumerator		*styleEnum = [styles objectEnumerator];
    NSMutableDictionary	*aStyleDictionary;

    while(aStyleDictionary = [styleEnum nextObject])
        [aStyleDictionary removeObjectForKey:CURRENT_STYLE_KEY];
    [self _showChangeStatus:YES];
    [styleMatrix selectCellWithTag:[self _tagForStyle:nil]];
}

- (int) _tagForStyle:(NSString *)aStyle
{
    if(!aStyle)
        return NSNoInterfaceStyle;
    if([aStyle caseInsensitiveCompare:MACINTOSH_STYLE] == NSOrderedSame)
        return NSMacintoshInterfaceStyle;
    if([aStyle caseInsensitiveCompare:NEXTSTEP_STYLE] == NSOrderedSame)
        return NSNextStepInterfaceStyle;
    if([aStyle caseInsensitiveCompare:WINDOWS_STYLE] == NSOrderedSame)
        return NSWindows95InterfaceStyle;
    return NSNoInterfaceStyle;
}

- (NSString *) _styleForTag:(int)aTag
{
    switch(aTag){
        case NSMacintoshInterfaceStyle:
            return MACINTOSH_STYLE;
        case NSNextStepInterfaceStyle:
            return NEXTSTEP_STYLE;
        case NSWindows95InterfaceStyle:
            return WINDOWS_STYLE;
    }
    return nil;
}

- (IBAction) setStyles:(id)sender
{
    NSEnumerator	*styleEnum = [styles objectEnumerator];
    NSDictionary	*aStyleDict;
    NSString		*aStyle;

    while(aStyleDict = [styleEnum nextObject]){
        aStyle = [aStyleDict objectForKey:CURRENT_STYLE_KEY];
        if(aStyle)
            [[NSUserDefaults standardUserDefaults] setObject:aStyle forKey:[aStyleDict objectForKey:INTERFACE_STYLE_KEY] inDomain:NSGlobalDomain];
        else
            [[NSUserDefaults standardUserDefaults] removeObjectForKey:[aStyleDict objectForKey:INTERFACE_STYLE_KEY] inDomain:NSGlobalDomain];
    }
    if([[NSUserDefaults standardUserDefaults] synchronize])
        [self revertStyles:nil];
}

- (IBAction) revertStyles:(id)sender
{
    NSEnumerator		*styleEnum = [styles objectEnumerator];
    NSMutableDictionary	*aStyleDictionary;
    NSString			*aStyle;

    while(aStyleDictionary = [styleEnum nextObject]){
        aStyle = [[NSUserDefaults standardUserDefaults] objectForKey:[aStyleDictionary objectForKey:INTERFACE_STYLE_KEY] inDomain:NSGlobalDomain];
        if(aStyle)
            [aStyleDictionary setObject:aStyle forKey:CURRENT_STYLE_KEY];
        else
            [aStyleDictionary removeObjectForKey:CURRENT_STYLE_KEY];
    }
    [self selectElementFromList:elementListPopup];
    [self _showChangeStatus:NO];
}

- (void) _userDefaultsDidChange:(NSNotification *)aNotif
{
    if(![setButtonCell isEnabled])
        // Do not revert styles if there are local modifications!
    	[self revertStyles:nil];
}

- (void) _showChangeStatus:(BOOL)hasChanges
{
    [setButtonCell setEnabled:hasChanges];
    [revertButtonCell setEnabled:hasChanges];
    if(hasChanges){
        if(![warningTextField superview])
            [[panel contentView] addSubview:warningTextField];
        if(![warningImageView superview])
            [[panel contentView] addSubview:warningImageView];
    }
    else{
        if([warningTextField superview])
            [warningTextField removeFromSuperview];
        if([warningImageView superview])
            [warningImageView removeFromSuperview];
    }
}

@end
