/*
	gifbmap.h

	gifheader.c, gifbmap.c, and gif.h are based on "giftoppm"
	of David Koblas.
	They are modified by T. Ogihara. (1995)
*/

/* +-------------------------------------------------------------------+ */
/* | Copyright 1990, David Koblas.                                     | */
/* |   Permission to use, copy, modify, and distribute this software   | */
/* |   and its documentation for any purpose and without fee is hereby | */
/* |   granted, provided that the above copyright notice appear in all | */
/* |   copies and that both that copyright notice and this permission  | */
/* |   notice appear in supporting documentation.  This software is    | */
/* |   provided "as is" without express or implied warranty.           | */
/* +-------------------------------------------------------------------+ */

#include  <signal.h>
#include  <setjmp.h>
#include  "gif.h"
#include  "strfunc.h"

static int DoExtension(FILE *, gifHeader *);
static int ReadColorMap(FILE *, gifHeader *);
static void write_header(const gifHeader *, FILE *);
static int write_image(const gifHeader *, FILE *, FILE *);

static void (*sav_violation)() = NULL;
static void (*sav_buserror)() = NULL;
static jmp_buf sav_env;

static void
catch_violation(int sig)
{
	longjmp(sav_env, sig);
}

int gif_to_pxo(gifHeader *gh, FILE *fp, FILE *fout)
{
	int	c, rtn = 0;

	if (gh->colormap) {	/* Global Colormap */
		if (ReadColorMap(fp, gh) == 0)
			return Err_FORMAT;
	}

	while ((c = getc(fp)) != ',' ) {	/* start character */
		if (c == EOF || c == ';')	/* GIF terminator */
			return Err_SHORT;
		if (c == '!') { 	/* Extension */
			if (DoExtension(fp, gh) < 0)
				return Err_ILLG;
		}
		/* other chars are illegal... ignore */
	}

	(void)get_long(fp); /* skip 4 bytes */
	gh->width = get_short(fp);
	gh->height = get_short(fp);
	c = getc(fp);
	gh->interlace = BitSet(c, INTERLACE);
	if (BitSet(c, LOCALCOLORMAP)) { /* Use Local Color Map */
		gh->colors = 2 << (c & 0x07);
		if (ReadColorMap(fp, gh) == 0)
			return Err_FORMAT;
	}
	if (gh->transp >= 0) {
		unsigned char *p = (unsigned char *)gh->palette[gh->transp];
		p[RED] = p[GREEN] = p[BLUE] = 255;
		if (gh->colors - 1 == gh->transp)
			--gh->colors;	/* if last of palette is transp. */
	}
	gh->bits = 8;  /* howManyBits(gh->palette, gh->colors); */
	gh->isgray = isGray(gh->palette, gh->colors);
	/* Initialize the Compression routines */
	if (initGifLZW(fp) == EOF)
		return Err_SHORT;
	if (fout == NULL)
		return 0;	/* Get Info. Only */

	sav_violation = signal(SIGSEGV, catch_violation);
	sav_buserror = signal(SIGBUS, catch_violation);
	if (setjmp(sav_env) == 0) {
		write_header(gh, fout);
		rtn = write_image(gh, fp, fout);
	}
	else /* return from Segmentation Error */
		rtn = Err_FORMAT;
	(void) signal(SIGSEGV, sav_violation);
	(void) signal(SIGBUS, sav_buserror);
	return rtn;
}

int ReadColorMap(FILE *fd, gifHeader *gh)
{
	int	i, number;
	unsigned char	*p;

	number = gh->colors;
	for (i = 0; i < number; ++i) {
		p = (unsigned char *)gh->palette[i];
		p[RED] = getc(fd);
		p[GREEN] = getc(fd);
		p[BLUE] = getc(fd);
		if (feof(fd))
			return 0;
	}
#ifdef REDUCE_MAP
	for (i = number - 1; i > 0; --i) {	/* skip stab */
		int	j;
		unsigned char *q = (unsigned char *)gh->palette[i - 1];
		p = (unsigned char *)gh->palette[i];
		for (j = 0; j < 3; j++)
			if (p[j] != q[j]) goto EXIT;
	}
EXIT:
	gh->colors = i + 1;
#endif
	return 1;
}

static int
DoExtension(FILE *fd, gifHeader *gh)
{
	int cc;
	unsigned char	buf[256];

	switch (cc = getc(fd)) {
	case 0x01:		/* Plain Text Extension */
	case 0x2c:		/* Image Descriptor */
	case 0x3b:		/* Trailer */
	case 0xff:		/* Application Extension */
		break;
	case 0xf9:		/* Graphic Control Extension */
		while (GetDataBlock(fd, buf) != 0) {
		    if (BitSet(buf[0], TRANSPARENCY))
			gh->transp = buf[3] & 0xff;	/* Transparent Index */
		}
		return cc;
	case 0xfe:		/* Comment Extension */
		while (GetDataBlock(fd, buf) != 0) {
		    if (gh->memo == NULL) {
			unsigned char *p;
			int	i;
			gh->memo = p = (unsigned char *)str_dup(buf);
			for (i = 0; *p; p++, i++) {
				if (*p < ' ') *p = ' ';
				if (i >= MAX_COMMENT-1) {
					*p = 0;
					break;
				}
			}
		    }
		}
		return cc;
	default:
		return -1;	/* ERROR */
	}
	while (GetDataBlock(fd, buf) != 0)
		;
	return cc;
}


static void
write_header(const gifHeader *gh, FILE *fout)
{
	int	i, j;
	int	memof;
	unsigned char *p;

/* PX original Format
    PX			: Header
    Width Height Colors	: Colors := Number of colors in palette - 1
    Count		: if Trans is there Count=1, otherwise Count=0.
    Trans		: Transparent + 256
    [Palette]		: Binary
    [Bitmap]		: Binary
*/
	fprintf(fout, "PX\n");
	memof = (gh->memo && gh->memo[0]);
	if (memof || gh->interlace) {
		putc('#', fout);
		if (gh->interlace)
			fprintf(fout, " Interlace");
		if (memof)
			fprintf(fout, " : %s", gh->memo);
		putc('\n', fout);
	}
	fprintf(fout, "%d %d %d ", gh->width, gh->height, gh->colors - 1);
	if (gh->transp >= 0)
		fprintf(fout, "1 %d\n", gh->transp + 256);
	else
		fprintf(fout, "0\n");
	for (i = 0; i < gh->colors; i++) {	/* Palette */
		p = (unsigned char *)gh->palette[i];
		for (j = 0; j < 3; j++)
			putc(p[j], fout);
	}
}

static int
write_image(const gifHeader *gh, FILE *fd, FILE *fout)
{
	int	i, xpos, ypos, pass;
	long	total;
	unsigned char	*plane, *map;

	if (gh->interlace) {
		total = gh->width * gh->height;
		if ((plane = (unsigned char *)malloc(total)) == NULL)
			return Err_MEMORY;
		pass = 0;
		for (ypos = 0; ; ) {
			map = plane + gh->width * ypos;
			for (xpos = 0; xpos < gh->width; xpos++)
				map[xpos] = LWZReadByte(fd);
			ypos += pass ? (0x10 >> pass) : 0x08;
			if (ypos >= gh->height) {
				if (++pass > 3)
					break;
				ypos = 8 >> pass;
			}
		}
		for (i = 0; i < total; i++)
			fputc(plane[i], fout);
		(void)free((void *)plane);
	}else {
		for (ypos = 0; ypos < gh->height; ypos++)
			for (xpos = 0; xpos < gh->width; xpos++)
				fputc(LWZReadByte(fd), fout);
	}
	return 0;
}
