
/* Copyright (c) Mark J. Kilgard, 1997. */
/* Copyright (c) Dietmar Planitzer, 1998 */

/* This program is freely distributable without licensing fees 
   and is provided without guarantee or warrantee expressed or 
   implied. This program is -not- in the public domain. */

#import "glut.h"
#import "macxglut_private.h"
#import "GLUTVisual.h"
#import "GLUTConixVisual.h"
#import "GLUTMesaVisual.h"







#if defined(VERBOSE1)

static char *compstr[] =
{
	"none", "=", "!=", "<=", ">=", ">", "<", "~"
};
static char *capstr[] =
{
	"rgba", "bufsize", "double", "stereo", "auxbufs", "red", "green", "blue", "alpha",
	"depth", "stencil", "acred", "acgreen", "acblue", "acalpha", "level", "xvisual",
	"transparent", "samples", "xstaticgray", "xgrayscale", "xstaticcolor", "xpseudocolor",
	"xtruecolor", "xdirectcolor", "slow", "conformant", "num"
};

#endif



static const GLUTVisualInfo *	_findMatchFromCriteria(const GLUTVisualInfo *fbmodes, int nfbmodes, GLUTCriterion *criteria, int ncriteria);
static int							_parseCriteria(char *word, GLUTCriterion * criterion, int *mask, BOOL * allowDoubleAsSingle);
static GLUTCriterion *			_parseModeString(char *mode, int *ncriteria, BOOL * allowDoubleAsSingle, GLUTCriterion * requiredCriteria, int nRequired, int requiredMask);
#if defined(VERBOSE1)
static void					_printCriteria(GLUTCriterion * criteria, int ncriteria);
#endif



@interface GLUTVisual(GLUTPrivate)

+ (void)initialize;
+ (id)allocWithZone: (NSZone *)zone;

@end





	/* *** GLUTVisual class implementation *** */


@implementation GLUTVisual


static const GLUTVisualInfo *		__glutVisualInfos = NULL;
static int								__glutVisualInfoCount = 0;
static Class							__glutConcreteVisualClass = Nil;



+ (BOOL)canCreateInstanceWithString: (const char *)dstring
{
	const GLUTVisualInfo *	vi = NULL;
	BOOL							treatAsSingle = NO;
	
//	vi = [self	visualInfoFromString: dstring
	vi = [__glutConcreteVisualClass visualInfoFromString: dstring
					treatAsSingle: &treatAsSingle
					requiredCriteria: NULL
					count: 0
					requiredCriteriaMask: 0];
		
	return (vi != NULL) ? YES : NO;
}

+ (BOOL)canCreateInstanceWithString: (const char *)dstring requiredCriteria: (const GLUTCriterion *)requiredCriteria count: (int)nRequired requiredCriteriaMask: (int)requiredMask
{
	const GLUTVisualInfo *	vi = NULL;
	BOOL							treatAsSingle = NO;
	
//	vi = [self	visualInfoFromString: dstring
	vi = [__glutConcreteVisualClass visualInfoFromString: dstring
					treatAsSingle: &treatAsSingle
					requiredCriteria: requiredCriteria
					count: nRequired
					requiredCriteriaMask: requiredMask];
		
	return (vi != NULL) ? YES : NO;
}

	/* Returns a visual info that matches the given display string. NULL if no matching visual info was found. */
+ (const GLUTVisualInfo *)visualInfoFromString: (const char *)string treatAsSingle: (BOOL *)yesno requiredCriteria: (const GLUTCriterion *)requiredCriteria count: (int)nRequired requiredCriteriaMask: (int)requiredMask
{
	GLUTCriterion *			criteria;
	const GLUTVisualInfo *	visinfo = NULL;
	BOOL							allowDoubleAsSingle;
	int							ncriteria, i;
	
	if(string == NULL) return NULL;
	
//	if(__glutVisualInfos == NULL)
//		__glutVisualInfos = [self makeVisualInfos: &__glutVisualInfoCount];
	if(__glutVisualInfos == NULL)
		__glutVisualInfos = [__glutConcreteVisualClass makeVisualInfos: &__glutVisualInfoCount];

	criteria = _parseModeString((char *) string, &ncriteria, &allowDoubleAsSingle, (GLUTCriterion *) requiredCriteria, nRequired, requiredMask);
	if(criteria == NULL)
	{
		__glutWarning("failed to parse mode string");
		return NULL;
	}
	
#if defined(VERBOSE1)
	_printCriteria(criteria, ncriteria);
#endif
	
	visinfo = _findMatchFromCriteria(__glutVisualInfos, __glutVisualInfoCount, criteria, ncriteria);
	if(visinfo)
		*yesno = NO;
	else
	{
		if(allowDoubleAsSingle)
		{
			/* Rewrite criteria so that we now look for a double
				buffered visual which will then get treated as a
				single buffered visual. */
			for(i = 0; i < ncriteria; i++)
			{
				if(criteria[i].capability == DS_DOUBLEBUFFER &&
					criteria[i].comparison == DS_EQ &&
					criteria[i].value == 0)
				{
					criteria[i].value = 1;
				}
			}
			visinfo = _findMatchFromCriteria(__glutVisualInfos, __glutVisualInfoCount, criteria, ncriteria);
			if(visinfo)
				*yesno = YES;
		}
	}
	free(criteria);
	
	return visinfo;
}

	/* Return an array of frame buffer mode structures, where each one describes one of the supported
		frame buffer configurations, possible with the receiving visual. */
+ (const GLUTVisualInfo *)makeVisualInfos: (int *)nummodes
{
	[NSException raise: NSInternalInconsistencyException format: @""];
	return NULL;
}

+ (id)visualWithString: (const char *)displayString pixelsWide: (int)width pixelsHigh: (int)height requiredCriteria: (const GLUTCriterion *)crits count: (unsigned int)critscount requiredCriteriaMask: (int)mask
{
//	return [[[self alloc]	initWithString: displayString
	return [[[__glutConcreteVisualClass alloc] initWithString: displayString
									pixelsWide: width
									pixelsHigh: height
									requiredCriteria: crits
									count: critscount
									requiredCriteriaMask: mask] autorelease];
}





- (id)initWithString: (const char *)displayString pixelsWide: (int)width pixelsHigh: (int)height requiredCriteria: (const GLUTCriterion *)crits count: (unsigned int)critscount requiredCriteriaMask: (int)mask
{
//	[self release];
//	self = [__glutConcreteVisualClass alloc];
	if((self = [super init]) != nil)
	{
		_width	= width;
		_height	= height;
		
		return self;
	}
	return nil;
}

	/* Returns the receiver's visual info structure. */
- (const GLUTVisualInfo *)visualInfo
{
	return _visualInfo;
}

	/* Returns an ID for the current visual info (corresponds to the visual info index in the visual infos array). */
- (int)visualInfoID
{
	return (((char *) _visualInfo) - ((char *) __glutVisualInfos)) / sizeof(GLUTVisualInfo);
}

	/* Sets the given visual info as the current visual info. It is assumed that the given visual info is
		one of the visual infos supported by the receiver. */
- (void)setVisualInfo: (const GLUTVisualInfo *)info
{
	_visualInfo = info;
}

	/* Returns the current size of the overlay plane in pixels */
- (NSSize)size
{
	return NSMakeSize((float) _width, (float) _height);
}

	/* Sets the overlay plane to the given size in pixels */
- (void)setSize: (NSSize)newSize
{
	_width	= (int) newSize.width;
	_height	= (int) newSize.height;
}

- (int)pixelsWide
{
	return _width;
}

- (int)pixelsHigh
{
	return _height;
}

- (BOOL)isDoubleBuffered
{
	return (BOOL) _visualInfo->cap[DS_DOUBLEBUFFER];
}

	/* Returns YES if the visual is in reality a double buffered visual, but should be treated as a single buffered one. */
- (BOOL)treatAsSingle
{
	return NO;
}

	/* Makes the receiver the current OpenGL context in town. */
- (void)makeCurrent
{
	[NSException raise: NSInternalInconsistencyException format: @""];
}

	/* Makes the receiver not the current GL context. */
- (void)makeNotCurrent
{
	[NSException raise: NSInternalInconsistencyException format: @""];
}

	/* Swaps the contents of the back buffer to the front buffer */
- (void)swapBuffers
{
	[NSException raise: NSInternalInconsistencyException format: @""];
}


/* *** private functions *** */


static const GLUTVisualInfo *_findMatchFromCriteria(const GLUTVisualInfo *fbmodes, int nfbmodes, GLUTCriterion *criteria, int ncriteria)
{
	const GLUTVisualInfo *	found;
	int *							bestScore, *thisScore;
	int							i, j, numok, result = 0, worse, better, foundIndex = 0;
	
	found = NULL;
	numok = 1;            /* "num" capability is indexed from 1,
                           not 0. */
	
		/* XXX alloca canidate. */
	bestScore = (int *) malloc(ncriteria * sizeof(int));
	if(!bestScore)
		__glutFatalError("out of memory.");

	for(j = 0; j < ncriteria; j++)
	{
			/* Very negative number. */
		bestScore[j] = -32768;
	}
	
		/* XXX alloca canidate. */
	thisScore = (int *) malloc(ncriteria * sizeof(int));
	if(!thisScore)
		__glutFatalError("out of memory.");
	
	for(i = 0; i < nfbmodes; i++)
	{
		if(fbmodes[i].valid)
		{		
			worse = 0;
			better = 0;
			
#if defined(VERBOSE1)
			NSLog(@"   visual info: %d\n", i);
#endif

			for(j = 0; j < ncriteria; j++)
			{
				int	cap, cvalue, fbvalue;
				
				cap = criteria[j].capability;
				cvalue = criteria[j].value;
				if(cap == DS_NUM)
					fbvalue = numok;
				else
					fbvalue = fbmodes[i].cap[cap];
#if defined(VERBOSE1)
				NSLog(@"  %s %s %d to %d\n", capstr[cap], compstr[criteria[j].comparison], cvalue, fbvalue);
#endif
				switch(criteria[j].comparison)
				{
					case DS_EQ:
								result = cvalue == fbvalue;
								thisScore[j] = 1;
								break;
					case DS_NEQ:
								result = cvalue != fbvalue;
								thisScore[j] = 1;
								break;
					case DS_LT:
								result = fbvalue < cvalue;
								thisScore[j] = fbvalue - cvalue;
								break;
					case DS_GT:
								result = fbvalue > cvalue;
								thisScore[j] = fbvalue - cvalue;
								break;
					case DS_LTE:
								result = fbvalue <= cvalue;
								thisScore[j] = fbvalue - cvalue;
								break;
					case DS_GTE:
								result = (fbvalue >= cvalue);
								thisScore[j] = fbvalue - cvalue;
								break;
					case DS_MIN:
								result = fbvalue >= cvalue;
								thisScore[j] = cvalue - fbvalue;
								break;
				}

#if defined(VERBOSE1)
				NSLog(@"                result=%d   score=%d   bestScore=%d\n", result, thisScore[j], bestScore[j]);
#endif
				
				if(result)
				{
					if(better || thisScore[j] > bestScore[j])
						better = 1;
					else if(thisScore[j] == bestScore[j])
						;/* Keep looking. */
					else
						goto nextFBM;
				}
				else
				{
					if(cap == DS_NUM)
						worse = 1;
					else
						goto nextFBM;
				}
			}
			
			if(better && !worse)
			{
				found = &fbmodes[i];
				foundIndex = i;
				for(j = 0; j < ncriteria; j++)
					bestScore[j] = thisScore[j];
			}
			numok++;
			
	nextFBM:;
		
		}
	}
	free(bestScore);
	free(thisScore);
	
#if defined(VERBOSE1)
				if(found)
					NSLog(@"\n   selected visual info: %d\n", foundIndex);
				else
					NSLog(@"\n   no matching visual info found.\n");
#endif
	
	return found;
}


static int _parseCriteria(char *word, GLUTCriterion * criterion, int *mask, BOOL * allowDoubleAsSingle)
{
	char *	cstr, *vstr, *response;
	int		comparator, value = 0;
	int		rgb, rgba, acc, acca, count, i;
	
	cstr = strpbrk(word, "=><!~");
	if(cstr)
	{
		switch(cstr[0])
		{
			case '=':
				comparator = DS_EQ;
				vstr = &cstr[1];
				break;
			case '~':
				comparator = DS_MIN;
				vstr = &cstr[1];
				break;
			case '>':
				if(cstr[1] == '=')
				{
					comparator = DS_GTE;
					vstr = &cstr[2];
				}
				else
				{
					comparator = DS_GT;
					vstr = &cstr[1];
				}
				break;
			case '<':
				if(cstr[1] == '=')
				{
					comparator = DS_LTE;
					vstr = &cstr[2];
				}
				else
				{
					comparator = DS_LT;
					vstr = &cstr[1];
				}
				break;
			case '!':
				if(cstr[1] == '=')
				{
					comparator = DS_NEQ;
					vstr = &cstr[2];
				}
				else
					return -1;
				break;
			default:
				return -1;
		}
		value = (int) strtol(vstr, &response, 0);
		if(response == vstr)
		{
				/* Not a valid number. */
			return -1;
		}
		*cstr = '\0';
	}
	else
	{
		comparator = DS_NONE;
	}
	
	switch(word[0])
	{
		case 'a':
			if(!strcmp(word, "alpha"))
			{
				criterion[0].capability = DS_ALPHA_SIZE;
				if(comparator == DS_NONE)
				{
					criterion[0].comparison = DS_GTE;
					criterion[0].value = 1;
				}
				else
				{
					criterion[0].comparison = comparator;
					criterion[0].value = value;
				}
				*mask |= (1 << DS_RGBA);
				*mask |= (1 << DS_ALPHA_SIZE);
				*mask |= (1 << DS_RGBA_MODE);
				return 1;
			}
			acca = !strcmp(word, "acca");
			acc = !strcmp(word, "acc");
			if(acc || acca)
			{
				criterion[0].capability = DS_ACCUM_RED_SIZE;
				criterion[1].capability = DS_ACCUM_GREEN_SIZE;
				criterion[2].capability = DS_ACCUM_BLUE_SIZE;
				criterion[3].capability = DS_ACCUM_ALPHA_SIZE;
				if(acca)
					count = 4;
				else
				{
					count = 3;
					criterion[3].comparison = DS_MIN;
					criterion[3].value = 0;
				}
				if(comparator == DS_NONE)
				{
					comparator = DS_GTE;
					value = 8;
				}
				for(i = 0; i < count; i++)
				{
					criterion[i].comparison = comparator;
					criterion[i].value = value;
				}
				*mask |= (1 << DS_ACCUM_RED_SIZE);
				return 4;
			}
			if(!strcmp(word, "auxbufs"))
			{
				criterion[0].capability = DS_AUX_BUFFERS;
				if(comparator == DS_NONE)
				{
					criterion[0].comparison = DS_MIN;
					criterion[0].value = 1;
				}
				else
				{
					criterion[0].comparison = comparator;
					criterion[0].value = value;
				}
				*mask |= (1 << DS_AUX_BUFFERS);
				return 1;
			}
			return -1;
			
		case 'b':
			if(!strcmp(word, "blue"))
			{
				criterion[0].capability = DS_BLUE_SIZE;
				if(comparator == DS_NONE)
				{
					criterion[0].comparison = DS_GTE;
					criterion[0].value = 1;
				}
				else
				{
					criterion[0].comparison = comparator;
					criterion[0].value = value;
				}
				*mask |= (1 << DS_RGBA);
				*mask |= (1 << DS_RGBA_MODE);
				return 1;
			}
			if(!strcmp(word, "buffer"))
			{
				criterion[0].capability = DS_BUFFER_SIZE;
				if(comparator == DS_NONE)
				{
					criterion[0].comparison = DS_GTE;
					criterion[0].value = 1;
				}
				else
				{
					criterion[0].comparison = comparator;
					criterion[0].value = value;
				}
				return 1;
			}
			return -1;

		case 'c':
			if(!strcmp(word, "conformant"))
			{
				criterion[0].capability = DS_CONFORMANT;
				if(comparator == DS_NONE)
				{
					criterion[0].comparison = DS_EQ;
					criterion[0].value = 1;
				}
				else
				{
					criterion[0].comparison = comparator;
					criterion[0].value = value;
				}
				*mask |= (1 << DS_CONFORMANT);
				return 1;
			}
			return -1;

		case 'd':
			if(!strcmp(word, "depth"))
			{
				criterion[0].capability = DS_DEPTH_SIZE;
				if(comparator == DS_NONE)
				{
					criterion[0].comparison = DS_GTE;
					criterion[0].value = 12;
				}
				else
				{
					criterion[0].comparison = comparator;
					criterion[0].value = value;
				}
				*mask |= (1 << DS_DEPTH_SIZE);
				return 1;
			}
			if(!strcmp(word, "double"))
			{
				criterion[0].capability = DS_DOUBLEBUFFER;
				if(comparator == DS_NONE)
				{
					criterion[0].comparison = DS_EQ;
					criterion[0].value = 1;
				}
				else
				{
					criterion[0].comparison = comparator;
					criterion[0].value = value;
				}
				*mask |= (1 << DS_DOUBLEBUFFER);
				return 1;
			}
			return -1;

		case 'g':
			if(!strcmp(word, "green"))
			{
				criterion[0].capability = DS_GREEN_SIZE;
				if(comparator == DS_NONE)
				{
					criterion[0].comparison = DS_GTE;
					criterion[0].value = 1;
				}
				else
				{
					criterion[0].comparison = comparator;
					criterion[0].value = value;
				}
				*mask |= (1 << DS_RGBA);
				*mask |= (1 << DS_RGBA_MODE);
				return 1;
			}
			return -1;
		case 'i':
			if(!strcmp(word, "index"))
			{
				criterion[0].capability = DS_RGBA;
				criterion[0].comparison = DS_EQ;
				criterion[0].value = 0;
				*mask |= (1 << DS_RGBA);
				*mask |= (1 << DS_CI_MODE);
				criterion[1].capability = DS_BUFFER_SIZE;
				if(comparator == DS_NONE)
				{
					criterion[1].comparison = DS_GTE;
					criterion[1].value = 1;
				}
				else
				{
					criterion[1].comparison = comparator;
					criterion[1].value = value;
				}
				return 2;
			}
			return -1;
		case 'l':
			if(!strcmp(word, "luminance"))
			{
				criterion[0].capability = DS_RGBA;
				criterion[0].comparison = DS_EQ;	
				criterion[0].value = 1;
				criterion[1].capability = DS_RED_SIZE;
				if(comparator == DS_NONE)
				{
					criterion[1].comparison = DS_GTE;
					criterion[1].value = 1;
				}
				else
				{
					criterion[1].comparison = comparator;
					criterion[1].value = value;
				}
				criterion[2].capability = DS_GREEN_SIZE;
				criterion[2].comparison = DS_EQ;
				criterion[2].value = 0;
				criterion[3].capability = DS_BLUE_SIZE;
				criterion[3].comparison = DS_EQ;
				criterion[3].value = 0;
				*mask |= (1 << DS_RGBA);
				*mask |= (1 << DS_RGBA_MODE);
				*mask |= (1 << DS_LUMINANCE_MODE);
				return 4;
			}
			return -1;

		case 'n':
			if(!strcmp(word, "num"))
			{
				criterion[0].capability = DS_NUM;
				if(comparator == DS_NONE)
					return -1;
				else
				{
					criterion[0].comparison = comparator;
 					criterion[0].value = value;
					return 1;
				}
			}
 			return -1;

		case 'r':
			if(!strcmp(word, "red"))
			{
				criterion[0].capability = DS_RED_SIZE;
				if(comparator == DS_NONE)
				{
					criterion[0].comparison = DS_GTE;
					criterion[0].value = 1;
				}
				else
				{
					criterion[0].comparison = comparator;
					criterion[0].value = value;
				}
				*mask |= (1 << DS_RGBA);
				*mask |= (1 << DS_RGBA_MODE);
				return 1;
			}
			rgba = !strcmp(word, "rgba");
			rgb = !strcmp(word, "rgb");
			if(rgb || rgba)
			{
				criterion[0].capability = DS_RGBA;
				criterion[0].comparison = DS_EQ;
 				criterion[0].value = 1;
				criterion[1].capability = DS_RED_SIZE;
				criterion[2].capability = DS_GREEN_SIZE;
				criterion[3].capability = DS_BLUE_SIZE;
				criterion[4].capability = DS_ALPHA_SIZE;
				if(rgba)
					count = 5;
				else
				{
					count = 4;
					criterion[4].comparison = DS_MIN;
					criterion[4].value = 0;
				}
				if(comparator == DS_NONE)
				{
					comparator = DS_GTE;
					value = 1;
				}
				for(i = 1; i < count; i++)
				{
					criterion[i].comparison = comparator;
					criterion[i].value = value;
				}
				*mask |= (1 << DS_RGBA);
				*mask |= (1 << DS_RGBA_MODE);
				return 5;
			}
			return -1;

		case 's':
			if(!strcmp(word, "stencil"))
			{
				criterion[0].capability = DS_STENCIL_SIZE;
				if(comparator == DS_NONE)
				{
					criterion[0].comparison = DS_MIN;
					criterion[0].value = 1;
				}
				else
				{
					criterion[0].comparison = comparator;
					criterion[0].value = value;
				}
				*mask |= (1 << DS_STENCIL_SIZE);
				return 1;
			}
			if(!strcmp(word, "single"))
			{
				criterion[0].capability = DS_DOUBLEBUFFER;
				if(comparator == DS_NONE)
				{
					criterion[0].comparison = DS_EQ;
					criterion[0].value = 0;
					*allowDoubleAsSingle = YES;
					*mask |= (1 << DS_DOUBLEBUFFER);
					return 1;
				}
				else
				return -1;
			}
			if(!strcmp(word, "stereo"))
			{
				criterion[0].capability = DS_STEREO;
				if(comparator == DS_NONE)
				{
					criterion[0].comparison = DS_EQ;
					criterion[0].value = 1;
				}
				else
				{
					criterion[0].comparison = comparator;
					criterion[0].value = value;
				}
				*mask |= (1 << DS_STEREO);
				return 1;
			}
			if(!strcmp(word, "samples"))
			{
				criterion[0].capability = DS_SAMPLES;
				if(comparator == DS_NONE)
				{
					criterion[0].comparison = DS_LTE;
					criterion[0].value = 4;
				}
				else
				{
					criterion[0].comparison = comparator;
					criterion[0].value = value;
				}
				*mask |= (1 << DS_SAMPLES);
				return 1;
			}
			if(!strcmp(word, "slow"))
			{
				criterion[0].capability = DS_SLOW;
				if(comparator == DS_NONE)
				{
						/* Just "slow" means permit fast visuals, but accept
							slow ones in preference. Presumably the slow ones
							must be higher quality or something else desirable. */
					criterion[0].comparison = DS_GTE;
					criterion[0].value = 0;
				}
				else
				{
					criterion[0].comparison = comparator;
					criterion[0].value = value;
				}
 				*mask |= (1 << DS_SLOW);
				return 1;
			}
			return -1;
#if defined(_WIN32)
		case 'w':
			if(!strcmp(word, "win32pfd"))
			{
				criterion[0].capability = DS_XVISUAL;
				if(comparator == DS_NONE)
					return -1;
				else
				{
					criterion[0].comparison = comparator;
					criterion[0].value = value;
					return 1;
				}
			}
			return -1;
#endif
#if !defined(_WIN32)
		case 'x':
			if(!strcmp(word, "xvisual"))
			{
				if(comparator == DS_NONE)
					return -1;
				else
				{
					criterion[0].capability = DS_XVISUAL;
					criterion[0].comparison = comparator;
					criterion[0].value = value;
					/* Set everything in "mask" so that no default criteria
						get used.  Assume the program really wants the
 						xvisual specified. */
					*mask |= ~0;
					return 1;
				}
			}
			/* Be a little over-eager to fill in the comparison and
				value so we won't have to replicate the code after each
				string match. */
			if(comparator == DS_NONE)
			{
				criterion[0].comparison = DS_EQ;
				criterion[0].value = 1;
			}
			else
			{
				criterion[0].comparison = comparator;
				criterion[0].value = value;
			}
			
			if(!strcmp(word, "xstaticgray"))
			{
				criterion[0].capability = DS_XSTATICGRAY;
				*mask |= (1 << DS_XSTATICGRAY);  /* Indicates _any_ visual class selected. */
				return 1;
			}
			if(!strcmp(word, "xgrayscale"))
			{
				criterion[0].capability = DS_XGRAYSCALE;
				*mask |= (1 << DS_XSTATICGRAY);  /* Indicates _any_ visual class selected. */
				return 1;
			}
			if(!strcmp(word, "xstaticcolor"))
			{
				criterion[0].capability = DS_XSTATICCOLOR;
				*mask |= (1 << DS_XSTATICGRAY);  /* Indicates _any_ visual class selected. */
				return 1;
			}
			if(!strcmp(word, "xpseudocolor"))
			{
				criterion[0].capability = DS_XPSEUDOCOLOR;
				*mask |= (1 << DS_XSTATICGRAY);  /* Indicates _any_ visual class selected. */
				return 1;
			}
			if(!strcmp(word, "xtruecolor"))
			{
				criterion[0].capability = DS_XTRUECOLOR;
				*mask |= (1 << DS_XSTATICGRAY);  /* Indicates _any_ visual class selected. */
				return 1;
			}
			if(!strcmp(word, "xdirectcolor"))
			{
				criterion[0].capability = DS_XDIRECTCOLOR;
				*mask |= (1 << DS_XSTATICGRAY);  /* Indicates _any_ visual class selected. */
				return 1;
			}
			return -1;
#endif
		default:
			return -1;
	}
}

static GLUTCriterion *_parseModeString(char *mode, int *ncriteria, BOOL * allowDoubleAsSingle, GLUTCriterion * requiredCriteria, int nRequired, int requiredMask)
{
	GLUTCriterion *	criteria = NULL;
	int					n, mask, parsed, i;
	char *				copy, *word;
	
	*allowDoubleAsSingle = NO;
	copy = __glutStrdup(mode);
		/* Attempt to estimate how many criteria entries should be needed. */
	n = 0;
	word = strtok(copy, " \t");
	while(word)
	{
		n++;
		word = strtok(NULL, " \t");	
	}
		/* Overestimate by 4 times ("rgba" might add four criteria
			entries) plus add in possible defaults plus space for
			required criteria. */
	criteria = (GLUTCriterion *) malloc((4 * n + 30 + nRequired) * sizeof(GLUTCriterion));
	if(!criteria)
		__glutFatalError("out of memory.");
	
		/* Re-copy the copy of the mode string. */
	strcpy(copy, mode);
	
		/* First add the required criteria (these match at the
			highest priority). Typically these will be used to force a
			specific level (layer), transparency, and/or visual type. */
	mask = requiredMask;
	for(i = 0; i < nRequired; i++)
		criteria[i] = requiredCriteria[i];
	n = nRequired;
	
	word = strtok(copy, " \t");
	while(word)
	{
		parsed = _parseCriteria(word, &criteria[n], &mask, allowDoubleAsSingle);
		if(parsed >= 0)
			n += parsed;
		else
			__glutWarning("Unrecognized display string word: %s (ignoring)\n", word);
		word = strtok(NULL, " \t");
	}
	
	if(!(mask & (1 << DS_ACCUM_RED_SIZE)))
	{
		criteria[n].capability = DS_ACCUM_RED_SIZE;
		criteria[n].comparison = DS_MIN;
		criteria[n].value = 0;
		criteria[n + 1].capability = DS_ACCUM_GREEN_SIZE;
		criteria[n + 1].comparison = DS_MIN;
		criteria[n + 1].value = 0;
		criteria[n + 2].capability = DS_ACCUM_BLUE_SIZE;
		criteria[n + 2].comparison = DS_MIN;	
		criteria[n + 2].value = 0;
		criteria[n + 3].capability = DS_ACCUM_ALPHA_SIZE;
		criteria[n + 3].comparison = DS_MIN;
		criteria[n + 3].value = 0;
		n += 4;
	}
	if(!(mask & (1 << DS_AUX_BUFFERS)))
	{
		criteria[n].capability = DS_AUX_BUFFERS;
		criteria[n].comparison = DS_MIN;
		criteria[n].value = 0;
		n++;
	}
	if(!(mask & (1 << DS_RGBA)))
	{
		criteria[n].capability = DS_RGBA;
		criteria[n].comparison = DS_EQ;
		criteria[n].value = 1;
		criteria[n + 1].capability = DS_RED_SIZE;
		criteria[n + 1].comparison = DS_GTE;
		criteria[n + 1].value = 1;
		criteria[n + 2].capability = DS_GREEN_SIZE;
		criteria[n + 2].comparison = DS_GTE;
		criteria[n + 2].value = 1;
		criteria[n + 3].capability = DS_BLUE_SIZE;
		criteria[n + 3].comparison = DS_GTE;
		criteria[n + 3].value = 1;
		criteria[n + 4].capability = DS_ALPHA_SIZE;
		criteria[n + 4].comparison = DS_MIN;
		criteria[n + 4].value = 0;
		n += 5;
		mask |= (1 << DS_RGBA_MODE);
	}
#if !defined(_WIN32)
	if(!(mask & (1 << DS_XSTATICGRAY)))
	{
	//	assert(isMesaGLX != -1);
		if((mask & (1 << DS_RGBA_MODE))/* && !isMesaGLX*/)
		{
			/* Normally, request an RGBA mode visual be TrueColor,
				except in the case of Mesa where we trust Mesa (and
				other code in GLUT) to handle any type of RGBA visual
				reasonably. */
			if(mask & (1 << DS_LUMINANCE_MODE))
			{
				/* If RGBA luminance was requested, actually go for a StaticGray visual. */
				criteria[n].capability = DS_XSTATICGRAY;
			}
			else
				criteria[n].capability = DS_XTRUECOLOR;
			criteria[n].value = 1;
			criteria[n].comparison = DS_EQ;
			n++;
		}
		if(mask & (1 << DS_CI_MODE))
		{
			criteria[n].capability = DS_XPSEUDOCOLOR;
			criteria[n].value = 1;
			criteria[n].comparison = DS_EQ;
			n++;
		}
	}
#endif
	if(!(mask & (1 << DS_STEREO)))
	{
		criteria[n].capability = DS_STEREO;
		criteria[n].comparison = DS_EQ;
		criteria[n].value = 0;
		n++;
	}
	if(!(mask & (1 << DS_DOUBLEBUFFER)))
	{
		criteria[n].capability = DS_DOUBLEBUFFER;
		criteria[n].comparison = DS_EQ;
		criteria[n].value = 0;
		*allowDoubleAsSingle = YES;
		n++;
	}
	if(!(mask & (1 << DS_DEPTH_SIZE)))
	{
		criteria[n].capability = DS_DEPTH_SIZE;
		criteria[n].comparison = DS_MIN;
		criteria[n].value = 0;
		n++;
	}
	if(!(mask & (1 << DS_STENCIL_SIZE)))
	{
		criteria[n].capability = DS_STENCIL_SIZE;
		criteria[n].comparison = DS_MIN;
		criteria[n].value = 0;
		n++;
	}
	if(!(mask & (1 << DS_LEVEL)))
	{
		criteria[n].capability = DS_LEVEL;
		criteria[n].comparison = DS_EQ;
		criteria[n].value = 0;
		n++;
	}
	if(n)
	{
		/* Since over-estimated the size needed; squeeze it down to reality. */
		criteria = (GLUTCriterion *) realloc(criteria, n * sizeof(GLUTCriterion));
		if(!criteria)
		{
			/* Should never happen since should be shrinking down! */
			__glutFatalError("out of memory.");
		}
	}
	else
	{
			/* For portability, avoid "realloc(ptr,0)" call. */
		free(criteria);
		criteria = NULL;
	}
	
	free(copy);
	*ncriteria = n;
	
	return criteria;
}

#if defined(VERBOSE1)

static void _printCriteria(GLUTCriterion * criteria, int ncriteria)
{
	int	i;
	
	NSLog(@"Criteria: %d\n", ncriteria);
	for(i = 0; i < ncriteria; i++)
		NSLog(@"  %s %s %d\n", capstr[criteria[i].capability], compstr[criteria[i].comparison], criteria[i].value);
}

#endif

@end



@implementation GLUTVisual(GLUTPrivate)

+ (void)initialize
{
#if defined(GFXLIB_CONIX)
	__glutConcreteVisualClass = [GLUTConixVisual class];
#elif defined(GFXLIB_MESA26)
	__glutConcreteVisualClass = [GLUTMesaVisual class];
#else
#	error no gl target specified
#endif
}

+ (id)allocWithZone: (NSZone *)zone
{
	if(self == [GLUTVisual class])
		return [__glutConcreteVisualClass allocWithZone: zone];
	else
		return [super allocWithZone: zone];
}

@end
