// MOFileCompletionStrategy.m
// MOKit - Yellow Box
//
// Copyright 1996-1999, Mike Ferris.
// All rights reserved.

// ABOUT MOKit
// 
// MOKit is a collection of useful and general objects.  Permission is 
// granted by the author to use MOKit in your own programs in any way 
// you see fit.  All other rights to the kit are reserved by the author 
// including the right to sell these objects as part of a LIBRARY or as 
// SOURCE CODE.  In plain English, I wish to retain rights to these 
// objects as objects, but allow the use of the objects as pieces in a 
// fully functional program.  NO WARRANTY is expressed or implied.  The author 
// will under no circumstances be held responsible for ANY consequences to 
// you from the use of these objects.  Since you don't have to pay for 
// them, and full source is provided, I think this is perfectly fair.

#import <MOKit/MOKit.h>

typedef enum {
    MOInitialVersion = 1,
} MOClassVersion;

static const MOClassVersion MOCurrentClassVersion = MOInitialVersion;

@implementation MOFileCompletionStrategy

+ (void)initialize {
    // Set the version.  Load classes, and init class variables.
    if (self == [MOFileCompletionStrategy class])  {
        [self setVersion:MOCurrentClassVersion];
    }
}

- (id)init {
    self = [super init];
    _fcsFlags.appendsSpaceOnFileMatch = NO;
    _fcsFlags.appendsSlashOnDirectoryMatch = YES;
    return self;
}

- (void)dealloc {
    [super dealloc];
}

- (BOOL)appendsSpaceOnFileMatch {
    return _fcsFlags.appendsSpaceOnFileMatch;
}

- (void)setAppendsSpaceOnFileMatch:(BOOL)flag {
    _fcsFlags.appendsSpaceOnFileMatch = flag;
}

- (BOOL)appendsSlashOnDirectoryMatch {
    return _fcsFlags.appendsSlashOnDirectoryMatch;
}

- (void)setAppendsSlashOnDirectoryMatch:(BOOL)flag {
    _fcsFlags.appendsSlashOnDirectoryMatch = flag;
}

- (NSString *)basePathFromProposedBasePath:(NSString *)basePath path:(NSString *)path {
    if (![path isAbsolutePath]) {
        if (basePath) {
#ifdef WIN32
            if ([path hasPrefix:@"/"] || [path hasPrefix:@"\\"]) {
                // Just take the drive letter from the basePath
                basePath = [[basePath pathComponents] objectAtIndex:0];
            }
#endif
            basePath = [basePath stringByStandardizingPath];
        }
    } else {
        basePath = nil;
    }
    return basePath;
}

- (void)addFilesMatchingPrefix:(NSString *)partialName forChildrenOfDirectory:(NSString *)dirPath toMutableArray:(NSMutableArray *)matches {
    NSFileManager *fm = [NSFileManager defaultManager];
    NSArray *children = [[fm directoryContentsAtPath:dirPath] sortedArrayUsingSelector:@selector(caseInsensitiveCompare:)];
    unsigned i, c;
    NSString *curChild;

    c = [children count];
    for (i=0; i<c; i++) {
        curChild = [children objectAtIndex:i];
        if (!partialName || ([curChild rangeOfString:partialName options:NSAnchoredSearch range:NSMakeRange(0, [curChild length])].length != 0)) {
            [matches addObject:curChild];
        }
    }

    // MF: Allow case-insensitive completions, but only if there aren't any for the correct case.  Allow this on Windows or Mach.  What the hell...
    if ([matches count] == 0) {
        for (i=0; i<c; i++) {
            curChild = [children objectAtIndex:i];
            if (!partialName || ([curChild rangeOfString:partialName options:(NSAnchoredSearch | NSCaseInsensitiveSearch) range:NSMakeRange(0, [curChild length])].length != 0)) {
                [matches addObject:curChild];
            }
        }
    }
}

- (NSArray *)matchesForPrefixString:(NSString *)path newPrefixString:(NSString **)newStr basePath:(NSString *)basePath {
    unichar ch;
    NSString *partialName = nil;
    NSMutableArray *matches = [[[NSMutableArray allocWithZone:[self zone]] init] autorelease];
    
    if (([path isEqualToString:@""]) || ((ch = [path characterAtIndex:[path length] - 1]) == (unichar)'/') || (ch == (unichar)'\\')) {
        partialName = nil;
    } else {
        partialName = [path lastPathComponent];
        path = [path stringByDeletingLastPathComponent];
    }
    path = [path stringByStandardizingPath];
    basePath = [self basePathFromProposedBasePath:basePath path:path];

    //NSLog(@"FileCompletionStrategy basePath: '%@', path: '%@', partialName: '%@'.", basePath, path, partialName);

    // Now find the matches
    [self addFilesMatchingPrefix:partialName forChildrenOfDirectory:(basePath ? [basePath stringByAppendingPathComponent:path] : path) toMutableArray:matches];

#if 0
    // MF: This code was supposed to allow completion based on a an array of paths as well as the default path.  It is disabled because I haven't really thought through the full implications of such a change
    if (basePath && _paths) {
        // This was a non-absolute path, also add matches for any directories in our path list
#ifdef WIN32
        // Make sure it wasn't semi-absolute...
        if ([path hasPrefix:@"/"] || [path hasPrefix:@"\\"]) {
#endif
            unsigned i, c = [_paths count];
            for (i=0; i<c; i++) {
                [self addFilesMatchingPrefix:partialName forChildrenOfDirectory:[_paths objectAtIndex:i] toMutableArray:matches];
            }
#ifdef WIN32
        }
#endif
    }
#endif
    
    if (newStr) {
        *newStr = [[path copyWithZone:[self zone]] autorelease];
    }
    //NSLog(@"FileCompletionStrategy matches: '%@'\nNew prefix string: '%@'.", matches, path);
    return matches;
}

- (NSString *)fullStringForPrefixString:(NSString *)prefixStr completionString:(NSString *)completionStr isInitialPrefixMatch:(BOOL)initialPrefixMatch basePath:(NSString *)basePath {
    NSString *fullStr;

    if ([completionStr isEqualToString:@""]) {
        if ([prefixStr isEqualToString:@""]) {
            fullStr = prefixStr;
        } else {
            fullStr = [prefixStr stringByAppendingString:@"/"];
        }
    } else {
        fullStr = [prefixStr stringByAppendingPathComponent:completionStr];
    }
    fullStr = [fullStr MO_stringByReplacingBackslashWithSlash];
    if (!initialPrefixMatch) {
        NSString *tempPath = fullStr;
        BOOL exists, isDir = NO;
        
        basePath = [self basePathFromProposedBasePath:basePath path:fullStr];
        if (basePath) {
            tempPath = [basePath stringByAppendingPathComponent:fullStr];
        }
        exists = [[NSFileManager defaultManager] fileExistsAtPath:tempPath isDirectory:&isDir];
#if 0
        // MF: This code was supposed to allow completion based on a an array of paths as well as the default path.  It is disabled because I haven't really thought through the full implications of such a change
        if (!exists && basePath && _paths) {
#ifdef WIN32
            // Make sure it wasn't semi-absolute...
            if ([fullStr hasPrefix:@"/"] || [fullStr hasPrefix:@"\\"]) {
#endif

                unsigned i, c = [_paths count];
                for (i=0; (!exists && i<c); i++) {
                    tempPath = [[_paths objectAtIndex:i] stringByAppendingPathComponent:fullStr];
                    exists = [[NSFileManager defaultManager] fileExistsAtPath:tempPath isDirectory:&isDir];
                }
#ifdef WIN32
            }
#endif
        }
#endif
        if (exists && isDir) {
            if (_fcsFlags.appendsSlashOnDirectoryMatch) {
                fullStr = [fullStr stringByAppendingString:@"/"];
            }
        } else if (exists) {
            if (_fcsFlags.appendsSpaceOnFileMatch) {
                fullStr = [fullStr stringByAppendingString:@" "];
            }
        }
    }
    return fullStr;
}

@end
