/*
    File:	ClockView.m

    Contains:	Subclass of view to implement a clock. You can set the
		clock type (analog, digital, or sundial) and choose to
		display seconds and the current date.

    Written By:	Ali T. Ozer

    Created:	June 25, 1997

    Copyright:	(c) 1997 by Apple Computer, Inc., all rights reserved.

    Change History:	Redesigned for 2.0 by Julie Zelenski.
			Modified for 3.0 by Ali Ozer.
			Redesigned for Rhapsody by Stephen Chick.
			Changed to use a repeating timer (7/15/97, SC).
			Changed to use NSString to draw text (7/15/97, SC).
			Changed to use NSDate to draw dates (7/15/97, SC).
			Changed to use NSCalendarDate to draw times (10/14/97, SC).
			Fixed bug in use of NSTimer (12/9/97, SC).

    You may incorporate this sample code into your applications without
    restriction, though the sample code has been provided "AS IS" and the
    responsibility for its operation is 100% yours.  However, what you are
    not permitted to do is to redistribute the source as "DSC Sample Code"
    after having made changes. If you're going to re-distribute the source,
    we require that you make it clear in the source that the code was
    descended from Apple Sample Code, but that you've made changes.
*/

#import "Clock.h"
#import "ClockView.h"	

@implementation ClockView

//  Constants.

#define PI (double)3.1415926535897

#define ANALOG	0	
#define DIGITAL	1
#define SUNDIAL 2

#define SHADOWRATIO .95		//  Sundial shadow length relative to radius
#define MARKERRATIO .15		//  Sundial marker height relative to radius

#define HOURRATIO 0.5		//  Analog hour hand length relative to radius
#define MINUTERATIO 0.85	//  Minute & seconds hands

//  String constants.

static NSString *romanDay [31] = {@"I", @"II", @"III", @"IV", @"V", @"VI",
				  @"VII", @"VIII", @"IX", @"X", @"XI", @"XII",
				  @"XIII", @"XIV", @"XV", @"XVI", @"XVII",
				  @"XVIII", @"XIX", @"XX", @"XXI", @"XXII",
				  @"XXIII", @"XXIV", @"XXV", @"XXVI", @"XXVII",
				  @"XXVIII", @"XXIX", @"XXX", @"XXXI"};

//-----------------------------------------------------------------------------
//
//  initWithFrame
//
//  Overrides NSView's initWithFrame method to initialize this custom view.
//
//  Initializes the face image, our fonts, and miscellaneous parameters. Then
//  starts the timer and draw ourselves.
//

- initWithFrame:(NSRect) frameRect
{
	NSRect bounds;

	[super initWithFrame:frameRect];

	//  Get our frame bounds.

	bounds = [self bounds];

	//  Create an image used to store the clock face.

	face = [[NSImage allocWithZone:[self zone]] initWithSize:bounds.size];

	//  Set up the fonts we'll be using.

	littleFont = [[NSFont fontWithName:@"Helvetica" size:9] retain];
	mediumFont = [[NSFont fontWithName:@"Times-Roman" size:12] retain];
	bigFont = [[NSFont fontWithName:@"Times-Roman" size:24] retain];
	
	//  Set the default state (analog face, no seconds, no date).

	clockType = ANALOG;
	showSeconds = NO;
	showDate = NO;

	//  Precalculate a bunch of geometricalstuff.

	center.x = bounds.size.width / 2.0;
	center.y = bounds.size.height / 2.0 + [mediumFont pointSize] / 2.0;
	radius = MIN (center.x, center.y - [mediumFont pointSize]);

	//  Register ourselves to receive notification when our view bounds
	//  change.

	[[NSNotificationCenter defaultCenter] addObserver:self
		selector:@selector (viewBoundsDidChange:)
		name:@"NSViewBoundsDidChangeNotification" object:nil];

	//  Predraw the face image.

	[self drawFace];

	//  Start the time entry. YES indicates that this is the first time

	[self startTimer:YES];
	[self setNeedsDisplay:YES];
	return self;
}

//-----------------------------------------------------------------------------
//
//  dealloc
//
//  Release any objects we created. Then call super.
//

- (void) dealloc
{
	[[NSNotificationCenter defaultCenter] removeObserver:self];
	[face release];
	[littleFont release];
	[mediumFont release];
	[bigFont release];
	[timer invalidate];
	[timer release];
	[super dealloc];
}

//-----------------------------------------------------------------------------
//
//  clockType
//

- (int) clockType
{
    return clockType;
}

//-----------------------------------------------------------------------------
//
//  showSeconds
//

- (BOOL) showSeconds 
{ 
    return showSeconds; 
}

//-----------------------------------------------------------------------------
//
//  showDate
//

- (BOOL) showDate
{ 
    return showDate; 
}

//-----------------------------------------------------------------------------
//
//  setClockType:
//
//  Sets the clock type (analog, digital, or sundial) and redraws the view.
//

- (void) setClockType:(int) newValue
{
	clockType = newValue;
	[self drawFace];
	[self setNeedsDisplay:YES];
}

//-----------------------------------------------------------------------------
//
//  setShowSeconds:
//
//  Sets whether or not the seconds hand is shown and redraws the view. The
//  timer must be reset because the time to the next firing depends on this
//  setting.
//

- (void) setShowSeconds:(BOOL) newValue
{ 
	showSeconds = newValue;
	[self startTimer:NO];
	[self setNeedsDisplay:YES];
}

//-----------------------------------------------------------------------------
//
//  setShowDate:
//
//  Sets whether or not the date is shown and redraws the view.
//

- (void) setShowDate:(BOOL) newValue
{ 
	showDate = newValue;
	[self setNeedsDisplay:YES];
}

//-----------------------------------------------------------------------------
//
//  changeShowDate:
//

- (void) changeShowDate:(id) sender;
{
	[self setShowDate:[[sender selectedCell] state]];
}

//-----------------------------------------------------------------------------
//
//  changeShowSeconds:
//

- (void) changeShowSeconds:(id) sender
{
	[self setShowSeconds:[[sender selectedCell] state]];
}

//-----------------------------------------------------------------------------
//
//  changeClockType:
//

- (void) changeClockType:(id) sender
{
	[self setClockType:[sender selectedTag]];
}

//-----------------------------------------------------------------------------
//
//  drawAnalog:
//
//  Draws the clock hands and date for the analog clock face
//

- (void) drawAnalog:(NSCalendarDate *) time;
{
	//  Draw the second, hour, and minute hands.

	if (showSeconds)
		PSWdrawClockHand (center.x, center.y, -6.0 * [time secondOfMinute],
				  radius * MINUTERATIO, NSDarkGray, 0.0);
	PSWdrawClockHand (center.x, center.y,
			  -([time hourOfDay] + [time minuteOfHour] / 60.0) * 30.0,
			  radius * HOURRATIO, NSBlack, 1.0);
	PSWdrawClockHand (center.x, center.y, -fmod ([time minuteOfHour], 60.0) * 6.0,
			  radius * MINUTERATIO, NSBlack, 1.0);

	//  Draw the date (day of week, month, day).

	if (showDate) {
		id object [2];
		id key [2];
		NSPoint point;
		NSSize size;
                NSDictionary *attributes;
		NSString *dateString;

		//  Build the date string.

                dateString = [time descriptionWithCalendarFormat:@"%A %B %d"];

		//  Build the date string attributes (little font, dark gray).

		object [0] = littleFont;
		key [0] = NSFontAttributeName;
		object [1] = [NSColor darkGrayColor];
		key [1] = NSForegroundColorAttributeName;
		attributes = [NSDictionary dictionaryWithObjects:object
			      forKeys:key count:2];

		//  Calculate the string position.

                size = [dateString sizeWithAttributes:attributes];
		point.x = center.x - size.width / 2.0;
		point.y = -7.0;

		//  Draw the string.

                [dateString drawAtPoint:point withAttributes:attributes];
	}
}

//-----------------------------------------------------------------------------
//
//  drawDigital:
//
//  drawDigital draws the time and date for the digital clock face
//

- (void) drawDigital:(NSCalendarDate *) time
{
	NSPoint point;
	NSSize size;
	NSDictionary *attributes;
	NSString *dateString, *timeString;
    
	//  Draw the time (hours, minutes, and seconds).

	if (showSeconds)
		timeString = [time descriptionWithCalendarFormat:@"%I:%M:%S"];
	else
		timeString = [time descriptionWithCalendarFormat:@"%I:%M"];

	//  Build the time string attributes (big font).

        attributes = [NSDictionary dictionaryWithObject:bigFont
                      forKey:NSFontAttributeName];

	//  Calculate the string position.

	size = [timeString sizeWithAttributes:attributes];
	point.x = center.x - size.width / 2.0;
	point.y = 10.0;

	//  Draw the string.

	[timeString drawAtPoint:point withAttributes:attributes];

	//  Draw the date (day of week, month, day).

	if (showDate) {

		//  Build the date string.

		dateString = [time descriptionWithCalendarFormat:@"%A %B %d"];

                //  Build the date string attributes (medium font).

                attributes = [NSDictionary dictionaryWithObject:littleFont
			      forKey:NSFontAttributeName];

                //  Calculate the string position.

                size = [dateString sizeWithAttributes:attributes];
                point.x = center.x - size.width / 2.0;
                point.y = 6.0;

                //  Draw the string.

                [dateString drawAtPoint:point withAttributes:attributes];
	}
}

//-----------------------------------------------------------------------------
//
//  drawSundial:
//
//  Draws the shadow and date for the sundial clock face
//  

- (void) drawSundial:(NSCalendarDate *) time;
{   
	float percentOfDay;
	NSPoint edge;

	//  Draw the sweep.

	if (showSeconds) 
		PSWdrawSweep (center.x, center.y, -6.0 * [time secondOfMinute],
			      radius * 0.75);

	//  Draw the shadow.

	percentOfDay = ([time hourOfDay] * 60 + [time minuteOfHour]) / (24.0 * 60.0);
	edge.x = sin (percentOfDay * PI * 2) * radius * SHADOWRATIO;
	edge.y = cos (percentOfDay * PI * 2) * radius * SHADOWRATIO;
	PSWdrawShadow (center.x, center.y, edge.x, edge.y,
		       radius * MARKERRATIO);

	//  Draw the date (month, day).

	if (showDate) {
                NSPoint point;
                NSSize size;
                NSDictionary *attributes;
                NSString *dateString;

                //  Build the date string.

		dateString = [time descriptionWithCalendarFormat:@"%B "];
		dateString = [dateString stringByAppendingString:romanDay [[time dayOfMonth] - 1]];

                //  Build the date string attributes (little font, dark gray).

                attributes = [NSDictionary dictionaryWithObject:mediumFont
			      forKey:NSFontAttributeName];

                //  Calculate the string position.

                size = [dateString sizeWithAttributes:attributes];
                point.x = center.x - size.width / 2.0;
                point.y = -3.0;

                //  Draw the string.

                [dateString drawAtPoint:point withAttributes:attributes];
	}
}

//-----------------------------------------------------------------------------
//
//  drawFace
//
//  drawFace draws the clock face image.  This
//  image is composited on screen and then the hands, shadow,
//  whatever is drawn on top of the face for the current time.
//

- (void) drawFace
{
	[face lockFocus];

	//  Fill the clock face image with the default background color for
	//  controls.

        [[NSColor whiteColor] set];
	NSRectFill ([self bounds]);
	[[NSColor blackColor] set];

	//  Draw the appropriate clock face.

	switch (clockType) {
		case ANALOG:
			PSWdrawAnalogFace (center.x, center.y,radius);
			break;
		case DIGITAL:		//  digital "face" is just blank
			break;
		case SUNDIAL:
			PSWdrawSundialFace (center.x, center.y, radius);
			break;
	}
	[face unlockFocus];
}

//-----------------------------------------------------------------------------
//
//  drawRect:
//
//  Draws the face and hands of the clock.
//

- (void) drawRect:(NSRect) rect
{
	//  First redraw the clock face by compositing the predrawn NSImage.

	[face compositeToPoint:[self bounds].origin operation:NSCompositePlusDarker];

	//  Draw the clock showing the current date and time.

	switch (clockType) {
		case ANALOG:
 			[self drawAnalog:[NSCalendarDate calendarDate]];
			break;
		case DIGITAL:
			[self drawDigital:[NSCalendarDate calendarDate]];
			break;
		case SUNDIAL:
			[self drawSundial:[NSCalendarDate calendarDate]];
			break;
	}
}

//-----------------------------------------------------------------------------
//
//  startTimer:
//
//  Creates a timer. If fireASAP is YES, the timer will be set to fire as soon
//  as possible (this would be the case at the start of the program, for
//  instance). If fireASAP is NO, then the timed entry is set to fire either in
//  one second (if seconds are being shown) or at the top of the next minute.
//

- (void) startTimer:(BOOL) fireASAP
{
	NSTimeInterval seconds;

	//  If a timer is already running, stop it (remove it from the
	//  NSRunLoop) and release it.

	[timer invalidate];
	[timer autorelease];

	//  If seconds are being shown, set up a repeating timer that fires
	//  once a second. Otherwise, set up a non-repeating timer that fires
	//  on the next minute.

	if (showSeconds) {
		seconds = 1.0;
		timerRepeats = YES;
	}
	else {
		seconds = 60.0 - ([[NSCalendarDate calendarDate] secondOfMinute] % 60);
		timerRepeats = NO;
	}
	timer = [NSTimer scheduledTimerWithTimeInterval:seconds target:self
		 selector:@selector (showTime:) userInfo:nil repeats:timerRepeats];

	//  Retain the timer.

	[timer retain];

        //  Send the receiver message to the target.

	if (fireASAP)
		[timer fire];
}

//-----------------------------------------------------------------------------
//
//  showTime:
//
//  Target action for the timer object. showTime: is called when the timer
//  fires. After redrawing the view, it calls startTimer: to prepare for the
//  next second or minute.
//

- (void) showTime:(NSTimer *) timer 
{
	[self setNeedsDisplay:YES];
	if (timerRepeats == NO)
		[self startTimer:NO];
}

//-----------------------------------------------------------------------------
//
//  viewBoundsDidChange:
//
//  viewBoundsDidChange: is invoked when this view's bounds change. It fixes
//  the clock to the new bounds and redraws the view.
//  

- (void) viewBoundsDidChange:(NSNotification *) aNotification
{
	NSRect bounds;

	bounds = [self bounds];
	center.x = bounds.size.width / 2.0;
	center.y = bounds.size.height / 2.0 + [mediumFont pointSize] / 2.0;
	radius = MIN (center.x, center.y- [mediumFont pointSize]);
	[face setSize:bounds.size];
	[self drawFace];
}

@end
