/**
 * Zephyr New Message UI Controller
 **
 * Wilfredo Sanchez | wsanchez@apple.com
 * Copyright 1998 Apple Computer, Inc.
 * All rights reserved.
 **/

#import <Foundation/Foundation.h>
#import <AppKit/AppKit.h>

#import "NSZephyr.h"
#import "ZephyrApplication.h"
#import "ZephyrMessage.h"
#import "NewMessageController.h"

enum _sendToOptions
{
    SEND_TO_USER = 0,
    SEND_TO_GROUP,
    SEND_TO_CUSTOM
};

@interface NewMessageController (Private)

- (void) setSendTo: (int) anOption;

- (void) verifyDestination;

@end

@implementation NewMessageController

/*******************
 * Class Variables *
 *******************/

static NSMutableArray* theRecentDestinations = nil;

+ (void) initialize
{
    if (self == [NewMessageController class])
      {
        theRecentDestinations = [[NSMutableArray array] retain];
      }
}

/********************
 * Instance Methods *
 ********************/

////
// Inits
////

- (id) init
{
    return [self initWithMessage: nil];
}

- (id) initWithMessage: (ZephyrMessage*) aMessage
{
    if ((self = [super init]))
      {
        // Load new message interface
        if (! [NSBundle loadNibNamed: @"NewMessage"
                           owner: self])
          {
            NSLog(@"Failed to open MessageBrowser interface.");
            [self release];
            return nil;
          }

        if (! window             ||
            ! destinationForm    ||
            ! classCell          ||
            ! instanceCell       ||
            ! recipientCell      ||
            ! sendToMatrix       ||
            ! urgentButton       ||
            ! authenticateButton ||
            ! favoritesPopUp     ||
            ! recentPopUp        ||
            ! replyPopUp         ||
            ! findButton         ||
            ! sendButton         ||
            ! messageTextView    ||
            ! statusTextField    )
          {
            NSLog(@"WARNING: NewMessageController interface failed to load properly.");
          }

        // Init window
        [window makeKeyAndOrderFront: self];

        // Init buttons
        [ findButton setEnabled: YES ];
        [ sendButton setEnabled: NO  ];

        if (aMessage)
          {
            NSString* aClass     = [aMessage zephyrClass   ];
            NSString* anInstance = [aMessage zephyrInstance];
            NSString* aRecipient = [aMessage recipient     ];

            if ([[aMessage recipient] isEqualToString: NS_GROUP_RECIPIENT])
              {
                [self setRecipient : aRecipient   ];
                [self setSendTo    : SEND_TO_GROUP];
              }
            else
              {
                [self setRecipient: [aMessage sender]];

                if ([aClass     isEqualToString: NS_DEFAULT_CLASS   ] &&
                    [anInstance isEqualToString: NS_DEFAULT_INSTANCE])
                  {
                    [self setSendTo: SEND_TO_USER];
                  }
                else
                  {
                    [self setSendTo: SEND_TO_CUSTOM];
                  }
              }

            // Init destination
            [ self setZephyrClass    : aClass     ];
            [ self setZephyrInstance : anInstance ];
          }
        else
          {
            // Init destination
            [ self setZephyrClass    : NS_DEFAULT_CLASS     ];
            [ self setZephyrInstance : NS_DEFAULT_INSTANCE  ];
            [ self setRecipient      : NS_DEFAULT_RECIPIENT ];
          }

        // Init popups
        [ self setFavorites : nil                   ];
        [ self setRecents   : theRecentDestinations ];
        [ self setReplies   : nil                   ];

        // Init message text
        [messageTextView setString: @""];

        // Init status text
        [self setStatus: @""];
      }
    return self;
}

////
// Accessors
////

- (NSString*) zephyrClass    { return [ classCell     title ]; }
- (NSString*) zephyrInstance { return [ instanceCell  title ]; }
- (NSString*) recipient      { return [ recipientCell title ]; }

- (void) setZephyrClass    : (NSString*) aClass     { [ classCell     setStringValue: aClass     ]; }
- (void) setZephyrInstance : (NSString*) anInstance { [ instanceCell  setStringValue: anInstance ]; }
- (void) setRecipient      : (NSString*) aRecipient { [ recipientCell setStringValue: aRecipient ]; }

- (void) setList: (NSArray*      ) aList
        forPopUp: (NSPopUpButton*) aPopUp
{
    int aCount = [aPopUp numberOfItems];

    while (aCount-- > 1) [aPopUp removeItemAtIndex: 1];

    if (aList) [aPopUp addItemsWithTitles: aList];
}

- (void) setFavorites : (NSArray*) aList { [ self setList:aList forPopUp:favoritesPopUp ]; }
- (void) setRecents   : (NSArray*) aList { [ self setList:aList forPopUp:recentPopUp    ]; }
- (void) setReplies   : (NSArray*) aList { [ self setList:aList forPopUp:replyPopUp     ]; }

- (void) setStatus: (NSString*) aMessage { [statusTextField setStringValue: aMessage]; }

////
// IB Targets
////

- (void) editDestinationTarget: (id) aSender { [self verifyDestination]; }

- (void) selectSendToTarget: (NSMatrix*) aSendToMatrix
{
    [self setSendTo: [aSendToMatrix selectedRow]];
}

- (void) checkUrgentTarget: (NSButton*) anUrgentButton
{
    if ([anUrgentButton state] == NSOnState)
      {
        [sendToMatrix setEnabled: NO];

        [ self setSendTo         : SEND_TO_USER       ];
        [ self setZephyrClass    : NS_URGENT_CLASS    ];
        [ self setZephyrInstance : NS_URGENT_INSTANCE ];
      }
    else
      {
        [sendToMatrix setEnabled: YES];

        [ self setZephyrClass    : NS_DEFAULT_CLASS    ];
        [ self setZephyrInstance : NS_DEFAULT_INSTANCE ];
      }
}

- (void) selectFromPopUpTarget: (NSPopUpButton*) aPopUp
{
    // Parse destination from title

    // Update destination
}

- (void) locateUsersTarget: (id) aSender { [(ZephyrApplication*)NSApp locateUsers]; }

- (void) sendMessageTarget: (id) aSender
{
    NSString*     anAuth = ([authenticateButton state] == NSOnState) ? @"yes":@"no";

    NSDictionary* anInfo =
        [NSDictionary dictionaryWithObjectsAndKeys:
            [self zephyrClass   ] , @"class"     ,
            [self zephyrInstance] , @"instance"  ,
            [self recipient     ] , @"recipient" ,
            NS_DEFAULT_OPCODE     , @"opcode"    ,
            anAuth                , @"auth"      ,
            nil];

    ZephyrMessage* aMessage = [ZephyrMessage messageWithInfo: anInfo
                                                        text: [messageTextView textStorage]];

    [aMessage send];

    [window performClose: self];
}

////
// Private Methods
////

- (void) setSendTo: (int) anOption
{
    [sendToMatrix selectCellAtRow:anOption column:0];

    switch (anOption)
      {
        case SEND_TO_USER:
            [ self setZephyrClass    : NS_USER_CLASS        ];
            [ self setZephyrInstance : NS_USER_INSTANCE     ];
            //[ self setRecipient      : NS_DEFAULT_RECIPIENT ];

            [     classCell setEnabled: NO  ];
            [  instanceCell setEnabled: NO  ];
            [ recipientCell setEnabled: YES ];
            [  urgentButton setEnabled: YES ];

            break;

        case SEND_TO_GROUP:
            //[ self setZephyrClass    : NS_DEFAULT_CLASS    ];
            //[ self setZephyrInstance : NS_DEFAULT_INSTANCE ];
            [ self setRecipient      : NS_GROUP_RECIPIENT  ];

            [     classCell setEnabled: YES ];
            [  instanceCell setEnabled: YES ];
            [ recipientCell setEnabled: NO  ];
            [  urgentButton setEnabled: NO  ];

            break;

        case SEND_TO_CUSTOM:
            //[ self setZephyrClass    : NS_DEFAULT_CLASS     ];
            //[ self setZephyrInstance : NS_DEFAULT_INSTANCE  ];
            //[ self setRecipient      : NS_DEFAULT_RECIPIENT ];

            [     classCell setEnabled: YES ];
            [  instanceCell setEnabled: YES ];
            [ recipientCell setEnabled: YES ];
            [  urgentButton setEnabled: NO  ];

            break;

        default:
            NSLog(@"ERROR: Send to option out of bounds.");
            break;
      }

    [self verifyDestination];
}

- (void) verifyDestination
{
    NSString* aClass     = [    classCell stringValue];
    NSString* anInstance = [ instanceCell stringValue];
    NSString* aRecipient = [recipientCell stringValue];

    NSLog(@"Destination is %@,%@,%@", aClass, anInstance, aRecipient);

    if ([aClass     isEqualToString: @""] ||
        [anInstance isEqualToString: @""] ||
        [aRecipient isEqualToString: @""])
      {
        [sendButton setEnabled: NO];
        [self setStatus: @"Destination incomplete."];
        return;
      }

    if ([aRecipient isEqualToString: NS_GROUP_RECIPIENT])
      {
        [sendButton setEnabled: YES];
        [self setStatus: [NSString stringWithFormat: @"Ready to send %@,%@ message.", aClass, anInstance]];
        return;
      }

    // Send a ping
    [self setStatus: [NSString stringWithFormat: @"Sending a PING to %@...", aRecipient]];

    // Disable send if ping failed
    if (NO)
      {
        [self setStatus: [NSString stringWithFormat: @"%@ is not logged in or is not suscribed to %@,%@ messages.",
                                                     aRecipient, aClass, anInstance]];
        [sendButton setEnabled: NO];
      }

    // Update status
    [self setStatus: [NSString stringWithFormat: @"Ready to send message to %@.", aRecipient]];
    [sendButton setEnabled: YES];
}

@end
