/**
 * Zephyr Message
 **
 * Wilfredo Sanchez | wsanchez@apple.com
 *
 * Copyright 1998 Apple Computer, Inc.
 *
 * Copyright (c) 1989 by the Massachusetts Institute of Technology.
 * For copying and distribution information, see the file
 * "mit-copyright.h".
 *
 * All rights reserved.
 **/

#include <zephyr/mit-copyright.h>

#include "NSZephyr.h"

#import <Foundation/Foundation.h>

#import "ZephyrClient.h"

#import "ZephyrMessage.h"

@interface ZephyrMessage (Private)

- (BOOL) sendWithPingOnly: (BOOL) aPingOption;

@end

@implementation ZephyrMessage

/*****************
 * Class Methods *
 *****************/

////
// Factories
///

+ (ZephyrMessage*) messageWithInfo: (NSDictionary      *) anInfo
                              text: (NSAttributedString*) aText
{
    return [[[self alloc] initWithInfo: anInfo
                                  text: aText] autorelease];
}

/********************
 * Instance Methods *
 ********************/

////
// Inits
////

- (id) initWithInfo: (NSDictionary*      ) anInfo
               text: (NSAttributedString*) aText
{
    if ((self = [super init]))
      {
        myInfo = [anInfo retain];
        myText = [aText  retain];
      }
    return self;
}

- (void) dealloc
{
    [ myInfo release ];
    [ myText release ];

    [super dealloc];
}

////
// Accessors
////

- (NSDictionary*      ) info { return myInfo; }
- (NSAttributedString*) text { return myText; }

- (NSString*) date            { return [myInfo objectForKey: @"date"     ]; }
- (NSString*) time            { return [myInfo objectForKey: @"time"     ]; }
- (NSString*) zephyrClass     { return [myInfo objectForKey: @"class"    ]; }
- (NSString*) zephyrInstance  { return [myInfo objectForKey: @"instance" ]; }
- (NSString*) recipient       { return [myInfo objectForKey: @"recipient"]; }
- (NSString*) opcode          { return [myInfo objectForKey: @"opcode"   ]; }
- (NSString*) auth            { return [myInfo objectForKey: @"auth"     ]; }
- (NSString*) sender          { return [myInfo objectForKey: @"sender"   ]; }
- (NSString*) signature       { return [myInfo objectForKey: @"signature"]; }
- (NSString*) fromhost        { return [myInfo objectForKey: @"fromhost" ]; }

- (NSString*) description
{
    return [NSString stringWithFormat: @"%@\n%@", myText, myInfo];
}

////
// Actions
////

- (BOOL) ping { return [ self sendWithPingOnly: YES ]; }
- (BOOL) send { return [ self sendWithPingOnly: NO  ]; }

////
// Private Methods
////

- (BOOL) sendWithPingOnly: (BOOL) aPingOption
{
    ZNotice_t  aNotice;
    Z_AuthProc anAuthOption;
    NSString*  aClass        = [self zephyrClass   ];
    NSString*  anInstance    = [self zephyrInstance];
    NSString*  aRecipient    = [self recipient     ];
    NSString*  anOpcode      = [self opcode        ];

    ////
    // Validate message
    ////
    {
        if (! aClass || ! anInstance || ! aRecipient)
          {
            NSLog(@"ERROR: Incomplete destination while sending message.");
            return NO;
          }

        if (aPingOption) anOpcode = NS_PING_OPCODE;

        if (! anOpcode)
          {
            NSLog(@"ERROR: No opcode while sneding message.");
            return NO;
          }
    }

    ////
    // Set up the notice
    ////
    {
        anAuthOption = ([[[self auth] lowercaseString] isEqualToString: @"yes"]) ? ZAUTH:ZNOAUTH;

        aNotice.z_class          = (char*)[aClass     cString];
        aNotice.z_class_inst     = (char*)[anInstance cString];
        aNotice.z_opcode         = (char*)[anOpcode   cString];
        aNotice.z_recipient      = (char*)[aRecipient cString];
        aNotice.z_port           = 0;
        aNotice.z_sender         = 0;
        aNotice.z_kind           = ACKED;
        aNotice.z_default_format = "Class $class, Instance $instance:\n"
                                   "To: @bold($recipient) at $time $date\n"
                                   "From: @bold($1) <$sender>\n\n$2";

        if (aPingOption)
          {
            aNotice.z_message     = NULL;
            aNotice.z_message_len = 0;
          }
        else
          {
            ////
            // ** FIX ME **
            // We need to unparse the attributed string into a scribe-format C string.
            ////

            NSString* aMessageString = [myText string];

            aNotice.z_message     = (char*)[aMessageString cString      ];
            aNotice.z_message_len =        [aMessageString cStringLength];
          }
    }

    ////
    // Send the notice
    ////
    {
        int       aReturnCode;
        ZNotice_t aReply;

        if ( (aReturnCode = ZSendNotice(&aNotice, anAuthOption) ) != ZERR_NONE)
          {
            NSLog(@"%@ while sending (%@,%@) message to %@.",
                  [ZephyrClient errorMessageForCode: aReturnCode],
                  aClass, anInstance, aRecipient);
            return NO;
          }

        if ( (aReturnCode = ZIfNotice(&aReply,
                                      (struct sockaddr_in*)0,
                                      ZCompareUIDPred,
                                      (char*)&(&aNotice)->z_uid) ) != ZERR_NONE)
          {
            NSLog(@"%@ while waiting for reply for (%@,%@) message to %@.",
                  [ZephyrClient errorMessageForCode: aReturnCode],
                  aClass, anInstance, aRecipient);
            ZFreeNotice(&aReply);
            return NO;
          }

        if (aReply.z_kind == SERVNAK)
          {
            NSLog(@"Authorization failure while sending (%@,%@) message to %@.",
                  aClass, anInstance, aRecipient);
            ZFreeNotice(&aReply);
            return NO;
          }

        if (aReply.z_kind != SERVACK || aReply.z_message_len == 0)
          {
            NSLog(@"Server failure while sending (%@,%@) message to %@.",
                  aClass, anInstance, aRecipient);
            ZFreeNotice(&aReply);
            return NO;
          }

        if (!strcmp(aReply.z_message, ZSRVACK_SENT))
          {
            NSLog(@"(%@,%@) message sent to %@.", aClass, anInstance, aRecipient);
          }
        else if (!strcmp(aReply.z_message, ZSRVACK_NOTSENT))
          {
            if ([aRecipient isEqualToString: NS_GROUP_RECIPIENT])
              {
                NSLog(@"No one is subscribing to (%@,%@) messages.", aClass, anInstance);
              }
            else
              {
                NSLog(@"%@ is not logged in or not subscribing to (%@,%@) messages.",
                      aRecipient, aClass, anInstance);
              }
          }
        else
          {
            NSLog(@"ERROR: Unknown response from server while sending (%@,%@) message to %@.",
                  aClass, anInstance, aRecipient);
            ZFreeNotice(&aReply);
            return NO;
          }

        ZFreeNotice(&aReply);
    }

    return YES;
}

@end
