CREATE OR REPLACE PACKAGE BODY "MGMT_EMREP_METRICS" AS

g_config_handle UTL_FILE.FILE_TYPE := NULL;
g_dbID v$database.DBID%TYPE := NULL;

METRIC_END_MARKER constant VARCHAR2(4) := ':End';
METRIC_BEGIN_MARKER constant VARCHAR2(6) := ':Begin';
ORACLE_EMREP_META_VER CONSTANT VARCHAR(17) := '10.3.5.0.0';

g_em_version sysman.mgmt_versions.version%TYPE := NULL;

/*
Write em_error record
*/
PROCEDURE write_error(p_error_msg VARCHAR2)
IS
BEGIN
  UTL_FILE.PUT_LINE(g_config_handle,'em_error=' || p_error_msg);
  UTL_FILE.FFLUSH(g_config_handle);
END write_error;

/*
Put marker for the metric	
*/
PROCEDURE  put_metric_marker( marker in VARCHAR2, metric in VARCHAR2,instance_name in VARCHAR2 default null)
IS
BEGIN
    UTL_FILE.PUT( g_config_handle, metric );
    IF instance_name is not NULL  THEN
    	UTL_FILE.PUT( g_config_handle, ':'||instance_name);
    END IF; 
    UTL_FILE.PUT_LINE(g_config_handle,marker);
END put_metric_marker;

/*
 Generic function to write results of the query to the config dump file
*/
PROCEDURE  write_results( query  IN VARCHAR2, separator IN VARCHAR2 default '|')
IS
    l_em_result_cur INTEGER DEFAULT DBMS_SQL.OPEN_CURSOR;
    l_col_cnt       NUMBER DEFAULT 0;
    l_columnValue   VARCHAR2(4000);
    l_status        NUMBER ;

BEGIN
   BEGIN
    dbms_sql.parse(l_em_result_cur,  query, dbms_sql.native );

   /* define all the columns */
    FOR i IN 1 .. 255 LOOP
        BEGIN
            dbms_sql.define_column( l_em_result_cur, i, l_columnValue, 4000 );
            l_col_cnt := i;
        EXCEPTION
            WHEN OTHERS THEN
                IF ( sqlcode = -1007 ) THEN exit; 
                ELSE
                    RAISE;
                END IF;
        END;
    END LOOP;

    dbms_sql.define_column( l_em_result_cur, 1, l_columnValue, 4000 );
    l_status := DBMS_SQL.EXECUTE (l_em_result_cur);

    LOOP
        exit when ( dbms_sql.fetch_rows(l_em_result_cur) <= 0 );
        UTL_FILE.PUT( g_config_handle, 'em_result=');
        FOR i IN 1 .. l_col_cnt LOOP
	    IF i != 1 THEN
            	UTL_FILE.PUT( g_config_handle, separator);
	    END IF;
            dbms_sql.column_value( l_em_result_cur, i, l_columnValue );
             /* replace seperators with escaped separators */
            l_columnValue := replace(l_columnValue,'#','##');
            l_columnValue := replace(l_columnValue,separator,'#'|| separator);
            UTL_FILE.PUT( g_config_handle, l_columnValue );
        END LOOP;
        UTL_FILE.NEW_LINE(g_config_handle );
    END LOOP;

    dbms_sql.close_cursor(l_em_result_cur);

  EXCEPTION
	WHEN UTL_FILE.INVALID_FILEHANDLE 
         OR UTL_FILE.INVALID_OPERATION 
         OR UTL_FILE.WRITE_ERROR THEN
		IF DBMS_SQL.IS_OPEN(l_em_result_cur) = TRUE
		THEN
    			dbms_sql.close_cursor(l_em_result_cur);
		END IF;
		RAISE;
    	WHEN OTHERS THEN
		IF DBMS_SQL.IS_OPEN(l_em_result_cur) = TRUE
		THEN
    			dbms_sql.close_cursor(l_em_result_cur);
		END IF;
		/*
		On any non-utl file exceptions, log as em_error
		for the metric.
		*/
                write_error('SQLERRM: ' || SQLERRM 
                     || ' SQLCODE: ' || SQLCODE);
  END;
END write_results;

/*
 Generic function to write to the config dump file
*/
PROCEDURE  write_metric( metric IN VARCHAR2,query  IN VARCHAR2, instance_name in VARCHAR2 default null, separator IN VARCHAR2 default '|')
IS
    l_end_done      BOOLEAN DEFAULT FALSE;
BEGIN
    put_metric_marker(METRIC_BEGIN_MARKER,metric,instance_name);
    write_results(query,separator);
    put_metric_marker(METRIC_END_MARKER,metric,instance_name);
    l_end_done   := TRUE;
    UTL_FILE.FFLUSH(g_config_handle);
EXCEPTION
    	WHEN OTHERS THEN
          IF NOT l_end_done  THEN
            put_metric_marker(METRIC_END_MARKER,metric,instance_name);
            UTL_FILE.FFLUSH(g_config_handle);
          END IF;
	  RAISE;
END write_metric;

/*
Private procedure
Collect metric=emrep_config
*/
procedure collect_repository_config IS
BEGIN
        write_metric('emrep_config','SELECT DECODE(component_mode, ''CENTRAL'',''Grid Control'', ''SYSAUX'',''DB CONTROL'',component_mode),version from sysman.mgmt_versions where component_name=''CORE'' ');
END collect_repository_config;

/*
Private procedure
Collect metric=emrep_oms_name
*/
procedure collect_oms_names IS
BEGIN
        write_metric('emrep_omsname','select distinct a.url, u.oracle_home from (select distinct(host_url) url from mgmt_oms_parameters where name = ''OMS_URL'') a, (select b.host_url url,b.value oracle_home from (select host_url,name,value , max(timestamp) timestamp from mgmt_oms_parameters group by host_url,name,value) b, mgmt_oms_parameters op where op.name  =''ORACLE_HOME'' and op.host_url = b.host_url and op.name = b.name and op.value = b.value and op.timestamp = b.timestamp) u where a.url = u.url(+) ');
END collect_oms_names;

/*
Private procedure
Collect metric=emrep_agents
*/
procedure collect_agents IS
BEGIN
        write_metric('emrep_agents','SELECT target_name from mgmt$target where target_type = ''oracle_emd'' ');
END collect_agents;
 
/*
Private procedure
Collect metric=emrep_beacons
*/
procedure collect_beacons IS
BEGIN
        write_metric('emrep_beacons','SELECT target_name from mgmt$target where target_type = ''oracle_beacon'' ');
END collect_beacons; 

/*
Private procedure
Collect metric=emrep_oms_secure
*/
procedure collect_oms_secure IS
BEGIN
 	write_metric('emrep_oms_secure','SELECT DISTINCT p.host_url,  DECODE(p.value,0,''Unsecure'',1,''Secure Unlocked'',2,''Locked'',''Unknown:''||p.value) lock_status FROM mgmt_oms_parameters p, ( SELECT host_url, name, MAX(timestamp) last_time FROM mgmt_oms_parameters WHERE name = ''oms_secure_status'' GROUP BY host_url, name) l WHERE p.host_url = l.host_url AND p.timestamp = l.last_time AND p.name = l.name ORDER BY p.host_url'); 
END collect_oms_secure;

/*
Private procedure
Collect metric=emrep_utl_privs
*/
procedure collect_utl_privs IS
BEGIN
        write_metric('emrep_utl_privs','SELECT grantee, privilege, table_name, grantor, owner FROM sys.dba_tab_privs WHERE grantee =''SYSMAN'' AND privilege   = ''EXECUTE'' AND table_name IN (''UTL_FILE'', ''UTL_SMTP'', ''UTL_TCP'') ');
END collect_utl_privs;

/*
Private procedure
Collect metric=emrep_dbAssoc
*/
procedure collect_dbAssoc IS
BEGIN
        write_metric('emrep_dbAssoc','SELECT NAME from v$database');
END collect_dbAssoc;
 
procedure write_option_result(p_name VARCHAR2,p_version VARCHAR2,
                 p_isInstalled INTEGER,p_isUsed VARCHAR2,
                 p_feature_info VARCHAR2)
IS
BEGIN
  -- write a record for this option, if its installed
  IF p_isInstalled = 1 THEN
    -- replace '|' char in feature into with '#'
    write_results(
      'select ' ||
      '''' || g_dbID || ''', ' || -- DBID
      '''' ||  substr('OCM ' || p_name,0,64) || ''',' || -- NAME
      '''' || p_version || ''',' || -- Version
      'decode(''' || p_isUsed || ''',''TRUE'',''1'',''''),' || -- Detected Usages
      '1,' || -- Total Samples
      '''' || p_isUsed || ''' ,' || -- Currently Used
      ''''',' || -- First Usage Date
      ''''',' || -- Last Usage Date
      ''''',' || -- Aux Count
      ''''',' || -- Last Sample Date
      ''''',' || -- Last Sample Period
      '''' || replace(p_feature_info,'|','#') || '''' || -- Feature Info
      ' from dual');
  END IF;
END write_option_result;

/*
collect feature usage
*/
PROCEDURE collect_feature_usage
IS
  l_isUsed VARCHAR2(5);
  l_end_done      BOOLEAN DEFAULT FALSE;
  type cursor_type IS ref CURSOR;
  l_packs_cursor cursor_type;
  l_pack_name VARCHAR2(64);
  l_feature_info VARCHAR2(4000);
  l_temp_version sysman.mgmt_versions.version%TYPE := NULL;
BEGIN
   put_metric_marker(METRIC_BEGIN_MARKER,'feature_usage',null);
   l_temp_version := SUBSTR(g_em_version,1,4);
 
   IF l_temp_version = '10.1' THEN
     OPEN l_packs_cursor FOR
     '
      SELECT ''gc_hosts'',
        ''TRUE'',
        ''''||CNT
      FROM
        (SELECT COUNT(*) CNT
         FROM mgmt_targets
         WHERE target_type = ''host'')
      UNION ALL
      SELECT pack_label ,''UNK'',Decode(
                  Nvl(pack_name,''NO''),''NO'','''',''Y'')
      FROM    mgmt_admin_licenses mal,  
            mgmt_license_Definitions ml
      WHERE mal.pack_name(+)=ml.pack_label
     ';
   ELSE

     OPEN l_packs_cursor FOR
   '
    SELECT ''gc_hosts'',
      ''TRUE'',
      ''''||cnt
    FROM
      (SELECT COUNT(*) cnt
       FROM mgmt_targets
       WHERE target_type = ''host'')
    UNION ALL
    SELECT DISTINCT pack_label AS
    "Pack Name",
      CASE
    WHEN (nvl(no_conf,   0) + nvl(conf,   0)) > 0 THEN
      ''TRUE''
      ELSE
      ''FALSE''
    END, decode(nvl(no_conf,   0) ||'':''|| nvl(conf,   0),''0:0'','''',
     nvl(no_conf,   0) ||'':''|| nvl(conf,   0))
      FROM
      (SELECT mlt.pack_name,
         SUM(decode(nvl(mlc.confirmation, ''N''),''N'',1,    0)) no_conf,
         SUM(decode(nvl(mlc.confirmation, ''N''),''Y'', 1,    0)) conf
       FROM mgmt_licensed_targets mlt,
         mgmt_license_confirmation mlc,
         mgmt_targets mt
       WHERE mlt.target_guid = mlc.target_guid(+)
       AND mlt.target_guid = mt.target_guid
       AND(mt.target_type,    mlt.pack_name) IN
        (SELECT target_type,
           pack_label
         FROM mgmt_license_definitions)
      GROUP BY mlt.pack_name)
    license,
      mgmt_license_definitions mld
    WHERE license.pack_name(+) = mld.pack_label
   ';
   END IF;

    LOOP
      FETCH l_packs_cursor INTO
        l_pack_name, l_isUsed, l_feature_info ;
      EXIT WHEN l_packs_cursor%NOTFOUND;

      write_option_result(l_pack_name ,g_em_version, 
                 1 , 
                 -- pack is used if there is any target that 
                 -- has accepted the license or not
                 l_isUsed,
                 l_feature_info);

   END LOOP;
   put_metric_marker(METRIC_END_MARKER,'feature_usage',null);
   l_end_done   := TRUE;
   UTL_FILE.FFLUSH(g_config_handle);
EXCEPTION
 WHEN OTHERS THEN
   IF NOT l_end_done  THEN
     put_metric_marker(METRIC_END_MARKER,'feature_usage',null);
     UTL_FILE.FFLUSH(g_config_handle);
   END IF;
 --raise any error
   RAISE;
end collect_feature_usage;

/**
Write file header
*/
PROCEDURE  write_file_header
IS
l_par v$instance.PARALLEL%TYPE;
l_db_characterset VARCHAR2(20);
BEGIN
  select value into l_db_characterset from NLS_DATABASE_PARAMETERS where parameter = 'NLS_CHARACTERSET';
  UTL_FILE.PUT_LINE(g_config_handle,'NLS_CHARACTERSET=' || l_db_characterset);
  select PARALLEL into l_par  from v$instance;
  UTL_FILE.PUT( g_config_handle, 'OCM_CROSS_HOST='||l_par );
  UTL_FILE.NEW_LINE(g_config_handle );
END;

/**
Write file footer
*/
PROCEDURE  write_file_footer
IS
BEGIN
   UTL_FILE.PUT_LINE(g_config_handle,'_CCR_EOF_');
END;

/*
Puts the config data into the file
*/
procedure collect_config_metrics(directory_location IN VARCHAR2,
                                 raise_exp BOOLEAN DEFAULT FALSE) IS
l_db_name  v$database.name%TYPE;
BEGIN
  BEGIN
  SELECT  name into l_db_name FROM v$database ;

  BEGIN
    g_config_handle := UTL_FILE.FOPEN(directory_location,l_db_name || 'db.emrep','W',32767);
    EXCEPTION
      WHEN UTL_FILE.INVALID_FILEHANDLE OR 
           UTL_FILE.INVALID_PATH OR
           UTL_FILE.INVALID_OPERATION OR 
           UTL_FILE.WRITE_ERROR THEN
        -- Just bail out, we cannot open or write to the ll file(s)
        RETURN;
  END;

  write_file_header();

  collect_repository_config; 
  collect_oms_names;
  collect_agents;
  collect_beacons;
  collect_utl_privs;
  collect_oms_secure;
  collect_dbAssoc;

  write_file_footer();

  UTL_FILE.FFLUSH(g_config_handle);
  UTL_FILE.FCLOSE(g_config_handle);
  g_config_handle := NULL;

  EXCEPTION
      WHEN OTHERS THEN
      IF UTL_FILE.IS_OPEN (g_config_handle) = TRUE
      THEN
          UTL_FILE.FFLUSH(g_config_handle);
          UTL_FILE.FCLOSE(g_config_handle);
      END IF;
      g_config_handle := null;
      IF (raise_exp)
      THEN
          /* Needs to modify this to log the error into alert log and don't rais any exception*/
          RAISE_APPLICATION_ERROR(-20000,'SQLERRM: ' || SQLERRM || ' SQLCODE: '|| SQLCODE);
      END IF;
  END;
END collect_config_metrics;

/*
Puts the statistics config data into the file
*/
procedure collect_stats_metrics(directory_location IN VARCHAR2,
                                 raise_exp BOOLEAN DEFAULT FALSE) IS
l_db_name  v$database.name%TYPE;
l_par v$instance.PARALLEL%TYPE;
BEGIN
  BEGIN
  SELECT  name into l_db_name FROM v$database ;
  select dbid into g_dbID from v$database;

  BEGIN
    execute immediate 'select version from sysman.mgmt_versions where component_name=''CORE''' into g_em_version;
  EXCEPTION
    WHEN OTHERS THEN
      null;
  END;

  BEGIN
    g_config_handle := UTL_FILE.FOPEN(directory_location,l_db_name || 'db.emrep-stat','W',32767);
    EXCEPTION
      WHEN UTL_FILE.INVALID_FILEHANDLE OR 
           UTL_FILE.INVALID_PATH OR
           UTL_FILE.INVALID_OPERATION OR 
           UTL_FILE.WRITE_ERROR THEN
        -- Just bail out, we cannot open or write to the ll file(s)
        RETURN;
  END;

  write_file_header();
  collect_feature_usage; 
  write_file_footer();
  UTL_FILE.FFLUSH(g_config_handle);
  UTL_FILE.FCLOSE(g_config_handle);
  g_config_handle := NULL;

  EXCEPTION
      WHEN OTHERS THEN
      IF UTL_FILE.IS_OPEN (g_config_handle) = TRUE
      THEN
          UTL_FILE.FFLUSH(g_config_handle);
          UTL_FILE.FCLOSE(g_config_handle);
      END IF;
      g_config_handle := null;
      IF (raise_exp)
      THEN
      /* Needs to modify this to log the error into alert log and don't rais any exception*/
          RAISE_APPLICATION_ERROR(-20000,'SQLERRM: ' || SQLERRM || ' SQLCODE: '|| SQLCODE);
      END IF;
  END;
END collect_stats_metrics; 

END MGMT_EMREP_METRICS;
/
show errors package MGMT_EMREP_METRICS;
