#!/bin/sh
#
# $Header: emll/bin/ebs_createCtxLocXml.sh /main/10 2009/05/22 11:23:51 rlandows Exp $
#
# Script for creating pointer XML document under the iAS Oracle Home 
# that contains the location of the Applications context file. This script
# was created in order to facilitate CCR to discover Oracle Applications 
# Infrastructure targets on Apps middle-tier nodes running on shared 
# filesystem (techstack and/or APPL_TOP).
#
# This script should be run from each Applications middle-tier node after 
# sourcing the Applications environment (APPSORA.env).
#
# The script also accepts the following optional arguments
#  ./ebs_createCtxLocXml [ -silent | -debug ]
# where -silent indicates run in silent mode with no output and always exits
#               with status code 0.
#       -debug  indicates run in debug mode with debug output.
#
# Return codes 
#   0 - Successfully wrote pointer XML
#   1 - Applications environment was not set
#   2 - Applications environment was set but unable to locate context file
#   3 - Location of script does not match the context file
#
# In the future, we should templatize this script and make it part of
# Autoconfig so that this manual step will not be neccessary.
#

#
# Reset the locale to ENGLISH for command line operations.
LC_ALL=C
export LC_ALL

# Initial settings
_binDir=/bin
_usrBinDir=/usr/bin

#
# Set the umask for all operations
umask 077

#
# Define commands that are explicit such that behavior of the commands is
# deterministic
#
if [ -f ${_binDir}/cut ]
then
    CUT=${_binDir}/cut
elif [ -f ${_usrBinDir}/cut ]
then
    CUT=${_usrBinDir}/cut
fi

AWK=${_binDir}/awk
CHMOD=${_binDir}/chmod
ECHO=${_binDir}/echo
DIRNAME=${_usrBinDir}/dirname
TR=${_usrBinDir}/tr
if [ -f ${_binDir}/uname ]
then
    UNAME=${_binDir}/uname
elif [ -f ${_usrBinDir}/uname ]
then
    UNAME=${_usrBinDir}/uname
fi
MKDIRP="${_binDir}/mkdir -p"
MKDIR=${_binDir}/mkdir
PRINTF=${_usrBinDir}/printf
HOSTNAME_CMD=${_binDir}/hostname
CAT=${_binDir}/cat
GREP=${_binDir}/grep
CMPS="${_usrBinDir}/cmp -s"
PLATFORM=`$UNAME | $CUT -f1`

ROOTCHAR=`$DIRNAME $0 | $CUT -b1`
if [ "${ROOTCHAR}" != "/" -a -z "${PWD}" ]
then
    $ECHO "The current shell is not supported for executing commands using relative paths."
    $ECHO "Either execute the command under the bash or ksh shells, or execute the "
    $ECHO "command using the absolute path."
    exit 1
fi

# Usage
usage()
{
  cat <<EOF
Usage: $0 [ -debug or -silent ]

  where:

    -debug  To run in debug mode
    -silent To run in silent mode

EOF
}

printDisclaimer()
{ 
  $ECHO
  $ECHO "POSSIBLE ERROR "
  $ECHO "This script it designed to run from the IAS_ORACLE_HOME "
  $ECHO "on an EBS middle tier.  If it is running from any other "
  $ECHO "location (such as an 8.0.6 or 10.1.2 ORACLE_HOME) possibly "
  $ECHO "by autoconfig, the above message can be ignored.  Otherwise,"
  $ECHO "please verify environment settings and "
  $ECHO "run IAS_ORACLE_HOME/bin/ccr/ebs_createCtxLocXml.sh "
  $ECHO "manually to configure OCM for your EBS middle tier."
}


# Set the debug/silent flag
# Need to call just once as currently only one option is possible
getopts ":d:s" _option
if [ $? -eq 0 ]
then
  case ${_option}
  in 
    d) 
       _debug="TRUE";;
    s) 
       _silent="TRUE";;
    *) 
       $ECHO "Invalid command line option."
       $ECHO ""
       usage
       exit;;
  esac
fi

# Extract the binary directory specification where this script resides.
# The enclosed code will come up with an absolute path.
_liveLinkBinDir=`$DIRNAME $0 | $TR -s '/'`
_liveLinkBinDir=`$ECHO $_liveLinkBinDir | $AWK -f ${_liveLinkBinDir}/strip_path.awk PWD=$PWD`

# Construct the liveLink installation directory root based upon the bin
# directory being a child.
_rootDir=`$DIRNAME ${_liveLinkBinDir}`
_libDir=${_rootDir}/lib

# LiveLink is expanded into a home, so the the parent of the install *IS* the
# Oracle home
_oracleHome=`$DIRNAME ${_rootDir}`
_appsutil_context_dir=${_oracleHome}/appsutil/context

# Perl bin directory 
_perlBinDir=${_rootDir}/engines/${PLATFORM}/perl/bin
_txkLibDir=${_rootDir}/sysman/admin/scripts
_perlLibDir=${_rootDir}/engines/${PLATFORM}/perl/lib/5.8.3

# Debug API
debug()
{
  if [ ! -z "$_debug" ]
  then
   $ECHO "DEBUG: $1"
  fi
}

# Temporary Variable to hold context value
_tmpOAVarVal=

#
# Context File Lookup API
#   Argument 1: Context File Path
#   Argument 2: OA Variable
#   Output: OA Variable Value stored in _tmpOAVarVal
#
getOAVar()
{
  # Launch perl inline only of context file exists and context variable
  # was passed
  if [ ! -z "$1" ] && [ -f $1 ] && [ ! -z "$2" ]
  then
    _tmpOAVarVal=`${_perlBinDir}/perl -I${_txkLibDir} -I${_perlLibDir} -e '
	use TXK::XML(); 
	use TXK::Common(); 
	my $contextFile = $ARGV[0]; 
	my $contextVar = $ARGV[1];
	my $FILE_NAME = "file"; 
	my $xmlObj = TXK::XML->new(); 
	$xmlObj->loadDocument( { $FILE_NAME => $contextFile  } ); 
	my $contextVarVal = $xmlObj->getOAVar("$contextVar"); 
	print "$contextVarVal\n"; 
    ' $1 $2`
  fi
}


# Check if environment was sourced - check if expected environment
# variables are set.
debug "env APPL_TOP: $APPL_TOP"
if [ -z "$APPL_TOP" ]
then
  if [ ! -z "$_silent" ]
  then
    exit 1
  fi
  $ECHO "Your Applications environment is not sourced. "
  $ECHO ""
  $ECHO "Please source your Applications environment before running "
  $ECHO "this script. The Applications environment file can be found "
  $ECHO "at \$APPL_TOP/APPS<s_contextname>.env (e.g. "
  $ECHO "\$APPL_TOP/APPStestsid_testhost.env) "
  printDisclaimer
  exit 1
fi

# Determine context file location
debug "env CONTEXT_FILE: $CONTEXT_FILE"
if [ -z "$CONTEXT_FILE" ]
then
# This means CONTEXT_FILE environment variable does not exist so try
# another route
  _hostname=${HOSTNAME}
  debug "env HOSTNAME: $HOSTNAME"
  if [ -z "$HOSTNAME" ]
  then
    debug "env HOST: $HOST"
    _hostname=${HOST}   
    if [ -z "$HOST" ]
    then
      debug "executing $HOSTNAME_CMD command to get hostname"
      _hostname=`$HOSTNAME_CMD`
    fi
  fi
  debug "env TWO_TASK: $TWO_TASK"
  _twoTask=${TWO_TASK}
  _applTop=${APPL_TOP}  
  if [ ! -z "${_hostname}" ] && [ ! -z "${_twoTask}" ] && [ ! -z "${_applTop}" ] 
  then    
    debug "Found _hostname, _twoTask, _applTop"
    if [ -f ${_applTop}/admin/${_twoTask}_${_hostname}.xml ]
    then
      _contextFileLoc=${_applTop}/admin/${_twoTask}_${_hostname}.xml 
    elif [ -f ${_applTop}/admin/${_twoTask}.xml ]
    then
      _contextFileLoc=${_applTop}/admin/${_twoTask}.xml
    fi
  fi
elif [ -f ${CONTEXT_FILE} ]
 then
   _contextFileLoc=${CONTEXT_FILE}
fi

# Verify if we were able to get a context file location
debug "Context File Location: $_contextFileLoc"
if [ -z "${_contextFileLoc}" ]
then
  if [ ! -z "$_silent" ]
  then
    exit 2
  fi
  $ECHO "This script was unable to locate the Applications context file. "
  $ECHO "Please try setting the CONTEXT_FILE environment variable and "
  $ECHO "re-running the script."
  printDisclaimer

  exit 2;
fi

debug "env CONTEXT_NAME: $CONTEXT_NAME"
_ctxLocXMLFile=${_appsutil_context_dir}/${CONTEXT_NAME}_`$HOSTNAME_CMD`_ctx.xml
if [ -z "$CONTEXT_NAME" ] 
then
 debug "Did not find CONTEXT_NAME. Using ${TWO_TASK}_`$HOSTNAME_CMD`_ctx.xml"
 _ctxLocXMLFile=${_appsutil_context_dir}/${TWO_TASK}_`$HOSTNAME_CMD`_ctx.xml
#elif [ -z  `$ECHO $CONTEXT_NAME | $GREP \`$HOSTNAME_CMD\`` ]
#then 
# debug "CONTEXT_NAME does not contain hostname. Using ${TWO_TASK}_`$HOSTNAME_CMD`_ctx.xml"
# _ctxLocXMLFile=${_appsutil_context_dir}/${CONTEXT_NAME}_`$HOSTNAME_CMD`_ctx.xml
fi

#
# Verify that we are in the right oracle home that matches the current
# environment context
#
# Validate by checking current root directory (i.e. oracle_home) against
# ias_oracle_home or ias_config_home inferred from the context file. 
#
getOAVar "${_contextFileLoc}" "s_weboh_oh"
_iasHome=${_tmpOAVarVal}
getOAVar "${_contextFileLoc}" "s_iASconfig_home"
_iasConfigHome=${_tmpOAVarVal}

# bug 5620076- Changed syntax for /dev/null so it works for AIX/HP as well
$CMPS ${_oracleHome} ${_iasHome} > /dev/null 2>&1
_retCodeIasHome=$?
$CMPS ${_oracleHome}/appsutil ${_iasConfigHome}/appsutil > /dev/null 2>&1
_retCodeIasConfigHome=$?

if [ ${_retCodeIasHome} -ne 0 -a ${_retCodeIasConfigHome} -ne 0 ]
then
  if [ ! -z "$_silent" ]
  then
    exit 3
  fi
  $ECHO "This script was run from a location that does not match the  "
  $ECHO "environment settings."
  printDisclaimer
  exit 3;
fi


# Write XML file with context file location to 
# $ORACLE_HOME/appsutil/context directory
if [ ! -d ${_appsutil_context_dir} ] 
then
  $MKDIRP ${_appsutil_context_dir}
  $CHMOD 740 ${_appsutil_context_dir}
fi

$ECHO "<oa_context>" > ${_ctxLocXMLFile}
$ECHO "  <oa_context_file_loc oa_var=\"s_contextfile\">${_contextFileLoc}</oa_context_file_loc>" >> ${_ctxLocXMLFile}
$ECHO "</oa_context>" >> ${_ctxLocXMLFile}
$CHMOD 775 ${_ctxLocXMLFile}

# We're done
if [ -z "$_silent" ]
then
  $ECHO "Successfully wrote Applications context file location to file: "
  $ECHO "${_ctxLocXMLFile}"
fi
exit 0
