# $Header: getdbinfo.pl 18-oct-2007.20:54:37 ndutko Exp $
#
# getdbinfo.pl
# 
# Copyright (c) 2006, 2007, Oracle. All rights reserved.  
#
#    NAME
#      getdbinfo.pl - returns the GUID for a database given DBNAME
#
#    DESCRIPTION
#      Returns the GUID for the most recent uploaded database configuration
#      element specified by DBNAME as an argument.
#
#    NOTES
#      Relies on the installation of the OCM collector and having performed
#      a configuration collection. 
#
#    MODIFIED   (MM/DD/YY)
#    ndutko      10/19/07 - XbranchMerge ndutko_bug-6467643 from st_emll_10.2.7
#    ndutko      10/18/07 - Use CCR_CONFIG_HOME for location of config tree
#    pparida     03/06/07 - Fix bug 5918617: Check file existence of
#                           $collectionFullFileSpec in getTargetGUID()
#    ndutko      01/22/07 - Do not return a GUID if the collector is configured
#                           for disconnected configurations
#    ndutko      07/11/06 - Change to use OCM_HOME env variable 
#    ndutko      06/27/06 - Perl script that returns the GUID for a DBNAME 
#    ndutko      06/27/06 - Creation
# 

# Given the name of the database passed as an argument, return the GUID 
# computed in the last database collection that was performed. 
#
# This implementation relies on the review files containing the contents of
# the targets collected.
#
# Usage:
#      getdbinfo.pl <dbname>
#
#   Arguments: 
#            <dbname>    - dbname of the instance in the home that is of
#                          interest
#
#   Returns:
#            GUID        - 128 bit GUID used to identify the configuration 
#                         for the database instance.
#
#   Errors:
#            0            - Success
#            1            - targetMap does not exist - no configuration
#                           collection information is available.
#            2            - no read access to the configuration data. Not the
#                           ORACLE_HOME owner.
#            3            - dbname not specified.
#            4            - OCM configuration configured for disconnected
#                           collections. No GUID available.

use strict;
use English;
use File::Spec;
use ias::simpleXPath;

use constant SUCCESS => 0;
use constant NO_SUCH_FILE => 1;
use constant NO_READ_ACCESS => 2;
use constant DBNAME_NOT_SPECIFIED => 3;
use constant DISCONNECTED_MODE => 4;

# 
# Determines if the collector was configured for disconnected mode.
#
sub isDisconnectedConfig()
{
    my $defaultConfig = File::Spec->catfile( $ENV{CCR_CONFIG_HOME}, "config",
                            "collector.properties" );

    # Default the connection mode to connected.
    my $disconnectedMode = 0;

    if ( -f $defaultConfig )
    {
        open CONFIG, "<$defaultConfig"
            or die "Unable to access collector configuration file, $!\n";
        while (<CONFIG>)
        {
           my $configRec = $_;
           chomp $configRec;

           if ( $configRec =~ /^\s*ccr\.disconnected\s*=\s*(.*)?\s*$/ )
           {
               if ( $1 =~ /true/i )
               {
                   $disconnectedMode = 1;
               }
           }
        }
        close CONFIG;
    }
    return $disconnectedMode;
}

#
# Given the database DBName, get the name of the oracle_database target that
# matches.
sub getDBTargetName($)
{
    my $dbname = uc(shift);
    my $targetXML = 
        File::Spec->catfile($ENV{CCR_CONFIG_HOME}, "config", "default", "targets.xml");
        
    if ( ! -f $targetXML )
    {
        exit NO_SUCH_FILE;
    }
    elsif ( ! -r $targetXML )
    {
        warn "Unable to read the list of discovered OCM targets.\n";
        warn $CHILD_ERROR,"\n";
        exit NO_READ_ACCESS;
    }
    else
    {   
        # Get the nodes for all the database targets.     
        my @dbRefNodes = simpleXPathQuery( $targetXML,
                                      'Targets/Target[@TYPE="oracle_database"]');
                                      
          # For each database target node, find the one that has a target
          # property that matches the DBName value specified.
        foreach my $refDbNode (@dbRefNodes)
        {
            my $databaseName = $refDbNode->{'NAME'};
            my @dbNameRefNodes = simpleXPathQuery( $targetXML,
                         'Targets/Target[@TYPE="oracle_database"][@NAME="'.$databaseName.
                         '"]/Property[@NAME="DBName"][@VALUE="'.$dbname.'"]');
                         
             # A match is found is more than one result is returned. In that
             # case, return the database name found.
             if (scalar(@dbNameRefNodes) > 0)
             {
                 return $databaseName;
             }
        }
    }

    # No database target was found.    
    return;
}    

#
# Given the database name, type oracle_database, get the review file from
# the state/review directory and extract the collection review file name
# 
sub getCollectionFNM($)
{
    my $databaseName = shift;
    my $targetMAP = 
        File::Spec->catfile($ENV{CCR_CONFIG_HOME}, "state", "review", "targetMap.xml");

    # Insure that accessibility is permitted for the targetMap.        
    if ( ! -f $targetMAP )
    {
        exit NO_SUCH_FILE;
    }
    elsif ( ! -r $targetMAP )
    {
        warn "Unable to read the OCM collection results table of contents.\n";
        warn $CHILD_ERROR,"\n";
        exit NO_READ_ACCESS;
    }
    else
    {
        # Get the collection Nodes for the database target. within the 
        # collection node resides the file specification for the file
        # that contains the GUID.
        my @dbCollectionRefs = simpleXPathQuery( $targetMAP,
                                      'Targets/Target[@name="'.$databaseName.
                                      '"][@type="Database Instance"]/Collection');
        if (scalar(@dbCollectionRefs) > 0)
        {
            foreach my $dbCollectionRef (@dbCollectionRefs)
            {
                return $dbCollectionRef->{'file'};
            }
        }
    }
}

#
# Given the collection name, extract the target GUID from the MGMT_TARGETS
# ROWSET.
#
sub getTargetGUID($)
{
    my $collectionBaseFile = shift;
    my $collectionFullFileSpec = File::Spec->catfile($ENV{CCR_CONFIG_HOME},
                        "state","review",$collectionBaseFile);

    if ( ! -f $collectionFullFileSpec )
    {
        exit NO_SUCH_FILE;
    }                        
    elsif ( ! -r $collectionFullFileSpec )
    {
        warn "Unable to read collection details information.\n";
        warn $CHILD_ERROR,"\n";
        exit NO_READ_ACCESS;
    }
    
    my @GUIDS = simpleXPathQueryForText( $collectionFullFileSpec,
                                      'UPLOAD/ROWSET[@TABLE="MGMT_TARGETS"]/ROW/TARGET_GUID');
    if (scalar(@GUIDS) > 0)
    {
        return $GUIDS[0];
    }    
}

# This is the main code line for this request.

# Get the collector.properties file contents and return if this is a
# disconnected configuration collection.
if (isDisconnectedConfig)
{
    exit DISCONNECTED_MODE;
}

my $dbname = shift;
if (!defined $dbname)
{
    warn "The Database DBNAME was not specified.\n";
    exit DBNAME_NOT_SPECIFIED;
}

my $databaseName = getDBTargetName($dbname);
my $collectionFnm = getCollectionFNM($databaseName);
print getTargetGUID($collectionFnm),"\n";
exit SUCCESS;
