#
# $Header: emll/sysman/admin/discover/fmw/Targets.pm /main/2 2010/07/30 10:48:17 jsmoler Exp $
#
# Targets.pm
# 
# Copyright (c) 2010, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      Targets.pm
#
#    DESCRIPTION
#      Routines to manage Fusion Middleware discovered targets
#
#    NOTES
#      <other useful comments, qualifications, etc.>
#
#    MODIFIED   (MM/DD/YY)
#    jsmoler     07/26/10 - ON_HOST target attribute
#    jsmoler     04/21/10 - Creation
# 

package fmw::Targets;

use strict;
use Exporter 'import';

our @EXPORT = qw(
    createTarget
    getTargets
    getTarget
    removeTarget
    getTargetType
    setTargetType
    getTargetName
    setTargetName
    getTargetDisplayName
    setTargetDisplayName
    getTargetHost
    setTargetHost
    getTargetProperties
    getTargetProperty
    setTargetProperty
    removeTargetProperty
);

my %targetTypes;

#
# Creates a new target and adds it to the discovered targets list.
#
# Arguments
#   targetType: Target type
#   targetName: Full target name
#   displayName (optional): Target display name
#   props (optional): Reference to a hash of target properties
#
# Returns: A reference to the created target
#
sub createTarget
{
    my ( $targetType, $targetName, $displayName, $props ) = @_;
    if (!defined($props))
    {
        $props = {};
    }
    my $target = { 'TYPE' => $targetType,
                   'NAME' => $targetName,
                   'DISPLAY_NAME' => $displayName,
                   'properties' => $props };
    my $targets = $targetTypes{$targetType};
    if (!defined($targets))
    {
        $targets = {};
        $targetTypes{$targetType} = $targets;
    }
    $$targets{$targetName} = $target;
    return $target;
}

#
# Gets a list of discovered targets, optionally filtered by target type.
# 
# Arguments
#   targetType (optional): If specified, only targets of this type will be returned
#
# Returns: A list of targets
#
sub getTargets
{
    my ( $targetType ) = @_;
    if (!defined($targetType))
    {
        my @allTargets;
        for my $targets (values %targetTypes)
        {
            push(@allTargets, values %{$targets});
        }
        return @allTargets;
    }
    my $targets = $targetTypes{$targetType};
    if (!defined($targets))
    {
        return ();
    }
    return values %{$targets};
}

#
# Gets a specific target from the discovered list.
#
# Arguments
#   targetType: Type of the target to search for
#   targetName: Name of the target to search for
#
# Returns: A reference to the target, if it is found in the discovered list
#
sub getTarget
{
    my ( $targetType, $targetName ) = @_;
    my $targets = $targetTypes{$targetType};
    return $$targets{$targetName};
}

#
# Removes a specific target from the discovered list.
#
# Arguments
#   targetType: Type of the target to remove
#   targetName: Name of the target to remove
#
sub removeTarget
{
    my ( $targetType, $targetName ) = @_;
    my $targets = $targetTypes{$targetType};
    delete $$targets{$targetName};
}

#
# Gets the type of a target.
#
# Arguments
#   target: Reference to a target
#
# Returns: The type of the target
#
sub getTargetType
{
    my ( $target ) = @_;
    return $$target{'TYPE'};
}

#
# Sets the type of a target.
#
# Arguments
#   target: Reference to a target
#   targetType: The new target type
#
sub setTargetType
{
    my ( $target, $targetType ) = @_;
    $$target{'TYPE'} = $targetType;
}

#
# Gets the name of a target.
#
# Arguments
#   target: Reference to a target
#
# Returns: The name of the target
#
sub getTargetName
{
    my ( $target ) = @_;
    return $$target{'NAME'};
}

#
# Sets the name of a target.
#
# Arguments
#   target: Reference to a target
#   targetName: The new target name
#
sub setTargetName
{
    my ( $target, $targetName ) = @_;
    $$target{'NAME'} = $targetName;
}

#
# Gets the display name of a target.
#
# Arguments
#   target: Reference to a target
#
# Returns: The display name of the target
#
sub getTargetDisplayName
{
    my ( $target ) = @_;
    return $$target{'DISPLAY_NAME'};
}

#
# Sets the display name of a target.
#
# Arguments
#   target: Reference to a target
#   displayName: The new target display name
#
sub setTargetDisplayName
{
    my ( $target, $displayName ) = @_;
    $$target{'DISPLAY_NAME'} = $displayName;
}
#
# Gets the host of a target.
#
# Arguments
#   target: Reference to a target
#
# Returns: The host of the target
#
sub getTargetHost
{
    my ( $target ) = @_;
    return $$target{'ON_HOST'};
}

#
# Sets the host of a target.
#
# Arguments
#   target: Reference to a target
#   host: The new target host
#
sub setTargetHost
{
    my ( $target, $host ) = @_;
    $$target{'ON_HOST'} = $host;
}

#
# Gets a hash containing the properties of a target.
#
# Arguments
#   target: Reference to a target
#
# Returns: A reference to a hash containing the properties of the target. The hash may be modified.
#
sub getTargetProperties
{
    my ( $target ) = @_;
    return $$target{'properties'};
}

#
# Gets a specific target property.
#
# Arguments
#   target: Reference to a target
#   propName: Name of the property to look up
#
# Returns: The value of the given property for the given target.
#
sub getTargetProperty
{
    my ( $target, $propName ) = @_;
    return $$target{'properties'}{$propName};
}

#
# Sets the value of a target property.
#
# Arguments
#   target: Reference to a target
#   propName: Name of the property to set
#   propVal: The new value of the property
#
sub setTargetProperty
{
    my ( $target, $propName, $propVal ) = @_;
    $$target{'properties'}{$propName} = $propVal;
}

#
# Removes a target property from a target.
#
# Arguments
#   target: Reference to a target
#   propName: Name of the property to remove
#
sub removeTargetProperty
{
    my ( $target, $propName ) = @_;
    delete $$target{'properties'}{$propName};
}

return 1;
