#!/usr/local/bin/perl
# 
#
# oracle_custom_apps.pl
# 
# Copyright (c) 2010, 2011, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      oracle_custom_apps.pl - Generate XML stream describing custom Oracle application targets
#
#    DESCRIPTION
#      Uses the targets.xml stream from FMW discovery results, customizes FMW targets as desired  
#      and returns the resulting XML stream.
#
#    NOTES
#      This is a plugin script which will be invoked by FMW discovery 
#
# 

use strict;
use File::Spec;
use fmw::Targets;
use ias::simpleXPath;

BEGIN
{
    require "$ENV{'CCR_HOME'}/sysman/admin/scripts/emd_common.pl";
}

# WebLogic target types
my $j2eeAppTargetType = 'j2ee_application';

my ($emdRoot,$hostName) = @ARGV;

my $middlewareHome = $ENV{'BEA_HOME'};
if (!defined($middlewareHome))
{
  EMD_PERL_DEBUG("No BEA_HOME defined, aborting fusion middleware application discovery");
  exit;
}

my @wl_j2ee_apps = getTargets($j2eeAppTargetType);
my $applicationsFilePath = "$emdRoot/sysman/admin/discover/fmw/apps.xml";
my %oracle_home_components_files;
my $applicationsFile;

eval {$applicationsFile = ias::simpleXPath::parseFile($applicationsFilePath); };
if ($@) {
  EMD_PERL_DEBUG("File $applicationsFilePath is not well-formed");
  exit;
}

foreach my $wl_j2ee_app (@wl_j2ee_apps)
{
  my $wl_j2ee_app_name = getTargetProperty($wl_j2ee_app, 'appName');
  
  # Here we strip off version number appended to the app name (if any)
  if ($wl_j2ee_app_name =~/#(\d+|\.+|\d+)+$/)
  {
    $wl_j2ee_app_name = substr $wl_j2ee_app_name, 0, rindex($wl_j2ee_app_name, '#');
  }
  my @applications = ias::simpleXPath::queryForNodes($applicationsFile, 'Applications/Application[@name="' . $wl_j2ee_app_name . '"]');
  if(defined(@applications))
  {
    foreach my $application (@applications)
    {
      my $name = ((ias::simpleXPath::queryForAttributes($application, 'Application'))[0])->{'name'};
      my $type = ((ias::simpleXPath::queryForAttributes($application, 'Application'))[0])->{'type'};
      my @components = ias::simpleXPath::queryForNodes($application, 'Application/Components/*');
      
      if(defined(@components))
      {
        foreach my $component (@components)
        {
          # We parse comps.xml once and only if there's one or more components found 
          if (!defined(%oracle_home_components_files))
          {
            %oracle_home_components_files = %{get_oracle_home_components_files()};
          }
          
          my $component_name = ((ias::simpleXPath::queryForAttributes($component, 'Component'))[0])->{'name'};
          
          my $oracle_home_component_found;
          if (defined(%oracle_home_components_files))
          {
            while(my ($oracle_home, $oracle_home_components_file) = each(%oracle_home_components_files))
            { 
              my @oracle_home_top_list_components = ias::simpleXPath::queryForNodes($oracle_home_components_file, 'PRD_LIST/TL_LIST/COMP[@NAME="' . $component_name . '"]');
              my @oracle_home_comp_list_components = ias::simpleXPath::queryForNodes($oracle_home_components_file, 'PRD_LIST/COMP_LIST/COMP[@NAME="' . $component_name . '"]');
              my @oracle_home_components = (@oracle_home_top_list_components, @oracle_home_comp_list_components);
              if (defined(@oracle_home_components))
              {
                foreach my $oracle_home_component (@oracle_home_components)
                {
                  $oracle_home_component_found = 1;
                
                  # Change target type
                  setTargetType($wl_j2ee_app, $type);
                
                  # Set Oracle home
                  setTargetProperty($wl_j2ee_app, 'OcmProxyOracleHome', $oracle_home);
                
                  # Query component attributes, set on the target
                  
                  # Determine component version
                  my $version_value = get_version_value($oracle_home_component, $oracle_home_components_file);
                  if (defined($version_value))
                  {
                    setTargetProperty($wl_j2ee_app, 'version_value', $version_value);
                  }
                 
                  clone_target($wl_j2ee_app, $type);
 
                  # Reset each(%oracle_home_components_files)
                  keys %oracle_home_components_files;
                  
                  # Only one component with the specified name is expected
                  last;
                }
              }
              # No need to process other files if component is found
              last if (defined($oracle_home_component_found));
            }
          }
          # No need to process other components associated with the application if one is found
          last if (defined($oracle_home_component_found));
        }
      }
    }
  }
}

# subroutine get_oracle_homes
# Determines the location of all oracle homes under widdleware home
# Returns reference to the array
sub get_oracle_homes
{
  my @oracle_homes;
  
  if (opendir(MW_HOME, $middlewareHome))
  {
    while (my $dirEntry = readdir(MW_HOME))
    {
      # skip "." and ".." directories
      next if ($dirEntry eq "." || $dirEntry eq "..");
      # check in directories only
      next unless (-d "$middlewareHome/$dirEntry");
      # look for the comps.xml file; add to OH list if it exists
      if (-f "$middlewareHome/$dirEntry/inventory/ContentsXML/comps.xml")
      {
        push (@oracle_homes, File::Spec->catfile("$middlewareHome", "$dirEntry"));
      }
    }
    closedir(MW_HOME);
  }

  return \@oracle_homes;
}

# subroutine get_oracle_home_components_file_path
# Returns the file path to the comps.xml file for the given Oracle home
sub get_oracle_home_components_file_path
{
  my $oracle_home = @_[0];
  return "$oracle_home/inventory/ContentsXML/comps.xml";
}

# subroutine get_oracle_home_components_file
# Returns parsed comps.xml file for the given Oracle home
sub get_oracle_home_components_file
{
  my $components_file_path = @_[0];
  return ias::simpleXPath::parseFile($components_file_path);
}

# subroutine get_oracle_home_components_files
# Returns reference to the hash map of Oracle home to parsed comps.xml files
sub get_oracle_home_components_files
{
  my %oracle_home_components_files = ();
  
  my @oracle_homes = @{get_oracle_homes()};
  if(defined(@oracle_homes))
  {
    foreach my $oracle_home (@oracle_homes)
    {
      my $oracle_home_components_file_path = get_oracle_home_components_file_path($oracle_home);
      my $oracle_home_components_file = get_oracle_home_components_file($oracle_home_components_file_path);
      $oracle_home_components_files{$oracle_home} = $oracle_home_components_file;
    }
  }
  
  return \%oracle_home_components_files;
}

# subroutine get_version_value
# Obtains component version value
sub get_version_value
{
    my $oracle_home_component = @_[0];
    my $oracle_home_components_file = @_[1];
    my $component_name = ((ias::simpleXPath::queryForAttributes($oracle_home_component, 'COMP'))[0])->{'NAME'};
    my $component_version = ((ias::simpleXPath::queryForAttributes($oracle_home_component, 'COMP'))[0])->{'VER'};
    
    # If there are patches, correct version is specified by one of them
    my @oracle_home_patches = ias::simpleXPath::queryForNodes($oracle_home_components_file, 'PRD_LIST/TL_LIST/PATCHSET/PS_PATCH[@NAME="' . $component_name . '"][@BASE_VER="' . $component_version . '"]');
    if (defined(@oracle_home_patches))
    {
        foreach my $oracle_home_patch (@oracle_home_patches)
        {
            my $patch_component_version = ((ias::simpleXPath::queryForAttributes($oracle_home_patch, 'PS_PATCH'))[0])->{'VER'};
            $component_version = $patch_component_version;
        }
    }

    
    return $component_version;
}


# subroutine clone_target
# Clones target if its type contains multiple pipe delimited types
sub clone_target
{
    my $wl_j2ee_app = @_[0]; 

    my @types = split(/\|/, @_[1]);
    if (defined(@types))
    {
	my $size = scalar(@types);
        if ($size > 1) {
           
           setTargetType($wl_j2ee_app, $types[0]);
           splice(@types, 0, 1);
           foreach my $type (@types)
           {
               createTarget($type, getTargetName($wl_j2ee_app), getTargetDisplayName($wl_j2ee_app),
                                 {
                                     'appName' => getTargetProperty($wl_j2ee_app, 'appName'),
                                     'version_value' => getTargetProperty($wl_j2ee_app, 'version_value'),
                                     'version' => getTargetProperty($wl_j2ee_app, 'version'),
                                     'serverTarget' => getTargetProperty($wl_j2ee_app, 'serverTarget'),
                                     'domainHome' => getTargetProperty($wl_j2ee_app, 'domainHome'),
                                     'OcmProxyOracleHome' => getTargetProperty($wl_j2ee_app, 'OcmProxyOracleHome'),
                                     'adminServerHost' => getTargetProperty($wl_j2ee_app, 'adminServerHost'),
                                     'appVersion' => getTargetProperty($wl_j2ee_app, 'appVersion')
                                 });
           }
        }
    }
}
