#
# Parse OUI XML files to get info for the oracle_home target
#
use File::Basename;
use File::stat;
use File::Spec();
use File::Spec::Functions;
use XML::Parser;
use Config;

package main;

my ($oracleHome, $hostName) = @ARGV;

my %homes;
my $Hfound = 0;

my $ouiInvLoc="";

my $OSNAME = $Config{'osname'};
my $IsWin32 = ($OSNAME eq 'MSWin32');

if (exists($ENV{LL_ORACLE_HOME}))
{
  $oracleHome = dirname($oracleHome);
}

# NESTED home search
# we now use BEA_HOME, this is supposed to be changed at some point to MW_HOME
#
if (defined ($ENV{BEA_HOME}))
{
  # we will recurse search for {BEA_HOME}/*/inventory/ContentsXML/comps.xml;
  # if one exists the parent should be located in the central inventory
  my $mwHome = $ENV{BEA_HOME};
  if (opendir(MW_HOME, $mwHome))
  {
    while (my $dirEntry = readdir(MW_HOME))
    {
      # skip "." and ".." directories
      next if ($dirEntry eq "." || $dirEntry eq "..");
      # check in directories only
      next unless (-d "$mwHome/$dirEntry");
      # look for the comps.xml file; add to OH list if it exists
      if (-f "$mwHome/$dirEntry/inventory/ContentsXML/comps.xml")
      {
        discoverHome("$mwHome/$dirEntry", 1);
      }
    }
    closedir(MW_HOME);
  }
}
# discover the incoming OH IFF comps.xml exists && BEA_HOME not set
elsif (-f "$oracleHome/inventory/ContentsXML/comps.xml")
{
  discoverHome($oracleHome, 0);
}

sub discoverHome
{
  my ($findHome, $isBeaHome) = @_;

  my $p1 = new XML::Parser(ErrorContext =>2);
  $p1->setHandlers(Start => \&handle_inventory_start);

  if($IsWin32)
  {
    eval 'use Win32::TieRegistry';
    $Registry->Delimiter("/");
    $ouiInvLoc = $Registry->{"HKEY_LOCAL_MACHINE/Software/Oracle/inst_loc"};
    chomp($ouiInvLoc);
  }
  else
  {
    my @ouiLocFileList = ($findHome."/oraInst.loc", "/var/opt/oracle/oraInst.loc", "/etc/oraInst.loc");
    my $ouiInstFile = "";

    # using an environment variable?
    if (defined $ENV{ORAINST_LOC})
    {
      $ouiInvLoc = $ENV{ORAINST_LOC};
    }
    else
    {
      # Locate the OUI base inventory file
      foreach $ouiFile (@ouiLocFileList) 
      {
        if ( -e $ouiFile ) 
        {
          $ouiInstFile = $ouiFile;
          last;
        }
      }
      if ($ouiInstFile ne "")
      {
        if (open(OUIFILE, "<$ouiInstFile"))
        {
          while (<OUIFILE>)
          {
            if (/^\s*inventory_loc/)
            {
              (undef,$ouiInvLoc) = split(/=/, $_);
            }
          }
          close OUIFILE;
        }
      }
      chomp $ouiInvLoc;
    }
    # fall through - use user's home as base for inventory
    if ($ouiInvLoc eq "")
    {
      $ouiInvLoc = $ENV{HOME}."/oraInventory";
    }
  }
  # split on comment character
  ($ouiInvLoc,undef) = split(/#/, $ouiInvLoc);

  $ouiInvLoc = $ouiInvLoc . "/ContentsXML/inventory.xml";

  # parse the central inventory file so we can find the info for this home
  # what if this home isn't in the inventory we're parsing??? (fake it!)
  #
  my $homeFound = 0;
  my $ohName;
  if (-e $ouiInvLoc)
  {
    $p1->parsefile($ouiInvLoc);
    my ($home, $ref, $homeLoc) = undef;
    my ($sb0) = stat($findHome);

    foreach $home(keys %homes)
    {
      $ref = $homes{$home};
      if (-e ($ref->{_LOC}))
      {
        if ($IsWin32)
        {
          if (lc(Win32::GetShortPathName($ref->{_LOC})) eq lc(Win32::GetShortPathName($findHome)))
          {
            $homeFound = 1;
          }
        }
        else
        {
          my ($sb1) = stat($ref->{_LOC});
          # if the device and inodes match, we found a match (dealing with symlinks)
          if (defined($sb1->dev) && ($sb1->dev == $sb0->dev) && ($sb1->ino == $sb0->ino))
          {
            $homeFound = 1;
          }
        }
        if ($homeFound == 1)
        {
          $ohName = ((defined($ref->{_NAME}) && length($ref->{_NAME}) > 0) ? $ref->{_NAME} : $findHome);
          print "<Target TYPE=\"oracle_home\" NAME=\"".($isBeaHome ? $findHome : $ohName)."\""
                .($isBeaHome ? " ON_HOST=\"$hostName\"": "")." VERSION=\"1.0\">\n";
          if ($isBeaHome)
          {
            print "  <Property NAME=\"OcmProxyOracleHome\" VALUE=\"$ref->{_LOC}\" />\n";
          }
          print "  <Property NAME=\"host\" VALUE=\"" . $hostName . "\" />\n";
          print "  <Property NAME=\"Location\" VALUE=\"" . $ref->{_LOC} . "\" />\n";
          print "  <Property NAME=\"HomeName\" VALUE=\"" . $ohName . "\" />\n";
          print "  <Property NAME=\"Type\" VALUE=\"" . $ref->{_TYPE} . "\" />\n";
          print "  <Property NAME=\"Index\" VALUE=\"0\" />\n";
          print "  <Property NAME=\"OUI_Platform\" VALUE=\"" . get_oui_plat($ref->{_LOC}) . "\" />\n";
          print "  <Property NAME=\"Clonable\" VALUE=\"" . $ref->{_CLONABLE} . "\" />\n";
          print "</Target>\n";
          last;
        }
      }
    }
  }

  # either there's no inventory file found, or the current home is not listed there
  # so let's look in the Registry if we're on Windows!
  if (($IsWin32) && ($homeFound == 0))
  {
    my ($oracleRegBase,$oracleRegSub,$homeRegInfo);
    $oracleRegBase = $Registry->{"HKEY_LOCAL_MACHINE/Software/Oracle/"};
    foreach (keys %$oracleRegBase)
    {
      if (($_ =~ m|^HOME|) || ($_ =~ m|^KEY_|))
      {
        $homeRegInfo = $oracleRegBase->{$_};
        if (lc(Win32::GetShortPathName($homeRegInfo->{ORACLE_HOME})) eq lc(Win32::GetShortPathName($findHome)))
        {
          print "<Target TYPE=\"oracle_home\" NAME=\"" . $homeRegInfo->{ORACLE_HOME_NAME} . "\""
                .($isBeaHome ? " ON_HOST=\"$hostName\"": "")." VERSION=\"1.0\">\n";
          if ($isBeaHome)
          {
            print "  <Property NAME=\"OcmProxyOracleHome\" VALUE=\"$homeRegInfo->{_LOC}\" />\n";
          }
          print "  <Property NAME=\"host\" VALUE=\"" . $hostName . "\" />\n";
          print "  <Property NAME=\"Location\" VALUE=\"" . $findHome . "\" />\n";
          print "  <Property NAME=\"HomeName\" VALUE=\"" . $homeRegInfo->{ORACLE_HOME_NAME} . "\" />\n";
          print "  <Property NAME=\"Type\" VALUE=\"O\" />\n";  # Assume HOME_TYPE_ORACLE
          print "  <Property NAME=\"Index\" VALUE=\"" . $homeRegInfo->{ID} . "\" />\n";
          print "  <Property NAME=\"OUI_Platform\" VALUE=\"" . get_oui_plat($findHome) . "\" />\n";
          print "  <Property NAME=\"Clonable\" VALUE=\"Y\" />\n";
          print "</Target>\n";
          $homeFound = 1;
          last;
        }
      }
    }
  }

  # either there's no inventory file found, or the current home is not listed there
  if ($homeFound == 0)
  {
    # use path for name & use known info
    print "<Target TYPE=\"oracle_home\" NAME=\"" . $findHome . "\""
          .($isBeaHome ? " ON_HOST=\"$hostName\"": "")." VERSION=\"1.0\">\n";
    if ($isBeaHome)
    {
      print "  <Property NAME=\"OcmProxyOracleHome\" VALUE=\"$findHome\" />\n";
    }
    print "  <Property NAME=\"host\" VALUE=\"" . $hostName . "\" />\n";
    print "  <Property NAME=\"Location\" VALUE=\"" . $findHome . "\" />\n";
    print "  <Property NAME=\"HomeName\" VALUE=\"Unknown\" />\n";
    print "  <Property NAME=\"Type\" VALUE=\"O\" />\n";  # Assume HOME_TYPE_ORACLE
    print "  <Property NAME=\"Index\" VALUE=\"0\" />\n";
    print "  <Property NAME=\"OUI_Platform\" VALUE=\"" . get_oui_plat($findHome) . "\" />\n";
    print "  <Property NAME=\"Clonable\" VALUE=\"Y\" />\n";
    print "</Target>\n";
  }
}

1;

sub handle_inventory_start
{
  my $p = shift;
  my $el = shift;

  if ($el eq "HOME")
  {
    my $homeInf = $homes{$Hfound};
    if (not defined($homeInf))
    {
      $homes{$Hfound} = $homeInf = new HomeInfo;
      $Hfound++;
    }

    $homeInf->{_IDX} = $Hfound-1;
    $homeInf->{_CLONABLE} = "Y";

    my ($att, $val);
    while (@_)
    {
      $att = shift;
      $val = shift;
      if ($att eq "NAME")
      {
        $homeInf->{_NAME} = $val;
      }
      elsif ($att eq "LOC")
      {
        $homeInf->{_LOC} = $val;
      }
      elsif ($att eq "TYPE")
      {
        $homeInf->{_TYPE} = $val;
      }
    }
  }
}

my %OHPropXML;
my $inOHPropAruID = "FALSE";

# Expat built-in encodings are UTF-8, iso-8859-1 [west european], UTF-16, and US-ASCII
# the remainder are from the various .enc files included with our perl distribution

my @encTable = (
    'US-ASCII',
    'iso-8859-1',        # [west european, a.k.a. Latin1]
    'windows-1252',      # [win Latin1 (8859-1 superset)]
    'iso-8859-2',        # [east european]
    'windows-1250',      # [win Latin2 (8859-2 superset)]
    'iso-8859-3',        # [south european]
    'iso-8859-4',        # [north european]
    'iso-8859-5',        # [cyrillic]
    'iso-8859-7',        # [greek]
    'iso-8859-8',        # [hebrew]
    'iso-8859-9',        # [turkish]
    'UTF-8',
    'UTF-16',
    'big5',              # [traditional chinese]
    'euc-kr',            # [extended unix code for korean]
    'x-euc-jp-jisx0221', # [extended unix code for japanese, JIS X0221]
    'x-euc-jp-unicode',  # [extended unix code for japanese, unicode]
    'x-sjis-cp932',      # [shift-jis win codepage 932]
    'x-sjis-jdk117',     # [shift-jis JDK117]
    'x-sjis-jisx0221',   # [shift-jis JIS X0221]
    'x-sjis-unicode',    # [shift-jis unicode]
  );

sub get_oui_plat
{
  # perhaps we can read the install.platform in the current
  # oracle_home; the ID=<nn> nv pair holds what we want!
  # or, in ContentsXML/oraclehomeproperties.xml, the ARU_ID
  # information holds this same value...
  my $oracleHome = shift;

  
  my $infoFile = $oracleHome . "/install.platform";
  my $platID="";

  $platID = getaruIDfromPropXML($oracleHome);
  if(!$platID)
  {
   if (-e $infoFile)
   {
     if (open(INFO, "<$infoFile"))
     {
       while (<INFO>)
       {
         if (/ID=/)
         {
           (undef, $platID) = split(/=/, $_);
           chomp $platID;
           return $platID;
         }
       }
     }
   }
   else
   {
     my $osName = $Config{'osname'};
     my $osArch = $Config{'archname'};
     my $osVers = $Config{'osvers'};

     if ($osName eq "MSWin32")
     {
       return 912;
     }
     elsif ($osName eq "linux")
     {
       return 46;
     }
     elsif ($osName eq "solaris")
     {
       return 453;
     }
   }
  }
  return $platID;
}

# Function to parse oraclehomeproperties.xml and extract ARU_ID information

sub getaruIDfromPropXML
{
  my $oracleHome = shift ;
  %OHPropXML = ();
  my ($OHPropFile) = File::Spec->catfile( $oracleHome,"inventory","ContentsXML","oraclehomeproperties.xml") ;
  if(open(OHFile, $OHPropFile))
  {
    close OHFile;
    my $p = new XML::Parser(ErrorContext=>2);
    
    $p->setHandlers(Start=> \&OHProp_start,
                    End=> \&OHProp_end,
                    Char=> \&OHProp_char);
    
    eval '$p->parsefile($OHPropFile)';
    
    if($@)
    { 
      foreach $_ (@encTable)
      {
        %OHPropXML = ();
        
        eval '$p->parsefile($OHPropFile,ProtocolEncoding=> $_ )';
  
        if(!$@)
        {
          last;
        }
      }
    }
    if($@)
    {
      return 0;
    }
  }
  else
  {
    return 0;
  }
  
  if(($OHPropXML{_ARU_ID}) !~ m/^\d+$/)
  {
    return 0;
  }
  return $OHPropXML{_ARU_ID} ;
	
}

# Start handler for getaruIDfromPropXML
sub OHProp_start
{
  my $p = shift;
  my $el = shift;
  if($el eq "ARU_ID")
  {
    $inOHPropAruID = "TRUE";
    $OHPropXML{_ARU_ID} = undef;
  }
}

#End handler for getaruIDfromPropXML
sub OHProp_end
{
  my $p = shift;
  my $el = shift;
  if($el eq "ARU_ID")
  {
    $inOHPropAruID = "FALSE" ;
  }
}

# Character handler for getaruIDfromPropXML
sub OHProp_char
{
  my $p = shift;
  my $info = shift;
  $info =~ s/\n/ /g ;
  if($inOHPropAruID eq "TRUE")
  {
    $OHPropXML{_ARU_ID} .= $info;
  }
}

package HomeInfo;
sub new
{
  my $class = shift;
  my $self = {};
  bless $self, $class;
}

1;
