#!/usr/local/bin/perl
# 
# $Header: oracle_ias.pl 03-aug-2005.08:54:50 ndutko Exp $
#
# oracle_ias.pl
# 
# Copyright (c) 2002, 2005, Oracle. All rights reserved.  
#
#    NAME
#      oracle_ias.pl - get XML stream describing iAS-related targets
#
#    DESCRIPTION
#      reads in the targets.xml from the "active" emd home, pulls out the
#      targets of the desired types (as specified in iasTypes.lst), adds
#      new properties as needed, returns the resulting XML stream.
#
#    NOTES
#
#    MODIFIED   (MM/DD/YY)
#    ndutko      08/03/05 - ndutko_code_checkin
#    jsutton     11/18/04 - TargetInstaller API change 
#    djoly       08/18/04 - Break up cemd 
#    djoly       05/14/04 - Update core reference 
#    jsutton     04/15/04 - Clean up warnings 
#    jsutton     01/23/04 - Further OSD code 
#    jsutton     01/22/04 - Platform-dependent changes in iAS discovery 
#    aaitghez    12/29/03 - delete EMDROOT from environment 
#    aaitghez    12/22/03 - pass extra args to emConfigInstall.jar 
#    jsutton     12/03/03 - Skip Metadata Repository-only iAS installs 
#    klmichae    12/02/03 - add host assoc target for 9.0.2/3 installs 
#    jsutton     11/19/03 - Handle LDAP targets (bug 3245997) 
#    jsutton     10/28/03 - Handle SSO targets; remove emCORE.jar reference
#    dsinha      09/11/03 - fix JRE reference 
#    jsutton     07/28/03 - Use java from JAVA_HOME if possible
#    jsutton     07/07/03 - fix 904 discovery
#    jsutton     07/02/03 - TargetInstaller now handles version-specific encryption issues
#    jsutton     06/02/03 - Check against iasTypes.lst for 904+ as well
#    jsutton     05/30/03 - Filter out ias-only targets for 904+
#    jsutton     05/21/03 - Check inventory for pre-904 instances if no emtab
#    lyang       04/29/03 - move listhomes to InventoryLoader class
#    jsutton     04/16/03 - Get list of OracleHomes; look for 904+ instances
#    jsutton     02/18/03 - Graceful handling of no emtab
#    jsutton     11/22/02 - Fix webcache properties (bug 2662578)
#    jsutton     11/01/02 - Add jserv and oc4j to the list of components needing version
#    jsutton     10/30/02 - Apache must have version corresponding to iAS version
#    jsutton     09/04/02 - Make helper 2 pass to tweak properties
#    jsutton     08/28/02 - StandaloneConsoleURL vs. EMD_URL
#    jsutton     08/22/02 - jsutton_ias_discovery_helper
#    jsutton     08/22/02 - Creation
# 
## use Config;
use Config;
my ($emdRoot,$hostName) = @ARGV;

#jsutton - force EMDROOT before require of oracledbUtl.pl
$ENV{EMDROOT}=$emdRoot;

require "$emdRoot/sysman/admin/discover/utl/oracledbUtl.pl";
require "$emdRoot/sysman/admin/scripts/emd_common.pl";

my $OSNAME = $Config{'osname'};
my $IsWin32 = ($OSNAME eq 'MSWin32');
if($IsWin32)
{
  eval 'use Win32::TieRegistry';
  $Registry->Delimiter("/");
}

$LOG_CATEGORY = "ORACLE_IAS_DISCOVERY: ";

# get list of "supported" target types
my %iasTargetTypes=();
my $typeLine;
my $iasTargetTypeFile = $emdRoot . "/sysman/config/iasTypes.lst";
if (-e $iasTargetTypeFile)
{
  open (TYPES_FILE_READER, $iasTargetTypeFile);
  while($typeLine = <TYPES_FILE_READER>)
  {
    $typeLine =~ s/^\s*//;
    chomp($typeLine);
    $iasTargetTypes{$typeLine} = 1;
  }
  close TYPES_FILE_READER;
}

# use java from 'appropriate' location
my $JAVA_HOME="";

if (defined($ENV{JRE_HOME}))
{
  $JAVA_HOME = $ENV{JRE_HOME};
}
if (($JAVA_HOME eq "") || (! -e "$JAVA_HOME/bin"))
{
  if (defined($ENV{JAVA_HOME}))
  {
    $JAVA_HOME = $ENV{JAVA_HOME};
  }
  if (($JAVA_HOME eq "") || (! -e "$JAVA_HOME/bin"))
  {
    if (-e "$emdRoot/jre/1.4.1/bin")
    {
      $JAVA_HOME="$emdRoot/jre/1.4.1";
    }
    elsif (-e "$emdRoot/jdk/bin")
    {
      $JAVA_HOME="$emdRoot/jdk";
    }
  }
}
die "Cannot determine JAVA_HOME\n" if (($JAVA_HOME eq "") || (! -e "$JAVA_HOME/bin"));

# start the output array
my @processedLines;
if (!exists $ENV{LL_ORACLE_HOME}) {push (@processedLines, "\<Targets\>\n")};

# 904+ home support
my ($iasVersion, $iasInstallType);
my $cpSep = ":";

if($IsWin32)
{
  $cpSep = ";";
}

# get list of homes 
my $allHomesRef;
my @pre904Homes;
my $numPre904Homes = 0;

if ($IsWin32)
{
  $allHomesRef = getIasOracleHomesWin32();
}
else
{
  $allHomesRef = getIasOracleHomesUnix();
}

# check through homes for iAS installs
foreach $homeLine (@$allHomesRef)
{
  chomp($homeLine);

  # check if iAS install, if version >= 904, pull ALL targets in...
  if (-e $homeLine . "/config/ias.properties")
  {
    $iasVersion = getPropertyFromFile($homeLine, "/config/ias.properties", "Version");
    if (substr($iasVersion, 0, index($iasVersion, ".")) eq "9")
    {
      # check for 902/903; skip this home if so
      if ((substr($iasVersion, 0, 5) eq "9.0.2") || 
          (substr($iasVersion, 0, 5) eq "9.0.3"))
      {
        $numPre904Homes = push (@pre904Homes, $homeLine);
        next;
      }
    }

    # got past version check; so we *must* be 904+ - check for MR-only, skip if it is
    $iasInstallType = getPropertyFromFile($homeLine, "/config/ias.properties", "InstallType");
    if ($iasInstallType eq "Infrastructure_Meta")
    {
      next;
    }

    EMD_PERL_DEBUG("$LOG_CATEGORY Use OH=$homeLine for 904+ discovery");
    push (@processedLines, get904PlusTargets($homeLine, $emdRoot, \%iasTargetTypes));
  }
}

#Get Active EMD Home (for 902/903 instances)
my $emdHomeRef = getActiveEmdHome($emdRoot);

# handle case of missing emtab IFF one existing pre-904 home
if (($emdHomeRef eq "") && ($numPre904Homes == 1))
{
  $emdHomeRef = pop(@pre904Homes);
}

# Bail out if no active home! 
if ($emdHomeRef ne "")
{
  # do the processing, add to the end of the output
  EMD_PERL_DEBUG("$LOG_CATEGORY Use OH=$emdHomeRef for 902/903 discovery");
  push (@processedLines, processPre904Targets($emdRoot, $hostName, $emdHomeRef, \%iasTargetTypes));
}

if (!exists $ENV{LL_ORACLE_HOME}) { push (@processedLines, "\<\/Targets\>\n") };
print @processedLines;
exit 0;


sub get904PlusTargets
{
  my ($iasHome, $emdRoot, $iasTargetTypes) = @_;
  my %iasMembers = ();
  my %otherMembers = ();

  my (@handledLines, @targetLines);

  my $inTarget = 0;
  my $isCompositeMember = 0;
  my $isIASTarget = 0;

  # read in targets.xml from given home
  my $targetsFile = $iasHome . "/sysman/emd/targets.xml";

  if (-e $targetsFile)
  {
    my $savedEmdRoot=$ENV{EMDROOT};
    delete($ENV{EMDROOT});
    # get full target listing - pass source, destination, extensionRoot
    open (TARGETS_FILE_READER, "$JAVA_HOME/bin/java -jar $emdRoot/jlib/emConfigInstall.jar listtargetsfully $iasHome $emdRoot $emdRoot |");
    $ENV{EMDROOT}=$savedEmdRoot;

    while($targetLine = <TARGETS_FILE_READER>)
    {
      $targetLine =~ s/^\s*//;

      # skip non-XML stufff...
      if (index($targetLine, "\<") != 0) 
      {
        next;
      }

      # skip the header/footer
      if (($targetLine eq "\<Targets\>\n") || ($targetLine eq "\<\/Targets\>\n"))
      {
        next;
      }
      # check - are we scanning a target?  
      # if so, look for the end target tag
      if ($inTarget > 0)
      {
        if (index($targetLine, "\/Target") >= 0)
        {
          # reset flags
          $inTarget = 0;
          $isCompositeMember = 0;
        }
        # are we handling a target that has composite membership?
        if ($isCompositeMember > 0)
        {
          if (index($targetLine, "\<MemberOf") >= 0)
          {
            # get name/type of "containing" target
            $tgtContainerType = substr ($targetLine, index ($targetLine, "TYPE")+length("TYPE=\""));
            $tgtContainerType = substr ($tgtContainerType, 0, index($tgtContainerType, "\""));
            $tgtContainerName = substr ($targetLine, index ($targetLine, "NAME")+length("NAME=\""));
            $tgtContainerName = substr ($tgtContainerName, 0, index($tgtContainerName, "\""));
      
            # add to appropriate membership hash
            if ($tgtContainerType eq "oracle_ias")
            {
              $iasMembers{$currTargetName . ":" . $currTargetType} = $tgtContainerName . ":" . $tgtContainerType;
            }
            else
            {
              $otherMembers{$currTargetName . ":" . $currTargetType} = $tgtContainerName . ":" . $tgtContainerType;
            }
          }
        }
        # check for CompositeMembership tag
        if (index($targetLine, "\<CompositeMembership") >= 0)
        {
          $isCompositeMember = 1;
        }
      }
      # look for the start tag for a new target
      if ($inTarget == 0)
      {
        if (index($targetLine, "\<Target TYPE") >= 0)
        {
          $currTargetType = substr ($targetLine, index ($targetLine, "\<Target TYPE")+length("\<Target TYPE=\""));
          $currTargetType = substr ($currTargetType, 0, index($currTargetType, "\""));
          $currTargetName = substr ($targetLine, index ($targetLine, "NAME")+length("NAME=\""));
          $currTargetName = substr ($currTargetName, 0, index($currTargetName, "\""));
          $inTarget = 1;
        }
        # watch out for one-liners!
        if ((index($targetLine, "\<Target") >= 0) && (index($targetLine, "\/\>") >= 0))
        {
          # reset flags
          $inTarget = 0;
          $isCompositeMember = 0;
        }
      }
      push (@targetLines, $targetLine);
      next;
    }

    # pass 2: collect iAS-related targets
    foreach $line (@targetLines)
    {
      # find the start tag for a target
      if (index ($line, "Target TYPE=") >= 0 )
      {
        $currTargetName = substr $line, index($line, "NAME=\"")+length("NAME=\"");
        $currTargetName = substr ($currTargetName, 0, index($currTargetName, "\""));
        $currTargetType = substr ($line, index ($line, "Target TYPE")+length("Target TYPE=\""));
        $currTargetType = substr ($currTargetType, 0, index($currTargetType, "\""));
      }

      # check to see if it's a type we care about; first if in the list
      if (defined ($iasTargetTypes->{$currTargetType}))
      {
        $isIASTarget = 1;
      }

      # check whether this is an IAS target by checking memberships
      if ($isIASTarget == 0)
      {
        # the fun part - check, recursively, if this target is a member or submember of iAS
        $targetMemberCheck = $currTargetName . ":" . $currTargetType;

        # the easy case - direct membership in an oracle_ias target
        if (defined($iasMembers{$targetMemberCheck}))
        {
          $isIASTarget = 1;
        }

        # still checking; look for membership in other targets, 
        # then check those targets' membership in an iAS -- recurse...
        if ($isIASTarget == 0)
        {
          $moreChecks = 1;
          while (($moreChecks > 0) && ($isIASTarget == 0))
          {
            if (defined($otherMembers{$targetMemberCheck}))
            {
              $otherMemberName = $otherMembers{$targetMemberCheck};
              if (defined($iasMembers{$otherMemberName}))
              {
                $isIASTarget = 1;
              }
              $targetMemberCheck = $otherMemberName;
            }
            else
            {
              $moreChecks = 0;
            }
          }
        }
      }

      # we've verified this is an iAS-related target; push its info
      if ($isIASTarget > 0)
      {
        push (@handledLines, $line);
      }

      # hit an end-of-target tag; reset flag
      if (index($line, "\/Target") >= 0)
      {
        $isIASTarget = 0;
      }
    }
  }
  return @handledLines;
}

sub processPre904Targets
{
  my ($emdRoot, $hostName, $emdHomeRef, $iasTargetTypes) = @_;

  # read in targets.xml from active home
  my $targetsFile = $emdHomeRef . "/sysman/emd/targets.xml";

  my $statusUrlPlaceholder = "STATUS_URL_PLACEHOLDER\n";
  my $oracleHomePlaceholder = "ORACLE_HOME_PLACEHOLDER\n";
  my $iasVersionPlaceholder = "IAS_VERSION_PLACEHOLDER\n";
  my $machinePlaceholder = "MACHINE_PLACEHOLDER\n";
  my $logFileDirPlaceholder = "LOGFILEDIR_PLACEHOLDER\n";
  my $iasDisplayNamePlaceholder = "IAS_DISPLAY_NAME_PLACEHOLDER\n";
  my $startComponentPlaceholder = "START_COMPONENT_PLACEHOLDER\n";
  my $hostAssocTargetPlaceholder = "HOST_ASSOC_TARGET_PLACEHOLDER\n";

  my $ignoringTarget = 0;
  my $skipVersionProperty = 0;
  my $needIasDisplayName = 0;
  my $needIasApacheInstancePort = 0;
  my $needIasOracleHome = 0;
  my $needHTTPMachine = 0;
  my $needConnectDesc = 0;
  my $hostTargetName;

  my $skipLogFileDir = 0;
  my $skipLogFileName = 0;
  my %iasDisplayNames = ();
  my %iasVersions = ();
  my %iasOracleHomes = ();
  my %iasContainers = ();
  my %iasApacheInstanceNames = ();
  my %iasApacheInstancePorts = ();
  my %httpMachines = ();
  my (@handledLines, @pass2Lines, $targetLine, $currTargetDisplayName, $currTargetName, $currTargetType);

  if (-e $targetsFile)
  {
    # get the EMD_URL and strip off the "/emd/main" part, add "/emd/console", make XML property
    my $emdUrl = getPropertyFromFile($emdHomeRef, "/sysman/config/emd.properties", "EMD_URL");
    $emdUrl = substr ($emdUrl, 0, index($emdUrl, "/emd/main"))."/emd/console";
    $emdUrl =  "<Property NAME=\"StandaloneConsoleURL\" VALUE=\"$emdUrl\"/>\n";

    my $savedEmdRoot=$ENV{EMDROOT};
    delete($ENV{EMDROOT});
    # get full target listing - pass source, destination, extensionRoot
    open (TARGETS_FILE_READER, "$JAVA_HOME/bin/java -jar $emdRoot/jlib/emConfigInstall.jar listtargetsfully $emdHomeRef $emdRoot $emdRoot |");
    $ENV{EMDROOT}=$savedEmdRoot;

    while($targetLine = <TARGETS_FILE_READER>)
    {
      $targetLine =~ s/^\s*//;

      # skip non-XML stufff...
      if (index($targetLine, "\<") != 0)
      {
        next;
      }

      if (($targetLine eq "\<Targets\>\n") || ($targetLine eq "\<\/Targets\>\n"))
      {
        next;
      }
      # check - are we skipping the current target?  
      # if so, look for the end target tag
      if ($ignoringTarget > 0)
      {
        if (index($targetLine, "\/Target") >= 0)
        {
          $ignoringTarget = 0;
        }
        next;
      }
      # is this the start of a new target?
      if (index($targetLine, "\<Target TYPE") >= 0)
      {
        $ignoringTarget = 1;
        $currTargetType = substr ($targetLine, index ($targetLine, "\<Target TYPE")+length("\<Target TYPE=\""));
        $currTargetType = substr ($currTargetType, 0, index($currTargetType, "\""));
        # check to see if it's a type we care about
        if (defined ($iasTargetTypes->{$currTargetType}))
        {
          $ignoringTarget = 0;
        }
        # reset flags (in case they got set but not reset for previous target)
        $skipVersionProperty = 0;
        $needIasDisplayName = 0;
        $needIasApacheInstancePort = 0;
        $needIasOracleHome = 0;
        $skipLogFileDir = 0;
        $skipLogFileName = 0;
        $needHTTPMachine = 0;
        $needConnectDesc = 0;
      }
      # we need to get the host name for later use in the AssocTargetInstance tag
      if ($currTargetType eq "host")
      {
        $currTargetName = substr $targetLine, index($targetLine, "NAME=\"")+length("NAME=\"");
        $currTargetName = substr ($currTargetName, 0, index($currTargetName, "\""));
        $hostTargetName = $currTargetName;
      }
      # one last check: are we going to skip the current target?
      # (may have been set by the test block above)
      if ($ignoringTarget > 0)
      {
        next; # next iteration thru while loop
      }

      # is this the version property we've elected to skip
      if (($skipVersionProperty > 0) &&
          (index($targetLine, "\<Property NAME=\"version\"") >= 0))
      {
        $skipVersionProperty = 0;
        next;
      }
      # is this the logFileDir property we've elected to skip
      if (($skipLogFileDir > 0) &&
          (index($targetLine, "\<Property NAME=\"logFileDir\"") >= 0))
      {
        $skipLogFileDir = 0;
        next;
      }
      # is this the logFileName property we've elected to skip
      if (($skipLogFileName > 0) &&
          (index($targetLine, "\<Property NAME=\"logFileName\"") >= 0))
      {
        $skipLogFileName = 0;
        push (@handledLines, "<Property NAME=\"logFileName\" VALUE=\"access_log\"/>\n");
        next;
      }
      # if we need the connect descriptor for LDAP, look for the TNSAddress property
      if (($needConnectDesc > 0) &&
          (index($targetLine, "\<Property NAME=\"TNSAddress\"") >= 0))
      {
        $ldapConnectDesc = substr $targetLine, index($targetLine, "VALUE=\"")+length("VALUE=\"");
        $ldapConnectDesc = substr $ldapConnectDesc, 0, index($ldapConnectDesc, "\"");
        push (@handledLines, "<Property NAME=\"ConnectDescriptor\" VALUE=\"$ldapConnectDesc\"/>\n");
        $needConnectDesc = 0;
        next;
      }

      # add the current line to the output
      push (@handledLines, $targetLine);

      # check: is this the start of a target requiring special handling?
      if (index ($targetLine, "Target TYPE=") >= 0 )
      {
        # get target name for later use
        $currTargetName = substr $targetLine, index($targetLine, "NAME=\"")+length("NAME=\"");
        $currTargetName = substr ($currTargetName, 0, index($currTargetName, "\""));
        if ($currTargetType eq "oracle_ias")
        {
          # add the EMD_URL to the output
          push (@handledLines, $emdUrl);
          # push status URL placeholder for phase 2
          push (@handledLines, $statusUrlPlaceholder);
          $needIasOracleHome = 1;
          # push associated target tag for phase 2
          push (@handledLines, $hostAssocTargetPlaceholder);
        }
        else
        {
          # not an oracle_ias target; need target display name to generate the IAS_DISPLAY_NAME property
          $currTargetDisplayName = substr($targetLine, index($targetLine, "DISPLAY_NAME=\"")+length("DISPLAY_NAME=\""));
          $currTargetDisplayName = substr $currTargetDisplayName, 0, index($currTargetDisplayName, "\"");
          $needIasDisplayName = 1;
          push (@handledLines, $iasDisplayNamePlaceholder);
        }
        # we get the Apache port for later use in the iAS status URL
        # we also need to force the version number to the iAS version
        if ($currTargetType eq "oracle_apache")
        {
          $needIasApacheInstancePort = 1;
          $skipVersionProperty = 1;
          push (@handledLines, $iasVersionPlaceholder);
          # push associated target tag for phase 2
          push (@handledLines, $hostAssocTargetPlaceholder);
        }
        # older webcache targets need OracleHome and version properties
        if ($currTargetType eq "oracle_webcache")
        {
          $needHTTPMachine = 1;
          $skipLogFileDir = 1;
          $skipLogFileName = 1;
          push (@handledLines, $oracleHomePlaceholder);
          push (@handledLines, $iasVersionPlaceholder);
          push (@handledLines, $machinePlaceholder);
          push (@handledLines, $logFileDirPlaceholder);
          # push associated target tag for phase 2
          push (@handledLines, $hostAssocTargetPlaceholder);
        }
        # jserv targets need version property
        if (($currTargetType eq "oracle_jserv") ||
            ($currTargetType eq "oc4j"))
        {
          push (@handledLines, $iasVersionPlaceholder);
          # push associated target tag for phase 2
          push (@handledLines, $hostAssocTargetPlaceholder);
        }
        # SSO targets have their own set of required info
        if ($currTargetType eq "oracle_sso_server")
        {
          push (@handledLines, $oracleHomePlaceholder);
          push (@handledLines, $iasVersionPlaceholder);
          push (@handledLines, $startComponentPlaceholder);
        }
        # LDAP targets need version and connect descriptor
        if ($currTargetType eq "oracle_ldap")
        {
          push (@handledLines, $iasVersionPlaceholder);
          $needConnectDesc = 1;
        }
        next; # go to next input line
      }
      # if we're processing an iAS composite target, get the name of the "containing" iAS instance
      # and use it to generate the IAS_DISPLAY_NAME property for the current target
      if ($needIasDisplayName > 0)
      {
        if ((index $targetLine, "MemberOf TYPE=\"oracle_ias\"") >= 0 )
        {
          $iasInstanceName = substr $targetLine, index($targetLine, "NAME=\"")+length("NAME=\"");
          $iasInstanceName = substr ($iasInstanceName, 0, index($iasInstanceName, "\""));
          $iasContainers{$currTargetName} = $iasInstanceName;
          $iasDisplayNames{$currTargetName} = $iasInstanceName . " " . $currTargetDisplayName;
          if ($currTargetType eq "oracle_apache")
          {
            $iasApacheInstanceNames{$iasInstanceName} = $currTargetName;
          }
          $needIasDisplayName = 0;
          next; # get next input line
        }
      }
      # get the Apache port number; we'll use this later for the iAS status URL
      if ($needIasApacheInstancePort > 0)
      {
        if ((index $targetLine, "Property NAME=\"HTTPPort\"") >= 0 )
        {
          $iasApacheInstancePort = substr $targetLine, index($targetLine, "VALUE=\"")+length("VALUE=\"");
          $iasApacheInstancePort = substr ($iasApacheInstancePort, 0, index($iasApacheInstancePort, "\""));
          $iasApacheInstancePorts{$currTargetName} = $iasApacheInstancePort;
          $needIasApacheInstancePort = 0;
          next; # get next input line
        }
      }
      # get the HTTPMachine property; we'll use this later for webcache 
      if ($needHTTPMachine > 0)
      {
        if ((index $targetLine, "Property NAME=\"HTTPMachine\"") >= 0 )
        {
          $httpMachine = substr $targetLine, index($targetLine, "VALUE=\"")+length("VALUE=\"");
          $httpMachine = substr ($httpMachine, 0, index($httpMachine, "\""));
          $httpMachines{$currTargetName} = $httpMachine;
          $needHTTPMachine = 0;
          next; # get next input line
        }
      }
      # for a given iAS, get the OracleHome property
      if ($needIasOracleHome > 0)
      {
        if ((index $targetLine, "Property NAME=\"OracleHome\"") >= 0 )
        {
          $iasOracleHome = substr $targetLine, index($targetLine, "VALUE=\"")+length("VALUE=\"");
          $iasOracleHome = substr ($iasOracleHome, 0, index($iasOracleHome, "\""));
          $iasOracleHomes{$currTargetName} = $iasOracleHome;
          # get the version of this iAS instance
          $iasVersions{$currTargetName} = getPropertyFromFile($iasOracleHome, "/config/ias.properties", "Version");
          $needIasOracleHome = 0;
          next; # get next input line
        }
      }
    }
    close TARGETS_FILE_READER;
  }
  #
  # Pass 2: plug in missing/generated properties
  #
  foreach $line (@handledLines)
  {
    if (index ($line, "Target TYPE=") >= 0 )
    {
      $currTargetName = substr $line, index($line, "NAME=\"")+length("NAME=\"");
      $currTargetName = substr ($currTargetName, 0, index($currTargetName, "\""));
    }
    if ($line eq $statusUrlPlaceholder)
    {
      if (defined($iasApacheInstanceNames{$currTargetName}))
      {
        $apacheName = $iasApacheInstanceNames{$currTargetName};
        $line = "<Property NAME=\"StatusURL\" VALUE=\"http://$hostName:$iasApacheInstancePorts{$apacheName}/\"/>\n";
      }
      else
      {
        $line = "";
      }
    }
    if ($line eq $iasDisplayNamePlaceholder)
    {
      if (defined($iasDisplayNames{$currTargetName}))
      {
        $line = "<Property NAME=\"IAS_DISPLAY_NAME\" VALUE=\"$iasDisplayNames{$currTargetName}\"/>\n";
      }
      else
      {
        $line = "";
      }
    }
    if ($line eq $machinePlaceholder)
    {
      if (defined($httpMachines{$currTargetName}))
      {
        $line = "<Property NAME=\"HTTPMachineForAdmin\" VALUE=\"$httpMachines{$currTargetName}\"/>\n";
      }
      else
      {
        $line = "";
      }
    }
    if ($line eq $logFileDirPlaceholder)
    {
      if (defined($iasContainers{$currTargetName}))
      {
        $iasContainer = $iasContainers{$currTargetName};
        $line = "<Property NAME=\"logFileDir\" VALUE=\"$iasOracleHomes{$iasContainer}/webcache/logs\"/>\n";
      }
      else
      {
        $line = "";
      }
    }
    if ($line eq $iasVersionPlaceholder)
    {
      if (defined($iasContainers{$currTargetName}))
      {
        $iasContainer = $iasContainers{$currTargetName};
        $line = "<Property NAME=\"version\" VALUE=\"$iasVersions{$iasContainer}\"/>\n";
      }
      else
      {
        $line = "";
      }
    }
    if ($line eq $oracleHomePlaceholder)
    {
      if (defined($iasContainers{$currTargetName}))
      {
        $iasContainer = $iasContainers{$currTargetName};
        $line = "<Property NAME=\"OracleHome\" VALUE=\"$iasOracleHomes{$iasContainer}\"/>\n";
      }
      else
      {
        $line = "";
      }
    }
    if ($line eq $startComponentPlaceholder)
    {
      if (defined($iasContainers{$currTargetName}))
      {
        $httpServer = $iasContainers{$currTargetName}."_HTTP Server";
        $line = "<Property NAME=\"startComponent\" VALUE=\"$httpServer\"/>\n";
      }
      else
      {
        $line = "";
      }
    }
    if ($line eq $hostAssocTargetPlaceholder)
    {
      $line = "<AssocTargetInstance ASSOC_TARGET=\"host\" TYPE=\"host\" NAME=\"$hostTargetName\"/>\n";
    }
    push (@pass2Lines, $line);
  }

  return @pass2Lines;
}

sub getPropertyFromFile
{
  my ($emdHomeRef, $fileSpec, $property) = @_;

  my $propertyValue = "";
  my $propertiesFile = $emdHomeRef . $fileSpec;
  my ($propertiesLine, @propInfo);

  if (-e $propertiesFile)
  {
    open (PROPERTIES_FILE_READER, $propertiesFile);
    while($propertiesLine = <PROPERTIES_FILE_READER>)
    {
      chomp($propertiesLine);
      if (index($propertiesLine, "$property") eq 0)
      {
        @propInfo = split(/=/ ,$propertiesLine);
        if ($propInfo[0] eq $property)
        {
          $propertyValue = $propInfo[1];
          last;
        }
      }
    }
    close PROPERTIES_FILE_READER;
  }
  return $propertyValue;
}

#Strips leading and trailing spaces and returns the string
sub trim 
{
  my $origStr = $_[0];
  #Strip trailing and leading
  $origStr =~ s/^\s*|\s*$//g;
  return $origStr;
}

#
# returns the oracleHome corresponding to the DEFAULT entry in emtab
#
sub getActiveEmdHome
{
  my ($emdRoot) = @_;
  my (@entries) = getEmtabEntries($emdRoot);
  my $emdHome = "";
  foreach $emTabEntry (@entries)
  {
    my ($emdName, $oracleHome) = split(/=/ , $emTabEntry);
    if ($emdName eq "DEFAULT")
    {
      $emdHome = $oracleHome;
      last;
    }
  }
  return $emdHome;
}

#Get the emtab file entries
# A typical emtab entry is of the form
# DEFAULT=/private1/oracle/902orcl
#
# Returns list of the form
# { Emd1=Oracle_home1 Emd2=Oracle_home2 ... EmdN=Oracle_homeN }
sub getEmtabEntries 
{
  my ($emdRoot) = @_;
  my (@entries,$emtabFile,$Emtab,$oraLine); 

  if (( $^O =~ "Windows" ) || ($^O =~ "MSWin32"))
  {
    my $emLocValue = $Registry->{"HKEY_LOCAL_MACHINE/Software/Oracle//em_loc"};
    chomp($emLocValue);
    push (@entries, "DEFAULT=$emLocValue");
  }
  else
  {
    $emtabFile = getEmtabFile();
  
    if ( $emtabFile ne "" ) 
    {
      if ( open(Emtab, $emtabFile) ) 
      {
        while ($oraLine=<Emtab>) 
        {
          chomp($oraLine);
          #strip all leading  white space characters.
          $oraLine =~ s/^\s*//;
          if( ($oraLine =~ /^\#/ ) || ( length($oraLine) <= 0 ) ) 
          {
            next;
          }
          push (@entries,$oraLine);
        } 
        close(Emtab);
      }
    }
  }
  return @entries;
}

# Get the emtab file location.
# The emtab file contains the oracle home of the active EMD
# Note: The location of emtab file is port specific
sub getEmtabFile 
{
  my @emtabFileList = ("/var/opt/oracle/emtab", "/etc/emtab");
  my $emtabFile = "";

  #$^O is supposed to yield the name of the OS, like Solaris , but 
  #currently it is not giving any value. This value probably should come 
  #from emctl.
  if ( $^O =~ "VMS" )
  {
    #TODO VMS specific code
  }
  else
  {
    # Check if EMTAB env is specified
    if ( defined $ENV{EMTAB} )
    {
      $emtabFile = trim ( $ENV{EMTAB} );
      if (! -e $emtabFile )
      {
        $emtabFile = "";
      } 
      elsif ( -d $emtabFile ) 
      {
        $emtabFile = "";
      }
      else 
      {
        #log "Using EMTAB file $emtabFile";
        return $emtabFile;
      }
    }
  }
  #trace Processing prospective EMTAB locations of %s $emtabFileList
  foreach $emtabFile (@emtabFileList) 
  {
    if ( -d $emtabFile ) 
    {
      #warning "$emtabFile can not be a directory";
    }
    elsif ( -e $emtabFile )
    {
      #log "Using emtab file $emtabFile";
      return $emtabFile
    }
  }
  #log "Warning: Found no valid emtab file";
  $emtabFile = "" ;
  return $emtabFile;
}

#Get iAS Oracle Homes for Unix
#Returns oracle Home references
#OhomeRef---> Array of Homes like ("/private/oracle817", "/private/oracle92");
# THIS IS AN OS DEPENDENT Function.
sub getIasOracleHomesUnix
{
  my @oratabEntries = getOratabEntries();

  my @ouiOracleHomes;
  my @iASOracleHomes;

  # get OUI homes for iAS installs
  @ouiOracleHomes = getOuiIasHomes();
  push (@iASOracleHomes, @ouiOracleHomes);

  foreach $entry (@oratabEntries) 
  {
    my ($sid, $oracleHome, $remain) = split(/:/ , $entry , 3);
    #remove leading and trailing white space.
    EMD_PERL_DEBUG("$LOG_CATEGORY Oratab entry=$entry: SID=$sid; OH=$oracleHome");

    $oracleHome =~ s/^\s*|\s*$//g;
    if(!(-e $oracleHome . "/config/ias.properties"))
    {
      EMD_PERL_DEBUG("$LOG_CATEGORY $oracleHome/config/ias.properties does not exist, excluded from discovery");
      next;
    }

    $processHome = 1;
    foreach $oHome (@iASOracleHomes)
    {
      if($oHome eq $oracleHome)
      {
        $processHome = 0;
        last;
      }
    }
    if($processHome)
    {
      push(@iASOracleHomes,$oracleHome);
      EMD_PERL_DEBUG("$LOG_CATEGORY Add OH=$oracleHome for discovery");
    }
  }
  return (\@iASOracleHomes);
}

#Get iAS Oracle Homes for Win32 platforms
#Returns oracle Home references
#OhomeRef---> Array of Homes like ("/private/oracle817", "/private/oracle92");
# THIS IS AN OS DEPENDENT Function.
# (wrapper to turn array from getOracleHomesNT to array ref)
sub getIasOracleHomesWin32
{
  my @oracleHomes;
  my @ouiOracleHomes;
  my @iASOracleHomes;

  # get OUI homes for iAS installs
  @ouiOracleHomes = getOuiIasHomes();
  push (@iASOracleHomes, @ouiOracleHomes);

  @oracleHomes = getOracleHomesNT();
  # check homes coming from NT registry
  foreach $oracleHome (@oracleHomes)
  {
    $processHome = 1;
    # skip duplicates 
    foreach $oHome(@iASOracleHomes)
    {
      if ($oHome eq $oracleHome)
      {
        $processHome = 0;
        last;
      } 
    }
    # skip homes which don't have ias.properties
    if(!(-e $oracleHome . "/config/ias.properties"))
    {
      $processHome = 0;
    }

    if($processHome)
    {
      push(@iASOracleHomes,$oracleHome);
      EMD_PERL_DEBUG("$LOG_CATEGORY Add OH=$oracleHome for discovery");
    }
  }
  return (\@iASOracleHomes);
}


sub getOuiIasHomes
{
  my @oracleHomes;

  my $savePath = $ENV{PATH};
  if($IsWin32)
  {
    $ENV{PATH} = "$emdRoot/oui/lib/win32\;$savePath";
  }
  open (HOMES_READER, "$JAVA_HOME/bin/java -cp $emdRoot/jlib/emConfigInstall.jar$cpSep$emdRoot/jlib/share.jar$cpSep$emdRoot/oui/jlib/OraInstaller.jar$cpSep$emdRoot/jlib/srvm.jar$cpSep$emdRoot/sysman/jlib/emcoreAgent.jar$cpSep$emdRoot/sysman/jlib/emagentSDK.jar oracle.sysman.emSDK.conf.InventoryLoader listhomes |");
  $ENV{PATH} = $savePath;
  while($homeLine = <HOMES_READER>)
  {
    $homeLine =~ s/^\s*//;
    chomp($homeLine);

    # check if iAS install
    if (-e $homeLine . "/config/ias.properties")
    {
      push(@oracleHomes,$homeLine);
      EMD_PERL_DEBUG("$LOG_CATEGORY Add OUI OH=$homeLine for discovery");
    }
  }
  close HOMES_READER;
  return (@oracleHomes);
}
