#!/usr/local/bin/perl
# 
# $Header: emll/sysman/admin/discover/oracle_ias2.pl /main/33 2012/06/14 11:40:58 fmorshed Exp $
#
# oracle_ias.pl
# 
# Copyright (c) 2002, 2012, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      oracle_ias.pl - get XML stream describing iAS-related targets
#
#    DESCRIPTION
#      reads in the targets.xml from the "active" emd home, pulls out the
#      targets of the desired types (as specified in iasTypes.lst), adds
#      new properties as needed, returns the resulting XML stream.
#
#    NOTES
#
#    MODIFIED   (MM/DD/YY)
#    fmorshed    01/19/12 - In advent of central collector, set emdroot, 
#                           and oracle home properly.
#    kotkumar    04/11/11 - Fix bug - 12339007
#    nmittal     02/16/11 - bug 11778471
#    kotkumar    12/30/10 - Adding version_value property for BPEL targets
#    ckalivar    07/01/10 - Fix bug 9856540: added version_value property for
#                           all oc4j targets
#    aghanti     06/17/10 - Pass ohs/webcache instance id to collection scripts
#    jsmoler     04/22/10 - move weblogic discovery to fmw_discover.pl
#    aghanti     04/20/10 - XbranchMerge aghanti_bug-9612827 from
#                           st_emll_10.3.3
#    aghanti     04/20/10 - Bug 9612827 - Check for opmnctl.bat for windows
#    aghanti     04/14/10 - XbranchMerge aghanti_bug-9557712 from
#                           st_emll_10.3.3
#    aghanti     04/14/10 - Bug 9557712 - Support discovery of multiple AS11g
#                           ohs and webcache targets
#    jsutton     03/30/10 - AS11 component discovery
#    jsmoler     10/09/09 - JRF template check
#    jsmoler     06/09/09 - grab fixes from 10.2.0.5 GC
#    jsmoler     03/13/09 - case insensitive host name match
#    jsmoler     12/10/08 - change app version display
#    jsmoler     11/25/08 - change domainlocation.properties format, BEAHOME -> BEA_HOME
#    jsmoler     11/19/08 - change weblogic discovery
#    jsmoler     11/06/08 - additional weblogic targets
#    jsmoler     10/27/08 - AS11 weblogic discovery
#    jsutton     06/25/08 - Fix iAS discovery in EBS env; make more selective
#    jsutton     12/10/07 - Use ORACLE_CONFIG_HOME vs. LL_ORACLE_HOME, as
#                           appropriate
#    jsutton     11/30/07 - Need else clause when checking for iASR1
#    jsutton     11/30/07 - Need to wrap calls to Win32 APIs
#    jsmoler     07/12/07 - support OPMN variables
#    jsutton     05/14/07 - XbranchMerge jsutton_bug-6052079 from
#                           st_emll_10.2.6
#    jsutton     02/08/07 - XbranchMerge jsutton_bug-5872565 from
#                           st_emll_10.2.5
#    jsutton     02/08/07 - Fix discovery of 10.1.3 where Apache not configured
#    jsmoler     07/14/06 - use parseApacheConf (bug 5224028) 
#    pparida     04/20/06 - Fix bug 5172723: Add support for iAS 10.1.3+ discovery 
#    jsutton     06/28/06 - Handle shortened home path 
#    jsutton     06/02/06 - Fixups for bug 5264089 
#    jsutton     05/16/06 - Fix pre904 discovery of ias instances 
#    ndutko      08/03/05 - ndutko_code_checkin
#    jsutton     11/18/04 - TargetInstaller API change 
#    djoly       08/18/04 - Break up cemd 
#    djoly       05/14/04 - Update core reference 
#    jsutton     04/15/04 - Clean up warnings 
#    jsutton     01/23/04 - Further OSD code 
#    jsutton     01/22/04 - Platform-dependent changes in iAS discovery 
#    aaitghez    12/29/03 - delete EMDROOT from environment 
#    aaitghez    12/22/03 - pass extra args to emConfigInstall.jar 
#    jsutton     12/03/03 - Skip Metadata Repository-only iAS installs 
#    klmichae    12/02/03 - add host assoc target for 9.0.2/3 installs 
#    jsutton     11/19/03 - Handle LDAP targets (bug 3245997) 
#    jsutton     10/28/03 - Handle SSO targets; remove emCORE.jar reference
#    dsinha      09/11/03 - fix JRE reference 
#    jsutton     07/28/03 - Use java from JAVA_HOME if possible
#    jsutton     07/07/03 - fix 904 discovery
#    jsutton     07/02/03 - TargetInstaller now handles version-specific encryption issues
#    jsutton     06/02/03 - Check against iasTypes.lst for 904+ as well
#    jsutton     05/30/03 - Filter out ias-only targets for 904+
#    jsutton     05/21/03 - Check inventory for pre-904 instances if no emtab
#    lyang       04/29/03 - move listhomes to InventoryLoader class
#    jsutton     04/16/03 - Get list of OracleHomes; look for 904+ instances
#    jsutton     02/18/03 - Graceful handling of no emtab
#    jsutton     11/22/02 - Fix webcache properties (bug 2662578)
#    jsutton     11/01/02 - Add jserv and oc4j to the list of components needing version
#    jsutton     10/30/02 - Apache must have version corresponding to iAS version
#    jsutton     09/04/02 - Make helper 2 pass to tweak properties
#    jsutton     08/28/02 - StandaloneConsoleURL vs. EMD_URL
#    jsutton     08/22/02 - jsutton_ias_discovery_helper
#    jsutton     08/22/02 - Creation
# 

use Config;
use File::Spec;
use ias::oc4j::getJVMData;

my ($emdRoot,$hostName) = @ARGV;
my $ORACLEHOME = $ENV{'ORACLE_HOME'};

# if LL_ORACLE_HOME is set, get targets ONLY from that home
my $onlyThisHome = (defined($ENV{LL_ORACLE_HOME}) ? "TRUE" : "FALSE");

my $oracleConfigHome = "";
if (defined($ENV{ORACLE_CONFIG_HOME}))
{
    $oracleConfigHome = $ENV{ORACLE_CONFIG_HOME};
}
elsif (defined($ENV{IAS_CONFIG_HOME}))
{
    $oracleConfigHome = $ENV{IAS_CONFIG_HOME};
}
elsif (defined($ENV{LL_ORACLE_HOME}))
{
    $oracleConfigHome = $ENV{LL_ORACLE_HOME};
}

#jsutton - force EMDROOT before require of oracledbUtl.pl
$ENV{EMDROOT}=$emdRoot;

use ias::simpleXPath;
use ias::asecm;
use ias::Apache_confFileParser;
use ias::propertiesFileParser;

require "$emdRoot/sysman/admin/scripts/emd_common.pl";
require "$emdRoot/sysman/admin/scripts/semd_common.pl";
require "$emdRoot/sysman/admin/discover/utl/oracledbUtl.pl";
require "$emdRoot/sysman/admin/scripts/LLInventoryParser.pl";

my $OSNAME = $Config{'osname'};
my $IsWin32 = ($OSNAME eq 'MSWin32');
if($IsWin32)
{
    eval 'use Win32::TieRegistry';
    $Registry->Delimiter("/");
}

$LOG_CATEGORY = "ORACLE_IAS_DISCOVERY: ";

# get list of "supported" target types
my %iasTargetTypes = (oracle_ias => 1,
                      oracle_apache => 1,
                      oracle_jserv => 1,
                      oracle_webcache => 1,
                      oc4j => 1);
my $typeLine;
my $iasTargetTypeFile = $ORACLEHOME . "/sysman/config/iasTypes.lst";
if (-e $iasTargetTypeFile)
{
    open (TYPES_FILE_READER, $iasTargetTypeFile);
    while($typeLine = <TYPES_FILE_READER>)
    {
        $typeLine =~ s/^\s*//;
        chomp($typeLine);
        $iasTargetTypes{$typeLine} = 1;
    }
    close TYPES_FILE_READER;
}

my $statusUrlPlaceholder = "STATUS_URL_PLACEHOLDER";
my $oracleHomePlaceholder = "ORACLE_HOME_PLACEHOLDER";
my $iasVersionPlaceholder = "IAS_VERSION_PLACEHOLDER";
#added VERSION_VALUE_PLACEHOLDER for version_value target property
my $versionValuePlaceholder = "VERSION_VALUE_PLACEHOLDER";
my $machinePlaceholder = "MACHINE_PLACEHOLDER";
my $logFileDirPlaceholder = "LOGFILEDIR_PLACEHOLDER";
my $iasDisplayNamePlaceholder = "IAS_DISPLAY_NAME_PLACEHOLDER";
my $startComponentPlaceholder = "START_COMPONENT_PLACEHOLDER";
my $hostAssocTargetPlaceholder = "HOST_ASSOC_TARGET_PLACEHOLDER";

# start the output array
my @processedLines;
push (@processedLines, "\<Targets\>\n") if (!exists $ENV{LL_ORACLE_HOME});

# 904+ home support
my ($iasVersion, $iasInstallType);

# get list of homes 
my $allHomesRef;
my @pre904Homes;
my $numPre904Homes = 0;

$allHomesRef = getIasOracleHomes($onlyThisHome);

my $isOMS1013PlusCapable = checkIfOMSIs1013PlusCapable($ORACLEHOME);
# check through homes for iAS installs
foreach $homeLine (@$allHomesRef)
{
    chomp($homeLine);
    EMD_PERL_DEBUG("$LOG_CATEGORY checking $homeLine for config/ias.properties.file");
    # check if iAS install, if version >= 904, pull ALL targets in...
    if (-e $homeLine . "/config/ias.properties")
    {
        $iasVersion = getPropertyFromFile($homeLine, "/config/ias.properties", "Version");
        if (substr($iasVersion, 0, index($iasVersion, ".")) eq "9")
        {
            # check for 902/903; skip this home if so
            if ((substr($iasVersion, 0, 5) eq "9.0.2") || 
                (substr($iasVersion, 0, 5) eq "9.0.3"))
            {
                EMD_PERL_DEBUG("$LOG_CATEGORY $homeLine is pre-904 iAS");
                $numPre904Homes = push (@pre904Homes, $homeLine);
                next;
            }
        }

        # got past version check; so we *must* be 904+ - check for MR-only, skip if it is
        $iasInstallType = getPropertyFromFile($homeLine, "/config/ias.properties", "InstallType");
        if ($iasInstallType eq "Infrastructure_Meta")
        {
            EMD_PERL_DEBUG("$LOG_CATEGORY skipping $homeLine; install type is Infrastructure_Meta");
            next;
        }
        my ($majorVer,$firstSub,$secondSub) =  split(/\./ , $iasVersion);

        if (($majorVer > 10) ||
           (($majorVer == 10) && ($firstSub > 1)) ||
           (($majorVer == 10) && ($firstSub == 1) && ($secondSub >= 3)))
        {
            # if OMS is not atleast 10.2.0.2 or above don't discover 10.1.3 IAS
            if ($isOMS1013PlusCapable == 0)
            {
                EMD_PERL_WARN("1013Plus AS instance found at $homeLine but OMS is not capable of managing this.");
                next;
            }
            my $iasHostName = getPropertyFromFile($homeLine, "/config/ias.properties", "VirtualHostName");
            my $iasComponents = getPropertyFromFile($homeLine, "/config/ias.properties", "Components");
            $iasComponents = trim($iasComponents);
            EMD_PERL_DEBUG("$LOG_CATEGORY Use OH=$homeLine for 10.1.3 PLUS discovery");
            EMD_PERL_DEBUG("$LOG_CATEGORY iasHostName = $iasHostName");
            EMD_PERL_DEBUG("$LOG_CATEGORY iasComponents = $iasComponents");
            push (@processedLines, get1013PlusTargets($homeLine, $ORACLEHOME, \%iasTargetTypes,$iasVersion,$iasHostName,$iasComponents));
        }
        else
        {
            EMD_PERL_DEBUG("$LOG_CATEGORY Use OH=$homeLine for 904+ discovery");
            push (@processedLines, get904PlusTargets($homeLine, $ORACLEHOME, \%iasTargetTypes));
        }
    }
    elsif (-e $homeLine . "/bin/opmnctl" || -e $homeLine . "/bin/opmnctl.bat")
    {
        # get iAS version from inventory
        my $tmpInfo = LLParseInventory::getInfo($ENV{USER_ORACLE_HOME},TOPLEVEL);
        $tmpInfo =~ /^\s*([^\s]*)\s+([^\s]*)/;
        $iasVersion = $2;
        EMD_PERL_DEBUG("$LOG_CATEGORY Use OH=$homeLine for AS11 discovery; iAS Version is $iasVersion");
        push (@processedLines, getAS11Targets($homeLine, $ORACLEHOME));
    }
    elsif (-e $homeLine . "/Apache/Apache/conf/apps.conf")
    {
        EMD_PERL_DEBUG("$LOG_CATEGORY checking OUI inventory in $homeLine");
        # no ias.properties file; check if we have an iappserver component
        my $iasBinHome = (defined($ENV{LL_ORACLE_HOME}) ? $ENV{LL_ORACLE_HOME} : $homeLine);
        # Invoke parser to get the top level OUI component from the inventory
        my $topLevelComp = LLParseInventory::getInfo($iasBinHome,TOPLEVEL);
        if ($topLevelComp =~ "oracle.iappserver" || $topLevelComp =~ "oracle.ias.apps.toplevel")
        {
            # looks like an Apps/iAS 1.0.2.x home ??
            push (@processedLines, getIASR1Targets($iasBinHome, $homeLine, $hostName));
        }
    }
}

if ($numPre904Homes > 0)
{
    #Get Active EMD Home (for 902/903 instances)
    my $emdHomeRef = getActiveEmdHome($ORACLEHOME);

    # handle case of missing emtab IFF one existing pre-904 home
    $emdHomeRef = pop(@pre904Homes) if (($emdHomeRef eq "") && ($numPre904Homes == 1));

    # Bail out if no active home! 
    if ($emdHomeRef ne "")
    {
        # do the processing, add to the end of the output
        EMD_PERL_DEBUG("$LOG_CATEGORY Use OH=$emdHomeRef for 902/903 discovery");
        push (@processedLines, processPre904Targets($ORACLEHOME, $hostName, $emdHomeRef, $onlyThisHome, \%iasTargetTypes));
    }
}

push (@processedLines, "\<\/Targets\>\n") if (!exists $ENV{LL_ORACLE_HOME});
print @processedLines;

exit 0;

sub get1013PlusTargets
{
    my ($iasHome, $oraclehome, $iasTargetTypes,$iasVersion, $iasHostName,$iasComponents) = @_;
    my @targetLines;
    my @componentArray;
    my @results = simpleXPathQuery(
                      "$iasHome/opmn/conf/opmn.xml",
                      "opmn/process-manager/ias-instance");

    my $iasInstanceName = $results[-1]{'name'};
    $iasInstanceName = $results[-1]{'id'} if ($iasInstanceName eq "");

    @componentArray = split( ',', $iasComponents );
    my $hasOC4J = 0;
    my $hasOHS = 0;
    my @oc4jInstances;
    for my $iasComponent (@componentArray)
    {
        if (lc($iasComponent) eq "j2ee")
        {
            $hasOC4J = 1;
            @oc4jInstances = list1013PlusOC4JInstances($iasHome);
        }
        elsif (lc($iasComponent) eq "apache")
        {
            $hasOHS = 1;
        }
        else 
        {
            EMD_PERL_DEBUG("Unexpected iAS component = $iasComponent");
        }
    }
    # default Http Port to EMPTY (we expect a numeric)
    my $httpPORT = "";
    my $ohsVirtualHostName;
    if ($hasOHS)
    {
        setOracleHome($iasHome);
        my $hash = parseApacheConf();
        $httpPORT = peek($$hash{'Port'});
        $ohsVirtualHostName = peek($$hash{'ServerName'});
    }
    else
    {
        @results = simpleXPathQuery(
                      "$iasHome/opmn/conf/opmn.xml",
                      'opmn/process-manager/ias-instance/ias-component/process-type[@module-id="OC4J"]/port[@id="default-web-site"]');
        for my $resultRef (@results)
        {
            $httpPORT = $$resultRef{'range'};
        }
    }
    $ohsVirtualHostName = $iasHostName if ($ohsVirtualHostName eq "");

    EMD_PERL_DEBUG("iAS HostName = $iasHostName");
    EMD_PERL_DEBUG("iAS Port  = $httpPORT");
    EMD_PERL_DEBUG("iAS Oracle Home = $iasHome");
    EMD_PERL_DEBUG("iAS Instance Name = $iasInstanceName");
    EMD_PERL_DEBUG("OC4J Instance Names = @oc4jInstances");
#   push(@targetLines,add1013PlusIASTarget($iasHome, $iasInstanceName, $iasHostName,$httpPORT));

    # Get the 1013plus iAS component (like BPEL, content services etc) targets
    # which are present in the $OH/sysman/emd/targets.xml file
    push(@targetLines,add1013PlusNonInfraTargets($iasHome, $emdRoot));

    # Pass blank string as iasComponentName for pre-11g OHS targets
    push(@targetLines,add1013PlusOHSTarget($iasHome, $iasInstanceName, "", $iasHostName, $httpPORT, $iasVersion)) if ($hasOHS);

    push(@targetLines,add1013PlusOC4JTargets($iasHome, $iasInstanceName, $iasHostName,\$httpPORT,$iasVersion, $hasOHS, @oc4jInstances)) if ($hasOC4J);

    # Should we add ias targets at the end as before??
    # unshift(@targetLines,add1013PlusIASTarget($iasHome, $iasInstanceName, $iasHostName,$httpPORT));
    push(@targetLines,add1013PlusIASTarget($iasHome, $iasInstanceName, $iasHostName,$httpPORT,$ohsVirtualHostName));
    EMD_PERL_DEBUG("***** Processed UNSHIFTED 1013 targets lines = @targetLines");
    return @targetLines;
}

sub findTargets
{
    my ($ref_iasMembers, $ref_otherMembers, $ref_targetLines, $fileHandle) = @_;

    my $inTarget = 0;
    my $isCompositeMember = 0;
 
    my $targetLine;
    my $currTargetName;
    my $currTargetType;
    my $tgtContainerName;
    my $tgtContainerType;
    #indicates are we at the target end tag
    my $targetEnd;
    #indicates weather version_value property already populated or not
    my $vValueIsIn = 0; 
    #loops through each line of targetlines(of target.xml given)
    while($targetLine = <$fileHandle>)
    {
        $targetLine =~ s/^\s*//;
        # reset targetEnd because it should only be set if current interation is target end line
        $targetEnd =0;

        # skip non-XML stuff, header, footer
        next if (index($targetLine, "\<") != 0);
        next if (($targetLine =~ m?<Targets?) || ($targetLine =~ m?</Targets?));

        # check - are we scanning a target?  
        # if so, look for the end target tag
        if ($inTarget > 0)
        {
            if (index($targetLine, "\/Target") >= 0)
            {
                # reset flags
                $inTarget = 0;
                $isCompositeMember = 0;
                #This indicates we are in target end line
                $targetEnd = 1;
               
            }
            # are we handling a target that has composite membership?
            if ($isCompositeMember > 0)
            {
                if (index($targetLine, "\<MemberOf") >= 0)
                {
                    # get name/type of "containing" target
                    $tgtContainerType = substr ($targetLine, index ($targetLine, "TYPE")+length("TYPE=\""));
                    $tgtContainerType = substr ($tgtContainerType, 0, index($tgtContainerType, "\""));
                    $tgtContainerName = substr ($targetLine, index ($targetLine, "NAME")+length("NAME=\""));
                    $tgtContainerName = substr ($tgtContainerName, 0, index($tgtContainerName, "\""));
                    # add to appropriate membership hash
                    if ($tgtContainerType eq "oracle_ias")
                    {
                        ${$ref_iasMembers}{$currTargetName . ":" . $currTargetType} = $tgtContainerName . ":" . $tgtContainerType;
                    }
                    else
                    {
                        ${$ref_otherMembers}{$currTargetName . ":" . $currTargetType} = $tgtContainerName . ":" . $tgtContainerType;
                    }
                }
            }
            # check for CompositeMembership tag
            $isCompositeMember = 1 if (index($targetLine, "\<CompositeMembership") >= 0);
        }
        # look for the start tag for a new target
        if ($inTarget == 0)
        {
            if (index($targetLine, "\<Target TYPE") >= 0)
            {
                $currTargetType = substr ($targetLine, index ($targetLine, "\<Target TYPE")+length("\<Target TYPE=\""));
                $currTargetType = substr ($currTargetType, 0, index($currTargetType, "\""));
                $currTargetName = substr ($targetLine, index ($targetLine, "NAME")+length("NAME=\""));
                $currTargetName = substr ($currTargetName, 0, index($currTargetName, "\""));
                $targetLine =~ s/\sON_HOST=".*?"//;
                $inTarget = 1;
                #reset vValueIsIn; because we havnt inserted version_value yet
                $vValueIsIn = 0;
            }
            # watch out for one-liners!
            if ((index($targetLine, "\<Target") >= 0) && (index($targetLine, "\/\>") >= 0))
            {
                # reset flags
                $inTarget = 0;
                $isCompositeMember = 0;
            }
        }
        #if its oracle_ias target  and we did not comeacross version perameter, then it is time to add version_value
        if ( ($targetEnd > 0) && ($currTargetType eq "oracle_ias") && ($vValueIsIn == 0) )
        {
             push (@{$ref_targetLines}, "<Property NAME=\"version_value\" VALUE=\"$iasVersion\"/>\n");
             #Note version value is inserted (prevents duplication)
             $vValueIsIn =1;
        }

        push (@{$ref_targetLines}, $targetLine);

	 # print version_value for BPEL target (oracle_integrationbpm)

        if( ($targetLine =~ /^\s*<\s*Property\s*NAME=\"OracleHome\"\s*VALUE=\"(.*?)\".*$/i ) && ($vValueIsIn ==0) && ($currTargetType eq "oracle_integrationbpm"))
        {
             push (@{$ref_targetLines}, "<Property NAME=\"version_value\" VALUE=\"$iasVersion\"/>\n");
             #Note version value is inserted (prevents duplication)
             $vValueIsIn =1;
        }

        # print version_value for targets otherthan oracle_ias
        # target.xml will already have version as an atribute
        if( ($targetLine =~ /^\s*<\s*Property\s*NAME=\"version\"\s*VALUE=\"(.*?)\".*$/i ) && ($vValueIsIn ==0) )
        {
            #Note version value is inserted(prevents duplication)
            $vValueIsIn =1;
            push (@{$ref_targetLines}, "<Property NAME=\"version_value\" VALUE=\"$1\"/>\n");
        }

        next;
    }
}

# Get the non Infrastructure iAS targets like BPEL etc.
# from the OH/sysman/emd/targets.xml
# Does some additional validation to make sure that the
# EM central agent does know abt these target types by verifying
# existence of the metadata files 
#
sub add1013PlusNonInfraTargets
{
    my ($iasHome, $emdRoot) = @_;
    my %iasMembers = ();
    my %otherMembers = ();

    my (@handledLines, @targetLines);

    my $isIASTarget = 0;

    # read in targets.xml from given home
    my $targetsFile = $iasHome . "/sysman/emd/targets.xml";
    my $metadataDir = $emdRoot . "/sysman/admin/metadata/";

    if (-e $targetsFile)
    {
        EMD_PERL_DEBUG("$targetsFile FOUND  in $iasHome\n"); 
        my $savedEmdRoot=$ENV{EMDROOT};
        delete($ENV{EMDROOT});
        # get full target listing - pass source, destination, extensionRoot
        open (TARGETS_FILE_READER, "<$targetsFile");
        $ENV{EMDROOT}=$savedEmdRoot;

        findTargets(\%iasMembers, \%otherMembers, \@targetLines, \*TARGETS_FILE_READER);    

        close(TARGETS_FILE_READER);    

        my $currTargetName = "";
        my $currTargetType = "";

        # pass 2: collect iAS-related targets
        my $line;
        foreach $line (@targetLines)
        {
            # find the start tag for a target
            if (index ($line, "Target TYPE=") >= 0 )
            {
                $currTargetName = substr $line, index($line, "NAME=\"")+length("NAME=\"");
                $currTargetName = substr ($currTargetName, 0, index($currTargetName, "\""));
                $currTargetType = substr ($line, index ($line, "Target TYPE")+length("Target TYPE=\""));
                $currTargetType = substr ($currTargetType, 0, index($currTargetType, "\""));
            }
            my $metaFile = $metadataDir . $currTargetType . ".xml" ;
            if (-e  $metaFile && $currTargetType ne "host")
            {
                EMD_PERL_DEBUG("$metaFile WAS found. \n");
                $isIASTarget = 1;
            }
            else
            {
                EMD_PERL_DEBUG("$metaFile NOT found. \n");
            }
            # we've verified this is a discoverable iAS-related target; push its info
            push (@handledLines, $line) if ($isIASTarget > 0);

            # hit an end-of-target tag; reset flag
            $isIASTarget = 0 if (index($line, "\/Target") >= 0);
        }
    }
    return @handledLines;
}

sub checkIfOMSIs1013PlusCapable
{
    # Get the OMS version
    my ($oraclehome) = @_;
    my $proto_ini_file = $oraclehome . "/sysman/emd/protocol.ini";
    # If file does not exist, the agent-OMS handshake has not yet happened.
    # We take the optimistic approach and assume the OMS version would
    # be atleast this version or higher
    return 1 if (! -e $proto_ini_file);

    my $omsVersion = getPropertyFromFile($oraclehome, "/sysman/emd/protocol.ini", "OMS_VERSION");
    EMD_PERL_DEBUG("OMS version is $omsVersion\n");
    my ($omsMajorVer,$omsFirstSub,$omsSecondSub,$omsThirdSub) =  split(/\./ , $omsVersion);
    EMD_PERL_DEBUG("OMS sub versions are $omsMajorVer : $omsFirstSub : $omsSecondSub : $omsThirdSub \n");
    my $isOMS1013PlusCapable = 0;
    if (($omsMajorVer > 10) ||
        (($omsMajorVer == 10) && ($omsFirstSub > 2)) ||
        (($omsMajorVer == 10) && ($omsFirstSub == 2) && ($omsSecondSub >0)) ||
        (($omsMajorVer == 10) && ($omsFirstSub == 2) && ($omsSecondSub ==0) && ($omsThirdSub >=2 )))
     {
         $isOMS1013PlusCapable = 1;
     }
     return $isOMS1013PlusCapable;
}

sub createTargetTypeXMLTag
{
    my ($targetType, $targetName, $displayName) = @_;
    my $targetTypeTag = "<Target TYPE=\"$targetType\" NAME=\"$targetName\"";

    $targetTypeTag = "$targetTypeTag DISPLAY_NAME=\"$displayName\"" if ($displayName ne "");

    $targetTypeTag = "$targetTypeTag>\n";
    return $targetTypeTag;
}

sub createTargetPropertyXMLTag
{
    my ($propName, $propValue) = @_;
    my $xmlstring = "\t<Property NAME=\"$propName\" VALUE=\"$propValue\"/>\n";
    return $xmlstring;
}

sub createAssocTargetXMLTag
{
    my ($assocName, $assocTargetType, $assocTargetName) = @_;
    my $xmlstring = "\t<AssocTargetInstance ASSOCIATION_NAME=\"$assocName\" ASSOC_TARGET_TYPE=\"$assocTargetType\" ASSOC_TARGET_NAME=\"$assocTargetName\"/>\n";
    return $xmlstring;
}

sub createCompositeMembershipXMLTag
{
    my ($memberOfType, $memberOfName, $associationName) = @_;
    my $xmlstring = "\t<CompositeMembership>\n";
    $xmlstring = $xmlstring . "\t\t<MemberOf TYPE=\"$memberOfType\" NAME=\"$memberOfName\" ASSOCIATION=\"$associationName\"/>\n";
    $xmlstring = $xmlstring . "\t</CompositeMembership>\n";
    return $xmlstring;
}

sub createOC4JJVMTargetInstanceString
{
    my ($iasHostName, $iasHome,$oc4jInstanceName,$oc4jInstanceDisplayName,$jvmid) = @_;
    my $jvmInstanceDisplayName = "JVM_" . $jvmid;
    my $jvmInstanceName = $oc4jInstanceName . "_" . $jvmInstanceDisplayName;
    #get version value from ias::oc4j::getJVMData::getJVMVersion
    my $version_value = getJVMVersion($iasHome,$oc4jInstanceDisplayName);
    my $targetTypeTag = createTargetTypeXMLTag("oc4jjvm",$jvmInstanceName ,$jvmInstanceDisplayName);
    my $host_PropTag = createTargetPropertyXMLTag("HTTPMachine",$iasHostName);
    my $uri_PropTag = createTargetPropertyXMLTag("URI","/JMXSoapAdapter/JMXSoapAdapter");
    my $OH_PropTag   = createTargetPropertyXMLTag("OracleHome",$iasHome);
    my $oc4jInstance_PropTag   = createTargetPropertyXMLTag("OC4JInstanceName",$oc4jInstanceDisplayName);
    my $jvmid_PropTag   = createTargetPropertyXMLTag("JVMId",$jvmid);
    my $mgmtWebSite_PropTag   = createTargetPropertyXMLTag("MgmtWebSite","default-web-site");
    my $ver_PropTag = '';
    if (defined $version_value && $version_value ne "")
    {
    $ver_PropTag = createTargetPropertyXMLTag("version_value",$version_value);
    }
    my $user_PropTag = createTargetPropertyXMLTag("authUser","oc4jadmin");
    # User would have to enter password via Monitoring properties for 
    # the target in OMS
#    my $pwd_PropTag = createTargetPropertyXMLTag("authPasswd","xyz");
    my $Assoc_PropTag   = createAssocTargetXMLTag("oc4j","oc4j",$oc4jInstanceName);
    my $Composite_PropTag   = createCompositeMembershipXMLTag("oc4j",$oc4jInstanceName,"JVM");

    my $jvmTargetString = $targetTypeTag .
                          $host_PropTag .
                          $uri_PropTag .
                          $OH_PropTag .
                          $oc4jInstance_PropTag .
                          $jvmid_PropTag .
                          $mgmtWebSite_PropTag .
                          $ver_PropTag .
                          $user_PropTag .
                          $Assoc_PropTag .
                          $Composite_PropTag .
                          getTargetEndTag();
    EMD_PERL_DEBUG("JVM target string is $jvmTargetString");
    return $jvmTargetString;
}

sub getTargetEndTag
{
    return "</Target>\n";
}

sub add1013PlusIASTarget
{
    my ($iasHome, $iasInstanceName, $iasHostName, $httpPORT, $ohsVirtualHostName) = @_;
    my $targetTypeTag = createTargetTypeXMLTag("oracle_ias",$iasInstanceName);
    my $OH_PropTag   = createTargetPropertyXMLTag("OracleHome",$iasHome);
    my $host_PropTag = createTargetPropertyXMLTag("host",$iasHostName);
    # add version_value target property
    my $version_PropTag = createTargetPropertyXMLTag("version_value",$iasVersion);

    my $statusURL = "http://$ohsVirtualHostName:$httpPORT";
    my $sa_ConsoleURL = "http://$ohsVirtualHostName:$httpPORT/em";
    my $StatusURL_PropTag = createTargetPropertyXMLTag("StatusURL",$statusURL);
    my $SAConsole_PropTag = createTargetPropertyXMLTag("StandaloneConsoleURL",$sa_ConsoleURL);
    EMD_PERL_DEBUG("OH Prop tag = $OH_PropTag");
    EMD_PERL_DEBUG("Host Prop tag = $host_PropTag");
    EMD_PERL_DEBUG("StatusURL Prop tag = $StatusURL_PropTag");
    EMD_PERL_DEBUG("SAConsole Prop tag = $SAConsole_PropTag");
    my $iasTargetLines = $targetTypeTag .
                         $version_PropTag .
                         $OH_PropTag .
                         $host_PropTag .
                         $StatusURL_PropTag .
                         $SAConsole_PropTag .
                         getTargetEndTag();
    EMD_PERL_DEBUG("1013 iAS targets lines = $iasTargetLines");
    return $iasTargetLines;
}

sub add1013PlusOHSTarget
{
    my ($iasHome, $iasInstanceName, $iasComponentName, $iasHostName, $httpPORT, $iasVersion) = @_;

    my $compId_PropTag  = "";
    my $ohsInstanceName  = "";
    if ($iasComponentName ne "") {
        # ohsInstanceName is now made up of the iasInstanceName and the iasComponentName
        $ohsInstanceName = $iasInstanceName . "_" . $iasComponentName . "_HTTP Server";
        # Add a new target property for IAS Component ID
        $compId_PropTag  = createTargetPropertyXMLTag("IASComponentID",$iasComponentName);
    } else {
        $ohsInstanceName = $iasInstanceName . "_HTTP Server";
    }
    my $targetTypeTag = createTargetTypeXMLTag("oracle_apache",$ohsInstanceName,"HTTP_Server");
    my $host_PropTag = createTargetPropertyXMLTag("HTTPMachine",$iasHostName);
    my $port_PropTag = createTargetPropertyXMLTag("HTTPPort",$httpPORT);
    my $verVal_PropTag = createTargetPropertyXMLTag("version_value",$iasVersion);
    my $ver_PropTag = createTargetPropertyXMLTag("version",$iasVersion);
    my $OH_PropTag   = createTargetPropertyXMLTag("OracleHome",$iasHome);
    my $Assoc_PropTag   = createAssocTargetXMLTag("ias","oracle_ias",$iasInstanceName);
    my $Composite_PropTag   = createCompositeMembershipXMLTag("oracle_ias",$iasInstanceName,"Apache target");

    my $ohsTargetLine = $targetTypeTag .
                        $host_PropTag .
                        $port_PropTag .
                        $ver_PropTag .
                        $verVal_PropTag .
                        $OH_PropTag .
                        $compId_PropTag .
                        $Assoc_PropTag .
                        $Composite_PropTag .
                        getTargetEndTag();
    EMD_PERL_DEBUG("1013 OHS targets lines = $ohsTargetLine");
    return $ohsTargetLine;
}

sub getNumJVMsFor1013PlusOC4J
{
    my ($iasHome, $oc4jInstanceDisplayName) = @_;
    my $numJVMs = "1";
    my @processSet = simpleXPathQuery(
                      "$iasHome/opmn/conf/opmn.xml",
                      'opmn/process-manager/ias-instance/ias-component/process-type[@module-id="OC4J"][@id="' . $oc4jInstanceDisplayName . '"]/process-set');
    for my $resultRef (@processSet)
    {
        $numJVMs = $$resultRef{'numprocs'};
    }
     return $numJVMs;
}

sub isJVMMBeansEnabledOn1013PlusOC4J
{
    my ($iasHome, $oc4jInstanceDisplayName) = @_;
    my $isEnabled = "0";
    EMD_PERL_DEBUG("is JVMs enabled for $oc4jInstanceDisplayName");
    setOracleHome($iasHome);
    my %moduleData = ias::asecm::getOpmnModuleData({
      'module' => 'OC4J',
      'process-type' => $oc4jInstanceDisplayName,
    });
    my $javaStartOptions = $moduleData{'start-parameters'}{'java-options'};
    if (defined($javaStartOptions) && index($javaStartOptions,"-Dcom.sun.management.jmxremote") >= 0)
    {
        $isEnabled = "1";
        EMD_PERL_DEBUG("JVM Mbeans enabled for $oc4jInstanceDisplayName");
    }
    else
    {
        EMD_PERL_DEBUG("NO JVM Mbeans enabled for $oc4jInstanceDisplayName");
    }
    return $isEnabled; 
}

sub does1013PlusOC4JHaveAscontrolEnabled
{
    my ($iasHome, $oc4jInstanceDisplayName) = @_;
    my $isEnabled = "0";
    my @results = simpleXPathQuery(
                      "$iasHome/j2ee/$oc4jInstanceDisplayName/config/server.xml",
                      'application-server/application[@name="ascontrol"]');
    for my $resultRef (@results)
    {
        my $isStarted = $$resultRef{'start'};
        $isEnabled = "1" if ($isStarted eq "true");
    }
    EMD_PERL_DEBUG("Status of ascontrol enabled on this oc4j $oc4jInstanceDisplayName is  = $isEnabled");
    return $isEnabled;
}

sub get1013PlusOC4J_LM_ASControl_Port
{
    my $ascontrolPort = "UNKNOWN";
    my ($iasHome, $oc4jInstanceName) = @_;
    my $ascWebSite = getWebSiteFor1013PlusAscontrol($iasHome,$oc4jInstanceName);
    my @results = simpleXPathQuery(
                      "$iasHome/opmn/conf/opmn.xml",
                      'opmn//ias-component/process-type[@module-id="OC4J"][@id="' . "$oc4jInstanceName" . '"]/port[@id="' . "$ascWebSite" . '"]');
    for my $resultRef (@results)
    {
        my $ascWebSitePort = $$resultRef{'range'};
        my $ascWebSiteProtocol = $$resultRef{'protocol'};
        if (lc($ascWebSiteProtocol) eq lc('http'))
        {
            if (index($ascWebSitePort,"-") >= 0)
            {
                $ascWebSitePort = substr($ascWebSitePort, 0, index($ascWebSitePort, "-"));
            }
            $ascontrolPort = $ascWebSitePort;
        }
    }
    return $ascontrolPort;
}

sub getWebSiteFor1013PlusAscontrol
{
    my ($iasHome, $oc4jInstanceName) = @_;
    #find all websites since ascontrol may not be on the default one?
    my @results = simpleXPathQuery(
                      "$iasHome/j2ee/$oc4jInstanceName/config/server.xml",
                      'application-server/web-site');
    my $websiteXmlFile;
    my $websiteXmlFileName;
    my $ascWebSite;
    WEBSITE: for my $resultRef (@results)
    {
        $websiteXmlFileName = $$resultRef{'path'};
        $websiteXmlFile = "$iasHome/j2ee/$oc4jInstanceName/config/" . $websiteXmlFileName;
        EMD_PERL_DEBUG ("website xml file for oc4j instance $oc4jInstanceName is $websiteXmlFile");

        my @ascontrolApp = simpleXPathQuery(
                      $websiteXmlFile,
                      'web-site/web-app[@application="ascontrol"]');
        for my $ascRef (@ascontrolApp)
        {
            EMD_PERL_DEBUG("Found ascontrol in website");
            #my $rootURL = $$ascRef{'root'};
            #EMD_PERL_DEBUG("Got rootURL as $rootURL");
            $ascWebSite = substr ($websiteXmlFileName, 0, index($websiteXmlFileName, ".xml"));
            if (index($ascWebSite,".") == 0)
            {
                $ascWebSite = substr ($ascWebSite, 2);
            }
            last WEBSITE;
        }
    }
    return $ascWebSite;
}

sub add1013PlusOC4JTargets
{
    (my $iasHome, my $iasInstanceName, my $iasHostName,my $httpPORT, my $iasVersion,my $hasOHS, my @oc4jInstances) = @_;
    my $oc4jTargetString = "";
    my $oc4jInstanceName;
    my $oc4jInstanceDisplayName;

    EMD_PERL_DEBUG ("oc4j instances are  @oc4jInstances");
    foreach $oc4jInstanceDisplayName (@oc4jInstances)
    {
    # Should we have "_" or " " between iasInstance and oc4jInstance?
        $oc4jInstanceName = $iasInstanceName . "_$oc4jInstanceDisplayName";
        my $targetTypeTag = createTargetTypeXMLTag("oc4j",$oc4jInstanceName,$oc4jInstanceDisplayName);
        my $host_PropTag = createTargetPropertyXMLTag("HTTPMachine",$iasHostName);
        my $OH_PropTag   = createTargetPropertyXMLTag("OracleHome",$iasHome);
        my $ver_PropTag = createTargetPropertyXMLTag("version",$iasVersion);
        my $verVal_PropTag = createTargetPropertyXMLTag("version_value",$iasVersion);
        my $Assoc_PropTag   = createAssocTargetXMLTag("ias","oracle_ias",$iasInstanceName);
        my $Composite_PropTag   = createCompositeMembershipXMLTag("oracle_ias",$iasInstanceName,"Servlet Engine");

        $oc4jTargetString = $oc4jTargetString .
                            $targetTypeTag .
                            $host_PropTag .
                            $OH_PropTag .
                            $ver_PropTag .
                            $verVal_PropTag .
                            $Assoc_PropTag .
                            $Composite_PropTag .
                            getTargetEndTag();
        # Now check if jdk1.5 is enabled to expose jvm mbeans so that oc4jjvm
        # targets can also be auto-discovered

        # We unconditionally discover oc4jjvm for 1013plus oc4js
        # if (isJVMMBeansEnabledOn1013PlusOC4J($iasHome, $oc4jInstanceDisplayName))
        # {
        my $numJVMs = getNumJVMsFor1013PlusOC4J($iasHome, $oc4jInstanceDisplayName);
        EMD_PERL_DEBUG("Num JVMS for the oc4j instance $oc4jInstanceDisplayName is $numJVMs");
        for (my $i = 1; $i <= $numJVMs; $i += 1)
        {
            my $jvmTargetInstanceString = createOC4JJVMTargetInstanceString($iasHostName, $iasHome, $oc4jInstanceName,$oc4jInstanceDisplayName, $i);
            $oc4jTargetString = $oc4jTargetString . $jvmTargetInstanceString;
        }
        # }
 
        if (!$hasOHS)
        {
            # Check if http protocol is used... in that case this is a standalone oc4j
            # Can there be multiple oc4j instances here without OHS and with http??
            if (does1013PlusOC4JHaveAscontrolEnabled($iasHome, $oc4jInstanceDisplayName))
            {
                EMD_PERL_DEBUG("ascontrol is enabled on $oc4jInstanceDisplayName");
                # Get the ascontrol port for this Locally Managed OC4J instance
                my $ascPort = get1013PlusOC4J_LM_ASControl_Port($iasHome, $oc4jInstanceDisplayName);
                EMD_PERL_DEBUG("ascontrol port is $ascPort");
                $$httpPORT = $ascPort if ($ascPort ne "");
            }
            else
            {
                EMD_PERL_DEBUG("ascontrol is disabled on $oc4jInstanceDisplayName");
            }
        }
    }
    EMD_PERL_DEBUG("1013 OC4J targets lines = $oc4jTargetString");
    return $oc4jTargetString;
}

sub list1013PlusOC4JInstances
{
    my $oraclehome = $_[0];
    my @results;

    my @enabledOC4Js = simpleXPathQuery(
                      "$oraclehome/opmn/conf/opmn.xml",
                      'opmn/process-manager/ias-instance/ias-component/process-type[@module-id="OC4J"][@status="enabled"]');

    for my  $oc4jInstance (@enabledOC4Js)
    {
        my $oc4jInstanceDisplayName = $$oc4jInstance{'id'};
        push(@results, $oc4jInstanceDisplayName);
    }
    return @results;
}

sub get904PlusTargets
{
    my ($iasHome, $oraclehome, $iasTargetTypes) = @_;
    my %iasMembers = ();
    my %otherMembers = ();

    my (@handledLines, @targetLines);

    my $isIASTarget = 0;

    # read in targets.xml from given home
    my $targetsFile = $iasHome . "/sysman/emd/targets.xml";

    if (-e $targetsFile)
    {
        my $savedEmdRoot=$ENV{EMDROOT};
        delete($ENV{EMDROOT});
        open (TARGETS_FILE_READER, "<$targetsFile");
        $ENV{EMDROOT}=$savedEmdRoot;

        findTargets(\%iasMembers, \%otherMembers, \@targetLines, \*TARGETS_FILE_READER);

        close(TARGETS_FILE_READER);    

        my $currTargetName = "";
        my $currTargetType = "";

        # pass 2: collect iAS-related targets
        my $line;
        my $moreChecks;
        foreach $line (@targetLines)
        {
            # find the start tag for a target
            if (index ($line, "Target TYPE=") >= 0 )
            {
                $currTargetName = substr $line, index($line, "NAME=\"")+length("NAME=\"");
                $currTargetName = substr ($currTargetName, 0, index($currTargetName, "\""));
                $currTargetType = substr ($line, index ($line, "Target TYPE")+length("Target TYPE=\""));
                $currTargetType = substr ($currTargetType, 0, index($currTargetType, "\""));
            }

            # check to see if it's a type we care about; first if in the list
            $isIASTarget = 1 if (defined ($iasTargetTypes->{$currTargetType}));

            # check whether this is an IAS target by checking memberships
            if ($isIASTarget == 0)
            {
                # the fun part - check, recursively, if this target is a member or submember of iAS
                $targetMemberCheck = $currTargetName . ":" . $currTargetType;

                # the easy case - direct membership in an oracle_ias target
                $isIASTarget = 1 if (defined($iasMembers{$targetMemberCheck}));

                # still checking; look for membership in other targets, 
                # then check those targets' membership in an iAS -- recurse...
                if ($isIASTarget == 0)
                {
                    $moreChecks = 1;
                    while (($moreChecks > 0) && ($isIASTarget == 0))
                    {
                        if (defined($otherMembers{$targetMemberCheck}))
                        {
                            $otherMemberName = $otherMembers{$targetMemberCheck};
                            $isIASTarget = 1 if (defined($iasMembers{$otherMemberName}));
                            $targetMemberCheck = $otherMemberName;
                        }
                        else
                        {
                            $moreChecks = 0;
                        }
                    }
                }
            }

            # we've verified this is an iAS-related target; push its info
            push (@handledLines, $line) if ($isIASTarget > 0);

            # hit an end-of-target tag; reset flag, Name/Type values
            if (index($line, "\/Target") >= 0)
            {
                $isIASTarget = 0;
                $currTargetName = "";
                $currTargetType = "";
            }
        }
    }
    return @handledLines;
}

sub processPre904Targets
{
    my ($oraclehome, $hostName, $emdHomeRef, $onlyThisHome, $iasTargetTypes) = @_;

    # read in targets.xml from active home
    my $targetsFile = $emdHomeRef . "/sysman/emd/targets.xml";

    my $ignoringTarget = 0;
    my $skipVersionProperty = 0;
    my $needIasDisplayName = 0;
    my $needIasApacheInstancePort = 0;
    my $needIasOracleHome = 0;
    my $needHTTPMachine = 0;
    my $needConnectDesc = 0;
    my $hostTargetName;

    my $skipLogFileDir = 0;
    my $skipLogFileName = 0;
    my %iasDisplayNames = ();
    my %iasVersions = ();
    my %iasOracleHomes = ();
    my %iasInstances = ();
    my %iasContainers = ();
    my %iasApacheInstanceNames = ();
    my %iasApacheInstancePorts = ();
    my %httpMachines = ();
    my %allTargets = ();
    my (@handledLines, @pass2Lines, $targetLine, $currTargetDisplayName, $currTargetName, $currTargetType);
    my $currTarget = [];

    if (-e $targetsFile)
    {
        # get the EMD_URL and strip off the "/emd/main" part, add "/emd/console", make XML property
        my $emdUrl = getPropertyFromFile($emdHomeRef, "/sysman/config/emd.properties", "EMD_URL");
        $emdUrl = substr ($emdUrl, 0, index($emdUrl, "/emd/main"))."/emd/console";
        $emdUrl =  "<Property NAME=\"StandaloneConsoleURL\" VALUE=\"$emdUrl\"/>\n";

        my $savedEmdRoot=$ENV{EMDROOT};
        delete($ENV{EMDROOT});
        # get full target listing - pass source, destination, extensionRoot
        open (TARGETS_FILE_READER, "<$targetsFile");
        $ENV{EMDROOT}=$savedEmdRoot;

        while($targetLine = <TARGETS_FILE_READER>)
        {
            $targetLine =~ s/^\s*//;

            # skip non-XML stuff, header, footer
            next if (index($targetLine, "\<") != 0);
            next if (($targetLine eq "\<Targets\>\n") || ($targetLine eq "\<\/Targets\>\n"));

            # check - are we skipping the current target?  
            # if so, look for the end target tag
            if ($ignoringTarget > 0)
            {
                $ignoringTarget = 0 if (index($targetLine, "\/Target") >= 0);
                next;
            }
            # is this the start of a new target?
            if (index($targetLine, "\<Target TYPE") >= 0)
            {
                $ignoringTarget = 1;
                $currTargetType = substr ($targetLine, index ($targetLine, "\<Target TYPE")+length("\<Target TYPE=\""));
                $currTargetType = substr ($currTargetType, 0, index($currTargetType, "\""));
                # check to see if it's a type we care about
                $ignoringTarget = 0 if (defined ($iasTargetTypes->{$currTargetType}));

                # reset flags (in case they got set but not reset for previous target)
                $skipVersionProperty = 0;
                $needIasDisplayName = 0;
                $needIasApacheInstancePort = 0;
                $needIasOracleHome = 0;
                $skipLogFileDir = 0;
                $skipLogFileName = 0;
                $needHTTPMachine = 0;
                $needConnectDesc = 0;
            }
            # we need to get the host name for later use in the AssocTargetInstance tag
            if ($currTargetType eq "host")
            {
                $currTargetName = substr $targetLine, index($targetLine, "NAME=\"")+length("NAME=\"");
                $currTargetName = substr ($currTargetName, 0, index($currTargetName, "\""));
                $hostTargetName = $currTargetName;
            }
            # one last check: are we going to skip the current target?
            # (may have been set by the test block above)
            next if ($ignoringTarget > 0);

            # is this the version property we've elected to skip
            if (($skipVersionProperty > 0) &&
                (index($targetLine, "\<Property NAME=\"version\"") >= 0))
            {
                $skipVersionProperty = 0;
                next;
            }
            # is this the logFileDir property we've elected to skip
            if (($skipLogFileDir > 0) &&
                (index($targetLine, "\<Property NAME=\"logFileDir\"") >= 0))
            {
                $skipLogFileDir = 0;
                next;
            }
            # is this the logFileName property we've elected to skip
            if (($skipLogFileName > 0) &&
                (index($targetLine, "\<Property NAME=\"logFileName\"") >= 0))
            {
                $skipLogFileName = 0;
                push (@$currTarget, "<Property NAME=\"logFileName\" VALUE=\"access_log\"/>\n");
                next;
            }
            # if we need the connect descriptor for LDAP, look for the TNSAddress property
            if (($needConnectDesc > 0) &&
                (index($targetLine, "\<Property NAME=\"TNSAddress\"") >= 0))
            {
                $ldapConnectDesc = substr $targetLine, index($targetLine, "VALUE=\"")+length("VALUE=\"");
                $ldapConnectDesc = substr $ldapConnectDesc, 0, index($ldapConnectDesc, "\"");
                push (@$currTarget, "<Property NAME=\"ConnectDescriptor\" VALUE=\"$ldapConnectDesc\"/>\n");
                $needConnectDesc = 0;
                next;
            }

            # add the current line to the output
            push (@$currTarget, $targetLine);

            # if this is a target closure tag, add the current set of
            # info, as an array, to the allTargets hash -- for later
            if (index($targetLine, "\/Target") >= 0)
            {
                $allTargets{$currTargetName} = $currTarget;
                $currTarget = [];
            }

            # check: is this the start of a target requiring special handling?
            if (index ($targetLine, "Target TYPE=") >= 0 )
            {
                # get target name for later use
                $currTargetName = substr $targetLine, index($targetLine, "NAME=\"")+length("NAME=\"");
                $currTargetName = substr ($currTargetName, 0, index($currTargetName, "\""));
                if ($currTargetType eq "oracle_ias")
                {
                    # add version_value to the output
                    push (@$currTarget, $versionValuePlaceholder."\n");
                    # add the EMD_URL to the output
                    push (@$currTarget, $emdUrl);
                    # push status URL placeholder for phase 2
                    push (@$currTarget, $statusUrlPlaceholder."\n");
                    $needIasOracleHome = 1;
                    # push associated target tag for phase 2
                    push (@$currTarget, $hostAssocTargetPlaceholder."\n");
                    $iasInstances{$currTargetName} = $currTargetName;
                }
                else
                {
                    # not an oracle_ias target; need target display name to generate the IAS_DISPLAY_NAME property
                    $currTargetDisplayName = substr($targetLine, index($targetLine, "DISPLAY_NAME=\"")+length("DISPLAY_NAME=\""));
                    $currTargetDisplayName = substr $currTargetDisplayName, 0, index($currTargetDisplayName, "\"");
                    $needIasDisplayName = 1;
                    push (@$currTarget, $iasDisplayNamePlaceholder."\n");
                }
                # we get the Apache port for later use in the iAS status URL
                # we also need to force the version number to the iAS version
                if ($currTargetType eq "oracle_apache")
                {
                    $needIasApacheInstancePort = 1;
                    $skipVersionProperty = 1;
                    push (@$currTarget, $iasVersionPlaceholder."\n");
                    # add version_value to the output
                    push (@$currTarget, $versionValuePlaceholder."\n");
                    # push associated target tag for phase 2
                    push (@$currTarget, $hostAssocTargetPlaceholder."\n");
                }
                # older webcache targets need OracleHome and version properties
                if ($currTargetType eq "oracle_webcache")
                {
                    $needHTTPMachine = 1;
                    $skipLogFileDir = 1;
                    $skipLogFileName = 1;
                    push (@$currTarget, $oracleHomePlaceholder."\n");
                    push (@$currTarget, $iasVersionPlaceholder."\n");
                    # add version_value to the output
                    push (@$currTarget, $versionValuePlaceholder."\n");
                    push (@$currTarget, $machinePlaceholder."\n");
                    push (@$currTarget, $logFileDirPlaceholder."\n");
                    # push associated target tag for phase 2
                    push (@$currTarget, $hostAssocTargetPlaceholder."\n");
                }
                # jserv targets need version property
                if (($currTargetType eq "oracle_jserv") ||
                    ($currTargetType eq "oc4j"))
                {
                    push (@$currTarget, $iasVersionPlaceholder."\n");
                    # add version_value to the output
                    push (@$currTarget, $versionValuePlaceholder."\n");
                    # push associated target tag for phase 2
                    push (@$currTarget, $hostAssocTargetPlaceholder."\n");
                }
                # SSO targets have their own set of required info
                if ($currTargetType eq "oracle_sso_server")
                {
                    push (@$currTarget, $oracleHomePlaceholder."\n");
                    push (@$currTarget, $iasVersionPlaceholder."\n");
                    # add version_value to the output
                    push (@$currTarget, $versionValuePlaceholder."\n");
                    push (@$currTarget, $startComponentPlaceholder."\n");
                }
                # LDAP targets need version and connect descriptor
                if ($currTargetType eq "oracle_ldap")
                {
                    push (@$currTarget, $iasVersionPlaceholder."\n");
                    # add version_value to the output
                    push (@$currTarget, $versionValuePlaceholder."\n");
                    $needConnectDesc = 1;
                }
                next; # go to next input line
            }
            # if we're processing an iAS composite target, get the name of the "containing" iAS instance
            # and use it to generate the IAS_DISPLAY_NAME property for the current target
            if ($needIasDisplayName > 0)
            {
                if ((index $targetLine, "MemberOf TYPE=\"oracle_ias\"") >= 0 )
                {
                    $iasInstanceName = substr $targetLine, index($targetLine, "NAME=\"")+length("NAME=\"");
                    $iasInstanceName = substr ($iasInstanceName, 0, index($iasInstanceName, "\""));
                    $iasContainers{$currTargetName} = $iasInstanceName;
                    $iasDisplayNames{$currTargetName} = $iasInstanceName . " " . $currTargetDisplayName;

                    $iasApacheInstanceNames{$iasInstanceName} = $currTargetName if ($currTargetType eq "oracle_apache");

                    $needIasDisplayName = 0;
                    next; # get next input line
                }
            }
            # get the Apache port number; we'll use this later for the iAS status URL
            if ($needIasApacheInstancePort > 0)
            {
                if ((index $targetLine, "Property NAME=\"HTTPPort\"") >= 0 )
                {
                    $iasApacheInstancePort = substr $targetLine, index($targetLine, "VALUE=\"")+length("VALUE=\"");
                    $iasApacheInstancePort = substr ($iasApacheInstancePort, 0, index($iasApacheInstancePort, "\""));
                    $iasApacheInstancePorts{$currTargetName} = $iasApacheInstancePort;
                    $needIasApacheInstancePort = 0;
                    next; # get next input line
                }
            }
            # get the HTTPMachine property; we'll use this later for webcache 
            if ($needHTTPMachine > 0)
            {
                if ((index $targetLine, "Property NAME=\"HTTPMachine\"") >= 0 )
                {
                    $httpMachine = substr $targetLine, index($targetLine, "VALUE=\"")+length("VALUE=\"");
                    $httpMachine = substr ($httpMachine, 0, index($httpMachine, "\""));
                    $httpMachines{$currTargetName} = $httpMachine;
                    $needHTTPMachine = 0;
                    next; # get next input line
                }
            }
            # for a given iAS, get the OracleHome property
            if ($needIasOracleHome > 0)
            {
                if ((index $targetLine, "Property NAME=\"OracleHome\"") >= 0 )
                {
                    $iasOracleHome = substr $targetLine, index($targetLine, "VALUE=\"")+length("VALUE=\"");
                    $iasOracleHome = substr ($iasOracleHome, 0, index($iasOracleHome, "\""));
                    $iasOracleHomes{$currTargetName} = $iasOracleHome;
                    # get the version of this iAS instance
                    $iasVersions{$currTargetName} = getPropertyFromFile($iasOracleHome, "/config/ias.properties", "Version");
                    $needIasOracleHome = 0;
                    next; # get next input line
                }
            }
        }
        close TARGETS_FILE_READER;
    }

    # walk the allTargets hash, and if $oracleConfigHome is not set
    # OR the target's oracleHome matches $oracleConfigHome, add the
    # target info to the big array of handled lines
    my $addTarget = 0;
    foreach $target(keys %allTargets)
    {
        if($IsWin32)
        {
            $addTarget = (length($oracleConfigHome) == 0) ||
                          (lc(Win32::GetShortPathName($iasOracleHomes{$iasContainers{$target}})) eq lc($oracleConfigHome)) ||
                          (lc(Win32::GetShortPathName($iasOracleHomes{$iasInstances{$target}}))  eq lc($oracleConfigHome))
        }
        else
        {
            $addTarget = (length($oracleConfigHome) == 0) ||
                          ($iasOracleHomes{$iasContainers{$target}} eq $oracleConfigHome) ||
                          ($iasOracleHomes{$iasInstances{$target}}  eq $oracleConfigHome)
        }
        if ($addTarget)
        {
            foreach (@{$allTargets{$target}})
            {
                push (@handledLines, $_);
            }
        }
    }

    #
    # Pass 2: plug in missing/generated properties
    #
    foreach $line (@handledLines)
    {
        chomp($line);
        if (index ($line, "Target TYPE=") >= 0 )
        {
            $currTargetType = substr ($line, index ($line, "\<Target TYPE")+length("\<Target TYPE=\""));
            $currTargetType = substr ($currTargetType, 0, index($currTargetType, "\""));
            $currTargetName = substr $line, index($line, "NAME=\"")+length("NAME=\"");
            $currTargetName = substr ($currTargetName, 0, index($currTargetName, "\""));
        }
        if ($line eq $statusUrlPlaceholder)
        {
            if (defined($iasApacheInstanceNames{$currTargetName}))
            {
                $apacheName = $iasApacheInstanceNames{$currTargetName};
                $line = "<Property NAME=\"StatusURL\" VALUE=\"http://$hostName:$iasApacheInstancePorts{$apacheName}/\"/>";
            }
            else
            {
                $line = "";
            }
        }
        if ($line eq $iasDisplayNamePlaceholder)
        {
            if (defined($iasDisplayNames{$currTargetName}))
            {
                $line = "<Property NAME=\"IAS_DISPLAY_NAME\" VALUE=\"$iasDisplayNames{$currTargetName}\"/>";
            }
            else
            {
                $line = "";
            }
        }
        if ($line eq $machinePlaceholder)
        {
            if (defined($httpMachines{$currTargetName}))
            {
                $line = "<Property NAME=\"HTTPMachineForAdmin\" VALUE=\"$httpMachines{$currTargetName}\"/>";
            }
            else
            {
                $line = "";
            }
        }
        if ($line eq $logFileDirPlaceholder)
        {
            if (defined($iasContainers{$currTargetName}))
            {
                $iasContainer = $iasContainers{$currTargetName};
                $line = "<Property NAME=\"logFileDir\" VALUE=\"$iasOracleHomes{$iasContainer}/webcache/logs\"/>";
            }
            else
            {
                $line = "";
            }
        }
        if ($line eq $iasVersionPlaceholder)
        {
            if (defined($iasContainers{$currTargetName}))
            {
                $iasContainer = $iasContainers{$currTargetName};
                $line = "<Property NAME=\"version\" VALUE=\"$iasVersions{$iasContainer}\"/>";
            }
            else
            {
                $line = "";
            }
        }
        if ($line eq $versionValuePlaceholder)
        {
            $line ="";
            if (defined($iasContainers{$currTargetName}))
            {
                $iasContainer = $iasContainers{$currTargetName};
                if ((defined($iasVersions{$iasContainer})) && ($iasVersions{$iasContainer} ne ""))
                { 
                   $line = "<Property NAME=\"version_value\" VALUE=\"$iasVersions{$iasContainer}\"/>";
                }
            }
            # if its not one of ias sub target it might be ias
            elsif (($currTargetType eq "oracle_ias")  && (defined($iasVersions{$currTargetName})) && ($iasVersions{$currTargetName} ne ""))
            {
                $line = "<Property NAME=\"version_value\" VALUE=\"$iasVersions{$currTargetName}\"/>";
            }
        }
        if ($line eq $oracleHomePlaceholder)
        {
            if (defined($iasContainers{$currTargetName}))
            {
                $iasContainer = $iasContainers{$currTargetName};
                $line = "<Property NAME=\"OracleHome\" VALUE=\"$iasOracleHomes{$iasContainer}\"/>";
            }
            else
            {
                $line = "";
            }
        }
        if ($line eq $startComponentPlaceholder)
        {
            if (defined($iasContainers{$currTargetName}))
            {
                $httpServer = $iasContainers{$currTargetName}."_HTTP Server";
                $line = "<Property NAME=\"startComponent\" VALUE=\"$httpServer\"/>";
            }
            else
            {
                $line = "";
            }
        }
        if ($line eq $hostAssocTargetPlaceholder)
        {
            $line = "<AssocTargetInstance ASSOC_TARGET=\"host\" TYPE=\"host\" NAME=\"$hostTargetName\"/>";
        }
        push (@pass2Lines, $line . "\n");

    }
    return @pass2Lines;
}

sub getIASR1Targets
{
    my ($oraclehome, $cfgHome, $hostName) = @_;
    my (@handledLines, @pass2Lines);
    my $iasName = "iASR1_".$oraclehome;
    EMD_PERL_DEBUG("$LOG_CATEGORY Use OH=$oraclehome for iAS 1022 discovery");

    # oracle_ias
    push (@handledLines, "<Target TYPE=\"oracle_ias\" NAME=\"".$iasName."\" VERSION=\"1.0\">\n");
    push (@handledLines, " <Property NAME=\"version_value\" VALUE=\"1.0.2.2\"/>\n");
    push (@handledLines, " <Property NAME=\"OracleHome\" VALUE=\"".$cfgHome."\"/>\n");
    push (@handledLines, " <Property NAME=\"host\" VALUE=\"".$hostName."\"/>\n");
    push (@handledLines, " <Property NAME=\"StandaloneConsoleURL\" VALUE=\"\"/>\n");
    push (@handledLines, " <Property NAME=\"StatusURL\" VALUE=\"".$statusUrlPlaceholder."\"/>\n");
    push (@handledLines, "</Target>\n");

    # apache [guaranteed?]
    my $apacheConf = $cfgHome . "/Apache/Apache/conf/httpd.conf";
    my $httpPort = "";
    if (-e $apacheConf)
    {
        $httpPort = getPropertyFromFile($cfgHome, "/Apache/Apache/conf/httpd.conf", "Listen", " ");
        push (@handledLines, "<Target TYPE=\"oracle_apache\" NAME=\"".$iasName."_HTTP Server\" DISPLAY_NAME=\"HTTP_Server\" VERSION=\"2.0\">\n");
        push (@handledLines, " <Property NAME=\"version\" VALUE=\"1.0.2.2\"/>\n");
        push (@handledLines, " <Property NAME=\"version_value\" VALUE=\"1.0.2.2\"/>\n");
        push (@handledLines, " <Property NAME=\"HTTPPort\" VALUE=\"".$httpPort."\"/>\n");
        push (@handledLines, " <Property NAME=\"HTTPMachine\" VALUE=\"".$hostName."\"/>\n");
        push (@handledLines, " <Property NAME=\"OracleHome\" VALUE=\"".$cfgHome."\"/>\n");
        push (@handledLines, " <AssocTargetInstance ASSOC_TARGET=\"ias\" TYPE=\"oracle_ias\" NAME=\"".$iasName."\"/>\n");
        push (@handledLines, " <CompositeMembership>\n");
        push (@handledLines, "  <MemberOf TYPE=\"oracle_ias\" NAME=\"".$iasName."\"/>\n");
        push (@handledLines, " </CompositeMembership>\n");
        push (@handledLines, "</Target>\n");

        # jserv... need to see if jserv.conf is called out in httpd.conf 
        #  or one of the recursively included config files
        my @httpdIncludes = recurseParseInclude($apacheConf);
        my $jservConf = undef;
        foreach $include(@httpdIncludes)
        {
            $jservConf = $include if ($include =~ /jserv/);
        }

        if (defined ($jservConf) && -e $jservConf)
        {
            push (@handledLines, "<Target TYPE=\"oracle_jserv\" NAME=\"".$iasName."_Jserv\" VERSION=\"2.0\">\n");
            push (@handledLines, " <Property NAME=\"HTTPMachine\" VALUE=\"".$hostName."\" />\n");
            push (@handledLines, " <Property NAME=\"version_value\" VALUE=\"1.0.2.2\"/>\n");
            push (@handledLines, " <Property NAME=\"OracleHome\" VALUE=\"".$cfgHome."\" />\n");
            push (@handledLines, " <CompositeMembership>\n");
            push (@handledLines, "  <MemberOf TYPE=\"oracle_ias\" NAME=\"".$iasName."\"/>\n");
            push (@handledLines, " </CompositeMembership>\n");
            push (@handledLines, "</Target>\n");
        }
    }

#  if (0)
#  {
#    # oc4j
#    # ebs claims OC4J is NOT used in ebs installs
#    my $oc4jRoot = $emdRoot . "/j2ee";
#    if (-e $oc4jRoot)
#    {
#      my $oc4jName = "";
#      push (@handledLines, "<Target TYPE=\"oc4j\" NAME=\"".$iasName.$oc4jName."\" DISPLAY_NAME=\"home\" VERSION=\"1.3\">\n");
#      push (@handledLines, "<Property NAME=\"HTTPMachine\" VALUE=\"".$hostName."\"/>\n");
#      push (@handledLines, "<Property NAME=\"OracleHome\" VALUE=\"".$emdRoot."\"/>\n");
#      push (@handledLines, "<Property NAME=\"version\" VALUE=\"1.0.2.2\"/>\n");
#      push (@handledLines, "<AssocTargetInstance ASSOC_TARGET=\"ias\" TYPE=\"oracle_ias\" NAME=\"".$iasName."\"/>\n");
#      push (@handledLines, "<CompositeMembership>\n");
#      push (@handledLines, "<MemberOf TYPE=\"oracle_ias\" NAME=\"".$iasName."\"/>\n");
#      push (@handledLines, "</CompositeMembership>\n");
#      push (@handledLines, "</Target>\n");
#    }
#
#  # ldap
#    push (@handledLines, "<Target TYPE=\"oracle_ldap\" NAME=\"".$iasName."_LDAP\" DISPLAY_NAME=\"OID\" VERSION=\"2.5\" ON_HOST=\"".$hostName."\">\n");
#    push (@handledLines, "<Property NAME=\"OracleHome\" VALUE=\"".$emdRoot."\"/>\n");
#    push (@handledLines, "<Property NAME=\"password\" VALUE=\"".$ldapPasswd."\" ENCRYPTED=\"FALSE\"/>\n");
#    push (@handledLines, "<Property NAME=\"LDAPScriptsPath\" VALUE=\"/sysman/admin/scripts\"/>\n");
#    push (@handledLines, "<Property NAME=\"host\" VALUE=\"".$hostName."\"/>\n");
#    push (@handledLines, "<Property NAME=\"UserName\" VALUE=\"".$ldapUser."\" ENCRYPTED=\"FALSE\"/>\n");
#    push (@handledLines, "<Property NAME=\"LDAPBindDN\" VALUE=\"".$ldapBindDN."\" ENCRYPTED=\"FALSE\"/>\n");
#    push (@handledLines, "<Property NAME=\"LDAPBindPwd\" VALUE=\"\"/>\n");
#    push (@handledLines, "<Property NAME=\"version\" VALUE=\"1.0.2.2\"/>\n");
#    push (@handledLines, "<Property NAME=\"ConnectDescriptor\" VALUE=\"".$ldapConnectDesc."\"/>\n");
#    push (@handledLines, "<CompositeMembership>\n");
#    push (@handledLines, "<MemberOf TYPE=\"oracle_ias\" NAME=\"".$iasName."\" ASSOCIATION=\" \"/>\n");
#    push (@handledLines, "</CompositeMembership>\n");
#    push (@handledLines, "</Target>\n");
#
#
#  # webcache
#  # ebs claims web cache is NOT used in ebs installs
#    push (@handledLines, "<Target TYPE=\"oracle_webcache\" NAME=\"".$iasName."_Web Cache\" DISPLAY_NAME=\"Web Cache\">\n");
#    push (@handledLines, "<Property NAME=\"HTTPProtocol\" VALUE=\"http\"/>\n");
#    push (@handledLines, "<Property NAME=\"HTTPMachine\" VALUE=\"localhost\"/>\n");
#    push (@handledLines, "<Property NAME=\"HTTPMachineForAdmin\" VALUE=\"".$hostName."\"/>\n");
#    push (@handledLines, "<Property NAME=\"HTTPPath\" VALUE=\"/\"/>\n");
#    push (@handledLines, "<Property NAME=\"HTTPQuery\" VALUE=\"\"/>\n");
#    push (@handledLines, "<Property NAME=\"OracleHome\" VALUE=\"".$emdRoot."\"/>\n");
#    push (@handledLines, "<Property NAME=\"MonitorPort\" VALUE=\"".$webcacheMonitorPort."\"/>\n");
#    push (@handledLines, "<Property NAME=\"AdminPort\" VALUE=\"".$webcacheAdminPort."\"/>\n");
#    push (@handledLines, "<Property NAME=\"logFileName\" VALUE=\"".$webcacheLogFileName."\"/>\n");
#    push (@handledLines, "<Property NAME=\"logFileDir\" VALUE=\"".$webcacheLogFileDir."\"/>\n");
#    push (@handledLines, "<Property NAME=\"version\" VALUE=\"1.0.2.2\"/>\n");
#    push (@handledLines, "<Property NAME=\"IASInternalName\" VALUE=\"WebCache\"/>\n");
#    push (@handledLines, "<AssocTargetInstance ASSOC_TARGET=\"ias\" TYPE=\"oracle_ias\" NAME=\"".$iasName."\"/>\n");
#    push (@handledLines, "<CompositeMembership>\n");
#    push (@handledLines, "<MemberOf TYPE=\"oracle_ias\" NAME=\"".$iasName."\" ASSOCIATION=\"null\"/>\n");
#    push (@handledLines, "</CompositeMembership>\n");
#    push (@handledLines, "</Target>\n");
#
#
#  # ifs
#    push (@handledLines, "<Target TYPE=\"oracle_ifs\" NAME=\"".$iasName."_IFS\">\n");
#    push (@handledLines, "<Property NAME=\"Version\" VALUE=\"1.0.2.2\"/>\n");
#    push (@handledLines, "<Property NAME=\"RunReposMetric\" VALUE=\"FALSE\" />\n");
#    push (@handledLines, "<Property NAME=\"RunResponseTimeMetric\" VALUE=\"FALSE\" />\n");
#    push (@handledLines, "<Property NAME=\"RunLoadBalancerResponseTimeMetric\" VALUE=\"FALSE\"/>\n");
#    push (@handledLines, "<Property NAME=\"IfsRootHome\" VALUE=\"".$iFSRootHome."\" />\n");
#    push (@handledLines, "<Property NAME=\"IfsHome\" VALUE=\"".$iFSHome."\" />\n");
#    push (@handledLines, "<Property NAME=\"DomainName\" VALUE=\"".$ifsDomainName."\" />\n");
#    push (@handledLines, "<Property NAME=\"DomainType\" VALUE=\"".$ifsDomainType."\" />\n");
#    push (@handledLines, "<Property NAME=\"SchemaPassword\" VALUE=\"".$ifsSchemaPassword."\" ENCRYPTED=\"FALSE\" />\n");
#    push (@handledLines, "<Property NAME=\"ApplicationUrl\" VALUE=\"".$ifsApplicationURL."\" />\n");
#    push (@handledLines, "<Property NAME=\"LoadBalancerUrl\" VALUE=\"".$ifsLoadBalancerURL."\" />\n");
#    push (@handledLines, "<CompositeMembership>\n");
#    push (@handledLines, "<MemberOf TYPE=\"oracle_ias\" NAME=\"".$iasName."\" ASSOCIATION=\"\"/>\n");
#    push (@handledLines, "</CompositeMembership>\n");
#    push (@handledLines, "</Target>\n");
#
#
#  # wireless
#    push (@handledLines, "<Target TYPE=\"oracle_wireless\" NAME=\"".$iasName."_Wireless\" DISPLAY_NAME=\"Wireless\" VERSION=\"1.0\" >\n");
#    push (@handledLines, "  <Property NAME=\"host\" VALUE=\"".$hostName."\"/>\n");
#    push (@handledLines, "  <Property NAME=\"startComponent\" VALUE=\"".$iasName."_OC4J_Wireless\"/>\n");
#    push (@handledLines, "  <Property NAME=\"OidRepSchemaName\" VALUE=\"wireless\"/>\n");
#    push (@handledLines, "  <Property NAME=\"IASInternalName\" VALUE=\"wireless\"/>\n");
#    push (@handledLines, "  <Property NAME=\"OracleHome\" VALUE=\"".$emdRoot."\"/>\n");
#    push (@handledLines, "  <AssocTargetInstance ASSOC_TARGET=\"ias\" TYPE=\"oracle_ias\" NAME=\"".$iasName."\"/>\n");
#    push (@handledLines, "  <CompositeMembership>\n");
#    push (@handledLines, "    <MemberOf TYPE=\"oracle_ias\" NAME=\"".$iasName."\" ASSOCIATION=\"null\"/>\n");
#    push (@handledLines, "  </CompositeMembership>\n");
#    push (@handledLines, "</Target>\n");
#
#
#  # forms
#    push (@handledLines, "<Target TYPE=\"oracle_forms\" NAME=\"".$iasName."_Forms\" DISPLAY_NAME=\"Forms\" VERSION=\"9.0\" >\n");
#    push (@handledLines, "  <Property NAME=\"ServletUrl\" VALUE=\"http://".$hostName.":80/forms90/l90servlet\"/>\n");
#    push (@handledLines, "  <Property NAME=\"oracle_home\" VALUE=\"".$emdRoot."\"/>\n");
#    push (@handledLines, "  <Property NAME=\"host\" VALUE=\"".$hostName."\"/>\n");
#    push (@handledLines, "  <Property NAME=\"configFilePath\" VALUE=\"".$emdRoot."/forms90/server/formsweb.cfg\"/>\n");
#    push (@handledLines, "  <CompositeMembership>\n");
#    push (@handledLines, "    <MemberOf TYPE=\"oracle_ias\" NAME=\"".$iasName."\" ASSOCIATION=\"null\"/>\n");
#    push (@handledLines, "  </CompositeMembership>\n");
#    push (@handledLines, "</Target>\n");
#
#
#  # reports
#    push (@handledLines, "<Target TYPE=\"oracle_repserv\" NAME=\"".$iasName."_Repsrv\" DISPLAY_NAME=\"Reports\" VERSION=\"9.0\" >\n");
#    push (@handledLines, "  <Property NAME=\"OracleHome\" VALUE=\"".$emdRoot."\"/>\n");
#    push (@handledLines, "  <Property NAME=\"Password\" VALUE=\"".$repPwd."\" ENCRYPTED=\"FALSE\" />\n");
#    push (@handledLines, "  <Property NAME=\"Server\" VALUE=\"".$hostName."\" />\n");
#    push (@handledLines, "  <Property NAME=\"Servlet\" VALUE=\"http://".$hostName.":".$httpPort."reports/rwservlet\" />\n");
#    push (@handledLines, "  <Property NAME=\"UserName\" VALUE=\"repadmin\" />\n");
#    push (@handledLines, "  <Property NAME=\"host\" VALUE=\"".$hostName."\"/>\n");
#    push (@handledLines, "  <Property NAME=\"HTTPProtocol\" VALUE=\"http\"/>\n");
#    push (@handledLines, "  <Property NAME=\"HTTPMachine\" VALUE=\"".$hostName."\"/>\n");
#    push (@handledLines, "  <Property NAME=\"HTTPPort\" VALUE=\"".$httpPort."\"/>\n");
#    push (@handledLines, "  <CompositeMembership>\n");
#    push (@handledLines, "    <MemberOf TYPE=\"oracle_ias\" NAME=\"".$iasName."\" ASSOCIATION=\"null\"/>\n");
#    push (@handledLines, "  </CompositeMembership>\n");
#    push (@handledLines, "</Target>\n");
#
#
#  # portal
#    push (@handledLines, "<Target TYPE=\"oracle_portal\" NAME=\"".$iasName."_Portal\" VERSION=\"1.0\">");
#    push (@handledLines, "  <Property NAME=\"version\" VALUE=\"1.0.2.2\" />");
#    push (@handledLines, "  <Property NAME=\"HTTPMachine\" VALUE=\"".$hostName."\" />");
#    push (@handledLines, "  <Property NAME=\"HTTPPort\" VALUE=\"".$portalPort."\" />");
#    push (@handledLines, "  <Property NAME=\"HTTPProtocol\" VALUE=\"http\" />");
#    push (@handledLines, "  <Property NAME=\"OidRepSchemaName\" VALUE=\"portal\" />");
#    push (@handledLines, "  <Property NAME=\"OracleHome\" VALUE=\"".$emdRoot."\" />");
#    push (@handledLines, "  <Property NAME=\"PPESuccessfulResponsesCriticalThreshold\" VALUE=\"80\" />");
#    push (@handledLines, "  <Property NAME=\"PPESuccessfulResponsesWarningThreshold\" VALUE=\"90\" />");
#    push (@handledLines, "  <Property NAME=\"PortalListeningHostPort\" VALUE=\"".$portalLHPort."\" />");
#    push (@handledLines, "  <Property NAME=\"portal_DAD\" VALUE=\"".$portalDAD."\" />");
#    push (@handledLines, "  <Property NAME=\"portletResponseCriticalThreshold\" VALUE=\"4500\" />");
#    push (@handledLines, "  <Property NAME=\"portletResponseWarningThreshold\" VALUE=\"4000\" />");
#    push (@handledLines, "  <CompositeMembership>");
#    push (@handledLines, "    <MemberOf TYPE=\"oracle_ias\" NAME=\"".$iasName."\"/>");
#    push (@handledLines, "  </CompositeMembership>");
#    push (@handledLines, "</Target>");
#  }
#
# others??

    return @handledLines;
}


sub getPropertyFromFile
{
    my ($emdHomeRef, $fileSpec, $property, $delim) = @_;

    my $propertyValue = "";
    my $propertiesFile = $emdHomeRef . $fileSpec;
    my ($propertiesLine, @propInfo);

    $delim = "=" if (!defined($delim));

    if (-e $propertiesFile)
    {
        open (PROPERTIES_FILE_READER, $propertiesFile);
        while($propertiesLine = <PROPERTIES_FILE_READER>)
        {
            chomp($propertiesLine);
            if (index($propertiesLine, "$property") eq 0)
            {
                @propInfo = split(/$delim/ ,$propertiesLine);
                if ($propInfo[0] eq $property)
                {
                    $propertyValue = $propInfo[1];
                    last;
                }
            }
        }
        close PROPERTIES_FILE_READER;
    }
    return $propertyValue;
}

##
# recursively parse given file for included files
# push found include directives into returned array
#
sub recurseParseInclude
{
    my ($confFile) = @_;

    my @foundInclude;
    if (-e $confFile)
    {
        open (CFILE, $confFile);
        my @cLines = <CFILE>;
        close CFILE;
        foreach $cLine (@cLines)
        {
            if ($cLine =~ /^\s*include/)
            {
                my @bits = split(/\"/,$cLine); # split on "
                push (@foundInclude, $bits[1]);
                push (@foundInclude, recurseParseInclude($bits[1]));
            }
        }
    }
    return @foundInclude;
}

#Strips leading and trailing spaces and returns the string
sub trim 
{
    my $origStr = $_[0];
    #Strip trailing and leading
    $origStr =~ s/^\s*|\s*$//g;
    return $origStr;
}

#
# returns the oracleHome corresponding to the DEFAULT entry in emtab
#
sub getActiveEmdHome
{
    my ($oraclehome) = @_;
    my (@entries) = getEmtabEntries($oraclehome);
    my $emdHome = "";
    foreach $emTabEntry (@entries)
    {
      my ($emdName, $oraclehome) = split(/=/ , $emTabEntry);
      if ($emdName eq "DEFAULT")
      {
          $emdHome = $oraclehome;
          last;
      }
    }
    return $emdHome;
}

#Get the emtab file entries
# A typical emtab entry is of the form
# DEFAULT=/private1/oracle/902orcl
#
# Returns list of the form
# { Emd1=Oracle_home1 Emd2=Oracle_home2 ... EmdN=Oracle_homeN }
sub getEmtabEntries 
{
    my ($oraclehome) = @_;
    my (@entries,$emtabFile,$Emtab,$oraLine); 

    if (( $^O =~ "Windows" ) || ($^O =~ "MSWin32"))
    {
        my $emLocValue = $Registry->{"HKEY_LOCAL_MACHINE/Software/Oracle/em_loc"};
        chomp($emLocValue);
        push (@entries, "DEFAULT=$emLocValue");
    }
    else
    {
        $emtabFile = getEmtabFile();
        if ( $emtabFile ne "" ) 
        {
            if ( open(Emtab, $emtabFile) ) 
            {
                while ($oraLine=<Emtab>) 
                {
                    chomp($oraLine);
                    # strip all leading  white space characters.
                    $oraLine =~ s/^\s*//;
                    next if (($oraLine =~ /^\#/ ) || ( length($oraLine) <= 0 ));
                    push (@entries,$oraLine);
                }
                close(Emtab);
            }
        }
    }
    return @entries;
}

# Get the emtab file location.
# The emtab file contains the oracle home of the active EMD
# Note: The location of emtab file is port specific
sub getEmtabFile 
{
    my @emtabFileList = ("/var/opt/oracle/emtab", "/etc/emtab");
    my $emtabFile = "";

    #$^O is supposed to yield the name of the OS, like Solaris , but 
    #currently it is not giving any value. This value probably should come 
    #from emctl.
    if ( $^O =~ "VMS" )
    {
        #TODO VMS specific code
    }
    else
    {
        # Check if EMTAB env is specified
        if ( defined $ENV{EMTAB} )
        {
            $emtabFile = trim ( $ENV{EMTAB} );
            if (! -e $emtabFile )
            {
                $emtabFile = "";
            }
            elsif ( -d $emtabFile ) 
            {
                $emtabFile = "";
            }
            else 
            {
                #log "Using EMTAB file $emtabFile";
                return $emtabFile;
            }
        }
    }
    #trace Processing prospective EMTAB locations of %s $emtabFileList
    foreach $emtabFile (@emtabFileList) 
    {
        if ( -d $emtabFile ) 
        {
            #warning "$emtabFile can not be a directory";
        }
        elsif ( -e $emtabFile )
        {
            #log "Using emtab file $emtabFile";
            return $emtabFile
        }
    }
    #log "Warning: Found no valid emtab file";
    $emtabFile = "" ;
    return $emtabFile;
}

#Get iAS Oracle Homes
#Returns oracle Home references
#OhomeRef---> Array of Homes like ("/private/oracle817", "/private/oracle92");
sub getIasOracleHomes
{
    my $onlyThisHome = shift;
    my @oratabEntries;
    my @ouiOracleHomes;
    my @iASOracleHomes;
    my @oracleHomes;

    if ($onlyThisHome eq "TRUE")
    {
        push(@iASOracleHomes,$oracleConfigHome);
    }
    else
    {
        # get OUI homes for iAS installs
        @ouiOracleHomes = getOuiIasHomes();
        push (@iASOracleHomes, @ouiOracleHomes);

        if ($IsWin32)
        {
            @oracleHomes = getOracleHomesNT();
            # check homes coming from NT registry
            REGHOME: foreach $oraclehome (@oracleHomes)
            {
                # skip homes which don't have ias.properties
                next if (!(-e $oraclehome . "/config/ias.properties"));

                # skip duplicates 
                foreach $oHome(@iASOracleHomes)
                {
                    next REGHOME if ($oHome eq $oraclehome)
                }
                push(@iASOracleHomes,$oraclehome);
                EMD_PERL_DEBUG("$LOG_CATEGORY Add OH=$oraclehome for discovery");
            }
        }
        else
        {
            @oratabEntries = getOratabEntries();
            ORATABHOME: foreach $entry (@oratabEntries) 
            {
                my ($sid, $oraclehome, $remain) = split(/:/ , $entry , 3);
                #remove leading and trailing white space.
                EMD_PERL_DEBUG("$LOG_CATEGORY Oratab entry=$entry: SID=$sid; OH=$oraclehome");

                $oraclehome =~ s/^\s*|\s*$//g;
                next if (!(-e $oraclehome . "/config/ias.properties"));

                foreach $oHome (@iASOracleHomes)
                {
                    next ORATABHOME if ($oHome eq $oraclehome)
                }
                push(@iASOracleHomes,$oraclehome);
                EMD_PERL_DEBUG("$LOG_CATEGORY Add OH=$oraclehome for discovery");
            }
        }
    }
    return (\@iASOracleHomes);
}

sub getOuiIasHomes
{
    my @oracleHomes;
    my $inventoryXmlFile = "";

    my $savePath = $ENV{PATH};

    if ( defined $ENV{INVENTORY_LOC} )
    {
        $inventoryXmlFile = checkInventoryXml($ENV{INVENTORY_LOC});
    }
    if($IsWin32)
    {
        $inventoryXmlFile = checkInventoryXml($Registry->{"LMachine/Software/Oracle/inst_loc"});
    }
    else
    {
        my ($f, $ln);
        my @lns;
        
        my @oraInstFileList = ("/var/opt/oracle/oraInst.loc", "/etc/oraInst.loc");
        
        foreach $f (@oraInstFileList) 
        {
            if ( open(ORAINST, "<$f") )
            {
                @lns = <ORAINST>;
                close(ORAINST);
                foreach $ln (@lns)
                {
                    $ln =~ s/\n$//;
                    if ($ln =~ /inventory_loc\s*=\s*([^\s]+)\s*$/i)
                    {
                        $inventoryXmlFile = checkInventoryXml($1);
                    }
                }
            }
        }
    }
    if ($inventoryXmlFile ne "")
    {
        if (open(INVXML, "<$inventoryXmlFile"))
        {
            my @lns = <INVXML>;
            close(INVXML);
            my $bigL;
      
            my $beginHomeList = 0;
            my $endHomeList = 0;
      
            my $ln;
            foreach $ln (@lns)
            {
                $ln =~ s/\n$//;
                if ($ln =~ /<HOME_LIST>(.*)/i)
                {
                    $ln = $1;
                    $beginHomeList = 1;
                }
                if ($ln =~ /(.*)<\/HOME_LIST>/i)
                {
                    $ln = $1;
                    $endHomeList = 1;
                }
                $bigL .= $ln if ($beginHomeList);
                last if ($endHomeList);
            }

            # rid of comments
            $bigL =~ s/<!--.*?-->//g;

            my @homes = split /<HOME\s+/, $bigL;
            my $h;
            foreach $h (@homes)
            {
                my $hLoc = substr($h, index($h,'LOC="')+length('LOC="'));
                $hLoc = substr($hLoc, 0, index($hLoc,'"'));
                # check if iAS install
                push(@oracleHomes,$hLoc) if (-e $hLoc . "/config/ias.properties");
            }
        }
    }
    return (@oracleHomes);
}

#"
# AS11 targets are found by parsing opmn.xml under the config home
#
sub getAS11Targets
{
    my ($configHome, $emdRoot) = @_;
    my @targetLines;
    my @componentArray;

    # setOracleHome($emdRoot . "/../");
    setOracleHome($configHome);

    my $opmnConfFile = "$configHome/config/OPMN/opmn/opmn.xml";

    # bail if no file to be found
    return if (! -e $opmnConfFile);

    # get ias instance info
    my @results = simpleXPathQuery(
                      $opmnConfFile,
                      "opmn/process-manager/ias-instance");

    my $iasInstanceName = $results[-1]{'name'};
    $iasInstanceName = $results[-1]{'id'} if ($iasInstanceName eq "");
    EMD_PERL_DEBUG("found ias instance $iasInstanceName");

    my $hasOHS = 0;
    my $httpPORT = "";
    my $iasHostName;

    # get components
    @results = simpleXPathQuery(
                      $opmnConfFile,
                      "opmn/process-manager/ias-instance/ias-component");
    for my $as11Comp (@results)
    {
        while( my ($k, $v) = each %$as11Comp )
        {
            if ($k =~ /^id$/)
            {
                @compInfo = simpleXPathQuery(
                          $opmnConfFile,
                          'opmn/process-manager/ias-instance/ias-component[@id="'.$v.'"]/process-type');
                for my $comp (@compInfo)
                {
                    while( my ($kp, $vp) = each %$comp )
                    {
                        if ($kp =~ /^id$/)
                        { 
                            if ($vp =~ /^OHS$/)
                            {
                                # locate ohs config files, parse, populate and print target snippet
                                $hasOHS = 1;
                                EMD_PERL_DEBUG("found OHS component $v");
                                # Set OHS instance ID in the ENV. It will be used by asecm::parseApacheConf()
                                $ENV{'CCR_OHS_ID'} = $v;
                                my $hash = parseApacheConf();
                                EMD_PERL_DEBUG("parsed apache conf for $v");
                                $httpPORT = peek($$hash{'Port'});
                                $httpPORT = (defined($httpPORT) ? $httpPORT : $$hash{'Listen'}[0]);
                                $iasHostName = peek($$hash{'ServerName'});
                                # print the target
                                push(@targetLines,add1013PlusOHSTarget($configHome, $iasInstanceName, $v, $iasHostName, $httpPORT, $iasVersion));
                            }
                            elsif ($vp =~ /^WebCache$/)
                            {
                                # todo: locate webcache config files, parse, populate and print target snippet
                                EMD_PERL_DEBUG("found WebCache component $v");
                                # The name is made up of the iasInstanceName and the iasComponentName
                                push (@targetLines, createTargetTypeXMLTag("oracle_webcache",$iasInstanceName."_".$v."_Web Cache","Web Cache"));
                                push (@targetLines, createTargetPropertyXMLTag("HTTPProtocol","http"));
                                push (@targetLines, createTargetPropertyXMLTag("HTTPMachine",$iasHostName));
                                push (@targetLines, createTargetPropertyXMLTag("HTTPMachineForAdmin",$iasHostName));
                                push (@targetLines, createTargetPropertyXMLTag("HTTPPath","/"));
                                push (@targetLines, createTargetPropertyXMLTag("HTTPQuery",""));
                                push (@targetLines, createTargetPropertyXMLTag("OracleHome",$configHome));
                                push (@targetLines, createTargetPropertyXMLTag("MonitorPort",""));
                                push (@targetLines, createTargetPropertyXMLTag("AdminPort",""));
                                push (@targetLines, createTargetPropertyXMLTag("logFileName",""));
                                push (@targetLines, createTargetPropertyXMLTag("logFileDir",""));
                                push (@targetLines, createTargetPropertyXMLTag("version",$iasVersion));
                                # add version_value property for normalized target version metric
                                push (@targetLines, createTargetPropertyXMLTag("version_value",$iasVersion));
                                push (@targetLines, createTargetPropertyXMLTag("IASInternalName","WebCache"));
                                push (@targetLines, createTargetPropertyXMLTag("IASComponentID",$v));
                                push (@targetLines, createAssocTargetXMLTag("ias","oracle_ias",$iasInstanceName));
                                push (@targetLines, createCompositeMembershipXMLTag("oracle_ias",$iasInstanceName,"null"));
                                push (@targetLines, getTargetEndTag());
                            }
                        }
                    }
                }
            }
        }
    }
    # print out the ias target
    push(@targetLines,add1013PlusIASTarget($configHome, $iasInstanceName, $iasHostName, $httpPORT, $iasHostName));
    return @targetLines;
}
