################################################################################
#
#  $Header: ChronosHash.pm 10-jun-2005.17:51:52 adosani Exp $
#
#
# Copyright (c) 2001, 2005, Oracle. All rights reserved.  
#
#    NAME
#      ChronosHash.pm - Chronos Hash
#
#    DESCRIPTION
#
#    NOTES
#
#    MODIFIED   (MM/DD/YY)
#     adosani    06/14/05 - add get_keys and is_key_valid functions
#     snakhoda   12/17/04 - abstracted generic hash functionality
#                           from ChronosPageHash into separate class
#
################################################################################

package ChronosHash;
use strict;
use Exporter;
use ChronosDbmTypes;
use ChronosLogging;
use File::Copy;
use vars qw(@ISA @EXPORT_OK);


@ISA = qw(Exporter);

@EXPORT_OK = qw(hashExists deleteHash moveHash getNumHashFiles);


my $chunk_marker_char = ' ';
my $placeholder_char = '^';

my $MAX_CHUNKS = 50; #protect against bad data set (should probably be a config parameter)
my $MAX_PLACEHOLDER_TRIES = 10;


sub new
{
  my ($class, $debug, $file_name, $persistent_file_name, $dbmInfo, $errorStrRef) = @_;
  my %ext_hash;
  my $max_entry_size = $dbmInfo->{'maxdatasize'};

  my $self =
  {
     ext_hash_ref => \%ext_hash,
     file_name => $file_name,
     persistent_file_name => $persistent_file_name,
     debug => $debug,
     max_entry_size => $max_entry_size,
     dbmInfo => $dbmInfo
  };
  if($persistent_file_name && (hashExists($persistent_file_name, $dbmInfo)  == getNumHashFiles($dbmInfo)))
  {
      # copy the page hash from the persistent dir to the tmp dir
      unless(copyHash($debug, $persistent_file_name, $file_name, $dbmInfo))
      {
        $$errorStrRef = "Failed copying page hash file(s) from persistent to tmp dir. $!\n";
	return undef;
    }
  }
  unless( dbmopen(%ext_hash, $file_name, 0666) )
  {
     my $errorStr = "Error in dbmopen opening page hash. $file_name, $!\n";
     $debug->PrintLog($DEBUG_FATAL, $errorStr);
     $$errorStrRef = $errorStr;
     return undef;
  }
  bless $self, $class;
  return $self;
}

sub fresh_hash
{
    my ($self) = @_;
    $self->close_hash();
    deleteHash($self->{file_name}, $self->{dbmInfo});
    unless(dbmopen(%{$self->{ext_hash_ref}},  $self->{file_name}, 0666) )
    {
	$self->{debug}->PrintLog($DEBUG_FATAL, "Error re-opening hash. $!\n");
	return 1;
    }
    return 0;
}

sub delete_entry_by_key
{
    my ($self, $key) = @_;
    my $hashRef = $self->{ext_hash_ref};
    my $allDeleted = 0;
    while(!$allDeleted)
    {
	my $thisKey = $key;
	my $value = get_single_chunk($hashRef, $thisKey);
	if(!$value || !(substr($value, 0, 1) eq $chunk_marker_char))
	{
	    $allDeleted = 1;
	    
	} else
	{
	    $key = $key.$chunk_marker_char;
	}
	# if the entry does not exist, no error is thrown by delete
	delete_single_chunk($hashRef, $thisKey);
    }
}


sub get_value
{
    my($self, $key) = @_;
    return get_value_($self->{ext_hash_ref}, $key);
}

sub get_value_
{
    my ($hashRef, $key) = @_;
    my $value = "";
    my $valueComplete = 0;
    while(!$valueComplete)
    {
	my $tmpValue = get_single_chunk($hashRef, $key);   
#	print "key: $key, tmpValue: $tmpValue, size: " . (length($key) + length($tmpValue)) . "\n";
	
	if( substr($tmpValue, 0,1) eq $chunk_marker_char)
	{
	    $tmpValue = substr($tmpValue, 1);
	    $key = $key.$chunk_marker_char;      
	} else {
	    $valueComplete = 1;
	}      
	$value = $value.$tmpValue;
    }
    return $value;

}

sub get_single_chunk
{
    my($hashRef, $chunk_key) = @_;
    my $numTries = 0;
    my $tmpValue = '';

    for($numTries = 0; $numTries < $MAX_PLACEHOLDER_TRIES; $numTries ++)
    {
	$tmpValue = $hashRef->{$chunk_key}; 
	last unless ($tmpValue eq $placeholder_char);
	$chunk_key .= $placeholder_char;
    }
    return $tmpValue;
}

sub insert_value
{
    my ($self, $key, $data, $max_entry_size, $debug) = @_;
    #insert is a replacement operation, so first delete any existing entry
    my $hashRef = $self->{ext_hash_ref};
    delete_entry_by_key($hashRef, $key);
    my $retval = 0;

    # Check if this entry is in the max size limitations of the OS
    my $keyLength = length($key);
    my $keyDataLength = $keyLength + length($data);
    if($max_entry_size && ($keyDataLength >= $max_entry_size))
    {
	my $allDataWritten = 0;
	my $thisKey = $key;
	my $thisKeyLength = $keyLength;
	my $thisDataStart = 0;
	my $dataLenLeft = length($data);
	my $numChunksInserted = 0;

	$debug->PrintTrace("Key+Data length ($keyDataLength) >= maximum allowed ($max_entry_size). Splitting Entry. Key \"$key\", Data: \"$data\"");

	while($dataLenLeft > 0)
	{	  
	    if($numChunksInserted >= $MAX_CHUNKS)
	    {
		$debug->PrintLog($DEBUG_HIGH, "Reached maximum chunks limit ($MAX_CHUNKS), Key: $key. Abandoning....");
		delete_entry_by_key($self->{ext_hash_ref}, $key);
		return 1;
	    }
	    #maxDataLen+keylength is one less than max_entry_size because
            #some implementations break with exactly max_entry_size
	    my $maxDataLen = $max_entry_size - $thisKeyLength -1;
	    # calculate the size for the chunk of data to insert
	    # if there is overflow, then we need to save one byte for 
	    # the $chunk_marker_char that indicates that there is more data 
	    # for this entry under a different key.
	    my $thisDataLen =
		($dataLenLeft <= $maxDataLen)?  $dataLenLeft : ($maxDataLen -1);
	    
	    my $thisDataChunk  = substr($data, $thisDataStart, $thisDataLen);
	    if($dataLenLeft > $maxDataLen)
	    {
		# add the
		$thisDataChunk = $chunk_marker_char.$thisDataChunk;
	    }
	    my $retval = insert_single_chunk($hashRef, $thisKey, $thisDataChunk, $max_entry_size, $debug);
	    if ($retval < 0) {
		$debug->PrintLog($DEBUG_HIGH, "Error doing partial hash key assignment: $@. Key: $thisKey, data: $data, thisDataChunk: $thisDataChunk. Abandoning....");
		delete_entry_by_key($self->{ext_hash_ref}, $key);
		return 1;
	    } elsif ($retval > 0)
	    {
		# $retval bytes were left over. put them back in the 
		# pool by adjusting $thisDataLen
		$thisDataLen -= $retval;
	    }            
	    $thisKey = $thisKey.$chunk_marker_char;
	    $thisKeyLength ++;
	    $dataLenLeft -= $thisDataLen;
	    $thisDataStart += $thisDataLen;
	    $numChunksInserted ++;
	}
    } else {
	$retval = insert_single_chunk($hashRef, $key, $data, $max_entry_size, $debug);
    }
    return $retval;
}

sub each_key_value
{
    my ($self) = @_;
    my $hashRef = $self->{ext_hash_ref};
    my $done = 0;
    my $value = "";
    my $key = "";
    while(!$done)
    {
	($key, ) = each(%{$hashRef});
	return () unless ($key);
	my $lastChar = substr($key, -1, 1);
	unless (($lastChar eq $placeholder_char) || ($lastChar eq $chunk_marker_char))
	{
	    $value = get_value_($hashRef, $key);
	    $done = 1;
	}	    
    }
    
    return ($key, $value);
}

sub get_keys
{
    my ($self) = @_;
    return keys(%{$self->{ext_hash_ref}});
}

sub is_key_valid
{
    my ($key) = @_;
    my $lastChar = substr($key, -1, 1);
    return (($lastChar ne $placeholder_char) && ($lastChar ne $chunk_marker_char));
}

sub delete_single_chunk
{
    my ($hashRef, $chunk_key) = @_;
    while($hashRef->{$chunk_key})
    {
	delete($hashRef->{$chunk_key});
	$chunk_key .= $placeholder_char;
    }
}

sub insert_single_chunk
{
    my ($hashRef, $chunk_key, $orig_chunk_data, $max_entry_size, $debug) = @_;
    my $overFlow = 0;
    my $chunk_data = $orig_chunk_data;

    my $numTries = 0;
    for($numTries = 0; $numTries < $MAX_PLACEHOLDER_TRIES; $numTries ++)
    {
	eval { $hashRef->{$chunk_key} = $chunk_data; };
	if ($@) {
	    $debug->PrintLog($DEBUG_HIGH, "Error doing hash key assignment: $@. Data: $chunk_data, numTries: $numTries");
	    eval {$hashRef->{$chunk_key} = $placeholder_char;};
	    if($@) {
		$debug->PrintLog($DEBUG_HIGH, "Error inserting placeholder: $@. Key: $chunk_key, Data: $chunk_data. ");
		delete_single_chunk($hashRef, $chunk_key);
		return -1;
	    }
	    $chunk_key .= $placeholder_char;
	    my $overFlowSize = length($chunk_key) + length($chunk_data) - 
 		($max_entry_size - 1);
	    #check if key+data would exceed or equal max_entry_size
	    if($overFlowSize > 0)
	    {
		$debug->PrintLog($DEBUG_HIGH, "Overflow of $overFlowSize would have occurred. Chopping off $overFlowSize bytes from chunk_data");
		$chunk_data = substr($chunk_data, 0, - $overFlowSize);
		$overFlow += $overFlowSize;
	    }	    
	} else {
	    return $overFlow;
	}
    }
    if($numTries == $MAX_PLACEHOLDER_TRIES) {
	$debug->PrintLog($DEBUG_HIGH, "Reached max tries ($MAX_PLACEHOLDER_TRIES) for doing doing hash key assignment: Key: $chunk_key, Data: $orig_chunk_data. Abandoning this assignment.");
	return -1;
    }
}


#####################################################################
# DESCRIPTION : closes hash 
#####################################################################

sub close_hash
{
  my ($self) = @_;
  unless(dbmclose(%{$self->{ext_hash_ref}}))
  {
    $self->{debug}->PrintLog($DEBUG_FATAL, "Error closing hash. $!\n");
    return 1;
  }
  return 0;
}


###########################################################################
# Usage - copy a hash from one file to another. Pass in old and new file
#         names (without the .dir and .pag extensions)
#
# IN:
#     $oldHash - file name of old hash
#     $newHash - file name of new hash
# OUT: 
# RETURN: 1 on success, 0 on error
###########################################################################
sub copyHash
{
    my ($debug, $oldHash, $newHash, $dbmInfo) = @_;
    my %oldHashTable;
    my %newHashTable;

    $debug->PrintLog($DEBUG_HIGH, "Entered copyHash: Copying $oldHash to $newHash");
    $debug->PrintLog($DEBUG_HIGH, "First delete $newHash");
    deleteHash($newHash, $dbmInfo);

    unless(dbmopen(%oldHashTable, $oldHash, 0444))
    {
	$debug->PrintLog($DEBUG_LOW, "copyHash: Could not open hash $oldHash for reading");
	return 0;
    }

    unless(dbmopen(%newHashTable, $newHash, 0666))
    {
	$debug->PrintLog($DEBUG_LOW, "copyHash: Could not open hash $newHash for writing");
	return 0;
    }

    if ($dbmInfo->{implname} eq "sdbm")      # bug 3236776
    {
	foreach my $key (keys %oldHashTable)
	{
	    eval {
		$newHashTable{$key} = $oldHashTable{$key};
	    };
	    if($@) {
		$debug->PrintLog($DEBUG_HIGH, "Error assigning key in copyHash: $key: $@");
	    }
	}
    } 
    else 
    {
	%newHashTable = %oldHashTable;
    }
    
    unless((dbmclose(%newHashTable)) && dbmclose(%oldHashTable)) {
	return 0;
    }
    $debug->PrintLog($DEBUG_HIGH, "Done with copyHash: Copying $oldHash to $newHash");
    return 1;
}

###########################################################################
# Usage - move a hash from one location(name) to another. Pass in old and new file
#         names (without the .dir and .pag extensions)
#
# IN:
#     $oldHashName - file name of old hash
#     $newHashName - file name of new hash
#     $dbmInfo - the refrence to the dbmInfo about this dbm type
# OUT: 
# RETURN: 1 on success, 0 on error
###########################################################################
sub moveHash
{
    my ($debug, $oldHashName, $newHashName, $dbmInfo) = @_;
    my $success = 1;
    $debug->PrintLog($DEBUG_HIGH, "Entered moveHash: Moving $oldHashName to $newHashName");

    foreach(@{$dbmInfo->{exts}})
    {
	$success = 0 unless(move($oldHashName."$_", $newHashName."$_"));
    }
    $debug->PrintLog($DEBUG_HIGH, "Done with moveHash: returning $success");
    return $success;
}


###########################################################################
# Usage - get the number of files that the given hash implementation uses
#
# IN:
#     $hashName - file name of  hash
#     $dbmInfo - the refrence to the dbmInfo about this dbm type
# OUT: 
# RETURN: number of files that exist for this hash implementation
###########################################################################
sub getNumHashFiles
{
    my ($dbmInfo) = @_;

    my $numHashFiles = scalar @{$dbmInfo->{exts}};
    return $numHashFiles;

}


###########################################################################
# Usage - check whether a given hash exists on the file system
#
# IN:
#     $hashName - file name of  hash
#     $dbmInfo - the refrence to the dbmInfo about this dbm type
# OUT: 
# RETURN: number of files that exist for this hash implementation
###########################################################################
sub hashExists
{
    my ($hashName,  $dbmInfo) = @_;
    my $numExist = 0;

    foreach(@{$dbmInfo->{exts}})
    {
	$numExist++ if( -e $hashName."$_");
    }
    return $numExist;
}

###########################################################################
# Usage - delete a hash file. Pass in hash name (without the .dir 
#         and .pag extensions)
#
# IN:
#     $hashName - file name of hash
#     $dbmInfo - the refrence to the dbmInfo about this dbm type
# OUT: 
# RETURN: number of files deleted for the hash
###########################################################################
sub deleteHash
{
    my ($hashName,  $dbmInfo) = @_;
    my $numDeleted = 0;

    foreach(@{$dbmInfo->{exts}})
    {
	$numDeleted ++  if(unlink($hashName."$_"));
    }
    return $numDeleted;
    
}
