##############################################################################
#  $Header:
#
#
# Copyright (c) 2001, 2005, Oracle. All rights reserved.  
#
#    NAME
#      ChronosLogging.pm - Chronos Logging
#
#    DESCRIPTION
#     This class provies the chronos mining engine (CME for short) a
#     centralized logging and tracing facility.
#
#    NOTES
#
#    MODIFIED   (MM/DD/YY)
#       adosani  01/06/05 - rotate log file
#       snakhoda 09/15/04 - add (and export) debug level variables 
#       snakhoda 10/14/03 - bugs 3186461, 3178092 - allow logging to stderr 
#                           for non-fatal errors 
#       asawant  11/19/01 - Fixing indentation..
#       mashukla 09/17/01 - changes to fix logging
#       mashukla 09/14/01 - changes to merge trace and debug options
#       asawant  09/10/01 - Enhancing error reporting.
#       jmarfati 09/05/01 - fix tracing
#       jmarfati 09/05/01 - do not print to stdout if not initialized
#       mashukla 08/20/01 - add perl module to support CME logging
#
################################################################################

package ChronosLogging;
use strict;
use IO::File;
use File::Basename;
use File::Copy;
use Exporter;
use vars qw(@ISA @EXPORT $DEBUG_FATAL $DEBUG_LOW $DEBUG_MEDIUM $DEBUG_HIGH);

@ISA = qw(Exporter);

@EXPORT = qw($DEBUG_FATAL $DEBUG_LOW $DEBUG_MEDIUM $DEBUG_HIGH);


$DEBUG_FATAL = 0;
$DEBUG_LOW = 1;
$DEBUG_MEDIUM = 2;
$DEBUG_HIGH = 3;


# constructor for logging object
sub new{
  my ($class) = @_;
  my $self =
  {
    log_file_name => '',
    debug_level   => '',
    trace_file_name => '',
    log_file_object => '',
    trace_file_object => '',
    trace_info => 0,
    target_runid => '',
    initialized => 0 
  };

  bless $self,$class;
  return $self;
}
#
# initialize logging object
# open log and trace files files
#
sub Initialize
{
  my ($self,$err_ref,$traceFileName,$logFileName,$debugLevel,$fatalLevel,
      $traceInfo,$targetID,$startTimeStamp,$maxLogFileSizeKB,$numLogFiles,$numTraceFiles) = @_;
  my $fileSize = 0;
  my $logDir = '';
  my $logFile = '';
  my $traceDir = '';
  my $traceFile = '';
  my @traceFileTimeStamps = ();
  my $fileName = '';
  my $numFiles = 1;
  my $fileNum;
  my $fileTimeStamp = '';
  my $targetRunID = $targetID.":".$startTimeStamp;
  $startTimeStamp =~ s/[\/:]//g; # remove / and : from timestamp
  $startTimeStamp =~ s/\s/_/;    # replace space with underscore

  $self->{log_file_name} = $logFileName;
  $self->{debug_level}  = $debugLevel;
  $self->{trace_file_name} = $traceFileName.".".$startTimeStamp;
  $self->{log_file_object} = new IO::File;
  $self->{trace_file_object} = new IO::File;
  $self->{target_runid} = $targetRunID;
  $self->{trace_info} = $traceInfo;
  $self->{fatal_level} = $fatalLevel;

  if (-e $self->{log_file_name})
  {
    # Check size of log file
    my @fileStat = stat($self->{log_file_name});
    $fileSize = $fileStat[7];

    if ($fileSize > ($maxLogFileSizeKB * 1024))
    {
      # Rotate log file
      ($logFile, $logDir) = fileparse($self->{log_file_name});
      opendir(LOGFILEDIR, $logDir);

      # count number of files in disk which have same format as rotated log files,
      # that we are going to keep
      while ($fileName=readdir(LOGFILEDIR))
      {
	if (($fileNum) = ($fileName =~ m/^$logFile\.(\d+)/o))
	{	
	  if ($fileNum >= $numLogFiles)
	  {
	    unlink $logDir.$fileName;
	  }
	  else
	  {
	    $numFiles++;
	  }
	}
      }
      closedir(LOGFILEDIR);

      if ($numFiles >= $numLogFiles)
      {
        $numFiles = $numLogFiles - 1;
      }
      while (--$numFiles) 
      {	
	unless (move($self->{log_file_name}.".".$numFiles, $self->{log_file_name}.".".($numFiles+1)))
	{
	  $$err_ref = "$!\nFailed to rotate log file $self->{log_file_name}\n";
	  print STDERR $$err_ref;
	}	
      }
      unless (move($self->{log_file_name}, $self->{log_file_name}.".".($numFiles+1)))
      {
	$$err_ref = "$!\nFailed to rotate log file $self->{log_file_name}\n";
	print STDERR $$err_ref;
      }
    }
  }

  # Open file in append mode
  unless($self->{log_file_object}->open(">>".$self->{log_file_name}))
  {
    $$err_ref = "$!\nFailed opening log file $self->{log_file_name}\n";
    print STDERR $$err_ref;
    return undef;
  }

  # if trace on open trace file
  if ($self->{"trace_info"})
  {
    ($traceFile, $traceDir) = fileparse($traceFileName);
    opendir(TRACEFILEDIR, $traceDir); 
    # count number of files in disk which have same format as trace file
    $numFiles = 1;
    while ($fileName=readdir(TRACEFILEDIR))
    {
      if (($fileTimeStamp) = $fileName=~ m/^$traceFile\.(\d{8}_\d{6})/o)
      {
	  $traceFileTimeStamps[$numFiles-1] = $fileTimeStamp;
	  $numFiles++;
      }
    }
    closedir(LOGFILEDIR);

    if ($numFiles >= $numTraceFiles)
    {
	my @sortedFileTimeStamps = sort {my $a2 = $a; $a2 =~ s/_//g; my $b2 = $b; $b2 =~ s/_//g; $b2 <=> $a2} @traceFileTimeStamps;
	for (my $i = $numTraceFiles-1; $i < $numFiles-1; $i++)
	{
	  unlink ($traceFileName.".".$sortedFileTimeStamps[$i]);
	}
    }

    unless($self->{trace_file_object}->open(">".$self->{trace_file_name}))
    {
      $$err_ref = "$!Failed opening trace file.\n";
      print STDERR $$err_ref;
      close $self->{log_file_object};
      return undef;
    }
  }

  $self->{initialized} = 1;
  return 1;
}

#
# Print to trace file
#
sub PrintTrace 
{
  my ($self,$errorString)=@_;
  # check if trace was turned in command line parameters
  if ($self->{initialized})
  {
    if ($self->{"trace_info"})
    {
      if (!print {$self->{trace_file_object}} "$errorString\n")
      {
        close({$self->{trace_file_object}});
        # something very drastic went wrong so exit.
        # Only place where this is done
        print STDERR "FATAL error on writing to trace file $!";
        exit 1;
      }
    }
  }
  return 1;
}
#
# Print to log file
# logLevel determines if log message has high enough debug level to be
# printed
#
sub PrintLog
{
  my ($self,$logLevel,$errorString,$force_stderr)=@_;

  my $stderrPrefix = "";
  if($logLevel == $self->{fatal_level}) {
      $force_stderr  = 1;
      $stderrPrefix = "Fatal error: ";
  }
      
  if($force_stderr) {
     print STDERR "$stderrPrefix$errorString\n";
  }

  if ($self->{initialized})
  {
    # check if trace was turned in command line parameters
    if ($self->{"trace_info"})
    {
      if (!print {$self->{trace_file_object}} "$errorString\n")
      {
        close({$self->{trace_file_object}});
        # something very drastic went wrong so exit.
        # Only place where this is done
        print STDERR "FATAL error on writing to trace file $!";
        exit 1;
      }
    }
    # print to debug file
    if ($logLevel <= $self->{debug_level})
    {
      if (!print {$self->{log_file_object}} $self->{target_runid}.":".
        ":"."$errorString\n")
      {
        close({$self->{log_file_object}});
        # something very drastic went wrong so exit.
        # Only place where this is done
        print "FATAL error on writing to log file $!";
        exit 1;
      }
    }
  }
  return 1;
}

# close both log and trace files and set initialized to false
sub Close 
{
  my ($self) = @_;
  if ($self->{initialized})
  {
    close $self->{log_file_object};
    if ($self->{"trace_info"})
    {
      close $self->{trace_file_object};
    }
  }
  $self->{initialized}=0;
  return 1;
}

# Make use return 1 
1;
