#
#  $Header: ChronosTableOut.pm 29-aug-2005.15:36:20 adosani Exp $
#
#
# Copyright (c) 2001, 2005, Oracle. All rights reserved.  
#
#    NAME
#      ChronosTableOut.pm - Chronos Table Out
#
#    DESCRIPTION
#      This class provies the chronos mining engine (CME for short) 
#    "an interface" with the MultiTableOut class. It's main 
#    functionality is to encapsulate all possible output formats that the 
#    CME supports and allow the CME to output its data without worrying about
#    the details of the file format itself.
#      Currently the supported formats are "XML" and "SQLLDR". In the XML format
#    tables are written to one file only, while in the SQLLDR format, tables
#    are written to seperate files.
#
#    NOTES
#
#    MODIFIED   (MM/DD/YY)
#     adosani    08/29/05 - add xml output formats for 10.1.0.3 and 10.1.0.4 
#                           oms 
#     adosani    05/18/05 - add CDATA to urls in pr_mapping and 
#                           incomplete_loads tables 
#     snakhoda   01/26/05 - fix bug 4143912: add DATABASE_TIME column 
#     snakhoda   12/15/04 - Add pr_mapping and incomplete_loads_tables 
#     eporter    12/02/04 - bug 3849061: add visitor_ip_num column 
#     mashukla   02/04/03 - remove origin server trips
#     mashukla   11/19/02 - add origin server trips
#     rpinnama   10/21/02 - Use upper case attribute names
#     asawant    01/04/02 - Adding target_guid (removing target_name and target_type)..
#     asawant    11/02/01 - Fix for bug 2093979 (adding CDATA to url field).
#     njagathe   11/07/01 - Change in delivery of files to EMD (bug # 1980893)..
#     asawant    09/10/01 - Removing 'ENDDATA' for sqlldr output.
#     asawant    08/31/01 - Adding timezone changes.
#     asawant    08/27/01 - Changing from view names to table names.
#     asawant    06/27/01 - Changing file names to be passed in as full names.
#     asawant    06/18/01 - Creation
#
#

package ChronosTableOut;
use MultiTableOut;
use strict;

# Output description for different OMS versions
#
# OMS 10.2 and above: completed loads table (mgmt_rt_metrics_raw),
#   incomplete loads (mgmt_rt_incomplete_loads), and page-request
#   mappings (mgmt_rt_pr_mapping)
# OMS 10.1.0.4: only completed loads table, no database_time column
# OMS 10.1.0.3: only completed loads table, no database_time and 
#   visitor_ip_num columns.


######################## Configurations for XML format #########################

# Configuration hash for XML data file 

my $xml_data_col_desc_ref_10200 = 
[
  ['      <TARGET_GUID>',             '</TARGET_GUID>'],
  ['      <METRIC_NAME>',             '</METRIC_NAME>'],
  ['      <COLLECTION_TIMESTAMP>',    '</COLLECTION_TIMESTAMP>'],
  ['      <SUBMIT_ACTION_TIMESTAMP>', '</SUBMIT_ACTION_TIMESTAMP>'],
  ['      <LOAD_ACTION_TIMESTAMP>',   '</LOAD_ACTION_TIMESTAMP>'],
  ['      <ELAPSED_TIME>',            '</ELAPSED_TIME>'],
  ['      <URL_FILENAME><![CDATA[',   ']]></URL_FILENAME>'],
  ['      <URL_BASE>',                '</URL_BASE>'],
  ['      <VISITOR_NODE>',            '</VISITOR_NODE>'],
  ['      <VISITOR_DOMAIN>',          '</VISITOR_DOMAIN>'],
  ['      <VISITOR_IP>',              '</VISITOR_IP>'],
  ['      <VISITOR_IP_NUM>',          '</VISITOR_IP_NUM>'],
  ['      <COOKIE_INDEX>',            '</COOKIE_INDEX>'],
  ['      <SERVER_IN_TIMESTAMP>',     '</SERVER_IN_TIMESTAMP>'],
  ['      <SERVER_OUT_TIMESTAMP>',    '</SERVER_OUT_TIMESTAMP>'],
  ['      <SERVER_LATENCY_TIME>',     '</SERVER_LATENCY_TIME>'],
  ['      <DATABASE_TIME>',           '</DATABASE_TIME>'],
  ['      <BROWSER_NAME>',            '</BROWSER_NAME>'],
  ['      <BROWSER_VERSION>',         '</BROWSER_VERSION>'],
  ['      <OS_NAME>',                 '</OS_NAME>'],
  ['      <OS_VERSION>',              '</OS_VERSION>']
];
my $xml_data_col_desc_ref_10104 = 
[
  ['      <TARGET_GUID>',             '</TARGET_GUID>'],
  ['      <METRIC_NAME>',             '</METRIC_NAME>'],
  ['      <COLLECTION_TIMESTAMP>',    '</COLLECTION_TIMESTAMP>'],
  ['      <SUBMIT_ACTION_TIMESTAMP>', '</SUBMIT_ACTION_TIMESTAMP>'],
  ['      <LOAD_ACTION_TIMESTAMP>',   '</LOAD_ACTION_TIMESTAMP>'],
  ['      <ELAPSED_TIME>',            '</ELAPSED_TIME>'],
  ['      <URL_FILENAME><![CDATA[',   ']]></URL_FILENAME>'],
  ['      <URL_BASE>',                '</URL_BASE>'],
  ['      <VISITOR_NODE>',            '</VISITOR_NODE>'],
  ['      <VISITOR_DOMAIN>',          '</VISITOR_DOMAIN>'],
  ['      <VISITOR_IP>',              '</VISITOR_IP>'],
  ['      <VISITOR_IP_NUM>',          '</VISITOR_IP_NUM>'],
  ['      <COOKIE_INDEX>',            '</COOKIE_INDEX>'],
  ['      <SERVER_IN_TIMESTAMP>',     '</SERVER_IN_TIMESTAMP>'],
  ['      <SERVER_OUT_TIMESTAMP>',    '</SERVER_OUT_TIMESTAMP>'],
  ['      <SERVER_LATENCY_TIME>',     '</SERVER_LATENCY_TIME>'],
  ['      <BROWSER_NAME>',            '</BROWSER_NAME>'],
  ['      <BROWSER_VERSION>',         '</BROWSER_VERSION>'],
  ['      <OS_NAME>',                 '</OS_NAME>'],
  ['      <OS_VERSION>',              '</OS_VERSION>']
];
my $xml_data_col_desc_ref_10103 = 
[
  ['      <TARGET_GUID>',             '</TARGET_GUID>'],
  ['      <METRIC_NAME>',             '</METRIC_NAME>'],
  ['      <COLLECTION_TIMESTAMP>',    '</COLLECTION_TIMESTAMP>'],
  ['      <SUBMIT_ACTION_TIMESTAMP>', '</SUBMIT_ACTION_TIMESTAMP>'],
  ['      <LOAD_ACTION_TIMESTAMP>',   '</LOAD_ACTION_TIMESTAMP>'],
  ['      <ELAPSED_TIME>',            '</ELAPSED_TIME>'],
  ['      <URL_FILENAME><![CDATA[',   ']]></URL_FILENAME>'],
  ['      <URL_BASE>',                '</URL_BASE>'],
  ['      <VISITOR_NODE>',            '</VISITOR_NODE>'],
  ['      <VISITOR_DOMAIN>',          '</VISITOR_DOMAIN>'],
  ['      <VISITOR_IP>',              '</VISITOR_IP>'],
  ['      <COOKIE_INDEX>',            '</COOKIE_INDEX>'],
  ['      <SERVER_IN_TIMESTAMP>',     '</SERVER_IN_TIMESTAMP>'],
  ['      <SERVER_OUT_TIMESTAMP>',    '</SERVER_OUT_TIMESTAMP>'],
  ['      <SERVER_LATENCY_TIME>',     '</SERVER_LATENCY_TIME>'],
  ['      <BROWSER_NAME>',            '</BROWSER_NAME>'],
  ['      <BROWSER_VERSION>',         '</BROWSER_VERSION>'],
  ['      <OS_NAME>',                 '</OS_NAME>'],
  ['      <OS_VERSION>',              '</OS_VERSION>']
];

# default oms 10.2
my $xml_data_col_desc_ref = $xml_data_col_desc_ref_10200;

my $xml_data_config_hash = 
{
  table_header => "  <ROWSET TABLE=\"MGMT_RT_METRICS_RAW\" ".
    "PRIMARYKEY=\"TARGET_GUID,COLLECTION_TIMESTAMP,".
    "URL_FILENAME,METRIC_NAME,SUBMIT_ACTION_TIMESTAMP,VISITOR_IP\" ".
    "DATECOLUMNS=\"COLLECTION_TIMESTAMP\">\n",
  table_footer => "  </ROWSET>\n",
  line_prefix => "    <ROW>\n",
  line_sufix => "\n    </ROW>\n",
  line_separator => "",
  columns => $xml_data_col_desc_ref,
  column_separator => "\n",
};

# Configuration hash for XML cookies file 

my $xml_cookies_col_desc_ref = 
[
  ['      <RAW_INDEX>', '</RAW_INDEX>'],
  ['      <NAME>',      '</NAME>'],
  ['      <VALUE>',     '</VALUE>']
];

my $xml_cookies_config_hash =
{
  table_header => "  <ROWSET TABLE=\"MGMT_RT_COOKIE_DATA\" ".
    "PRIMARYKEY=\"RAW_INDEX,NAME,VALUE\">\n",
  table_footer => "  </ROWSET>\n",
  line_prefix => "    <ROW>\n",
  line_sufix => "\n    </ROW>\n",
  line_separator => "",
  columns => $xml_cookies_col_desc_ref,
  column_separator => "\n",
};


my $xml_incompleteloads_data_col_desc_ref =
    [
     ['      <TARGET_GUID>',              '</TARGET_GUID>'],
     ['      <PAGE_URL><![CDATA[',        ']]></PAGE_URL>'],
     ['      <NUM_INCOMPLETE_LOADS>',     '</NUM_INCOMPLETE_LOADS>'],
     ['      <AVG_SERVER_TIME>',          '</AVG_SERVER_TIME>'],
     ['      <AGGREGATE_HOUR_TIMESTAMP>', '</AGGREGATE_HOUR_TIMESTAMP>']
     ];

my $xml_incomplete_loads_config_hash =
{
    "table_header" => '  <ROWSET TABLE="MGMT_RT_INCOMPLETE_LOADS" PRIMARYKEY="TARGET_GUID,'.
	'PAGE_URL,AGGREGATE_HOUR_TIMESTAMP" DATECOLUMNS="AGGREGATE_HOUR_TIMESTAMP">'.
	"\n",
	"table_footer" => "  </ROWSET>\n",
	"line_prefix" => "    <ROW>\n",
	"line_sufix" => "\n    </ROW>\n",
	"line_separator" => "",
	"columns" => $xml_incompleteloads_data_col_desc_ref,
	"column_separator" => "\n",
    };

my $xml_prmapping_data_col_desc_ref =
    [
     ['      <TARGET_GUID>',                 '</TARGET_GUID>'],
     ['      <PAGE_URL><![CDATA[',           ']]></PAGE_URL>'],
     ['      <REQUEST_URL><![CDATA[',        ']]></REQUEST_URL>'],
     ['      <NUM_CACHE_HITS>',              '</NUM_CACHE_HITS>'],
     ['      <CACHE_HITS_AVG_SVR_TIME>',     '</CACHE_HITS_AVG_SVR_TIME>'],
     ['      <NUM_NON_CACHE_HITS>',          '</NUM_NON_CACHE_HITS>'],
     ['      <NON_CACHE_HITS_AVG_SVR_TIME>', '</NON_CACHE_HITS_AVG_SVR_TIME>'],
     ['      <AGGREGATE_HOUR_TIMESTAMP>',    '</AGGREGATE_HOUR_TIMESTAMP>']
     ];

my  $xml_pr_mapping_table_config_hash =
{
    "table_header" => '  <ROWSET TABLE="MGMT_RT_PR_MAPPING" PRIMARYKEY="TARGET_GUID,'.
	'PAGE_URL,REQUEST_URL,AGGREGATE_HOUR_TIMESTAMP" DATECOLUMNS="AGGREGATE_HOUR_TIMESTAMP">'.
	"\n",
	"table_footer" => "  </ROWSET>\n",
	"line_prefix" => "    <ROW>\n",
	"line_sufix" => "\n    </ROW>\n",
	"line_separator" => "",
	"columns" => $xml_prmapping_data_col_desc_ref,
	"column_separator" => "\n",
    };
################################################################################


####################### Configurations for SQLLDR format #######################

# Configuration hash for the SQLLDR data file

my $sqlldr_data_config_hash = 
{
  table_header => "LOAD DATA\nINFILE *\nAPPEND\nINTO TABLE ".
    "mgmt_rt_metrics_raw\nfields terminated by ',' optionally enclosed ".
    "by '\"'\ntrailing nullcols\n(target_guid,metric_name,".
    "collection_timestamp TIME \"YYYY/MM/DD/ HH24:MI:SS TZD\",".
    "submit_action_timestamp,load_action_timestamp,elapsed_time,".
    "url_filename char(1024),url_base char(1024) ,visitor_node char(1024),".
    "visitor_domain,visitor_ip,cookie_index,server_in_timestamp,".
    "server_out_timestamp,server_latency_time,browser_name,".
    "browser_version,os_name,os_version)\nBEGINDATA\n",
  table_footer => "\n",
  line_prefix => "",
  line_sufix => "",
  line_separator => "\n",
  columns => "",
  column_separator => ","
};

# Configuration hash for the SQLLDR cookies file

my $sqlldr_cookies_config_hash = 
{
  table_header => "LOAD DATA\nINFILE *\nAPPEND\nINTO TABLE ".
    "MGMT_RT_COOKIE_DATA\nfields terminated by ',' optionally enclosed ".
    "by '\"'\ntrailing nullcols (raw_index,name,value)\nBEGINDATA\n",
  table_footer => "\n",
  line_prefix => "",
  line_sufix => "",
  line_separator => "\n",
  columns => "",
  column_separator => ","
};

################################################################################


# Usage new(class, err_ref, file_type, cookies_enabled, data_file_name, 
#   cookies_file_name)
# Returns undef on failure and a valid object of this class on success
# Current supported file types are : XML and SQLLDR
sub new 
{
  my ($class, $err_ref, $emdURL, $file_type, $gu_enabled, $cookies_enabled, $data_file_name, 
    $tmp_file_name, $cookies_file_name, $oms_version) = @_;
  my $self = 
  {
    file_type => $file_type,
    cookies_enabled => $cookies_enabled,
    gu_enabled => $gu_enabled,
    data_file_name => $data_file_name,
    tmp_file_name => $tmp_file_name,
    cookies_file_name => $cookies_file_name,
    cookies_enabled => undef,   # 0 = disabled, else = enabled
    multi_table_object => undef,
    oms_version => $oms_version
  };

  $self->{'cookies_enabled'} = $cookies_enabled;

  unless(defined($data_file_name) && defined($file_type) &&
   defined($cookies_enabled) && defined($emdURL))
  {
    $$err_ref = "Incomplete argument list.\n";
    return(undef);
  };

  if(($file_type =~ m/XML/i) && (! defined($tmp_file_name)))
  {
    $$err_ref = "Incomplete argument list (missing tmp_file_name).\n";
    return(undef);
  };

  if(($cookies_enabled) && ($file_type =~ m/SQLLDR/i) && 
   (! defined($cookies_file_name)))
  {
    $$err_ref = "Incomplete argument list (missing cookies_file_name).\n";
    return(undef);
  };

  # Create object
  bless $self, $class;

  my $config_hash;
  my $data_table_config;
  my $cookies_table_config;
  my $incomplete_loads_table_config;
  my $pr_mapping_table_config;
  if($file_type =~ m/SQLLDR/i)  	# SQLLDR
  {
    $config_hash = 
    {
      merged_tables => 0
    };
    $data_table_config = $sqlldr_data_config_hash;
    $cookies_table_config = $sqlldr_cookies_config_hash;
  }
  elsif($file_type =~ m/XML/i)      	# XML
  {
    $config_hash = 
    {
      file_name => $data_file_name,
      tmp_file_name => $tmp_file_name,
      merged_tables => 1,
      file_header => "<UPLOAD EMD_URL=\"$emdURL\" VERSION=\"1.0\">\n",
      file_footer => "</UPLOAD>\n",
      table_separator => ""
    };
    $data_table_config = $xml_data_config_hash;

    # if oms version less than 10.1.0.4.0, use 10.1.0.3 format
    if($oms_version < 101040)
    {
      $data_table_config->{columns} = $xml_data_col_desc_ref_10103;
    }
    # else if oms version less than 10.2.0.0.0, use 10.1.0.4 format
    elsif($oms_version < 102000)
    {
      $data_table_config->{columns} = $xml_data_col_desc_ref_10104;
    }
    # else use 10.2 format by default (set earlier)
    
    $cookies_table_config = $xml_cookies_config_hash;
    $incomplete_loads_table_config = $xml_incomplete_loads_config_hash;
    $pr_mapping_table_config = $xml_pr_mapping_table_config_hash;
  }
  else						# Unknown
  {
    $$err_ref = "Unknown file type: $self->{file_type}.\n";
    return undef;
  };

  # Create MultiTableOut
  unless($self->{multi_table_object} = new MultiTableOut $err_ref,$config_hash)
  {
    $$err_ref = "$${err_ref}Failed creating MultiTableOut.\n";
    return undef;
  };

  # Create data table
  unless(defined($self->{multi_table_object}->add_table($err_ref,
    $data_table_config, $data_file_name)))
  {
    $$err_ref = "$${err_ref}Failed creating data table.\n";
    return undef;
  };

  if($self->{gu_enabled})
  {
      #Create page-request mapping table
      unless(defined($self->{multi_table_object}->add_table($err_ref,
			   $pr_mapping_table_config, $data_file_name)))
      {
	  $$err_ref = "$${err_ref}Failed creating prmapping table.\n";
	  return undef;
      };  
      
      #Create incomplete loads table
      unless(defined($self->{multi_table_object}->add_table($err_ref,
			    $incomplete_loads_table_config, $data_file_name)))
      {
	  $$err_ref = "$${err_ref}Failed creating incomplete loads table.\n";
	  return undef;
      };  
  }

  unless($self->{cookies_enabled})		# Cookies are not enabled.
  {
    return $self;
  };

  # Create cookies table
  unless(defined($self->{multi_table_object}->add_table($err_ref,
    $cookies_table_config, $cookies_file_name)))
  {
    $$err_ref = "$${err_ref}Failed creating cookies table.\n";
    return undef;
  };

  return $self;
};

# Usage : add_data_line(self, err_ref, data_row)
# Returns 1 on success and 0 on failure
sub add_data_line 
{
  my ($self, $err_ref, $data_row) = @_;
  if($self->{oms_version} < 101040)      # 10.1.0.3 format
  {
    # remove database_time and visitor_ip_num cols from input row
    splice(@{$data_row}, 16, 1);
    splice(@{$data_row}, 11, 1);
  }
  elsif($self->{oms_version}  < 102000)  # 10.1.0.4 format
  {
    # remove database_time col from input row
    splice(@{$data_row}, 16, 1); 
  }
  unless($self->{multi_table_object}->add_line($err_ref, 0, $data_row))
  {
    $$err_ref = "$${err_ref}Failed adding data row.\n";
    return 0;
  };
  return 1;
};

# Usage : add_cookie_line(self, err_ref, data_row)
# Returns 1 on success and 0 on failure
sub add_cookie_line 
{
  my ($self, $err_ref, $data_row) = @_;
  if($self->{cookies_enabled})
  {
    unless($self->{multi_table_object}->add_line($err_ref, ($self->{gu_enabled}) ? 3 : 1, $data_row))
    {
      $$err_ref = "$${err_ref}Failed adding cookie row.\n";
      return 0;
    };
    return 1;
  };
  $$err_ref = "Cookies are disabled.\n";
  return 0;
};


# Usage : add_data_line(self, err_ref, data_row)
# Returns 1 on success and 0 on failure
sub add_incomplete_loads_line 
{
  my ($self, $err_ref, $data_row) = @_;
  if($self->{gu_enabled})
  {
      unless($self->{multi_table_object}->add_line($err_ref, 2, $data_row))
      {
	  $$err_ref = "$${err_ref}Failed adding data row.\n";
	  return 0;
      };
      return 1;
  }
  $$err_ref = "Grand Unification is disabled.\n";
  return 1;
};

# Usage : add_data_line(self, err_ref, data_row)
# Returns 1 on success and 0 on failure
sub add_pr_mapping_line 
{
    my ($self, $err_ref, $data_row) = @_;
    if($self->{gu_enabled})
    {
	unless($self->{multi_table_object}->add_line($err_ref, 1, $data_row))
	{
	    $$err_ref = "$${err_ref}Failed adding data row.\n";
	    return 0;
	};
	return 1;
    }
    $$err_ref = "Grand Unification is disabled.\n";
    return 1;
}


# Usage : end_data(self, err_ref)
# Returns 1 on success and 0 on failure
sub end_data 
{
  my ($self, $err_ref) = @_;
  unless($self->{multi_table_object}->end_data($err_ref))
  {
    $$err_ref = "$${err_ref}Failed end_data.\n";
    return 0;
  };
  return 1;
};

# Usage : end_force(self)
# Returns 1 on success and 0 on failure
sub end_force 
{
  my ($self) = @_;
  $self->{multi_table_object}->end_force();
  return 1;
}

# Usage: lookup_file_extension(file_type)
# Returns the file extension for a known type or "" for unknown types
# See that this is a Static method.
sub lookup_file_extension {
  my ($file_type) = @_;
  if($file_type =~ m/XML/i) {
    return ".xml";
  } elsif ($file_type =~ m/SQLLDR/i) {
    return ".ctl";
  };
  return "";
}

# So the require or use succeeds
1;

