# Copyright (c) 2003, 2005, Oracle. All rights reserved.  
#
# NAME
#   IFSProcessInfo.pl
#     - gets the pids for all iFS processes of all the iFS domains under the
#       given Oracle Home
#                       
# DESCRIPTION
#   IFSProcessInfo.pl <script_dir> <oracle_home> 
#
#    where:
#      <script_dir> is filepath of the scripts directory
#      <oracle_home> is the oracle home in which the iAS is installed
#
#    returns: 
#      [display-name]|[start-pid]|[comma-separated-pid-list]
#      one line per iFS domain
#
#      where:
#        <display-name> the target display name for the iFS domain
#        <start-pid> the pid for the iFS process that has the earlist
#          start time of all the iFS processes of the domain
#        <comma-separated-pid-list> a comma separated list of pids for all the
#          iFS processes of the domain 
#
#
# HISTORY
#   02/18/2004    mhong - fix transformDomainName subroutine to replace ' '
#                         with '_' as well
#   01/22/2004    mhong - support windows platform
#   11/03/2003    mhong - fix start time for files and websites targets
#                       - handle multiple domains under same OH properly
#   10/13/2003    mhong - fix for linux platform
#   10/01/2003    mhong - fix iteration on hash %ifs_targets so it can handle
#                         multiple iFS domains under same Oracle Home
#   06/18/2003    mhong - consider OC4J_iFS_cmsdk and OC4J_iFS_webstarterapp
#                         pids for start-pid
#   06/12/2003    mhong - created


use strict;
require "emd_common.pl";
require "iasresourceusage.pl";
require "iasntresourceusage.pl";

my %ias_names;
my %ifs_targets;
my %ifs_domain_types;
my %domains;
my $oracle_home;
my $the_ias_name;
my $target_info;
my $ias_name;
my $domain_name;
my $domain_type;
my $domain_info;
my $display_name;
my $min_start_time;
my $start_pid;
my $pid_list;

my $transformed_domain_name;
my @dc_node_pids;
my @http_node_pids;

$oracle_home = $ARGV[1];

# parse the targets.xml under the given Oracle Home to populate the hashes:
# %ias_names maps oracle homes to oracle_ias target names;
# %ifs_targets maps ifs domain names to oracle_ifs target display names and
# parent oracle_ias target names;
# %ifs_domain_types maps ifs domain names to domain types
parseTargetsXml($oracle_home);

# get the oracle_ias target name for the given oracle home
$the_ias_name = $ias_names{$oracle_home} || "";

executePSCommand(); # required by getTimeInformation()

if (($^O =~ "Windows") || ($^O =~ "MSWin32"))  # windows platforms
{
    while (($domain_name, $domain_type) = each(%ifs_domain_types))
    {
	if (isDomainUnderCurrentOracleHome($domain_name))
	{
	    $transformed_domain_name = transformDomainName($domain_name);

	    # this populates two pid lists: dc_node_pids and http_node_pids
	    getPidsForDomain($oracle_home,
			     $transformed_domain_name,
			     $domain_type);

	    $domain_info = $domains{$domain_name} || "0||";
	    ($min_start_time, $start_pid, $pid_list) =
		split('\|', $domain_info);

	    foreach my $pid (@dc_node_pids)
	    {
		my $time_info = getTimeInformation($pid);
		if ($time_info ne "|")
		{
		    my ($up_time, $start_time) = split('\|', $time_info);
		    if ($start_pid eq "" || $min_start_time > $start_time)
		    {
			$start_pid = $pid;
			$min_start_time = $start_time;
		    }

		    if ($pid_list eq "")
		    {
			$pid_list = $pid;
		    }
		    else
		    {
			$pid_list = "$pid_list,$pid";
		    }
		}
	    }

	    foreach my $pid (@http_node_pids)
	    {
		my $time_info = getTimeInformation($pid);
		if ($time_info ne "|")
		{
		    my ($up_time, $start_time) = split('\|', $time_info);
		    if ($start_pid eq "" || $min_start_time > $start_time)
		    {
			$start_pid = $pid;
			$min_start_time = $start_time;
		    }

		    # $pid_list is untouched
		}
	    }

	    $domains{$domain_name} = "$min_start_time|$start_pid|$pid_list";
	}
    }
}
else  # non-windows platforms
{
    # find all the iFS processes under the Oracle Home, they could belong to
    # multiple iFS domains
    my $ps_command = getPsCommand();
    my $ps_output = `$ps_command | grep java | grep "$oracle_home" | grep oracle.ifs.management.domain | grep -v grep`;

    my @lines = split( "\n", $ps_output );

    foreach my $line (@lines)
    {
	$domain_name = getIfsDomainName($line);
	$domain_info = $domains{$domain_name} || "0||";
	($min_start_time, $start_pid, $pid_list) = split('\|', $domain_info);

	# the output is in the form: 
	# {USER} {PID} {CPU} {MEM} {TT} {??} {START TIME} {??} {COMMAND}
	my @tokens = split(" ", $line);

	my $pid = $tokens[1];
	my $time_info = getTimeInformation($pid);
	my ($up_time, $start_time) = split('\|', $time_info);

	if ($start_pid eq "" || $min_start_time > $start_time)
	{
	    $start_pid = $pid;
	    $min_start_time = $start_time;
	}

	if ($pid_list eq "")
	{
	    $pid_list = $pid;
	}
	else
	{
	    $pid_list = "$pid_list,$pid";
	}

	$domains{$domain_name} = "$min_start_time|$start_pid|$pid_list";
    } # end of foreach

    # evaluate well known OC4J instance pids for start times when there is one iFS
    # domain under the given oracle home/ias instance or when there are multiple
    # iFS domains under the same oracle home but the domains have different types.
    # If there are multiple same domain typed iFS domains under the same
    # oracle home/ias instance, we can not tell which OC4J instances are for which
    # iFS domain by just checking their ps output. We have to igonre the OC4J
    # instances in that case.
    while (($domain_name, $domain_type) = each(%ifs_domain_types))
    {
	if ($domain_name = getUniqueIfsDomainName($the_ias_name, $domain_type))
	{
	    if ($domain_type eq "cmsdk")
	    {
		checkOc4jInstancePid($ps_command, $oracle_home, $domain_name,
				     "OC4J_iFS_cmsdk");
		checkOc4jInstancePid($ps_command, $oracle_home, $domain_name,
				     "OC4J_iFS_webstarterapp");
	    }
	    elsif ($domain_type eq "files")
	    {
		checkOc4jInstancePid($ps_command, $oracle_home, $domain_name,
				     "OC4J_iFS_files");
	    }
	    elsif ($domain_type eq "websites")
	    {
		checkOc4jInstancePid($ps_command, $oracle_home, $domain_name,
				     "OC4J_iFS_websites");
	    }
	}
    }
} # end of if (($^O =~ "Windows") || ($^O =~ "MSWin32"))

# print the display name, start pid, and the pid list, one line per domain
while (($domain_name, $target_info) = each(%ifs_targets))
{
    ($display_name, $ias_name) = split('\|', $target_info);

    # make sure the returned oracle_ifs targets are member of the oracle_ias
    # target under the given oracle home
    if ($ias_name eq $the_ias_name)
    {
	$domain_info = $domains{$domain_name} || "0||";
	($min_start_time, $start_pid, $pid_list) = split('\|', $domain_info);

	print "$display_name|$start_pid|$pid_list\n";
    }
}

# end of main

###########################################################################

# Parse the targets.xml under the given oracle home to get the mapping
# between the oracle homes and the target names for all the oracle_ias targets,
# the mapping between the domain names and the target display names plus
# the parent oracle_ias target names of all the oracle_ifs targets, and the
# mapping between the domain names and the domain types for all the
# oracle_ifs targets.
# Parameters
#   oracle_home: the Oracle Home under which the targets.xml is located
# Return
#   no explicit return, but it updates the hashes:
#   %ias_names, %ifs_targets, and %ifs_domain_types
sub parseTargetsXml
{
    my $oracle_home = shift(@_);

    # open $ORACLE_HOME/sysman/emd/targets.xml
    my $filename = $oracle_home . "/sysman/emd/targets.xml";
    if (!open(TARGETS, $filename))
    {
	EMD_PERL_ERROR("Cannot open $filename: $!");
	return;
    }

    # track if the line is for oracle_ias, oracle_ifs, or other targets:
    #   0 - neigher oracle_ias nor oracle_ifs
    #   1 - oracle_ias target
    #   2 - oracle_ifs target
    #   3 - oracle_ifs target composite membership
    my $which_target_entry = 0;
    my $ias_name;
    my $ias_oracle_home;
    my $ifs_display_name;
    my $ifs_domain_name;
    my $ifs_domain_type;

    while (<TARGETS>)
    {
	chomp;

	if ($which_target_entry == 0)
	{
	    # check for the start of an oracle_ias target and
            # get its NAME attribute value
	    $ias_name = getIasName($_);
	    if ($ias_name ne "")
	    {
		$which_target_entry = 1;
	    }
	    else
	    {
		# check for the start of an oracle_ifs target and
		# get its DISPLAY_NAME attribute value
		$ifs_display_name = getIfsDisplayName($_);
		if ($ifs_display_name ne "")
		{
		    $which_target_entry = 2;
		}
	    }
	}
	elsif ($which_target_entry == 1) # oracle_ias target
	{
	    # check the end of the oracle_ias target
	    if (index($_, "</Target>") != -1)
	    {
		$which_target_entry = 0;
		$ias_name = "";
	    }
	    else
	    {
		# check for the OracleHome instance property and get its value
		my $tmp_oracle_home = getIasOracleHomePropertyValue($_);

		if ($tmp_oracle_home ne "" && $ias_name ne "")
		{
		    $ias_oracle_home = $tmp_oracle_home;
		    $ias_names{$ias_oracle_home} = $ias_name;
		}
	    }
	}
	elsif ($which_target_entry == 2) # oracle_ifs target
	{
	    # check the end of the oracle_ifs target
	    if (index($_, "</Target>") != -1)
	    {
		$which_target_entry = 0;
		$ifs_display_name = $ifs_domain_name = $ifs_domain_type = "";
	    }
	    elsif (index($_, "<CompositeMembership>") != -1)
	    {
		$which_target_entry = 3;
	    }
	    else
	    {
		# check for the DomainName or the DomainType instance
                # property and get its value
		my $tmp_domain_name = getIfsDomainNamePropertyValue($_);
		my $tmp_domain_type = getIfsDomainTypePropertyValue($_);

		if ($tmp_domain_name ne "" && $ifs_display_name ne "")
		{
		    $ifs_domain_name = $tmp_domain_name;
		    $ifs_targets{$ifs_domain_name} = $ifs_display_name;
		}

		if ($tmp_domain_name ne "" && $ifs_domain_type ne "")
		{
		    $ifs_domain_name = $tmp_domain_name;
		    $ifs_domain_types{$ifs_domain_name} = $ifs_domain_type;
		}

		if ($tmp_domain_type ne "" && $ifs_domain_name ne "")
		{
		    $ifs_domain_type = $tmp_domain_type;
		    $ifs_domain_types{$ifs_domain_name} = $ifs_domain_type;
		}
	    }
	}
	elsif ($which_target_entry == 3) # oracle_ifs target membership
	{
	    # check the end of the oracle_ifs target membership
	    if (index($_, "</CompositeMembership>") != -1)
	    {
		$which_target_entry = 2;
	    }
	    else
	    {
		# check for parent oracle_ias target name
		my $ias_name = getParentIasName($_);
		if ($ias_name ne "" && $ifs_domain_name ne "")
		{
		    my $ifs_target_info = $ifs_targets{$ifs_domain_name};
		    my ($my_display_name, $my_ias_name) =
			split('\|', $ifs_target_info);
		    $ifs_targets{$ifs_domain_name} =
			"$my_display_name|$ias_name";
		}
	    }
	}
    } # end of while

    close(TARGETS) || EMD_PERL_ERROR("Cannot close $filename: $!");
}


# Gets the NAME attribute value for the oracle_ias target.
# Parameters
#   line: a line that is read from targets.xml
# Return
#   the value of the NAME attribute, if the line contains the open
#   tag of the oracle_ias target and the NAME attribute; or empty
#   string, otherwise
sub getIasName
{
    my $line = shift(@_);
    my $name = "";

    # oracle_ias target starts with "<Target TYPE="oracle_ias" ..."
    if (index($line, "<Target TYPE=\"oracle_ias\"") != -1)
    {
	my $pos = index($line, "NAME=\"");
	if ($pos != -1)
	{
	    $line = substr($line, $pos + 6);
	    $line =~ s/\".*//;

	    $name = $line;
	}
    }

    return $name;
}


# Gets the OracleHome instance property value of the oracle_ias target.
# Parameters
#   line: a line that is read from targets.xml
# Return
#  the value of the OracleHome instance property, if the line contains that
#  property; or empty string, otherwise
sub getIasOracleHomePropertyValue
{
    my $line = shift(@_);
    my $ias_oracle_home = "";

    if (index($line, "<Property NAME=\"OracleHome\"") != -1)
    {
	my $pos = index($line, "VALUE=\"");
	if ($pos != -1)
	{
	    $line = substr($line, $pos + 7);
	    $line =~ s/\".*//;

	    $ias_oracle_home = $line;
	}
    }

    return $ias_oracle_home;
}


# Gets the DISPLAY_NAME attribute value for the oracle_ifs target.
# Parameters
#   line: a line that is read from targets.xml
# Return
#   the value of the DISPLAY_NAME attribute, if the line contains the open
#   tag of the oracle_ifs target and the DISPLAY_NAME attribute; or empty
#   string, otherwise
sub getIfsDisplayName
{
    my $line = shift(@_);
    my $display_name = "";

    # oracle_ifs target starts with "<Target TYPE="oracle_ifs" ..."
    if (index($line, "<Target TYPE=\"oracle_ifs\"") != -1)
    {
	my $pos = index($line, "DISPLAY_NAME=\"");
	if ($pos != -1)
	{
	    $line = substr($line, $pos + 14);
	    $line =~ s/\".*//;

	    $display_name = $line;
	}
    }

    return $display_name;
}


# Gets the parent oracle_ias name for the oracle_ifs target.
# Parameters
#   line: a line that is read from targets.xml
# Return
#   the value of the NAME attribute, if the line indicates a memberof
#   relationship with an oracle_ias target; or empty string, otherwise
sub getParentIasName
{
    my $line = shift(@_);
    my $ias_name = "";

    if (index($line, "<MemberOf TYPE=\"oracle_ias\"") != -1)
    {
	my $pos = index($line, "NAME=\"");
	if ($pos != -1)
	{
	    $line = substr($line, $pos + 6);
	    $line =~ s/\".*//;

	    $ias_name = $line;
	}
    }

    return $ias_name;
}


# Gets the DomainName instance property value of the oracle_ifs target.
# Parameters
#   line: a line that is read from targets.xml
# Return
#  the value of the DomainName instance property, if the line contains that
#  property; or empty string, otherwise
sub getIfsDomainNamePropertyValue
{
    my $line = shift(@_);
    my $domain_name = "";

    if (index($line, "<Property NAME=\"DomainName\"") != -1)
    {
	my $pos = index($line, "VALUE=\"");
	if ($pos != -1)
	{
	    $line = substr($line, $pos + 7);
	    $line =~ s/\".*//;

	    $domain_name = $line;
	}
    }

    return $domain_name;
}


# Gets the DomainType instance property value of the oracle_ifs target.
# Parameters
#   line: a line that is read from targets.xml
# Return
#  the value of the DomainType instance property, if the line contains that
#  property; or empty string, otherwise
sub getIfsDomainTypePropertyValue
{
    my $line = shift(@_);
    my $domain_type = "";

    if (index($line, "<Property NAME=\"DomainType\"") != -1)
    {
	my $pos = index($line, "VALUE=\"");
	if ($pos != -1)
	{
	    $line = substr($line, $pos + 7);
	    $line =~ s/\".*//;

	    $domain_type = $line;
	}
    }

    return $domain_type;
}


# Get the iFS domain name from the given process' command line.
# Parameters
#   ps_output: the complete ps output for the process
# Return
#   the iFS domain name, which is in the format of
#   ifs://<dbhost>:<dbport>:<dbservice>:<schema>
sub getIfsDomainName
{
    my $ps_output = shift(@_);
    my $domain_name = "";

    # domain name starts with "ifs://"
    my $pos = index($ps_output, "ifs://");
    if ($pos != -1)
    {
	$ps_output = substr($ps_output, $pos);

	# domain name ends with " "
	$ps_output =~ s/\s.*//;

	# remove the trailing '"' if present
	my $end_char = substr($ps_output, -1, 1);
	if ($end_char eq "\"")
	{
	    chop($ps_output);
	}

	$domain_name = $ps_output;
    }

    return $domain_name;
}


# Gets the domain name for the one and only iFS domain configured with the
# given domain type and under the given oracle home/ias instance.
# Parameters
#   the_ias_name: the oracle_ias target name for the given ias instance
#   the_domain_type: the given domain type
# Return
#   the iFS domain name, when there is one and only one iFS domain configured
#   for the given ias instance and the given domain type
sub getUniqueIfsDomainName
{
    my ($the_ias_name, $the_domain_type) = @_;
    my $unique_domain_name = "";
    my $dup = "false";

    while (my ($domain_name, $target_info) = each(%ifs_targets))
    {
	my ($display_name, $ias_name) = split('\|', $target_info);
	my $domain_type = $ifs_domain_types{$domain_name} || "";

	if ($ias_name eq $the_ias_name &&
	    $domain_type eq $the_domain_type)
	{
	    if ($dup eq "false" && $unique_domain_name eq "")
	    {
		$unique_domain_name = $domain_name;
	    }
	    else
	    {
		$unique_domain_name = "";
		$dup = "true";
	    }
	}
    }

    return $unique_domain_name;
}


# Evaluate the OC4J instance's pid to see if it has an earlier start time
# compared with the other iFS processes in the given iFS domain.
# Parameters
#   ps_command: the ps command, ported for different UNIX platforms
#   oracle_home: the oracle home
#   domain_name: the name of the domain this OC4J instance is associated with
#   oc4j_name: the OC4J instance name
# Return
#   no explicit return, it updates the hash %domains when necessary
sub checkOc4jInstancePid
{
    my ($ps_command, $oracle_home, $domain_name, $oc4j_name) = @_;
    my $domain_info;
    my $min_start_time;
    my $start_pid;
    my $pid_list;

    my $ps_output = `$ps_command | grep java | grep "oracle.home=$oracle_home" | grep "oracle.oc4j.instancename=$oc4j_name" | grep -v grep`;
    my @lines = split("\n", $ps_output);

    foreach my $line (@lines)
    {
	$domain_info = $domains{$domain_name} || "0||";
	($min_start_time, $start_pid, $pid_list) = split('\|', $domain_info);

	# the output is in the form: 
	# {USER} {PID} {CPU} {MEM} {TT} {??} {START TIME} {??} {COMMAND}
	my @tokens = split(" ", $line);

	my $pid = $tokens[1];
	my $time_info = getTimeInformation($pid);
	my ($up_time, $start_time) = split('\|', $time_info);

	if ($start_pid eq "" || $min_start_time > $start_time)
	{
	    $start_pid = $pid;
	    $min_start_time = $start_time;
	}

	# $pid_list is untouched
	$domains{$domain_name} = "$min_start_time|$start_pid|$pid_list";
    }
}

#########################################################################

# Check whether the given domain is under the current oracle home
# Parameters
#   domain_name the name of the domain
# Return
#   true if the domain is under the current oracle home; false, otherwise
sub isDomainUnderCurrentOracleHome
{
    my $domain_name = shift(@_);
    my $target_info = $ifs_targets{$domain_name};
    my ($display_name, $ias_name) = split('\|', $target_info);

    return ($ias_name eq $the_ias_name);
}


# Transform the domain name by removing the prefix "ifs://" and replacing any
# ' ', ':', or '.' with '_'.
# Parameters
#   domain_name: the name of the domain, in the format of
#                ifs://<db-host>:<db-port>:<db-service>:<schema>
# Return
#   the transformed domain name, which could be used 
sub transformDomainName
{
    my $domain_name = shift(@_);

    # remove leading ifs://
    $domain_name =~ s/ifs:\/\///;

    # replace ' ', ':', or '.' with '_'
    $domain_name =~ s/[ :\.]/_/g;

    return $domain_name;
}


# Gather all the process ids stored in .pid files for this domain instance
# Parameters
#   oracle_home: the oracle home
#   transformed_domain_name: the tranformed domain name - no prefix ifs:// and
#                            ' ' and ':' and '.' being replaced with '_'
#   domain_type: the type of the domain: cmsdk or files
# Return
#   no explicit return, it populates two pid lists:
#     dc_node_pids contains the process ids of the Domain Controller and the
#     non-HTTP Node processes
#     http_node_pids contains the process ids of the HTTP Node processes
sub getPidsForDomain
{
    my ($oracle_home, $transformed_domain_name, $domain_type) = @_;
    my $dir =
	$oracle_home . "/ifs/" . $domain_type . "/log/" . $transformed_domain_name;
    my $basename;
    my $filename;
    my $pid;

    # reset the pid lists first
    @dc_node_pids = ();
    @http_node_pids = ();

    # replace any '\' to '/'
    $dir =~ s/\\/\//g;

    if (!opendir(PIDDIR, $dir))
    {
	EMD_PERL_ERROR("Cannot list directory $dir: $!");
	return;
    }

    while ($basename = readdir(PIDDIR))
    {
	$filename = $dir . '/' . $basename;

	if ((-f $filename) && ($basename =~ "\.pid"))
	{
	    $pid = getProcessIdFromPidFile($filename);
	    if ($pid ne "")
	    {
		if ($basename eq "DomainController\.pid")
		{
		    push(@dc_node_pids, $pid);
		}
		elsif ($basename =~ "_HTTPNodeGuardian\.pid")
		{
		    push(@http_node_pids, $pid);
		}
		elsif ($basename =~ "_NodeGuardian\.pid")
		{
		    push(@dc_node_pids, $pid);
		}
		elsif ($basename =~ "_NodeManager\.pid")
		{
		    push(@dc_node_pids, $pid);
		}
	    }
	}
    }

    closedir(PIDDIR);
}


# Retrieve the process id from the given .pid file. If the .pid file contains
# more than one process id, we only care for the first one.
# Parameters
#   pid_filename: the absolute path of the .pid file
# Return
#   the pid
sub getProcessIdFromPidFile
{
    my $pid_filename = shift(@_);

    # checkPIDFile is defined in iasntresourceusage.pl
    my @pids = checkPIDFile($pid_filename);

    if (@pids && $#pids >= 0)
    {
	return $pids[0];
    }
    else
    {
	return "";
    }
}


# EOF
