#!/usr/local/bin/perl
# 
# $Header: 
#
# collect_props.pl
# 
# Copyright (c) 2009, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      collect_props.pl
#
#    DESCRIPTION
#      Generates output for P6 OCM collection
#
#    NOTES
#      This script uses a data driver file (P6DataDriver.xml) plus the available P6 .properties files in order to generate the output.
#
#   Usage:
#       collect_props <P6HomeDir> <TableName> <CollectionType>
#           where CollectionType is "db", "prefs", or "admin"
#
#    MODIFIED   (MM/DD/YY)
#    snathan     07/30/09 - Fixed handling of buildid.properties file as per
#                           Nestor
#    nmittal     07/03/09 - Creation


    use XML::Parser;

    if (@ARGV < 3)
    {
        print "\nUsage: collect_props <P6HomeDir> <TableName> <CollectionType>\n";
        print "  where CollectionType is \"admin\", \"db\", or \"prefs\"\n";
        print "\n";
        exit(1);
    }
    my $P6_Home_Dir=$ARGV[0];
    $ThisTable = $ARGV[1];

    my $collectType = $ARGV[2];
    my $propsFilename;

    $bGetColumns = 0;
    $allColumnsList = ();

    ## Read the version buildid into $P6BuildId
    if (!open(BUILDID, $P6_Home_Dir."/buildid.properties"))
    {
       print "em_error=can't read P6 version number file";
       exit(1);
    }
    $propline = <BUILDID>;
    chomp($propline);

    ## Always only one line in this file
    my ($key, $value) = split(/=/, $propline);
    $P6BuildId = $value;
    $version621 = (index($P6BuildId, "6.2.1") >= 0);
    
    ## Populate allColumnsList with column names as defined in the data driver file for the given table name $ThisTable
    
    $slashpos = rindex($0, "/");
    $P6DataDriverFile = substr($0, 0, $slashpos+1)."P6DataDriver.xml";
    
    parseDataDriver($P6DataDriverFile);
    
    %coltypes = qw (db DB_Config prefs GlobalPrefs admin Admin_Config);
    $propsFilename = $coltypes{$collectType};
    if (length($propsFilename) == 0)
    {
        print "em_error=Bad or missing collection type param '".$collectType."'";
        exit(1);
    }

    ## Print out the values associated with each required column

    my $fileNum = "";    
    if ($collectType eq "db" || $collectType eq "prefs")
    {
        $fileNum = 0;
    }
    while (1)
    {
        $fullPropsPath = $P6_Home_Dir."/".$propsFilename.$fileNum.".properties";
        if (-e $fullPropsPath)
        {
            printColumnValues($fullPropsPath);
            $fileNum++;
        }
        else
        {
            last;
        }
    }
    

    ###############################################################
    
    sub parseDataDriver
    {
        my ($dataDriverFilename) = @_;
        
        $parser = new XML::Parser(Style => 'Subs');
        open(TEXT, $dataDriverFilename) or die ("can't open P6 data driver file '".$dataDriverFilename."'");

        $parser->parsefile($dataDriverFilename);
    }
    
    ###############################################################
    
    sub printColumnValues
    {
        my ($filename) = @_;
        
        open(FILE,$filename) or die("Unable to open file ".$filename);

        ## Read properties file into %mappings array
        my %mappings;
        my $line;
        foreach $line (<FILE>)
        {
            chomp($line);
            # NOTE: can't use split() here because '=' sometimes appears within a value
            $i = index($line, "=");
            if ($i >= 0)
            {
                $key = substr($line, 0, $i);
                $value = substr($line, $i+1);
                
                # get rid of any pipe chars
                $value =~ tr/|/!/;
                
                #put it in the hash map
                $mappings{uc($key)} = $value;
            }
        }

        print "em_result=";
        ## Now print out each column value
        my $first = 1;
        foreach $colname (@allColumnsList)
        {
#print " ".$colname."=";
            if (!$first)
            {
                print "|";
            }
            $first = 0;
            my $lookupKey = getLookupKey($colname);
            
            $val = truncate_string($mappings{$lookupKey}, $colname);

            print $val;

#print "\n";
            
        }
        print "\n";
    }

    ###########################################################
    ##  Given the column name, get the lookup key
    ##  In most cases, it will just be the uppercase of the
    ##  column name.  However sometimes it will be adjusted
    ##  based on the version of P6 being collected from.
    ###########################################################
    sub getLookupKey
    {
        my ($colname) = @_;


        ## Handle special cases
        my $i = index($colname, "ContentRepository.JackRabbit");
        if ($i == 0 && $version621 == 1)
        {
            ## Remove 'JackRabbit' from name
            $colname = "ContentRepository".substr($colname, 28);
        }

        return uc($colname);
    }
    
    ###############################################################
    ## Called each time a TableDescriptor node is parsed from XML
    ###############################################################
    sub TableDescriptor
    {
#print "In TableDescriptor\n";
        my $count = @_;
        for ($k = 0; $k < $count; $k++)
        {
            $attr = $_[$k];
            if ($attr eq "TABLE_NAME")
            {
                if ($_[++$k] eq $ThisTable)
                {
                    $bGetColumns = 1;
                    last;
                }
            }
        }
    }
    
    ###############################################################
    ## Called each time an end tag for TableDescriptor is parsed from XML
    ###############################################################
    
    sub TableDescriptor_
    {
        $bGetColumns = 0;
    }
    
    ###############################################################
    ## Called each time a ColumnDescriptor node is parsed from XML
    ###############################################################
    sub ColumnDescriptor
    {
        if (!$bGetColumns)
        {
            return;
        }
        my $count = @_;
        for ($k = 0; $k < $count; $k++)
        {
            $attr = $_[$k];
            if ($attr eq "NAME")
            {
                $k++;
                $columnName = $_[$k];
#print $_[$k]."\n";
                $allColumnsList[@allColumnsList] = $columnName;
                last;
            }

        }
    }
    
    ########################################################
    ##  Truncate string according to rules
    ##     
    ########################################################
    sub truncate_string
    {
        my ($string, $col) = @_;

        ## If column name has 'path' in it, truncate to 1000 chars, otherwise truncate everything else to 100 chars.

        if((rindex $col,ucfirst('path'))>-1 || (rindex $col,uc('path'))>-1 || (rindex $col,'path')>-1)
        {
            $string = substr($string, 0, 1000);
        }
        else
        {
            $string = substr($string, 0, 100);
        }
        return $string;
    }

