# $Header: Ptdpm3.pm 13-may-2005.00:29:16 mgoodric Exp $
#
# Package : Ptdpm3.pm
#
# Copyright (c) 2002, 2005, Oracle. All rights reserved.  
#
#   NAME
#       Ptdpm3.pm
#
#   DESCRIPTION
#       Common utilities
#
#   NOTES
#
#   MODIFIED     (MM/DD/YY)
#      mgoodric   05/13/05 - fix warning with bitwise operator on 5.8.3 
#      mgoodric   09/27/04 - code cleanup 
#      mgoodric   05/27/04 - Added trunc()
#      mgoodric   03/17/04 - Added left() and right()
#      mgoodric   06/16/03 - fix converting Solaris timezone to JAVA timezones
#      mgoodric   05/11/03 - added -debug tracing
#      mgoodric   04/08/03 - fix finding MAC address
#      goramamo   08/16/02 - 904 Changes
#      xxu        06/25/02 - remove /usr/local/bin/perl
#      goramamo   06/02/02 - Added sortuniq sub
#      mgoodric   02/14/02 - Make scripts more portable
#      goramamo   12/15/01 - Creation
#
##*************************************************************

use strict;
use FileHandle;

package Ptdpm3;

use Carp;
require 5.005;

#require 5.6.1;
require Exporter;

use Ptdpm2;

#******************************************
#   Export Subroutines
#******************************************

@Ptdpm3::ISA    = ('Exporter');
@Ptdpm3::EXPORT = qw(
  &trim
  &trim1DimArrayAfterSplit
  &trim2DimArrayAfterSplit
  &sortuniq
  &joinValue
  &setOutputAutoflush
  &debugTrace
  &equalsIgnoreCase
  &isNumeric
  &isDay
  &isMonth
  &isYear
  &isHour
  &isMins
  &isSecs
  &isInRange
  &isTime
  &zeroPad
  &left
  &right
  &trunc
  );

#******************************************
#     Global Variables
#******************************************

use constant NIL => '';

#******************************************
#     Exported Subroutines
#******************************************

sub trim
{

    #Arguments:  string or array
    #Outputs  :  string or array
    #Function :  trim

    my (@out) = @_;
    for (@out)
    {
        s/^\s+//;
        s/\s+$//;
    }

    return wantarray ? @out : $out[0];
}

sub trim1DimArrayAfterSplit
{

    #Arguments: 1 dim array
    #Outputs  : 1 dim array
    #Function : trim

    my (@out) = @_;
    my @outtrim = ();
    my $i       = 0;
    my $tag     = NIL;
    my $value   = NIL;
    my $outsize = 0;
    $outsize = scalar(@out);

    for ($i = 0 ; $i < $outsize ; $i++)
    {
        ($tag, $value) = split ($Ptdpm2::SEP, $out[$i], 2);
        chomp($tag);
        $value = NIL if (!defined $value);
        chomp($value);
        $outtrim[$i] = joinValue($tag, $value);
    }

    return @outtrim;
}

sub trim2DimArrayAfterSplit
{

    #Arguments: 2 dim array
    #Outputs  : 2 dim array
    #Function : trim

    my (@out) = @_;
    my @outtrim = ();
    my $i       = 0;
    my $j       = 0;
    my $tag     = NIL;
    my $value   = NIL;
    my $outsize = 0;
    $outsize = scalar(@out);

    for ($i = 0 ; $i < $outsize ; $i++)
    {
        for ($j = 0 ; $j < 7 ; $j++)
        {
            ($tag, $value) = split ($Ptdpm2::SEP, $out[$i][$j], 2);
            chomp($tag);
            $value = NIL if (!defined $value);
            chomp($value);
            $outtrim[$i][$j] = joinValue($tag, $value);
        }
    }

    return @outtrim;
}

sub sortuniq
{
    my (@in) = @_;
    my @tmp  = ();
    my @out  = ();
    my $rest = 'nonesuch';

    @tmp = sort (@in);
    @out = grep ($_ ne $rest && (($rest) = $_), @tmp);

    return @out;
}

sub joinValue($;$)
{
    my ($tag, $value) = @_;
    $value = NIL if (!defined $value);

    return join ($Ptdpm2::SEP, trim($tag), trim($value));
}

sub setOutputAutoflush
{
    my $outHandle = select(STDOUT);
    $| = 1;    # set OUTPUT_AUTOFLUSH
    select(STDERR);
    $| = 1;                # flush std error as well
    select($outHandle);    #reset handle back to original
}

sub debugTrace($;$)
{
    my ($value, $out) = @_;
    if (defined $main::DEBUG && $main::DEBUG > 0)
    {
        if (defined $out)
        {
            printf($out "%s\n", $value);
        }
        else
        {
            printf("%s\n", $value);
        }
    }
}

sub isNumeric($)
{
    my ($value)  = @_;
    my $result = 0;
    if (defined $value)
    {
        $result = $value =~ m/^[0-9]+$/;
    }

    return $result;
}

sub isDay($)
{
    return isInRange($_[0], 1, 31);
}

sub isMonth($)
{
    return isInRange($_[0], 1, 12);
}

sub isYear($)
{
    return isInRange($_[0], 0, 2038);
}

sub isHour($)
{
    return isInRange($_[0], 0, 23);
}

sub isMins($)
{
    return isInRange($_[0], 0, 59);
}

sub isSecs($)
{
    return isInRange($_[0], 0, 59);
}

sub isTime($)
{
    my ($value) = @_;
    my @in     = ();
    my $result = 0;

    $value = NIL if (!defined $value);
    if (length($value) > 0)
    {
        @in = split ($Ptdpm2::SEP, $value);
        if (scalar(@in) == 2)
        {
            $result = (isHour($in[0]) & isMins($in[1]));
        }
        elsif (scalar(@in) == 3)
        {
            $result = (isHour($in[0]) & isMins($in[1]) & isSecs($in[2]));
        }
    }

    return $result;
}

sub isInRange($$;$)
{
    my ($value, $lowRange, $highRange) = @_;
    my $result = 0;
    if (isNumeric($value))
    {
        if (!defined $highRange)
        {
            $result = 1 if ($value >= $lowRange);
        }
        else
        {
            $result = 1 if (($value >= $lowRange) && ($value <= $highRange));
        }
    }

    return $result;
}

sub zeroPad($;$)
{
    my ($value, $n) = @_;
    $value = NIL if (!defined $value);
    $n     = 2  if (!defined $n);

    $value = trim($value);
    while (length($value) < $n)
    {
        $value = '0' . $value;
    }

    return $value;
}

sub equalsIgnoreCase($$)
{
    my ($s1, $s2) = @_;

    return (lc($s1) eq lc($s2));
}

sub left($$;$)
{
    my ($tag, $text, $ignorecase) = @_;
    $ignorecase = 0 if (!defined $ignorecase);

    my $value = $text;
    if ($ignorecase)
    {
        $tag = lc($tag);
        $text = lc($text);
    }
    my $loc = index($text, $tag);
    if ($loc != -1)
    {
        $value = substr($value, 0, $loc);
        $value = trim($value);
        $value =~ s/\s+/ /g;
    }

    return $value;
}

sub right($$;$)
{
    my ($tag, $text, $ignorecase) = @_;
    $ignorecase = 0 if (!defined $ignorecase);

    my $value = $text;
    if ($ignorecase)
    {
        $tag = lc($tag);
        $text = lc($text);
    }
    my $loc = index($text, $tag);
    if ($loc != -1)
    {
        $value = substr($value, $loc + length($tag));
        $value = trim($value);
        $value =~ s/\s+/ /g;
    }

    return $value;
}

sub trunc($;$)
{
    my ($value,$len) = @_;
    $value = NIL if (!defined $value);
    $len = 2000 if (!defined $len);

    $value = trim($value);
    $value =~ s/\s+/ /g;
    if (length($value) > $len)
    {
        $value = substr($value, 0, $len-3) . '...';
    }

    return $value;
}

1;

# End of the Program
