# $Header: Ptdpm9.pm 03-may-2005.08:09:10 mgoodric Exp $
#
# Package : Ptdpm9.pm
#
# Copyright (c) 2002, 2005, Oracle. All rights reserved.  
#
#   NAME
#       Ptdpm9.pm
#
#   DESCRIPTION
#       Collects OS Configuration Properties and Parameters
#
#   NOTES
#
#   MODIFIED     (MM/DD/YY)
#      mgoodric   05/03/05 - add fs.aio-pinned to exclusions 
#      mgoodric   09/27/04 - code cleanup 
#      mgoodric   09/23/04 - Add fs.quota.syncs to property exclusion list
#      mgoodric   04/06/04 - Linux cleanup
#      mgoodric   03/19/04 - Fix rounding to 2 decimals
#      mgoodric   01/30/04 - fix parsing for "nameserver"
#      mgoodric   06/16/03 - fix converting Solaris timezone to JAVA timezones
#      mgoodric   05/11/03 - added -debug tracing
#      mgoodric   04/15/03 - fix using uname for arch
#      mgoodric   04/07/03 - fix finding MAC address
#      rlal       02/25/03 - EM 4.0 changed for Linux
#      goramamo   08/16/02 - 904 Changes
#      xxu        06/25/02 - remove /usr/local/bin/perl
#      vkhizder   06/20/02 - vkhizder_compare_fix_2420795_collection_script_changes
#      goramamo   06/12/02 - Added more parm collection
#      goramamo   06/02/02 - Moved Boot Parm Collection from Ptdpm7.pm
#      goramamo   06/02/02 - Creation
#
#*************************************************************

use strict;

package Ptdpm9;

use Carp;
require 5.005;

#require 5.6.1;
require Exporter;

use Ptdpm0;
use Ptdpm2;
use Ptdpm3;

#******************************************
#   Export Subroutines
#******************************************

@Ptdpm9::ISA    = ('Exporter');
@Ptdpm9::EXPORT = qw(
  &getOSConfiguration
  &getMaxSwapSpace
  );

#******************************************
#     Global Variables
#******************************************

use constant NIL => '';

%Ptdpm9::OSConf = ();
@Ptdpm9::PropertyExcludes = ('random', '-nr', '-state',
                             'fs.quota.syncs', 'fs.aio-pinned');

#******************************************
#     Exported Subroutines
#******************************************

sub getOSConfiguration
{
    debugTrace('Calling... getOSConfiguration()');

    getCurrSysDefn();
    getSysConfig();
    getDNSNameServers();

    return %Ptdpm9::OSConf;
}

#******************************************
#     Non-exported Subroutines
#******************************************

sub getCurrSysDefn
{
    debugTrace('Calling... getCurrSysDefn()');

    my $ipcs = getIPCSStatus();

    my @parm = getCurrSysDef();
    my $size     = scalar(@parm);
    my $i        = 0;
    my $j        = 0;
    my @parms    = ();
    my @value    = ();
    my $procflag = 0;
    my $maxswap  = getMaxSwapSpace();
    my $exclude  = NIL;

    LOOP: for ($i = 0 ; $i < $size ; $i++)
    {
        foreach $exclude (@Ptdpm9::PropertyExcludes)
        {
            next LOOP if ($parm[$i] =~ m/$exclude/);
        }
        ($parms[$j], $value[$j]) = split ('=', $parm[$i], 2);
        $j++;
    }

    $parms[$j]   = "Maximum Swap Space (MB)";
    $value[$j++] = sprintf("%.2f", $maxswap / (1024 * 1024));

    loadKernelParmsToHash($Ptdpm0::SYSCTL, \@parms, \@value);
}

sub getSysConfig
{
    debugTrace('Calling... getSysConfig()');

    my @parmlist = ('OPEN_MAX', 'CHILD_MAX');

    #  If you want to collect all the params under getconf,
    #  uncommnet next line;  comment previous line.
    #  my @parmlist  = ('-a');

    my $i     = 0;
    my $j     = 0;
    my $k     = 0;
    my $rest1 = NIL;
    my $rest2 = NIL;
    my @value = ();
    my @parms = ();
    my @sdef  = getSysConf();

    my $size2 = scalar(@sdef);
    my $size1 = scalar(@parmlist);

    if (($size1 == 1) && ($parmlist[0] eq '-a'))
    {
        for ($i = 0 ; $i < $size2 ; $i++)
        {
            ($parms[$i], $value[$i]) = split ($Ptdpm2::SEP, $sdef[$i], 2);
        }
    }
    else
    {
        for ($j = 0 ; $j < $size1 ; $j++)
        {
            for ($i = 0 ; $i < $size2 ; $i++)
            {
                ($rest1, $rest2) = split ($Ptdpm2::SEP, $sdef[$i], 2);
                if (defined $rest1)
                {
                    $rest1 = trim($rest1);
                    if ($rest1 eq $parmlist[$j])
                    {
                        $parms[$k] = $rest1;
                        $value[$k++] = $rest2;
                        last;
                    }
                }
            }
        }
    }

    loadKernelParmsToHash($Ptdpm0::GETCONF, \@parms, \@value);
}

sub loadKernelParmsToHash($$$)
{
    my ($source, $parms, $value) = @_;

    my $size1 = 0;
    $size1 = scalar(@$parms);
    my $size2 = 0;
    $size2 = scalar(@$value);
    my $i = 0;
    my $j = 0;

    for ($i = 0 ; $i < $size1 ; $i++)
    {
        if (!defined @$parms[$i] || @$parms[$i] eq NIL) { next; }
        $Ptdpm9::OSConf{$source}{@$parms[$i]}{$Ptdpm2::TYPE_TAG}  = $source;
        $Ptdpm9::OSConf{$source}{@$parms[$i]}{$Ptdpm2::NAME_TAG}  = @$parms[$i];
        $Ptdpm9::OSConf{$source}{@$parms[$i]}{$Ptdpm2::VALUE_TAG} = trim(@$value[$j]);
        if (index($source, $Ptdpm0::NDD) > -1)
        {
            $j++;
        }
        $j++;
    }
}

sub getMaxSwapSpace
{
    debugTrace('Calling... getMaxSwapSpace()');

    #Function: Calculate maximum swap space

    my $swapspace = 0;
    my @swap   = getSwapSpaceInfo();
    $swapspace = $swap[0] * 1024;

    return $swapspace;
}

sub getDNSNameServers
{
    debugTrace('Calling... getDNSNameServers()');

    my $file  = getResolvConf();
    my @parms = ();
    my @value = ();
    my @val   = ();
    my $i     = 0;

    if ($file ne NIL)
    {
        debugTrace('Opening... ' . $file);
        open(ES, "$file");
        while (<ES>)
        {
            if ($_ =~ m/^\s*#/) { next; }
            if ($_ =~ m/^\s*nameserver\s/)
            {
                @val = split (' ', $_, 2);
                $value[$i++] = $val[1];
                $parms[$i] = "nameserver $i";
            }
        }
        close(ES);

        loadKernelParmsToHash($Ptdpm0::ETCRESOLVCONFFILE, \@parms, \@value);
    }
}

1;

# End of the Program
