# 
# RpasDomainInfo.pm
# 
# Copyright (c) 2007, 2008, Oracle. All rights reserved.  
#
#    NAME
#      RpasDomainInfo.pm - Subroutines to return information about 
#      an Rpas Domain
#
#    DESCRIPTION
#    This file contains subroutines to fetch information about Rpas
#     domains.  All functions will require RPAS_HOME to be set and 
#     the path  to the domain will need to be passed in order to work.
#

use strict;
use POSIX;
package RpasDomainInfo;

sub new
{
   my $class = shift;
   my $self = {};
   bless($self, $class);
   return $self;
}

# Set up the RPAS environment including RPAS_HOME and the lib paths.
sub setupEnvironment
{
   my @envvars = ('RPAS_HOME', 'LL_ORACLE_HOME', 'ORACLE_HOME');
   foreach my $var (@envvars)
   {
      if( (exists $ENV{$var}) && (defined $ENV{$var}) )
      {
         $ENV{RPAS_HOME} = $ENV{$var};
         last;
      }
   }
   die "One or more of the following variables must point to the home directory: @{envvars}\n"
      unless( (exists $ENV{RPAS_HOME}) && (defined $ENV{RPAS_HOME}) );

   # The lib path is OS dependent.
   my $libPath = $ENV{RPAS_HOME} . "/lib";
   my ($system, $node, $release, $version, $machine) = POSIX::uname();
   if (($system eq "SunOS") or ($system eq "Linux") or ($system eq "HP-UX"))
   {
      if( exists $ENV{LD_LIBRARY_PATH} )
      {
         $ENV{LD_LIBRARY_PATH} = $ENV{LD_LIBRARY_PATH} . ":" . $libPath;
      }
      else
      {
         $ENV{LD_LIBRARY_PATH} = $libPath;
      }
   }
   elsif ($system eq "AIX")
   {
      if( exists $ENV{LIBPATH} )
      {
         $ENV{LIBPATH} = $ENV{LIBPATH} . ":" . $libPath;
      }
      else
      {
         $ENV{LIBPATH} = $libPath;
      }
   }
   elsif (($system eq "Windows_NT") or ($system eq "Windows NT"))
   {
      if( exists $ENV{LIB} )
      {
         $ENV{LIB} = $ENV{LIB} . ";" . $libPath;
      }
      else
      {
         $ENV{LIB} = $libPath;
      }
   }
   else
   {
      return 1; 
   }
   return 0;
}

# Return the domains of a given type.  If there is no type provided, or the type is invalid, simply return all domains.
sub getDomains {
   my $rpasHome = shift;
   die "RPAS_HOME not provided\n" unless $rpasHome;
   my $type = shift;
   if( ($type ne 'Simple') && ($type ne 'Global') ) {
      $type = "All";
   }
   
   # Pull in the list of domains.
   # Can't open the file?  No domains returned
   exit if (! -f "${rpasHome}/bin/.private/listDomains"); 
 
   open DOMAINS, "${rpasHome}/bin/.private/listDomains |";
      my @domains = <DOMAINS>;
   close DOMAINS;

   my @retval;
   foreach my $domain (@domains) {
      chomp $domain;
      # If we are in Windows, replace all \ with / in the path.
      my ($system, $node, $version, $release, $machine) = POSIX::uname();
      if($system eq 'Windows_NT') {
         $domain =~ s/\\/\//g;
      }
      
      if( ($type eq "All") || (RpasDomainInfo::getType($domain) eq $type) ) {
         push(@retval, $domain);
      }
   }
   
   return @retval;
}

# Return the type of a domain.  Output will be either: Simple, Global, or Subdomain.
sub getType
{
   my $domain = shift;
   die "The domain path must be passed in.\n" unless $domain;

   # The output of -type should simply be "Simple", "Global", and "Subdomain".
   my @info = getDomainInfo($domain, "type");

   return $info[0];
}

# Returns an array containing all the subdomain paths under a global domain.
sub getSubdomainsShell {
   my $domain = shift;
   die "The domain path must be passed in.\n" unless $domain;

   # The output of -shellsubdomains lists each subdomain path on a separate line.
   my @subdomains = getDomainInfo($domain, "shellsubdomains");

   # Make sure there are no backslashes in the path name.
   foreach my $subdomain (@subdomains)
   {
      $subdomain =~ s/\\/\//g;
   }
   return @subdomains;
}

# Returns the Partition Dimension, Subdomains, and Partition Positions for those domains for any
# given Global domain.  The return value is a hash that contains the following:
# {dim} - Partition dimension
# {subdomains} - An array of hashes containing information on each sub domain.
#                          These hashes are of the format:
#                          {path} - Path to the subdomain
#                          {pos} - Partition position for this domain
#
sub getSubdomainsFull {
   my $domain = shift;
   die "The domain path must be passed in.\n" unless $domain;

   # The output of -listsubdomains is in the following format:
   # Partition Dim: <dimension>
   # <path to subdomain> <partition position>
   # <path to subdomain> <partition position>
   # ...
   # We are going to create a hash to hold all these values.
   my @info = getDomainInfo($domain, "listsubdomains");
   my %retval;
   
   # Strip off the partition dimension and store it in the hash.
   my ($label, $dim) = split(/:/, $info[0]);
   $dim =~ s/^\s*(.*?)\s*$/$1/g;
   $retval{ 'dim' } = $dim;
   shift(@info);
   
   # Now we just have the subdomains.  The path and position are separated by whitespace.
   # We're going to store them in a new hash.
   my @subdomains;
   foreach my $line (@info) {
      my ($path, $pos) = split ' ', $line, 2;

      # Make sure there are no baskslashes in the path name.
      $path =~ s/\\/\//g;
      
      # Stuff the information into a hash and store it in the array.
      my %subdomain;
      $subdomain{ 'path' } = $path;
      $subdomain{ 'pos'  } = $pos;
      push( @subdomains, \%subdomain );
   }
   $retval{ 'subdomains' } = \@subdomains;
   return %retval;
}

# Returns the domain size information.  The return value is a hash in the followint format:
# {'Directories'} - Number of directories used.
# {'Files'}             - Number of files used.
# {'File Size'}      - Accumulated size of all files.
# {'Databases'}    - Number of databases used.
# {'Arrays'}          - Number of arrays used.
# {'Array Size'}   - Accumulated size of all arrays.
#
sub getDomainSize {
   my $domain = shift;
   die "The domain path must be passed in.\n" unless $domain;

   # The output of domainsize is in the following format:
   # Directories: <numDirectories>
   # Files: <numFiles>
   # File Size: <numBytes>
   # Databases: <numDatabases>
   # Arrays: <numArrays>
   # Array Size: <numBytes>
   my @info = getDomainInfo($domain, "domainsize");
   my %retval;

   # Split the data on the colon.  The head will be the label and the tail will be the value.  Trim the whitespace after each
   # result.
   foreach my $line (@info) {
      my ($label, $value) = split(/:/, $line, 2);
      $label =~ s/^\s*(.*?)\s*$/$1/g;
      $value =~ s/^\s*(.*?)\s*$/$1/g;
      $retval{ $label } = $value;
   }
   
   return %retval;
}

# Gets the domain history.  Output is an array of hashes containing the revision information in the following format:
# {version} - Version of the domain after the action took place.
# {action}   - The action performed on the domain.
# {date}       - The date that the action occurred.
#
sub getHistory
{
   my $domain = shift;
   die "The domain path must be passed in.\n" unless $domain;

   # The output of history is in the following format:
   # {version}:{action}:{date}
   my @info = getDomainInfo($domain, "history");
   my @retval;

   # Split the data on the colon.  The head will be the label and the tail will be the value.  Trim the whitespace after each
   # result.
   foreach my $line (@info) {
      my ($version, $action, $date) = split(/:/, $line, 3);   
      my %history;
      $history{ 'version' } = $version;
      $history{ 'action'  } = $action;
      $history{ 'date'    } = $date;
      push (@retval, \%history);
   }
   return @retval;
}

# Returns the domain information of a subdomain's master.  The return value is a hash in the following format:
# {'Master'}            - Path to master.
# {'Partition Dim'} - Partition Dimension.
# {'Partition Pos'}  - Partition Position.
#
sub getMasterInfo {
   my $domain = shift;
   die "The domain path must be passed in.\n" unless $domain;

   # The output of masterdomaininfo is in the following format:
   # Master: <path to master>
   # Partition Dim: <partition dimension>
   # Partition Pos: <partition position>
   my @info = getDomainInfo($domain, "masterdomaininfo");
   my %retval;

   # Split the data on the colon.  The head will be the label and the tail will be the value.  Trim the whitespace after each
   # result.
   foreach my $line (@info) {
      my ($label, $value) = split(/:/, $line, 2);
      $label =~ s/^\s*(.*?)\s*$/$1/g;
      $value =~ s/^\s*(.*?)\s*$/$1/g;
      $retval{ $label } = $value;
   }   
   return %retval;
}

# Gets the terse domain information for one or more options and strips blank lines from output.
sub getDomainInfo {
   my ($domain, $option) = @_;
   die "RPAS_HOME must be set.\n" unless $ENV{RPAS_HOME};
   die "usage: getDomainInfo(domain, option)\n" unless ($domain and $option);

   open IN, "$ENV{RPAS_HOME}/bin/domaininfo -d ${domain} -${option} -terse |";
   my @lines = <IN>;
   close IN;

   my @retval;   
   foreach my $line (@lines) {
      chomp $line;
      if ($line) {
         push(@retval, ${line});
      }
   }
   return @retval;
}

1;
