
# +===========================================================================+
# |   Copyright (c) 2003 Oracle Corporation, Redwood Shores, California, USA
# |                         All Rights Reserved
# |                        Applications Division
# +===========================================================================+
# |
# | FILENAME
# |   ARGS.pm
# |
# | DESCRIPTION
# |	 TXK ARGS package
# |
# | USAGE
# |       See ARGS.html
# |
# | PLATFORM
# |
# | NOTES
# |
# +===========================================================================+

# $Header: ARGS.pm 03-aug-2005.08:54:52 ndutko Exp $

package TXK::ARGS;

@ISA = qw( TXK::Common );

######################################
# Standard Modules
######################################

use strict;
use English;

require 5.005;

######################################
# Package Specific Modules
######################################

use TXK::Error();
use TXK::Util();
use TXK::FileSys();
use TXK::XML();

######################################
# Public Constants
######################################

use constant NAME        	=> 'name';
use constant DESC           	=> 'desc';
use constant TYPE           	=> 'type';
use constant ALLOWABLE_VALUES   => 'allowable_values';
use constant REQUIRED	    	=> 'required';
use constant PROMPT	    	=> 'prompt';
use constant CONFIRM  		=> 'confirm';
use constant DEFAULT   	        => 'default';
use constant CONDITION 	        => 'condition';
use constant INNER		=> 'inner';
use constant SEQUENCE		=> 'seq';
use constant YES		=> 'Yes';
use constant NO			=> 'No';

use constant STRING		=> 'String';
use constant NUMBER		=> 'Number';
use constant LIST		=> 'List';
use constant YESNO		=> 'YesNo';
use constant NON_EMPTY	        => '-AnyString-';
use constant PREFIX	        => '-';

use constant HASH_TABLE		=> 'hashTable';
use constant XML_STRING	        => 'xmlString';
use constant XML_FILE		=> 'xmlFile';

######################################
# Package Variables 
######################################

my $PACKAGE_ID = "TXK::ARGS";

######################################
# Object Keys
######################################

my $ARGS_ARRAY			 = "args";
my $ARGS_DEFN			 = "definition";
my $ARGS_DEFN_TYPE		 = "type";
my $ARGS_DATA			 = "data";
my $ARGS_UNDEF_DATA		 = "undefinedData";
my $ARGS_USE_PROMPT		 = "useprompt";
my $ARGS_ALLOW_UNDEF		 = "allowUndefinedArgs";
my $ARGS_XML			 = "xmlObj";
my $ARGS_COMMON_DEFN	         = "commonDefn";

######################################
#
# Object Structure
# ----------------
#
#  Hash Array
#
######################################

######################################
# Package Methods 
#
# Public
#
#	new 	- build empty object
#
######################################

sub new;
sub DESTROY;
sub getArgValue;
sub getArgTable;
sub validateArgs;
sub printUsage;

######################################
# Package Methods
# 
# Private
#       All private methods are marked with a leading underscore.
#
######################################

sub _buildCommonDefinition;
sub _convertXMLToHashFormat;
sub _doConvertXMLToHashFormat;
sub _doBuildCommonDefinition;
sub _printPublicFormat;

######################################
# Constructor
######################################

sub new {
  my $type = $ARG[0];

  my $self = TXK::Common->new();

  bless $self, $PACKAGE_ID ;

  my $key;

  my %INIT_OBJ = (
  		  PACKAGE_IDENT   	      => $PACKAGE_ID,
  	          $ARGS_ARRAY		      => [ ],
		  $ARGS_DEFN		      => { },
		  $ARGS_DEFN_TYPE	      => TXK::ARGS::HASH_TABLE,
                  $ARGS_DATA		      => { },
	          $ARGS_UNDEF_DATA	      => { },
 	 	  $ARGS_USE_PROMPT	      => TXK::Util::TRUE,
 	 	  $ARGS_ALLOW_UNDEF             => TXK::Util::FALSE,
	          $ARGS_XML		      => TXK::XML->new(),
		  $ARGS_COMMON_DEFN	      => [ ],
                 );

  foreach $key (keys %INIT_OBJ)
   {
     $self->{$key} = $INIT_OBJ{$key};
   }

  return $self;
}

######################################
# Destructor
######################################

sub DESTROY
{
}

######################################
# validateArgs
######################################

sub validateArgs
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  TXK::Util->isValidArgs({args=>$args,reqd=>["$ARGS_ARRAY","$ARGS_DEFN"]});

  $self->{$ARGS_ARRAY} = TXK::Util->getArrayRef
					    ($ARGS_ARRAY,$args->{$ARGS_ARRAY});

  $self->{$ARGS_USE_PROMPT} = TXK::Util::TRUE;
  $self->{$ARGS_ALLOW_UNDEF} = TXK::Util::FALSE;
  $self->{$ARGS_DEFN_TYPE} = TXK::ARGS::HASH_TABLE;

  my $key;

  foreach $key ("$ARGS_USE_PROMPT","$ARGS_ALLOW_UNDEF")
   {
     $self->{$key} = ( exists $args->{$key}
                          ? TXK::Util->getBooleanArg($key,$args->{$key})
                          : $self->{$key}
                     );
   }

  if ( exists $args->{$ARGS_DEFN_TYPE} )
   {
     return $self->setError("Invalid <${ARGS_DEFN_TYPE}> - must be one of : " .
                            HASH_TABLE . "," . XML_STRING . "," . XML_FILE)
            unless ( $args->{$ARGS_DEFN_TYPE} eq TXK::ARGS::HASH_TABLE ||
                     $args->{$ARGS_DEFN_TYPE} eq TXK::ARGS::XML_STRING ||
                     $args->{$ARGS_DEFN_TYPE} eq TXK::ARGS::XML_FILE );

     $self->{$ARGS_DEFN_TYPE} = $args->{$ARGS_DEFN_TYPE};
   }

  $self->{$ARGS_DEFN} = 
      ( $self->{$ARGS_DEFN_TYPE} eq TXK::ARGS::HASH_TABLE 
          ? TXK::Util->getHashRef($ARGS_DEFN,$args->{$ARGS_DEFN})
          : TXK::Util->getScalarArg($ARGS_DEFN,$args->{$ARGS_DEFN})
      );

#	Load the Args into an XML object

  my $fsys = TXK::FileSys->new();

  if ( $self->{$ARGS_DEFN_TYPE} eq TXK::ARGS::XML_FILE ) 
   {
     $fsys->access({ fileName=>$self->{$ARGS_DEFN},
                     type=>TXK::FileSys::FILE,
                     checkMode=>TXK::FileSys::READ_ACCESS,
                   })
         or return $self->setError("Unable to access XML definition file " .
                                           $self->{$ARGS_DEFN});

     $self->{$ARGS_XML}->loadDocument({ file => $self->{$ARGS_DEFN} });

   }
  elsif ( $self->{$ARGS_DEFN_TYPE} eq TXK::ARGS::XML_STRING )
   {
     $self->{$ARGS_XML}->loadString($self->{$ARGS_DEFN});
   }
  else
   {
   }
   
  _buildCommonDefinition($self) or return TXK::Error::FAIL;

#
#	Parse the ARGS data and copy to the undefined table.
#

  $self->{$ARGS_DATA} = { };
  $self->{$ARGS_UNDEF_DATA} = { };

  my $prefix = TXK::ARGS::PREFIX();
  my ($data,$key,$value);

#
#	Handle the special case of -?. If there is just one arg, -?, then
#	printUsage and stop.
#

  if ( scalar(@{$self->{$ARGS_ARRAY}}) == 1 &&
       $self->{$ARGS_ARRAY}->[0] =~ m/^${prefix}\?$/ )
   {
     $self->printUsage();
     TXK::Error->stop();
   }

  for $data ( @{$self->{$ARGS_ARRAY}} )
   {
     next unless ( defined $data && $data ne "" );

     next unless $data =~ m/^${prefix}(\w+)=(.*)$/ ;

     $self->{$ARGS_DATA}->{$1} = $2;
     $self->{$ARGS_UNDEF_DATA}->{$1} = $2;
   }

  _doValidateArgs($self,$self->{$ARGS_DATA},$self->{$ARGS_COMMON_DEFN})
         or return TXK::Error::FAIL;

   my $undef_args = undef;

   foreach $key (keys %{$self->{$ARGS_UNDEF_DATA}})
    {
      $undef_args .= "$key ";
    }

   return $self->setStop("Undefined args are not allowed. The following " .
                         " args are not required : " . $undef_args )
          if ( !$self->{$ARGS_ALLOW_UNDEF} && $undef_args );

  return TXK::Error::SUCCESS;
}

######################################
# getArgValue
######################################

sub getArgValue
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return $self->{$ARGS_DATA}->{$args} 
         if ( defined $args && !ref($args) &&
              exists $self->{$ARGS_DATA}->{$args} );

  return undef;
}

######################################
# getArgTable
######################################

sub getArgTable
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return $self->{$ARGS_DATA};
}

######################################
# printUsage
######################################

sub printUsage
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  print "\nUsage: \n\n";

  _doPrintUsage($self->{$ARGS_COMMON_DEFN},0);

}

######################################
# End of Public methods
######################################

# ==========================================================================

sub _doValidateArgs
{
  my $self  = $ARG[0];
  my $argdata=$ARG[1];
  my $argdefn=$ARG[2];

  my ($fmt,$name);

  foreach $fmt (@$argdefn) 
   {
     $name = $fmt->{TXK::ARGS::NAME()};

#	Remove from the undefined args table.

     delete $self->{$ARGS_UNDEF_DATA}->{$name};

     if ( exists($fmt->{TXK::ARGS::DEFAULT()})  &&
          defined $fmt->{TXK::ARGS::DEFAULT()} ) 
      {
          $argdata->{$name} = $fmt->{TXK::ARGS::DEFAULT()}
            unless (exists($argdata->{$name}) && _isArgData($argdata->{$name}));
      }

     my $validateError = TXK::Util::FALSE;

     for ( ;; )
      {
	  # Need to prompt if either required and data is empty OR
	  # validation error. Previously we only did this for required 
	  # which meant that invalid values for optional fields caused
	  # the code to loop.

        if ( 
             ( 
               ( uc($fmt->{TXK::ARGS::REQUIRED()}) ne uc(TXK::ARGS::NO()) &&
                 !_isArgData($argdata->{$name})
               )
              || $validateError 
             )
            && $self->{$ARGS_USE_PROMPT} 
          )
         {
           my $input;
           my $prompt = ( $fmt->{TXK::ARGS::PROMPT()} &&
                          $fmt->{TXK::ARGS::PROMPT()} ne "" 
                              ? $fmt->{TXK::ARGS::PROMPT()} 
                              : ( $fmt->{TXK::ARGS::DESC()} &&
                                  $fmt->{TXK::ARGS::DESC()} ne ""
                                  ? "Enter " . $fmt->{TXK::ARGS::DESC()}
                                  : "Enter " . $name 
                                )
                        );

           print "$prompt ? ";

           $input = <STDIN>;

#	   Need to deal with special case where STDIN comes from a file.
#	   If STDIN is undefined then we know we must be using redirection.
#	   This is most likely to happen with RTs.

           return $self->setStop("ERROR processing <arg> $name : End of STDIN")
           			unless ( defined $input );
           
           chomp($input);

           $argdata->{$name} = $input;
         }

        $argdata->{$name} = ""  unless ( _isArgData($argdata->{$name}) );

	# Since we now loop for optional fields, reapply defaults if
	# data is empty. 

        if ( exists($fmt->{TXK::ARGS::DEFAULT()})  &&
             defined $fmt->{TXK::ARGS::DEFAULT()} ) 
         {
             $argdata->{$name} = $fmt->{TXK::ARGS::DEFAULT()}
                     unless ( exists($argdata->{$name}) && 
                                      _isArgData($argdata->{$name}) );
         }

	# Optional String/Number fields can be null. All others must
	# match specified type - YESNO, LIST etc.

        last if ( uc($fmt->{TXK::ARGS::REQUIRED()}) eq uc(TXK::ARGS::NO()) &&
                  !_isArgData($argdata->{$name}) &&
                  ( !$fmt->{TXK::ARGS::TYPE()} ||
                    uc($fmt->{TXK::ARGS::TYPE()}) eq uc(TXK::ARGS::NUMBER()) ||
                    uc($fmt->{TXK::ARGS::TYPE()}) eq uc(TXK::ARGS::STRING()) 
                  ) );

        my $errmsg = undef;

        if (  !$fmt->{TXK::ARGS::TYPE()} )
         {
           $errmsg = "Argument value cannot be an empty string"
                          unless ( _isArgData($argdata->{$name}) );
         }
        elsif (  uc($fmt->{TXK::ARGS::TYPE()}) eq uc(TXK::ARGS::YESNO()) )
         {
           $errmsg = "Argument value must be either Yes or No"
                    unless ( uc($argdata->{$name}) eq uc(TXK::ARGS::YES()) ||
                             uc($argdata->{$name}) eq uc(TXK::ARGS::NO()) );
         }
        elsif ( uc($fmt->{TXK::ARGS::TYPE()}) eq uc(TXK::ARGS::NUMBER()) )
         {
           $errmsg = "Argument value must be a number"
                     unless ( $argdata->{$name} =~ m/^-?\d+\.?\d*$/ );
         }
        elsif ( uc($fmt->{TXK::ARGS::TYPE()}) eq uc(TXK::ARGS::STRING()) )
         {
           $errmsg = "Argument value cannot be an empty string"
                          unless ( _isArgData($argdata->{$name}) );
         }
        elsif ( uc($fmt->{TXK::ARGS::TYPE()}) eq uc(TXK::ARGS::LIST()) )
         {
           $errmsg = "Argument value cannot be an empty string"
                       unless ( _isArgData($argdata->{$name}) );

           if ( exists($fmt->{TXK::ARGS::ALLOWABLE_VALUES()}) )
            {
              my @list = split(/,/,$fmt->{TXK::ARGS::ALLOWABLE_VALUES()});
              my $listkey;
              my $found = TXK::Util::FALSE;

              foreach $listkey (@list)
               {
                 $found=TXK::Util::TRUE,last 
                         if ( $listkey eq $argdata->{$name} );
               }

              $errmsg = ( $found ? undef
                                 : "Argument value not in list of " .
                                   "allowable values (" . join(',',@list) . ")" ) ;
            }
         }

        print "ERROR processing <arg> $name : $errmsg\n" if defined $errmsg;

        return $self->setStop("ERROR processing <arg> $name : $errmsg")
                  if ( !$self->{$ARGS_USE_PROMPT} && $errmsg );

        $validateError=TXK::Util::TRUE, next if $errmsg;

        if ( uc($fmt->{TXK::ARGS::CONFIRM()}) eq uc(TXK::ARGS::YES())  
                      && $self->{$ARGS_USE_PROMPT} )
         {
           my $input;
           my $prompt = "Confirm <${name}> value " . $argdata->{$name} .
                        " : Y/N ? ";

           print $prompt;

           chomp($input = <STDIN>);

#          Need to deal with special case where STDIN comes from a file.
#          If STDIN is undefined then we know we must be using redirection.
#          This is most likely to happen with RTs.

           return $self->setStop("ERROR confirming <arg> $name : End of STDIN")
                               unless ( defined $input );

           $validateError=TXK::Util::TRUE, next 
			  unless (uc($input) eq "Y");
         }
     
        last;
      }

     if ( exists($fmt->{TXK::ARGS::CONDITION()})  && 
          ref($fmt->{TXK::ARGS::INNER()}) eq "ARRAY" )
      {
        _doValidateArgs($self,$argdata,$fmt->{TXK::ARGS::INNER()})
                      or return TXK::Error::FAIL
            if ( ( uc($fmt->{TXK::ARGS::CONDITION()}) 
                              eq uc(TXK::ARGS::NON_EMPTY()) &&
                   _isArgData($argdata->{$name})
                 ) 
                || !_isArgData($fmt->{TXK::ARGS::CONDITION()})
                || $fmt->{TXK::ARGS::CONDITION()} eq $argdata->{$name} 
                || ( uc($fmt->{TXK::ARGS::CONDITION()}) eq uc($argdata->{$name})
                         && 
                     uc($fmt->{TXK::ARGS::TYPE()}) eq uc(TXK::ARGS::YESNO())
                   ) );
      }
   }

  return TXK::Error::SUCCESS;
}

# ==========================================================================

sub _buildCommonDefinition
{
  my $self  = $ARG[0];

  my ($pubfmt);

  if ( $self->{$ARGS_DEFN_TYPE} eq TXK::ARGS::HASH_TABLE )
   {
     $pubfmt = $self->{$ARGS_DEFN};
   }
  else
   { 
     $pubfmt = {};
     _convertXMLToHashFormat($self,$pubfmt) or return TXK::Error::FAIL;
   }

#  print "Print PublicFormat \n";
#  _printPublicFormat($pubfmt,1);

#
#	We need to build an array based common format to ensure
#	we process args in the correct order, either by sequence or by
#	name.
#

  $self->{$ARGS_COMMON_DEFN} = [ ];

  _doBuildCommonDefinition($self,$pubfmt,$self->{$ARGS_COMMON_DEFN}) 
			or return TXK::Error::FAIL;


#  print "Print CommonFormat \n";
#  _printCommonFormat($self->{$ARGS_COMMON_DEFN},1);

  return TXK::Error::SUCCESS;
}

# ==========================================================================

sub _doBuildCommonDefinition
{
  my $self  = $ARG[0];
  my $pubfmt= $ARG[1];
  my $comfmt= $ARG[2];

  my $key;

  foreach $key ( keys %$pubfmt )
   {
     return $self->setError(
            "Missing hash table for key $key found when building common definition")
            unless ( ref($pubfmt->{$key}) eq "HASH" );
   }

  foreach $key ( sort { my ($lhs_seq,$rhs_seq);
                        $lhs_seq = $pubfmt->{$a}->{TXK::ARGS::SEQUENCE()};
                        $rhs_seq = $pubfmt->{$b}->{TXK::ARGS::SEQUENCE()};

                        $lhs_seq = "999999" unless defined $lhs_seq;
		        $rhs_seq = "999999" unless defined $rhs_seq;

                        $lhs_seq <=> $rhs_seq or $a cmp $b;
                      }
                 ( keys %$pubfmt ) )
   {
     my $entry;

     return $self->setError(
              "Missing <name> entry in <arg> while building common definition")
            unless ( defined $key && $key ne "" );

     $entry =              {
       TXK::ARGS::NAME()    => $key,
       TXK::ARGS::DESC()    => $pubfmt->{$key}->{TXK::ARGS::DESC()},
       TXK::ARGS::TYPE()    => $pubfmt->{$key}->{TXK::ARGS::TYPE()},
       TXK::ARGS::ALLOWABLE_VALUES()
                            => $pubfmt->{$key}->{TXK::ARGS::ALLOWABLE_VALUES()},
       TXK::ARGS::REQUIRED()=> $pubfmt->{$key}->{TXK::ARGS::REQUIRED()},
       TXK::ARGS::PROMPT()  => $pubfmt->{$key}->{TXK::ARGS::PROMPT()},
       TXK::ARGS::CONFIRM() => $pubfmt->{$key}->{TXK::ARGS::CONFIRM()},
       TXK::ARGS::DEFAULT() => $pubfmt->{$key}->{TXK::ARGS::DEFAULT()},
       TXK::ARGS::CONDITION()=>$pubfmt->{$key}->{TXK::ARGS::CONDITION()},
       TXK::ARGS::SEQUENCE()=> $pubfmt->{$key}->{TXK::ARGS::SEQUENCE()},
       TXK::ARGS::INNER()   => undef,
                           };

     push @$comfmt, $entry;

     if ( ref($pubfmt->{$key}->{TXK::ARGS::INNER()}) eq "HASH" )
      {
        $entry->{TXK::ARGS::INNER()} = [ ];

        _doBuildCommonDefinition($self,$pubfmt->{$key}->{TXK::ARGS::INNER()},
                                 $entry->{TXK::ARGS::INNER()})
              or return TXK::Error::FAIL;
      }
   }

  return TXK::Error::SUCCESS;
}

# ==========================================================================

sub _doConvertXMLToHashFormat
{
  my $self  = $ARG[0];
  my $xml   = $ARG[1];
  my $xmldoc= $ARG[2];
  my $pubfmt= $ARG[3];

  my ($obj,$sequence,$argName);

  $sequence = 0;

  foreach $obj (@$xmldoc)
   {
     next unless ( $obj->{TXK::XML::HEAD()} eq "arg" );

     my $names = $obj->{TXK::XML::NAMES()};

     $sequence ++;

     $argName = $xml->getXMLName($names,TXK::ARGS::NAME());

     return $self->setError(
              "Missing <name> entry in <arg> while processing <ScriptArgs>")
            unless ( defined $argName && $argName ne "" );

     return $self->setError("Name $argName has multiple entries in <arg>")
            if ( exists($pubfmt->{$argName}) );

     $pubfmt->{$argName} = {
       TXK::ARGS::DESC()     => $xml->getXMLName($names,TXK::ARGS::DESC()),
       TXK::ARGS::TYPE()     => $xml->getXMLName($names,TXK::ARGS::TYPE()),
       TXK::ARGS::ALLOWABLE_VALUES()   
                 => $xml->getXMLName($names,TXK::ARGS::ALLOWABLE_VALUES()),
       TXK::ARGS::REQUIRED() => $xml->getXMLName($names,TXK::ARGS::REQUIRED()),
       TXK::ARGS::PROMPT()   => $xml->getXMLName($names,TXK::ARGS::PROMPT()),
       TXK::ARGS::CONFIRM()  => $xml->getXMLName($names,TXK::ARGS::CONFIRM()),
       TXK::ARGS::DEFAULT()  => $xml->getXMLName($names,TXK::ARGS::DEFAULT()),
       TXK::ARGS::CONDITION()=> $xml->getXMLName($names,TXK::ARGS::CONDITION()),
       TXK::ARGS::SEQUENCE() => "$sequence",
       TXK::ARGS::INNER()    => undef,
                           };

     if ( $obj->{TXK::XML::IS_INNER()} )
      {
        $pubfmt->{$argName}->{TXK::ARGS::INNER()} = {};

        _doConvertXMLToHashFormat($self,$xml,$obj->{TXK::XML::BODY()},
                                  $pubfmt->{$argName}->{TXK::ARGS::INNER()})
              or return TXK::Error::FAIL;
      }
   }

  return TXK::Error::SUCCESS;
}

# ==========================================================================

sub _convertXMLToHashFormat
{
  my $self  = $ARG[0];
  my $pubfmt= $ARG[1];

  my $xmldoc = $self->{$ARGS_XML}->getDocument();

  my ($obj,$scriptArg);

  foreach $obj (@$xmldoc)
   {
     $scriptArg = $obj, last
                   if ( $obj->{TXK::XML::HEAD()} eq "ScriptArgs" &&
                        $obj->{TXK::XML::IS_INNER()} );
   }

  return $self->setError("No <ScriptArgs> entry in XML ARGS definition file")
         unless defined $scriptArg;

  _doConvertXMLToHashFormat($self,$self->{$ARGS_XML},
                            $scriptArg->{TXK::XML::BODY()},$pubfmt)
     or return TXK::Error::FAIL;

  return TXK::Error::SUCCESS;
}

# ==========================================================================

sub _printPublicFormat
{
  my $pubfmt= $ARG[0];
  my $level = $ARG[1];

  my $key;

  foreach $key (keys %$pubfmt)
   {
     print ( ("   " x $level) , "Key=$key \n") ;

     my $data = $pubfmt->{$key};
     my $id;

     foreach $id (keys %$data)
      {
        print ( ("   " x $level) , "  Id=$id, Value=$data->{$id} \n");
      }

     _printPublicFormat($data->{TXK::ARGS::INNER()},$level+1)
           if ( ref($data->{TXK::ARGS::INNER()}) eq "HASH" );

   }
}

# ==========================================================================

sub _printCommonFormat
{
  my $comfmt= $ARG[0];
  my $level = $ARG[1];

  my $data;

  foreach $data (@$comfmt)
   {
     print ( ("   " x $level) , "Key=", $data->{TXK::ARGS::NAME()}, "\n") ;

     my $id;

     foreach $id (keys %$data)
      {
        print ( ("   " x $level) , "  Id=$id, Value=$data->{$id} \n");
      }

     _printCommonFormat($data->{TXK::ARGS::INNER()},$level+1)
           if ( ref($data->{TXK::ARGS::INNER()}) eq "ARRAY" );
   }
}

# ==========================================================================

sub _doPrintUsage
{
  my $comfmt= $ARG[0];
  my $level = $ARG[1];

  my $data;

  foreach $data (@$comfmt)
   {
     my $ident = ("  " x $level);

     print $ident,TXK::ARGS::PREFIX, $data->{TXK::ARGS::NAME()},"\n";

     $ident .= "  ";

     my $type = ( $data->{TXK::ARGS::TYPE()} 
                     ? $data->{TXK::ARGS::TYPE()} : TXK::ARGS::STRING );

     my $reqd = ( $data->{TXK::ARGS::REQUIRED()} 
                     ? $data->{TXK::ARGS::REQUIRED()} : TXK::ARGS::YES );

     my $lov  = ( $data->{TXK::ARGS::ALLOWABLE_VALUES()} 
                     ? $data->{TXK::ARGS::ALLOWABLE_VALUES()} : undef );

     my $dflt = ( $data->{TXK::ARGS::DEFAULT()} 
                     ? $data->{TXK::ARGS::DEFAULT()} : undef );

     my $desc = ( $data->{TXK::ARGS::DESC()} 
                     ? $data->{TXK::ARGS::DESC()} : undef );

     my $cond = ( $data->{TXK::ARGS::CONDITION()}
                     ? $data->{TXK::ARGS::CONDITION()} : undef );

     my $inner=  $data->{TXK::ARGS::INNER()};

     print $ident,"Type        = ", $type , "\n";
     print $ident,"Required    = ", $reqd , "\n";
     print $ident,"List_Values = ", $lov  , "\n"    if ( defined $lov );
     print $ident,"Default     = ", $dflt , "\n"    if ( defined $dflt );
     print $ident,"Description = ", $desc , "\n"    if ( defined $desc );
     print $ident,"Condition   = ", $cond , "\n"    if ( defined $cond );

     _doPrintUsage($inner,$level+1) if ( ref($inner) eq "ARRAY" );
   }
}

# ==========================================================================

sub _isArgData
{
 my $data = $ARG[0];

# _isArgData is used to deal with the special case of "0". Testing
# the string will fail if entered value equals "0".

# This test methods returns true ( i.e. arg data entered ) if arg equals "0".
# Otherwise is just returns the arg value, which may be undef ,"" or non-empty.

 return "1" if ( $data eq "0" );

 return $data;
} 

# ==========================================================================

1;


