
# +===========================================================================+
# |   Copyright (c) 2003 Oracle Corporation, Redwood Shores, California, USA
# |                         All Rights Reserved
# |                        Applications Division
# +===========================================================================+
# |
# | FILENAME
# |   AutoBuild.pm
# |
# | DESCRIPTION
# |      TXK AutoBuild package
# |
# | USAGE
# |       See AutoBuild.html
# |
# | PLATFORM
# |
# | NOTES
# |
# +===========================================================================+

# $Header: AutoBuild.pm 03-aug-2005.08:54:53 ndutko Exp $

package TXK::AutoBuild;

@ISA = qw( TXK::Common );

######################################
# Standard Modules
######################################

use strict;
use English;

require 5.005;

######################################
# Package Specific Modules
######################################

use TXK::Error();
use TXK::Util();
use TXK::FileSys();
use TXK::IO();

######################################
# Public Constants
######################################

use constant BASE_NO        => 'base';
use constant PORT_NO        => 'port';
use constant PATCH_LEVEL    => 'level';
use constant PLATFORM       => 'platform';
use constant NAME	    => 'name';
use constant TYPE	    => 'type';
use constant DOWNLOAD	    => 'download';
use constant DB_UPDATE      => 'dbupdate';
use constant FILESYS_UPDATE => 'filesysupdate';
use constant APPROVED       => 'approved';
use constant ACTION	    => 'action';
use constant PATH	    => 'path';
use constant VERSION        => 'version';
use constant COMMENT        => 'comment';
use constant BUILD          => 'build';
use constant SERVER_BASE    => 'serverbase';

######################################
# Package Variables 
######################################

my $PACKAGE_ID = "TXK::AutoBuild";

my @TECHSTACK_TYPE_LIST = ( "8i", "9iR2", "10G", "10g", "iAS", "ias" );

######################################
# Object Keys
######################################

my $XML_OBJ	                 = "xmlObj";
my $DEFINITION_FILE              = "fileName";
my $TECHSTACK_TYPE               = "techStackType";
my $TECHSTACK_BUILD		 = "techStackBuild";
my $TECHSTACK_SERVER_BASE	 = "techStackServerBase";
my $PRODUCT_TYPE                 = "productType";
my $PRODUCT_VERSION              = "productVersion";
my $PRODUCT_LOCATION_PATH        = "productLocation";

my $OUI_PRODUCT_TYPE             = "OUIproductType";
my $OUI_PRODUCT_VERSION          = "OUIproductVersion";
my $OUI_PRODUCT_LOCATION_PATH    = "OUIproductLocation";

my $PATCH_LIST			 = "patchList";
my $PATCHSET_LIST		 = "patchSetList";
my $INSTALL_LIST		 = "installList";

######################################
#
# Object Structure
# ----------------
#
#  Hash Array
#
######################################

######################################
# Package Methods 
#
# Public
#
#	new 	- build empty object
#
######################################

sub new;
sub DESTROY;
sub loadDefinition;
sub isValidTechStack;
sub isOUIIncluded;
sub isProductIncluded;
sub getTechStackType;
sub getTechStackBuild;
sub getTechStackServerBase;
sub getProductType;
sub getProductVersion;
sub getProductLocationPath;
sub getOUIProductType;
sub getOUIProductVersion;
sub getOUIProductLocationPath;
sub getPatchList;
sub getPatchSetList;
sub getInstallList;
sub getRefPatchList;
sub getRefPatchSetList;
sub getRefInstallList;
sub getTechStackInfo;
sub getPatchListInfo;
sub getPatchSetListInfo;
sub getInstallListInfo;
sub getAllPatches;
sub getPatch;

######################################
# Package Methods
# 
# Private
#       All private methods are marked with a leading underscore.
#
######################################

sub _getXMLName;

######################################
# Constructor
######################################

sub new {
  my $type = $ARG[0];

  my $self = TXK::Common->new();

  bless $self, $PACKAGE_ID ;

  my $key;

  my %INIT_OBJ = (
  		   PACKAGE_IDENT   	      => $PACKAGE_ID,
  	           $XML_OBJ		      => undef,
		   $DEFINITION_FILE	      => undef,
                   $TECHSTACK_TYPE	      => "",
                   $TECHSTACK_BUILD	      => "",
                   $TECHSTACK_SERVER_BASE     => "",
                   $PRODUCT_TYPE              => "",
                   $PRODUCT_VERSION           => "",
                   $PRODUCT_LOCATION_PATH     => "",
                   $OUI_PRODUCT_TYPE          => "",
                   $OUI_PRODUCT_VERSION       => "",
                   $OUI_PRODUCT_LOCATION_PATH => "",
		   $PATCH_LIST		      => [ ],
                   $PATCHSET_LIST  	      => [ ],
                   $INSTALL_LIST		      => [ ],
                  );

  foreach $key (keys %INIT_OBJ)
   {
     $self->{$key} = $INIT_OBJ{$key};
   }

  return $self;
}

######################################
# Destructor
######################################

sub DESTROY
{
}

######################################
# loadDefinition
######################################

sub loadDefinition
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  TXK::Util->isValidArgs({args=>$args,reqd=>["$DEFINITION_FILE"]});

  $self->{$DEFINITION_FILE} = TXK::Util->getScalarArg
                                  ($DEFINITION_FILE,$args->{$DEFINITION_FILE});

  $self->{$XML_OBJ}         = TXK::XML->new();

  my $fsys = TXK::FileSys->new();

  $fsys->access({ fileName=>$self->{$DEFINITION_FILE},
                  type=>TXK::FileSys::FILE,
                  checkMode=>TXK::FileSys::READ_ACCESS,
                })
      or return $self->setError("Unable to access XML definition file " .
                                $self->{$DEFINITION_FILE});

  $self->{$XML_OBJ}->loadDocument({ file => $self->{$DEFINITION_FILE} });

  my $xmldoc = $self->{$XML_OBJ}->getDocument();

  my $autobuild;
  my $obj;

  foreach $obj (@$xmldoc)
   {
     $autobuild = $obj, last
                   if ( $obj->{TXK::XML::HEAD()} eq "autobuild" &&
                        $obj->{TXK::XML::IS_INNER()} );
   }

  return $self->setError("No <autobuild> entry in XML definition file")
         unless defined $autobuild;

  my $techstack;

  foreach $obj (@{$autobuild->{TXK::XML::BODY()}})
   {
     $techstack = $obj, last
                   if ( $obj->{TXK::XML::HEAD()} eq "TechStack" &&
                        $obj->{TXK::XML::IS_INNER()} );
   }

  return $self->setError("No <TechStack> entry in XML definition file")
         unless defined $techstack;

  $self->{$TECHSTACK_TYPE} = _getXMLName
				  ($techstack->{TXK::XML::NAMES()},"type");

  return $self->setError(
         "Invalid <TechStack> type $self->{$TECHSTACK_TYPE}" . 
         " : Allowable values = @TECHSTACK_TYPE_LIST " )
                unless($self->isValidTechStack());

  $self->{$TECHSTACK_BUILD} = _getXMLName
			  ($techstack->{TXK::XML::NAMES()},"build");

  return $self->setError("<TechStack> build version is undefined" )
                unless($self->{$TECHSTACK_BUILD});

  $self->{$TECHSTACK_SERVER_BASE} = _getXMLName
                          ($techstack->{TXK::XML::NAMES()},"serverbase");

  return $self->setError("<TechStack> serverbase version is undefined" )
                unless($self->{$TECHSTACK_SERVER_BASE});

  my ($shiphome,$ouiShiphome);

  foreach $obj (@{$techstack->{TXK::XML::BODY()}})
   {
     if ( $obj->{TXK::XML::HEAD()} eq "shiphome" &&
                        $obj->{TXK::XML::IS_INNER()} )
      {
        my ($product,$version);

        $product = _getXMLName($obj->{TXK::XML::NAMES()},"product");
        $version = _getXMLName($obj->{TXK::XML::NAMES()},"version");

        return $self->setError(
                "Invalid <TechStack> <shiphome> product : $product")
                   unless ( $product eq "db" || $product eq "iAS" ||
                            $product eq "OUI" );

        if ( $product eq "db" || $product eq "iAS" )
         {
            $self->{$PRODUCT_TYPE} = $product;
            $self->{$PRODUCT_VERSION} = $version;
            $shiphome    = $obj;
         }
        elsif ( $product eq "OUI" )
         {
            $self->{$OUI_PRODUCT_TYPE} = $product;
            $self->{$OUI_PRODUCT_VERSION} = $version;
            $ouiShiphome    = $obj;
         }
      }
   }

  return $self->setError("TechStack OUI <shiphome> <version> cannot be null")
                   if ( $self->{$OUI_PRODUCT_TYPE} &&
                        $self->{$OUI_PRODUCT_VERSION} eq "" );

  my $location;

  if ( defined $shiphome )
   {
     return $self->setError(
                        "TechStack product <shiphome><version> cannot be null")
                unless ( $self->{$PRODUCT_VERSION} );

     foreach $obj (@{$shiphome->{TXK::XML::BODY()}})
      {
        $location = $obj, last
                      if ( $obj->{TXK::XML::HEAD()} eq "location" );
      }

     return $self->setError(
                "No product <shiphome> <location> entry in XML definition file")
            unless defined $location;

     $self->{$PRODUCT_LOCATION_PATH} =
                            _getXMLName($location->{TXK::XML::NAMES()},"path");

     return $self->setError(
           "product <shiphome> <location> path in XML definition file is empty")
         unless ( defined $self->{$PRODUCT_LOCATION_PATH} &&
                  $self->{$PRODUCT_LOCATION_PATH} ne "" );
    }

  $location = undef;

  if ( defined $ouiShiphome )
   {
     foreach $obj (@{$ouiShiphome->{TXK::XML::BODY()}})
      {
        $location = $obj, last
                      if ( $obj->{TXK::XML::HEAD()} eq "location" );
      }

     return $self->setError(
                   "No OUI <shiphome> <location> entry in XML definition file")
            unless defined $location;

     $self->{$OUI_PRODUCT_LOCATION_PATH} =
                          _getXMLName($location->{TXK::XML::NAMES()},"path");
   }

  return $self->setError(
       "OUI <shiphome> <location> path in XML definition file is empty")
       if ( $self->{$OUI_PRODUCT_TYPE} &&
            !$self->{$OUI_PRODUCT_LOCATION_PATH} );

#
#   Now traverse the patch list.
#

  my ($patch,$patchPath,$baseNo,$level,$platform,$actualNo,$name,
      $p_version,$p_action,$p_type,$p_comment,$p_download,$p_dbupdate,
      $p_filesysupdate,$p_approved);

  my %patchTable;

  foreach $obj (@{$techstack->{TXK::XML::BODY()}})
   {
     next unless ( $obj->{TXK::XML::HEAD()} eq "patchno" );

     $patch = $obj;

     return $self->setError
               ("TechStack <patchno> entry cannot contain an inner node")
            if ( $patch->{TXK::XML::IS_INNER()} );

     $patchPath = _getXMLName($patch->{TXK::XML::NAMES()},"path");

     $level    = _getXMLName($patch->{TXK::XML::NAMES()},"level");
     $platform = _getXMLName($patch->{TXK::XML::NAMES()},"platform");
     $name     = _getXMLName($patch->{TXK::XML::NAMES()},"name");
     $p_type   = _getXMLName($patch->{TXK::XML::NAMES()},"type");
     $p_version= _getXMLName($patch->{TXK::XML::NAMES()},"version");
     $p_action = _getXMLName($patch->{TXK::XML::NAMES()},"action");
     $p_comment= _getXMLName($patch->{TXK::XML::NAMES()},"comment");
     $p_download= _getXMLName($patch->{TXK::XML::NAMES()},"download");
     $p_approved= _getXMLName($patch->{TXK::XML::NAMES()},"approved");
     $p_dbupdate= _getXMLName($patch->{TXK::XML::NAMES()},"dbupdate");
     $p_filesysupdate
                = _getXMLName($patch->{TXK::XML::NAMES()},"filesysupdate");

     return $self->setError("TechStack <patchno> name not valid - $name")
               unless ( $name eq "install" ||
                        $name =~ m/standalone/ ||
                        $name =~ m/patchset/ );

     $level = "100" unless ( defined $level );

     $p_download      = "Yes" unless ( defined $p_download );
     $p_dbupdate      = "No"  unless ( defined $p_dbupdate );
     $p_approved      = "No"  unless ( defined $p_approved );
     $p_filesysupdate = "Yes" unless ( defined $p_filesysupdate );

     return $self->setError
                  ("TechStack <patchno> level must be an integer - $level")
            unless ( $level =~ m/^\s*(\d+)\s*$/ );

     $actualNo = $patch->{TXK::XML::BODY()}->[0];

     return $self->setError(
               "TechStack <patchno> OS patch number not valid - $actualNo")
               unless ( $actualNo =~ m/^(\d+)$/ );

     $platform = TXK::OSD::GENERIC unless(defined $platform && $platform ne "");

     return $self->setError
                   ("TechStack <patchno> invalid platform : $platform")
            unless ( TXK::OSD->isValidAutoBuildPlatform($platform) );

     next unless ($platform eq TXK::OSD::GENERIC ||
                  ($platform eq TXK::OSD::UNIX_GENERIC && TXK::OSD->isUNIX()) ||
                  $platform eq TXK::OSD->getAutoBuildName() );

     if ( $name eq "install" || $name =~ m/standalone/ )
      {
        return $self->setError(
                     "TechStack <patchno> path entry not valid - $patchPath")
                unless ( $patchPath =~ m/^(patch)\/(\d+)$/ );

        $baseNo = $2;   # Second parameter from preceeding m/ operator.
      }
     else
      {
        $baseNo = $actualNo;
      }

     $patchTable{$baseNo} =
                  { TXK::AutoBuild::BASE_NO()     => $baseNo,
                    TXK::AutoBuild::PORT_NO()     => $actualNo,
                    TXK::AutoBuild::PATCH_LEVEL() => $level, 
                    TXK::AutoBuild::PLATFORM()    => $platform ,
                    TXK::AutoBuild::NAME()        => $name,
                    TXK::AutoBuild::PATH()        => $patchPath,
                    TXK::AutoBuild::TYPE()        => $p_type,
                    TXK::AutoBuild::ACTION()      => $p_action,
                    TXK::AutoBuild::VERSION()     => $p_version,
		    TXK::AutoBuild::COMMENT()	  => $p_comment,
		    TXK::AutoBuild::DOWNLOAD()	  => $p_download,
		    TXK::AutoBuild::APPROVED()	  => $p_approved,
		    TXK::AutoBuild::DB_UPDATE()	  => $p_dbupdate,
		    TXK::AutoBuild::FILESYS_UPDATE()
                                                  => $p_filesysupdate,
                  }
            unless ( exists($patchTable{$baseNo}) &&
                     $platform eq TXK::OSD::GENERIC );
   }

#
#  Create PATCH_LIST in level order.
#
  my $data;

  foreach $data ( sort { $a->{TXK::AutoBuild::PATCH_LEVEL()} 
                            <=> $b->{TXK::AutoBuild::PATCH_LEVEL()} }
                       ( values %patchTable )
                )
   {
     push @{$self->getRefPatchList()}, $data 
                        if ( $data->{TXK::AutoBuild::NAME()} =~ m/standalone/ );
     push @{$self->getRefPatchSetList()},$data 
		        if ( $data->{TXK::AutoBuild::NAME()} =~ m/patchset/ );
     push @{$self->getRefInstallList()}, $data 
			if ( $data->{TXK::AutoBuild::NAME()} eq "install" );
   }

  return $self->setError
         ("Can only have one <patchno> install entry per definition file")
       if ( scalar($self->getInstallList()) > 1 );

  return $self->setError("<techstack> <patchno> must have an install entry")
       if ( scalar($self->getInstallList()) == 0 );
  
  return TXK::Error::SUCCESS;
}

######################################
# isValidTechStack
######################################

sub isValidTechStack
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  my $type;

  foreach $type (@TECHSTACK_TYPE_LIST)
   {
     return TXK::Util::TRUE if ( $self->{$TECHSTACK_TYPE} eq $type) ;
   }

  return TXK::Util::FALSE;
}

######################################
# isOUIIncluded
######################################

sub isOUIIncluded
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return TXK::Util::TRUE if ( $self->{$OUI_PRODUCT_TYPE} );

  return TXK::Util::FALSE;
}

######################################
# isProductIncluded
######################################

sub isProductIncluded
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return TXK::Util::TRUE if ( $self->{$PRODUCT_TYPE} );

  return TXK::Util::FALSE;
}

######################################
# getTechStackType
######################################

sub getTechStackType
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return $self->{$TECHSTACK_TYPE};
}

######################################
# getTechStackBuild
######################################

sub getTechStackBuild
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return $self->{$TECHSTACK_BUILD};
}

######################################
# getTechStackServerBase
######################################

sub getTechStackServerBase
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return $self->{$TECHSTACK_SERVER_BASE};
}

######################################
# getProductType
######################################

sub getProductType
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return $self->{$PRODUCT_TYPE};
}

######################################
# getProductVersion
######################################

sub getProductVersion
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return $self->{$PRODUCT_VERSION};
}

######################################
# getProductLocationPath
######################################

sub getProductLocationPath
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return $self->{$PRODUCT_LOCATION_PATH};
}

######################################
# getOUIProductType
######################################

sub getOUIProductType
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return $self->{$OUI_PRODUCT_TYPE};
}

######################################
# getOUIProductVersion
######################################

sub getOUIProductVersion
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return $self->{$OUI_PRODUCT_VERSION};
}

######################################
# getOUIProductLocationPath
######################################

sub getOUIProductLocationPath
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return $self->{$OUI_PRODUCT_LOCATION_PATH};
}

######################################
# getPatchList
######################################

sub getPatchList
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return ( @{$self->{$PATCH_LIST}} );
}

######################################
# getPatchSetList
######################################

sub getPatchSetList
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return ( @{$self->{$PATCHSET_LIST}} );
}

######################################
# getInstallList
######################################

sub getInstallList
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return ( @{$self->{$INSTALL_LIST}} );
}

######################################
# getRefPatchList
######################################

sub getRefPatchList
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return $self->{$PATCH_LIST};
}

######################################
# getRefPatchSetList
######################################

sub getRefPatchSetList
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return $self->{$PATCHSET_LIST};
}

######################################
# getRefInstallList
######################################

sub getRefInstallList
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return $self->{$INSTALL_LIST};
}

######################################
# getAllPatches
######################################

sub getAllPatches
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return ( @{$self->{$INSTALL_LIST}}, @{$self->{$PATCHSET_LIST}} , 
           @{$self->{$PATCH_LIST}} );

}

######################################
# getPatch
######################################

sub getPatch
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  my $patch;

  foreach $patch ($self->getAllPatches())
   {
     return $patch if ( $patch->{TXK::AutoBuild::BASE_NO()} eq $args );
   }

  return undef;
}

######################################
# getPatchListInfo
######################################

sub getPatchListInfo
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  my ($str,$data);

  $str = "Patch List\n";

  foreach $data (@{$self->getRefPatchList()})
   {
     $str .= "Patch="    . $data->{TXK::AutoBuild::BASE_NO()}.
             " Port="    . $data->{TXK::AutoBuild::PORT_NO()}.
             " Platform=". $data->{TXK::AutoBuild::PLATFORM()}.
             " Level="   . $data->{TXK::AutoBuild::PATCH_LEVEL()} .
             " Path="    . $data->{TXK::AutoBuild::PATH()} .
             " Type="    . $data->{TXK::AutoBuild::TYPE()} .
             " Version=" . $data->{TXK::AutoBuild::VERSION()} .
             " Action="  . $data->{TXK::AutoBuild::ACTION()} .
             " Comment=" . $data->{TXK::AutoBuild::COMMENT()} .
             " Download=". $data->{TXK::AutoBuild::DOWNLOAD()} .
             " DB_Update=" . $data->{TXK::AutoBuild::DB_UPDATE()} .
             " FileSys_Update=" . $data->{TXK::AutoBuild::FILESYS_UPDATE()} .
             " Approved=" . $data->{TXK::AutoBuild::APPROVED()} .
             "\n";
   }

  return $str;
}

######################################
# getPatchSetListInfo
######################################

sub getPatchSetListInfo
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  my ($str,$data);

  $str = "PatchSet List\n";

  foreach $data (@{$self->getRefPatchSetList()})
   {
     $str .= "Patch="    . $data->{TXK::AutoBuild::BASE_NO()}.
             " Port="    . $data->{TXK::AutoBuild::PORT_NO()}.
             " Platform=". $data->{TXK::AutoBuild::PLATFORM()}.
             " Level="   . $data->{TXK::AutoBuild::PATCH_LEVEL()} .
             " Path="    . $data->{TXK::AutoBuild::PATH()} .
             " Type="    . $data->{TXK::AutoBuild::TYPE()} .
             " Version=" . $data->{TXK::AutoBuild::VERSION()} .
             " Action="  . $data->{TXK::AutoBuild::ACTION()} .
             " Comment=" . $data->{TXK::AutoBuild::COMMENT()} .
             " Download=". $data->{TXK::AutoBuild::DOWNLOAD()} .
             " DB_Update=" . $data->{TXK::AutoBuild::DB_UPDATE()} .
             " FileSys_Update=" . $data->{TXK::AutoBuild::FILESYS_UPDATE()} .
             " Approved=" . $data->{TXK::AutoBuild::APPROVED()} .
             "\n";
   }

  return $str;
}

######################################
# getInstallListInfo
######################################

sub getInstallListInfo
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  my ($str,$data);

  $str = "Install List\n";

  foreach $data (@{$self->getRefInstallList()})
   {
     $str .= "Patch="    . $data->{TXK::AutoBuild::BASE_NO()}.
             " Port="    . $data->{TXK::AutoBuild::PORT_NO()}.
             " Platform=". $data->{TXK::AutoBuild::PLATFORM()}.
             " Level="   . $data->{TXK::AutoBuild::PATCH_LEVEL()} .
             " Path="    . $data->{TXK::AutoBuild::PATH()} .
             " Type="    . $data->{TXK::AutoBuild::TYPE()} .
             " Version=" . $data->{TXK::AutoBuild::VERSION()} .
             " Action="  . $data->{TXK::AutoBuild::ACTION()} .
             " Comment=" . $data->{TXK::AutoBuild::COMMENT()} .
             " Download=". $data->{TXK::AutoBuild::DOWNLOAD()} .
             " DB_Update=" . $data->{TXK::AutoBuild::DB_UPDATE()} .
             " FileSys_Update=" . $data->{TXK::AutoBuild::FILESYS_UPDATE()} .
             " Approved=" . $data->{TXK::AutoBuild::APPROVED()} .
             "\n";
   }

  return $str;
}

######################################
# getTechStackInfo
######################################

sub getTechStackInfo
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return 
       "TechStack    Type    = ". $self->getTechStackType() . "\n" .
       "             Build   = ". $self->getTechStackBuild() . "\n" .
       "             ServerBase = ". $self->getTechStackServerBase() . "\n" .
       "             Product = ". $self->getProductType() . "\n" .
       "                Version = ". $self->getProductVersion() . "\n" .
       "                Path    = ". $self->getProductLocationPath() . "\n" .
       "             Product = ". $self->getOUIProductType() . "\n" .
       "                Version = ". $self->getOUIProductVersion() . "\n" .
       "                Path    = ". $self->getOUIProductLocationPath()
                  ;
}

######################################
# End of Public methods
######################################

# ==========================================================================

sub _getXMLName
{
 my $names = $ARG[0];
 my $id    = $ARG[1];

 my $entry;
 my $data;

 foreach $entry ( @$names )
  {
    if ( exists $entry->{$id} )
     {
       $data = $entry->{$id} ;
       $data =~ s/\'//g;
       $data =~ s/\"//g;

       return $data;
     }
  }

 return undef;
}

# ==========================================================================

1;

