
# +===========================================================================+
# |   Copyright (c) 2003 Oracle Corporation, Redwood Shores, California, USA
# |                         All Rights Reserved
# |                        Applications Division
# +===========================================================================+
# |
# | FILENAME
# |   AutoConfig.pm
# |
# | DESCRIPTION
# |      TXK AutoConfig package
# |
# | USAGE
# |       See AutoConfig.html
# |
# | PLATFORM
# |
# | NOTES
# |
# +===========================================================================+

# $Header: AutoConfig.pm 03-aug-2005.08:54:53 ndutko Exp $

package TXK::AutoConfig;

@ISA = qw( TXK::Common );

######################################
# Standard Modules
######################################

use strict;
use English;
use Carp;

require 5.005;

######################################
# Package Specific Modules
######################################

use TXK::Error();
use TXK::Util();
use TXK::FileSys();
use TXK::IO();
use TXK::Techstack();

######################################
# Public Constants
######################################

######################################
# Package Variables 
######################################

my $PACKAGE_ID = "TXK::AutoConfig";

######################################
# Object Keys
######################################

my $TEMPLATE_FILE  = "template";
my $OUTPUT_FILE    = "output";
my $XML_OBJ	   = "xmlObj";
my $OA_VARS        = "OAVars";
my $FILE_NAME      = "file";
my $CMD            = "cmd";
my $STORE          = "store";
my $DERIVED        = "derived";
my $JDK118         = "jdk118";

######################################
#
# Object Structure
# ----------------
#
#  Hash Array
#
######################################

######################################
# Package Methods 
#
# Public
#
#	new 	- build empty object
#
######################################

sub new;
sub DESTROY;
sub instantiate;
sub loadContext;
sub getCtxValue;
sub setOAVar;
sub getOAVar;
sub isDerivedMode;
sub setDerivedMode;
sub storeContext;

######################################
# Package Methods
# 
# Private
#       All private methods are marked with a leading underscore.
#
######################################
sub _updatecontext;
sub _setCmd;
sub _checkAccess;
sub _flush;

######################################
# Constructor
######################################

sub new {
  my $type = $ARG[0];

  my $self = TXK::Common->new();

  bless $self, $PACKAGE_ID ;

  my $key;

  my %INIT_OBJ = (
		   PACKAGE_IDENT   => $PACKAGE_ID,
		   $TEMPLATE_FILE  => undef,
		   $OUTPUT_FILE    => undef,
                   $FILE_NAME      => undef,
                   $XML_OBJ        => TXK::XML->new(), 
		   $JDK118         => TXK::Util::FALSE,
                   $STORE          => {},
                   $DERIVED        => " ",
                  );

  foreach $key (keys %INIT_OBJ)
   {
     $self->{$key} = $INIT_OBJ{$key};
   }

  return $self;
}

######################################
# Destructor
######################################

sub DESTROY
{
}

######################################
# instantiate
######################################

sub instantiate
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  my %oaVars;
  my $useOAVars = TXK::Util::FALSE;

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  TXK::Util->isValidArgs({args=>$args,
                          reqd=>["$TEMPLATE_FILE","$OUTPUT_FILE",
			         "$XML_OBJ"]});

  return $self->setError("$XML_OBJ parameter must be an instance of TXK::XML")
                     unless (  ref($args->{$XML_OBJ}) eq "TXK::XML" );

  $self->{$TEMPLATE_FILE} = TXK::Util->getScalarArg
                                  ($TEMPLATE_FILE,$args->{$TEMPLATE_FILE});
  $self->{$OUTPUT_FILE}   = TXK::Util->getScalarArg
                                  ($OUTPUT_FILE,$args->{$OUTPUT_FILE});

  my $xml = $args->{$XML_OBJ};

  if ( exists $args->{$OA_VARS} && ref($args->{$OA_VARS}) eq "ARRAY" )
   {
     my $i;
 
     $useOAVars = TXK::Util::TRUE;

     foreach $i (@{$args->{$OA_VARS}})
      {
        $oaVars{$i} = " ";
      }
   }

  my $fsys = TXK::FileSys->new();

  $fsys->access({ fileName=>$self->{$TEMPLATE_FILE},
                  type=>TXK::FileSys::FILE,
                  checkMode=>TXK::FileSys::READ_ACCESS,
                })
      or return $self->setError("No read access for $self->{$TEMPLATE_FILE}");

  $fsys->access({ fileName=>$self->{$OUTPUT_FILE},
                  type=>TXK::FileSys::FILE,
                  checkMode=>TXK::FileSys::CREATE_ACCESS,
                })
      or return $self->setError("No create access for $self->{$OUTPUT_FILE}");

#       All OK , so create the target file in case we need intermediate
#       directories.

  $fsys->create({ fileName=>$self->{$OUTPUT_FILE},
                  type=>TXK::FileSys::FILE,
                })
         or return $self->setError("Cannot create $self->{$OUTPUT_FILE}");

  my $tmpl_io  = TXK::IO->new();
  my $output_io= TXK::IO->new();
  my $buffer;

  $tmpl_io->open({ mode => TXK::IO::READ,
                  fileName => $self->{$TEMPLATE_FILE}
                 });

  $output_io->open({ mode => TXK::IO::WRITE,
                     fileName => $self->{$OUTPUT_FILE}
                   });

  my $tmpl_fh = $tmpl_io->getFileHandle();
  my $output_fh=$output_io->getFileHandle();

  while ($buffer = <$tmpl_fh>)
   {
     my @varList = ( $buffer =~ m/%[a-zA-Z0-9_\/]+%/g );
     my $varId;

     foreach $varId (@varList)
      {
        my $varName = $varId;

        $varName =~ s/%//g;

        my $varValue = undef;

        $varValue = $xml->getOAVar($varName)
		        if (   !$useOAVars || 
                             ( $useOAVars && exists($oaVars{$varName}) ) );

        $buffer =~ s/${varId}/${varValue}/ if defined $varValue;
      }
     print $output_fh $buffer;
   }
 
  $tmpl_io->close();
  $output_io->close();

  return TXK::Error::SUCCESS;
}



######################################
# loadContext
######################################
sub loadContext{
 my $self=$ARG[0];
 my $args=$ARG[1];

 TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

 TXK::Util->isValidArgs({args=>$args,reqd=>[$FILE_NAME]});

 my $fileName= TXK::Util->getScalarArg($FILE_NAME,$args->{$FILE_NAME});
 my $xmlObj = TXK::XML->new();

 $self->{$XML_OBJ}=$xmlObj;
 $self->{$FILE_NAME}=$fileName;
 $self->{$XML_OBJ}->loadDocument( { $FILE_NAME => $fileName } );

 return TXK::Error::SUCCESS;
}


######################################
# getOAVar
######################################
sub getOAVar{
 my $self=$ARG[0];
 my $args=$ARG[1];

 TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

 my $oaVar = TXK::Util->getScalarArg(q(ctx variable),$args);

 return ( $self->{$STORE}->{$oaVar}  ? $self->{$STORE}->{$oaVar}
                                     : $self->{$XML_OBJ}->getOAVar($oaVar));
}


######################################
# getCtxValue
######################################

sub getCtxValue{
  my $self = $ARG[0];
  my $oaVar = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return $self->getOAVar($oaVar);
}

######################################
# isDerivedMode
######################################
sub isDerivedMode{

 my $self=$ARG[0];

 TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

 return TXK::Util::TRUE if ( $self->{$DERIVED} eq q(-derived) );

 return TXK::Util::FALSE; 
}

######################################
# storeContext
######################################
sub storeContext{
 my $self=$ARG[0];

 TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

 my $store=$self->{$STORE};

 $self->_checkAccess($self->{$FILE_NAME},TXK::FileSys::WRITE_ACCESS);
 $self->_setCmd() unless $self->{$CMD};
 $self->_updateContext(%$store);   # Hash expanded into key/value list.
 $self->{$XML_OBJ}->setOAVar($store);
 $self->{$STORE}={};

 return TXK::Error::SUCCESS;
}

######################################
# setOAVar
######################################
sub setOAVar{
 my $self=$ARG[0];
 my $args=$ARG[1];

 TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

 my $name;

 foreach $name (keys %$args)
  {
    $self->{$STORE}->{$name} = $args->{$name}
                    if ( defined ($self->{$XML_OBJ}->getOAVar($name)) );
  } 

 $self->_flush() if $self->isDerivedMode();

 return TXK::Error::SUCCESS;
}

######################################
# setDerivedMode
######################################
sub setDerivedMode{
 my $self=$ARG[0];
 my $args=$ARG[1];
 TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

 my $derived= TXK::Util->getBooleanArg(q(is derived),$args);
 if ($derived){
   if (! $self->isDerivedMode()){
       $self->_flush();
       $self->{$DERIVED}=q(-derived); 
     }
 }else{
   if ( $self->isDerivedMode()){
       $self->_flush();
       $self->{$DERIVED}=" ";
     }
 }
 
 return TXK::Error::SUCCESS;
}


######################################
# End of Public methods
######################################
sub _flush{
 my $self=$ARG[0];
 $self->storeContext();
 $self->loadContext({$FILE_NAME,$self->{$FILE_NAME}});
}

sub _updateContext{
 my $self = shift;
 my $cmdRef = $self->{$CMD};
 $cmdRef->{arg8} =$self->{$DERIVED}; 
 my $proc = TXK::Process->new(); 

 while(@_){
    $cmdRef->{arg6} =shift; 
    $cmdRef->{arg7}=shift; 
    $cmdRef->{arg7} = q("") unless ( $cmdRef->{arg7} );

    $proc->abortOnError ( { enable => TXK::Util::FALSE } ) 
	if ( TXK::OSD->isWindows() and $self->{$JDK118} );

    $proc->run($cmdRef);

    $proc->abortOnError ( { enable => TXK::Util::TRUE } ) 
	if ( TXK::OSD->isWindows() and $self->{$JDK118} );

 }

 return TXK::Error::SUCCESS;
}

sub _setCmd{
  my $self=$ARG[0];
  my $jretop = TXK::OSD->trDirPathFromBase($self->getCtxValue(q(s_jretop)));
  my $ext = TXK::OSD->getExecutableExt();
  $ext = q(.) . $ext if ( $ext );
  my $nojit=" ";
  my $javaCommand;
  my $arg2=" ";
  my $className = q(oracle.apps.ad.context.UpdateContext);
  my $java = "$jretop/bin/java$ext";
  my $jre  = "$jretop/bin/jre$ext";

  TXK::OSD->setClassPath("");
  TXK::OSD->addClassPath($self->getCtxValue(q(s_jretop)) . "/lib/rt.jar");
  TXK::OSD->addClassPath($self->getCtxValue(q(s_javatop)));
  TXK::OSD->addClassPath($self->getCtxValue(q(s_javatop)) . "/apps.zip");
  TXK::OSD->addClassPath($self->getCtxValue(q(s_adtop)) . "/java/adconfig.zip");

  if( -x $jre ){
      $nojit =  TXK::OSD->isNT() ? q(-nojit) : " ";
      $javaCommand = $jre;
      $arg2=q(-mx256M);
      TXK::OSD->addClassPath($self->getCtxValue(q(s_javatop))."/appsborg.zip");
  } elsif (-x $java) {
      $javaCommand = $java;
      $arg2=q(-Xmx256M);
      TXK::OSD->addClassPath($self->getCtxValue(q(s_javatop))."/appsborg2.zip");
  } else {
      TXK::Error::abort("Valid java executable not found\n");
  }

  my $ctx_file_key  = TXK::Techstack::APPS_CTX_FILE_KEY;
  my $techstack     = TXK::Techstack->new({ $ctx_file_key => $self->{$FILE_NAME} } );
  my $jdkversion    = $techstack->txkGetJDKVersion( { 'javacmd'  => $javaCommand } );
  $self->{$JDK118}  = TXK::Util::TRUE if ( $jdkversion =~ m/1\.1\.8/ );

  my $cmdRef= {
                command => TXK::OSD->trDirPathToBase($javaCommand),
                arg1 => $nojit,
                arg2 => $arg2,
                arg3 => "-classpath " . TXK::OSD->getClassPath(),
                arg4 => "$className",
                arg5 => $self->{$FILE_NAME}
              };
  $self->{$CMD} = $cmdRef;
}

sub _checkAccess{
  my $self=$ARG[0];
  my $file=$ARG[1];
  my $access=$ARG[2];
  my $fsys = TXK::FileSys->new();

  $fsys->access({fileName=>$file,
               type=>TXK::FileSys::FILE,
               checkMode=>$access})
   or TXK::Error::abort($fsys->getError());
}

# ==========================================================================

1;

