
# +===========================================================================+
# |   Copyright (c) 2003 Oracle Corporation, Redwood Shores, California, USA
# |                         All Rights Reserved
# |                        Applications Division
# +===========================================================================+
# |
# | FILENAME
# |   Common.pm
# |
# | DESCRIPTION
# |      TXK Common package
# |
# | USAGE
# |       See Common.html
# |
# | PLATFORM
# |
# | NOTES
# |
# +===========================================================================+

# $Header: Common.pm 03-aug-2005.08:54:53 ndutko Exp $

package TXK::Common;

######################################
#
#!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
#
# Common should NEVER use class methods
# as TXK::Util::isValidObj will fail
# for derived classes.
#
#!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
#
######################################

######################################
# Standard Modules
######################################

use strict;
use English;
use Carp;

require 5.005;

######################################
# Package Specific Modules
######################################

use TXK::Util();

######################################
# Public Constants
######################################

######################################
# Private Constants
######################################

######################################
# Package Variables 
######################################

my $PACKAGE_ID = "TXK::COMMON";

######################################
# Object Keys
######################################

my $ABORT_ON_ERROR = "abortOnError";
my $STOP_ON_ERROR  = "stopOnError";
my $ERROR_INFO     = "errorInfo";
my $DEBUG          = "debug";
my $PACKAGE_IDENT  = "PACKAGE_IDENT";

######################################
#
# Object Structure
# ----------------
#
#  Hash Array
#
######################################

######################################
# Package Methods 
#
# Public
#
#	new 	- build empty object
#
#   Unless otherwise stated all other public methods pass one argument,
#   either a scalar or hash table/
#
######################################

sub new;
sub DESTROY;
sub debug;
sub abortOnError;
sub getError;
sub clone;

######################################
# Package Methods
# 
# Private
#       All private methods are marked with a leading underscore.
#
######################################

sub setErrorNoAbort;
sub setAbort;
sub doError;
sub setError;
sub clearErrorMessage;

######################################
# Constructor
######################################

sub new {
  my $type = $ARG[0];

  my $self = {} ;

  bless $self , $PACKAGE_ID;

  my $key;

  my %INIT_OBJ = (
                   $ABORT_ON_ERROR => "true",
		   $STOP_ON_ERROR  => "0",
                   $ERROR_INFO     => {},
                   $DEBUG          => "0",
                   $PACKAGE_IDENT  => $PACKAGE_ID,
                  );

  foreach $key (keys %INIT_OBJ)
   { 
     $self->{$key} = $INIT_OBJ{$key};
   } 

  return $self;
}

######################################
# Destructor
######################################

sub DESTROY
{
}

######################################
# Set Debug 
######################################

sub debug
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  $self->{$DEBUG} = TXK::Util->setDebug($args);
}

######################################
# Set abortOnError
######################################

sub abortOnError
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];
  
  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});
  
  $self->{$ABORT_ON_ERROR} = TXK::Util->setAbortOnError($args)
                                if ( defined $args );
}                               

######################################
# getError
######################################

sub getError
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];
  
  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});
  
  return $self->{$ERROR_INFO};
} 

######################################
# clearErrorMessage
######################################

sub clearErrorMessage
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  $self->{$ERROR_INFO}->{'message'} = "";
  
  return TXK::Error::SUCCESS;
}

######################################
# End of Public methods
######################################

# ==========================================================================

sub setErrorNoAbort
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];
  my $caller_level = $ARG[2];
  
  my $oldvalue = $self->{$ABORT_ON_ERROR} ;
  
  $self->{$ABORT_ON_ERROR} = TXK::Util::FALSE;
  
  doError($self,$args,$caller_level);
  
  $self->{$ABORT_ON_ERROR} = $oldvalue;
  
  return TXK::Error::FAIL;
} 

# ==========================================================================

sub setAbort
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];
  my $caller_level = $ARG[2];
  
#       No need to keep old value as we won't be coming back !

  $self->{$ABORT_ON_ERROR} = TXK::Util::TRUE;
  
  doError($self,$args,$caller_level);
} 

# ==========================================================================

sub setStop
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];
  my $caller_level = $ARG[2];

  my $oldvalue = $self->{$STOP_ON_ERROR} ;

  $self->{$STOP_ON_ERROR} = TXK::Util::TRUE;

  doError($self,$args,$caller_level);

  $self->{$STOP_ON_ERROR} = $oldvalue;

  return TXK::Error::FAIL;
}

# ==========================================================================

sub doError
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];
  my $caller_level = $ARG[2];
  my %msg ;

  if ( ref($args) eq "HASH" )
   {
     $self->{$ERROR_INFO} = %$args ;

     %msg = %$args;
     $msg{"_caller_level"} = ( defined $caller_level ? $caller_level : 3 );
   }
  else
   {
     $self->{$ERROR_INFO}->{'message'}
             = $args .  "\n" . $self->{$ERROR_INFO}->{'message'};

     %msg = ( _msg => $self->{$ERROR_INFO}->{'message'},
              _caller_level => ( defined $caller_level ? $caller_level : 3 ) );
   }

  TXK::Error->stop($self->{$ERROR_INFO})  if ( $self->{$STOP_ON_ERROR} );
  TXK::Error->abort(\%msg) if ( $self->{$ABORT_ON_ERROR} );

  return TXK::Error::FAIL;
}

# ==========================================================================

sub setError
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];
  my $caller_level = $ARG[2];

#  setError is a dummy wrapper over doError to keep caller_level consistent.

  return doError($self,$args,$caller_level);
}

sub clone
{
    my $self = $ARG[0];

    my $clone = { };
    
    bless $clone, ref($self);

    TXK::Util->copyHash($self,$clone);

    return $clone;
}

# ==========================================================================

1;
