
# +===========================================================================+
# |   Copyright (c) 2003 Oracle Corporation, Redwood Shores, California, USA
# |                         All Rights Reserved
# |                        Applications Division
# +===========================================================================+
# |
# | FILENAME
# |   Inventory.pm
# |
# | DESCRIPTION
# |      TXK Inventory package
# |
# | USAGE
# |       See Inventory.html
# |
# | PLATFORM
# |
# | NOTES
# |
# +===========================================================================+

# $Header: emll/sysman/admin/scripts/TXK/Inventory.pm /main/3 2008/11/16 06:44:07 ndutko Exp $

package TXK::Inventory;

@ISA = qw( TXK::Common );

#------------------------------------------------------------------------------
# Standard Modules
#------------------------------------------------------------------------------

use strict;
use English;
use Carp;

require 5.005;

#------------------------------------------------------------------------------
# Public Constants
#------------------------------------------------------------------------------

use constant NULL		=> ""			;
use constant APPS_USER		=> "apps"		;
use constant APPS_PASSWORD_KEY	=> "appspass"		;
use constant APPS_DEFAULT_PWD	=> "UNKNOWN_PWD"        ;
use constant SYS_PASSWORD_KEY 	=> "syspass"		;
use constant SYS_DEFAULT_PWD	=> "UNKNOWN_PWD"	;
use constant APPS_CTX_FILE_KEY	=> "contextfile"	;
use constant JDKVERSION_KEY	=> "jdkversion"		;
use constant DEFAULT_JDKVERSION	=> "1.3.1"		;
use constant ADIDENT_EXE        => "adident"		;
use constant ADXMLCTX_FILE      => "adxmlctx.tmp"	;
use constant WEB_PATCH_LIST_13	=> "webPatchListJDK13"	;
use constant WEB_PATCH_LIST_14	=> "webPatchListJDK14"	;
use constant CONC_PATCH_LIST_13	=> "concPatchListJDK13"	;
use constant CONC_PATCH_LIST_14	=> "concPatchListJDK14"	;

use constant TXK_VALIDATE_MESSAGE  => "validateMsg"	;

#------------------------------------------------------------------------------
# Private Constants
#------------------------------------------------------------------------------

use constant JAVACMD_NOT_FOUND	=> "-1"			;
use constant JDK_NOT_REQUIRED	=> "1"			;

#------------------------------------------------------------------------------
# Package Specific Modules
#------------------------------------------------------------------------------

use TXK::IO();
use TXK::Log();
use TXK::OSD();
use TXK::XML();
use TXK::Util();
use TXK::Error();
use TXK::Runtime();
use TXK::Process();
use TXK::SQLPLUS();
use TXK::RunScript();

#------------------------------------------------------------------------------
# Package Variables 
#------------------------------------------------------------------------------

my $PACKAGE_ID = "TXK::Inventory";

#------------------------------------------------------------------------------
# Object Keys
#------------------------------------------------------------------------------

my $APPS_VERSION	= "AppsVersion"		;
my $DB_VERSION     	= "DBVersion"		;
my $JDK_VERSION    	= "JDKVersion"		;
my $IAS_VERSION    	= "iASVersion"		;
my $TOOLS_VERSION  	= "toolsVersion"	;
my $LOG_FILE       	= "logFile"		;
my $REP_FILE       	= "reportFile"		;
my $JDK_TOP		= "JDK_TOP"		;
my $JAVA_CMD		= "JavaExecutable"	;
my $JAVA_TOP		= "JAVA_TOP"		;
my $CONF_WITH_JDK  	= "configuredWithJDK"	;
my $HOST_NAME      	= "hostName"		;
my $APPL_TOP	   	= "APPL_TOP"		;
my $TWO_TASK            = "TWO_TASK"		;
my $ORACLE_HOME	   	= "ORACLE_HOME"		;
my $IAS_ORACLE_HOME	= "IAS_ORACLE_HOME"	;
my $APPS_CTX_FILE  	= "appsContextFile"  	;
my $APPS_CTX_VERSION  	= "appsCtxVersion"  	;
my $CTXTMPL_VERSION  	= "ctxTmplVersion"  	;
my $ADCONFIG_ENABLED	= "adconfigEnabled"	;
my $IS_WEB_NODE		= "isWebNode"		;
my $IS_FORMS_NODE	= "isFormsNode"		;
my $IS_CONC_NODE	= "isConcNode"		;
my $IS_DB_NODE		= "isDBNode"		;
my $IS_ADMIN_NODE	= "isAdminNode"		;
my $ERROR_MESSAGE	= "errorMessage"	;
my $ERROR_STATUS	= "errorStatus"		;
my $APPS_PASSWORD	= TXK::Inventory::APPS_PASSWORD_KEY;
my $SYS_PASSWORD	= TXK::Inventory::SYS_PASSWORD_KEY;
my $VALIDATE_MESSAGE 	= TXK_VALIDATE_MESSAGE;

#
# The following patterns can change
#

my $JSERV_PATTERN_1       = qr/\s*wrapper\.classpath=/			;
my $JSERV_DT_JAR          = qr/lib(\/|\\)dt\.jar/			;
my $JSERV_TOOLS_JAR       = qr/lib(\/|\\)tools\.jar/			;
my $JSERV_RT_JAR          = qr/jre(\/|\\)lib(\/|\\)rt\.jar/		;
my $JSERV_CHARSETS_JAR    = qr/jre(\/|\\)lib(\/|\\)charsets\.jar/	;
my $JSERV_I18N_JAR        = qr/jre(\/|\\)lib(\/|\\)i18n\.jar/		;

my $JDBC_FILE_12          = "jdbc12\.zip"            			;
my $JDBC_FILE_14          = "jdbc14\.zip"            			;
my $APPSBORG_FILE         = "appsborg2\.zip"         			;

my $JDK_13_PATTERN	  = qr/(1\.3.1)/				;
my $JDK_14_PATTERN	  = qr/(1\.4)/					;
my $JSERVJAVA_PATTERN     = qr/\s*JSERVJAVA\s*=\s*/			;
my $JDKVERSION_PATTERN_1  = qr/\s*java version\s+\"(1\.3.1|1\.4)/	;
my $JDKVERSION_PATTERN_2  = qr/\s*java version\s+\"/			;

#------------------------------------------------------------------------------
# Object Structure
#------------------------------------------------------------------------------

#------------------------------------------------------------------------------
#
#  Hash Array
#
#------------------------------------------------------------------------------

#------------------------------------------------------------------------------
# Package Methods 
#
# Public
#
#       new     - build empty object
#
#------------------------------------------------------------------------------

sub new;
sub getOA;
sub getiASversion;
sub getJDKversion;
sub getNodeInfo;
sub getAppsCtxLocation;
sub isConfiguredWithJDK;
sub isAdConfigEnabled;
sub isAdminNode;
sub isDBNode;
sub isWebNode;
sub isFormsNode;
sub isConcNode;
sub printStatus;

#------------------------------------------------------------------------------
# Package Methods
# 
# Private
#       All private methods are marked with a leading underscore.
#
#------------------------------------------------------------------------------

sub _isAdConfigEnabled;
sub _trimDomain;
sub _getAppsCtxLocation;
sub _getJDKversion;
sub _getNodeInfo;
sub _getCtxVersion;
sub _verifyConfigJDK;
sub _isFileFound;
sub _isPatchApplied;
sub _verifyPatch;

#------------------------------------------------------------------------------
# Constructor
#------------------------------------------------------------------------------

sub new
{

  my $type = $ARG[0];
  my $args = $ARG[1];
  
  my $self = TXK::Common->new();

  bless $self, $PACKAGE_ID ;

  my $key;

  my %INIT_OBJ = (
		   PACKAGE_IDENT   	=> $PACKAGE_ID			,
		   $APPS_VERSION	=> undef			,
		   $DB_VERSION     	=> undef			,
		   $JDK_VERSION    	=> undef			,
		   $IAS_VERSION    	=> undef			,
		   $TOOLS_VERSION  	=> undef			,
		   $APPS_PASSWORD	=> undef			,
		   $SYS_PASSWORD	=> undef			,
		   $ADCONFIG_ENABLED	=> undef			,
                   $JAVA_CMD		=> undef			,
                   $JDK_TOP		=> TXK::Inventory::NULL		,
                   $JAVA_TOP		=> TXK::Inventory::NULL		,
		   $LOG_FILE       	=> TXK::Inventory::NULL		,
		   $REP_FILE       	=> TXK::Inventory::NULL		,
		   $HOST_NAME      	=> TXK::Inventory::NULL		,
		   $APPL_TOP       	=> TXK::Inventory::NULL		,
		   $TWO_TASK       	=> TXK::Inventory::NULL		,
		   $ORACLE_HOME       	=> TXK::Inventory::NULL		,
		   $IAS_ORACLE_HOME	=> TXK::Inventory::NULL		,
		   $APPS_CTX_FILE  	=> TXK::Inventory::NULL		,
		   $APPS_CTX_VERSION  	=> TXK::Inventory::NULL		,
		   $CTXTMPL_VERSION  	=> TXK::Inventory::NULL		,
		   $VALIDATE_MESSAGE	=> TXK::Inventory::NULL		,
		   $ERROR_MESSAGE	=> TXK::Inventory::NULL		,
		   $ERROR_STATUS	=> TXK::Util::FALSE		,
		   $CONF_WITH_JDK  	=> TXK::Util::FALSE		,
		   $IS_WEB_NODE		=> TXK::Util::FALSE		,
		   $IS_FORMS_NODE	=> TXK::Util::FALSE		,
		   $IS_CONC_NODE	=> TXK::Util::FALSE		,
		   $IS_DB_NODE		=> TXK::Util::FALSE		,
		   $IS_ADMIN_NODE	=> TXK::Util::FALSE		,
                 );

  foreach $key (keys %INIT_OBJ)
  {
     $self->{$key} = $INIT_OBJ{$key};
  }

  $self->_init($args);

  return $self;
}

#------------------------------------------------------------------------------
# Internal Initialization
#------------------------------------------------------------------------------

sub _init
{
  my $self	= $ARG[0];
  my $args	= $ARG[1];


  $self->{$HOST_NAME}  	    = TXK::OSD->getHostName();
  $self->{$APPL_TOP} 	    = TXK::OSD->getEnvVar({name => "APPL_TOP"});
  $self->{$ORACLE_HOME}     = TXK::OSD->getEnvVar({name => "ORACLE_HOME"});
  $self->{$IAS_ORACLE_HOME} = TXK::OSD->getEnvVar({name => "IAS_ORACLE_HOME"});
  $self->{$JAVA_TOP}        = TXK::OSD->getEnvVar({name => "JAVA_TOP"});


  if ( TXK::OSD->isNT() )
   {
     $self->{$TWO_TASK} = TXK::OSD->getEnvVar({ name => "LOCAL"});
   }
  else
   {
     $self->{$TWO_TASK} = TXK::OSD->getEnvVar({ name => "TWO_TASK"});
   }  


  $self->_trimDomain();

  if ( ( ref( $args ) eq "HASH" ) && ( keys %{$args} ) )
   {
     foreach my $key ( keys %$args ) 
      {
	if ( $key eq TXK::Inventory::APPS_PASSWORD_KEY )
         {
	   $self->{$APPS_PASSWORD} = $args->{$key};
         }
        elsif ( $key eq TXK::Inventory::SYS_PASSWORD_KEY )
         {
	   $self->{$SYS_PASSWORD} = $args->{$key};
         }
        elsif ( $key eq TXK::Inventory::APPS_CTX_FILE_KEY )
         {
           $self->_isAdConfigEnabled($args);
         }
      }
   }

   if ( !defined($self->{$APPS_PASSWORD} ) )
    {
      $self->{$APPS_PASSWORD} = TXK::Inventory::APPS_DEFAULT_PWD;
    }

   if ( !defined($self->{$SYS_PASSWORD} ) )
    {
      $self->{$SYS_PASSWORD} = TXK::Inventory::SYS_DEFAULT_PWD;
    }

}

#------------------------------------------------------------------------------
# _trimDomain
#------------------------------------------------------------------------------

sub _trimDomain
{
  my $self = shift;

  if ( $self->{$HOST_NAME} =~ /^([^\.]+)\.(.*$)/ )
   {
     $self->{$HOST_NAME} =~ s/^([^\.]+)\.(.*$)/\1/;
   }
}

#------------------------------------------------------------------------------
# Destructor
#------------------------------------------------------------------------------

sub DESTROY
{
}

#------------------------------------------------------------------------------
# Retrieve iAS version
#------------------------------------------------------------------------------

#------------------------------------------------------------------------------
# Retrieve
#------------------------------------------------------------------------------

#------------------------------------------------------------------------------
## _getJDKversion()
#------------------------------------------------------------------------------

sub _getJDKversion
{
  my $self = $ARG[0];

  my $java_cmd = "";

  my $jdkversion = "0";

  ##--------------------------------------------------------
  # No need to check for JDK on the Web and Concurrent
  # processing nodes.
  ##--------------------------------------------------------

  if ( ( ! $self->{$IS_WEB_NODE} ) && ( ! $self->{$IS_CONC_NODE} ))
   {
     return TXK::Inventory::NULL;
   }

  ##--------------------------------------------------------
  # Constructing the full path to the log file
  ##--------------------------------------------------------

  my $appl_top  = TXK::OSD->getEnvVar({ name => "APPL_TOP"});

  my $appllog   = TXK::OSD->getEnvVar({ name => "APPLLOG"});

  my $log_f     = $appl_top . "/admin/" . $appllog . "/" . "java_ver.txt";

  my $logfile   = TXK::OSD->trDirPathToBase($log_f);

  ##--------------------------------------------------------
  # Now run the java executable to determine its version
  ##--------------------------------------------------------

  my $process   = TXK::Process->new();

  if ( !defined($self->{$JAVA_CMD}) )
   {
     if ( $self->isAdConfigEnabled() eq TXK::Util::TRUE )
      {
        my $xmlobj = TXK::XML->new();
        
        $xmlobj->loadDocument ( { file => $self->{$APPS_CTX_FILE} } );
   
        my $installed_jdk_location = $xmlobj->getOAVar('s_jdktop');
   
        $self->{$JDK_TOP} = TXK::OSD->trDirPathToBase($installed_jdk_location);
   
        my $java = $installed_jdk_location . "/bin/" . "java";
   
        $java_cmd = TXK::OSD->trDirPathToBase($java);

        $self->{$VALIDATE_MESSAGE} .= "\n[NOTE]\t The location of the JDK installation as defined by the value of\n\t \(s_jdktop\) in the Applications Context file is as follows:\n\n\t $self->{$JDK_TOP}\n";
   
      }
     else
      {
        #-----------------------------------------------------------------------
        # IO package takes care of any path translation.
        #-----------------------------------------------------------------------
   
        my $af_java_path = $self->{$IAS_ORACLE_HOME} . "/Apache/Apache/bin/";
   
        my $af_java_exe  = ( TXK::OSD->isNT() ? "java.cmd" : "java.sh" );
   
        my $af_java_cmd  = $af_java_path . $af_java_exe;
   
        my $s_io = TXK::IO->new();
   
        $s_io->open({ fileName => $af_java_cmd });

        my $s_io_ref  = $s_io->getFileHandle();
        my @s_io_data = <$s_io_ref>;
   
        $s_io->close();
        
        my @javaobj = grep(/$JSERVJAVA_PATTERN/,@s_io_data);
          
        foreach ( @javaobj )
         {
           chomp;
   
           s/$JSERVJAVA_PATTERN//;
   
           if ( TXK::OSD->isNT() )
            {
              s/\s*\\jre\\bin.*//;
            }
           else
            {
              s/\s*\/jre\/bin.*//;
            }         
   
           $self->{$JDK_TOP} = $_;
         }
   
        my $java = $self->{$JDK_TOP} . "/bin/" . "java";
   
        $java_cmd = TXK::OSD->trDirPathToBase($java);

        $self->{$VALIDATE_MESSAGE} .= "\n[NOTE]\t The location of the JDK installation as defined by\n\t the JSERVJAVA variable inside the script named\n\t $af_java_cmd\n\t is as follows:\n\n\t $self->{$JDK_TOP}\n";
   
      }
   }
  else
   {
     $java_cmd = $self->{$JAVA_CMD};
   }
  
  ##---------------------------------------------------------
  # Make sure java is 1.3.1 or higher.
  ##---------------------------------------------------------

  $self->{$JAVA_CMD} = $java_cmd;

  $process->abortOnError( { enable => TXK::Util::FALSE } );

  my $run_status = $process->run({ 
                                   command    => "$java_cmd -version ",
                                   stdout     => $logfile             ,
                                });

  if ( $run_status eq TXK::Error::FAIL )
   {
     return TXK::Inventory::JAVACMD_NOT_FOUND;
   }

  my $io = TXK::IO->new();

  $io->open({ fileName => $logfile });

  my $io_ref = $io->getFileHandle();
  my @io_data = <$io_ref>;

  $io->close();

  unlink($logfile);

  unless ( grep(/$JDKVERSION_PATTERN_1/,@io_data) )
   {
     return $jdkversion;

   }

  my @jdkobj = grep(/$JDKVERSION_PATTERN_1/,@io_data);
       
  foreach ( @jdkobj )
   {
     chomp;
     s/$JDKVERSION_PATTERN_2//;
     s/\s*\"$//;
     s/_/\./;

     $self->{$JDK_VERSION} = $_;
   }

  return $self->{$JDK_VERSION};
}

#------------------------------------------------------------------------------
## getJDKversion()
#------------------------------------------------------------------------------

sub getJDKversion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $installed_jdk_version;

  if ( !defined($self->{$JDK_VERSION}) )
   {
     if ( ( $self->isWebNode() eq TXK::Util::TRUE ) ||
          ( $self->isConcNode() eq TXK::Util::TRUE ) )
      {
        $installed_jdk_version = $self->_getJDKversion();

      }
     else
      {
#        $self->{$VALIDATE_MESSAGE} .=  "\n[WARN]\t It is not necessary to check for JDK on this node of the E-Business Suite\n\t Release 11i environment, as this node is neither a Web\n\t node nor a Concurrent processing node.\n\n";
   
        return TXK::Inventory::JDK_NOT_REQUIRED;

      }
      
     if ( $installed_jdk_version eq TXK::Inventory::JAVACMD_NOT_FOUND )
      {
        return TXK::Inventory::JAVACMD_NOT_FOUND;
      }
     else
      {
        return $installed_jdk_version;
      }
   }
  else
   {
     return $self->{$JDK_VERSION};
   }
}


#------------------------------------------------------------------------------
## isConfiguredWithJDK()
#------------------------------------------------------------------------------

sub isConfiguredWithJDK
{
  my $self   = $ARG[0];
  my $args   = $ARG[1];

  $self->isAdConfigEnabled($args);

  $self->_getNodeInfo($args);

  if ( ( $self->isWebNode() eq TXK::Util::TRUE ) ||
       ( $self->isConcNode() eq TXK::Util::TRUE ) )
   {
     my $web_status    = TXK::Error::SUCCESS;
     my $conc_status   = TXK::Error::SUCCESS;
     my $prereq_status = TXK::Error::SUCCESS;
     my $config_status = TXK::Error::SUCCESS;

     my $passed_in_jdk;

     my $JDBC_FILE;

     if ( keys(%$args))
      {
        my $jdkversion_key = TXK::Inventory::JDKVERSION_KEY;

        foreach my $key (keys %$args)
         {
           if ( $key eq $jdkversion_key )
            {
              $passed_in_jdk = $args->{$jdkversion_key};
              last;
            }
         }
      }

     if ( $passed_in_jdk == 0 )
      {
        $passed_in_jdk = TXK::Inventory::DEFAULT_JDKVERSION;
      }

     my $installed_jdk_version = $self->getJDKversion();
     my $pattern_1             = $self->{$JDK_TOP};
     my $pattern_2             = $APPSBORG_FILE;

     #
     # JAVA command not found!!
     #

     if ( $installed_jdk_version eq TXK::Inventory::JAVACMD_NOT_FOUND )
      {
        $self->{$VALIDATE_MESSAGE} .= "\n[FAIL]\t The Java executable shown below could not be located:\n\t $self->{$JAVA_CMD}\n\n";

        if ( $passed_in_jdk =~ $JDK_13_PATTERN )
         {
           $self->{$VALIDATE_MESSAGE} .= "\t The Web or Concurrent processing node of this E-Business Suite\n\t Release 11i system is not configured with JDK $passed_in_jdk.\n\t Please see the My Oracle Support Note 130091.1 for more information on\n\t Upgrading to JDK 1.3 with the Oracle E-Business Suite Release 11i.\n\n";

         }
        elsif ( $passed_in_jdk =~ $JDK_14_PATTERN )
         {
           $self->{$VALIDATE_MESSAGE} .= "\n[FAIL]\t The Web or Concurrent processing node of this E-Business Suite\n\t Release 11i system is not configured with JDK $passed_in_jdk.\n\t Please see the My Oracle Support Note 246105.1 for more information on\n\t Upgrading to J2SE 1.4 with the Oracle E-Business Suite Release 11i.\n\n";

         }

        return TXK::Error::FAIL;

      }
     elsif ( $installed_jdk_version eq TXK::Inventory::JDK_NOT_REQUIRED )
      {
        $self->{$VALIDATE_MESSAGE} .= "\n[WARN]\t It is not necessary to check for an installation of JDK on this node\n\t of the E-Business Suite Release 11i environment, as this node is\n\t neither a Web node nor a Concurrent processing node.\n\n";

        return TXK::Error::WARNING;

      }

     #
     # JAVA cmd found and executed. Now get the version returned by the cmd.
     #

     if ( $installed_jdk_version =~ $JDK_13_PATTERN )
      {
        $JDBC_FILE = $JDBC_FILE_12;
      }
     elsif ( $installed_jdk_version =~ $JDK_14_PATTERN )
      {
        $JDBC_FILE = $JDBC_FILE_14;
      }

     if ( $installed_jdk_version lt $passed_in_jdk )
      {
        if ( $passed_in_jdk =~ $JDK_13_PATTERN )
         {
           $self->{$VALIDATE_MESSAGE} .= "\n[FAIL]\t The version of the currently installed JDK \($installed_jdk_version\)\n\t is lower than $passed_in_jdk.\n\n\t The Web or Concurrent processing node of this E-Business Suite\n\t Release 11i system is not configured with JDK $passed_in_jdk.\n\t Please see the My Oracle Support Note 130091.1 for more information on\n\t Upgrading to JDK 1.3 with the Oracle E-Business Suite Release 11i.\n\n";

         }
        elsif ( $passed_in_jdk =~ $JDK_14_PATTERN )
         {
           $self->{$VALIDATE_MESSAGE} .= "\n[FAIL]\t The version of the currently installed JDK \($installed_jdk_version\)\n\t is lower than $passed_in_jdk.\n\n\t The Web or Concurrent processing node of this E-Business Suite\n\t Release 11i system is not configured with JDK $passed_in_jdk.\n\t Please see the My Oracle Support Note 246105.1 for more information on\n\t Upgrading to J2SE 1.4 with the Oracle E-Business Suite Release 11i.\n\n";

         }

        return TXK::Error::FAIL;

      }

     #
     # Verify if the pre-req patches, if any, were already installed.
     #

     if ( keys(%$args))
      {
	my $web_key;
	my $conc_key;
        my $web_prereq_status  = TXK::Error::SUCCESS;
        my $conc_prereq_status = TXK::Error::SUCCESS;

	if ( $installed_jdk_version =~ $JDK_13_PATTERN )
         {
	   $web_key  = TXK::Inventory::WEB_PATCH_LIST_13;
	   $conc_key = TXK::Inventory::CONC_PATCH_LIST_13;

         }
	elsif ( $installed_jdk_version =~ $JDK_14_PATTERN )
	 {
	   $web_key  = TXK::Inventory::WEB_PATCH_LIST_14;
	   $conc_key = TXK::Inventory::CONC_PATCH_LIST_14;

	 }

	if ( ( $self->isWebNode() ) && ( exists($args->{$web_key}) ) )
         {
           $web_prereq_status = $self->_verifyPatch($args->{$web_key});

         }

	if ( ( $self->isConcNode() ) && ( exists($args->{$conc_key}) ) )
         {
           $conc_prereq_status = $self->_verifyPatch($args->{$conc_key});

         }

        if ( ($web_prereq_status  eq TXK::Error::SUCCESS) &&
	     ($conc_prereq_status eq TXK::Error::SUCCESS)
           )
         {
           $prereq_status = TXK::Error::SUCCESS;

         }
        else
         {
           $prereq_status = TXK::Error::WARNING;

         }

      }

     #
     # Version of the installed JAVA is >= the one passed in.
     # Now, determine if the node is fully configured with the installed JAVA.
     #
     # Verify if the Web node is fully configured with the supplied JDK version
     #

     if ( $self->isWebNode() eq TXK::Util::TRUE )
      {
        $web_status = $self->_verifyConfigJDK($IS_WEB_NODE);

      }

     #
     # Verify if the Concurrent processing node is configured with the
     # supplied JDK version
     #

     if ( $self->isConcNode() eq TXK::Util::TRUE )
      {
        $conc_status = $self->_verifyConfigJDK($IS_CONC_NODE);

      }

     #-------------------------------------------------------------------------
     # Wrap up everything and get ready to return to the caller.
     #-------------------------------------------------------------------------

     if ( ( $web_status    eq TXK::Error::SUCCESS ) &&
          ( $conc_status   eq TXK::Error::SUCCESS )
        )
      {
	if ( $prereq_status eq TXK::Error::SUCCESS )
	 {
	   $config_status = TXK::Error::SUCCESS;
	 }
	else
	 {
	   $config_status = TXK::Error::WARNING;
	 }

      }
     else
      {
        $config_status = TXK::Error::FAIL;

      }

     if ( $config_status ne TXK::Error::SUCCESS )
      {
	my $FLAG;

	if ( ( $web_status  eq TXK::Error::SUCCESS ) &&
             ( $conc_status eq TXK::Error::SUCCESS )
 	   )
	 {
	   $FLAG = "WARN";
	 }
	else
	 {
	   $FLAG = "FAIL";
	 }

	if ( $installed_jdk_version =~ $JDK_13_PATTERN )
         {
           $self->{$VALIDATE_MESSAGE} .= "\n[$FLAG]\t The Web and\/or Concurrent processing node of this E-Business Suite\n\t Release 11i system does not appear already configured\n\t or ready to be configured with JDK $installed_jdk_version.\n\t Please see the My Oracle Support Note 130091.1 for more information on\n\t Upgrading to JDK 1.3 with the Oracle E-Business Suite Release 11i.\n\n";

         }
        elsif ( $installed_jdk_version =~ $JDK_14_PATTERN )
         {
           $self->{$VALIDATE_MESSAGE} .= "\n[$FLAG]\t The Web and\/or Concurrent processing node of this E-Business Suite\n\t Release 11i system does not appear already configured\n\t or ready to be configured with JDK $installed_jdk_version.\n\t Please see the My Oracle Support Note 246105.1 or more information on\n\t Upgrading to J2SE 1.4 with the Oracle E-Business Suite Release 11i.\n\n";

         }
      }
     else
      {
        $self->{$VALIDATE_MESSAGE} .= "\n[PASS]\t The Web and/or Concurrent processing node of this E-Business Suite\n\t Release 11i system appears configured with JDK $installed_jdk_version.\n\n";

      }

     return $config_status;

   }
  else
   { 
     $self->{$VALIDATE_MESSAGE} .= "\n[WARN]\t It is not necessary to verify the JDK version if this\n\t is not the Web or Concurrent processing node.\n";

     return TXK::Error::WARNING;
   }
   
  return TXK::Error::SUCCESS;
}

#------------------------------------------------------------------------------
## _getAppsCtxLocation()
#------------------------------------------------------------------------------

sub _getAppsCtxLocation
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $context_file = TXK::Inventory::NULL;
  my $oracle_sid   = TXK::Inventory::NULL;
  my $dirName      = TXK::Inventory::NULL;
  my $ctxfile_1    = TXK::Inventory::NULL;
  my $ctxfile_2    = TXK::Inventory::NULL;
  my $findStatus   = TXK::Inventory::NULL;

  if ( ( ref( $args ) eq "HASH" ) && ( keys %{$args} ) )
   {
     my $APPS_CTX_FILE_KEY = TXK::Inventory::APPS_CTX_FILE_KEY;

     if ( exists ( $args->{$APPS_CTX_FILE_KEY} ))
      {
	#
	# this means that the context file is specified.
	#

        $context_file = $args->{$APPS_CTX_FILE_KEY};

        my $ctx_status = $self->_isFileFound($context_file);

        if ( $ctx_status eq TXK::Util::FALSE )
         {
           $self->{$ADCONFIG_ENABLED} = TXK::Util::FALSE;

           $self->{$VALIDATE_MESSAGE} .= "\n[NOTE]\t It appears ADCONFIG is not enabled in this E-Business Release 11i\n\t environment, as the supplied Applications context file shown\n\t below cannot be located:\n\t $context_file\n";
 
           return TXK::Error::FAIL;
  
         }
        else
         {
           $self->{$APPS_CTX_FILE} = TXK::OSD->trDirPathToBase($context_file);
  
           return TXK::Error::SUCCESS;
  
         }
      }
   }


  #
  # If we come to this point, no context file is specified.
  # Will have to figure out by using some default settings:
  #
  # 1. First try to get it from the setting of CONTEXT_FILE
  # 2. Else, base it on <CONTEXTNAME>.xml and check for its existence.
  #

  $context_file = TXK::OSD->getEnvVar({ name => "CONTEXT_FILE"});

  if ( $context_file ne TXK::Inventory::NULL )
   {

     $findStatus = $self->_isFileFound($context_file);

     if ( $findStatus eq TXK::Util::TRUE )
      {
        #
        # If we found it, return immediately.
        #

        $self->{$APPS_CTX_FILE} = TXK::OSD->trDirPathToBase($context_file);

        return TXK::Error::SUCCESS;

      }
   }

  #
  # If we come here, CONTEXT_FILE is not defined.
  # Must base on the <CONTEXTNAME>.xml
  #
 
  $oracle_sid = $self->{$TWO_TASK};
   
  $dirName = $self->{$APPL_TOP} . "/admin/";

  $ctxfile_1 = $dirName . $oracle_sid . "_" . $self->{$HOST_NAME} . ".xml";
  $ctxfile_2 = $dirName . $oracle_sid . ".xml";

  #--------------------------------------------------------------------------
  # Our preferred format is <TWO_TASK>_<hostname>.xml
  # If found it, return immediately.
  #--------------------------------------------------------------------------
  
  $findStatus = $self->_isFileFound($ctxfile_1); 

  if ( $findStatus eq TXK::Util::TRUE )
   {
     $self->{$APPS_CTX_FILE} = TXK::OSD->trDirPathToBase($ctxfile_1);

     return TXK::Error::SUCCESS;
   }

  #-------------------------------------------------------------------------
  # Our preferred format is <TWO_TASK>_<hostname>.xml
  # But if it doesn't exist, we try <TWO_TASK>.xml
  #-------------------------------------------------------------------------

  $findStatus = $self->_isFileFound($ctxfile_2);

  if ( $findStatus eq TXK::Util::TRUE )
   {
     $self->{$APPS_CTX_FILE} = TXK::OSD->trDirPathToBase($ctxfile_2);

     return TXK::Error::SUCCESS;
   }

  if ( $findStatus eq TXK::Util::FALSE )
   {
     $self->{$VALIDATE_MESSAGE} .= "\n[NOTE]\t It appears ADCONFIG is not enabled in this E-Business Release 11i\n\t environment, as none of the Applications context files shown\n\t below can be located:\n\t $context_file\n\t $ctxfile_1\n\t $ctxfile_2\n";
 
     return TXK::Error::FAIL;

   }

}

#------------------------------------------------------------------------------
## _isAdConfigEnabled()
#------------------------------------------------------------------------------

sub _isAdConfigEnabled
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  #
  # 1. Get the version of the template first
  # 2. If successful, get the version of the context file
  # 3. If successful, compare the two versions
  # 3. If equal, it is adconfig enabled
  #
  #    Other than that, it is not adconfig enabled.
  #

  my $ad_top  = TXK::OSD->getEnvVar({ name => "AD_TOP"});

  my $adxmlfile = TXK::Inventory::ADXMLCTX_FILE;

  my $adctx = $ad_top . "/admin/template/" . "$adxmlfile";

  my $adctxtmpl = TXK::OSD->trDirPathToBase($adctx);

  my $ctxtmpl_version = $self->_getCtxVersion($adctxtmpl);

  if ( $ctxtmpl_version eq TXK::Inventory::NULL )
   {
     $self->{$ADCONFIG_ENABLED} = TXK::Util::FALSE;

     $self->{$VALIDATE_MESSAGE} .= "\n[NOTE]\t It appears ADCONFIG is not enabled in this E-Business Release 11i\n\t environment, as the version of the Applications Context template\n\t shown  below cannot be determined:\n\t $adctxtmpl.\n";

     return $self->{$ADCONFIG_ENABLED};

   }
  else
   {
     $self->{$CTXTMPL_VERSION} = $ctxtmpl_version;
   }

  my $loc_status = $self->_getAppsCtxLocation($args);

  if ( $loc_status eq TXK::Error::FAIL )
   {
     #
     # Error accessing the context file. Consider that as non-adconfig enabled
     # and return immediately.
     #
     
     $self->{$ADCONFIG_ENABLED} = TXK::Util::FALSE;

     return $self->{$ADCONFIG_ENABLED};     

   }

  my $apps_ctx_version = $self->_getCtxVersion($self->{$APPS_CTX_FILE});

  ##---------------------------------------------------------------------------
  # Now that we have the version of adxmlctx.tmp and the location of the Apps
  # context file. Let's compare the versions.
  # If they are equal, the environment is definitely adconfig enabled.
  ##---------------------------------------------------------------------------

  if ( $ctxtmpl_version ne $apps_ctx_version )
   {
     $self->{$ADCONFIG_ENABLED} = TXK::Util::FALSE;

     my $apps_ctx = $self->{$APPS_CTX_FILE};

     $self->{$VALIDATE_MESSAGE} .= "\n[NOTE]\t It appears ADCONFIG is not enabled in this E-Business Release 11i\n\t environment, as the versions of the Applications Context template\n\t and the Applications Context file shown below are not identical:\n\n\t $adctxtmpl\n\t Version: $ctxtmpl_version\n\n\t $apps_ctx\n\t Version: $apps_ctx_version\n";

     return $self->{$ADCONFIG_ENABLED};

   }
  else
   {
     $self->{$APPS_CTX_VERSION} = $apps_ctx_version;

     $self->{$ADCONFIG_ENABLED} = TXK::Util::TRUE;

     my $apps_ctx = $self->{$APPS_CTX_FILE};

     $self->{$VALIDATE_MESSAGE} .= "\n[NOTE]\t It appears ADCONFIG is enabled in this E-Business Release 11i\n\t environment, as the versions of the Applications Context template\n\t and the Applications Context file shown below are identical:\n\n\t $adctxtmpl\n\t Version: $ctxtmpl_version\n\n\t $apps_ctx\n\t Version: $apps_ctx_version\n";

   }

  return $self->{$ADCONFIG_ENABLED};

}

#------------------------------------------------------------------------------
## isAdConfigEnabled()
#------------------------------------------------------------------------------

sub isAdConfigEnabled
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( !defined($self->{$ADCONFIG_ENABLED}) )
   {
     $self->_isAdConfigEnabled($args);
   }

  return $self->{$ADCONFIG_ENABLED};
}

#------------------------------------------------------------------------------
## _getNodeInfo()
#------------------------------------------------------------------------------

sub _getNodeInfo
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $isDBNode;
  my $isAdminNode;
  my $isWebNode;
  my $isFormsNode;
  my $isConcNode;

  if ( $self->isAdConfigEnabled() eq TXK::Util::TRUE )
   {
     my $xmlobj = TXK::XML->new();
     
     $xmlobj->loadDocument ( { file => $self->{$APPS_CTX_FILE} } );

     $isDBNode    = $xmlobj->getOAVar('s_isDB');

     $isAdminNode = $xmlobj->getOAVar('s_isAdmin');

     $isWebNode   = $xmlobj->getOAVar('s_isWeb');

     $isFormsNode = $xmlobj->getOAVar('s_isForms');

     $isConcNode  = $xmlobj->getOAVar('s_isConc');
   }
  else
   {
     my $apps_user     = TXK::Inventory::APPS_USER;
     my $apps_password;

     if ( defined( $self->{$APPS_PASSWORD} ) )
      {
	$apps_password = $self->{$APPS_PASSWORD};
      }
     else
      {
	$apps_password = TXK::Inventory::APPS_DEFAULT_PWD;
      }

     my $appl_top  = TXK::OSD->getEnvVar({ name => "APPL_TOP"});

     my $appllog   = TXK::OSD->getEnvVar({ name => "APPLLOG"});

     my $log_f     = $appl_top . "/admin/" . $appllog . "/" . "node_info.txt";

     my $logfile   = TXK::OSD->trDirPathToBase($log_f);

     my $spath  = TXK::OSD->getEnvVar({ name => "FND_TOP"}) . "/patch/115/sql/";

     my $s_path = TXK::OSD->trDirPathToBase($spath);

     my $sql    = $s_path . "txkGetNodeInfo.sql";

     my $node   = $self->{$HOST_NAME};

     my $sid    = $self->{$TWO_TASK};

     my $sqlctx = TXK::SQLPLUS->new();

     my $fsys   = TXK::FileSys->new();

     $fsys->access ({
		      fileName	=> $sql,
                      type	=> TXK::FileSys::FILE,
		      checkMode	=> TXK::FileSys::READ_ACCESS
		   })
     or TXK::Error->stop("Check File(s) Error", $fsys->getError());

     $sqlctx->setConnectInfo({
				user     => "$apps_user",
				password => "$apps_password",
				two_task => "$sid"
			    });

     $sqlctx->clearCommand();

     $sqlctx->addCommand("spool " . "$logfile");

     $sqlctx->addCommand("@" . "$sql " . "$node");

     $sqlctx->execute({ showOutput =>TXK::Util::FALSE, });

     my $io = TXK::IO->new();

     $io->open({ fileName => $logfile });

     my $io_ref  = $io->getFileHandle();
     my @io_data = <$io_ref>;

     $io->close();

     unlink($logfile) or return TXK::Error::FAIL;

     foreach (@io_data)
      {

        if ( /\s*SUPPORT_CP\s*=\s*TRUE/ )
         {
           $isConcNode = "yes";
         }
      
        if ( /\s*SUPPORT_FORMS\s*=\s*TRUE/ )
         {
           $isFormsNode = "yes";
         }
        
        if ( /\s*SUPPORT_WEB\s*=\s*TRUE/ )
         {
           $isWebNode = "yes";
         }
        
        if ( /\s*SUPPORT_ADMIN\s*=\s*TRUE/ )
         {
           $isAdminNode = "yes";
         }
      }    
   }

  $isDBNode    =~ tr /a-z/A-Z/;
  $isAdminNode =~ tr /a-z/A-Z/;
  $isWebNode   =~ tr /a-z/A-Z/;
  $isFormsNode =~ tr /a-z/A-Z/;
  $isConcNode  =~ tr /a-z/A-Z/;


  if ( $isDBNode eq "YES" )
   {
     $self->{$IS_DB_NODE} = TXK::Util::TRUE;
   }

  if ( $isAdminNode eq "YES" )
   {
     $self->{$IS_ADMIN_NODE} = TXK::Util::TRUE;
   }

  if ( $isWebNode eq "YES" )
   {
     $self->{$IS_WEB_NODE} = TXK::Util::TRUE;
   }

  if ( $isFormsNode eq "YES" )
   {
     $self->{$IS_FORMS_NODE} = TXK::Util::TRUE;
   }

  if ( $isConcNode eq "YES" )
   {
     $self->{$IS_CONC_NODE} = TXK::Util::TRUE;
   }

}

#------------------------------------------------------------------------------
## isWebNode()
#------------------------------------------------------------------------------

sub isWebNode
{
  my $self = $ARG[0];

  return $self->{$IS_WEB_NODE};
}

#------------------------------------------------------------------------------
## isFormsNode()
#------------------------------------------------------------------------------

sub isFormsNode
{
  my $self = $ARG[0];

  return $self->{$IS_FORMS_NODE};
}

#------------------------------------------------------------------------------
## isAdminNode()
#------------------------------------------------------------------------------

sub isAdminNode
{
  my $self = $ARG[0];

  return $self->{$IS_ADMIN_NODE};
}

#------------------------------------------------------------------------------
## isDBNode()
#------------------------------------------------------------------------------

sub isDBNode
{
  my $self = $ARG[0];

  return $self->{$IS_DB_NODE};
}

#------------------------------------------------------------------------------
## isConcNode()
#------------------------------------------------------------------------------

sub isConcNode
{
  my $self = $ARG[0];

  return $self->{$IS_CONC_NODE};
}

#------------------------------------------------------------------------------
## getAppsCtxLocation()
#------------------------------------------------------------------------------

sub getAppsCtxLocation
{
  my $self = $ARG[0];

  return $self->{$APPS_CTX_FILE};
}

#------------------------------------------------------------------------------
## _verifyConfigJDK()
#------------------------------------------------------------------------------

sub _verifyConfigJDK
{
  my $self = $ARG[0];
  my $tier = $ARG[1];

  my $ret_status = TXK::Error::SUCCESS;

  my $web_ret_status  = TXK::Error::SUCCESS;
  my $conc_ret_status = TXK::Error::SUCCESS;

  my $JDBC_FILE;

  my $CHARSETS_FILE;

  #
  # Verify if the Web node is configured with JDK version x
  #

  if ( $tier eq $IS_WEB_NODE )
   {
     my $web_home = $self->{$IAS_ORACLE_HOME};
     my $jserv_propPath;
     my $jserv_propFile;
     my $pattern_1;
     my $webHeaderPrinted = TXK::Util::FALSE;

     my $installed_jdk = $self->_getJDKversion();

     if ( $installed_jdk =~ $JDK_13_PATTERN )
      {
        $JDBC_FILE = $JDBC_FILE_12;
      }
     elsif ( $installed_jdk =~ $JDK_14_PATTERN )
      {
        $JDBC_FILE = $JDBC_FILE_14;
      }

     #-------------------------------------------------------------------------
     # 1. Determine if the appropriate JDBC zip file is in the CLASSPATH
     #    as defined in the jserv.properties.
     #-------------------------------------------------------------------------

     if ( TXK::OSD->isNT() )
      {
        $pattern_1 = "${JSERV_PATTERN_1}.*\\${JDBC_FILE}";
      }
     else
      {
        $pattern_1 = "${JSERV_PATTERN_1}.*\/${JDBC_FILE}";
      }

     my $jserv_propPath_f = $web_home . "/Apache/Jserv/etc/";

     $jserv_propPath = TXK::OSD->trDirPathToBase($jserv_propPath_f);

     $jserv_propFile = $jserv_propPath . "jserv.properties";

     my $io = TXK::IO->new();

     $io->open({ fileName => $jserv_propFile });

     my $io_ref = $io->getFileHandle();

     my @io_data = <$io_ref>;

     $io->close();

     if ( grep(/$pattern_1/, @io_data ) )
      {
        $web_ret_status = ($web_ret_status && TXK::Error::SUCCESS);

        $webHeaderPrinted = TXK::Util::TRUE;

        $self->{$VALIDATE_MESSAGE} .= "\n[PASS]\t The setting of the CLASSPATH environment in\n\t $jserv_propFile\n\t contains the following value:\n\n\t wrapper.classpath=$self->{$JAVA_TOP}/$JDBC_FILE\n";

      }
     else
      {
        $web_ret_status = ($web_ret_status && TXK::Error::FAIL);

        $webHeaderPrinted = TXK::Util::TRUE;

        $self->{$VALIDATE_MESSAGE} .= "\n[FAIL]\t The setting of the CLASSPATH environment in\n\t $jserv_propFile\n\t does not contain the following value:\n\n\t wrapper.classpath=$self->{$JAVA_TOP}/$JDBC_FILE\n";

      }

     #-------------------------------------------------------------------------
     # 2. Determine if the appropriate jar files are in the CLASSPATH
     #    as defined in the jserv.properties.
     #-------------------------------------------------------------------------

     my $pattern_2;
     my $jdk_top = $self->{$JDK_TOP};

     if ( TXK::OSD->isNT() )
      {
	$jdk_top =~ s/\\/\\\\/g;

      }

     $jdk_top = qr/$jdk_top/;

     if ( $installed_jdk =~ $JDK_13_PATTERN )
      {
        $CHARSETS_FILE = $JSERV_I18N_JAR;
      }
     elsif ( $installed_jdk =~ $JDK_14_PATTERN )
      {
        $CHARSETS_FILE = $JSERV_CHARSETS_JAR;
      }

     #
     # Determine if CLASSPATH contains dt.jar
     #

     $pattern_2 = "${JSERV_PATTERN_1}.*${jdk_top}.*${JSERV_DT_JAR}";

     if ( grep(/$pattern_2/, @io_data ) )
      {
        $web_ret_status = ($web_ret_status && TXK::Error::SUCCESS);

        my $dt = TXK::OSD->trDirPathToBase("/lib/dt.jar");

        $self->{$VALIDATE_MESSAGE} .= "\n[PASS]\t The setting of the CLASSPATH environment in\n\t $jserv_propFile\n\t contains the following value:\n\n\t wrapper.classpath=$self->{$JDK_TOP}$dt\n";

      }
     else
      {
        $web_ret_status = ($web_ret_status && TXK::Error::FAIL);

        my $dt = TXK::OSD->trDirPathToBase("/lib/dt.jar");

        $self->{$VALIDATE_MESSAGE} .= "\n[FAIL]\t The setting of the CLASSPATH environment in\n\t $jserv_propFile\n\t does not contain the following value:\n\n\t wrapper.classpath=$self->{$JDK_TOP}$dt\n";

      }

     #
     # Determine if CLASSPATH contains tools.jar
     #

     $pattern_2 = "${JSERV_PATTERN_1}.*${jdk_top}.*${JSERV_TOOLS_JAR}";

     if ( grep(/$pattern_2/, @io_data ) )
      {
        $web_ret_status = ($web_ret_status && TXK::Error::SUCCESS);

        my $tools_jar = TXK::OSD->trDirPathToBase("/lib/tools.jar");

        $self->{$VALIDATE_MESSAGE} .= "\n[PASS]\t The setting of the CLASSPATH environment in\n\t $jserv_propFile\n\t contains the following value:\n\n\t wrapper.classpath=$self->{$JDK_TOP}$tools_jar\n";

      }
     else
      {
        $web_ret_status = ($web_ret_status && TXK::Error::FAIL);

        my $tools_jar = TXK::OSD->trDirPathToBase("/lib/tools.jar");

        $self->{$VALIDATE_MESSAGE} .= "\n[FAIL]\t The setting of the CLASSPATH environment in\n\t $jserv_propFile\n\t does not contain the following value:\n\n\t wrapper.classpath=$self->{$JDK_TOP}$tools_jar\n";

      }

     #
     # Determine if CLASSPATH contains rt.jar
     #

     $pattern_2 = "${JSERV_PATTERN_1}.*${jdk_top}.*${JSERV_RT_JAR}";

     if ( grep(/$pattern_2/, @io_data ) )
      {
        $web_ret_status = ($web_ret_status && TXK::Error::SUCCESS);

        my $rt = TXK::OSD->trDirPathToBase("/jre/lib/rt.jar");

        $self->{$VALIDATE_MESSAGE} .= "\n[PASS]\t The setting of the CLASSPATH environment in\n\t $jserv_propFile\n\t contains the following value:\n\n\t wrapper.classpath=$self->{$JDK_TOP}$rt\n";

      }
     else
      {
        $web_ret_status = ($web_ret_status && TXK::Error::FAIL);

        my $rt = TXK::OSD->trDirPathToBase("/jre/lib/rt.jar");

        $self->{$VALIDATE_MESSAGE} .= "\n[FAIL]\t The setting of the CLASSPATH environment in\n\t $jserv_propFile\n\t does not contain the following value:\n\n\t wrapper.classpath=$self->{$JDK_TOP}$rt\n";

      }

     #
     # Determine if CLASSPATH contains charsets.jar or i18n.jar
     #

     $pattern_2 = "${JSERV_PATTERN_1}.*${jdk_top}.*${CHARSETS_FILE}";

     if ( grep(/$pattern_2/, @io_data ) )
      {
        $web_ret_status = ($web_ret_status && TXK::Error::SUCCESS);

        my $charset;

        if ( $installed_jdk =~ $JDK_13_PATTERN )
         {
           $charset = TXK::OSD->trDirPathToBase("/jre/lib/i18n.jar");

         }
        elsif ( $installed_jdk =~ $JDK_14_PATTERN )
         {
	   $charset = TXK::OSD->trDirPathToBase("/jre/lib/charsets.jar");	

         }

        $self->{$VALIDATE_MESSAGE} .= "\n[PASS]\t The setting of the CLASSPATH environment in\n\t $jserv_propFile\n\t contains the following value:\n\n\t wrapper.classpath=$self->{$JDK_TOP}$charset\n";
	 
      }
     else
      {
        $web_ret_status = ($web_ret_status && TXK::Error::FAIL);

        my $charset;

        if ( $installed_jdk =~ $JDK_13_PATTERN )
         {
           $charset = TXK::OSD->trDirPathToBase("/jre/lib/i18n.jar");

         }
        elsif ( $installed_jdk =~ $JDK_14_PATTERN )
         {
	   $charset = TXK::OSD->trDirPathToBase("/jre/lib/charsets.jar");	

         }
	 
         $self->{$VALIDATE_MESSAGE} .= "\n[FAIL]\t The setting of the CLASSPATH environment in\n\t $jserv_propFile\n\t does not contain the following value:\n\n\t wrapper.classpath=$self->{$JDK_TOP}$charset\n";

      }
     

   }
  elsif ( $tier eq $IS_CONC_NODE )
   {
     #
     #    Verify if the Web node is configured with JDK version x
     #

     my $af_jvaprg     = TXK::OSD->getEnvVar({ name => "AFJVAPRG"});
     my $af_jre_top    = TXK::OSD->getEnvVar({ name => "AF_JRE_TOP"});
     my $af_classpath  = TXK::OSD->getEnvVar({ name => "AF_CLASSPATH"});

     my $pattern_1 = $self->{$JDK_TOP};
     my $pattern_2 = $APPSBORG_FILE;

     my $ConcHeaderPrinted = TXK::Util::FALSE;

     if ( TXK::OSD->isNT() )
      {
	$pattern_1    =~ s/\\/\\\\/g;
	$pattern_2    =~ s/\\/\\\\/g;

      }

     if ( $af_jvaprg =~ qr/${pattern_1}/ )
      {
        $self->{$VALIDATE_MESSAGE} .= "\n[PASS]\t The current setting of AFJVAPRG is:\n\n\t $self->{$JDK_TOP}\n";

        $conc_ret_status = ($conc_ret_status && TXK::Error::SUCCESS);

      }
     else
      {
        $self->{$VALIDATE_MESSAGE} .= "\n[FAIL]\t The current setting of AFJVAPRG is not:\n\n\t $self->{$JDK_TOP}\n";

        $conc_ret_status = ($conc_ret_status && TXK::Error::FAIL);

      }

     if ( $af_jre_top =~ qr/${pattern_1}/ )
      {
        $self->{$VALIDATE_MESSAGE} .= "\n[PASS]\t The current setting of AF_JRE_TOP is:\n\n\t $self->{$JDK_TOP}\n";

        $conc_ret_status = ($conc_ret_status && TXK::Error::SUCCESS);

      }
     else
      {
        $self->{$VALIDATE_MESSAGE} .= "\n[FAIL]\t The current setting of AF_JRE_TOP is not:\n\n\t $self->{$JDK_TOP}\n";

        $conc_ret_status = ($conc_ret_status && TXK::Error::FAIL);

      }

     if ( $af_classpath =~ qr/${pattern_2}/ )
      {
        $self->{$VALIDATE_MESSAGE} .= "\n[PASS]\t The current setting of AF_CLASSPATH contains\n\t the following value:\n\n\t $pattern_2\n";

        $conc_ret_status = ($conc_ret_status && TXK::Error::SUCCESS);

      }
     else
      {
        $self->{$VALIDATE_MESSAGE} .= "\n[FAIL]\t The current setting of AF_CLASSPATH does not contain\n\t the following value:\n\n\t $pattern_2\n";

        $conc_ret_status = ($conc_ret_status && TXK::Error::FAIL);

      }

   }

  $ret_status = ($web_ret_status && $conc_ret_status);

  return $ret_status;

}

#------------------------------------------------------------------------------
## _getCtxVersion()
#------------------------------------------------------------------------------

sub _getCtxVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $fsys = TXK::FileSys->new();  

  my @identData;       
  my $rc      = TXK::Inventory::NULL;

  if ( $args eq TXK::Inventory::NULL )
   {
     return TXK::Inventory::NULL;
   }

  $fsys->abortOnError ( { enable => TXK::Util::FALSE } );

  $rc = $fsys->ident({
                       fileName => $args,
                       identData => \@identData,
                    });

  if ( ( $rc eq TXK::Error::FAIL ) || ( $rc eq TXK::FileSys::NO_IDENT ) )
   {   
     return TXK::Inventory::NULL;
   }
  elsif ( $rc eq TXK::FileSys::MULTIPLE_IDENTS )
   {
     my $header;
       
     foreach $header (@identData)
      {
        $rc = $header->{TXK::FileSys::IDENT_VERSION};
      }

     return $rc;
   }
  else
   {
     return $rc;
   }

}

#------------------------------------------------------------------------------
## _isFileFound()
#------------------------------------------------------------------------------

sub _isFileFound
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $io;

  my $f        = TXK::Inventory::NULL;
  my $status   = TXK::Util::FALSE;

  if ( $args eq TXK::Inventory::NULL )
   {
     return TXK::Util::FALSE;
   }

  $f = TXK::OSD->trDirPathToBase($args);

  $io = TXK::IO->new();

  $io->abortOnError ({ enable => TXK::Util::FALSE });

  $status = $io->open ({
			 fileName	=> $f,
			 mode		=> TXK::IO::RDWR
		      });

  $io->abortOnError ({ enable => TXK::Util::TRUE }); 

  if ( $status eq TXK::Error::SUCCESS )
   {
     $io->close();
   }

  if ( $status eq TXK::Error::SUCCESS )
   {
     return TXK::Util::TRUE;
   }
  else
   {
     return TXK::Util::FALSE;
   }

}

#------------------------------------------------------------------------------
## printStatus()
#------------------------------------------------------------------------------

sub printStatus
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( $args eq $VALIDATE_MESSAGE )
   {
     print "$self->{$VALIDATE_MESSAGE}";
   }

}

#------------------------------------------------------------------------------
## _isPatchApplied()
#------------------------------------------------------------------------------

sub _verifyPatch
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $patch_status  = TXK::Error::SUCCESS;

  foreach my $patch ( @{$args} )
   {
     if ( $self->_isPatchApplied($patch) )
      {
	$patch_status = ($patch_status && TXK::Error::SUCCESS);

        $self->{$VALIDATE_MESSAGE} .= "\n[PASS]\t It appears that patch $patch was already applied.\n";
 
      }
     else
      {
        $patch_status = ($patch_status && TXK::Error::WARNING);

        $self->{$VALIDATE_MESSAGE} .= "\n[WARN]\t It appears that patch $patch has not been applied.\n";

      }

   }

  return $patch_status;

}

#------------------------------------------------------------------------------
## _isPatchApplied()
#------------------------------------------------------------------------------

sub _isPatchApplied
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $patch_applied = TXK::Util::FALSE;
  my $apps_user     = TXK::Inventory::APPS_USER;
  my $apps_password;

  if ( defined( $self->{$APPS_PASSWORD} ) )
   {
     $apps_password = $self->{$APPS_PASSWORD};
   }
  else
   {
     $apps_password = TXK::Inventory::APPS_DEFAULT_PWD;
   }

  my $appl_top  = TXK::OSD->getEnvVar({ name => "APPL_TOP"});

  my $appllog   = TXK::OSD->getEnvVar({ name => "APPLLOG"});

  my $log_f     = $appl_top . "/admin/" . $appllog . "/" . "patch_info.txt";

  my $logfile   = TXK::OSD->trDirPathToBase($log_f);

  my $spath  = TXK::OSD->getEnvVar({ name => "FND_TOP"}) . "/patch/115/sql/";

  my $s_path = TXK::OSD->trDirPathToBase($spath);

  my $sql    = $s_path . "txkVerifyPatch.sql";

  my $node   = $self->{$HOST_NAME};

  my $sid    = $self->{$TWO_TASK};

  my $sqlctx = TXK::SQLPLUS->new();

  my $fsys   = TXK::FileSys->new();

  $fsys->access ({
                   fileName  => $sql,
                   type      => TXK::FileSys::FILE,
                   checkMode => TXK::FileSys::READ_ACCESS
                })
  or TXK::Error->stop("Check File(s) Error", $fsys->getError());

  $sqlctx->setConnectInfo({
                             user     => "$apps_user",
                             password => "$apps_password",
                             two_task => "$sid"
                         });

  $sqlctx->clearCommand();

  $sqlctx->addCommand("spool " . "$logfile");

  $sqlctx->addCommand("@" . "$sql " . "$args " . "$node");

  $sqlctx->execute({ showOutput =>TXK::Util::FALSE, });

  my $io = TXK::IO->new();

  $io->open({ fileName => $logfile });

  my $io_ref  = $io->getFileHandle();
  my @io_data = <$io_ref>;

  $io->close();

  unlink($logfile);

  if ( grep(/$args/, @io_data ) )
   {
     $patch_applied = TXK::Util::TRUE;
     
   }
  else
   {
     $patch_applied = TXK::Util::FALSE;

   }

  return $patch_applied;

}

#------------------------------------------------------------------------------
## _getAppsVersion()
#------------------------------------------------------------------------------

sub _getAppsVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $apps_user     = TXK::Inventory::APPS_USER;
  my $apps_password;
  my $apps_version;
  my $pattern = qr/11\.5\.(\d)+/;

  if ( defined( $self->{$APPS_PASSWORD} ) )
   {
     $apps_password = $self->{$APPS_PASSWORD};
   }
  else
   {
     $apps_password = TXK::Inventory::APPS_DEFAULT_PWD;
   }

  my $appl_top  = TXK::OSD->getEnvVar({ name => "APPL_TOP"});

  my $appllog   = TXK::OSD->getEnvVar({ name => "APPLLOG"});

  my $log_f     = $appl_top . "/admin/" . $appllog . "/" . "apps_ver.txt";

  my $logfile   = TXK::OSD->trDirPathToBase($log_f);

  my $sid    = $self->{$TWO_TASK};

  my $sqlctx = TXK::SQLPLUS->new();

  $sqlctx->setConnectInfo({
                             user     => "$apps_user",
                             password => "$apps_password",
                             two_task => "$sid"
                         });

  $sqlctx->clearCommand();

  $sqlctx->addCommand("set heading off");
  $sqlctx->addCommand("set echo off");
  $sqlctx->addCommand("set verify off");

  $sqlctx->addCommand("spool " . "$logfile");

  $sqlctx->addCommand("select RELEASE_NAME from fnd_product_groups");

  $sqlctx->execute({ showOutput =>TXK::Util::FALSE, });

  my $io = TXK::IO->new();

  $io->open({ fileName => $logfile });

  my $io_ref  = $io->getFileHandle();
  my @io_data = <$io_ref>;

  $io->close();

  unlink($logfile);

  foreach (@io_data)
   {
     chomp;

     if ( /$pattern/ )
      {
        $apps_version = $_;
     
      }
   }

  return $apps_version;
}

#------------------------------------------------------------------------------
## getAppsVersion()
#------------------------------------------------------------------------------

sub getAppsVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( !defined($self->{$APPS_VERSION}) )
   {
     $self->{$APPS_VERSION} = $self->_getAppsVersion();

   }

  return $self->{$APPS_VERSION};

}
1;
