
# +===========================================================================+
# |   Copyright (c) 2003 Oracle Corporation, Redwood Shores, California, USA
# |                         All Rights Reserved
# |                        Applications Division
# +===========================================================================+
# |
# | FILENAME
# |   Log.pm
# |
# | DESCRIPTION
# |      TXK Log package
# |
# | USAGE
# |       See Log.html
# |
# | PLATFORM
# |
# | NOTES
# |
# +===========================================================================+

# $Header: Log.pm 03-aug-2005.08:54:53 ndutko Exp $

package TXK::Log;

@ISA = qw( TXK::Common );

######################################
# Standard Modules
######################################

use strict;
use English;
use Carp;

require 5.005;

######################################
# Package Specific Modules
######################################

use Cwd;

use TXK::Error();
use TXK::Util();
use TXK::IO();

######################################
# Public Constants
######################################

######################################
# Private Constants
######################################

######################################
# Package Variables 
######################################

my $PACKAGE_ID = "TXK::Log";

my $TXK_LOG_DIR = "TXK_LOG_DIR";

######################################
# Object Keys
######################################

my $IS_OPEN        = "isOpen";
my $FILE_NAME      = "fileName";
my $DIR_NAME       = "dirName";
my $AUTO_FLUSH     = "autoFlush";
my $IO_OBJ	   = "ioObj";
my $OPEN_MODE      = "mode";

######################################
#
# Object Structure
# ----------------
#
#  Hash Array
#
######################################

######################################
# Package Methods 
#
# Public
#
#	new 	- build empty object
#
######################################

sub new;
sub DESTROY;
sub open;
sub close;
sub getFileHandle;
sub isOpen;
sub print;
sub println;
sub getDefaultDir;

######################################
# Package Methods
# 
# Private
#       All private methods are marked with a leading underscore.
#
######################################

######################################
# Constructor
######################################

sub new {
  my $type = $ARG[0];

  my $self = TXK::Common->new();

  bless $self, $PACKAGE_ID ;

  my $key;

  my %INIT_OBJ = (
		   PACKAGE_IDENT   => $PACKAGE_ID,
                   $IS_OPEN        => "0",
                   $FILE_NAME      => undef,
                   $DIR_NAME       => undef,
                   $AUTO_FLUSH     => undef,
	           $IO_OBJ 	   => undef,
                  );

  foreach $key (keys %INIT_OBJ)
   { 
     $self->{$key} = $INIT_OBJ{$key};
   } 

  return $self;
}

######################################
# Destructor
######################################

sub DESTROY
{
}

######################################
# Open
######################################

sub open
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  TXK::Util->isValidArgs({args=>$args,reqd=>["$FILE_NAME"]});

  return $self->setError("Log file already opened") if ( $self->{$IS_OPEN} );

  $self->{$IO_OBJ} = TXK::IO->new();

  $args->{$AUTO_FLUSH} = TXK::Util::TRUE;

  $args->{$OPEN_MODE}  = TXK::IO::WRITE
                         unless ( exists $args->{$OPEN_MODE} &&
                                  $args->{$OPEN_MODE} == TXK::IO::APPEND );

  $self->{$IO_OBJ}->open($args);

  $self->{$IS_OPEN} = TXK::Util::TRUE;
 
  return TXK::Error::SUCCESS;
}

######################################
# close
######################################

sub close
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  return $self->setError("Log file not open") if ( ! $self->{$IS_OPEN} );

  $self->{$IO_OBJ}->close();

  $self->{$IS_OPEN} = TXK::Util::FALSE;

  return TXK::Error::SUCCESS;
}

######################################
# getFileHandle
######################################

sub getFileHandle
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  return $self->setError("Log file not open") if ( ! $self->{$IS_OPEN} );

  return $self->{$IO_OBJ}->getFileHandle();
}

######################################
# isOpen
######################################

sub isOpen
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];
  
  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});
  
  return $self->{$IS_OPEN};
} 

######################################
# print
######################################

sub print
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];
  
  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});
  
  return $self->setError("Log file not open") if ( ! $self->{$IS_OPEN} );
  
  my $msg = ( !ref($args) 
                 ? $args
                 : ( ref($args) eq "SCALAR" ? $$args : "" )
            );

  $self->{$IO_OBJ}->print($msg);
  
  return TXK::Error::SUCCESS;
}

######################################
# println
######################################

sub println
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  $self->print($args);
  $self->print("\n");
 
  return TXK::Error::SUCCESS;
}

######################################
# Get Default Log Dir
######################################

sub getDefaultDir
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID,
                         args=>$args});

  return $ENV{$TXK_LOG_DIR} if exists $ENV{$TXK_LOG_DIR};

  return Cwd->cwd();
}

######################################
# End of Public methods
######################################

# ==========================================================================

1;

