
# +===========================================================================+
# |   Copyright (c) 2003 Oracle Corporation, Redwood Shores, California, USA
# |                         All Rights Reserved
# |                        Applications Division
# +===========================================================================+
# |
# | FILENAME
# |   OSD.pm
# |
# | DESCRIPTION
# |      TXK OSD package
# |
# | USAGE
# |       See OSD.html
# |
# | PLATFORM
# |
# | NOTES
# |
# +===========================================================================+

# $Header: OSD.pm 03-aug-2005.08:54:53 ndutko Exp $

package TXK::OSD;

@ISA = qw( TXK::Common );

######################################
# Standard Modules
######################################

use strict;
use English;
use Carp;

require 5.005;

######################################
# Package Specific Modules
######################################

use Sys::Hostname();
use TXK::Common();

######################################
# Public Constants
######################################

use constant GENERIC      => "Generic";
use constant UNIX_GENERIC => "UNIX-G";

## OS specific constants

use constant UNKNOWN_OS_TYPE   => "UNKNOWN-OS-TYPE";
use constant UNKNOWN_OS_VERSION=> "UNKNOWN-OS-VERSION";
use constant UNKNOWN_OS_RELEASE=> "UNKNOWN-OS-RELEASE";
use constant UNKNOWN_OS_DISTRIBUTION => "UNKNOWN-OS-DISTRIBUTION";
use constant UNKNOWN_BUGDB_PORT_ID =>"UNKNOWN-BUGDB-PORT-ID";

#  Distribution

use constant LINUX_REDHAT      => "Red Hat Linux";
use constant LINUX_SUSE        => "SuSe Linux";

#  Version

use constant LINUX_REDHAT_AS21 => "2.1";
use constant LINUX_REDHAT_AS30 => "3.0";
use constant LINUX_REDHAT_AS40 => "4.0";
use constant LINUX_SUSE_70     => "7.0";

#  Type

use constant LINUX_32          => "LINUX 32-bit";
use constant LINUX_64	       => "LINUX 64-bit";
use constant LINUX_ITANIUM     => "LINUX Itanium";

use constant WINDOWS_32        => "Windows 32-bit";
use constant WINDOWS_64        => "Windows 64-bit";

use constant HPUX_ITANIUM      => "HPUX-Itanium";
use constant HPUX_32	       => "HPUX-32bit";
use constant HPUX_64           => "HPUX-64bit";

use constant AIX_5L	       => "AIX-5L";
use constant AIX_64	       => "AIX-64";
use constant AIX_32	       => "AIX-32bit";

use constant SOLARIS_32	       => "SOLARIS 32-bit";
use constant SOLARIS_64        => "SOLARIS 64-bit";

use constant TRU64	       => "TRU-64 OSF1";

######################################
# Package Variables 
######################################

my $PACKAGE_ID = "TXK::OSD";

######################################
# Object Keys
######################################

my $PLATFORM 		= "platform";
my $PERL_PLATFORM	= "perlPlatform";
my $AUTOBUILD_PLATFORM  = "autobuildPlatform";
my $AUTOCONFIG_PLATFORM = "autoConfigPlatform";
my $HOST_NAME	        = "hostName";
my $ENV_NAME		= "name";
my $ENV_VALUE		= "value";
my $ENV_TRANSLATE	= "translate";
my $OS_TYPE             = "os_type";
my $OS_VERSION          = "os_version";
my $OS_RELEASE          = "os_release";
my $OS_DISTRIBUTION     = "os_distribution";
my $BUGDB_PORTID	= "bugDB_portId";

my $IS_LINUX 		= "LINUX";
my $IS_HPUX  		= "HPUX";
my $IS_AIX     		= "AIX";
my $IS_OSF1    		= "OSF1";
my $IS_NT      		= "NT";
my $IS_WINDOWS 		= "WINDOWS";
my $IS_SOLARIS 		= "SOLARIS";
my $IS_UNIX_G		= "UNIX-Generic";
my $IS_GENERIC		= GENERIC;
my $DIR_SEPARATOR	= "DIR_SEPARATOR";
my $BASE_DIR_SEPARATOR	= "BASE_DIR_SEPARATOR";
my $PATH_SEPARATOR      = "PATH_SEPARATOR";
my $CLASS_PATH	        = "classPath";
my $CMD_PATH	        = "commandPath";
my $SCRIPT_EXT	        = "scriptExt";
my $ALT_SCRIPT_EXT	= "altScriptExt";
my $EXE_EXT		= "executableExt";

######################################
#
# Object Structure
# ----------------
#
#  Hash Array
#
######################################

######################################
# OSD has a default obj , as most
# methods should be static.
######################################

my $classSelf = TXK::OSD->new();
my $baseClassSelf = undef;

######################################
# Package Methods 
#
# Public
#
#	new 	- build empty object
#
######################################

sub new;
sub DESTROY;
sub getName;
sub getPerlName;
sub getHostName;
sub getAutoBuildName;
sub getAutoConfigGenericName;
sub getAutoConfigName;
sub getOSInfo;
sub getOSType;
sub getOSVersion;
sub getOSRelease;
sub getOSDistribution;
sub getBugDBPortId;
sub isLinux;
sub isSolaris;
sub isAIX;
sub isHPUX;
sub isTru64;
sub isNT;
sub isWindows;
sub isUNIX;
sub getDirSeparator;
sub getBaseDirSeparator;
sub getPathSeparator;
sub trDirPath;
sub trFileName;
sub trDirPathFromBase;
sub trDirPathToBase;
sub trFileDir;
sub getBaseName;
sub getDirName;
sub isValidAutoBuildPlatform;
sub setClassPath;
sub addClassPath;
sub getClassPath;
sub setCommandPath;
sub addCommandPath;
sub getCommandPath;
sub getEnvVar;
sub setEnvVar;
sub getCommandScriptExt;
sub getAltCommandScriptExt;
sub getExecutableExt;
sub chdir;

######################################
# Package Methods
# 
# Private
#       All private methods are marked with a leading underscore.
#
######################################

sub _findOSType;
sub _setOS;
sub _resetOS;

######################################
# Constructor
######################################

sub new {
  my $type = $ARG[0];

  my $self = TXK::Common->new();

  bless $self, $PACKAGE_ID ;

  my $key;

  my %INIT_OBJ = (
		   PACKAGE_IDENT   	=> $PACKAGE_ID,
                   $PLATFORM		=> undef,
	           $PERL_PLATFORM  	=> undef,
		   $HOST_NAME      	=> undef,
		   $AUTOBUILD_PLATFORM 	=> undef,
		   $AUTOCONFIG_PLATFORM => undef,
                   $DIR_SEPARATOR  	=> undef,
	           $BASE_DIR_SEPARATOR 	=> undef,
                   $PATH_SEPARATOR	=> undef,
		   $OS_TYPE             => TXK::OSD::UNKNOWN_OS_TYPE(),
		   $OS_VERSION          => TXK::OSD::UNKNOWN_OS_VERSION(),
		   $OS_RELEASE          => TXK::OSD::UNKNOWN_OS_RELEASE(),
		   $OS_DISTRIBUTION     => TXK::OSD::UNKNOWN_OS_DISTRIBUTION(),
		   $CLASS_PATH     	=> "",
		   $CMD_PATH       	=> "",
                   $SCRIPT_EXT     	=> "",
                   $ALT_SCRIPT_EXT 	=> "",
		   $EXE_EXT		=> "",
	           $BUGDB_PORTID        => TXK::OSD::UNKNOWN_BUGDB_PORT_ID(),
                  );

  foreach $key (keys %INIT_OBJ)
   {
     $self->{$key} = $INIT_OBJ{$key};
   }

  $self->{$PERL_PLATFORM} = $OSNAME;
  $self->{$HOST_NAME} = Sys::Hostname::hostname();

  $self->{$DIR_SEPARATOR} = "/";		# UNIX defaults
  $self->{$BASE_DIR_SEPARATOR} = "/";
  $self->{$PATH_SEPARATOR} = ':';
  $self->{$SCRIPT_EXT} = "sh";
  $self->{$ALT_SCRIPT_EXT} = "csh";
  $self->{$EXE_EXT} = "";

  if ( $OSNAME eq "MSWin32" )
   {
     $self->{$PLATFORM} = "$IS_NT";
     $self->{$AUTOBUILD_PLATFORM} = "winnt";
     $self->{$AUTOCONFIG_PLATFORM} = "NT";
     $self->{$DIR_SEPARATOR} = "/";
     $self->{$BASE_DIR_SEPARATOR} = "\\";
     $self->{$PATH_SEPARATOR} = ';';
     $self->{$SCRIPT_EXT} = "cmd";
     $self->{$ALT_SCRIPT_EXT} = "cmd";
     $self->{$EXE_EXT} = "exe";
     $self->{$OS_DISTRIBUTION} = "";

     $self->{$OS_RELEASE} = `ver`;
     $self->{$OS_RELEASE} = TXK::OSD::UNKNOWN_OS_RELEASE if ( $CHILD_ERROR );
     $self->{$OS_RELEASE} =~ s/\n//g; 
   }
  elsif ( $OSNAME eq "linux" )
   {
     $self->{$PLATFORM} = "$IS_LINUX";
     $self->{$AUTOBUILD_PLATFORM} = "linux";
     $self->{$AUTOCONFIG_PLATFORM} = "Linux";

     $self->{$OS_RELEASE} = `uname -r`;
     $self->{$OS_RELEASE} = TXK::OSD::UNKNOWN_OS_RELEASE if ( $CHILD_ERROR );

     if ( ( -f "/etc/redhat-release" ) and 
	  ( -r "/etc/redhat-release" ) )
      {
	  $self->{$OS_DISTRIBUTION} = LINUX_REDHAT ;
	  open(REL_FILE, "< /etc/redhat-release");
	  my @file_contents = <REL_FILE>;
	  $self->{$OS_VERSION} = LINUX_REDHAT_AS21 if ( scalar(grep (/2\.1AS/,@file_contents)) > 0 ) ;
	  close(REL_FILE);
	  $self->{$OS_VERSION} = LINUX_REDHAT_AS30 if ( scalar(grep (/Red Hat Enterprise Linux AS release 3/,@file_contents)) > 0 ) ;
	  $self->{$OS_VERSION} = LINUX_REDHAT_AS40 if ( scalar(grep (/Red Hat Enterprise Linux AS release 4/,@file_contents)) > 0 ) ;
      }
     elsif ( ( -f "/etc/SuSE-release" ) and
	     ( -r "/etc/SuSE-release" ) )
      {
	  $self->{$OS_DISTRIBUTION} = LINUX_SUSE ;	  
	  open(REL_FILE, "< /etc/SuSE-release");
	  my @file_contents = <REL_FILE>;
	  close(REL_FILE);
	  $self->{$OS_VERSION} = LINUX_SUSE_70 if ( scalar(grep (/SuSE Linux 7\.0/,@file_contents)) > 0 ) ;
      }
   }
  elsif ( $OSNAME eq "solaris" )
   {
     $self->{$PLATFORM} = "$IS_SOLARIS";
     $self->{$AUTOBUILD_PLATFORM} = "solaris";
     $self->{$AUTOCONFIG_PLATFORM} = "Solaris";

     $self->{$OS_RELEASE} = `uname -r`;
     $self->{$OS_RELEASE} = TXK::OSD::UNKNOWN_OS_RELEASE if ( $CHILD_ERROR );
     $self->{$OS_DISTRIBUTION} = "";
   }
  elsif ( $OSNAME eq "aix" )
   {
     $self->{$PLATFORM} = "$IS_AIX";
     $self->{$AUTOBUILD_PLATFORM} = "aix";
     $self->{$AUTOCONFIG_PLATFORM} = "IBM_AIX";

     $self->{$OS_RELEASE} = `uname -r`;
     $self->{$OS_RELEASE} = TXK::OSD::UNKNOWN_OS_RELEASE if ( $CHILD_ERROR );
     $self->{$OS_DISTRIBUTION} = "";
   }
  elsif ( $OSNAME eq "hpux" )
   {
     $self->{$PLATFORM} = "$IS_HPUX";
     $self->{$AUTOBUILD_PLATFORM} = "hpux";
     $self->{$AUTOCONFIG_PLATFORM} = "HP_UX";

     $self->{$OS_RELEASE} = `uname -r`;
     $self->{$OS_RELEASE} = TXK::OSD::UNKNOWN_OS_RELEASE if ( $CHILD_ERROR );
     $self->{$OS_DISTRIBUTION} = "";
   }
  elsif ( $OSNAME eq "dec_osf" )
   {
     $self->{$PLATFORM} = "$IS_OSF1";
     $self->{$AUTOBUILD_PLATFORM} = "tru64";
     $self->{$AUTOCONFIG_PLATFORM} = "UNIX_Alpha";

     $self->{$OS_RELEASE} = `uname -r`;
     $self->{$OS_RELEASE} = TXK::OSD::UNKNOWN_OS_RELEASE if ( $CHILD_ERROR );
     $self->{$OS_DISTRIBUTION} = "";
   }
  else
   {
     $self->{$PLATFORM} = "$IS_UNIX_G";
     $self->{$AUTOBUILD_PLATFORM} = "$IS_GENERIC";
     $self->{$AUTOCONFIG_PLATFORM} = "Linux";
   }

  chomp($self->{$OS_RELEASE});  # Remove new line, if any.

#  Now try and figure out the OS Type.

  _findOSType($self) unless ( $self->{$PLATFORM} eq "$IS_UNIX_G" );

  return $self;
}

######################################
# Destructor
######################################

sub DESTROY
{
}

######################################
# Get Platform Name
######################################

sub getName
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  return $classSelf->{$PLATFORM};
}

######################################
# Get AutoBuild Platform Name
######################################

sub getAutoBuildName
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  return $classSelf->{$AUTOBUILD_PLATFORM};
}

######################################
# Set AutoBuild Platform Name
######################################

sub setAutoBuildName
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  $classSelf->{$AUTOBUILD_PLATFORM} = $args;
}

######################################
# Get AutoConfig Platform Name
######################################

sub getAutoConfigName
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  return $classSelf->{$AUTOCONFIG_PLATFORM};
}

######################################
# Get AutoConfig Generic Platform Name
######################################

sub getAutoConfigGenericName
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  return "NT" if ( $classSelf->{$PLATFORM} eq "$IS_NT" );

  return "UNIX";
}

######################################
# Get Perl Platform Name
######################################

sub getPerlName
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  return $classSelf->{$PERL_PLATFORM};
}

######################################
# Get Host Platform Name
######################################

sub getHostName
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  return $classSelf->{$HOST_NAME};
}

######################################
# Get OS Info
######################################

sub getOSInfo
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  return "# Platform Info: " .  TXK::OSD->getName() .
         "(" . TXK::OSD->getPerlName() . ")" .
         " - " . TXK::OSD->getOSType . 
         ( TXK::OSD->getOSDistribution() 
              ? ( "(" . TXK::OSD->getOSDistribution() . ")" ) : "" ) ;
}

######################################
# Get OS Type
######################################

sub getOSType
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  return $classSelf->{$OS_TYPE};
}

######################################
# Get OS Version
######################################

sub getOSVersion
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  return $classSelf->{$OS_VERSION};
}

######################################
# Get OS Release
######################################

sub getOSRelease
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  return $classSelf->{$OS_RELEASE};
}

######################################
# Get OS Distribution
######################################

sub getOSDistribution
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  return $classSelf->{$OS_DISTRIBUTION};
}

######################################
# Get BUG DB Port Id
######################################

sub getBugDBPortId
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  return $classSelf->{$BUGDB_PORTID};
}

######################################
# Is LINUX
######################################

sub isLinux
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  return TXK::Util::TRUE if ( $classSelf->{$PLATFORM} eq "$IS_LINUX" );

  return TXK::Util::FALSE;
}


######################################
# Is Solaris
######################################

sub isSolaris
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  return TXK::Util::TRUE if ( $classSelf->{$PLATFORM} eq "$IS_SOLARIS" );

  return TXK::Util::FALSE;
}

######################################
# Is AIX
######################################

sub isAIX
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  return TXK::Util::TRUE if ( $classSelf->{$PLATFORM} eq "$IS_AIX" );

  return TXK::Util::FALSE;
}

######################################
# Is HPUX
######################################

sub isHPUX
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  return TXK::Util::TRUE if ( $classSelf->{$PLATFORM} eq "$IS_HPUX" );

  return TXK::Util::FALSE;
}

######################################
# Is Tru64
######################################

sub isTru64
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  return TXK::Util::TRUE if ( $classSelf->{$PLATFORM} eq "$IS_OSF1" );

  return TXK::Util::FALSE;
}

######################################
# Is NT
######################################

sub isNT
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  return TXK::Util::TRUE if ( $classSelf->{$PLATFORM} eq "$IS_NT" );

  return TXK::Util::FALSE;
}

######################################
# Is Windows
######################################

sub isWindows
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  return TXK::Util::TRUE if ( $classSelf->{$PLATFORM} eq "$IS_WINDOWS" ||
                              $classSelf->{$PLATFORM} eq "$IS_NT" );

  return TXK::Util::FALSE;
}

######################################
# Is UNIX
######################################

sub isUNIX
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];
  
  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});
  
  return TXK::Util::TRUE if ( $classSelf->{$PLATFORM} ne "$IS_NT" );
  
  return TXK::Util::FALSE;
}

######################################
# Is Valid AutoBuild Platform
######################################

sub isValidAutoBuildPlatform
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  my $platform = $args if ( defined $args && !ref($args) );

  return TXK::Util::FALSE unless ( defined $platform );

  return TXK::Util::TRUE 
         if (  $platform eq "winnt" 	||
               $platform eq "linux"	||
               $platform eq "hpux" 	||
               $platform eq "aix"  	||
               $platform eq "solaris"	||
               $platform eq "tru64"	||
               $platform eq GENERIC     ||
               $platform eq UNIX_GENERIC
            );

  return TXK::Util::FALSE;
}

######################################
# getDirSeparator
######################################

sub getDirSeparator
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  return $classSelf->{$DIR_SEPARATOR};
}

######################################
# getBaseDirSeparator
######################################

sub getBaseDirSeparator
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  return $classSelf->{$BASE_DIR_SEPARATOR};
}

######################################
# getPathSeparator
######################################

sub getPathSeparator
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});
  
  return $classSelf->{$PATH_SEPARATOR};
} 

######################################
# getCommandScriptExt
######################################

sub getCommandScriptExt
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  return $classSelf->{$SCRIPT_EXT};
}

######################################
# getAltCommandScriptExt
######################################

sub getAltCommandScriptExt
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  return $classSelf->{$ALT_SCRIPT_EXT};
}

######################################
# getExecutableExt
######################################

sub getExecutableExt
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  return $classSelf->{$EXE_EXT};
}

######################################
# getBaseName
######################################

sub getBaseName
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  my $file = $args;

  if ( defined $file && !ref($file) )
   {

#	Use both separators and choose the nearest.

     my  $r1 = rindex($file,$classSelf->{$BASE_DIR_SEPARATOR});
     my  $r2 = rindex($file,$classSelf->{$DIR_SEPARATOR});

     $file = substr($file,(($r1>$r2 ? $r1 : $r2)+1)) if ( $r1>=0 || $r2>=0 );
   }

  return $file;
}

######################################
# getDirName
######################################

sub getDirName
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  my $dir  = $args;
  
  if ( defined $dir  && !ref($dir ) )
   { 
  
#       Use both separators and choose the nearest.
  
     my  $r1 = rindex($dir ,$classSelf->{$BASE_DIR_SEPARATOR});
     my  $r2 = rindex($dir ,$classSelf->{$DIR_SEPARATOR});
  
     $dir  = substr($dir ,0,(($r1>$r2 ? $r1 : $r2))) if ( $r1>=0 || $r2>=0 );
   }

  return $dir ;
}

######################################
# setClassPath
######################################

sub setClassPath
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  my $path = $args;

  if ( defined $path  && !ref($path ) )
   {
     $classSelf->{$CLASS_PATH} = TXK::OSD->trDirPathToBase($path);
   }

  return TXK::Error::SUCCESS;
}

######################################
# getClassPath
######################################

sub getClassPath
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  return $classSelf->{$CLASS_PATH};
}

######################################
# addClassPath
######################################

sub addClassPath
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  my $path = $args;

  if ( defined $path  && !ref($path ) )
   {
     $classSelf->{$CLASS_PATH} .=  TXK::OSD->getPathSeparator() 
					if ( $classSelf->{$CLASS_PATH} );

     $classSelf->{$CLASS_PATH} .= TXK::OSD->trDirPathToBase($path);
   }

  return TXK::Error::SUCCESS;
}

######################################
# setCommandPath
######################################

sub setCommandPath
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  my $path = $args;

  if ( defined $path  && !ref($path ) )
   {
     $classSelf->{$CMD_PATH} = TXK::OSD->trDirPathToBase($path);
   }

  return TXK::Error::SUCCESS;
}

######################################
# getCommandPath
######################################

sub getCommandPath
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  return $classSelf->{$CMD_PATH};
}

######################################
# addCommandPath
######################################

sub addCommandPath
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  my $path = $args;

  if ( defined $path  && !ref($path ) )
   {
     $classSelf->{$CMD_PATH} .=  TXK::OSD->getPathSeparator() 
					if ( $classSelf->{$CMD_PATH} );

     $classSelf->{$CMD_PATH} .= TXK::OSD->trDirPathToBase($path);
   }

  return TXK::Error::SUCCESS;
}

######################################
# trDirPath
######################################

sub trDirPath
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  my $dirstr = "";

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

#  $basesep = $classSelf->{$BASE_DIR_SEPARATOR};
  my $basesep = $classSelf->{$DIR_SEPARATOR};

  $dirstr = $args if ( ! ref($args) );

  $dirstr =~ s#$classSelf->{$DIR_SEPARATOR}#${basesep}#g ;

  return $dirstr;
}

######################################
# trDirPathFromBase
######################################

sub trDirPathFromBase
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  my $dirstr = "";
  my $basesep;

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  $basesep = $classSelf->{$BASE_DIR_SEPARATOR};
  $basesep .= $basesep if ( $basesep eq "\\" ); 

  $dirstr = $args if ( ! ref($args) );

  $dirstr =~ s#${basesep}#$classSelf->{$DIR_SEPARATOR}#g;

  return $dirstr;
}

######################################
# trDirPathToBase
######################################

sub trDirPathToBase
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  my $dirstr = "";
  my $basesep;

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  $basesep = $classSelf->{$BASE_DIR_SEPARATOR};

  $dirstr = $args if ( ! ref($args) );

  $dirstr =~ s#$classSelf->{$DIR_SEPARATOR}#${basesep}#g;

  return $dirstr;
}

######################################
# trFileName
######################################

sub trFileName
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  my $filename = "";

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

#  my $basesep = $classSelf->{$BASE_DIR_SEPARATOR};
  my $basesep = $classSelf->{$DIR_SEPARATOR};

  $filename = $args if ( ! ref($args) ) ;

  $filename =~ s#$classSelf->{$DIR_SEPARATOR}#${basesep}#g;

  return $filename;
}

######################################
# trFileDir
######################################

sub trFileDir
{
  my $self  = $ARG[0];
  my $arg1  = $ARG[1];
  my $arg2  = $ARG[2];

  my ($fileId,$dirId);

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});
  
  $dirId = TXK::OSD->trDirPath($arg2) if (defined $arg2);
  $fileId = TXK::OSD->trFileName($arg1) if (defined $arg1);

  return $dirId . ( $dirId && $fileId ? TXK::OSD->getDirSeparator() : undef) .
         $fileId;
}

######################################
# getEnvVar
######################################

sub getEnvVar
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  my $translate;

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID,
                         args=>$args});

  TXK::Util->isValidArgs({args=>$args,reqd=>["$ENV_NAME"]});

  my $name = TXK::Util->getScalarArg($ENV_NAME,$args->{$ENV_NAME});
  my $value;

  $translate = ( exists $args->{$ENV_TRANSLATE}
                     ? TXK::Util->getBooleanArg($ENV_TRANSLATE,
                                                $args->{$ENV_TRANSLATE})
                     : TXK::Util::FALSE
               );

  $value = $ENV{$name};

  $value = TXK::OSD->trDirPathFromBase($value) if ( $translate );

  return $value;
}

######################################
# setEnvVar
######################################

sub setEnvVar
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  my $translate;

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID,
                         args=>$args});

  TXK::Util->isValidArgs({args=>$args,reqd=>["$ENV_NAME","$ENV_VALUE"]});

  my $name = TXK::Util->getScalarArg($ENV_NAME,$args->{$ENV_NAME});
  my $value= TXK::Util->getString($args->{$ENV_VALUE});

  $translate = ( exists $args->{$ENV_TRANSLATE}
                     ? TXK::Util->getBooleanArg($ENV_TRANSLATE,
                                                $args->{$ENV_TRANSLATE})
                     : TXK::Util::FALSE
               );

  if ( defined $value )
   {
     $ENV{$name} = ( $translate ? TXK::OSD->trDirPathToBase($value) : $value );
   }
  else
   {
     delete $ENV{$name};
   }

  return TXK::Error::SUCCESS;
}

######################################
# chdir
######################################

sub chdir
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  if ( defined $args && !ref($args) )
   {
     my $dir = TXK::OSD->trDirPathToBase($args);

     chdir($dir) or $classSelf->setError("Failed to chdir to $dir");
   }

  return TXK::Error::SUCCESS;
}

######################################
# End of Public methods
######################################

# ==========================================================================

sub _findOSType
{
  my $self  = $ARG[0];
  my $variant = $ARG[1];

#   As we're being called from new() it probably isn't wise to use
#   object methods, such as isLinux(). Mind you this is Perl, so I'm
#   sure it would be allowed!

#   Code based on Opatch::Command.pm->get_os_id().

#   Variant is only ever passed from _setOS(), to enable full RT simulation.

  my $force_variant = "";

  $force_variant = $variant if ( $variant );

  my $os_name = $self->{$PLATFORM};

  my $system_command;
  my $sys_call_result;
  my $status;

  if    ( $os_name eq "$IS_SOLARIS" )
   {
     $self->{$OS_TYPE} = TXK::OSD::SOLARIS_32;
     $self->{$BUGDB_PORTID} = "453";

     $system_command = "/bin/isainfo";
     $sys_call_result = qx/$system_command/;
     $status = $CHILD_ERROR;

     if (    ( ( $sys_call_result =~ m#^sparcv9 # ) 
                      && $status == 0 && !$force_variant) 
          || $force_variant eq TXK::OSD::SOLARIS_64  
        ) 
      {
         $self->{$OS_TYPE} = TXK::OSD::SOLARIS_64;
         $self->{$BUGDB_PORTID} = "23";
      } 
   }
  elsif ( $os_name eq "$IS_WINDOWS" || $os_name eq "$IS_NT" )
   {
     $self->{$OS_TYPE} = TXK::OSD::WINDOWS_32;
     $self->{$BUGDB_PORTID} = "912";

     $system_command = "ver";
     $sys_call_result = qx/$system_command/;
     $status = $CHILD_ERROR;

     if ( ( ( $sys_call_result =~ m#Microsoft Windows \[Version 5# ) &&
            ( $status == 0 ) && !$force_variant
          ) 
         || $force_variant eq TXK::OSD::WINDOWS_64 
        )
      {
        my $processor = "$ENV{'PROCESSOR_IDENTIFIER'}";

        if (   ( $processor =~ /ia64/ && !$force_variant )
            || $force_variant eq TXK::OSD::WINDOWS_64 ) 
         {
           $self->{$OS_TYPE} = TXK::OSD::WINDOWS_64;
           $self->{$BUGDB_PORTID} = "208";
         }
      } 
   }
  elsif ( $os_name eq "$IS_AIX" )
   {
     $self->{$OS_TYPE} = TXK::OSD::AIX_32;
     $self->{$BUGDB_PORTID} = "319";

     $system_command = "uname -v";
     $sys_call_result = qx/$system_command/;
     $status = $CHILD_ERROR;

     if    (   ( $sys_call_result == 5 && $status == 0 && !$force_variant ) 
            || $force_variant eq TXK::OSD::AIX_5L
           ) 
      {
        $self->{$OS_TYPE} = TXK::OSD::AIX_5L;
        $self->{$BUGDB_PORTID} = "212";
      } 
     elsif (   ( $sys_call_result == 4 && $status == 0 && !$force_variant ) 
            || $force_variant eq TXK::OSD::AIX_64 
           ) 
      { 
        $self->{$OS_TYPE} = TXK::OSD::AIX_64;
        $self->{$BUGDB_PORTID} = "38";
      }
   }
  elsif ( $os_name eq "$IS_HPUX" )
   {
     $self->{$OS_TYPE} = TXK::OSD::HPUX_32;
     $self->{$BUGDB_PORTID} = "2";

     $system_command = "uname -a";
     $sys_call_result = qx/$system_command/;
     $status = $CHILD_ERROR;

     if    (   ( ( $sys_call_result =~ m#64-bit# ) 
                           && $status == 0 && !$force_variant )
            || $force_variant eq TXK::OSD::HPUX_64 
           )
      {
         $self->{$OS_TYPE} = TXK::OSD::HPUX_64;
         $self->{$BUGDB_PORTID} = "59";
      }
     elsif (   ( ( $sys_call_result =~ m#ia64# ) 
                          && $status == 0 && !$force_variant ) 
            || $force_variant eq TXK::OSD::HPUX_ITANIUM
           )
      {
         $self->{$OS_TYPE} = TXK::OSD::HPUX_ITANIUM;
         $self->{$BUGDB_PORTID} = "197";
      }
   }
  elsif ( $os_name eq "$IS_OSF1" )
   {
     $self->{$OS_TYPE} = TXK::OSD::TRU64;
     $self->{$BUGDB_PORTID} = "87";
   }
  elsif ( $os_name eq "$IS_LINUX" )
   {
     $self->{$OS_TYPE} = TXK::OSD::LINUX_32;
     $self->{$BUGDB_PORTID} = "46";

     $system_command = "uname -a";
     $sys_call_result = qx/$system_command/;
     $status = $CHILD_ERROR;

     my @words = split ' ', $sys_call_result;
     my $word_count = 0;
     my $word;

     foreach $word (@words) 
      {
        $word_count++;
      }

     my $last_word = $words[$word_count-1];
     my $next_to_last_word = $words[$word_count-2];

     if (   ( $last_word =~ m#unknown# && $next_to_last_word =~ m#ia64# 
                                 && !$force_variant ) 
         || $force_variant eq TXK::OSD::LINUX_ITANIUM 
        )
      {
        $self->{$OS_TYPE} = TXK::OSD::LINUX_ITANIUM;
        $self->{$BUGDB_PORTID} = "214";
      } 
     elsif (   ( $last_word =~ m#unknown# && $next_to_last_word =~ m#[456]86# 
                                 && !$force_variant ) 
            || $force_variant eq TXK::OSD::LINUX_32
           )
      {
        $self->{$OS_TYPE} = TXK::OSD::LINUX_32;
        $self->{$BUGDB_PORTID} = "46";
      }
     		# How do we test for Platform 226 ( Linux 64 )
     elsif (   ( 0 && !$force_variant )
            || $force_variant eq TXK::OSD::LINUX_64 
           )
      {
        $self->{$OS_TYPE} = TXK::OSD::LINUX_64;
        $self->{$BUGDB_PORTID} = "226";
      }
   }
}

sub _setOS
{

 # Undocumented method to override the class object. It should only ever
 # be called by RT code.

  my $self      = $ARG[0];
  my $os_ident  = $ARG[1];
  my $os_name   = $ARG[2];
  my $os_variant= $ARG[3];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  return unless ( $os_ident && $os_ident eq "ORACLE::APPS::TXK::RT::IDENT" );

# Perl OS list.

  my @os_list = ( "MSWin32", "linux", "solaris", "aix", "hpux", "dec_osf",
                  "unknown" );

  my $os;
  my $found = 0;
  my $new_os;

  foreach $os (@os_list)
   {
     $found = 1, $new_os = $os, last 
           if ( $os_name && $os_name eq ( "RT-ONLY-OS-" . $os ) );
   }

  return unless $found;

# We don't check os_variant - we assume that the RT code will set it
# correctly.

# Remember the base OS unless it has already been set.

  $baseClassSelf = $classSelf unless ( defined $baseClassSelf );

# Remember the real OSNAME

  my $actual_os_name = $OSNAME;

# Set the new OSNAME;

  $OSNAME = $new_os;

# Create the new OSD object.

  my $osd = TXK::OSD->new();

# Reset the OSNAME.

  $OSNAME = $actual_os_name;

# Call _findOSType directly if we're forcing a variant.

  _findOSType($osd,$os_variant) if ($os_variant);

# Override the class object

  $classSelf = $osd;
}

sub _resetOS
{

 # Undocumented method to reset the class object to the base OS. It should
 # only ever be called by RT code. It does nothing unless _setOS has
 # been called.

 my $self      = $ARG[0];

 TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

 return unless ( defined $baseClassSelf ) ;  # _setOS has been called ?

 $classSelf = $baseClassSelf;
}

1;

