
# +===========================================================================+
# |   Copyright (c) 2003 Oracle Corporation, Redwood Shores, California, USA
# |                         All Rights Reserved
# |                        Applications Division
# +===========================================================================+
# |
# | FILENAME
# |   Process.pm
# |
# | DESCRIPTION
# |      TXK Process package
# |
# | USAGE
# |       See Process.html
# |
# | PLATFORM
# |
# | NOTES
# |
# +===========================================================================+

# $Header: Process.pm 03-aug-2005.08:54:53 ndutko Exp $

package TXK::Process;

@ISA = qw( TXK::Common );

######################################
# Standard Modules
######################################

use strict;
use English;
use Carp;

require 5.005;

######################################
# Package Specific Modules
######################################

use TXK::Error();
use TXK::Util();
use TXK::IO();
use TXK::FileSys();
use TXK::OSD();

######################################
# Public Constants
######################################

use constant NO_TRANSLATE   => 0;
use constant TRANSLATE_PATH => 1;

######################################
# Private Constants
######################################

######################################
# Package Variables 
######################################

my $PACKAGE_ID = "TXK::Process";

######################################
# Object Keys
######################################

my $COMMAND	   = "command";
my $STDIN_FILE	   = "stdin";
my $STDOUT_FILE    = "stdout";
my $STDERR_FILE    = "stderr";
my $ARG_PREFIX	   = "arg";
my $ARG_TEXT	   = "text";
my $ARG_TYPE	   = "type";
my $ARG_SECURE     = "secure";
my $ARGUMENT	   = "argument";
my $SHOW_COMMAND   = "showCommand";
my $SHOW_OUTPUT    = "showOutput";
my $RUNTIME_OBJ    = "runtimeObj";

######################################
#
# Object Structure
# ----------------
#
#  Hash Array
#
######################################

######################################
# Package Methods 
#
# Public
#
#	new 	- build empty object
#
######################################

sub new;
sub DESTROY;
sub run;
sub which;

######################################
# Package Methods
# 
# Private
#       All private methods are marked with a leading underscore.
#
######################################

######################################
# Constructor
######################################

sub new {
  my $type = $ARG[0];

  my $self = TXK::Common->new();

  bless $self, $PACKAGE_ID ;

  my $key;

  my %INIT_OBJ = (
		   PACKAGE_IDENT   => $PACKAGE_ID,
		   $COMMAND	   => undef,
	           $STDIN_FILE     => undef,
	           $STDOUT_FILE	   => undef,
	           $STDERR_FILE	   => undef,
	           $ARGUMENT	   => "",
		   $SHOW_COMMAND   => TXK::Util::FALSE,
                   $SHOW_OUTPUT    => TXK::Util::FALSE,
                   $RUNTIME_OBJ    => undef,
                  );

  foreach $key (keys %INIT_OBJ)
   {
     $self->{$key} = $INIT_OBJ{$key};
   }

  return $self;
}

######################################
# Destructor
######################################

sub DESTROY
{
}

######################################
# run
######################################

sub run
{   
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  TXK::Util->isValidArgs({args=>$args,reqd=>["$COMMAND"]});

#	Args can be string or hash array.

  return $self->setError("Command argument must be either a scalar or a hash")
             unless ( !ref($args->{$COMMAND}) || 
                      ref($args->{$COMMAND}) eq "SCALAR" ||
                      ref($args->{$COMMAND}) eq "HASH" );

  if ( ref($args->{$COMMAND}) eq "HASH" )
   {  
     $self->{$COMMAND} = TXK::Util->getScalarArg
                               ("$COMMAND : $ARG_TEXT",
                                $args->{$COMMAND}->{$ARG_TEXT});

     $self->{$COMMAND} = TXK::OSD->trDirPathToBase($self->{$COMMAND})
                         if ( exists ($args->{$COMMAND}->{$ARG_TYPE}) &&
                              $args->{$COMMAND}->{$ARG_TYPE} == 
				          TXK::Process::TRANSLATE_PATH );
   }
  else
   {
     $self->{$COMMAND} = TXK::Util->getScalarArg("$COMMAND",$args->{$COMMAND});
   }

	# Process ARG 1-N, if present.

  $self->{$ARGUMENT} = "";
  $self->{$SHOW_COMMAND} = TXK::Util::FALSE;
  $self->{$SHOW_OUTPUT}  = TXK::Util::FALSE;
  $self->{$STDIN_FILE} = "";
  $self->{$STDOUT_FILE} = "";
  $self->{$STDERR_FILE} = "";
  $self->{$RUNTIME_OBJ}  = undef;

  my $key;
  my %runArg;
  my $secure_arg;
  my $argstr = "";
  my $display_arg_buffer;

  foreach $key ( keys %$args )
   {
     $runArg{$key} = $args->{$key} 
               if ( defined $key && $key =~ m/^(${ARG_PREFIX})\d+$/ );
   }

  foreach $key (
                 sort { my ($rhs,$lhs);
                        $lhs = $a; $lhs =~ s/${ARG_PREFIX}//g;
                        $rhs = $b; $rhs =~ s/${ARG_PREFIX}//g;
                        $lhs <=> $rhs;
                      } ( keys %runArg )
               )
   {
     $secure_arg = TXK::Util::FALSE;

     if ( ref($runArg{$key}) eq "HASH" )
      {
        $argstr = TXK::Util->getScalarArg("$key : $ARG_TEXT", 
                                           $runArg{$key}->{$ARG_TEXT});

        $argstr = TXK::OSD->trDirPathToBase($argstr)
                            if ( exists ($runArg{$key}->{$ARG_TYPE}) &&
                                 $runArg{$key}->{$ARG_TYPE} ==
                                             TXK::Process::TRANSLATE_PATH );

        $secure_arg = TXK::Util->getBooleanArg("$key : $ARG_SECURE",
                                               $runArg{$key}->{$ARG_SECURE})
                      if ( exists($runArg{$key}->{$ARG_SECURE}) ) ;

      }
     else
      {
        $argstr = TXK::Util->getScalarArg("$key",$runArg{$key});
      }
 
     $self->{$ARGUMENT} .= $argstr . " ";     

        # Append to the display buffer.

     $display_arg_buffer .= 
                  ( $secure_arg ? ( "*" x length($argstr) ) : $argstr ) . " ";
   }

#	Now process stdin/out/err etc.

  foreach $key ("$STDIN_FILE","$STDOUT_FILE","$STDERR_FILE")
   {
     $self->{$key} = ( exists $args->{$key}
                          ? TXK::Util->getScalarArg($key,$args->{$key})
                          : undef
                     );
   }

  foreach $key ("$SHOW_COMMAND","$SHOW_OUTPUT")
   {
     $self->{$key} = ( exists $args->{$key}
                          ? TXK::Util->getBooleanArg($key,$args->{$key})
                          : $self->{$key}
                     );
   }

# Check for runtimeObj.

  $self->{$RUNTIME_OBJ} = $args->{$RUNTIME_OBJ}
                            if ( exists ($args->{$RUNTIME_OBJ}) &&
                                 ref($args->{$RUNTIME_OBJ}) eq "TXK::Runtime" );

#	Create file handles for redirection. No need to check for
#	errors as abortOnError defaults to true.

  my ($stdin_io,$stdout_io,$stderr_io);

  my $fsys = TXK::FileSys->new();

  if ( $self->{$STDIN_FILE} )
   {
     $fsys->access({ fileName=>$self->{$STDIN_FILE},
                     type=>TXK::FileSys::FILE,
                     checkMode=>TXK::FileSys::READ_ACCESS,
                   })
         or return $self->setError("No read access for <${STDIN_FILE}> " .
                                   "$self->{$STDIN_FILE}");

#    AutoBuild and adpatch just won't allow STDIN to be cloned on NT. This
#    is clearly a bug in the way they use Perl but we need to provide some
#    solution. So for NT, we use < format rather than STDIN handle.

     if ( ! TXK::OSD->isNT() )
      {
        $stdin_io = TXK::IO->new();

        $stdin_io->open({ fileName   => $self->{$STDIN_FILE},
                          handleName => "STDIN",
                          saveHandle => "true",
                          mode       => TXK::IO::READ,
                        });
      }
   }
  
#
# Output message before STDOUT is redirected. We use a separate buffer for
# stdout string, as we now hide SECURE args.
#

  my $system_cmd      = $self->{$COMMAND} . " " . $self->{$ARGUMENT};
  my $show_system_cmd = $self->{$COMMAND} . " " . $display_arg_buffer;

# Modified to support runtimeObj log files.

  if ( $self->{$SHOW_COMMAND} )
   {
     my $print_cmd = "Execute SYSTEM command : " . $show_system_cmd ;

     if ( $self->{$RUNTIME_OBJ} )
      {
        $self->{$RUNTIME_OBJ}->printLog($print_cmd . "\n");
      }
     else
      {
         print $print_cmd , "\n";
      }
  }

  if ( $self->{$STDOUT_FILE} )
   {
     $stdout_io = TXK::IO->new();

     $fsys->access({ fileName=>$self->{$STDOUT_FILE},
                     type=>TXK::FileSys::FILE,
                     checkMode=>TXK::FileSys::CREATE_ACCESS,
                   })
         or return $self->setError("No create access for <${STDOUT_FILE}> " .
                                   "$self->{$STDOUT_FILE}");

     $fsys->create( {  fileName=>$self->{$STDOUT_FILE},
                       type=>TXK::FileSys::FILE,
                    } );

     $stdout_io->open({ fileName   => $self->{$STDOUT_FILE},
                        handleName => "STDOUT",
                        saveHandle => "true",
                        mode       => TXK::IO::WRITE,
			autoFlush  => "true",
                     });
   }

  if ( $self->{$STDERR_FILE} || $self->{$STDOUT_FILE} )
   {
     $stderr_io = TXK::IO->new();

     $fsys->access({ fileName=>$self->{$STDERR_FILE},
                     type=>TXK::FileSys::FILE,
                     checkMode=>TXK::FileSys::CREATE_ACCESS,
                   })
         or return $self->setError("No create access for <${STDERR_FILE}> " .
                                   "$self->{$STDERR_FILE}")
            if ( $self->{$STDERR_FILE} );

     $fsys->create( {  fileName=>$self->{$STDERR_FILE},
                       type=>TXK::FileSys::FILE,
                    } )
              if ( $self->{$STDERR_FILE} );

     $stderr_io->open({ fileName   => ( $self->{$STDERR_FILE}
                                           ? $self->{$STDERR_FILE}
                                           : "&STDOUT" ) ,
                        handleName => "STDERR",
                        saveHandle => "true",
                        mode       => TXK::IO::WRITE,
			autoFlush  => "true",
                     });
   }

  my $command_err;

# Handle special case of NT.

  $system_cmd .= " < " . $self->{$STDIN_FILE}  
              if ( TXK::OSD->isNT() && $self->{$STDIN_FILE} );
   
  my $rc = system($system_cmd);

  $command_err = $ERRNO if $rc;

#	Close file handles and check for SHOW_OUTPUT before returning errors.

  $stdin_io->close()  if (defined $stdin_io);
  $stdout_io->close() if (defined $stdout_io);
  $stderr_io->close() if (defined $stderr_io);

#
#	If showOutput has been specified and STDERR/OUT were redirected,
#	then output to the reset STDERR/OUT.
#

  if ( $self->{$SHOW_OUTPUT} )
   {
     if ( $self->{$STDOUT_FILE} )
      {
        my $show_stdout = TXK::IO->new();

        $show_stdout->open({ fileName   => $self->{$STDOUT_FILE} });

        my $stdout_ref = $show_stdout->getFileHandle();
        my $rec;

        print $rec while ( $rec = <$stdout_ref> );

        $show_stdout->close();
      }

     if ( $self->{$STDERR_FILE} )
      {
        my $show_stderr = TXK::IO->new();

        $show_stderr->open({ fileName   => $self->{$STDERR_FILE} });
      
        my $stderr_ref = $show_stderr->getFileHandle();
        my $rec;
        
        print STDERR $rec while ( $rec = <$stderr_ref> );

        $show_stderr->close();
      }
   }

  return $self->setError(
                     "Command error: <rc> = $rc, <command> = $show_system_cmd")
                   if ( $rc );

  return TXK::Error::SUCCESS;
}

######################################
# which
######################################

sub which
{ 
  my $self  = $ARG[0];
  my $args  = $ARG[1];
  
  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});
  
  TXK::Util->isValidArgs({args=>$args,reqd=>["$COMMAND"]});

  my $cmd = TXK::Util->getScalarArg("$COMMAND",$args->{$COMMAND});

  my $pathsep = TXK::OSD->getPathSeparator();
  my $basesep = TXK::OSD->getBaseDirSeparator();
  my $exe_ext = TXK::OSD->getExecutableExt();

  my @path = split(/${pathsep}/,$ENV{'PATH'});

  my ($entry,$file);

  foreach $entry (@path)
   {
     my $file = $entry . $basesep . $cmd . ( $exe_ext ? "." : "" ) . $exe_ext;

     return $file if ( -x $file && ! -d $file );
   }

  return TXK::Error::FAIL;
}    

######################################
# End of Public methods
######################################

# ==========================================================================

sub _getString
{
  my $self     = $ARG[0];
  my $argName  = $ARG[1];
  my $args     = $ARG[2];

  my $str = "";

  return TXK::Util->getScalarArg($argName,$args)
         if ( !ref($args) || ref($args) eq "SCALAR" );


  return $self->setError(
		    "Parameter <${argName}> must be either a scalar or hash")
         unless ( !ref($args) || ref($args) eq "SCALAR" || 
                                 ref($args) eq "HASH" );
}

1;

