# $Header: emll/sysman/admin/scripts/TXK/Reports.pm /main/2 2008/11/16 06:44:07 ndutko Exp $
# +===========================================================================+
# |   Copyright (c) 2002 Oracle Corporation, Redwood Shores, California, USA
# |                         All Rights Reserved
# |                        Applications Division
# +===========================================================================+
# |
# | FILENAME
# |   Reports.pm
# |
# | DESCRIPTION
# |   
# |   Reports package to generate a text or an HTML report based on an XML
# |   definition.
# |
# |
# | PLATFORM
# |   Generic
# |
# +===========================================================================+

package TXK::Reports;

@ISA = qw( TXK::Common );


######################################
# Standard Modules
######################################

use strict;
use English;
use Carp;
use Text::Wrap qw(&wrap);

require 5.005;


#######################################
# Public Constants
######################################

use constant PROGRAM_DESC => 
    "**** TXK Reports package to generate text and html reports based on xml definition ****";

use constant REPORT_TEXT => "text";
use constant REPORT_HTML => "html";


#######################################
# Package Specific Modules
######################################


use TXK::AutoConfig;
use TXK::Error;
use TXK::FileSys;
use TXK::IO;
use TXK::Log;
use TXK::OSD;
use TXK::Process;
use TXK::Restart;
use TXK::Runtime;
use TXK::Util;
use TXK::XML;
use TXK::AutoBuild;
use TXK::SQLPLUS;
use TXK::Validate;





#####################################
# Argument Definition
#####################################


######################################
# Package Methods 
#
# Public
#
#	new 	- build empty object
#
#  <!-- the input xml file has the following str -->
#  <?xml version="1.0" encoding="iso-8859-1"?> 
#  <TXKReport>
#   
#     <Title>Technology Validation Utility report for 11.5.10 Maintainance Pack.</Title>
#
#     <Header>The latest version of the Technology Validation Utility and instructions are available in My Oracle Support Note 240744.1.  
#     </Header>
#    
#   <!-- then 
#   <Conc message="Validation of Cocurrent Processing Server Node (ap667wgs):">
#      <action name=CheckDisplay >
#         <status>PASS</status>
#         <version>135.35.55.244:26.0</version>
#         <message>DISPLAY variable from the Applications Context file is: 135.35.55.244:26.0</message>
#      </action>
#      <action name=CheckJDKOnConcTier >
#         <status>PASS</status>
#         <version>1.3.1.07</version>
#         <message>JDK on Concurrent Processing node is: 1.3.1.07</message>
#      </action>
#    </Conc>
#   <Web message="Validation of HTTP Server Node (ap667wgs):">
#      <action name=CheckDisplay >
#         <status>PASS</status>
#         <version>135.35.55.244:26.0</version>
#         <message>DISPLAY variable from the Applications Context file is: 135.35.55.244:26.0</message>
#      </action>
#      <action name=CheckJinitiator >
#         <status>PASS</status>
#         <version>1.3.1.14</version>
#         <message>Oracle JInitiator version from Applications Context file is: 1.3.1.14</message>
#      </action>
#    </Web>
#   <Admin message="Validation of Adminstration Server Node (ap667wgs)">
#      <action name=CheckJQPInitDBParameter >
#         <status>FAIL</status>
#         <version>0</version>
#         <message>JOB_QUEUE_PROCESSES in parameter table is:  
#	   11.5.10 Maintenance Pack requires a minimum value of 2
#	   for this parameter. Please follow My Oracle Support Note 216205.1 "Database
#	   Initialization Parameters and Configurations for Oracle Applications 11i"
#	   to update.</message>
#      </action>
#    </Admin>
#
#    <Summary>Summary Status [ALLPASS]:
#	     All the technology checks on this node needed for the 11.5.10 Maintenance Pack have passed. Once this is true of each node in your system, you may proceed with the next step in the "Oracle Applications Release 11.5.10 Maintenance Pack Installation Instructions".
  
#             Oracle highly recommends that you take the time now to review My Oracle Support Note 216205.1, "Database Initialization Parameters and Configuration for Oracle Applications 11i", to ensure that your database instance is prepared for use with Release 11.5.10.
#   </Summary>
#    </TXKReport>
#
#
######################################


sub new;
sub loadDocument;

######################################
# Package Methods
# 
# Private
#       All private methods are marked with a leading underscore.
#
######################################


sub _setClasspath;
sub _setJava;
sub _setJVMOptions;

######################################
# Variable Declarations
######################################


######################################
# Package Variables 
######################################

my $PACKAGE_ID = "TXK::Reports"   ;


my $INPUT_XML     = "InputXML"    ;     # currently takes an XML object
my $OUTPUT_REPORT = "OutputReport";
my $REPORT_TYPE   = "reporttype"  ;
my $REPORT_XSL    = "xslfile"   ;
my $REPORT_LINES  = "ReportLines" ;
my $SUMMARY       = "Summary"     ;
my $TECHSTACK     = "techstack"   ;
my $JAVA          = "java"        ;
my $JAVA_TOP      = "javatop"     ;
my $CLASSPATH     = "classpath"   ;
my $JVM_OPTIONS   = "jvmoptions"  ; 


my $DEFAULT_LINE_LENGTH      = 76;
my $DEFAULT_MESSAGE_LENGTH   = 50;  # leave 10 chars for versions
my $DEFAULT_HEADER_LENGTH    = 15;
my $DEFAULT_STATUS_LENTTH    = 6 ;  # for [PASS] or [FAIL] or [WARN]
my $DEFAULT_FIELD_IDENT      = 1 ;  # no .of spaces between Status and Message , Message and Version
my $MESSAGE_IDENT            = 10;
my $IGNORE_ERROR             = TXK::Util::FALSE;

$Text::Wrap::columns = $DEFAULT_LINE_LENGTH;

######################################
# Constructor
######################################


sub new
{
    my $type = $ARG[0];

    my $args = $ARG[1];

    my $self = TXK::Common->new();

    bless $self, $PACKAGE_ID ;

    my %INIT_OBJ = ( $INPUT_XML     => undef,
		     $OUTPUT_REPORT => undef,
		     $REPORT_TYPE   => undef,
		     $REPORT_XSL    => undef,
		     $REPORT_LINES  => undef,
		     $SUMMARY       => TXK::XML->new(),
		     $TECHSTACK     => undef,
		     PACKAGE_IDENT  => $PACKAGE_ID,
		     $JAVA          => undef,
		     $JVM_OPTIONS   => undef,
		     $JAVA_TOP      => undef,
		     $CLASSPATH     => undef
                    );

    TXK::Util->copyHash( \%INIT_OBJ , $self );

    return $self;   
}

##########################
# sub loadDocument
##########################

sub loadDocument
{
    my $self = $ARG[0];

    my $args = $ARG[1];

    TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

    TXK::Util->isValidArgs({args=>$args,
			    reqd=> ["$INPUT_XML","$OUTPUT_REPORT","$TECHSTACK"]} );

    $self->{$TECHSTACK} = $args->{$TECHSTACK};

    $self->{$OUTPUT_REPORT} = $args->{$OUTPUT_REPORT};

    if ( !defined $args->{$REPORT_TYPE} )
     {
	 $self->{$REPORT_TYPE} = REPORT_TEXT;
     }
    else
     {
	 $self->{$REPORT_TYPE} = $args->{$REPORT_TYPE};
     }

    if ( $self->{$REPORT_TYPE} eq REPORT_HTML )
     {
	 TXK::Util->isValidArgs({args=>$args,
			    reqd=> ["$REPORT_XSL"]} );

	 $self->{$REPORT_XSL} = $args->{$REPORT_XSL};
     }

    if ( ref($args->{$INPUT_XML}) eq 'TXK::XML' )
     {
	 $self->{$INPUT_XML} = $args->{$INPUT_XML};
     }

   $self->generateReport();
}

################################
# sub _generateReport
###############################

sub generateReport
{
    my $self = $ARG[0];

    TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

    $self->{$REPORT_LINES} = [];

    #---------------------------------------
    # Text report is always generated 
    #---------------------------------------

    $self->_generateReportText();

    if ( $self->{$REPORT_TYPE} eq REPORT_HTML )
     {
	 $self->_generateReportHTML();
     }
}

#################################
# sub getSummaryNode
#################################

sub getSummaryNode
{
    my $self = $ARG[0];

    TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

    $self->{$SUMMARY}->addNode ( { 'node' => { 'type' => 'xmlnode',
					  'value'=> $self->{$INPUT_XML}->getNode({'node'=>TXK::Validate::SUMMARY_TAG}) }
			  } );
}

#################################
# sub getSummaryMessage
#################################

sub getSummaryMessage
{
    my $self = $ARG[0];

    TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

    return _getTextFromNode( $self->{$SUMMARY}->getNode({'node'=>TXK::Validate::MESSAGE_TAG}) );
}

sub getSummaryStatus
{
    my $self = $ARG[0];

    TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

    return  $self->{$SUMMARY}->getNodeValue({'node'=>TXK::Validate::STATUS_TAG});
}

sub getSummaryStatusText
{
    my $self = $ARG[0];

    TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

    return  $self->{$SUMMARY}->getAttrValue({'node'=> TXK::Validate::SUMMARY_TAG,
					    'attrname'=>$TXK::Validate::SUMMARY_STATUS_TEXT});
}

#################################
# sub setSummary
#################################

sub setSummary
{
    my $self = $ARG[0];

    my $summary = $ARG[1];

    TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

    $self->{$INPUT_XML}->setNodeValue({'node'  => TXK::Validate::SUMMARY_TAG,
				       'value' => $summary });

}


#################################
# sub getReportFile
#################################

sub getReportFile
{

}

#################################
# sub printReport
#################################

sub printReport
{
    my $self = $ARG[0];

    TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});
    
    return unless ( $self->{$REPORT_TYPE} eq REPORT_TEXT );
	
    my $line;

    my $io = TXK::IO->new();

    $io->open( { 'fileName' => $self->{$OUTPUT_REPORT},
	         'mode'     => TXK::IO::WRITE} );

    foreach $line ( @{$self->{$REPORT_LINES}} )
     {
	 print $line ;

	 $io->print($line);
     }

    $io->close();
}

sub _generateReportText
{
    my $self = $ARG[0];

    $self->_generateTitle();

    $self->_generateHeader();
    
    $self->_generateBody();

    $self->_generateSummary();
}

sub _generateReportHTML
{
    my $self = $ARG[0];

    my $apps_ctx_file_key = TXK::Techstack::APPS_CTX_FILE_KEY;

    my $tmpFile = TXK::OSD->trDirPathToBase( TXK::Techstack->getAppsTempDir(
					     { $apps_ctx_file_key => $self->{$TECHSTACK}->txkGetAppsCtxLocation() } ) ."/txkreport.xml");

    $self->{$INPUT_XML}->storeDocument( { file => $tmpFile } );
 
    $self->_setJava();

    $self->_setClasspath();

    $self->_setJVMOptions();

    my $java = $self->{$JAVA};

    my $classpath = $self->{$CLASSPATH};

    my $fsys = TXK::FileSys->new();

    $fsys->access ( { fileName => $java,
		      type     => TXK::FileSys::FILE,
		      checkMode=>TXK::FileSys::READ_ACCESS } )
	or TXK::Error->stop("Unable to access $java",$fsys->getError());

    my $proc = TXK::Process->new();

    $proc->abortOnError({enable => TXK::Util::FALSE }) if ( $IGNORE_ERROR );

    $proc->run( { command => $java ,
		  arg1    => "$self->{$JVM_OPTIONS} -classpath $classpath",
		  arg2    => "oracle.xml.parser.v2.oraxsl",
		  arg3    => $tmpFile,
		  arg4    => $self->{$REPORT_XSL},
		  arg5    => $self->{$OUTPUT_REPORT},
		  showOutput => TXK::Util::TRUE } );

    $proc->abortOnError({enable => TXK::Util::TRUE }) ;

    $fsys->rmfile( { fileName => $tmpFile }) or TXK::Error->stop("can not delete file $tmpFile");
}

sub _setJVMOptions
{
    my $self = $ARG[0];

    my $apps_ctx_file_key = TXK::Techstack::APPS_CTX_FILE_KEY;

    $self->_setJava() unless ( defined $self->{$JAVA} );

    if ( TXK::OSD->isUNIX() )
     {
	 # no options for Unix

	 $self->{$JVM_OPTIONS} = "";
     }
    else
     {
	 #For NT , JRE 118 we need to set -nojit

	 if ( $self->{$JAVA} =~ m/jre\.exe$/ )
	  {
	      my $proc = TXK::Process->new();

	      my $outFile = TXK::OSD->trDirPathToBase( TXK::Techstack->getAppsTempDir(
					     { $apps_ctx_file_key => $self->{$TECHSTACK}->txkGetAppsCtxLocation() } ) ."/java_ver.out");

	      #---------------------------------------------------------------
	      # on NT, jre118 exist with 1 which translates to 256 in perl. 
	      # so we ignore the error here and set the "-nojit" for jre 118
	      #----------------------------------------------------------------

	      $proc->abortOnError({enable=>TXK::Util::FALSE});

	      $proc->run ( { command     => $self->{$JAVA},
			     arg1        => "-version",
			     stdout      => $outFile,
			     showOutput  => TXK::Util::TRUE } );

	      $proc->abortOnError({enable=>TXK::Util::TRUE});

	      my $io = TXK::IO->new();
	      my $fsys = TXK::FileSys->new();
	      $io->open({fileName  => $outFile});
	      my $io_handle = $io->getFileHandle();
	      my @io_data = <$io_handle>;
	      $io->close();
	      $fsys->rmfile({fileName => $outFile});

	      $self->{$JVM_OPTIONS} = "-nojit" if ( scalar ( grep ( /1\.1\.8/ ,@io_data ) ) > 0 ) ;

	      print ( "java is $self->{$JAVA}\n");
	      
	      $IGNORE_ERROR = TXK::Util::TRUE;
	  }
     }
}

sub _setJava
{
    my $self = $ARG[0];

    my $java = TXK::OSD->trFileName(TXK::OSD->getEnvVar({name => "AFJVAPRG"}));

    my $java_top = undef ;

    my $fsys = TXK::FileSys->new();

    my $apps_ctx_file_key = TXK::Techstack::APPS_CTX_FILE_KEY;

    if ( $self->{$TECHSTACK}->txkIsDBContext( { $apps_ctx_file_key => $self->{$TECHSTACK}->txkGetAppsCtxLocation() } ) )
     {
	 my $oracle_home = $self->{$TECHSTACK}->txkGetDBOracleHome();

	 # use the context variables first

	 my $jre_top_ctx = $self->{$TECHSTACK}->txkGetContextVar({ oavar => 's_jretop' });
	 my $jdk_top_ctx = $self->{$TECHSTACK}->txkGetContextVar({ oavar => 's_jdktop' });

	 my $dir;

	 my $dirList = [ TXK::OSD->trDirPathToBase($jre_top_ctx),
			 TXK::OSD->trDirPathToBase($jdk_top_ctx),
			 TXK::OSD->trDirPathToBase($oracle_home."/jre/1.1.8"),
			 TXK::OSD->trDirPathToBase($oracle_home."/jdk"),
			 TXK::OSD->trDirPathToBase($oracle_home."/JRE/1.1.8"),
			 TXK::OSD->trDirPathToBase($oracle_home."/JRE"),
			 TXK::OSD->trDirPathToBase($oracle_home."/jre"),
			 TXK::OSD->trDirPathToBase($oracle_home."/JDK") ] ;

	 foreach $dir ( @$dirList )
	  {
	      $java_top = $dir;
	      
	      $fsys->access( { dirName   => $java_top,
			       type      => TXK::FileSys::DIRECTORY,
			       checkMode => TXK::FileSys::READ_ACCESS})
		  or next;

	      last;
	  }

	 my $java_extension = undef;

	 $java_extension    = ".exe" if ( TXK::OSD->isNT() );

	 $java = TXK::OSD->trDirPathToBase($java_top."/bin/java" .$java_extension );

	 my $javaFound = TXK::Util::FALSE;

	 $fsys->access( { fileName  => $java,
			  type      => TXK::FileSys::FILE,
			  checkMode => TXK::FileSys::READ_ACCESS } ) 
	     and $javaFound = TXK::Util::TRUE;

	 unless ( $javaFound )
	  {
	      $java = TXK::OSD->trDirPathToBase($java_top."/bin/jre" .$java_extension );

	      $fsys->access( { fileName  => $java,
			       type      => TXK::FileSys::FILE,
			       checkMode => TXK::FileSys::READ_ACCESS } ) 
		  and $javaFound = TXK::Util::TRUE;
	  }

	 unless ( $javaFound )
	  {
	      my $proc = TXK::Process->new();

	      $java = $proc->which({ command => "java"}) or TXK::Error->stop("java comand not found in PATH");
	  }

     }
    else
     {
	 $java_top = $self->{$TECHSTACK}->txkGetJavaTop();
     }

    $java .= ".exe" if ( TXK::OSD->isNT() and ( $java !~ m/.exe$/ ) ) ;

    $self->{$JAVA} = $java;

    $self->{$JAVA_TOP} = $java_top;
}

sub _setClasspath
{
    my $self = $ARG[0];

    my $classpath = TXK::OSD->trFileName(TXK::OSD->getEnvVar({name => "CLASSPATH"}));

    my $java_top = $self->{$JAVA_TOP};

    my $apps_ctx_file_key = TXK::Techstack::APPS_CTX_FILE_KEY;

    if ( $self->{$TECHSTACK}->txkIsDBContext( { $apps_ctx_file_key => $self->{$TECHSTACK}->txkGetAppsCtxLocation() } ) )
     {
	 my $oracle_home = $self->{$TECHSTACK}->txkGetDBOracleHome();

	 TXK::OSD->addClassPath ($oracle_home."/lib/xmlparserv2.jar");

	 TXK::OSD->addClassPath ($java_top."/lib/rt.jar");

	 TXK::OSD->addClassPath ($java_top."/jre/lib/rt.jar");

	 TXK::OSD->addClassPath ($java_top."/lib/dt.jar");

	 TXK::OSD->addClassPath ($java_top."/jre/lib/i18n.jar");

	 $classpath = TXK::OSD->getClassPath("") ;
     }

    $self->{$CLASSPATH} = $classpath;
}


sub _generateTitle
{
    my $self = $ARG[0];

    $Text::Wrap::columns = $DEFAULT_LINE_LENGTH;

    my $title = $self->{$INPUT_XML}->getNodeValue({'node'=>TXK::Validate::TITLE_TAG}); 

    $title =~ s/\n//g;

    $title = wrap ("","",$title);  # the title is not supposed to span more than one line
                                   # but just in case
    my $lengthOfLastLine;

    foreach ( split("\n",$title) )
     {
	 $lengthOfLastLine = length($_);
     }
    
    push ( @{$self->{$REPORT_LINES}} , $title ."\n");        

    push ( @{$self->{$REPORT_LINES}} , "=" x $lengthOfLastLine ."\n" );
}

sub _generateBody
{
    my $self = $ARG[0];

    my $tree = $self->{$INPUT_XML}->getDocument();

    my $nodeType;

    foreach $nodeType (@{$tree->[1]->{'body'}})
     {
	 unless ( $nodeType->{'head'} eq TXK::Validate::TITLE_TAG  or
		  $nodeType->{'head'} eq TXK::Validate::HEADER_TAG or
		  $nodeType->{'head'} eq TXK::Validate::SUMMARY_TAG
		  )
	  {

		my $nodeXML = TXK::XML->new();

		$nodeXML->addNode( {'node' => {'type'  => 'xmlnode',
					       'value' => $nodeType } });

		my $nodeMessage = $nodeXML->getAttrValue ( {'node'     => $nodeType->{'head'},
							    'attrname' => TXK::Validate::SECTION_TITLE } );

		$nodeMessage =~ s/(^")|("$)//g;

		$Text::Wrap::columns = $DEFAULT_LINE_LENGTH;

		$nodeMessage = wrap ("" , "", $nodeMessage);

		push ( @{$self->{$REPORT_LINES}} , "\n".$nodeMessage."\n" );		

		$self->_generateNode($nodeXML);
	  }
     }
}

sub _generateNode
{
    my $self = $ARG[0];

    my $tree = $ARG[1]->getDocument();

    my $action;

   foreach $action (@{$tree->[0]->{'body'}})
    {
	 my $actionTag = TXK::Validate::REPORT_ACTION_TAG ;

	 if ( $action->{'head'} =~ m/^$actionTag$/ )
	 {
	     my $node = TXK::XML->new();

	     $node->addNode({'node' => {'type' => 'xmlnode',
					'value' => $action }
			    });

	     my $status = $node->getNodeValue({node=>TXK::Validate::STATUS_TAG});

	     my $message = _getTextFromNode($node->getNode({'node'=>'message'}));

	     my $first = TXK::Util::TRUE;
	     
	     my $line;

	     my $firstLine;

	     my $restLines = [];

	     ($firstLine, @$restLines ) = split ("\n",$message);

              #------------------------------------
    	      # check out the version and status here.
              # version should be of the type
   	      # message: <version> for the first line
	      #---------------------------------------

      	       if ( $status ne "" )
	        {
	       	    $firstLine= "  [$status]  ".$firstLine;
	      	}
	       else
	        {
		    $MESSAGE_IDENT = 2 ;

		    $firstLine= "  ".$firstLine;
		}

	        my $version = $firstLine ;

	        $version = $node->getNodeValue({node=>TXK::Validate::ACTION_VERSION_TAG});

                #------------------------------------------------------------------
	        # remove the version from the line if it exists.
	        # though it is not required to have the version in the message
	        # but it may be there in the definition to have clarity in the message
	        #------------------------------------------------------------------
	     
	        $firstLine =~ s/$version//g;

                if ( length ( $firstLine ) > $DEFAULT_MESSAGE_LENGTH )
                 {
	       	    $Text::Wrap::columns = $DEFAULT_MESSAGE_LENGTH;

	       	    $firstLine = wrap (""," " x $MESSAGE_IDENT, $firstLine);

	       	    my $innerLine;
	       	    
	       	    my $innerFirst = TXK::Util::TRUE;
			    
	       	    foreach $innerLine ( split ("\n",$firstLine ) )
	             {
		       	 if ( $innerFirst )
		       	  {
		              $innerFirst = TXK::Util::FALSE;

	       		      $innerLine .= " " x ($DEFAULT_MESSAGE_LENGTH - length($innerLine))." ".$version;
	       		  }

	       		 push ( @{$self->{$REPORT_LINES}} , $innerLine."\n" );
		     }
	      	}
              else
	       {
      		    $firstLine .=  " " x ($DEFAULT_MESSAGE_LENGTH - length($firstLine)). " ". $version;

		    push ( @{$self->{$REPORT_LINES}} , $firstLine ."\n");
	       }
	     
	     if ( scalar(@$restLines) )
	      {

	       	  $Text::Wrap::columns = $DEFAULT_MESSAGE_LENGTH;

		  $line = join ( "",@$restLines);

		  $line =~ s/^\s+//g;

		  $line =~ s/\s{2,}/ /g;
	     
		  $line =~ s/\n//g;  # remove new lines

		  $line = wrap(" " x $MESSAGE_IDENT ," " x $MESSAGE_IDENT , $line);

		  push ( @{$self->{$REPORT_LINES}} , $line ."\n" );
	      }
	 }
     }
}

sub _generateSummary
{
    my $self = $ARG[0];

    $self->getSummaryNode();

    my $summaryMessage = $self->getSummaryMessage();

    return if ( $summaryMessage eq '' ) ;

    $Text::Wrap::columns = $DEFAULT_LINE_LENGTH;

    my $summaryHeader = $self->getSummaryStatusText(). "  [" .$self->getSummaryStatus() . "]:";

    push ( @{$self->{$REPORT_LINES}} , "\n".$summaryHeader."\n" ); 

    my $line ;

    $summaryMessage =~ s/\s{2,}/ /g;

    $line = wrap(" " x 2," " x 2 , $summaryMessage);

    push ( @{$self->{$REPORT_LINES}} , $line."\n" );    
}


sub _generateHeader
{
    my $self = $ARG[0];

    my $line;

    my $Header = TXK::XML->new();

    $Header->addNode( { 'node' => { 'type'  => 'xmlnode',
				    'value' => $self->{$INPUT_XML}->getNode( { 'node' => TXK::Validate::HEADER_TAG } ) }
		    } );

    my $hline = _getTextFromNode( $Header->getNode({ 'node' => 'hline' }) );

    $line = wrap ('','',$hline);

    $self->{$REPORT_LINES} = [ ] unless ( defined $self->{$REPORT_LINES} );

    push ( @{$self->{$REPORT_LINES}} ,"\n". $line."\n\n" );

    $Text::Wrap::columns = $DEFAULT_LINE_LENGTH;

    my $tree = $Header->getDocument();

    my $element;

    foreach $element ( @{$tree->[0]->{'body'}} )
     {
	 my $xml = TXK::XML->new();

	 $xml->addNode ( { 'node' => { 'type' =>'xmlnode',
				       'value'=> $element }
		       } );

	 my $root = $xml->getRootNodeName();

         my $name = $xml->getAttrValue( { 'node'     => $root,
         				  'attrname' => 'name' } );

	 unless ( $name eq '' )
	  {
	      my $value = $xml->getNodeValue( { 'node' => $root } );

	      $line = wrap ( ' ' x 2 , ' ' x 2 , $name . ' ' x ( $DEFAULT_HEADER_LENGTH - length($name) ). ":$value" );

	      push ( @{$self->{$REPORT_LINES}} , $line."\n" );
	  }
     }
}

sub _getTextFromNode
{
    my $node = $ARG[0];

    my $xml = TXK::XML->new();

    $xml->addNode( {'node' => { 'type' => 'xmlnode',
				'value'=> $node } 
		   } );

    my $tree = $xml->getDocument();

    my ($element,$line);

    foreach $element ( @{$tree->[0]->{'body'}} )
     {
	 if ( ref ($element) eq 'HASH' )  # this is an xml element
	  {
	      if ( $element->{'is_inner'} )
	       {
		   $line .= "\n\n" if ( $element->{'head'} eq 'para' ); # support for paragraphs

		   $line .= _getTextFromNode($element). ' ';
	       }
	      else
	       {
		   $line .= $element->{'body'}->[0] . ' ' ;
	       }
	  }
	 else # this is just a scalar
	  {
	      $line .= $element ;
	  }
     }

    return $line;
}

#=================================================================
# End of script.
#=================================================================

1;
