
# +===========================================================================+
# |   Copyright (c) 2003 Oracle Corporation, Redwood Shores, California, USA
# |                         All Rights Reserved
# |                        Applications Division
# +===========================================================================+
# |
# | FILENAME
# |   Restart.pm
# |
# | DESCRIPTION
# |      TXK Restart package
# |
# | USAGE
# |       See Restart.html
# |
# | PLATFORM
# |
# | NOTES
# |
# +===========================================================================+

# $Header: Restart.pm 03-aug-2005.08:54:53 ndutko Exp $

package TXK::Restart;

@ISA = qw( TXK::Common );

######################################
# Standard Modules
######################################

use strict;
use English;
use Carp;

require 5.005;

######################################
# Package Specific Modules
######################################

use TXK::Error();
use TXK::FileSys();
use TXK::Util();
use TXK::XML();
use TXK::IO();

######################################
# Public Constants
######################################

use constant ACTION_SUCCESS => "Done";
use constant ACTION_PENDING => "ToDo";
use constant ACTION_RUNNING => "Running";
use constant ACTION_FAILED  => "Error";

use constant RUN_EVERYTIME  => "Run_Everytime";
use constant RUN_ONCE	    => "Run_Once";

######################################
# Package Variables 
######################################

my $PACKAGE_ID = "TXK::Restart";

######################################
# Object Keys
######################################

my $RESTART_FILE   = "restartFile";
my $RUNTIME_OBJ	   = "runtimeObj";
my $USE_RESTART    = "useRestart";
my $CLEAR_RESTART  = "clearRestartFile";
my $ACTION_TABLE   = "actionTable";
my $XML_OBJ	   = "xmlObj";
my $FILESYS_OBJ	   = "fileSys";
my $IO_OBJ	   = "ioOBJ";
my $TABLE_LOADED   = "tableLoaded";
my $ACTION_ITERATOR= "actionIterator";

my $ACTION_ID	   = "action";
my $DESC_ID	   = "desc";
my $STEP_ID	   = "step";
my $STATUS_ID	   = "status";
my $RUN_TYPE	   = "type";
my $USER_CONTEXT   = "context";

######################################
#
# Object Structure
# ----------------
#
#  Hash Array
#
######################################

######################################
# Package Methods 
#
# Public
#
#	new 	- build empty object
#
######################################

sub new;
sub DESTROY;
sub save;
sub load;
sub hasMoreActions;
sub nextAction;

######################################
# Package Methods
# 
# Private
#       All private methods are marked with a leading underscore.
#
######################################

######################################
# Constructor
######################################

sub new {
  my $type = $ARG[0];

  my $self = TXK::Common->new();

  bless $self, $PACKAGE_ID ;

  my $key;

  my %INIT_OBJ = (
		   PACKAGE_IDENT   => $PACKAGE_ID,
		   $USE_RESTART    => TXK::Util::TRUE,
                   $CLEAR_RESTART  => TXK::Util::FALSE,
                   $ACTION_TABLE   => undef,
                   $ACTION_ITERATOR=> -1,
		   $TABLE_LOADED   => "0",
		   $RESTART_FILE   => "",
	           $XML_OBJ	   => undef,
	           $FILESYS_OBJ    => TXK::FileSys->new(),
	           $IO_OBJ	   => TXK::IO->new(),
	           $RUNTIME_OBJ	   => undef,
                 );

  foreach $key (keys %INIT_OBJ)
   {
     $self->{$key} = $INIT_OBJ{$key};
   }

  return $self;
}

######################################
# Destructor
######################################

sub DESTROY
{
}

######################################
# load
######################################

sub load
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  TXK::Util->isValidArgs({args=>$args,reqd=>["$ACTION_TABLE"]});

  my $key;

  $self->{$USE_RESTART} = TXK::Util::TRUE;
  $self->{$CLEAR_RESTART} = TXK::Util::FALSE;
  $self->{$RESTART_FILE} = "";

  foreach $key ("$USE_RESTART","$CLEAR_RESTART")
   {
     $self->{$key} = ( exists $args->{$key}
                          ? TXK::Util->getBooleanArg($key,$args->{$key})
                          : $self->{$key}
                     );
   }

  if ( $self->{$USE_RESTART} )
   {
     foreach $key ("$RESTART_FILE")
      {
        $self->{$key} = ( exists $args->{$key}
                             ? TXK::Util->getScalarArg($key,$args->{$key})
                             : undef
                        );
      }
   }

  $self->{$RUNTIME_OBJ} = $args->{$RUNTIME_OBJ}
                            if ( exists ($args->{$RUNTIME_OBJ}) &&
                                 ref($args->{$RUNTIME_OBJ}) eq "TXK::Runtime" );

  my $acttab = $args->{$ACTION_TABLE};


#	Clear out old action table.

  $self->{$ACTION_TABLE} = [ ];

  foreach $key ( 
                 sort { my ($rhs,$lhs); 
                        $lhs = $a; $lhs =~ s/step//g;
                        $rhs = $b; $rhs =~ s/step//g;
                        $lhs <=> $rhs;
                      } 
                      ( keys %$acttab )
               )
   {
     return $self->setError("Invalid key in action table : $key")
             unless  ( defined $key &&
                       $key =~ m/^(step)\d+$/ );

     my $actinfo = $acttab->{$key};

     return $self->setError(
             "Missing action in actionTable for key = $key")
             unless  ( ref($actinfo) eq "HASH" &&
                       exists ( $actinfo->{$ACTION_ID} ) );

     return $self->setError("Action for <${key}> must be a defined subroutine")
                       unless  ( ref($actinfo->{$ACTION_ID}) eq "CODE" &&
                                 defined (&{$actinfo->{$ACTION_ID}}) );

     return $self->setError("Invalid <${RUN_TYPE}> in action table for $key " .
                            " - " .  $actinfo->{$RUN_TYPE} )
                    if ( exists($actinfo->{$RUN_TYPE}) &&
                         $actinfo->{$RUN_TYPE} ne TXK::Restart::RUN_ONCE &&
                         $actinfo->{$RUN_TYPE} ne TXK::Restart::RUN_EVERYTIME );

     return $self->setError("User context must be a reference to a hash table" .
                            " - <${key}> ")
                   if ( exists($actinfo->{$USER_CONTEXT}) &&
                        ref($actinfo->{$USER_CONTEXT}) ne "HASH" );

     my $actEntry = {};

     $actEntry->{$STEP_ID}   = $key;
     $actEntry->{$ACTION_ID} = $actinfo->{$ACTION_ID};
     $actEntry->{$DESC_ID}   = ( exists ($actinfo->{$DESC_ID}) 
                                   ? $actinfo->{$DESC_ID} : "" ) ;

     $actEntry->{$STATUS_ID} = TXK::Restart::ACTION_PENDING;
     $actEntry->{$RUN_TYPE}  = TXK::Restart::RUN_ONCE;

     $actEntry->{$RUN_TYPE}  = $actinfo->{$RUN_TYPE} 
                               if ( exists($actinfo->{$RUN_TYPE}) );
     $actEntry->{$USER_CONTEXT} = $actinfo->{$USER_CONTEXT}
                                  if ( exists($actinfo->{$USER_CONTEXT}) );
            
     push @{$self->{$ACTION_TABLE}}, $actEntry;
   }

  $self->{$TABLE_LOADED} = TXK::Util::TRUE;

  if ( $self->{$USE_RESTART} )
   {
     return $self->setError(
                      "Must specify restart file when using RESTART option")
             unless  ( $self->{$RESTART_FILE} );

     my $fsys = $self->{$FILESYS_OBJ};

     my $fileExists = $fsys->access({ fileName=>$self->{$RESTART_FILE},
                                      type=>TXK::FileSys::FILE,
                                      checkMode=>TXK::FileSys::READ_ACCESS,
                                    });
     unless ( $fileExists )
      {
         $fsys->access({ fileName=>$self->{$RESTART_FILE},
                         type=>TXK::FileSys::FILE,
                         checkMode=>TXK::FileSys::CREATE_ACCESS,
                       })
            or return $self->setError("No create access for <${RESTART_FILE}>" .
                                      " $self->{$RESTART_FILE}");

         $fsys->create( {  fileName=>$self->{$RESTART_FILE},
                           type=>TXK::FileSys::FILE,
                        } );
#
#	Create Initial restart file.
#
	$self->save();
      }

#	Open and load restart flags.

     my $io = $self->{$IO_OBJ};

     if ( $self->{$CLEAR_RESTART} )
      {
        $self->{$RUNTIME_OBJ}->printStdMsg(
               "\n++++ Warning  : " . 
               "Clearing Restart file - all saved actions will be lost, " .
               "file = " . $self->{$RESTART_FILE} ."\n")
          if ( defined $self->{$RUNTIME_OBJ} );

        $io->open({ fileName=> $self->{$RESTART_FILE} ,
                    mode    => TXK::IO::WRITE } );
        $io->close();
      }

     $io->open({ fileName=> $self->{$RESTART_FILE} ,
                 mode    => TXK::IO::READ } );

     my $io_fh = $io->getFileHandle();
     my $actTab = $self->{$ACTION_TABLE};
     my $rec;

     while ( $rec = <$io_fh> )
      {
       my ($idx,$step,$status,$desc) = split(/:/,$rec,5);

       $actTab->[$idx]->{$STATUS_ID} = $status 
                         if ( $idx < scalar(@$actTab) );
      }

     $io->close();
   }

#
#	Set the action iterator to -1.

  $self->{$ACTION_ITERATOR} = -1;

  return TXK::Error::SUCCESS;
}

######################################
# save
######################################

sub save
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

#	Do nothing if action table not loaded and restart enabled.

  return TXK::Error::SUCCESS unless ( $self->{$TABLE_LOADED} &&
                                      $self->{$USE_RESTART} );

  my $io = $self->{$IO_OBJ};

  $io->open({ fileName=> $self->{$RESTART_FILE} ,
              mode    => TXK::IO::WRITE } );

  my $io_fh = $io->getFileHandle();
  my $actTab = $self->{$ACTION_TABLE};
  my $i;

  for ( $i=0; $i<scalar(@$actTab); $i++ )
   {
     my $actEntry = $actTab->[$i];

     print $io_fh "${i}:$actEntry->{$STEP_ID}:$actEntry->{$STATUS_ID}:",
                  "$actEntry->{$RUN_TYPE}:", $actEntry->{$DESC_ID}, "\n";
   }

  $io->close();

  return TXK::Error::SUCCESS;
}

######################################
# hasMoreActions
######################################

sub hasMoreActions
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

#       Do nothing if action table not loaded.

  return TXK::Error::FAIL unless ( $self->{$TABLE_LOADED} );

  my $actTab = $self->{$ACTION_TABLE};
  my $idx ;

  if ( $self->{$ACTION_ITERATOR} >= 0 ) 
   {
	# Have previous action, mark as complete and save if restart enabled.

     $idx = $self->{$ACTION_ITERATOR};

     $actTab->[$idx]->{$STATUS_ID} = TXK::Restart::ACTION_SUCCESS;

     $self->save() if ( $self->{$USE_RESTART} );

     $self->{$RUNTIME_OBJ}->printStdMsg(
			"\n++++ Completed : " . $self->_getActionInfo() )
          if ( defined $self->{$RUNTIME_OBJ} );
   }

  return TXK::Error::FAIL unless ( $self->{$ACTION_ITERATOR} + 1 <
                                     scalar(@{$self->{$ACTION_TABLE}}) );                                 
  for ($idx = $self->{$ACTION_ITERATOR} + 1; $idx<scalar(@$actTab) ; $idx++ )
   {
     last if ( $actTab->[$idx]->{$STATUS_ID} eq TXK::Restart::ACTION_PENDING ||
               $actTab->[$idx]->{$STATUS_ID} eq TXK::Restart::ACTION_RUNNING ||
               $actTab->[$idx]->{$STATUS_ID} eq TXK::Restart::ACTION_FAILED  ||
               $actTab->[$idx]->{$RUN_TYPE}  eq TXK::Restart::RUN_EVERYTIME );

     $self->{$RUNTIME_OBJ}->printStdMsg(
			"\n++++ Skipping  : " . $self->_getActionInfo($idx) ."\n")
          if ( defined $self->{$RUNTIME_OBJ} );
   }

#	At end of action table ?

  return TXK::Error::FAIL unless ( $idx < scalar(@$actTab) );

  return TXK::Error::SUCCESS;
}

######################################
# nextAction
######################################

sub nextAction
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

#       Do nothing if action table not loaded.

  return TXK::Error::FAIL unless ( $self->{$TABLE_LOADED} );

  return TXK::Error::FAIL unless ( $self->{$ACTION_ITERATOR} + 1 <
                                     scalar(@{$self->{$ACTION_TABLE}}) );

  my $actTab = $self->{$ACTION_TABLE};
  my $idx ;

#
#	Get next pending action.
#

  for ($idx = $self->{$ACTION_ITERATOR} + 1; $idx<scalar(@$actTab) ; $idx++ )
   {
     last if ( $actTab->[$idx]->{$STATUS_ID} eq TXK::Restart::ACTION_PENDING ||
               $actTab->[$idx]->{$STATUS_ID} eq TXK::Restart::ACTION_RUNNING ||
               $actTab->[$idx]->{$STATUS_ID} eq TXK::Restart::ACTION_FAILED  ||
               $actTab->[$idx]->{$RUN_TYPE}  eq TXK::Restart::RUN_EVERYTIME );
   }

#	Should always call hasMoreActions so there should be something to do.
#	But just in case...

  return TXK::Error::FAIL unless ( $idx < scalar(@$actTab) );

  $self->{$ACTION_ITERATOR} = $idx;

  $self->{$RUNTIME_OBJ}->printStdMsg("\n++++ Running   : " . 
                                     $self->_getActionInfo() . "\n")
                           if ( defined $self->{$RUNTIME_OBJ} );

# Pass back user context.

  $$args = $actTab->[$self->{$ACTION_ITERATOR}]->{$USER_CONTEXT}
              if ( defined $args && ref($args) eq "SCALAR" );
                   
  return $actTab->[$self->{$ACTION_ITERATOR}]->{$ACTION_ID};
}

######################################
# End of Public methods
######################################

# ==========================================================================

sub _getActionInfo
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  my $idx    = ( defined $args ? $args : $self->{$ACTION_ITERATOR} );
  my $actTab = $self->{$ACTION_TABLE};

  my $step_str = sprintf("%-7s",$actTab->[$idx]->{$STEP_ID});

  return $step_str . " : " . $actTab->[$idx]->{$DESC_ID} .  "\n" . 
         ( " " x 27 ) . "<" . $actTab->[$idx]->{$STATUS_ID} . ">," . 
                        "<" . $actTab->[$idx]->{$RUN_TYPE}  . ">" . 
                        " @ " . TXK::Util->getTimestamp() ;
         
}

1;

