
# +===========================================================================+
# |   Copyright (c) 2003 Oracle Corporation, Redwood Shores, California, USA
# |                         All Rights Reserved
# |                        Applications Division
# +===========================================================================+
# |
# | FILENAME
# |   Runtime.pm
# |
# | DESCRIPTION
# |      TXK Runtime package
# |
# | USAGE
# |       See Runtime.html
# |
# | PLATFORM
# |
# | NOTES
# |
# +===========================================================================+

# $Header: Runtime.pm 03-aug-2005.08:54:54 ndutko Exp $

package TXK::Runtime;

@ISA = qw( TXK::Common );

######################################
# Standard Modules
######################################

use strict;
use English;
use Carp;

require 5.005;

######################################
# Package Specific Modules
######################################

use TXK::Error();
use TXK::Log();
use TXK::Util();
use TXK::IO();
use TXK::OSD();
use TXK::FileSys();
use TXK::Process();
use TXK::XML();
use TXK::AutoConfig();
use TXK::Restart();
use TXK::Script();
use TXK::RunScript();

######################################
# Public Constants
######################################
  
######################################
# Private Constants
######################################

######################################
# Global Variables
######################################

my $GLOBAL_EXEC_COMMAND = "";

######################################
# Package Variables 
######################################

my $PACKAGE_ID = "TXK::Runtime";

######################################
# Object Keys
######################################

my $RECORD_STDOUT  = "recordStdout";
my $RECORD_STDERR  = "recordStderr";
my $REDIRECT_STDIN = "redirectStdin";
my $USE_RESTART	   = "useRestart";
my $RESTART_DIR	   = "restartDir";
my $RESTART_FILE   = "restartFile";
my $LOG_DIR        = "logDir";
my $FILE_PREFIX    = "filePrefix";
my $USE_LOG	   = "useLog";
my $ERROR_OBJ	   = "errorObj";
my $STDIN_FILE     = "stdinFile";
my $STDOUT_FILE    = "stdoutFile";
my $STDERR_FILE    = "stderrFile";
my $OUT_DIR	   = "outDir";
my $STDIN_IO       = "stdinIO";
my $STDOUT_IO      = "stdoutIO";
my $STDERR_IO      = "stderrIO";
my $LOG_OBJ   	   = "logObj";
my $FILESYS_OBJ	   = "fileSysObj";
my $PROCESS_OBJ    = "processObj";
my $CONTEXT_OBJ    = "contextObj";
my $AUTOCFG_OBJ    = "autocfgObj";
my $RESTART_OBJ    = "restartObj";
my $LOG_FILE       = "logFile";
my $RUNTIME_INIT   = "runtimeInit";
my $ACTION_TABLE   = "actionTable";
my $CLEAR_RESTART  = "clearRestartFile";
my $EXEC_COMMAND   = "execCommand";
my $EXEC_NAME      = "execName";
my $TXK_TOP 	   = "TXK_TOP";
my $REPORT_DIR     = "reportDir";
my $REPORT_FILE    = "reportFile";
my $REPORT_OBJ     = "reportObj";
my $USE_REPORT     = "useReport";
my $USE_TIMESTAMP  = "useTimestamp";

##################################################################
#
#   ACTION_TABLE
#
#      Entry : { step => <step_name>,
#                action => <function>,
#	         complete => TRUE | FALSE
#                desc => "desc" 
#
##################################################################

######################################
#
# Object Structure
# ----------------
#
#  Hash Array
#
######################################

######################################
# Package Methods 
#
# Public
#
#	new 	- build empty object
#
######################################

sub new;
sub DESTROY;
sub init;
sub create;
sub getExecCommand;
sub getExecName;
sub getTXKTop;
sub getLogDirectory;
sub getRestartDirectory;
sub printLog;
sub printReport;
sub close;
sub getLog;
sub getFileSys;
sub getReport;
sub getProcess;
sub getContext;
sub getAutoConfig;
sub stop;
sub printStdMsg;
sub printTimestamp;
sub setActionTable;
sub hasMoreActions;
sub nextAction;

######################################
# Package Methods
# 
# Private
#       All private methods are marked with a leading underscore.
#
######################################

######################################
# Constructor
######################################

sub new {

  my $type = $ARG[0];

  my $self = TXK::Common->new();

  bless $self, $PACKAGE_ID ;

  my $key;

  my %INIT_OBJ =    (
			PACKAGE_IDENT   => $PACKAGE_ID,
 			$RECORD_STDOUT  => "true",
			$RECORD_STDERR  => "true",
			$USE_RESTART	=> "true",
			$RESTART_DIR	=> "",
			$RESTART_FILE   => "",
			$REDIRECT_STDIN => "0",
			$LOG_DIR        => "",
		        $FILE_PREFIX    => "",
			$USE_LOG	=> "true",
			$REPORT_DIR     => "",
			$USE_REPORT     => "0",
			$USE_TIMESTAMP  => "true",
		        $ERROR_OBJ	=> undef,
			$STDIN_FILE	=> "",
			$STDOUT_FILE	=> "",
			$STDERR_FILE	=> "",
			$OUT_DIR	=> "",
			$STDIN_IO  	=> undef,
			$STDOUT_IO  	=> undef,
			$STDERR_IO  	=> undef,
			$LOG_OBJ	=> undef,
			$FILESYS_OBJ	=> undef,
			$PROCESS_OBJ	=> undef,
			$CONTEXT_OBJ	=> undef,
			$AUTOCFG_OBJ	=> undef,
			$RESTART_OBJ	=> undef,
			$REPORT_OBJ	=> undef,
			$LOG_FILE	=> "",
			$REPORT_FILE    => "",
			$ACTION_TABLE   => { },
			$CLEAR_RESTART  => TXK::Util::FALSE,
		        $RUNTIME_INIT   => "0",
			$EXEC_COMMAND   => "",
			$EXEC_NAME      => "",
			$TXK_TOP 	=> "",
		  );

  foreach $key (keys %INIT_OBJ)
   {
     $self->{$key} = $INIT_OBJ{$key};
   }

  return $self;
}

######################################
# Destructor
######################################

sub DESTROY
{
}

######################################
# close
######################################

sub close
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

        # Do nothing unless runtime init was successful.

  return TXK::Error::SUCCESS unless ( $self->{$RUNTIME_INIT} );

  $self->printStdMsg();
  $self->printTimestamp("Program : " . $self->getExecCommand() . " completed");
  $self->printStdMsg();

  $self->{$LOG_OBJ}->close()    if ( $self->{$USE_LOG} );
  $self->{$REPORT_OBJ}->close() if ( $self->{$USE_REPORT} );
  $self->{$STDIN_IO}->close()   if ( $self->{$REDIRECT_STDIN} );
  $self->{$STDOUT_IO}->close()  if ( $self->{$RECORD_STDOUT} );
  $self->{$STDERR_IO}->close()  if ( $self->{$RECORD_STDERR} );

  return TXK::Error::SUCCESS;
}

######################################
# Init
######################################

sub init
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];
  my $key;

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  TXK::Util->isValidArgs({args=>$args,reqd=>[]}) if defined $args;

  $self->{$ERROR_OBJ}   = TXK::Error->new();
  $self->{$FILESYS_OBJ} = TXK::FileSys->new();
  $self->{$PROCESS_OBJ} = TXK::Process->new();
  $self->{$CONTEXT_OBJ} = TXK::XML->new();
  $self->{$AUTOCFG_OBJ} = TXK::AutoConfig->new();
  $self->{$RESTART_OBJ} = TXK::Restart->new();

  my $script = TXK::Script->getClassObj();

  if ( ref($script) eq "TXK::Script" )
   {
     $self->{$EXEC_COMMAND} = $script->getScriptCommand();
     $self->{$EXEC_NAME}    = $script->getScriptName();
     $self->{$TXK_TOP}      = $script->getTXKTop();

	# Keep global copy for TXK::ERROR:

     $GLOBAL_EXEC_COMMAND   = $self->{$EXEC_COMMAND};

     $script->setRuntime($self);
   }

  foreach $key ("$RECORD_STDOUT","$RECORD_STDERR","$REDIRECT_STDIN","$USE_LOG",
                "$USE_RESTART","$USE_REPORT","$USE_TIMESTAMP")
   {
     $self->{$key} = ( exists $args->{$key}
                          ? TXK::Util->getBooleanArg($key,$args->{$key})
                          : $self->{$key}
                     );
   }

  foreach $key ("$LOG_DIR","$FILE_PREFIX","$STDIN_FILE","$STDOUT_FILE",
	        "$STDERR_FILE","$OUT_DIR","$RESTART_DIR","$RESTART_FILE",
                "$EXEC_COMMAND","$EXEC_NAME","$TXK_TOP",
                "$REPORT_FILE", "$REPORT_DIR","$LOG_FILE" )
   {
     $self->{$key} = ( exists $args->{$key}
                          ? TXK::Util->getScalarArg($key,$args->{$key})
                          : $self->{$key}
                     );
   }

  $GLOBAL_EXEC_COMMAND   = $self->{$EXEC_COMMAND};

#	Parameters have to be processed in order.

  my $dirsep = TXK::OSD->getDirSeparator();

  $self->{$LOG_DIR} = $self->getTXKTop() . 
                             $dirsep . "admin" . 
                             $dirsep . $self->getExecName() .
                             $dirsep . "log" 
                      unless ( $self->{$LOG_DIR} || 
                               $self->getTXKTop() eq "" );

  $self->{$REPORT_DIR} = $self->getTXKTop() . 
                             $dirsep . "admin" . 
                             $dirsep . $self->getExecName() .
                             $dirsep . "rep" 
                      unless ( $self->{$REPORT_DIR} || 
                               $self->getTXKTop() eq "" );

  $self->{$OUT_DIR} = $self->getTXKTop() .
                             $dirsep . "admin" .
                             $dirsep . $self->getExecName() .
                             $dirsep . "out"
                      unless ( $self->{$OUT_DIR} ||
                               $self->getTXKTop() eq "" );

  $self->{$RESTART_DIR} = $self->getTXKTop() .
                                 $dirsep . "admin" .
                                 $dirsep . $self->getExecName() .
                                 $dirsep . "restart"
                          unless ( $self->{$RESTART_DIR} ||
                               $self->getTXKTop() eq "" );

  $self->{$FILE_PREFIX} = $self->getExecName()
                                 unless $self->{$FILE_PREFIX} ;

  my $filePrefixNoTS     = $self->{$FILE_PREFIX};

  $self->{$FILE_PREFIX} .= "_" . TXK::Util->getFileNameTimestamp()
                          if ( $self->{$USE_TIMESTAMP} );

  $self->{$FILE_PREFIX} =~ s/:/_/g;
  $self->{$FILE_PREFIX} =~ s/__/_/g;

  my $fsys = TXK::FileSys->new();

  if ( $self->{$USE_RESTART} )
   {
     $self->{$RESTART_FILE} = TXK::OSD->trFileDir
                                   ( $filePrefixNoTS . ".rsf",
                                     $self->{$RESTART_DIR} )
                              unless ( $self->{$RESTART_FILE} );

     return $self->setError("Cannot specify empty <${RESTART_FILE}> when " .
                            "<${USE_RESTART}> is set to true")
             unless  ( defined $self->{$RESTART_FILE} &&
                       $self->{$RESTART_FILE} ne "" );

   }

  if ( $self->{$REDIRECT_STDIN} )
   {
     return $self->setError("Must specify <${STDIN_FILE}> when " .
                            "<${REDIRECT_STDIN}> is set to true")
             unless  ( defined $self->{$STDIN_FILE} ) ;
                      
     $self->{$STDIN_FILE} = TXK::OSD->trFileDir($self->{$STDIN_FILE});

     $self->{$STDIN_IO} = TXK::IO->new();

     $fsys->access({ fileName=>$self->{$STDIN_FILE},
                     type=>TXK::FileSys::FILE,
                     checkMode=>TXK::FileSys::READ_ACCESS,
                   })
         or return $self->setError("No read access for <${STDIN_FILE}> " .
                                   "$self->{$STDIN_FILE}");

     $self->{$STDIN_IO}->open({ fileName   => $self->{$STDIN_FILE},
                       	        handleName => "STDIN",
                       	        saveHandle => "true",
                       	        mode       => TXK::IO::READ,
                     	        });
   }

  if ( $self->{$RECORD_STDOUT} )
   {
     $self->{$STDOUT_IO} = TXK::IO->new();

     $self->{$STDOUT_FILE} 
		= TXK::OSD->trFileDir($self->{$FILE_PREFIX} . "_stdout.log",
                                      $self->{$OUT_DIR})
		      unless ( $self->{$STDOUT_FILE} );
                                      
     $fsys->access({ fileName=>$self->{$STDOUT_FILE},
                     type=>TXK::FileSys::FILE,
                     checkMode=>TXK::FileSys::CREATE_ACCESS,
                   })
         or return $self->setError("No create access for <${STDOUT_FILE}> " .
                                   "$self->{$STDOUT_FILE}");

     $fsys->create( {  fileName=>$self->{$STDOUT_FILE},
                       type=>TXK::FileSys::FILE,
                    } );

     $self->{$STDOUT_IO}->open({ fileName   => $self->{$STDOUT_FILE},
                        	 handleName => "STDOUT",
                        	 saveHandle => "true",
                        	 autoFlush  => "true",
                        	 mode       => TXK::IO::WRITE,
                     	       });
   }
  
  if ( $self->{$USE_LOG} )
   {
     $self->{$LOG_OBJ} = TXK::Log->new();

     $self->{$LOG_FILE} = $self->{$FILE_PREFIX} . ".log"
                             unless ( defined $self->{$LOG_FILE} &&
                                              $self->{$LOG_FILE} ne "" );
                                      
     $self->{$LOG_FILE} 
		= TXK::OSD->trFileDir($self->{$LOG_FILE},$self->{$LOG_DIR});
                                      
     $fsys->access({ fileName=>$self->{$LOG_FILE},
                     type=>TXK::FileSys::FILE,
                     checkMode=>TXK::FileSys::CREATE_ACCESS,
                   })
         or return $self->setError("No create access for <${LOG_FILE}> " .
                                   "$self->{$LOG_FILE}");

     $fsys->create( {  fileName=>$self->{$LOG_FILE},
                       type=>TXK::FileSys::FILE,
                    } );

     $self->{$LOG_OBJ}->open(  { fileName   => $self->{$LOG_FILE} });
   }
  
  if ( $self->{$USE_REPORT} )
   {
     $self->{$REPORT_OBJ} = TXK::Log->new();

     $self->{$REPORT_FILE} = $self->{$FILE_PREFIX} . ".rep"
                             unless ( defined $self->{$REPORT_FILE} &&
                                              $self->{$REPORT_FILE} ne "" );
                                      
     $self->{$REPORT_FILE}
                = TXK::OSD->trFileDir($self->{$REPORT_FILE},
                                      $self->{$REPORT_DIR});

     $fsys->access({ fileName=>$self->{$REPORT_FILE},
                     type=>TXK::FileSys::FILE,
                     checkMode=>TXK::FileSys::CREATE_ACCESS,
                   })
         or return $self->setError("No create access for <${REPORT_FILE}> " .
                                   "$self->{$REPORT_FILE}");

     $fsys->create( {  fileName=>$self->{$REPORT_FILE},
                       type=>TXK::FileSys::FILE,
                    } );

     $self->{$REPORT_OBJ}->open(  { fileName   => $self->{$REPORT_FILE} });
   }

#	Before STDERR disappears, display the names of the outputfiles.

  print STDERR 
        "*** ALL THE FOLLOWING FILES ARE REQUIRED FOR RESOLVING RUNTIME ERRORS\n",
        _getFileInfo($self);

  if ( $self->{$RECORD_STDERR} )
   {
     $self->{$STDERR_IO} = TXK::IO->new();

     $fsys->access({ fileName=>$self->{$STDERR_FILE},
                     type=>TXK::FileSys::FILE,
                     checkMode=>TXK::FileSys::CREATE_ACCESS,
                   })
         or return $self->setError("No create access for <${STDERR_FILE}> " .
                                   "$self->{$STDERR_FILE}")
            if ( $self->{$STDERR_FILE} );

     $fsys->create( {  fileName=>$self->{$STDERR_FILE},
                       type=>TXK::FileSys::FILE,
                    } )
            if ( $self->{$STDERR_FILE} );

     $self->{$STDERR_IO}->open({ fileName   => ( $self->{$STDERR_FILE}
                                           ? $self->{$STDERR_FILE}
                                           : "&STDOUT" ) ,
                                 handleName => "STDERR",
                                 saveHandle => "true",
                                 autoFlush  => "true",
                                 mode       => TXK::IO::WRITE,
                                   });
   }

  $self->{$RUNTIME_INIT} = TXK::Util::TRUE;

  $self->printTimestamp("Program : " . $self->getExecCommand() . " started");
  $self->printStdMsg();

  $self->printStdMsg(_getFileInfo($self));

  return TXK::Error::SUCCESS;
}

######################################
# printLog
######################################

sub printLog
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  $self->{$LOG_OBJ}->print($args) if ( $self->{$USE_LOG} );

  return TXK::Error::SUCCESS;
}

######################################
# printReport
######################################

sub printReport
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  $self->{$REPORT_OBJ}->print($args) if ( $self->{$USE_REPORT} );

  return TXK::Error::SUCCESS;
}

######################################
# printStdMsg
######################################

sub printStdMsg
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  $self->{$LOG_OBJ}->println($args) if ( $self->{$USE_LOG} );

  print $args, "\n";

  return TXK::Error::SUCCESS;
}

######################################
# printTimestamp
######################################

sub printTimestamp
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  my $msg = TXK::Util->getString($args);

  $self->printStdMsg(
                      ( $msg eq "" ? "# *** Timestamp: " : "$msg @ " ) . 
                      TXK::Util->getTimestamp()
                    );

  return TXK::Error::SUCCESS;
}

######################################
# getLog
######################################

sub getLog
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return $self->{$LOG_OBJ};
}

######################################
# getReport
######################################
  
sub getReport
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return $self->{$REPORT_OBJ};
} 

######################################
# getFileSys
######################################

sub getFileSys
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return $self->{$FILESYS_OBJ};
}

######################################
# getProcess
######################################

sub getProcess
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return $self->{$PROCESS_OBJ};
}

######################################
# getContext
######################################

sub getContext
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return $self->{$CONTEXT_OBJ};
}

######################################
# getAutoConfig
######################################

sub getAutoConfig
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return $self->{$AUTOCFG_OBJ};
}

######################################
# setActionTable
######################################

sub setActionTable
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  TXK::Util->isValidArgs({args=>$args,reqd=>["$ACTION_TABLE"]});

  my $acttab = $args->{$ACTION_TABLE};

  return $self->setError("Argument <${ACTION_TABLE}> must be a HASH table")
              unless ( ref($acttab) eq "HASH" );

  my $key;

  $self->{$CLEAR_RESTART} = TXK::Util::FALSE;

  foreach $key ("$CLEAR_RESTART")
   {
     $self->{$key} = ( exists $args->{$key}
                          ? TXK::Util->getScalarArg($key,$args->{$key})
                          : $self->{$key}
                     );
   }

  $self->{$RESTART_OBJ}->load( { restartFile => $self->{$RESTART_FILE},
                                 useRestart  => $self->{$USE_RESTART},
                                 actionTable => $acttab,
			         clearRestartFile => $self->{$CLEAR_RESTART},
                                 runtimeObj  => $self } );

  return TXK::Error::SUCCESS;
}

######################################
# hasMoreActions
######################################

sub hasMoreActions
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return $self->{$RESTART_OBJ}->hasMoreActions();
}

######################################
# nextAction
######################################

sub nextAction
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return $self->{$RESTART_OBJ}->nextAction();
}

######################################
# stop
######################################

sub stop
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  TXK::Error->stop();
}

######################################
# Create
######################################

sub create
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  my $rt = TXK::Runtime->new();

  $rt->init( ( defined $args ? $args : {} ) );

  return $rt;
}

######################################
# getExecCommand
######################################

sub getExecCommand
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,
                         package=>$PACKAGE_ID,
                         mode=>(ref($self) eq $PACKAGE_ID ? "instance":"class"),
                        });

  return $self->{$EXEC_COMMAND} if ( ref($self) eq $PACKAGE_ID );
  return $GLOBAL_EXEC_COMMAND;
}

######################################
# getExecName
######################################

sub getExecName
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return ( $self->{$EXEC_NAME} eq "" ? $PROGRAM_NAME : $self->{$EXEC_NAME} );
}

######################################
# getTXKTop
######################################

sub getTXKTop
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return $self->{$TXK_TOP};
}

######################################
# getLogDirectory
######################################

sub getLogDirectory
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return $self->{$LOG_DIR};
}

######################################
# getRestartDirectory
######################################

sub getRestartDirectory
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return $self->{$RESTART_DIR};
}

######################################
# End of Public methods
######################################

sub _getFileInfo
{
  my $self  = $ARG[0];

  my $str = "";

  $str .= "*** Log File = $self->{$LOG_FILE} \n" if $self->{$LOG_FILE};
  $str .= "*** STDOUT   = $self->{$STDOUT_FILE} \n" if $self->{$STDOUT_FILE}; 
  $str .= "*** Restart  = $self->{$RESTART_FILE} \n" if $self->{$RESTART_FILE};
  $str .= "*** Report   = $self->{$REPORT_FILE} \n" if $self->{$REPORT_FILE};

  return $str;
}

1;


