
# +===========================================================================+
# |   Copyright (c) 2003 Oracle Corporation, Redwood Shores, California, USA
# |                         All Rights Reserved
# |                        Applications Division
# +===========================================================================+
# |
# | FILENAME
# |   SQLPLUS.pm
# |
# | DESCRIPTION
# |      TXK SQLPLUS package
# |
# | USAGE
# |       See SQLPLUS.html
# |
# | PLATFORM
# |
# | NOTES
# |
# +===========================================================================+

# $Header: SQLPLUS.pm 03-aug-2005.08:54:54 ndutko Exp $

package TXK::SQLPLUS;

@ISA = qw( TXK::Common );

######################################
# Standard Modules
######################################

use strict;
use English;

use Symbol;

require 5.005;

######################################
# Package Specific Modules
######################################

use TXK::Error();
use TXK::Util();
use TXK::IO();
use TXK::FileSys();
use TXK::OSD();
use TXK::Process();

######################################
# Public Constants
######################################

######################################
# Private Constants
######################################

######################################
# Package Variables 
######################################

my $PACKAGE_ID = "TXK::SQLPLUS";

######################################
# Object Keys
######################################

my $COMMAND	   = "command";
my $COMMAND_ARRAY  = "commandArray";
my $RUNTIME_OBJ    = "runtimeObj";
my $SHOW_OUTPUT    = "showOutput";
my $SHOW_COMMAND   = "showCommand";
my $ORA_USER	   = "user";
my $ORA_PASSWORD   = "password";
my $ORA_TWO_TASK   = "two_task";
my $ORA_DB_VERSION = "version";
my $ORA_CONNECT_STR= "connectString";
my $SQLPLUS_EXE    = "sqlplus";
my $VALID_CONNECT  = "validConnectString";
my $VALID_VERSION  = "validVersion";
my $ERROR_OUTPUT   = "errorOutput";
my $ID_NUMBER      = "IdNumber";
my $PROCESS_OBJ    = "processObj";
my $FILESYS_OBJ    = "fileSysObj";

######################################
#
# Object Structure
# ----------------
#
#  Hash Array
#
######################################

######################################
# Package Methods 
#
# Public
#
#	new 	- build empty object
#
######################################

sub new;
sub DESTROY;
sub execute;
sub addCommand;
sub clearCommand;
sub setCommand;
sub getCommandArray;
sub getErrorOutput;
sub setConnectInfo;
sub validateConnectInfo;
sub setRuntime;

######################################
# Package Methods
# 
# Private
#       All private methods are marked with a leading underscore.
#
######################################

sub _doExecute;

######################################
# Constructor
######################################

sub new {
  my $type = $ARG[0];

  my $self = TXK::Common->new();

  bless $self, $PACKAGE_ID ;

  my $key;

  my %INIT_OBJ = (
		  PACKAGE_IDENT   => $PACKAGE_ID,
		  $COMMAND	=> undef,
		  $COMMAND_ARRAY  => [ ],
		  $SHOW_COMMAND   => TXK::Util::FALSE,
                  $SHOW_OUTPUT    => TXK::Util::FALSE,
	          $ORA_USER	=> "",
		  $ORA_PASSWORD   => "",
		  $ORA_TWO_TASK	=> "",
                  $ORA_DB_VERSION =>"",
		  $ORA_CONNECT_STR=> "",
		  $VALID_CONNECT  => "",
	          $VALID_VERSION  => "",
	          $SQLPLUS_EXE	=> "",
		  $ERROR_OUTPUT   => "",
		  $RUNTIME_OBJ    => undef,
		  $ID_NUMBER	=> "0",
		  $PROCESS_OBJ	=> TXK::Process->new(),
		  $FILESYS_OBJ	=> TXK::FileSys->new(),
                 );

  foreach $key (keys %INIT_OBJ)
   {
     $self->{$key} = $INIT_OBJ{$key};
   }

  return $self;
}

######################################
# Destructor
######################################

sub DESTROY
{
}

######################################
# execute 
######################################

sub execute
{   
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  $args = {} unless ( defined $args );

  TXK::Util->isValidArgs({args=>$args,reqd=>[]});

  $self->{$ERROR_OUTPUT} = "";

  $self->{$SHOW_COMMAND} = TXK::Util::FALSE;
  $self->{$SHOW_OUTPUT}  = TXK::Util::FALSE;

  my $key;

#	Now process string args.

  foreach $key ("$COMMAND","$ORA_USER","$ORA_PASSWORD","$ORA_TWO_TASK",
                "$ORA_DB_VERSION")
   {
     $self->{$key} = "", next 
                         if ( ( $key eq "$ORA_TWO_TASK" ||
                                $key eq "$ORA_DB_VERSION" ) && 
                              exists($args->{$key}) && $args->{$key} eq "" );

     $self->{$key} = ( exists $args->{$key}
                          ? TXK::Util->getScalarArg($key,$args->{$key})
                          : $self->{$key}
                     );
   }

  foreach $key ("$SHOW_COMMAND","$SHOW_OUTPUT")
   {
     $self->{$key} = ( exists $args->{$key}
                          ? TXK::Util->getBooleanArg($key,$args->{$key})
                          : $self->{$key}
                     );
   }

  $self->setCommand($self->{$COMMAND}) if ( exists($args->{$COMMAND}) );

  return $self->setError("Command Buffer is empty - nothing to execute")
			unless (scalar(@{$self->{$COMMAND_ARRAY}}) > 0 );

  $self->setRuntime($args) if ( exists($args->{$RUNTIME_OBJ}) );

  $self->validateConnectInfo() or return TXK::Error::FAIL;

  return _doExecute($self,TXK::Util::FALSE);
}

######################################
# setConnectInfo
######################################

sub setConnectInfo
{ 
  my $self  = $ARG[0];
  my $args  = $ARG[1];
  
  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});
  
  TXK::Util->isValidArgs({args=>$args,reqd=>["$ORA_USER","$ORA_PASSWORD"]});

  my $key;

  foreach $key ("$ORA_USER","$ORA_PASSWORD","$ORA_TWO_TASK","$ORA_DB_VERSION")
   {
     $self->{$key} = "", next 
                         if ( ( $key eq "$ORA_TWO_TASK" ||
                                $key eq "$ORA_DB_VERSION" ) &&
                              exists($args->{$key}) && $args->{$key} eq "" );

     $self->{$key} = ( exists $args->{$key}
                          ? TXK::Util->getScalarArg($key,$args->{$key})
                          : $self->{$key}
                     );
   }

  $self->setRuntime($args) if ( exists($args->{$RUNTIME_OBJ}) );

  return $self->validateConnectInfo();
}    

######################################
# validateConnectInfo
######################################

sub validateConnectInfo
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  my $key;

  my $is_sysdba = TXK::Util::FALSE;

  $is_sysdba = TXK::Util::TRUE 
               if ( uc($self->{$ORA_USER}) eq "SYS" &&
                    ( uc($self->{$ORA_PASSWORD}) =~ m/^\s*AS SYSDBA\s*$/ ||
                      uc($self->{$ORA_PASSWORD}) =~ m/^\s*AS_SYSDBA\s*$/ ) );

#   We no longer validate ORA_TWO_TASK, as some callers pass in a 
#   full TNS descriptor. It's not clear what would be the set of possible 
#   characters to match against - at least [a-zA-Z0-9_\-\.\,\=\(\)] 
#   We keep the ORA_TWO_TASK in the validation loop in case we do decide
#   to add a match list in the future.

  foreach $key ("$ORA_USER","$ORA_PASSWORD","$ORA_TWO_TASK")
   {
     return $self->setError("Must specify database <${key}> " )
                         unless ( $self->{$key} || $key eq "$ORA_TWO_TASK" );

     return $self->setError("Only alphanumeric characters allowed for <${key}>")
                         unless (    !$self->{$key} 
                                  || $self->{$key} =~ m/^\w+$/
                                  || $is_sysdba 
                                  || $key eq "$ORA_TWO_TASK" );
   }

  if ( $is_sysdba )
   {
     $self->{$ORA_CONNECT_STR}  = "/ " . $self->{$ORA_PASSWORD} ;

     $self->{$ORA_CONNECT_STR} =~ s/_/ /g;
   }
  else
   {
     $self->{$ORA_CONNECT_STR}  = $self->{$ORA_USER} ;
     $self->{$ORA_CONNECT_STR} .= "/" . $self->{$ORA_PASSWORD} ;

     $self->{$ORA_CONNECT_STR} .= '@' . $self->{$ORA_TWO_TASK}
                                        if ($self->{$ORA_TWO_TASK});
   }

#
#  Check for executable and ORACLE_HOME.
#

  my $ora_home = TXK::OSD->getEnvVar({ name => "ORACLE_HOME"});

  return $self->setError(
         "ORACLE_HOME env. variable must be set to validate sqlplus connection")
         unless ($ora_home);

  my $ora_home = TXK::OSD->trDirPathFromBase($ora_home);
  my $exe_ext  = TXK::OSD->getExecutableExt();

  $exe_ext = "." . $exe_ext if ( $exe_ext );

  my $sqlplus = "PLUS80" if ( -x ("$ora_home/bin/PLUS80" . $exe_ext) &&
                              TXK::OSD->isNT() );

  $sqlplus = "sqlplus" if ( -x ("$ora_home/bin/sqlplus" . $exe_ext)  
                             && !$sqlplus );

  return $self->setError("SQLPLUS executable not found in ORACLE_HOME/bin")
                        unless ( $sqlplus );

  my $proc = $self->{$PROCESS_OBJ};

  my $plus_path = $proc->which({command => $sqlplus});

  return $self->setError("SQLPLUS executable not in PATH variable")
                        unless ( $plus_path );

  my $ora_home_path = TXK::OSD->getEnvVar({ name => "ORACLE_HOME"}) .
                          TXK::OSD->getBaseDirSeparator() . "bin" .
                          TXK::OSD->getBaseDirSeparator() . $sqlplus .
                          $exe_ext ;

  return $self->setError ("SQLPLUS in PATH does not point to ORACLE_HOME/bin")
         unless ( $plus_path eq $ora_home_path ||
                  ( uc($plus_path) eq uc($ora_home_path) && TXK::OSD->isNT() ) 
                );

  $self->{$SQLPLUS_EXE} = $plus_path;

#
#       No need to run sqlplus if already validated.
#

  return TXK::Error::SUCCESS
         if ( $self->{$VALID_CONNECT} eq $self->{$ORA_CONNECT_STR} &&
              $self->{$VALID_VERSION} eq $self->{$ORA_DB_VERSION} );

  return $self->setError("Invalid connect string - cannot connect to database")
         unless ( _doExecute($self,TXK::Util::TRUE) );

  $self->{$VALID_CONNECT} = $self->{$ORA_CONNECT_STR};
  $self->{$VALID_VERSION} = $self->{$ORA_DB_VERSION};

  return TXK::Error::SUCCESS;
}

######################################
# addCommand
######################################

sub addCommand
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  push @{$self->{$COMMAND_ARRAY}}, $args if ( defined $args && !ref($args) );

  return TXK::Error::SUCCESS;
}

######################################
# setCommand
######################################

sub setCommand
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  $self->{$COMMAND_ARRAY} = [ ];

  $self->addCommand($args);

  return TXK::Error::SUCCESS;
}

######################################
# clearCommand
######################################

sub clearCommand
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  $self->{$COMMAND_ARRAY} = [ ];

  return TXK::Error::SUCCESS;
}

######################################
# getCommandArray
######################################

sub getCommandArray
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];
  
  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});
  
  return  $self->{$COMMAND_ARRAY} ;
}

######################################
# getErrorOutput
######################################

sub getErrorOutput
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return $self->{$ERROR_OUTPUT};
}

######################################
# setRuntime
######################################

sub setRuntime
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  TXK::Util->isValidArgs({args=>$args,reqd=>["$RUNTIME_OBJ"]});

  $self->{$RUNTIME_OBJ} = ( ref($args->{$RUNTIME_OBJ}) eq "TXK::Runtime" 
                                 ? $args->{$RUNTIME_OBJ} : undef );

  return TXK::Error::SUCCESS;
}

######################################
# End of Public methods
######################################

#-------------------------------------
# _doExecute
#-------------------------------------

sub _doExecute
{
  my $self        = $ARG[0];
  my $doIdCheck   = $ARG[1];

  my $rt = $self->{$RUNTIME_OBJ};
  my $fsys = $self->{$FILESYS_OBJ};

  $self->{$PROCESS_OBJ}->abortOnError({ enable => TXK::Util::FALSE });

  $self->{$ERROR_OUTPUT} = "";

  my $plus_top;

  if ( defined $rt )
   {
     $plus_top = $rt->getTXKTop() .
                     TXK::OSD->getDirSeparator() . "admin" . 
                     TXK::OSD->getDirSeparator() . $rt->getExecName() .
                     TXK::OSD->getDirSeparator() . "sqlplus" ;
   }
  else
   {
     $plus_top = TXK::OSD->getEnvVar({ name => "TXK_TOP" });

     $plus_top .= TXK::OSD->getDirSeparator() . "admin" .
                  TXK::OSD->getDirSeparator() . 
                                 TXK::OSD->getBaseName($PROGRAM_NAME) .
                  TXK::OSD->getDirSeparator() . "sqlplus"
             if ( $plus_top );
   }

  my ($plus_file,$plus_out,$plus_in);

  if ( !$doIdCheck )
   {
     $plus_file = $plus_top . ($plus_top ? TXK::OSD->getDirSeparator() :"").
                    "plus_" . TXK::Util->getFileNameTimestamp() .
                        "_" . $self->{$ID_NUMBER} . ".sql";

     $plus_out  = $plus_top . ($plus_top ? TXK::OSD->getDirSeparator() :"").
                    "plus_" . TXK::Util->getFileNameTimestamp() .
                        "_" . $self->{$ID_NUMBER} . ".out";

     $fsys->access({ fileName=>$plus_out,
                     type=>TXK::FileSys::FILE,
                     checkMode=>TXK::FileSys::CREATE_ACCESS,
                   })
       or return $self->setError("No create access for <PLUS_OUT> ". $plus_out);

     $fsys->access({ fileName=>$plus_file,
                     type=>TXK::FileSys::FILE,
                     checkMode=>TXK::FileSys::CREATE_ACCESS,
                   })
        or return $self->setError("No create access for <PLUS_FILE> " .
                                        $plus_file);
   }

#    AutoBuild and adpatch just won't allow STDIN to be cloned on NT. This
#    is clearly a bug in the way they use Perl but we need to provide some
#    solution. So for NT, we have to create a stdin file.

  if ( TXK::OSD->isNT() ) 
   {
     $plus_in  = $plus_top . ($plus_top ? TXK::OSD->getDirSeparator() :"").
                    "plus_" . TXK::Util->getFileNameTimestamp() .
                        "_" . $PROCESS_ID . ".in";

     $fsys->access({ fileName=>$plus_in,
                     type=>TXK::FileSys::FILE,
                     checkMode=>TXK::FileSys::CREATE_ACCESS,
                   })
       or return $self->setError("No create access for <PLUS_IN> ". $plus_in);
   }

#
#	Create IO handles
#

  my ($pipe_1_in,$pipe_1_out,$pipe_2_in,$pipe_2_out);

  $pipe_1_in = gensym;
  $pipe_1_out= gensym;
  $pipe_2_in = gensym;
  $pipe_2_out= gensym;

  pipe($pipe_1_in,$pipe_1_out)
          or return $self->setError("Unable to create sqlplus input pipes");

  pipe($pipe_2_in,$pipe_2_out)
          or return $self->setError("Unable to create sqlplus output pipes");

  my ($io_1_in,$io_1_out,$io_2_in,$io_2_out);

  $io_2_in = TXK::IO->new();
  $io_2_out= TXK::IO->new();

#
#	No need to worry about dangling STDIN/OUT as the IO objects
#	will abort on error.
#

#    AutoBuild and adpatch just won't allow STDIN to be cloned on NT. This
#    is clearly a bug in the way they use Perl but we need to provide some
#    solution. So for NT, we have to create a stdin file.

  if ( ! TXK::OSD->isNT() )
   {
     $io_1_in = TXK::IO->new();
     $io_1_out= TXK::IO->new();

     $io_1_in->open({ handleRef  => $pipe_1_in,
                      handleName => "STDIN",
                      saveHandle => "true",
                      mode       => TXK::IO::READ,
                    });

     $io_1_out->open({ handleRef  => $pipe_1_out,
                       autoFlush  => "true",
                       mode       => TXK::IO::WRITE,
                     });
   }
  else
   {
     $fsys->create( {  fileName=>$plus_in,  type=>TXK::FileSys::FILE, } );

     $io_1_out = TXK::Log->new();

     $io_1_out->open({ fileName=> $plus_in });
   }

  $io_2_in->open({  handleRef  => $pipe_2_in,
                    mode       => TXK::IO::READ,
                 })
            if ( $doIdCheck );

  $io_2_out->open({ handleRef  => $pipe_2_out,
                    handleName => "STDOUT",
                    saveHandle => "true",
                    autoFlush  => "true",
                    mode       => TXK::IO::WRITE,
                  })
            if ( $doIdCheck );

  close($pipe_1_in);
  close($pipe_1_out);
  close($pipe_2_in);
  close($pipe_2_out);

#
#  Write connect data
#

  $io_1_out->print("whenever sqlerror exit failure\n");
  $io_1_out->print("connect " . $self->{$ORA_CONNECT_STR} . "\n");

  if ( $doIdCheck )
   {
     $io_1_out->print("set head off\n");
     $io_1_out->print("select 'SESSION_ID:'||sys_guid() h1 from dual;\n");
     $io_1_out->print("select 'VERSION_ID:'||version h2 from v\$instance;\n")
                if ( $self->{$ORA_DB_VERSION} );

     $io_1_out->print("exit success\n");
   }
  else
   {
     $io_1_out->print("\@" . $plus_file . "\n");
     $io_1_out->print("exit success\n");

     $fsys->create( {  fileName=>$plus_file, type=>TXK::FileSys::FILE, } );
     $fsys->create( {  fileName=>$plus_out,  type=>TXK::FileSys::FILE, } );

     my $plus_log = TXK::Log->new();

     $plus_log->open({ fileName=> $plus_file });

     my $cmd_rec;

     foreach $cmd_rec (@{$self->{$COMMAND_ARRAY}})
      {
        $plus_log->println($cmd_rec);

        print $cmd_rec, "\n" if ($self->{$SHOW_COMMAND});
      }

#
#     Make sure there's a trailing / or otherwise script could hang.
#

     $plus_log->println("commit");
     $plus_log->println("/");

     $plus_log->close();
   }

  my $rc;

#	Close stdin file for NT.

  $io_1_out->close() if ( TXK::OSD->isNT() );

  if ( $doIdCheck )
   {
     $rc = $self->{$PROCESS_OBJ}->run({ command => $self->{$SQLPLUS_EXE} ,
                                        arg1    => "-silent /nolog",
                                        stdin   => ( TXK::OSD->isNT()
                                                      ? $plus_in : "0" ) });
   }
  else
   {
     $rc = $self->{$PROCESS_OBJ}->run({ command => $self->{$SQLPLUS_EXE} ,
                                        arg1    => "/nolog",
                                        showOutput => $self->{$SHOW_OUTPUT},
                                        stdout  => $plus_out ,
                                        stdin   => ( TXK::OSD->isNT()
                                                      ? $plus_in : "0" ) });
   }

  if ( ! TXK::OSD->isNT() )
   {
     $io_1_in->close();
     $io_1_out->close();
   }
  else
   {
#	Make sure we remove stdin file on NT. It contains user/pass info.

     $self->{$FILESYS_OBJ}->rmfile({ fileName => $plus_in });
   }

  $io_2_out->close() if ( $doIdCheck ) ;

  if ( $doIdCheck )
   {
     my $io_hndl = $io_2_in->getFileHandle();

     my @io_data = <$io_hndl>;
     my $rec;

     $io_2_in->close();
     
     unless ( $rc )
      {
        my $io_error;

        foreach $rec (@io_data)
         {
           $io_error .= $rec;
         }
    
        return $self->setError(
		"Invalid Connect string - cannot connect to database\n" .
                $io_error );
      }

     my $id_number;

     foreach $rec (@io_data)
      {
        next unless ( $rec =~ m/SESSION_ID:(\w+)/ );

        $id_number=$1, last;        
      }
 
     return $self->setError("Cannot find SESSION_ID: in connect buffer")
            unless ($id_number);

     $self->{$ID_NUMBER} = $id_number;

     if ( $self->{$ORA_DB_VERSION} ) 
      {
        my $db_version=undef;

        foreach $rec (@io_data)
         {
           next unless ( $rec =~ m/VERSION_ID:(\w+\.\w+\.\w+\.\w+\.\w*)/ );

           $db_version=$1, last;
         }

        my $reqd_version = $self->{$ORA_DB_VERSION};

        return $self->setError("DB Version not found - cannot validate server")
               unless (defined $db_version);

        
        return $self->setError(
			"DB Version is $db_version, expecting $reqd_version")
               unless ( $reqd_version eq 
                           substr($db_version,0,length($reqd_version)) );
      }

     return TXK::Error::SUCCESS;
   }

#
#	Get last n lines from stdout for error info.
#

  unless ( $rc )
   {
     my $io = TXK::IO->new();

     $io->open({ fileName => $plus_out });

     my $io_ref = $io->getFileHandle();
     my $io_data;
     my $no_rows = 0;
     my $error_lines = 50;

     $no_rows++ while ($io_data=<$io_ref>);

     $io->close();
     
     $io->open({ fileName => $plus_out });
     $io_ref = $io->getFileHandle();

     my $start_pt = $no_rows - $error_lines;

     $start_pt = 0 if ( $start_pt < 0 );
     $no_rows = 0;

     $self->{$ERROR_OUTPUT} = "";

     while ( $io_data=<$io_ref> )
      {
        $no_rows++;

        $self->{$ERROR_OUTPUT} .= $io_data if ($no_rows>$start_pt);
      }

#	Make sure Error_Output contains something.

     $self->{$ERROR_OUTPUT} = " " unless ( $self->{$ERROR_OUTPUT} );

     $io->close();
   }

  $self->{$FILESYS_OBJ}->rmfile({ fileName => $plus_out });
  $self->{$FILESYS_OBJ}->rmfile({ fileName => $plus_file});

  return $self->setError("SQLPLUS error: buffer=\n" .  $self->{$ERROR_OUTPUT})
               unless ( $rc );

  return TXK::Error::SUCCESS;
}

############################
#
############################

1;

