# +===========================================================================+
# |   Copyright (c) 2003 Oracle Corporation, Redwood Shores, California, USA
# |                         All Rights Reserved
# |                        Applications Division
# +===========================================================================+
# |
# | FILENAME
# |   SSL.pm
# |
# | DESCRIPTION
# |      TXK SSL package
# |
# | USAGE
# |       See SSL.html
# |
# | PLATFORM
# |
# | NOTES
# |
# +===========================================================================+

# $Header: emll/sysman/admin/scripts/TXK/SSL.pm /main/2 2008/11/16 06:44:07 ndutko Exp $

package TXK::advconfig::SSL;

@ISA = qw( TXK::Common);
@ISA = qw( TXK::Techstack);

#-------------------------------------
# Standard Modules
#-------------------------------------

use strict;
use English;
use Carp;

require 5.005;

#-------------------------------------
# Public Constants
#-------------------------------------

use constant SSL_CONFIGURED	   => "sslConfigured";
use constant SSL_ACCEL_CONFIGURED  => "sslAccelConfigured";
use constant FORMS_CONFIG_WITH_SSL => "formsConfiguredWithSSL";

use constant CONFIG_MESSAGE	   => TXK::Techstack::TXK_CONFIG_MESSAGE;
use constant VALIDATE_MESSAGE	   => TXK::Techstack::TXK_VALIDATE_MESSAGE;
use constant OUTPUT_MESSAGE	   => TXK::Techstack::TXK_OUTPUT_MESSAGE;

use constant WEBSSL_PORT	   => TXK::Techstack::TXK_WEBSSL_PORT;
use constant WEB_SSL_KEYFILE	   => TXK::Techstack::TXK_WEB_SSL_KEYFILE;
use constant WEB_SSL_CERTFILE	   => TXK::Techstack::TXK_WEB_SSL_CERTFILE;
use constant WEB_SSL_CERTCHAINFILE => TXK::Techstack::TXK_WEB_SSL_CERTCHAINFILE;

use constant SSL_CFG_ENABLE	   => TXK::Techstack::TXK_CFG_ENABLE;
use constant SSL_CFG_DISABLE	   => TXK::Techstack::TXK_CFG_DISABLE;

use constant URL_PROTOCOL	   => TXK::Techstack::TXK_URL_PROTOCOL;
use constant LOCAL_URL_PROTOCOL    => TXK::Techstack::TXK_LOCAL_URL_PROTOCOL;
use constant SSL_DEFAULT_PROTOCOL  => TXK::Techstack::TXK_SSL_DEFAULT_PROTOCOL;
use constant URL_DEFAULT_PROTOCOL  => TXK::Techstack::TXK_URL_DEFAULT_PROTOCOL;
use constant DEFAULT_WEBSSL_PORT   => TXK::Techstack::TXK_DEFAULT_WEBSSL_PORT;

use constant WEBPORT		    => TXK::Techstack::TXK_WEBPORT;
use constant LOGIN_PAGE		    => TXK::Techstack::TXK_LOGIN_PAGE;
use constant ACTIVE_WEBPORT	    => TXK::Techstack::TXK_ACTIVE_WEBPORT;
use constant WEB_ENTRY_URL_PROTOCOL => TXK::Techstack::TXK_WEB_ENTRY_URL_PROTOCOL;

use constant SSL_ACCELERATOR	   => "t_ssl_accelerator";
use constant DISCO_PROTOCOL	   => "s_disco_protocol";
use constant WEB_ENTRY_HOST	   => TXK::Techstack::TXK_WEB_ENTRY_HOST;
use constant WEB_ENTRY_DOMAIN	   => TXK::Techstack::TXK_WEB_ENTRY_DOMAIN;
use constant WEB_HOST		   => "s_webhost";
use constant DOMAIN_NAME	   => "s_domainname";
use constant MULTI_WEBNODE	   => "s_multiwebnode";
use constant WEB_SSL_DIR	   => "s_web_ssl_directory";
use constant APACHE_TOP		   => "s_tp";
use constant FRM_WALLET_DIR	   => "s_frmWalletDir";
use constant DB_WALLET_DIR	   => "t_dbWalletDir";
use constant USE_DB_WALLET	   => "t_useDBWallet";
use constant JINIT_MIN_VERSION	   => "1.1.8.03";
use constant CWALLET		   => "/cwallet.sso";
use constant EWALLET		   => "/ewallet.der";

#
# the ones below can be moved to either Inventory or Techstack modules
#

use constant FILE_CHECKMODE	   => "checkMode";
use constant FILE_TYPE		   => "type";
use constant PORT_IN_USE_ERROR     => "-14";

#-------------------------------------
# Private Constants
#-------------------------------------

use constant PORT_IN_USE_ERROR_MSG => "It appears that the specified port is already in use by another service.";

use constant PORT_IN_USE_SUCCESS_MSG => "It appears that the specified port is not currently in use by another service.";

use constant MISSING_WEBPORT => "Optional Web port is not provided. Default value of s_webport will be used.";

use constant MISSING_ACTIVE_WEBPORT => "Optional Active Web port is not provided. Value of s_webport will be used.";

use constant MISSING_ENTRY_HOST => "Optional Web entry point host is not provided. Value of s_webhost will be used.";

use constant MISSING_ENTRY_DOMAIN => "Optional Web entry point domain is not provided. Value of s_domainname will be used.";

use constant UPLOAD_AS_IS_INFO => "Value will be uploaded as shown.";

use constant VALUE_UNCHANGED => "Value remains unchanged, for it appears that your E-Business Suite Release 11i system is configured with HTTP Load-Balancer information.";

use constant MISSING_PARAMETER  => "Required input parameter and/or value is not provided.";

use constant PRESENT_PARAMETER  => "Required input parameter and/or value is provided.";

use constant FORMS_CONFIG_ERROR => "The E-Business Suite Release 11i system that is configured with SSL will not be fully secure until either the Forms Listener Servlet is enabled or the Forms server is configured with SSL using the Wallet Manager. For additional information, please see the My Oracle Support Note 123718.1 - A Guide to Understanding and Implementing SSL with Oracle Applications 11i; and the My Oracle Support Note 201340.1 - Using Forms Listener Servlet with Oracle Applications 11i.";

use constant FRMS_LSNR_SERVLET_SUCCESS	=> "It appears that this E-Business Suite Release 11i system is using the Forms Listener Servlet.";

use constant FRMS_LSNR_SERVLET_ERROR	=> "It appears that this E-Business Suite Release 11i system is not using the Forms Listener Servlet.";

use constant FRMS_SERVER_WALLET_SUCCESS => "It appears that the Forms Server is configured with SSL using the Wallet Manager.";

use constant FRMS_SERVER_WALLET_ERROR => "It appears that the Forms Server is not configured with SSL using the Wallet Manager.";

use constant DB_WALLET_WARNING_MESSAGE => "You must use Oracle Wallet Manager to set up an Oracle Wallet on the database tier for any communication in SSL mode between the database and middle tiers. For additional information, please see the My Oracle Support Note 123718.1 - A Guide to Understanding and Implementing SSL with Oracle Applications 11i.";

use constant NOT_ALLOWED_IF_MULTI_WEBNODE => "It appears that this E-Business Suite 11i system is using a multi Web-node configuration. In order to disable the E-Business Suite Release 11i system with the SSL accelerator, you must enable the HTTP load balancing first.";

use constant VALIDATION_ERROR	=> "The validation of the input parameters and other dependencies that are required for the SSL configuration has failed.";

use constant VALIDATION_SUCCESS => "The validation of the input parameters and other dependencies that are required for the SSL configuration has completed successfully.";

use constant VALIDATION_WARNING => "The validation of the input parameters and other dependencies that are required for the SSL configuration has completed with warning(s).";

use constant VALIDATION_ACCEL_ERROR => "The validation of the input parameters and other dependencies that are required for the configuration with SSL accelerator has failed.";

use constant VALIDATION_ACCEL_SUCCESS => "The validation of the input parameters and other dependencies that are required for the configuration with SSL accelerator has completed successfully.";

use constant VALIDATION_ACCEL_WARNING => "The validation of the input parameters and other dependencies that are required for the configuration with SSL accelerator has completed with warning(s).";

use constant MISSING_FILE_MSG	=> "File not found. Please verify the file exists under the indicated location.";

use constant VALID_FILE_MSG	=> "File exists.";

use constant CONFIG_SUCCESS	=> "The SSL configuration has completed successfully.";

use constant CONFIG_WARNING	=> "The SSL configuration has completed with warning(s).";

use constant CONFIG_ERROR	=> "The SSL configuration has completed with errors.";

use constant CONFIG_ACCEL_SUCCESS => "The configuration with SSL accelerator has completed successfully.";

use constant CONFIG_ACCEL_WARNING => "The configuration with SSL accelerator has completed with warning(s).";

use constant CONFIG_ACCEL_ERROR	=> "The configuration with SSL accelerator has completed with errors.";

use constant DISABLE_SUCCESS    => "The SSL was disabled successfully.";

use constant DISABLE_ERROR      => "The SSL was not disabled successfully.";

use constant DISABLE_ACCEL_SUCCESS => "The configuration with SSL accelerator was disabled successfully.";

use constant DISABLE_ACCEL_ERROR   => "The configuration with SSL accelerator was not disabled successfully.";

use constant FILE_CREATE_ERROR	=> "Cannot generate the file under the indicated location. Please verify that this user account has the permission to write to this location.";

use constant UPLOAD_SUCCESS	=> "Value was uploaded successfully to the Applications context file.";

use constant UPLOAD_ERROR	=> "An error was encountered while uploading certain context value(s) to the Applications context file.";

use constant JINIT_ERROR => "You must use JInitiator 1.1.8.3 or above which has the capability to download jar files over an SSL Web connection. For additional information, please see the My Oracle Support Note 124606.1 - Upgrading Oracle JInitiator with Oracle Applications 11i.";


#-------------------------------------
# Package Specific Modules
#-------------------------------------

use TXK::IO();
use TXK::Log();	
use TXK::OSD();
use TXK::XML();
use TXK::Util();
use TXK::Error();
use TXK::Runtime();
use TXK::Process();
use TXK::SQLPLUS();
use TXK::RunScript();
use TXK::Techstack();

#-------------------------------------
# Package Variables 
#-------------------------------------

my $PACKAGE_ID            = "TXK::advconfig::SSL";

my $DEFAULT_CERTCHAINFILE  = "/ssl.crt/ca.crt";
my $DEFAULT_CA_BUNDLE_FILE = "/ssl.crt/ca-bundle.crt";
my $DEFAULT_KEY_FILE	   = "/ssl.key/server.key";
my $DEFAULT_CERT_FILE	   = "/ssl.crt/server.crt";

my $FORMS_LSNR_SERVLET		= TXK::Techstack::TXK_FORMS_LSNR_SERVLET;
my $WALLET_MGR			= TXK::Techstack::TXK_WALLET_MGR;
my $FORMS_CONFIGURATION		= TXK::Techstack::TXK_FORMS_CONFIGURATION;

my $OUTPUT			= TXK::Techstack::TXK_WR_OUTPUT;
my $MESSAGE			= TXK::Techstack::TXK_WR_MESSAGE;
my $PARAMETER			= TXK::Techstack::TXK_WR_PARAMETER;
my $VALUE			= TXK::Techstack::TXK_WR_VALUE;
my $ERROR			= TXK::Techstack::TXK_WR_ERROR;
my $WARNING			= TXK::Techstack::TXK_WR_WARNING;
my $NOTE			= TXK::Techstack::TXK_WR_NOTE;
my $SUCCESS			= TXK::Techstack::TXK_WR_SUCCESS;
my $FAILURE			= TXK::Techstack::TXK_WR_FAILURE;
my $WARN_MSG			= TXK::Techstack::TXK_WR_WARN_MSG;

my $ENABLE			= SSL_CFG_ENABLE;
my $DISABLE			= SSL_CFG_DISABLE;

my $CONFIG_MESSAGE		= CONFIG_MESSAGE;
my $VALIDATE_MESSAGE		= VALIDATE_MESSAGE;
my $OUTPUT_MESSAGE		= OUTPUT_MESSAGE;

my $VALIDATION_ERROR		= VALIDATION_ERROR;
my $VALIDATION_SUCCESS		= VALIDATION_SUCCESS;
my $VALIDATION_WARNING		= VALIDATION_WARNING;

my $VALIDATION_ACCEL_ERROR	= VALIDATION_ACCEL_ERROR;
my $VALIDATION_ACCEL_SUCCESS	= VALIDATION_ACCEL_SUCCESS;
my $VALIDATION_ACCEL_WARNING	= VALIDATION_ACCEL_WARNING;

my $CONFIG_SUCCESS		= CONFIG_SUCCESS;
my $CONFIG_WARNING		= CONFIG_WARNING;
my $CONFIG_ERROR		= CONFIG_ERROR;

my $CONFIG_ACCEL_SUCCESS	= CONFIG_ACCEL_SUCCESS;
my $CONFIG_ACCEL_WARNING	= CONFIG_ACCEL_WARNING;
my $CONFIG_ACCEL_ERROR		= CONFIG_ACCEL_ERROR;

my $DISABLE_SUCCESS		= DISABLE_SUCCESS;
my $DISABLE_ERROR		= DISABLE_ERROR;

my $DISABLE_ACCEL_SUCCESS	= DISABLE_ACCEL_SUCCESS;
my $DISABLE_ACCEL_ERROR		= DISABLE_ACCEL_ERROR;

my $MISSING_FILE_MSG		= MISSING_FILE_MSG;
my $VALID_FILE_MSG		= VALID_FILE_MSG;
my $FILE_CREATE_ERROR		= FILE_CREATE_ERROR;
my $UPLOAD_SUCCESS		= UPLOAD_SUCCESS;
my $UPLOAD_ERROR		= UPLOAD_ERROR;
my $FORMS_CONFIG_ERROR		= FORMS_CONFIG_ERROR;
my $FRMS_LSNR_SERVLET_SUCCESS	= FRMS_LSNR_SERVLET_SUCCESS;
my $FRMS_LSNR_SERVLET_ERROR	= FRMS_LSNR_SERVLET_ERROR;
my $FRMS_SERVER_WALLET_SUCCESS	= FRMS_SERVER_WALLET_SUCCESS;
my $FRMS_SERVER_WALLET_ERROR	= FRMS_SERVER_WALLET_ERROR;
my $DB_WALLET_WARNING_MESSAGE	= DB_WALLET_WARNING_MESSAGE;
my $NOT_ALLOWED_IF_MULTI_WEBNODE = NOT_ALLOWED_IF_MULTI_WEBNODE;
my $PORT_IN_USE_ERROR		= PORT_IN_USE_ERROR;
my $PORT_IN_USE_ERROR_MSG	= PORT_IN_USE_ERROR_MSG;
my $PORT_IN_USE_SUCCESS_MSG	= PORT_IN_USE_SUCCESS_MSG;
my $MISSING_WEBPORT		= MISSING_WEBPORT;
my $MISSING_ACTIVE_WEBPORT	= MISSING_ACTIVE_WEBPORT;
my $MISSING_ENTRY_HOST		= MISSING_ENTRY_HOST;
my $MISSING_ENTRY_DOMAIN	= MISSING_ENTRY_DOMAIN;
my $UPLOAD_AS_IS_INFO		= UPLOAD_AS_IS_INFO;
my $VALUE_UNCHANGED		= VALUE_UNCHANGED;
my $MISSING_PARAMETER		= MISSING_PARAMETER;
my $PRESENT_PARAMETER		= PRESENT_PARAMETER;
my $JINIT_ERROR			= JINIT_ERROR;
my $AUTOCONFIG_MESSAGE		= TXK::Techstack::AUTOCONFIG_MESSAGE;

my $LOGIN_PAGE_PATTERN          = qr/^(http)(s?)(.+)(:[0-9]+)([^\/])(\/)(.*$)/;

#-------------------------------------
# Object Keys
#-------------------------------------

my $WEBPORT			= WEBPORT;
my $WEBSSL_PORT			= WEBSSL_PORT;
my $WEB_SSL_KEYFILE		= WEB_SSL_KEYFILE;
my $WEB_SSL_CERTFILE		= WEB_SSL_CERTFILE;
my $WEB_SSL_CERTCHAINFILE	= WEB_SSL_CERTCHAINFILE;
my $ACTIVE_WEBPORT		= ACTIVE_WEBPORT;
my $LOGIN_PAGE			= LOGIN_PAGE;
my $URL_PROTOCOL		= URL_PROTOCOL;
my $LOCAL_URL_PROTOCOL		= LOCAL_URL_PROTOCOL;
my $WEB_ENTRY_URL_PROTOCOL	= WEB_ENTRY_URL_PROTOCOL;
my $SSL_CONFIGURED		= SSL_CONFIGURED;
my $SSL_ACCEL_CONFIGURED	= SSL_ACCEL_CONFIGURED;
my $SSL_ACCELERATOR		= SSL_ACCELERATOR;
my $WEB_ENTRY_HOST		= WEB_ENTRY_HOST;
my $WEB_ENTRY_DOMAIN		= WEB_ENTRY_DOMAIN;
my $URL_DEFAULT_PROTOCOL	= URL_DEFAULT_PROTOCOL;
my $SSL_DEFAULT_PROTOCOL	= SSL_DEFAULT_PROTOCOL;
my $DISCO_PROTOCOL		= DISCO_PROTOCOL;
my $WEB_HOST			= WEB_HOST;
my $DOMAIN_NAME			= DOMAIN_NAME;
my $MULTI_WEBNODE		= MULTI_WEBNODE;
my $WEB_SSL_DIR			= WEB_SSL_DIR;
my $APACHE_TOP			= APACHE_TOP;
my $FRM_WALLET_DIR		= FRM_WALLET_DIR;
my $DB_WALLET_DIR		= DB_WALLET_DIR;
my $USE_DB_WALLET		= USE_DB_WALLET;
my $JINIT_MIN_VERSION		= JINIT_MIN_VERSION;
my $FORMS_CONFIG_WITH_SSL	= FORMS_CONFIG_WITH_SSL;
my $CWALLET			= CWALLET;
my $EWALLET			= EWALLET;

#-------------------------------------
#
# Object Structure
# ----------------
#
#  Hash Array
#
#-------------------------------------

#-------------------------------------
# Package Methods 
#
# Public
#
#       new     - build empty object
#
#-------------------------------------

sub new;
sub enable;
sub disable;
sub validate;
sub isSSLEnabled;
sub isFormsConfigWithSSL;

#-------------------------------------
# Package Methods
# 
# Private
#       All private methods are marked with a leading underscore.
#
#-------------------------------------

sub _config_stub;
sub _configSSL;
sub _configSSLAccel;
sub _disable;
sub _validate;
sub _validateWithSSLAccel;
sub _validateFile;
sub _verifyPort;
sub _modifyCtxFile;
sub _modifyCtxFile_SSLAccel;
sub _writeToLog;
sub _isFormsConfigWithSSL;
sub _verifyDBWallet;

#-------------------------------------
# Constructor
#-------------------------------------

sub new
{
  my $type = $ARG[0];
  my $args = $ARG[1];

  my $self = TXK::Techstack->new($args);

  bless $self, $PACKAGE_ID ;

  my $key;

  my %INIT_OBJ = (
		   PACKAGE_IDENT   		=> $PACKAGE_ID		,
		   $WEBPORT			=> TXK::Techstack::NULL ,
		   $WEBSSL_PORT			=> TXK::Techstack::NULL ,
		   $WEB_SSL_KEYFILE		=> TXK::Techstack::NULL ,
		   $WEB_SSL_CERTFILE		=> TXK::Techstack::NULL ,
		   $WEB_SSL_CERTCHAINFILE	=> TXK::Techstack::NULL ,
		   $WEB_ENTRY_URL_PROTOCOL	=> TXK::Techstack::NULL ,
		   $WEB_ENTRY_HOST		=> TXK::Techstack::NULL ,
		   $WEB_ENTRY_DOMAIN		=> TXK::Techstack::NULL ,
		   $URL_PROTOCOL		=> TXK::Techstack::NULL ,
		   $LOCAL_URL_PROTOCOL		=> TXK::Techstack::NULL ,
		   $ACTIVE_WEBPORT		=> TXK::Techstack::NULL ,
		   $LOGIN_PAGE			=> TXK::Techstack::NULL ,
		   $FORMS_CONFIGURATION		=> TXK::Techstack::NULL ,
		   $DISCO_PROTOCOL		=> TXK::Techstack::NULL	,
		   $WEB_HOST			=> TXK::Techstack::NULL ,
		   $DOMAIN_NAME			=> TXK::Techstack::NULL ,
		   $WEB_SSL_DIR			=> TXK::Techstack::NULL ,
		   $FRM_WALLET_DIR		=> TXK::Techstack::NULL ,
		   $DB_WALLET_DIR		=> TXK::Techstack::NULL ,
		   $SSL_ACCELERATOR		=> TXK::Util::FALSE	,
		   $USE_DB_WALLET		=> TXK::Util::FALSE	,
		   $SSL_CONFIGURED		=> undef		,
		   $SSL_ACCEL_CONFIGURED	=> undef		,
		   $FORMS_CONFIG_WITH_SSL	=> undef		,
		   $OUTPUT			=> undef		,
		   $MESSAGE			=> undef		,
		   $PARAMETER			=> undef		,
		   $VALUE			=> undef		,
		   $ERROR			=> undef		,
		   $WARNING			=> undef		,
		   $WARN_MSG			=> undef		,
		   $NOTE			=> undef		,
                 );

  foreach $key (keys %INIT_OBJ)
  {
     $self->{$key} = $INIT_OBJ{$key};
  }

  return $self;
}

#-------------------------------------
# Destructor
#-------------------------------------

sub DESTROY
{
}

#-------------------------------------
# enable
#-------------------------------------

sub enable
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  #
  # Start the validation
  #
  
  my $validate_status = $self->validate($args);

  if ( ( $validate_status eq TXK::Techstack::TXK_VLD_FAIL ) ||
       ( $validate_status eq TXK::Techstack::TXK_VLD_MISSING_FILE_ERROR )
     )
   {
     return $validate_status;

   }
  elsif ( $validate_status eq TXK::Techstack::TXK_VLD_FRMSLSNR_SERVL_WARNING )
   {
     #
     # do nothing for now.
     #

   }
  else
   {
     #
     # do nothing for now.
     #

   }

  #
  # Start the configuration
  #

  my $ssl_accelerator = TXK::Util::FALSE;

  if ( keys(%$args) )
   {
     ( $ssl_accelerator = $args->{$SSL_ACCELERATOR} ) =~ tr /a-z/A-Z/;

   }

  my $config_status = $self->_config_stub($args);

  if ( $config_status eq TXK::Techstack::TXK_CFG_FAIL )
   {
     if ( $self->txkIsPrintMsgOn() eq TXK::Util::TRUE )
      {
        print "$args->{$CONFIG_MESSAGE}";

      }

     if ( ( $ssl_accelerator =~ /(YES|Y|TRUE|T)/ ) ||
          ( $ssl_accelerator ne TXK::Techstack::NULL )
	)
      {
        $self->_writeToLog ({
			      output   => $args->{$OUTPUT_MESSAGE},
			      message  => $CONFIG_ACCEL_ERROR,
			   });

      }
     else
      {
        $self->_writeToLog ({
			      output   => $args->{$OUTPUT_MESSAGE},
			      message  => $CONFIG_ERROR,
			   });

      }

     return $config_status;
   }
  else
   {
     if ( $validate_status eq TXK::Techstack::TXK_VLD_FRMSLSNR_SERVL_WARNING )
      {

        $args->{$CONFIG_MESSAGE} = $args->{$CONFIG_MESSAGE} .
                                   "\n[PASS]\t The SSL configuration has completed with warning(s).\n";

        if ( $self->txkIsPrintMsgOn() eq TXK::Util::TRUE )
         {
           print "$args->{$CONFIG_MESSAGE}";

         }

        $self->_writeToLog ({
			      output   => $args->{$OUTPUT_MESSAGE},
			      message  => $CONFIG_WARNING,
			   });


        $config_status = TXK::Techstack::TXK_CFG_WARNING;

      }
     else
      {

        $args->{$CONFIG_MESSAGE} = $args->{$CONFIG_MESSAGE} .
                                   "\n[PASS]\t The SSL configuration has completed successfully.\n";

        if ( $self->txkIsPrintMsgOn() eq TXK::Util::TRUE )
         {
           print "$args->{$CONFIG_MESSAGE}";

         }

        if ( ( $ssl_accelerator =~ /(YES|Y|TRUE|T)/ ) ||
             ( $ssl_accelerator ne TXK::Techstack::NULL )
	   )
         {
           $self->_writeToLog ({
			         output   => $args->{$OUTPUT_MESSAGE},
			         message  => $CONFIG_ACCEL_SUCCESS,
			      });

         }
        else
         {
           $self->_writeToLog ({
			         output   => $args->{$OUTPUT_MESSAGE},
			         message  => $CONFIG_SUCCESS,
			      });

	 }
      }
   }

  return $config_status;
}

#-------------------------------------
# _config_stub
#-------------------------------------

sub _config_stub
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $config_status = TXK::Techstack::TXK_CFG_SUCCESS;

  my $SSL = TXK::Techstack::TXK_SSL;

  if ( ( keys(%$args) ) &&
       ( ( $args->{$ENABLE} =~ /$SSL/i ) || ( $args->{$DISABLE} =~ /$SSL/i ) )
     )
   {
     ( my $ssl_accelerator = $args->{$SSL_ACCELERATOR} ) =~ tr /a-z/A-Z/;

     if ( ( $ssl_accelerator =~ /(YES|Y|TRUE|T)/ ) ||
          ( $ssl_accelerator ne TXK::Techstack::NULL )
        )
      {
        $config_status = $self->_configSSLAccelerator($args);

      }
     else
      {
        $config_status = $self->_configSSL($args);

      }
   }

  return $config_status;

}

#-------------------------------------
# _configSSL
#-------------------------------------

sub _configSSL
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $config_status = TXK::Techstack::TXK_CFG_SUCCESS;

  my $SSL = TXK::Techstack::TXK_SSL;

  #
  # Also check if the Web node is configured with an HTTP Load Balancer.
  #

  my $httplb = TXK::advconfig::HttpLB->new($args);

  my $isHttpLBEnabled = $httplb->isHttpLBEnabled($args);

  if ( keys(%$args) )
   {

     if ( $args->{$ENABLE} =~ /$SSL/i )
      {
	my $ssl_keyfile = $args->{$WEB_SSL_DIR} . $DEFAULT_KEY_FILE;

	my $ssl_certfile = $args->{$WEB_SSL_DIR} . $DEFAULT_CERT_FILE;

        my $ssl_chainfile = $args->{$WEB_SSL_DIR} . $DEFAULT_CA_BUNDLE_FILE;

        $self->{$WEB_SSL_DIR} =
			TXK::OSD->trDirPathToBase($args->{$WEB_SSL_DIR});

        $self->{$WEB_SSL_KEYFILE} =
			TXK::OSD->trDirPathToBase($ssl_keyfile);

        $self->{$WEB_SSL_CERTFILE} =
			TXK::OSD->trDirPathToBase($ssl_certfile);
   
   	$self->{$WEB_SSL_CERTCHAINFILE} =
			TXK::OSD->trDirPathToBase($ssl_chainfile);

        #
        # check if the Cert Chainfile exists and is accessible
        #
   
#   	my $web_ssl_certchainfile = $self->{$WEB_SSL_CERTCHAINFILE};
#   
#        if ($self->_validateFile($web_ssl_certchainfile) eq TXK::Util::FALSE)
#         {
#   
#           my $fsys = TXK::FileSys->new();
#   
#           $fsys->abortOnError ( { enable => TXK::Util::FALSE } );
#   
#           my $return_status = $fsys->create ({
#                                           fileName  => $web_ssl_certchainfile,
#                                           type      => TXK::FileSys::FILE,
#                                     });
#   
#           if ( $return_status eq TXK::Error::FAIL )
#            {
#              $self->_writeToLog ({
#   				    output	=> $args->{$OUTPUT_MESSAGE},
#   				    note	=> $FILE_CREATE_ERROR,
#   				    parameter	=> $WEB_SSL_CERTCHAINFILE,
#   				    value	=> $web_ssl_certchainfile,
#				 });
#   
#   	      $config_status = TXK::Techstack::TXK_CFG_FAIL;
#   
#              return $config_status;
#   
#            }
#         }

        $self->{$WEBSSL_PORT} = $args->{$WEBSSL_PORT};

	#
	# If the Web node is configured with an HTTP Load Balancer,
	# do not change s_active_webport.
	#

	if ( $isHttpLBEnabled )
	 {
	   $self->{$ACTIVE_WEBPORT} = $self->txkGetContextVar({
						oavar => $ACTIVE_WEBPORT
					});
	 }
	else
	 {
	   $self->{$ACTIVE_WEBPORT} = $self->{$WEBSSL_PORT};

	 }

        $self->{$SSL_CONFIGURED} = TXK::Util::TRUE;
   
        $self->{$URL_PROTOCOL} = TXK::Techstack::TXK_SSL_DEFAULT_PROTOCOL;
   
        $self->{$LOCAL_URL_PROTOCOL} = $self->{$URL_PROTOCOL};
   
        $self->{$WEB_ENTRY_URL_PROTOCOL} = $self->{$URL_PROTOCOL};

	$self->{$WEBPORT} = $self->{$WEBSSL_PORT};

	my $host_name =
                $self->txkGetContextVar({ oavar => $WEB_ENTRY_HOST });

	my $domain_name =
                $self->txkGetContextVar({ oavar => $WEB_ENTRY_DOMAIN });

	my $login_page = $self->{$URL_PROTOCOL}         . '://' .
                         $host_name                     . "."   .
                         $domain_name                   . ":"   .
                         $self->{$ACTIVE_WEBPORT}       . "/";

	my $ctx_login_page = $self->txkGetContextVar({ oavar => $LOGIN_PAGE });

	$ctx_login_page =~ s/$LOGIN_PAGE_PATTERN/$login_page\7/gi;

	$login_page = $ctx_login_page;

	$self->{$LOGIN_PAGE} = $login_page;

      }
     elsif ( $args->{$DISABLE} =~ /$SSL/i )
      {
	if ( ( exists  ( $args->{$WEBPORT} )) &&
	     ( defined ( $args->{$WEBPORT} )) &&
	     ( $args->{$WEBPORT} ne TXK::Techstack::NULL )
	   )
	 {
	   $self->{$WEBPORT} = $args->{$WEBPORT};

	 }
        else
	 {
	   $self->{$WEBPORT} =
			$self->txkGetContextVar({ oavar => $WEBPORT });

	 }

	$self->{$SSL_CONFIGURED} = TXK::Util::FALSE;

	$self->{$URL_PROTOCOL} = TXK::Techstack::TXK_URL_DEFAULT_PROTOCOL;

	$self->{$LOCAL_URL_PROTOCOL} = $self->{$URL_PROTOCOL};

	$self->{$WEB_ENTRY_URL_PROTOCOL} = $self->{$URL_PROTOCOL};

	#
	# If the Web node is configured with an HTTP Load Balancer,
	# do not change s_active_webport.
	#

	if ( $isHttpLBEnabled )
	 {
	   $self->{$ACTIVE_WEBPORT} = $self->txkGetContextVar({
						oavar => $ACTIVE_WEBPORT
					});
	 }
	else
	 {
	   $self->{$ACTIVE_WEBPORT} = $self->{$WEBPORT};

	 }

	$self->{$WEBSSL_PORT} = TXK::Techstack::TXK_DEFAULT_WEBSSL_PORT;
   
	my $host_name =
		$self->txkGetContextVar({ oavar => $WEB_ENTRY_HOST });

	my $domain_name =
		$self->txkGetContextVar({ oavar => $WEB_ENTRY_DOMAIN });

	my $login_page = $self->{$URL_PROTOCOL}		. '://'	.
			 $host_name			. "."	.
			 $domain_name			. ":"	.
			 $self->{$ACTIVE_WEBPORT}	. "/";

	my $ctx_login_page = $self->txkGetContextVar({ oavar => $LOGIN_PAGE });

	$ctx_login_page =~ s/$LOGIN_PAGE_PATTERN/$login_page\7/gi;

	$login_page = $ctx_login_page;

	$self->{$LOGIN_PAGE} = $login_page;

      }


     if ( $self->_modifyCtxFile($args) eq TXK::Error::FAIL )
      {

        $args->{$CONFIG_MESSAGE} = $args->{$CONFIG_MESSAGE} .
                                   "\n[FAIL]\t The SSL configuration has failed.\n\t Please verify if the input parameters are valid.\n";

        $self->_writeToLog ({
			      output    => $args->{$OUTPUT_MESSAGE},
			      message	=> $UPLOAD_ERROR,
		           });

        $config_status = TXK::Techstack::TXK_CFG_FAIL;

      }

   }

  return $config_status;
}

#-------------------------------------
# _configSSLAccelerator
#-------------------------------------

sub _configSSLAccelerator
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $config_status = TXK::Techstack::TXK_CFG_SUCCESS;

  my $SSL = TXK::Techstack::TXK_SSL;

  if ( keys(%$args) )
   {

     if ( $args->{$ENABLE} =~ /$SSL/i )
      {
	$self->{$ACTIVE_WEBPORT} = $args->{$ACTIVE_WEBPORT};

	$self->{$WEBPORT} = $args->{$ACTIVE_WEBPORT};

	$self->{$WEB_ENTRY_HOST} = $args->{$WEB_ENTRY_HOST};

        $self->{$WEB_ENTRY_DOMAIN} = $args->{$WEB_ENTRY_DOMAIN};

	$self->{$WEB_ENTRY_URL_PROTOCOL} = $SSL_DEFAULT_PROTOCOL;

	$self->{$SSL_ACCEL_CONFIGURED} = TXK::Util::TRUE;

      }
     elsif ( $args->{$DISABLE} =~ /$SSL/i )
      {
        if ( ( exists  ( $args->{$ACTIVE_WEBPORT} )) &&
             ( defined ( $args->{$ACTIVE_WEBPORT} )) &&
             ( $args->{$ACTIVE_WEBPORT} ne TXK::Techstack::NULL )
           )
         {
	   $self->{$ACTIVE_WEBPORT} = $args->{$ACTIVE_WEBPORT};

	 }
	else
	 {
	   $self->{$ACTIVE_WEBPORT} =
			$self->txkGetContextVar({oavar => $WEBPORT});

	 }

        if ( ( exists  ( $args->{$WEB_ENTRY_HOST} )) &&
             ( defined ( $args->{$WEB_ENTRY_HOST} )) &&
             ( $args->{$WEB_ENTRY_HOST} ne TXK::Techstack::NULL )
           )
         {
           $self->{$WEB_ENTRY_HOST} = $args->{$WEB_ENTRY_HOST};

         }
        else
         {
           $self->{$WEB_ENTRY_HOST} =
                        $self->txkGetContextVar({ oavar => $WEB_HOST });

         }

        if ( ( exists  ( $args->{$WEB_ENTRY_DOMAIN} )) &&
             ( defined ( $args->{$WEB_ENTRY_DOMAIN} )) &&
             ( $args->{$WEB_ENTRY_DOMAIN} ne TXK::Techstack::NULL )
           )
         {
           $self->{$WEB_ENTRY_DOMAIN} = $args->{$WEB_ENTRY_DOMAIN};

         }
        else
         {
           $self->{$WEB_ENTRY_DOMAIN} =
                        $self->txkGetContextVar({ oavar => $DOMAIN_NAME });

         }

	$self->{$WEBPORT} = $args->{$ACTIVE_WEBPORT};

	$self->{$WEB_ENTRY_URL_PROTOCOL} = $URL_DEFAULT_PROTOCOL;

	$self->{$SSL_ACCEL_CONFIGURED} = TXK::Util::FALSE;

      }

     $self->{$URL_PROTOCOL} = $URL_DEFAULT_PROTOCOL;

     $self->{$LOCAL_URL_PROTOCOL} = $URL_DEFAULT_PROTOCOL;

     my $login_page = $self->{$WEB_ENTRY_URL_PROTOCOL}	. '://'	.
                      $self->{$WEB_ENTRY_HOST}		. "."	.
                      $self->{$WEB_ENTRY_DOMAIN}	. ":"	.
                      $self->{$ACTIVE_WEBPORT}		. "/";

     my $ctx_login_page = $self->txkGetContextVar({ oavar => $LOGIN_PAGE });

     $ctx_login_page =~ s/$LOGIN_PAGE_PATTERN/$login_page\7/gi;

     $login_page = $ctx_login_page;

     $self->{$LOGIN_PAGE} = $login_page;

     if ( $self->_modifyCtxFile_SSLAccel($args) eq TXK::Error::FAIL )
      {

	$args->{$CONFIG_MESSAGE} = $args->{$CONFIG_MESSAGE} .
                                   "\n[FAIL]\t The configuration with SSL accelrator has failed.\n\t Please verify if the input parameters are valid.\n";

	$self->_writeToLog ({
			      output    => $args->{$OUTPUT_MESSAGE},
			      message   => $UPLOAD_ERROR,
			   });

        $config_status = TXK::Techstack::TXK_CFG_FAIL;

      }
   }

  return $config_status;
}

#-------------------------------------
# disable
#-------------------------------------

sub disable
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $SSL = TXK::Techstack::TXK_SSL;

  if ( keys(%$args) )
   {
     ( my $ssl_accelerator = $args->{$SSL_ACCELERATOR} ) =~ tr /a-z/A-Z/;

     if ( ( $ssl_accelerator =~ /(YES|Y|TRUE|T)/ ) ||
          ( $ssl_accelerator ne TXK::Techstack::NULL )
        )
      {
        my $validate_status = $self->_validateWithSSLAccel($args);

	if ( $validate_status eq TXK::Techstack::TXK_VLD_FAIL )
	 {
	   return $validate_status;

	 }
      }
   }

  #
  # Start the configuration
  #

  my $ssl_accelerator = TXK::Util::FALSE;

  if ( keys(%$args) )
   {
     ( $ssl_accelerator = $args->{$SSL_ACCELERATOR} ) =~ tr /a-z/A-Z/;

   }

  my $config_status = $self->_config_stub($args);

  if ( $config_status eq TXK::Techstack::TXK_CFG_FAIL )
   {
     if ( $self->txkIsPrintMsgOn() eq TXK::Util::TRUE )
      {
        print "$args->{$CONFIG_MESSAGE}";

      }

     if ( ( $ssl_accelerator =~ /(YES|Y|TRUE|T)/ ) ||
          ( $ssl_accelerator ne TXK::Techstack::NULL )
        )
      {
        $self->_writeToLog ({
                              output   => $args->{$OUTPUT_MESSAGE},
                              message  => $DISABLE_ACCEL_ERROR,
                           });

      }
     else
      {
        $self->_writeToLog ({
			      output   => $args->{$OUTPUT_MESSAGE},
			      message  => $DISABLE_ERROR,
		   	   });

      }

     return $config_status;
   }
  else
   {
     $args->{$CONFIG_MESSAGE} = $args->{$CONFIG_MESSAGE} .
                              "\n[PASS]\t The SSL was disabled successfully.\n";

     if ( $self->txkIsPrintMsgOn() eq TXK::Util::TRUE )
      {
        print "$args->{$CONFIG_MESSAGE}";

      }

     if ( ( $ssl_accelerator =~ /(YES|Y|TRUE|T)/ ) ||
          ( $ssl_accelerator ne TXK::Techstack::NULL )
        )
      {
        $self->_writeToLog ({
                              output   => $args->{$OUTPUT_MESSAGE},
                              message  => $DISABLE_ACCEL_SUCCESS,
                           });

      }
     else
      {
        $self->_writeToLog ({
			      output   => $args->{$OUTPUT_MESSAGE},
			      message  => $DISABLE_SUCCESS,
			   });

      }

   }

  return $config_status;
}


#-------------------------------------
# _validateWithSSLAccel
#-------------------------------------

sub _validateWithSSLAccel
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $SSL			= TXK::Techstack::TXK_SSL;
  my $webport			= TXK::Techstack::NULL;
  my $active_webport		= TXK::Techstack::NULL;
  my $active_webport_status	= TXK::Techstack::TXK_VLD_SUCCESS;
  my $webentryhost_status	= TXK::Techstack::TXK_VLD_SUCCESS;
  my $webentrydomain_status	= TXK::Techstack::TXK_VLD_SUCCESS;
  my $disable_multiwebnode	= TXK::Techstack::TXK_VLD_SUCCESS;

  if ( keys(%$args) )
   {
     if ( $args->{$ENABLE} =~ /$SSL/i )
      {

        if ( ( exists  ( $args->{$ACTIVE_WEBPORT} )) &&
             ( defined ( $args->{$ACTIVE_WEBPORT} )) &&
             ( $args->{$ACTIVE_WEBPORT} ne TXK::Techstack::NULL )
           )
         {
           $active_webport = $args->{$ACTIVE_WEBPORT};
          
           if ( ! $self->_isPortFree($active_webport) )
            {
              $active_webport_status = PORT_IN_USE_ERROR;
          
              $self->_writeToLog({
                                   output    => $args->{$OUTPUT_MESSAGE},
                                   note      => $PORT_IN_USE_ERROR_MSG,
                                   parameter => $ACTIVE_WEBPORT,
                                   value     => $active_webport,
                                });
          	      
            }
           else
            {
	      $webport = $args->{$ACTIVE_WEBPORT};

              $self->_writeToLog({
                                   output    => $args->{$OUTPUT_MESSAGE},
                                   note      => $UPLOAD_AS_IS_INFO,
                                   parameter => $ACTIVE_WEBPORT,
                                   value     => $active_webport,
                                });

              $self->_writeToLog({
                                   output    => $args->{$OUTPUT_MESSAGE},
                                   note      => $UPLOAD_AS_IS_INFO,
                                   parameter => $WEBPORT,
                                   value     => $webport,
                                });
          
            }
         }
        else
         {
           $active_webport_status = TXK::Techstack::TXK_VLD_FAIL;

           $self->_writeToLog ({
                                 output    => $args->{$OUTPUT_MESSAGE},
                                 note      => $MISSING_PARAMETER,
                                 parameter => $ACTIVE_WEBPORT,
                                 value     => "",
                              });

         }

        if ( ( exists  ( $args->{$WEB_ENTRY_HOST} )) &&
             ( defined ( $args->{$WEB_ENTRY_HOST} )) &&
             ( $args->{$WEB_ENTRY_HOST} ne TXK::Techstack::NULL )
           )
         {
           $self->_writeToLog({
                                output    => $args->{$OUTPUT_MESSAGE},
                                note      => $PRESENT_PARAMETER,
                                parameter => $WEB_ENTRY_HOST,
                                value     => $args->{$WEB_ENTRY_HOST},
                             });

         }
        else
         {
  	   $webentryhost_status	= TXK::Techstack::TXK_VLD_FAIL;

           $self->_writeToLog({
                                output    => $args->{$OUTPUT_MESSAGE},
                                note      => $MISSING_PARAMETER,
                                parameter => $WEB_ENTRY_HOST,
                                value     => "",
                             });

         }

        if ( ( exists  ( $args->{$WEB_ENTRY_DOMAIN} )) &&
             ( defined ( $args->{$WEB_ENTRY_DOMAIN} )) &&
             ( $args->{$WEB_ENTRY_DOMAIN} ne TXK::Techstack::NULL )
           )
         {
           $self->_writeToLog({
                                output    => $args->{$OUTPUT_MESSAGE},
                                note      => $PRESENT_PARAMETER,
                                parameter => $WEB_ENTRY_DOMAIN,
                                value     => $args->{$WEB_ENTRY_DOMAIN},
                             });

         }
        else
         {
  	   $webentrydomain_status = TXK::Techstack::TXK_VLD_FAIL;

           $self->_writeToLog({
                                output    => $args->{$OUTPUT_MESSAGE},
                                note      => $MISSING_PARAMETER,
                                parameter => $WEB_ENTRY_DOMAIN,
                                value     => "",
                             });

         }
      }
     elsif ( $args->{$DISABLE} =~ /$SSL/i )
      {
	#
	# Disabling is allowed only if this is single Web-node configuration.
	#

	my $is_multiwebnode =
			$self->txkGetContextVar({ oavar => $MULTI_WEBNODE });

        $is_multiwebnode =~ tr /a-z/A-Z/;

	if ( $is_multiwebnode =~ /(YES|Y)/ )
         {
	   $disable_multiwebnode = TXK::Techstack::TXK_VLD_FAIL;

           $self->_writeToLog({
                                output    => $args->{$OUTPUT_MESSAGE},
                                message   => $NOT_ALLOWED_IF_MULTI_WEBNODE,
                             });


         }

        if ( ( exists  ( $args->{$ACTIVE_WEBPORT} )) &&
             ( defined ( $args->{$ACTIVE_WEBPORT} )) &&
             ( $args->{$ACTIVE_WEBPORT} ne TXK::Techstack::NULL )
           )
         {
           $active_webport = $args->{$ACTIVE_WEBPORT};
          
           if ( ! $self->_isPortFree($active_webport) )
            {
              $active_webport_status = PORT_IN_USE_ERROR;
          
              $self->_writeToLog({
                                   output    => $args->{$OUTPUT_MESSAGE},
                                   note      => $PORT_IN_USE_ERROR_MSG,
                                   parameter => $ACTIVE_WEBPORT,
                                   value     => $active_webport,
                                });
          	      
            }
           else
            {
	      $webport = $args->{$ACTIVE_WEBPORT};

              $self->_writeToLog({
                                   output    => $args->{$OUTPUT_MESSAGE},
                                   note      => $UPLOAD_AS_IS_INFO,
                                   parameter => $ACTIVE_WEBPORT,
                                   value     => $active_webport,
                                });

              $self->_writeToLog({
                                   output    => $args->{$OUTPUT_MESSAGE},
                                   note      => $UPLOAD_AS_IS_INFO,
                                   parameter => $WEBPORT,
                                   value     => $webport,
                                });
          
            }
         }
        else
         {
           $active_webport =
			$self->txkGetContextVar({oavar => $WEBPORT});

	   $webport = $active_webport;

           $self->_writeToLog ({
                                 output    => $args->{$OUTPUT_MESSAGE},
                                 note      => $MISSING_ACTIVE_WEBPORT,
                                 parameter => $ACTIVE_WEBPORT,
                                 value     => $active_webport,
                              });

           $self->_writeToLog({
                                output    => $args->{$OUTPUT_MESSAGE},
                                note      => $UPLOAD_AS_IS_INFO,
                                parameter => $WEBPORT,
                                value     => $webport,
                             });

         }

        if ( ( exists  ( $args->{$WEB_ENTRY_HOST} )) &&
             ( defined ( $args->{$WEB_ENTRY_HOST} )) &&
             ( $args->{$WEB_ENTRY_HOST} ne TXK::Techstack::NULL )
           )
         {
           $self->_writeToLog({
                                output    => $args->{$OUTPUT_MESSAGE},
                                note      => $PRESENT_PARAMETER,
                                parameter => $WEB_ENTRY_HOST,
                                value     => $args->{$WEB_ENTRY_HOST},
                             });

         }
        else
         {
	   my $web_entry_host =
			$self->txkGetContextVar({ oavar => $WEB_HOST });

           $self->_writeToLog({
                                output    => $args->{$OUTPUT_MESSAGE},
                                note      => $MISSING_ENTRY_HOST,
                                parameter => $WEB_ENTRY_HOST,
                                value     => $web_entry_host,
                             });

         }

        if ( ( exists  ( $args->{$WEB_ENTRY_DOMAIN} )) &&
             ( defined ( $args->{$WEB_ENTRY_DOMAIN} )) &&
             ( $args->{$WEB_ENTRY_DOMAIN} ne TXK::Techstack::NULL )
           )
         {
           $self->_writeToLog({
                                output    => $args->{$OUTPUT_MESSAGE},
                                note      => $PRESENT_PARAMETER,
                                parameter => $WEB_ENTRY_DOMAIN,
                                value     => $args->{$WEB_ENTRY_DOMAIN},
                             });

         }
        else
         {
  	   my $webentrydomain =
			$self->txkGetContextVar({ oavar => $DOMAIN_NAME });

           $self->_writeToLog({
                                output    => $args->{$OUTPUT_MESSAGE},
                                note      => $MISSING_ENTRY_DOMAIN,
                                parameter => $WEB_ENTRY_DOMAIN,
                                value     => $webentrydomain,
                             });

         }
      }
   }


  if ( keys(%$args) )
   {
     if ( ( $args->{$ENABLE} =~ /$SSL/i ) || ( $args->{$DISABLE} =~ /$SSL/i ) )
      {
	if ( ( $active_webport_status ne TXK::Techstack::TXK_VLD_SUCCESS ) ||
             ( $webentryhost_status   ne TXK::Techstack::TXK_VLD_SUCCESS ) ||
             ( $webentrydomain_status ne TXK::Techstack::TXK_VLD_SUCCESS ) ||
             ( $disable_multiwebnode  ne TXK::Techstack::TXK_VLD_SUCCESS )
           )
         {
	   return TXK::Techstack::TXK_VLD_FAIL;

         }
        else
         {
	   my $url_protocol = $URL_DEFAULT_PROTOCOL;

	   my $local_url_protocol = $URL_DEFAULT_PROTOCOL;

	   my $web_entry_url_protocol;

	   my $disco_protocol;

           if ( $args->{$ENABLE} =~ /$SSL/i )
	    {
	      $web_entry_url_protocol = $SSL_DEFAULT_PROTOCOL;

	      $disco_protocol = $SSL_DEFAULT_PROTOCOL;

	    }
	   elsif ( $args->{$DISABLE} =~ /$SSL/i )
	    {
	      $web_entry_url_protocol = $URL_DEFAULT_PROTOCOL;

	      $disco_protocol = $URL_DEFAULT_PROTOCOL;

	    }


	   my $host_name;

	   if ( ( exists  ( $args->{$WEB_ENTRY_HOST} )) &&
		( defined ( $args->{$WEB_ENTRY_HOST} )) &&
		( $args->{$WEB_ENTRY_HOST} ne TXK::Techstack::NULL )
	      )
	    {
	      $host_name = $args->{$WEB_ENTRY_HOST};

	    }
	   else
	    { 
	      $host_name = $self->txkGetContextVar({ oavar => $WEB_HOST });

	    }

	   my $domain_name;

	   if ( ( exists  ( $args->{$WEB_ENTRY_DOMAIN} )) &&
		( defined ( $args->{$WEB_ENTRY_DOMAIN} )) &&
		( $args->{$WEB_ENTRY_DOMAIN} ne TXK::Techstack::NULL )
	      )
	    {
	      $domain_name = $args->{$WEB_ENTRY_DOMAIN};

	    }
	   else
	    { 
	      $domain_name = $self->txkGetContextVar({ oavar => $DOMAIN_NAME });

	    }

           my $login_page = $web_entry_url_protocol	. '://' .
                            $host_name			. "."   .
                            $domain_name		. ":"   .
                            $active_webport		. "/";

           my $ctx_login_page = $self->txkGetContextVar({
					oavar => $LOGIN_PAGE
					});

           $ctx_login_page =~ s/$LOGIN_PAGE_PATTERN/$login_page\7/gi;

           $login_page = $ctx_login_page;

           $self->_writeToLog({
           		        output    => $args->{$OUTPUT_MESSAGE},
           		        note      => $UPLOAD_AS_IS_INFO,
           		        parameter => $URL_PROTOCOL,
           		        value     => $url_protocol,
           		     });
          
           $self->_writeToLog({
           		        output    => $args->{$OUTPUT_MESSAGE},
           		        note      => $UPLOAD_AS_IS_INFO,
           		        parameter => $LOCAL_URL_PROTOCOL,
           		        value     => $local_url_protocol,
           		     });
          
           $self->_writeToLog({
           		        output    => $args->{$OUTPUT_MESSAGE},
           		        note      => $UPLOAD_AS_IS_INFO,
           		        parameter => $WEB_ENTRY_URL_PROTOCOL,
           		        value     => $web_entry_url_protocol,
           		     });
          
           $self->_writeToLog({
           		        output    => $args->{$OUTPUT_MESSAGE},
           		        note      => $UPLOAD_AS_IS_INFO,
           		        parameter => $LOGIN_PAGE,
           		        value     => $login_page,
           		     });
	   return TXK::Techstack::TXK_VLD_SUCCESS;

         }
      }
     else
      {
	#
	# Just a place holder for now.
	#
      }


   }


  return TXK::Techstack::TXK_VLD_SUCCESS;
}

#-------------------------------------
# _validate
#-------------------------------------

sub _validate
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $webssl_port		= TXK::Techstack::NULL;
  my $web_ssl_dir		= TXK::Techstack::NULL;
  my $web_ssl_keyfile		= TXK::Techstack::NULL;
  my $web_ssl_certfile		= TXK::Techstack::NULL;
  my $web_ssl_certchainfile	= TXK::Techstack::NULL;
  my $web_ssl_ca_bundle_file	= TXK::Techstack::NULL;

  my $webssl_port_status		= TXK::Techstack::TXK_VLD_SUCCESS;
  my $web_ssl_keyfile_status		= TXK::Techstack::TXK_VLD_SUCCESS;
  my $web_ssl_certfile_status		= TXK::Techstack::TXK_VLD_SUCCESS;
  my $web_ssl_certchainfile_status	= TXK::Techstack::TXK_VLD_SUCCESS;
  my $web_ssl_ca_bundle_file_status	= TXK::Techstack::TXK_VLD_SUCCESS;
  my $formsLsnrServlet_status		= TXK::Techstack::TXK_VLD_SUCCESS;
  my $formsServer_status		= TXK::Techstack::TXK_VLD_SUCCESS;
  my $dbWallet_status			= TXK::Techstack::TXK_VLD_SUCCESS;
  my $validate_status			= TXK::Techstack::TXK_VLD_SUCCESS;

  my $SSL			= TXK::Techstack::TXK_SSL;
  my $active_webport		= TXK::Techstack::NULL;
  my $webport			= TXK::Techstack::NULL;
  my $webport_status		= TXK::Techstack::TXK_VLD_SUCCESS;

  my $jinit_version_status	= TXK::Techstack::TXK_VLD_SUCCESS;
  my $jinit_version		= $self->txkGetJinitVersion($args);

  if ( $jinit_version lt $JINIT_MIN_VERSION )
   {
     $self->_writeToLog ({
			   output   => $args->{$OUTPUT_MESSAGE},
			   message  => $JINIT_ERROR,
			});

     $jinit_version_status = TXK::Techstack::TXK_VLD_FAIL;

   }

  #
  # Also check if the Web node is configured with an HTTP Load Balancer.
  #

  my $httplb = TXK::advconfig::HttpLB->new($args);

  my $isHttpLBEnabled = $httplb->isHttpLBEnabled($args);

  if ( keys(%$args) )
   {
     if ( $args->{$ENABLE} =~ /$SSL/i )
      {
	if ( ( exists  ( $args->{$WEB_SSL_DIR} )) &&
	     ( defined ( $args->{$WEB_SSL_DIR} )) &&
	     ( $args->{$WEB_SSL_DIR} ne TXK::Techstack::NULL )
	   )	
	 {
	   $web_ssl_dir = $args->{$WEB_SSL_DIR};

	   my $ssl_keyfile = $web_ssl_dir . $DEFAULT_KEY_FILE;

	   my $ssl_certfile = $web_ssl_dir . $DEFAULT_CERT_FILE;

	   my $ssl_certchainfile = $web_ssl_dir . $DEFAULT_CERTCHAINFILE;

	   my $ssl_ca_bundle_file = $web_ssl_dir . $DEFAULT_CA_BUNDLE_FILE;

	   $web_ssl_keyfile =
			TXK::OSD->trDirPathToBase($ssl_keyfile);

	   $web_ssl_certfile =
			TXK::OSD->trDirPathToBase($ssl_certfile);

	   $web_ssl_certchainfile =
			TXK::OSD->trDirPathToBase($ssl_certchainfile);

	   $web_ssl_ca_bundle_file =
			TXK::OSD->trDirPathToBase($ssl_ca_bundle_file);
         	   
           #
           # check if s_web_ssl_keyfile exists.
           #
         
           if ( $self->_validateFile($web_ssl_keyfile) eq TXK::Error::FAIL )
            {
              $web_ssl_keyfile_status =
         		  TXK::Techstack::TXK_VLD_MISSING_FILE_ERROR;
         
              $self->_writeToLog({
         			   output    => $args->{$OUTPUT_MESSAGE},
         			   note      => $MISSING_FILE_MSG,
         			   parameter => $WEB_SSL_KEYFILE,
         			   value     => $web_ssl_keyfile,
         			});
         
            }
           else
            {
              $self->_writeToLog({
                                   output    => $args->{$OUTPUT_MESSAGE},
                                   note      => $VALID_FILE_MSG,
                                   parameter => $WEB_SSL_KEYFILE,
                                   value     => $web_ssl_keyfile,
                                });
         
            }
         	   
           #
           # check if s_web_ssl_certfile exists.
           #
         
           if ( $self->_validateFile($web_ssl_certfile) eq TXK::Error::FAIL )
            {
              $web_ssl_certfile_status =
         		  TXK::Techstack::TXK_VLD_MISSING_FILE_ERROR;
         
              $self->_writeToLog({
				   output    => $args->{$OUTPUT_MESSAGE},
         			   note	     => $MISSING_FILE_MSG,
         			   parameter => $WEB_SSL_CERTFILE,
         			   value     => $web_ssl_certfile,
         			});
         
            }
           else
            {
              $self->_writeToLog({
                                   output    => $args->{$OUTPUT_MESSAGE},
                                   note      => $VALID_FILE_MSG,
                                   parameter => $WEB_SSL_CERTFILE,
                                   value     => $web_ssl_certfile,
                                });
         
            }

           #
           # check if ca.crt exists.
           #
         	   
           if ( $self->_validateFile($web_ssl_certchainfile)
							eq TXK::Error::FAIL )
            {
              my $fsys = TXK::FileSys->new();
      
              $fsys->abortOnError ({ enable => TXK::Util::FALSE });
      
              my $return_status = $fsys->create ({
					fileName  => $web_ssl_certchainfile,
					type      => TXK::FileSys::FILE,
					});
      
              if ( $return_status eq TXK::Error::FAIL )
               {
                 $web_ssl_certchainfile_status = TXK::Techstack::TXK_VLD_FAIL;

                 $self->_writeToLog ({
      				    output	=> $args->{$OUTPUT_MESSAGE},
      				    note	=> $FILE_CREATE_ERROR,
      				    parameter	=> $WEB_SSL_CERTCHAINFILE,
      				    value	=> $web_ssl_certchainfile,
   				 });
      
               }
            }

           #
           # check if ca-bundle.crt exists.
           #

           if ( $self->_validateFile($web_ssl_ca_bundle_file)
							eq TXK::Error::FAIL )
            {
	      my $apache_top =
			$self->txkGetContextVar({ oavar => $APACHE_TOP });

	      my $tmp_web_ssl_ca_bundle_file = $apache_top .
					       "/Apache/Apache/conf" .
					       $DEFAULT_CA_BUNDLE_FILE;

	      my $src_web_ssl_ca_bundle_file =
			TXK::OSD->trDirPathToBase($tmp_web_ssl_ca_bundle_file);

	      if ( $self->_validateFile($src_web_ssl_ca_bundle_file)
							eq TXK::Error::SUCCESS )
	       {
		 my $fsys = TXK::FileSys->new();
      
		 $fsys->abortOnError ({ enable => TXK::Util::FALSE });
      
		 my $return_status = $fsys->copy ({
					source => $src_web_ssl_ca_bundle_file,
					dest   => $web_ssl_ca_bundle_file
					});
      
		 if ( $return_status eq TXK::Error::FAIL )
		  {
		    $web_ssl_ca_bundle_file_status =
						TXK::Techstack::TXK_VLD_FAIL;

		    $self->_writeToLog ({
					  output    => $args->{$OUTPUT_MESSAGE},
					  note	    => $FILE_CREATE_ERROR,
					  parameter => $WEB_SSL_CERTCHAINFILE,
					  value	    => $web_ssl_ca_bundle_file,
					});
      
                  }
		 else
		  {
		    $self->_writeToLog({
					 output    => $args->{$OUTPUT_MESSAGE},
					 note      => $VALID_FILE_MSG,
					 parameter => $WEB_SSL_CERTCHAINFILE,
					 value     => $web_ssl_ca_bundle_file,
				      });
         
		  }

	       }
	      else
	       {
                 $web_ssl_ca_bundle_file_status =
            			TXK::Techstack::TXK_VLD_MISSING_FILE_ERROR;
            
                 $self->_writeToLog({
				      output    => $args->{$OUTPUT_MESSAGE},
				      note      => $MISSING_FILE_MSG,
				      parameter => $WEB_SSL_CERTCHAINFILE,
				      value     => $src_web_ssl_ca_bundle_file,
				   });
	       }	
            }
           else
            {
              $self->_writeToLog({
                                   output    => $args->{$OUTPUT_MESSAGE},
                                   note      => $VALID_FILE_MSG,
                                   parameter => $WEB_SSL_CERTCHAINFILE,
                                   value     => $web_ssl_ca_bundle_file,
                                });
         
            }
	 }

	if ( ( exists  ( $args->{$WEBSSL_PORT} )) &&
	     ( defined ( $args->{$WEBSSL_PORT} )) &&
	     ( $args->{$WEBSSL_PORT} ne TXK::Techstack::NULL )
	   )	
	 {
           $webssl_port = $args->{$WEBSSL_PORT};
          
           if ( ! $self->_isPortFree($webssl_port) )
            {
              $webssl_port_status = PORT_IN_USE_ERROR;
          
              $self->_writeToLog({
                                   output    => $args->{$OUTPUT_MESSAGE},
                                   note      => $PORT_IN_USE_ERROR_MSG,
                                   parameter => $WEBSSL_PORT,
                                   value     => $webssl_port,
                                });
          	      
            }
           else
            {
              $self->_writeToLog({
                                   output    => $args->{$OUTPUT_MESSAGE},
                                   note      => $UPLOAD_AS_IS_INFO,
                                   parameter => $WEBSSL_PORT,
                                   value     => $webssl_port,
                                });
	      #
	      # If the Web node is configured with an HTTP Load Balancer,
	      # do not change s_active_webport.
	      #
          
	      if ( $isHttpLBEnabled )
	       {
		 $active_webport = $self->txkGetContextVar({
						oavar => $ACTIVE_WEBPORT
						});

		 $self->_writeToLog({
				      output    => $args->{$OUTPUT_MESSAGE},
				      note      => $VALUE_UNCHANGED,
				      parameter => $ACTIVE_WEBPORT,
				      value     => $active_webport,
				   });

	       }
	      else
	       {
		 $active_webport = $webssl_port;

		 $self->_writeToLog({
				      output    => $args->{$OUTPUT_MESSAGE},
				      note      => $UPLOAD_AS_IS_INFO,
				      parameter => $ACTIVE_WEBPORT,
				      value     => $active_webport,
				   });

	       }
          
          
	      $webport = $webssl_port;

              $self->_writeToLog({
                                   output    => $args->{$OUTPUT_MESSAGE},
                                   note      => $UPLOAD_AS_IS_INFO,
                                   parameter => $WEBPORT,
                                   value     => $webport,
                                });
          
          
            }
	 }

	if ( ( exists  ( $args->{$USE_DB_WALLET} )) &&
	     ( defined ( $args->{$USE_DB_WALLET} )) &&
	     ( $args->{$USE_DB_WALLET} ne TXK::Techstack::NULL )
	   )	
	 {
	   ( my $use_DBWallet = $args->{$USE_DB_WALLET} ) =~ tr /a-z/A-Z/;

	   if ( $use_DBWallet =~ (/YES|Y|TRUE|t/) )
	    {
	      $self->_writeToLog({
				   output    => $args->{$OUTPUT_MESSAGE},
				   message   => $DB_WALLET_WARNING_MESSAGE,
				});

	    }
	 }

      }
     elsif ( $args->{$DISABLE} =~ /$SSL/i )
      {
	#
	#
	#
        if ( ( exists  ( $args->{$WEBPORT} )) &&
             ( defined ( $args->{$WEBPORT} )) &&
             ( $args->{$WEBPORT} ne TXK::Techstack::NULL )
           )
         {
	   $webport = $args->{$WEBPORT};

           if ( ! $self->_isPortFree($webport) )
	    {
              $webport_status = PORT_IN_USE_ERROR;
          
              $self->_writeToLog({
                                   output    => $args->{$OUTPUT_MESSAGE},
                                   note      => $PORT_IN_USE_ERROR_MSG,
                                   parameter => $WEBPORT,
                                   value     => $webport,
                                });

            }
	   else
	    {
              $self->_writeToLog({
                                   output    => $args->{$OUTPUT_MESSAGE},
                                   note      => $UPLOAD_AS_IS_INFO,
                                   parameter => $WEBPORT,
                                   value     => $webport,
                                });
          
	    }
         }
        else
         {
	   $args->{$WEBPORT} =
				$self->txkGetContextVar({oavar => $WEBPORT});

	   $webport = $args->{$WEBPORT};

           $self->_writeToLog ({
                                 output    => $args->{$OUTPUT_MESSAGE},
                                 note      => $MISSING_WEBPORT,
                                 parameter => $WEBPORT,
                                 value     => $webport,
                              });

         }
      }
   }

  #
  # check if Forms Listener Servlet is currently set up
  #

  if ( keys(%$args) )
   {
     if ( $args->{$ENABLE} =~ /$SSL/i )
      {
	#
	# Verify if the Forms Listener Servlet is in use.
	#

        if ( $self->txkIsFormsLsnrServlet($args) )
         {
           $formsLsnrServlet_status = TXK::Techstack::TXK_VLD_SUCCESS;

	 }
	else
	 {
           $formsLsnrServlet_status = TXK::Techstack::TXK_VLD_FAIL;

         }
	
	#
	# If the Forms Listener Servlet is not in use, verify if the
	# Forms Server is configured with SSL using the Wallet Manager.
	#

	if ( $formsLsnrServlet_status ne TXK::Techstack::TXK_VLD_SUCCESS )
	 {
	   if ( $self->isFormsConfigWithSSL($args) )
	    {
	      $formsServer_status = TXK::Techstack::TXK_VLD_SUCCESS;

	    }
	   else
	    {
              $formsServer_status = TXK::Techstack::TXK_VLD_FAIL;

	    }

	 }



        if ( ( $jinit_version_status
				ne TXK::Techstack::TXK_VLD_SUCCESS ) ||
             ( $web_ssl_keyfile_status
				ne TXK::Techstack::TXK_VLD_SUCCESS ) ||
             ( $web_ssl_certfile_status
				ne TXK::Techstack::TXK_VLD_SUCCESS ) ||
             ( $web_ssl_ca_bundle_file_status
				ne TXK::Techstack::TXK_VLD_SUCCESS ) ||
             ( $web_ssl_certchainfile_status
				ne TXK::Techstack::TXK_VLD_SUCCESS ) ||
             ( $webssl_port_status
				ne TXK::Techstack::TXK_VLD_SUCCESS ) ||
             ( $dbWallet_status
				ne TXK::Techstack::TXK_VLD_SUCCESS ) ||
	     ( ( $formsLsnrServlet_status
				ne TXK::Techstack::TXK_VLD_SUCCESS ) &&
	       ( $formsServer_status
				ne TXK::Techstack::TXK_VLD_SUCCESS )
	     )
           )
         {

	   if ( ( $formsLsnrServlet_status
				ne TXK::Techstack::TXK_VLD_SUCCESS ) &&
		( $formsServer_status
				ne TXK::Techstack::TXK_VLD_SUCCESS )
	      )
	    {
	      $self->_writeToLog ({
				    output   => $args->{$OUTPUT_MESSAGE},
				    message  => $FRMS_LSNR_SERVLET_ERROR,
				 });

	      $self->_writeToLog ({
				    output   => $args->{$OUTPUT_MESSAGE},
				    message  => $FRMS_SERVER_WALLET_ERROR,
				 });

	      $self->_writeToLog ({
                                    output   => $args->{$OUTPUT_MESSAGE},
                                    message  => $FORMS_CONFIG_ERROR,
                                 });

	    }

	   $validate_status = TXK::Techstack::TXK_VLD_FAIL;

         }
        else
         {
	   my $url_protocol = $SSL_DEFAULT_PROTOCOL;

	   my $local_url_protocol = $SSL_DEFAULT_PROTOCOL;

	   my $web_entry_url_protocol = $SSL_DEFAULT_PROTOCOL;

	   my $host_name = $self->txkGetContextVar({
					oavar => 's_webentryhost'
					});

	   my $domain_name = $self->txkGetContextVar({
					oavar => 's_webentrydomain'
					});


	   my $login_page = $url_protocol 	. '://'	.
			    $host_name	  	. "."	.
			    $domain_name	. ":"	.
			    $active_webport	. "/";
			    

	   my $ctx_login_page = $self->txkGetContextVar({oavar => $LOGIN_PAGE});

	   $ctx_login_page =~ s/$LOGIN_PAGE_PATTERN/$login_page\7/gi;

	   $login_page = $ctx_login_page;

           $self->_writeToLog({
                                output    => $args->{$OUTPUT_MESSAGE},
                                note      => $UPLOAD_AS_IS_INFO,
                                parameter => $WEB_SSL_DIR,
                                value     => $web_ssl_dir,
                             });

           $self->_writeToLog({
           		        output    => $args->{$OUTPUT_MESSAGE},
           		        note      => $UPLOAD_AS_IS_INFO,
           		        parameter => $URL_PROTOCOL,
           		        value     => $url_protocol,
           		     });
          
           $self->_writeToLog({
           		        output    => $args->{$OUTPUT_MESSAGE},
           		        note      => $UPLOAD_AS_IS_INFO,
           		        parameter => $LOCAL_URL_PROTOCOL,
           		        value     => $local_url_protocol,
           		     });
          
           $self->_writeToLog({
           		        output    => $args->{$OUTPUT_MESSAGE},
           		        note      => $UPLOAD_AS_IS_INFO,
           		        parameter => $WEB_ENTRY_URL_PROTOCOL,
           		        value     => $web_entry_url_protocol,
           		     });
          
           $self->_writeToLog({
           		        output    => $args->{$OUTPUT_MESSAGE},
           		        note      => $UPLOAD_AS_IS_INFO,
           		        parameter => $LOGIN_PAGE,
           		        value     => $login_page,
           		     });

	   $validate_status = TXK::Techstack::TXK_VLD_SUCCESS;

         }

	return $validate_status;
      }
     elsif ( $args->{$DISABLE} =~ /$SSL/i )
      {
        if ( $webport_status ne TXK::Techstack::TXK_VLD_SUCCESS )
         {
	   return TXK::Techstack::TXK_VLD_FAIL;

	 }

	my $url_protocol = $URL_DEFAULT_PROTOCOL;

	my $local_url_protocol = $URL_DEFAULT_PROTOCOL;

	my $web_entry_url_protocol = $URL_DEFAULT_PROTOCOL;

	my $webssl_port = TXK::Techstack::TXK_DEFAULT_WEBSSL_PORT;

	#
	# If the Web node is configured with an HTTP Load Balancer,
	# do not change s_active_webport.
	#

	my $active_webport;

	if ( $isHttpLBEnabled )
	 {
	   $active_webport = $self->txkGetContextVar({
						oavar => $ACTIVE_WEBPORT
					});
	 }
	else
	 {
	   $active_webport = $webport;

	 }

	my $host_name = $self->txkGetContextVar({
					oavar => 's_webentryhost'
					});

	my $domain_name = $self->txkGetContextVar({
					oavar => 's_webentrydomain'
					});


	my $login_page = $url_protocol 		. '://'	.
			 $host_name	  	. "."	.
			 $domain_name	  	. ":"	.
			 $active_webport	. "/";
			    

	my $ctx_login_page = $self->txkGetContextVar({oavar => $LOGIN_PAGE});

	$ctx_login_page =~ s/$LOGIN_PAGE_PATTERN/$login_page\7/gi;

	$login_page = $ctx_login_page;

        $self->_writeToLog({
        		     output    => $args->{$OUTPUT_MESSAGE},
        		     note      => $UPLOAD_AS_IS_INFO,
           		     parameter => $URL_PROTOCOL,
        		     value     => $url_protocol,
        		  });
          
        $self->_writeToLog({
        		     output    => $args->{$OUTPUT_MESSAGE},
        		     note      => $UPLOAD_AS_IS_INFO,
        		     parameter => $LOCAL_URL_PROTOCOL,
        		     value     => $local_url_protocol,
        		  });
          
        $self->_writeToLog({
        		     output    => $args->{$OUTPUT_MESSAGE},
        		     note      => $UPLOAD_AS_IS_INFO,
        		     parameter => $WEB_ENTRY_URL_PROTOCOL,
        		     value     => $web_entry_url_protocol,
        		  });

        $self->_writeToLog({
        		     output    => $args->{$OUTPUT_MESSAGE},
        		     note      => $UPLOAD_AS_IS_INFO,
        		     parameter => $WEBSSL_PORT,
        		     value     => $webssl_port,
        		  });

	#
	# If the Web node is configured with an HTTP Load Balancer,
	# do not change s_active_webport.
	#
          
	if ( $isHttpLBEnabled )
	 {
	   $self->_writeToLog({
	  	  	        output    => $args->{$OUTPUT_MESSAGE},
	  	  	        note      => $VALUE_UNCHANGED,
	  	  	        parameter => $ACTIVE_WEBPORT,
	  	  	        value     => $active_webport,
	  	  	     });

	 }
	else
	 {
	   $self->_writeToLog({
			        output    => $args->{$OUTPUT_MESSAGE},
			        note      => $UPLOAD_AS_IS_INFO,
			        parameter => $ACTIVE_WEBPORT,
			        value     => $active_webport,
			     });

	 }
          
        $self->_writeToLog({
        		     output    => $args->{$OUTPUT_MESSAGE},
        		     note      => $UPLOAD_AS_IS_INFO,
        		     parameter => $LOGIN_PAGE,
        		     value     => $login_page,
        		  });

        return TXK::Techstack::TXK_VLD_SUCCESS;

      }
   }
}

#-------------------------------------
# validate
#-------------------------------------

sub validate
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $SSL		= TXK::Techstack::TXK_SSL;
  my $validate_status	= TXK::Techstack::TXK_VLD_SUCCESS;
  my $ssl_accelerator	= TXK::Util::FALSE;

  if ( ( keys(%$args) ) &&
       ( ( $args->{$ENABLE} =~ /$SSL/i ) || ( $args->{$DISABLE} =~ /$SSL/i ) )
     )
   {
     ( $ssl_accelerator = $args->{$SSL_ACCELERATOR} ) =~ tr /a-z/A-Z/;

     if ( ( $ssl_accelerator =~ /(YES|Y|TRUE|T)/ ) ||
          ( $ssl_accelerator ne TXK::Techstack::NULL )
        )
      {
	$validate_status = $self->_validateWithSSLAccel($args);

      }
     else
      {
	$validate_status = $self->_validate($args);

      }
   }


  if ( ( $validate_status eq TXK::Techstack::TXK_VLD_FAIL ) ||
       ( $validate_status eq TXK::Techstack::TXK_VLD_MISSING_FILE_ERROR )
     )
   {

     if ( $self->txkIsPrintMsgOn() eq TXK::Util::TRUE )
      {
        print "$args->{$VALIDATE_MESSAGE}";
      }

     if ( ( $ssl_accelerator =~ /(YES|Y|TRUE|T)/ ) ||
          ( $ssl_accelerator ne TXK::Techstack::NULL )
        )
      {
	$self->_writeToLog ({
                              output   => $args->{$OUTPUT_MESSAGE},
                              message  => $VALIDATION_ACCEL_ERROR,
                           });

      }
     else
      {
     
	$self->_writeToLog ({
                              output   => $args->{$OUTPUT_MESSAGE},
                              message  => $VALIDATION_ERROR,
                           });
      }


   }
  elsif ( $validate_status eq TXK::Techstack::TXK_VLD_FRMSLSNR_SERVL_WARNING )
   {

     if ( $self->txkIsPrintMsgOn() eq TXK::Util::TRUE )
      {
        print "$args->{$VALIDATE_MESSAGE}";
      }
     
     $self->_writeToLog ({
                           output   => $args->{$OUTPUT_MESSAGE},
                           message  => $VALIDATION_WARNING,
                        });
     
   }
  else
   {

     if ( $self->txkIsPrintMsgOn() eq TXK::Util::TRUE )
      {
        print "$args->{$VALIDATE_MESSAGE}";
      }

     if ( ( $ssl_accelerator =~ /(YES|Y|TRUE|T)/ ) ||
          ( $ssl_accelerator ne TXK::Techstack::NULL )
        )
      {
        $self->_writeToLog ({
                              output   => $args->{$OUTPUT_MESSAGE},
                              message  => $VALIDATION_ACCEL_SUCCESS,
                           });

      }
     else
      {
	if ( ( exists  ( $args->{$USE_DB_WALLET} )) &&
	     ( defined ( $args->{$USE_DB_WALLET} )) &&
	     ( $args->{$USE_DB_WALLET} ne TXK::Techstack::NULL )
	   )	
	 {
	   ( my $use_DBWallet = $args->{$USE_DB_WALLET} ) =~ tr /a-z/A-Z/;

	   if ( $use_DBWallet =~ (/YES|Y|TRUE|t/) )
	    {
	      $validate_status = TXK::Techstack::TXK_VLD_WARNING;

	      $self->_writeToLog ({
				    output  => $args->{$OUTPUT_MESSAGE},
				    message => $VALIDATION_WARNING,
				 });

	    }
           else
	    {
	      $validate_status = TXK::Techstack::TXK_VLD_SUCCESS;

	      $self->_writeToLog ({
				    output  => $args->{$OUTPUT_MESSAGE},
				    message => $VALIDATION_SUCCESS,
				 });

	    }
	 }
        else
         {
	   $self->_writeToLog ({
				 output  => $args->{$OUTPUT_MESSAGE},
				 message => $VALIDATION_SUCCESS,
			      });
	 }
      }

   }

 return $validate_status;

}

#-------------------------------------
# _validateFile
#-------------------------------------

sub _validateFile
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $fstatus = TXK::Error::FAIL;

  my $fsys = TXK::FileSys->new();

  $fstatus = $fsys->access({
		     	     fileName   => $args,
		     	     type       => TXK::FileSys::FILE,
		     	     checkMode  => TXK::FileSys::READ_ACCESS
                  	  });

  return $fstatus;

}

#-------------------------------------
# _modifyCtxFile
#-------------------------------------

sub _modifyCtxFile
{

  my $self = $ARG[0];
  my $args = $ARG[1];

  my $store_status = TXK::Error::SUCCESS;

  my $SSL = TXK::Techstack::TXK_SSL;

  my $ctxfile = $self->txkGetAppsCtxFile();

  my $ctxobj = TXK::AutoConfig->new();

  $ctxobj->loadContext({ file => $ctxfile });

  $ctxobj->setDerivedMode('true');

  #
  # Also check if the Web node is configured with an HTTP Load Balancer.
  #

  my $httplb = TXK::advconfig::HttpLB->new($args);

  my $isHttpLBEnabled = $httplb->isHttpLBEnabled($args);


  if ( $args->{$ENABLE} =~ /$SSL/i )
   {
     $ctxobj->setOAVar ({
		$WEB_SSL_DIR		=> $self->{$WEB_SSL_DIR},
		$WEB_SSL_KEYFILE	=> $self->{$WEB_SSL_KEYFILE},
		$WEB_SSL_CERTFILE	=> $self->{$WEB_SSL_CERTFILE},
		$WEB_SSL_CERTCHAINFILE	=> $self->{$WEB_SSL_CERTCHAINFILE},
		$WEBSSL_PORT		=> $self->{$WEBSSL_PORT},
		$LOGIN_PAGE		=> $self->{$LOGIN_PAGE},
		$URL_PROTOCOL		=> $self->{$URL_PROTOCOL},
		$LOCAL_URL_PROTOCOL	=> $self->{$LOCAL_URL_PROTOCOL},
		$WEB_ENTRY_URL_PROTOCOL	=> $self->{$WEB_ENTRY_URL_PROTOCOL},
		$ACTIVE_WEBPORT		=> $self->{$ACTIVE_WEBPORT},
		$WEBPORT		=> $self->{$WEBPORT}
		});

     $self->_writeToLog ({
   			   output    => $args->{$OUTPUT_MESSAGE},
   			   note      => $UPLOAD_SUCCESS,
   			   parameter => $WEB_SSL_DIR,
   			   value     => $self->{$WEB_SSL_DIR},
   			});
   
     $self->_writeToLog ({
   			   output    => $args->{$OUTPUT_MESSAGE},
   			   note      => $UPLOAD_SUCCESS,
   			   parameter => $WEB_SSL_KEYFILE,
   			   value     => $self->{$WEB_SSL_KEYFILE},
   			});
   
     $self->_writeToLog ({
			   output    => $args->{$OUTPUT_MESSAGE},
			   note      => $UPLOAD_SUCCESS,
			   parameter => $WEB_SSL_CERTFILE,
			   value     => $self->{$WEB_SSL_CERTFILE},
			});
   
     $self->_writeToLog({
			  output    => $args->{$OUTPUT_MESSAGE},
			  note      => $UPLOAD_SUCCESS,
			  parameter => $WEB_SSL_CERTCHAINFILE,
			  value     => $self->{$WEB_SSL_CERTCHAINFILE},
			});
   
     $self->_writeToLog ({
			   output    => $args->{$OUTPUT_MESSAGE},
			   note      => $UPLOAD_SUCCESS,
			   parameter => $WEBSSL_PORT,
			   value     => $self->{$WEBSSL_PORT},
			});
   
   }
  elsif ( $args->{$DISABLE} =~ /$SSL/i )
   {
     $ctxobj->setOAVar ({
		$WEBSSL_PORT		=> $self->{$WEBSSL_PORT},
		$LOGIN_PAGE		=> $self->{$LOGIN_PAGE},
		$URL_PROTOCOL		=> $self->{$URL_PROTOCOL},
		$LOCAL_URL_PROTOCOL	=> $self->{$LOCAL_URL_PROTOCOL},
		$WEB_ENTRY_URL_PROTOCOL	=> $self->{$WEB_ENTRY_URL_PROTOCOL},
		$ACTIVE_WEBPORT		=> $self->{$ACTIVE_WEBPORT},
		$WEBPORT		=> $self->{$WEBPORT}
		});

   }

  #
  # The following settings are common whether ENABLE or DISABLE
  #

  $self->_writeToLog ({
                        output    => $args->{$OUTPUT_MESSAGE},
                        note      => $UPLOAD_SUCCESS,
                        parameter => $WEBSSL_PORT,
                        value     => $self->{$WEBSSL_PORT},
                     });

  $self->_writeToLog ({
                        output    => $args->{$OUTPUT_MESSAGE},
                        note      => $UPLOAD_SUCCESS,
                        parameter => $LOGIN_PAGE,
                        value     => $self->{$LOGIN_PAGE},
                     });

  $self->_writeToLog ({
                        output    => $args->{$OUTPUT_MESSAGE},
                        note      => $UPLOAD_SUCCESS,
                        parameter => $URL_PROTOCOL,
                        value     => $self->{$URL_PROTOCOL},
                     });

  $self->_writeToLog ({
                        output    => $args->{$OUTPUT_MESSAGE},
                        note      => $UPLOAD_SUCCESS,
                        parameter => $LOCAL_URL_PROTOCOL,
                        value     => $self->{$LOCAL_URL_PROTOCOL},
                     });

  $self->_writeToLog ({
                        output    => $args->{$OUTPUT_MESSAGE},
                        note      => $UPLOAD_SUCCESS,
                        parameter => $WEB_ENTRY_URL_PROTOCOL,
                        value     => $self->{$WEB_ENTRY_URL_PROTOCOL},
                     });

  #
  # If the Web node is configured with an HTTP Load Balancer,
  # do not change s_active_webport.
  #

  if ( $isHttpLBEnabled )
   {
     $self->_writeToLog ({
                           output    => $args->{$OUTPUT_MESSAGE},
                           note      => $VALUE_UNCHANGED,
                           parameter => $ACTIVE_WEBPORT,
                           value     => $self->{$ACTIVE_WEBPORT},
		        });

   }
  else
   {
     $self->_writeToLog ({
                           output    => $args->{$OUTPUT_MESSAGE},
                           note      => $UPLOAD_SUCCESS,
                           parameter => $ACTIVE_WEBPORT,
                           value     => $self->{$ACTIVE_WEBPORT},
		        });

   }

  $self->_writeToLog ({
                        output    => $args->{$OUTPUT_MESSAGE},
                        note      => $UPLOAD_SUCCESS,
                        parameter => $WEBPORT,
                        value     => $self->{$WEBPORT},
                     });


  return $store_status;

}

#-------------------------------------
# _modifyCtxFile_SSLAccel
#-------------------------------------

sub _modifyCtxFile_SSLAccel
{

  my $self = $ARG[0];
  my $args = $ARG[1];

  my $store_status = TXK::Error::SUCCESS;

  my $SSL = TXK::Techstack::TXK_SSL;

  my $ctxfile = $self->txkGetAppsCtxFile();

  my $ctxobj = TXK::AutoConfig->new();

  $ctxobj->loadContext({ file => $ctxfile });

  $ctxobj->setDerivedMode('true');

  if ( ( $args->{$ENABLE} =~ /$SSL/i ) || ( $args->{$DISABLE} =~ /$SSL/i ) )
   {
     $ctxobj->setOAVar ({
		$WEBPORT		=> $self->{$WEBPORT},
		$ACTIVE_WEBPORT		=> $self->{$ACTIVE_WEBPORT},
		$WEB_ENTRY_HOST		=> $self->{$WEB_ENTRY_HOST},
		$WEB_ENTRY_DOMAIN	=> $self->{$WEB_ENTRY_DOMAIN},
		$WEB_ENTRY_URL_PROTOCOL	=> $self->{$WEB_ENTRY_URL_PROTOCOL},
		$LOGIN_PAGE		=> $self->{$LOGIN_PAGE},	
		$URL_PROTOCOL		=> $self->{$URL_PROTOCOL},
		$LOCAL_URL_PROTOCOL	=> $self->{$LOCAL_URL_PROTOCOL},
		});

  $self->_writeToLog ({
                        output    => $args->{$OUTPUT_MESSAGE},
                        note      => $UPLOAD_SUCCESS,
                        parameter => $WEBPORT,
                        value     => $self->{$WEBPORT},
                     });

  $self->_writeToLog ({
                        output    => $args->{$OUTPUT_MESSAGE},
                        note      => $UPLOAD_SUCCESS,
                        parameter => $ACTIVE_WEBPORT,
                        value     => $self->{$ACTIVE_WEBPORT},
		     });

  $self->_writeToLog ({
                        output    => $args->{$OUTPUT_MESSAGE},
                        note      => $UPLOAD_SUCCESS,
                        parameter => $WEB_ENTRY_HOST,
                        value     => $self->{$WEB_ENTRY_HOST},
                     });

  $self->_writeToLog ({
                        output    => $args->{$OUTPUT_MESSAGE},
                        note      => $UPLOAD_SUCCESS,
                        parameter => $WEB_ENTRY_DOMAIN,
                        value     => $self->{$WEB_ENTRY_DOMAIN},
                     });

  $self->_writeToLog ({
                        output    => $args->{$OUTPUT_MESSAGE},
                        note      => $UPLOAD_SUCCESS,
                        parameter => $WEB_ENTRY_URL_PROTOCOL,
                        value     => $self->{$WEB_ENTRY_URL_PROTOCOL},
                     });

  $self->_writeToLog ({
                        output    => $args->{$OUTPUT_MESSAGE},
                        note      => $UPLOAD_SUCCESS,
                        parameter => $LOGIN_PAGE,
                        value     => $self->{$LOGIN_PAGE},
                     });

  $self->_writeToLog ({
                        output    => $args->{$OUTPUT_MESSAGE},
                        note      => $UPLOAD_SUCCESS,
                        parameter => $URL_PROTOCOL,
                        value     => $self->{$URL_PROTOCOL},
                     });

  $self->_writeToLog ({
                        output    => $args->{$OUTPUT_MESSAGE},
                        note      => $UPLOAD_SUCCESS,
                        parameter => $LOCAL_URL_PROTOCOL,
                        value     => $self->{$LOCAL_URL_PROTOCOL},
                     });

   }
  else
   {
     #
     # Place holder.
     #
   }

  return $store_status;

}



#-------------------------------------
# printSSLArg
#-------------------------------------

sub printSSLArg
{
  my $self = $ARG[0];

  print "\n$WEBPORT = $self->{$WEBPORT}\n";

  print "$WEBSSL_PORT = $self->{$WEBSSL_PORT}\n";

  print "$WEB_SSL_KEYFILE = $self->{$WEB_SSL_KEYFILE}\n";

  print "$WEB_SSL_CERTFILE = $self->{$WEB_SSL_CERTFILE}\n";

  print "$WEB_SSL_CERTCHAINFILE = $self->{$WEB_SSL_CERTCHAINFILE}\n";

  print "$ACTIVE_WEBPORT = $self->{$ACTIVE_WEBPORT}\n";

  print "$URL_PROTOCOL = $self->{$URL_PROTOCOL}\n";

  print "$WEB_ENTRY_URL_PROTOCOL = $self->{$WEB_ENTRY_URL_PROTOCOL}\n";

  print "$LOCAL_URL_PROTOCOL = $self->{$LOCAL_URL_PROTOCOL}\n";

  print "$LOGIN_PAGE = $self->{$LOGIN_PAGE}\n";

  print "$SSL_CONFIGURED = $self->{$SSL_CONFIGURED}\n";

}

#-------------------------------------
# isSSLEnabled
#-------------------------------------

sub isSSLEnabled
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( !defined($self->{$SSL_CONFIGURED}) )
   {
     $self->{$SSL_CONFIGURED} = $self->txkIsSSLConfigured($args);

   }

  return $self->{$SSL_CONFIGURED};

}

#-------------------------------------
# _writeToLog
#-------------------------------------

sub _writeToLog
{

  my $self      = $ARG[0];
  my $args      = $ARG[1];

  my $val	= TXK::Techstack::TXK_WR_VALUE;
  my $msg       = TXK::Techstack::TXK_WR_MESSAGE;
  my $warn_msg  = TXK::Techstack::TXK_WR_WARN_MSG;
  my $note      = TXK::Techstack::TXK_WR_NOTE;

  $self->{$NOTE}	= undef;
  $self->{$VALUE}	= undef;
  $self->{$ERROR}	= undef;
  $self->{$MESSAGE}	= undef;
  $self->{$WARNING}	= undef;
  $self->{$WARN_MSG}	= undef;
  $self->{$WARNING}	= undef;

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  TXK::Util->isValidArgs({args=>$args,reqd=>["$OUTPUT"]});

  foreach my $key ( keys %$args )
   {
     if ( $key eq $OUTPUT )
      {
	$self->{$OUTPUT} = $args->{$key};

      }
     elsif ( $key eq $ERROR )
      {
	$self->{$ERROR} = $args->{$key};

      }
     elsif ( $key eq $WARNING )
      {
	$self->{$WARNING} = $args->{$key};

      }
     elsif ( $key eq $WARN_MSG )
      {
	$self->{$WARN_MSG} = $args->{$key};

      }
     elsif ( $key eq $MESSAGE )
      {
	$self->{$MESSAGE} = $args->{$key};

      }
     elsif ( $key eq $NOTE )
      {
	$self->{$NOTE} = $args->{$key};

      }
     elsif ( $key eq $PARAMETER )
      {
	$self->{$PARAMETER} = $args->{$key};

      }
     elsif ( $key eq $VALUE )
      {
	$self->{$VALUE} = $args->{$key};

      }

   }

   if ( defined ( $self->{$MESSAGE} ))
    {
      $self->{$OUTPUT}->{$MESSAGE} = $self->{$OUTPUT}->{$MESSAGE} . " " .
			             $self->{$MESSAGE};
    }

   if ( defined ( $self->{$WARN_MSG} ))
    {
      $self->{$OUTPUT}->{$WARN_MSG} = $self->{$OUTPUT}->{$WARN_MSG} . " " .
				     $self->{$WARN_MSG};
    }

#   $self->{$OUTPUT}->{$ERROR}   = $self->{$ERROR};

   if ( ( defined $self->{$VALUE} ) && ( defined $self->{$PARAMETER} ) )
    {
      my $param = {
   		    $val	=>	$self->{$VALUE},
   		    $msg	=>	$self->{$NOTE},
                  };
   
      $self->{$OUTPUT}->{$self->{$PARAMETER}} = $param;
   
    }

}

sub _isPortFree()
{
  my $self      = $ARG[0];
  my $args      = $ARG[1];

  my $port_free = TXK::Util::TRUE;

  my $dbport	   = $self->txkGetContextVar({ oavar => 's_dbport' });
  my $repsport	   = $self->txkGetContextVar({ oavar => 's_repsport' });
  my $rpcport	   = $self->txkGetContextVar({ oavar => 's_rpcport' });
  my $formsport	   = $self->txkGetContextVar({ oavar => 's_formsport' });
  my $tcfport	   = $self->txkGetContextVar({ oavar => 's_tcfport' });
  my $metdataport  = $self->txkGetContextVar({ oavar => 's_metdataport' });
  my $metreqport   = $self->txkGetContextVar({ oavar => 's_metreqport' });
  my $mwaPortNo    = $self->txkGetContextVar({ oavar => 's_mwaPortNo' });
  my $mwaDispPort  = $self->txkGetContextVar({ oavar => 's_mwaDispatcherPort'});
  my $oemweb_port  = $self->txkGetContextVar({ oavar => 's_oemweb_port' });
  my $osagent_port = $self->txkGetContextVar({ oavar => 's_osagent_port' });
#  my $disco_port   = $self->txkGetContextVar({ oavar => 's_disco_port' });
  my $webport_pls  = $self->txkGetContextVar({ oavar => 's_webport_pls' });
  my $oprocmgrport = $self->txkGetContextVar({ oavar => 's_oprocmgr_port' });

  if ( ( $args eq $dbport       ) ||
       ( $args eq $repsport     ) ||
       ( $args eq $rpcport      ) ||
       ( $args eq $formsport    ) ||
       ( $args eq $tcfport      ) ||
       ( $args eq $metdataport  ) ||
       ( $args eq $metreqport   ) ||
       ( $args eq $mwaPortNo    ) ||
       ( $args eq $mwaDispPort  ) ||
       ( $args eq $oemweb_port  ) ||
       ( $args eq $osagent_port ) ||
#       ( $args eq $disco_port   ) ||
       ( $args eq $webport_pls  ) ||
       ( $args eq $oprocmgrport )
     )

   {
     return TXK::Util::FALSE;

   }
  else
   {
     return TXK::Util::TRUE;

   }

}

#-------------------------------------
# isFormsConfigWithSSL
#-------------------------------------

sub isFormsConfigWithSSL
{
  my $self      = $ARG[0];
  my $args      = $ARG[1];

  if ( !defined($self->{$FORMS_CONFIG_WITH_SSL}) )
   {
     $self->_isFormsConfigWithSSL($args);

   }

  return $self->{$FORMS_CONFIG_WITH_SSL};

}

#-------------------------------------
# _isFormsConfigWithSSL
#-------------------------------------

sub _isFormsConfigWithSSL
{
  my $self      = $ARG[0];
  my $args      = $ARG[1];

  my $SSL	   	= TXK::Techstack::TXK_SSL;
  my $cw	   	= $CWALLET;
  my $ew	   	= $EWALLET;
  my $cwallet	   	= TXK::Techstack::NULL;
  my $ewallet	   	= TXK::Techstack::NULL;
  my $frmWalletDir	= TXK::Techstack::NULL;
  my $cwallet_status	= TXK::Techstack::TXK_VLD_SUCCESS;
  my $ewallet_status	= TXK::Techstack::TXK_VLD_SUCCESS;

  if ( ! ( $self->txkIsAdConfigEnabled($args) ))
   {
     print "$AUTOCONFIG_MESSAGE\n";

     TXK::Error->stop("");

   }
  else
   {
     if ( ( $self->txkIsFormsNode() eq TXK::Util::TRUE ) && ( keys(%$args)) )
      {
	if ( ( exists  ( $args->{$FRM_WALLET_DIR} )) &&
	     ( defined ( $args->{$FRM_WALLET_DIR} )) &&
	     ( $args->{$FRM_WALLET_DIR} ne TXK::Techstack::NULL )
           )
	 {
	   $frmWalletDir = TXK::OSD->trDirPathToBase($args->{$FRM_WALLET_DIR});

	 }
	else
	 {
	   $frmWalletDir =
			$self->txkGetContextVar({ oavar => $FRM_WALLET_DIR });

	 }

	my $cw_tmp = $frmWalletDir . $cw;
	my $ew_tmp = $frmWalletDir . $ew;

	$cwallet   = TXK::OSD->trDirPathToBase($cw_tmp);
	$ewallet   = TXK::OSD->trDirPathToBase($ew_tmp);

	#
	# Check if the wallet files exist.
	#

	if ( $self->_validateFile($cwallet) eq TXK::Error::FAIL )
	 {
	   $cwallet_status = TXK::Techstack::TXK_VLD_MISSING_FILE_ERROR;

	 }

	if ( $self->_validateFile($ewallet) eq TXK::Error::FAIL )
	 {
	   $ewallet_status = TXK::Techstack::TXK_VLD_MISSING_FILE_ERROR;

	 }

	if ( ( $cwallet_status ne TXK::Techstack::TXK_VLD_SUCCESS ) ||
	     ( $ewallet_status ne TXK::Techstack::TXK_VLD_SUCCESS )
	   )
	 {
	   $self->{$FORMS_CONFIG_WITH_SSL} = TXK::Util::FALSE;

	 }
	else
	 {
	   $self->{$FORMS_CONFIG_WITH_SSL} = TXK::Util::TRUE;

	 }
      }
   }

}

#-------------------------------------
# _verifyDBWallet
#-------------------------------------

sub _verifyDBWallet
{
  my $self      = $ARG[0];
  my $args      = $ARG[1];

  my $SSL               = TXK::Techstack::TXK_SSL;
  my $cw                = $CWALLET;
  my $ew                = $EWALLET;
  my $cwallet           = TXK::Techstack::NULL;
  my $ewallet           = TXK::Techstack::NULL;
  my $frmWalletDir      = TXK::Techstack::NULL;
  my $cwallet_status    = TXK::Techstack::TXK_VLD_SUCCESS;
  my $ewallet_status    = TXK::Techstack::TXK_VLD_SUCCESS;

  my $tdb = TXK::TechstackDB->new($args);

  #
  #
  #
}
1;
