
# +===========================================================================+
# |   Copyright (c) 2003 Oracle Corporation, Redwood Shores, California, USA
# |                         All Rights Reserved
# |                        Applications Division
# +===========================================================================+
# |
# | FILENAME
# |   Script.pm
# |
# | DESCRIPTION
# |      TXK Script package
# |
# | USAGE
# |       See Script.html
# |
# | PLATFORM
# |
# | NOTES
# |
# +===========================================================================+

# $Header: Script.pm 03-aug-2005.08:54:54 ndutko Exp $

package TXK::Script;

@ISA = qw( TXK::Common );

######################################
# Standard Modules
######################################

use strict;
use English;
use Carp;

require 5.005;

######################################
# Package Specific Modules
######################################

use TXK::Common();
use TXK::RunScript();
use TXK::OSD();

######################################
# Public Constants
######################################

######################################
# Global Variables
######################################

my $CURRENT_INSTANCE = undef;

######################################
# Package Variables 
######################################

my $PACKAGE_ID = "TXK::Script";

######################################
# Object Keys
######################################

my $SCRIPT_COMMAND = "scriptCommand";
my $SCRIPT_NAME    = "scriptName";
my $TXK_TOP	   = "TXK_TOP";
my $RUNTIME_OBJ    = "runTimeObj";

######################################
# Package Methods
#
# Public
#
######################################

sub new;
sub DESTROY;
sub run;
sub getTXKTop;
sub getScriptName;
sub getScriptCommand;
sub getRuntime;
sub setScriptName;
sub setScriptCommand;
sub setTXKTop;
sub setRuntime;
sub setClassObj;
sub getClassObj;

######################################
# Package Methods
#
# Private
#       All private methods are marked with a leading underscore.
#
######################################

######################################
# Constructor
######################################

sub new {
  my $type = $ARG[0];

  my $self = TXK::Common->new();

  bless $self, $PACKAGE_ID ;

  my $key;

  my %INIT_OBJ =  (
			PACKAGE_IDENT     => $PACKAGE_ID,
                        $SCRIPT_COMMAND	  => "",
			$SCRIPT_NAME      => "",
			$TXK_TOP          => "",
			$RUNTIME_OBJ	  => undef,
                  ); 

  foreach $key (keys %INIT_OBJ)
   {    
     $self->{$key} = $INIT_OBJ{$key};
   }    

  return $self;
}

######################################
# Destructor
######################################

sub DESTROY
{
}

######################################
# run
######################################

sub run
{

        # This is the driver for all Perl scripts. It is mandatory because
        # we need to trap all errors using an eval due to STDERR redirection
        # errors on Linux.

  my $self  = $ARG[0];
  my $arg1  = $ARG[1];
  my $arg2  = $ARG[2];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  $self->setTXKTop($arg1);
  $self->setScriptCommand($arg2);

  my $execName = TXK::OSD->getBaseName($self->getScriptCommand());
  my $cmdName = $self->getScriptCommand();

  $execName =~ s/\.pl//g;
  $execName =~ s/\./_/g;

  $self->setScriptName($execName);

#       Store static version of $script object, so runtime can find
#       it.

  TXK::Script->setClassObj($self);

#
#       Require runs in the namespace of the current package. To avoid
#       conflict with this package, we do the actual call in a dummy
#       package.
#

  eval { TXK::RunScript->require($cmdName); };

# Get runtime log object.

  my $log = $self->getRuntime()->getLog()
               if ( ref($self->getRuntime()) eq "TXK::Runtime" );

  my $scriptError = $EVAL_ERROR;

  my $idx = index($scriptError,"Compilation failed in require");

  $scriptError = substr($scriptError,0,$idx) if ( $idx > -1 );

  print STDERR "\nErrors encountered running $cmdName \n\n"
               unless $EVAL_ERROR eq "";

  print STDERR $scriptError unless $EVAL_ERROR eq "";

  print "End of $cmdName : No Errors encountered\n"
                if ( $EVAL_ERROR eq "" );

  if ( ref($log) eq "TXK::Log" && $log->isOpen() )
   {
     my $log_fh = $log->getFileHandle();

     print $log_fh "\nErrors encountered running $cmdName \n\n"
                  unless $EVAL_ERROR eq "";

     print $log_fh $scriptError unless $EVAL_ERROR eq "";

     print $log_fh "End of $cmdName : No Errors encountered\n"
                       if ( $EVAL_ERROR eq "" );
   }

  return ( $EVAL_ERROR eq "" ? "0" : "1" );
}

######################################
# getTXKTop
######################################

sub getTXKTop
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return $self->{$TXK_TOP};
}

######################################
# getScriptName
######################################

sub getScriptName
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];
                     
  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return $self->{$SCRIPT_NAME};
}

######################################
# getScriptCommand
######################################

sub getScriptCommand
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];
                     
  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return $self->{$SCRIPT_COMMAND};
}

######################################
# getRuntime
######################################

sub getRuntime
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return $self->{$RUNTIME_OBJ};
}

######################################
# setScriptName
######################################

sub setScriptName
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];
                     
  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  $self->{$SCRIPT_NAME} = $args;
}

######################################
# setScriptCommand
######################################

sub setScriptCommand
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];
                     
  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  $self->{$SCRIPT_COMMAND} = $args;
}

######################################
# setTXKTop
######################################

sub setTXKTop
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  $self->{$TXK_TOP} = $args;
}

######################################
# setRuntime
######################################

sub setRuntime
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  $self->{$RUNTIME_OBJ} = ( ref($args) eq "TXK::Runtime" ? $args : undef );
}

######################################
# setClassObj
######################################

sub setClassObj
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  $CURRENT_INSTANCE = ( ref($args) eq $PACKAGE_ID ? $args : undef );
} 

######################################
# getClassObj
######################################

sub getClassObj
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  return $CURRENT_INSTANCE;
} 

1;

