# +===========================================================================+
# |   Copyright (c) 2003 Oracle Corporation, Redwood Shores, California, USA
# |                         All Rights Reserved
# |                        Applications Division
# +===========================================================================+
# |
# | FILENAME
# |   Techstack.pm
# |
# | DESCRIPTION
# |      TXK Techstack package
# |
# | USAGE
# |       See Techstack.html
# |
# | PLATFORM
# |
# | NOTES
# |
# +===========================================================================+

# $Header: emll/sysman/admin/scripts/TXK/Techstack.pm /main/5 2008/11/16 06:44:07 ndutko Exp $

package TXK::Techstack;

@ISA = qw( TXK::Common);

#-------------------------------------
# Standard Modules
#-------------------------------------

use strict;
use English;
use Carp;

require 5.005;

#-------------------------------------
# Public Constants
#-------------------------------------

	#
	# Applications User & Sys names
	#

use constant APPS_USER_KEY	=> "appsuser"		;
use constant APPS_DEFAULT_USER	=> "apps"               ;
use constant APPS_PASSWORD_KEY  => "appspass"           ;
use constant APPS_DEFAULT_PWD   => "UNKNOWN_PWD"        ;
use constant SYS_PASSWORD_KEY   => "syspass"            ;
use constant SYS_DEFAULT_PWD    => "UNKNOWN_PWD"        ;
use constant APPS_CTX_FILE_KEY  => "contextfile"        ;
use constant CONTEXT_TYPE_KEY   => "contexttype"        ;
use constant JDKVERSION_KEY     => "jdkversion"         ;
use constant DEFAULT_JDKVERSION => "1.3.1"              ;
use constant ADIDENT_EXE        => "adident"            ;
use constant ADXMLCTX_FILE      => "adxmlctx.tmp"       ;
use constant ADXMLCTX_FILE_DB   => "adxdbctx.tmp"       ;
use constant WEB_PATCH_LIST_13  => "webPatchListJDK13"  ;
use constant WEB_PATCH_LIST_14  => "webPatchListJDK14"  ;
use constant CONC_PATCH_LIST_13 => "concPatchListJDK13" ;
use constant CONC_PATCH_LIST_14 => "concPatchListJDK14" ;
	#
	# SSL
	#

use constant TXK_SSL			=> "SSL"			;
use constant TXK_WEBSSL_PORT            => "s_webssl_port"		;
use constant TXK_WEB_SSL_KEYFILE        => "s_web_ssl_keyfile"		;
use constant TXK_WEB_SSL_CERTFILE       => "s_web_ssl_certfile"		;
use constant TXK_WEB_SSL_CERTCHAINFILE  => "s_web_ssl_certchainfile"	;

use constant TXK_URL_PROTOCOL           => "s_url_protocol"		;
use constant TXK_LOCAL_URL_PROTOCOL     => "s_local_url_protocol"	;
use constant TXK_SSL_DEFAULT_PROTOCOL   => "https"			;
use constant TXK_URL_DEFAULT_PROTOCOL   => "http"			;
use constant TXK_DEFAULT_WEBSSL_PORT    => "443"			;

	#
	# Common between SSL and HTTP Hardware load balancing
	#

use constant TXK_WEBPORT		=> "s_webport"			;
use constant TXK_LOGIN_PAGE             => "s_login_page"		;
use constant TXK_ACTIVE_WEBPORT         => "s_active_webport"		;
use constant TXK_WEB_ENTRY_URL_PROTOCOL	=> "s_webentryurlprotocol"	;

	#
	# HTTP Hardware load balancing
	#

use constant TXK_HTTP_LB		=> "HttpLoadBal"		;
use constant TXK_WEB_HOST		=> "s_webhost"			;
use constant TXK_WEB_ENTRY_HOST		=> "s_webentryhost"		;
use constant TXK_WEB_ENTRY_DOMAIN	=> "s_webentrydomain"		;
use constant TXK_SESSION_PERSISTENT	=> "t_session_persistent"	;
use constant TXK_DNS_LAYER		=> "t_dns"			;
use constant TXK_DOMAIN_NAME		=> "s_domainname"		;


	#
	# Jserv Load balancing & Forms Listener Servlet
	#

use constant TXK_OPROCMGR_NLIST		=> "s_oacore_trusted_oproc_nodes";
use constant TXK_FORMS_FNDTOP		=> "s_formsfndtop"		 ;
use constant TXK_JSERV_LB		=> "JLoadBal"			 ;
use constant TXK_FORMS_LSNR_SERVLET	=> "FormsLsnrServlet"		 ;
use constant TXK_FORMS_METRIC_LB	=> "FormsMetLoadBal"		 ;
use constant TXK_FORMS_METRIC		=> "FormsMetric"		 ;
use constant TXK_WALLET_MGR		=> "WalletManager"		 ;
use constant TXK_FORMS_CONFIGURATION	=> "FormsConfig"		 ;

	#
	# Multi-purpose constants
	#

use constant NULL			=> ""				;
use constant TXK_ORACLE_TERM		=> "vt220"			;
use constant TXK_CURRENT_ERROR		=> "currentError"		;
use constant TXK_CURRENT_ERROR_MSG	=> "currentErrorMsg"		;
use constant TXK_OUTPUT_MESSAGE		=> "outputMsg"			;
use constant TXK_CONFIG_MESSAGE		=> "configMsg"			;
use constant TXK_VALIDATE_MESSAGE	=> "validateMsg"		;
use constant TXK_DISPLAY		=> "display"			;

use constant AUTOCONFIG_MESSAGE		=> "\n[FAIL]\t AutoConfig must be enabled on this system before proceeding with\n\t the specified operation. For more information, please see\n\t AutoConfig My Oracle Support Note 165195.1.\n\n";

	#
	# Validation & Configuration related constants
	#

use constant TXK_CFG_ENABLE			=> "enable"		;
use constant TXK_CFG_DISABLE			=> "disable"		;
use constant TXK_CFG_STATUS			=> "configStatus"	;
use constant TXK_VAL_STATUS			=> "validateStatus"	;

use constant TXK_WR_SUCCESS			=> "success"		;
use constant TXK_WR_FAILURE			=> "failure"		;
use constant TXK_WR_WARNING			=> "warning"		;
use constant TXK_WR_MESSAGE			=> "message"		;
use constant TXK_WR_PARAMETER			=> "parameter"		;
use constant TXK_WR_VALUE			=> "value"		;
use constant TXK_WR_ERROR			=> "error"		;
use constant TXK_WR_WARN_MSG			=> "warn_msg"		;
use constant TXK_WR_OUTPUT			=> "output"		;
use constant TXK_WR_OA_VAR			=> "oa_var"		;
use constant TXK_WR_NOTE			=> "note"		;
use constant TXK_WR_VLD_TAG			=> "validation"		;
use constant TXK_WR_CFG_TAG			=> "configuration"	;

use constant TXK_CFG_FAIL			=> "0"			;
use constant TXK_CFG_WARNING			=> "-1"			;
use constant TXK_CFG_SUCCESS			=> "1"			;

use constant TXK_VLD_FAIL			=> "00"			;
use constant TXK_VLD_WARNING			=> "-10"		;
use constant TXK_VLD_MISSING_FILE_ERROR		=> "-11"		;
use constant TXK_VLD_FRMSLSNR_SERVL_WARNING	=> "-12"		;
use constant TXK_VLD_MISSING_PARAMETER		=> "-13"		;
use constant TXK_VLD_SUCCESS			=> "11"			;

#-------------------------------------
# Private Constants
#-------------------------------------

use constant JAVACMD_NOT_FOUND  		=> "-1"                 ;
use constant JDK_NOT_REQUIRED   		=> "-2"			;
use constant JDK_NOT_SPECIFIED_NODE_TYPE	=> "-3"			;
use constant DISPLAY_NOT_SET			=> "-1"			;
use constant XDPYINFO_NOT_FOUND			=> "-2"			;
use constant NO_ACTIVE_DISPLAY			=> "-3"			;

#-------------------------------------
# Package Specific Modules
#-------------------------------------

use TXK::IO();
use TXK::Log();
use TXK::OSD();
use TXK::XML();
use TXK::Util();
use TXK::Error();
use TXK::Runtime();
use TXK::Process();
use TXK::SQLPLUS();
use TXK::RunScript();

#-------------------------------------
# Package Variables
#-------------------------------------

my $PACKAGE_ID = "TXK::Techstack";

#-------------------------------------
# Object Keys
#-------------------------------------

my $PRINTMSG		= "printMsg"		;
my $APPS_VERSION	= "AppsVersion"		;
my $JINIT_VERSION	= "JinitVersion"	;
my $DISCO_VERSION	= "DiscoVersion"	;
my $OA_VERSION		= "OAVersion"		;
my $OA_VERSION_IDENT	= "OAVersionIdent"	;
my $DB_VERSION     	= "DBVersion"		;
my $JDK_VERSION_WEB    	= "JDKVersionWeb"	;
my $JDK_VERSION_CP    	= "JDKVersionCP"	;
my $JDK_VERSION_ADMIN   = "JDKVersionAdmin"	;
my $OHS_VERSION    	= "OHSVersion"		;
my $OJSP_VERSION	= "OjspVersion"		;
my $FORMS_VERSION	= "FormsVersion"	;
my $DEV6I_PS_VERSION	= "Dev6iPsVersion"	;
my $ORACLE_EWT_VERSION	= "OracleEWTVersion"	;
my $IAS_PATCHSET	= "iASpatchset"		;
my $TOOLS_VERSION  	= "toolsVersion"	;
my $HAS_8063_BACKPORT  	= "has8063Backport"	;
my $WINDOWS_8063_VERSION= "windows8063version"  ;
my $TECHSTACK_VERSION	= "TechstackVersion"	;
my $LOG_FILE       	= "logFile"		;
my $REP_FILE       	= "reportFile"		;
my $JDK_TOP_WEB		= "JDK_TOP_WEB"		;
my $JDK_TOP_CP		= "JDK_TOP_CP"		;
my $OHS_TOP		= "APACHE_TOP"		;
my $JAVA_CMD_WEB        = "JavaExecutableWeb"   ;
my $JAVA_CMD_CP         = "JavaExecutableCP"    ;
my $JAVA_CMD_ADMIN      = "JavaExecutableAdmin" ;
my $JAVA_TOP		= "JAVA_TOP"		;
my $CONF_WITH_JDK_WEB  	= "configuredWithJDKWeb";
my $CONF_WITH_JDK_CP  	= "configuredWithJDKCP"	;
my $HOST_NAME      	= "hostName"		;
my $APPL_TOP	   	= "APPL_TOP"		;
my $APPLTMP	   	= "APPLTMP"		;
my $TWO_TASK            = "TWO_TASK"		;
my $TOOLS_ORACLE_HOME	= "ORACLE_HOME"		;
my $IAS_ORACLE_HOME	= "IAS_ORACLE_HOME"	;
my $DB_HOME		=  "DBHome"		;
my $APPS_CTX_FILE  	= "appsContextFile"  	;
my $APPS_CTX_VERSION  	= "appsCtxVersion"  	;
my $CONTEXT_XMLOBJ      = "contextXMLObject"	;
my $CTXTMPL_VERSION  	= "ctxTmplVersion"  	;
my $ADCONFIG_ENABLED	= "adconfigEnabled"	;
my $ADCONTEXT_UPTODATE	= "adcontextUptoDate"	;
my $IS_WEB_NODE		= "isWebNode"		;
my $IS_FORMS_NODE	= "isFormsNode"		;
my $IS_CONC_NODE	= "isConcNode"		;
my $IS_DB_NODE		= "isDBNode"		;
my $IS_ADMIN_NODE	= "isAdminNode"		;
my $ERROR_MESSAGE       = "errorMessage"        ;
my $ERROR_STATUS        = "errorStatus"         ;
my $FND_TOP             = "FND_TOP"             ;
my $FND_TMPL_DRV        = "FND_TMPL_DRV"        ;
my $AC_TEMP_ROLLUP_VER  = "AC_TEMP_ROLLUP_VER"  ;
my $AC_TEMP_ROLLUP_BUG  = "AC_TEMP_ROLLUP_BUG"  ;
my $AC_TEMP_ROLLUP_TXT  = "AC_TEMP_ROLLUP_TXT"  ;
my $AC_TEMP_RUP_ALL_TXT = "AC_TEMP_RUP_ALL_TXT" ;
my $APPS_USER	  	= APPS_USER_KEY		;
my $APPS_PASSWORD       = APPS_PASSWORD_KEY	;
my $SYS_PASSWORD        = SYS_PASSWORD_KEY	;
my $APPS_CTX_FILE_KEY	= APPS_CTX_FILE_KEY	;
my $OUTPUT_MESSAGE	= TXK_OUTPUT_MESSAGE	;
my $CONFIG_MESSAGE	= TXK_CONFIG_MESSAGE	;
my $VALIDATE_MESSAGE	= TXK_VALIDATE_MESSAGE	;
my $MODPLSQL_VERSION    = "modplsql_version"    ;
my $JDBC_DRIVER_VERSION = "jdbc_driver_version" ;
my $SYM_LINK_REPOS      = "symbolic_link_repository";

my $IS_SSL			= TXK_SSL			;
my $IS_HTTP_LB			= TXK_HTTP_LB			;
my $IS_JSERV_LB			= TXK_JSERV_LB			;
my $IS_FORMS_LSNR_SERVLET	= TXK_FORMS_LSNR_SERVLET	;
my $IS_FORMS_METRIC_LB		= TXK_FORMS_METRIC_LB		;
my $IS_FORMS_METRIC		= TXK_FORMS_METRIC		;
my $IS_WALLET_MGR		= TXK_WALLET_MGR		;
my $IS_DISPLAY			= TXK_DISPLAY			;
my $FORMS_CONFIGURATION		= TXK_FORMS_CONFIGURATION	;

my $WEBSSL_PORT			= TXK_WEBSSL_PORT		;
my $WEB_SSL_KEYFILE		= TXK_WEB_SSL_KEYFILE		;
my $WEB_SSL_CERTFILE		= TXK_WEB_SSL_CERTFILE		;
my $WEB_SSL_CERTCHAINFILE	= TXK_WEB_SSL_CERTCHAINFILE	;
my $URL_PROTOCOL		= TXK_URL_PROTOCOL		;
my $LOCAL_URL_PROTOCOL		= TXK_LOCAL_URL_PROTOCOL	;
my $LOGIN_PAGE			= TXK_LOGIN_PAGE		;
my $ACTIVE_WEBPORT		= TXK_ACTIVE_WEBPORT		;
my $WEB_ENTRY_URL_PROTOCOL	= TXK_WEB_ENTRY_URL_PROTOCOL	;
my $CURRENT_ERROR		= TXK_CURRENT_ERROR		;
my $CURRENT_ERROR_MSG		= TXK_CURRENT_ERROR_MSG		;
my $AUTOCONFIG_MESSAGE		= AUTOCONFIG_MESSAGE            ;
my $CONTEXT_TYPE                = CONTEXT_TYPE_KEY              ;

my $CONTEXT_VARS_TO_CHECK_FOR_SYM_LINK   = [ "s_html"    , 
					     "s_javatop" , 
					     "s_pt"      ]        ;

#
# The following patterns used for pattern matching can change
#

my $JSERV_PATTERN_1       = qr/\s*wrapper\.classpath=/			;
my $JSERV_DT_JAR          = qr/lib(\/|\\)dt\.jar/                       ;
my $JSERV_TOOLS_JAR       = qr/lib(\/|\\)tools\.jar/                    ;
my $JSERV_RT_JAR          = qr/jre(\/|\\)lib(\/|\\)rt\.jar/             ;
my $JSERV_CHARSETS_JAR    = qr/jre(\/|\\)lib(\/|\\)charsets\.jar/       ;
my $JSERV_I18N_JAR        = qr/jre(\/|\\)lib(\/|\\)i18n\.jar/           ;

my $JDBC_FILE_12          = "jdbc12\.zip"                               ;
my $JDBC_FILE_14          = "jdbc14\.zip"                               ;
my $APPSBORG_FILE         = "appsborg2\.zip"				;

my $JDK_13_PATTERN        = qr/(1\.3.1)/                                ;
my $JDK_14_PATTERN        = qr/(1\.4)/                                  ;
my $JSERVJAVA_PATTERN     = qr/\s*JSERVJAVA\s*=\s*/			;

my $SERVLET_SERVERURL     = qr/^(\/forms\/formservlet)\b/		;
my $SERVLET_COMMENT       = qr/([^\\#])+/				;

my $TECHSTACK_1000	  = qr/(ias10)/					;
my $TECHSTACK_1022	  = qr/(ias1022)/				;
my $TECHSTACK_1021	  = qr/(http1312)/				;
my $TECHSTACK_1000_VER	  = "ias10"					;
my $TECHSTACK_1022_VER	  = "ias1022"					;
my $TECHSTACK_1021_VER	  = "http1312"					;

my $OHS_PATTERN_1  = qr/^(Server\s*version.*Oracle.*HTTP.*\/)/		      ;
my $OHS_PATTERN_2  = qr/\s*\(.*\).*$/					      ;
my $OHS_PATTERN_3  = qr/^(Server\s*built.*\(iAS\s*1\.0\.2\.2.*rollup\s*)/     ;
my $OHS_PATTERN_4  = qr/(\s*|\))\s*$/					      ;
my $OHS_PATTERN_5  = qr/oracle.apache.apache/				      ;
my $OHS_PATTERN_6  = qr/(^\s+\<PS_PATCH NAME=\"oracle.apache.apache\" VER=\"|\" BASE_VER=\"1.3.19.0.0a\"\/>$)/ ;


my $OAJSP_26	= qr/(115\.26)/	;
my $OAJSP_27	= qr/(115\.27)/	;
my $OAJSP_36	= qr/(115\.36)/	;

my $OAFWK_552	= "5.5.2E";
my $OAFWK_560	= "5.6E";
my $OAFWK_570	= "5.7H";

my $OJSP_PATTERN_1  = qr/^(Oracle\s*JSP\s*Translator\s*Release\s*)/;
my $OJSP_PATTERN_2  = qr/1\.1\.(2|3).*/;

my $FORMS_PATTERN_1  = qr/^(Forms.*)(Form\s*Compiler\)\s*)(Version\s*)/;
my $FORMS_PATTERN_2  = qr/\(Production\)$/;

my $FORMS_NT_PATTERN_1 = qr/(854\s*)(w32forms60\s*)(tools\s*)(\"forms60\"\s*)/;
my $FORMS_NT_PATTERN_2 = qr/(\"Oracle\s*Forms\")/;

my $FORMS_DEV6I_PS9  = qr/(6\.0\.8\.18\.*)/;
my $FORMS_DEV6I_PS10 = qr/(6\.0\.8\.19\.*)/;
my $FORMS_DEV6I_PS11 = qr/(6\.0\.8\.20\.*)/;
my $FORMS_DEV6I_PS12 = qr/(6\.0\.8\.21\.*)/;

my $PLSQL_PATTERN    = qr/^PL\/SQL\s+Version\s+([\.\d]+)\s*/;
my $DBMS_PATTERN_1   = qr/ntrdbms80\s+root\s+"rdbms80"\s+"([\d\.]+)"/;
my $XDPY_PATTERN_1   = qr/^default\s+screen\s+number:\s+(\d+)/;
my $XDPY_PATTERN_2   = qr/^screen #/;
my $XDPY_PATTERN_3   = qr/^\s*default visual id:\s*(\w+)/;
my $XDPY_PATTERN_4   = qr/^\s*visual id:\s*(\w+)/;
my $XDPY_PATTERN_5   = qr/^\s*class:\s*(\w+)/;

my $XMLPARSER904_PATTERN = qr/xdkjava_version_9.0.4.0.0_production/;

my $JDBC_DRIVER_MAPPING_12 = 
                 { '8.1.6.2'     => [ 115.0 ],
		   '8.1.7'       => [ 115.1 ],
		   '8.1.7.1B'    => [ 115.2 , 115.3 , 115.4 ],
		   '8.1.7.2'     => [ 115.5 ],
		   '8.1.7.3'     => [ 115.6 ],
		   '9.2.0'       => [ 115.7 ,  115.8 , 115.9 ],
		   '9.2.0.2'     => [ 115.10 , 115.11 , 115.12 , 115.13 ,115.16 ],
		   '9.2.0.4'     => [ 115.14 , 115.15 , 115.17 , 115.18 , 115.19 , 115.20 , 115.21 ,
				    115.22 , 115.23 , 115.24 , 115.25 , 115.26 , 115.27 ]
		   };

my $JDBC_DRIVER_MAPPING_14 = 
                 { '9.2.0.4'     => [ 115.1 , 115.2 , 115.3 , 115.4 , 115.5 ,115.6 , 115.7 ,
				    115.8 , 115.9 , 115.10, 115.11 ],
		   '9.2.0.5'     => [ 115.12 , 115.13 ]
		 };

my $AIX_MODPLSQL_CHECKSUM_30985D = [ 4054870827 ] ;

# this contains the bug nos for each platform which contain
# patch for bug 3966092. the keys are the same as AUTOCONFIG_PLATFORM in OSD.pm

my $PLATFORM_8174_BUGNO_3966092 = { Solaris    => { bugno => [ 3973551 , 3973565 ],
                                                    objects => 29 },
				    IBM_AIX    => { bugno => [ 3975426 , 3973563 , 3973554 ],
						    objects => 1 },
				    HP_UX      => { bugno => [ 3973559 , 3975424 ],
						    objects => 1 },
				    UNIX_Alpha => { bugno => [ 3973560 ],
						    objects => 29 },
				    Linux      => { bugno => [ 3967806 ] ,
						    objects => 29 }
				  };

#------------------------------------------------------------------------------
# Object Structure
# -----------------------------------------------------------------------------

#------------------------------------------------------------------------------
#
#  Hash Array
#
#------------------------------------------------------------------------------

#-------------------------------------
# Package Methods
#
# Public
#
#       new     - build empty object
#
#------------------------------------------------------------------------------

sub new;

sub txkGetContextType;
sub txkIsDBContext;
sub txkGetContextVar;
sub txkGetOAVersionIdent;
sub txkGetOAVersion;
sub txkGetOHSVersion;
sub txkGetJDKVersion;
sub txkGetAppsVersion;
sub txkGetOjspVersion;
sub txkGetFormsVersion;
sub txkGetJinitVersion;
sub txkGetDiscoVersion;
sub txkGetDev6iPsVersion;
sub txkGetEWTVersion;
sub txkCheckiASExpiration;
sub txkGetIASpatchset;
sub txkGetRSFVersionForIASOracleHome;

sub txkGetOHSTop;
sub txkGetJDKTop;
sub txkGetJavaTop;
sub txkGetIASOracleHome;
sub txkGetToolsOracleHome;
sub txkGetToolsOracleHomeVersion;
sub txkGetAppsCtxLocation;
sub txkGetAppsCtxFile;
sub txkGetApplTmp;
sub txkGetApplTop;
sub txkGetACTemplateRollupVer;
sub txkGetACTemplateRollupBugNo;
sub txkGetACTemplateRollupBugtext;
sub txkGetFndTemplateDriverVersion;
sub txkGetTechstackVersion;
sub txkGetDBOracleHome;		   # get Oracle Home from db context file

sub txkIsConfiguredWithJDK;
sub txkIsConfiguredWithJDKWeb;
sub txkIsConfiguredWithJDKCP;
sub txkIsAdConfigEnabled;
sub txkIsContextUpToDate;
sub txkIsAdminNode;
sub txkIsDBNode;
sub txkIsWebNode;
sub txkIsFormsNode;
sub txkIsConcNode;
sub txkIsFormsLsnrServlet;
sub txkIsSSLConfigured;
sub txkIsCPJDKVersion131;
sub txkIsCPJDKVersion14;

sub txkCheckDisplay;
sub txkCheckOProcAndWebPort;
sub txkCheck8063Backport; # For bug 1227566
sub txkGet8063VersionOnWindows;
sub txkCheckXMLParser904;

sub txkSetError;
sub txkGetError;
sub txkGetErrorMsg;
sub txkPrintErrorMsg;

sub txkPrintMsg;
sub txkPrintStatus;
sub txkIsPrintMsgOn;

sub txkCheckForSymLinks;
sub txkCheckOptionsSymLink;
sub txkGetModPLSQLVersion;
sub txkGetJDBCDriverVersion;

#------------------------------------------------------------------------------
# Package Methods
#
# Private
#       All private methods are marked with a leading underscore.
#
#-------------------------------------

sub _txkIsDBContext;
sub _isFileFound;
sub _isPatchApplied;
sub _isAdConfigEnabled;

sub _init;
sub _validateChecksum;
sub _getRDBMSValueFromRGS;
sub _verifyConfigJDKCP;
sub _verifyConfigJDKWeb;

sub _trimDomain;
sub _setPrintMsg;
sub _verifyPatch;

sub _getOHSTop;
sub _getJDKTopWeb;
sub _getJDKTopCP;
sub _getJavaTop;
sub _getNodeInfo;
sub _getSSLConfig;
sub _getOAVersionIdent;
sub _getOAVersion;
sub _getOHSVersion;
sub _getCtxVersion;
sub _getAppsVersion;
sub _getOjspVersion;
sub _getIASpatchset;
sub _getFormsVersion;
sub _getJinitVersion;
sub _getDiscoVersion;
sub _getDev6iPsVersion;
sub _getIASOracleHome;
sub _getFormsVersionNT;
sub _getFormsVersionUnix;
sub _getToolsOracleHome;
sub _getAppsCtxLocation;
sub _getFormsLsnrServlet;
sub _getApplTmp;
sub _getApplTop;
sub _getJDKVersionCP;
sub _getJDKVersionWeb;
sub _getJDKVersionAdmin;
sub _getNodeInfoIfAdConfigEnabled;
sub _getTechstackVersion;
sub _getToolsOracleHomeVersionUnix;
sub _getToolsOracleHomeVersionNT;
sub _check8063Backport;
sub _checkDisplay;
sub _txkGetOAVersionFromJava;
sub _countSymLinksInPath;
sub _countSymLinks;
sub _getModPLSQLVersion;
sub _getModPLSQLVersionNT;
sub _getJDBCDriverVersion;
sub _doGetJDKVersion;
sub _getACTemplateRollupFullText;

#-------------------------------------
# Constructor
#-------------------------------------

sub new
{

  my $type = $ARG[0];
  my $args = $ARG[1];

  my $self = TXK::Common->new();

  bless $self, $PACKAGE_ID ;

  my $key;

  my %INIT_OBJ = (
		   PACKAGE_IDENT   		=> $PACKAGE_ID		,
		   $PRINTMSG			=> TXK::Util::FALSE	,
		   $APPS_VERSION		=> undef		,
		   $JINIT_VERSION		=> undef		,

		   $DISCO_VERSION		=> undef		,
		   $OA_VERSION     		=> undef		,
		   $OA_VERSION_IDENT   		=> undef		,
		   $DB_VERSION     		=> undef		,
		   $JDK_VERSION_WEB    		=> undef		,
		   $JDK_VERSION_CP    		=> undef		,
		   $JDK_VERSION_ADMIN    	=> undef		,
		   $OHS_VERSION    		=> undef		,
		   $OJSP_VERSION    		=> undef		,
		   $FORMS_VERSION    		=> undef		,
		   $DEV6I_PS_VERSION   		=> undef		,
		   $ORACLE_EWT_VERSION   	=> undef		,
		   $IAS_PATCHSET		=> undef		,
		   $TOOLS_VERSION  		=> undef		,
		   $HAS_8063_BACKPORT  		=> undef		,
		   $WINDOWS_8063_VERSION        => undef                ,
		   $TECHSTACK_VERSION		=> undef		,
		   $APPS_USER			=> undef		,
		   $APPS_PASSWORD		=> undef		,
		   $CONTEXT_TYPE		=> undef		,
		   $SYS_PASSWORD		=> undef		,
		   $ADCONFIG_ENABLED		=> undef		,
		   $ADCONTEXT_UPTODATE 		=> undef		,
		   $JAVA_CMD_WEB		=> undef		,
		   $JAVA_CMD_CP			=> undef		,
		   $JAVA_CMD_ADMIN		=> undef		,
                   $JDK_TOP_WEB			=> undef		,
                   $JDK_TOP_CP			=> undef		,
                   $OHS_TOP			=> undef		,
                   $JAVA_TOP			=> undef		,
		   $LOG_FILE       		=> NULL			,
		   $REP_FILE       		=> NULL			,
		   $HOST_NAME      		=> NULL			,
		   $APPL_TOP       		=> undef		,
		   $APPLTMP       		=> undef		,
                   $FND_TOP                     => undef                ,
                   $FND_TMPL_DRV                => undef                ,
                   $AC_TEMP_ROLLUP_VER          => undef                ,
                   $AC_TEMP_ROLLUP_BUG          => undef                ,
                   $AC_TEMP_ROLLUP_TXT          => undef                , 
                   $AC_TEMP_RUP_ALL_TXT         => undef                ,
		   $TWO_TASK       		=> NULL			,
		   $TOOLS_ORACLE_HOME       	=> NULL			,
		   $IAS_ORACLE_HOME		=> undef		,
		   $DB_HOME			=> undef		,
		   $APPS_CTX_FILE  		=> NULL			,
		   $APPS_CTX_VERSION  		=> undef		,
		   $CONTEXT_XMLOBJ		=> undef		,
		   $CTXTMPL_VERSION  		=> undef		,
                   $VALIDATE_MESSAGE		=> NULL			,
                   $ERROR_MESSAGE		=> NULL			,
                   $ERROR_STATUS		=> TXK::Util::FALSE	,
		   $CURRENT_ERROR		=> NULL			,
		   $CURRENT_ERROR_MSG		=> NULL			,
		   $CONF_WITH_JDK_WEB  		=> TXK::Util::FALSE	,
		   $CONF_WITH_JDK_CP  		=> TXK::Util::FALSE	,
		   $IS_WEB_NODE			=> undef		,
		   $IS_FORMS_NODE		=> undef		,
		   $IS_CONC_NODE		=> undef		,
		   $IS_DB_NODE			=> undef		,
		   $IS_ADMIN_NODE		=> undef		,
		   $IS_SSL			=> undef		,
		   $IS_HTTP_LB			=> undef		,
		   $IS_JSERV_LB			=> undef		,
		   $IS_FORMS_LSNR_SERVLET	=> undef		,
		   $IS_FORMS_METRIC_LB		=> undef		,
		   $IS_FORMS_METRIC		=> undef		,
		   $IS_WALLET_MGR		=> undef		,
		   $IS_DISPLAY                  => undef                ,
		   $JDBC_DRIVER_VERSION         => undef                ,
		   $MODPLSQL_VERSION            => undef                ,
		   $FORMS_CONFIGURATION		=> NULL			,
		   $WEBSSL_PORT			=> NULL			,
		   $WEB_SSL_KEYFILE		=> NULL			,
		   $WEB_SSL_CERTFILE		=> NULL			,
		   $WEB_SSL_CERTCHAINFILE	=> NULL			,
		   $URL_PROTOCOL		=> NULL			,
		   $LOCAL_URL_PROTOCOL		=> NULL			,
		   $LOGIN_PAGE			=> NULL			,
		   $ACTIVE_WEBPORT		=> NULL			,
		   $SYM_LINK_REPOS              => undef                ,
		   $WEB_ENTRY_URL_PROTOCOL	=> NULL
                 );

  foreach $key (keys %INIT_OBJ)
  {
     $self->{$key} = $INIT_OBJ{$key};
  }

  $self->_init($args);

  return $self;
}

#------------------------------------------------------------------------------
# Internal Initialization
#------------------------------------------------------------------------------

sub _init
{
  my $self	= $ARG[0];
  my $args      = $ARG[1];


  $self->{$HOST_NAME}  	      = TXK::OSD->getHostName();

  if ( TXK::OSD->isNT() )
   {
     $self->{$TWO_TASK} = TXK::OSD->getEnvVar({ name => "LOCAL"});
   }
  else
   {
     $self->{$TWO_TASK} = TXK::OSD->getEnvVar({ name => "TWO_TASK"});
   }

  $self->_trimDomain();

  if ( ( ref( $args ) eq "HASH" ) && ( keys %{$args} ) )
   {
     if ( ( exists  ( $args->{$APPS_CTX_FILE_KEY} )) &&
          ( defined ( $args->{$APPS_CTX_FILE_KEY} )) &&
          ( $args->{$APPS_CTX_FILE_KEY} ne NULL    )
        )
      {
	  $self->{$APPS_CTX_FILE} = $args->{$APPS_CTX_FILE_KEY};
      }

     $self->{$CONTEXT_TYPE} = "APPL_TOP Context";

    if (  $self->_isAdConfigEnabled($args) )
     {
	 $self->{$CONTEXT_TYPE} = $self->txkGetContextType ( { $APPS_CTX_FILE_KEY => $self->{$APPS_CTX_FILE} } );
     }

     if ( ( exists  ( $args->{$APPS_USER} )) &&
          ( defined ( $args->{$APPS_USER} )) &&
          ( $args->{$APPS_USER} ne NULL    )
        )
      {
        $self->{$APPS_USER} = $args->{$APPS_USER};
      }
     else
      {
	  $self->{$APPS_USER} = $self->txkGetContextVar({ oavar => 's_apps_user' });
      }

     if ( ( exists  ( $args->{$APPS_PASSWORD} )) &&
          ( defined ( $args->{$APPS_PASSWORD} )) &&
          ( $args->{$APPS_PASSWORD} ne NULL    )
        )
      {
        $self->{$APPS_PASSWORD} = $args->{$APPS_PASSWORD};

      }

     if ( ( exists  ( $args->{$SYS_PASSWORD} )) &&
          ( defined ( $args->{$SYS_PASSWORD} )) &&
          ( $args->{$SYS_PASSWORD} ne NULL    )
        )
      {
        $self->{$SYS_PASSWORD} = $args->{$SYS_PASSWORD};

      }

   }

  if ( !defined($self->{$APPS_USER} ) or
       $self-{$APPS_USER} eq ""        )
   {
     $self->{$APPS_USER} = APPS_DEFAULT_USER;
   }

  if ( !defined($self->{$APPS_PASSWORD} ) )
   {
     $self->{$APPS_PASSWORD} = APPS_DEFAULT_PWD;
   }

  if ( !defined($self->{$SYS_PASSWORD} ) )
   {
     $self->{$SYS_PASSWORD} = SYS_DEFAULT_PWD;
   }


  $self->_getApplTop($args);

  $self->_getApplTmp($args);

  $self->_getNodeInfo($args);

  $self->_getToolsOracleHome($args);

  $self->_getIASOracleHome($args);

#  $self->_getJavaTop($args);

#  $self->_getJDKTop($args);

##  $self->_getFormsLsnrServlet();

##  $self->_getSSLConfig();

}

#------------------------------------------------------------------------------
# _trimDomain
#------------------------------------------------------------------------------

sub _trimDomain
{
  my $self = shift;

  if ( $self->{$HOST_NAME} =~ /^([^\.]+)\.(.*$)/ )
   {
     $self->{$HOST_NAME} =~ s/^([^\.]+)\.(.*$)/\1/;
   }
}

#------------------------------------------------------------------------------
# _getNodeInfoIfAdConfigEnabled
#------------------------------------------------------------------------------

sub _getNodeInfoIfAdConfigEnabled
{
  my $self = shift;

  my $isWebNode		= $self->txkGetContextVar({ oavar => 's_isWeb' });
  my $isFormsNode	= $self->txkGetContextVar({ oavar => 's_isForms' });
  my $isConcNode	= $self->txkGetContextVar({ oavar => 's_isConc' });
  my $isAdminNode	= $self->txkGetContextVar({ oavar => 's_isAdmin' });
  my $isDBNode		= $self->txkGetContextVar({ oavar => 's_isDB' });

  $isDBNode    =~ tr /a-z/A-Z/;
  $isAdminNode =~ tr /a-z/A-Z/;
  $isWebNode   =~ tr /a-z/A-Z/;
  $isFormsNode =~ tr /a-z/A-Z/;
  $isConcNode  =~ tr /a-z/A-Z/;

  if ( $isDBNode eq "YES" )
   {
     $self->{$IS_DB_NODE} = TXK::Util::TRUE;
   }

  if ( $isAdminNode eq "YES" )
   {
     $self->{$IS_ADMIN_NODE} = TXK::Util::TRUE;
   }

  if ( $isWebNode eq "YES" )
   {
     $self->{$IS_WEB_NODE} = TXK::Util::TRUE;
   }

  if ( $isFormsNode eq "YES" )
   {
     $self->{$IS_FORMS_NODE} = TXK::Util::TRUE;
   }

  if ( $isConcNode eq "YES" )
   {
     $self->{$IS_CONC_NODE} = TXK::Util::TRUE;
   }

}

#------------------------------------------------------------------------------
# Retrieve iAS version
#------------------------------------------------------------------------------

#------------------------------------------------------------------------------
# Retrieve
#------------------------------------------------------------------------------

#------------------------------------------------------------------------------
## _getJDKVersionWeb()
#------------------------------------------------------------------------------

sub _getJDKVersionWeb
{
  my $self = $ARG[0];

  my $java_cmd;

  my $jdkversion = "0";

  ##--------------------------------------------------------
  # No need to check for JDK on the Web and Concurrent
  # processing nodes.
  ##--------------------------------------------------------

  if ( ! $self->{$IS_WEB_NODE} )
   {
     return NULL;
   }

  ##--------------------------------------------------------
  # Constructing the full path to the log file
  ##--------------------------------------------------------

  my $appltmp   = $self->txkGetApplTmp();

  my $log_f     = $appltmp . "/" . "java_ver.txt";

  my $logfile   = TXK::OSD->trDirPathToBase($log_f);

  ##--------------------------------------------------------
  # Now run the java executable to determine its version
  ##--------------------------------------------------------

  my $process   = TXK::Process->new();

  if ( !defined($self->{$JAVA_CMD_WEB}) )
   {
     if ( $self->txkIsAdConfigEnabled() eq TXK::Util::TRUE )
      {

        my $installed_jdk_location = $self->txkGetContextVar({ oavar => 's_jdktop' });

        $self->{$JDK_TOP_WEB} = TXK::OSD->trDirPathToBase($installed_jdk_location);

        my $java = $installed_jdk_location . "/bin/" . "java";

	$java .= ".exe" if ( TXK::OSD->isWindows() );

        $java_cmd = TXK::OSD->trDirPathToBase($java);

        $self->{$VALIDATE_MESSAGE} .= "\n[NOTE]\t The location of the JDK installation as defined by the value of\n\t \(s_jdktop\) in the Applications Context file is as follows:\n\n\t $self->{$JDK_TOP_WEB}\n";

      }
     else
      {
        #----------------------------------------------------------------------
        # IO package takes care of any path translation.
        #----------------------------------------------------------------------

	#
	# @@ Do nothing for now!
	#

      }
   }
  else
   {
     $java_cmd = $self->{$JAVA_CMD_WEB};
   }

  ##---------------------------------------------------------
  # Make sure java is 1.3.1 or higher.
  ##---------------------------------------------------------

  $self->{$JAVA_CMD_WEB} = $java_cmd;

  $self->{$JDK_VERSION_WEB} = $self->_doGetJDKVersion({'javacmd' => $self->{$JAVA_CMD_WEB} } );

  return $self->{$JDK_VERSION_WEB} ;
}

#------------------------------------------------------------------------------
## _getJDKVersionAdmin()
#------------------------------------------------------------------------------

sub _getJDKVersionAdmin
{
  my $self = $ARG[0];

  my $java_cmd;

  my $jdkversion = "0";

  ##--------------------------------------------------------
  # This version returns the Java version used by AD which is the only component in ADMIN node to 
  # use Java. Since AD is used on all nodes, no need to check for specific node types.
  ##--------------------------------------------------------

  # if ( ! $self->{$IS_ADMIN_NODE} )
  # {
  #   return NULL;
  # }

  ##--------------------------------------------------------
  # Constructing the full path to the log file
  ##--------------------------------------------------------

  my $appltmp   = $self->txkGetApplTmp();

  my $log_f     = $appltmp . "/" . "java_ver.txt";

  my $logfile   = TXK::OSD->trDirPathToBase($log_f);

  ##--------------------------------------------------------
  # Now run the java executable to determine its version
  ##--------------------------------------------------------

  my $process   = TXK::Process->new();

  if ( !defined($self->{$JAVA_CMD_ADMIN}) )
   {
     if ( $self->txkIsAdConfigEnabled() eq TXK::Util::TRUE )
      {

        my $installed_jdk_location = $self->txkGetContextVar({ oavar => 's_jretop' });

        #$self->{$JDK_TOP_ADMIN} = TXK::OSD->trDirPathToBase($installed_jdk_location);

        my $java = $self->txkGetContextVar({ oavar => 's_adjvaprg' });

        $java_cmd = TXK::OSD->trDirPathToBase($java);

        $self->{$VALIDATE_MESSAGE} .= "\n[NOTE]\t The location of the JDK installation as defined by the value of\n\t \(s_jdktop\) in the Applications Context file is as follows:\n\n\t $self->{$JDK_TOP_WEB}\n";

      }
     else
      {
        #----------------------------------------------------------------------
        # IO package takes care of any path translation.
        #----------------------------------------------------------------------

	#
	# @@ Do nothing for now!
	#

      }
   }
  else
   {
     $java_cmd = $self->{$JAVA_CMD_ADMIN};
   }

  ##---------------------------------------------------------
  # Make sure java is 1.3.1 or higher.
  ##---------------------------------------------------------

  $self->{$JAVA_CMD_ADMIN} = $java_cmd;

  $self->{$JDK_VERSION_ADMIN} = $self->_doGetJDKVersion({'javacmd' => $self->{$JAVA_CMD_ADMIN} } );

  return $self->{$JDK_VERSION_ADMIN};
}

#------------------------------------------------------------------------------
## returns the java version
## _doGetJDKVersion({jdk => <java cmd>})
## 
#------------------------------------------------------------------------------

sub _doGetJDKVersion
{
    my $self = $ARG[0];
    my $args = $ARG[1];

    my $javacmd = "javacmd";

    TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

    TXK::Util->isValidArgs( { args => $args, reqd => ["javacmd"] });
    
    my $fsys = TXK::FileSys->new();
    
    $fsys->access( { fileName   => $args->{$javacmd} ,
		     checkMode  => TXK::FileSys::READ_ACCESS,
		     type       => TXK::FileSys::FILE } )
	or TXK::Error->stop("Invalid Java cmd : $args->{$javacmd}" , $fsys->getError());

    my $proc = TXK::Process->new();

    ##--------------------------------------------------------
    # Constructing the full path to the log file
    ##--------------------------------------------------------

    my $logfile   = TXK::OSD->trDirPathToBase($self->txkGetApplTmp()."/java_ver.txt");

    $proc->abortOnError( { enable => TXK::Util::FALSE } );  # since jre 1.1.8 returns non-zero status for success

    my $run_status = $proc->run({
	                          command    => "$args->{$javacmd}"  ,
				  arg1       => "-version"           ,
                                  stdout     => $logfile             ,
                                });

    $proc->abortOnError( { enable => TXK::Util::TRUE } );

    my $io = TXK::IO->new();

    $io->open({ fileName => $logfile });

    my $io_ref = $io->getFileHandle();
    my @io_data = <$io_ref>;

    $io->close();

    $fsys->rmfile({ fileName => $logfile}) or TXK::Error->stop("Can not delete $logfile\n");

    my ($jdk_version, $version_output);

    foreach ( @io_data ) { $version_output .= $_;  }

    if ($version_output =~ /version \"(.*)\"/) 
     {
	 $jdk_version = $1;
     } 
    elsif ($version_output =~ /Version (.*)\n/) 
     {    # special entry for 1.1.8!
	 $jdk_version = $1;
      } 

    #
    # HP 1.1.8 JRE returns a character string with the following format:
    #  jre version "HP-UX Java C.01.18.05 01/25/2001 17:59:59 jind04"
    #
    if (TXK::OSD->isHPUX()) 
     {   
	 # special entry for HP-UX 1.1.8!
	 if ($io_data[0] =~ /C.01.18/) 
	  { 
	      $jdk_version = "1.1.8";
	  }
     }

    return $jdk_version;
}


#------------------------------------------------------------------------------
## _getJDKVersionCP()
#------------------------------------------------------------------------------

sub _getJDKVersionCP
{
  my $self = $ARG[0];

  my $java_cmd;

  my $jdkversion = "0";

  ##--------------------------------------------------------
  # No need to check for JDK on the Web and Concurrent
  # processing nodes.
  ##--------------------------------------------------------

  if ( ! $self->{$IS_CONC_NODE} )
   {
     return NULL;
   }

  ##--------------------------------------------------------
  # Constructing the full path to the log file
  ##--------------------------------------------------------

  my $appltmp  = $self->txkGetApplTmp();

  my $log_f    = $appltmp . "/" . "java_ver.txt";

  my $logfile   = TXK::OSD->trDirPathToBase($log_f);

  ##--------------------------------------------------------
  # Now run the java executable to determine its version
  ##--------------------------------------------------------

  my $process   = TXK::Process->new();

  if ( !defined($self->{$JAVA_CMD_CP}) )
   {
     if ( $self->txkIsAdConfigEnabled() eq TXK::Util::TRUE )
      {
        my $installed_jdk_location = $self->txkGetContextVar({ oavar => 's_afjretop' });

        $self->{$JDK_TOP_CP} = TXK::OSD->trDirPathToBase($installed_jdk_location);

        my $java = $installed_jdk_location . "/bin/" . "java";

	$java .= ".exe" if ( TXK::OSD->isWindows() );

        $java_cmd = TXK::OSD->trDirPathToBase($java);

        $self->{$VALIDATE_MESSAGE} .= "\n[NOTE]\t The location of the JDK installation as defined by the value of\n\t \(s_afjretop\) in the Applications Context file is as follows:\n\n\t $self->{$JDK_TOP_CP}\n";

      }
     else
      {
        #----------------------------------------------------------------------
        # IO package takes care of any path translation.
        #----------------------------------------------------------------------

	#
	# @@ Do nothing for now!
	#

      }
   }
  else
   {
     $java_cmd = $self->{$JAVA_CMD_CP};
   }

  ##---------------------------------------------------------
  # Make sure java is 1.3.1 or higher.
  ##---------------------------------------------------------

  $self->{$JAVA_CMD_CP} = $java_cmd;

  $self->{$JDK_VERSION_CP} = $self->_doGetJDKVersion({'javacmd' => $self->{$JAVA_CMD_CP} } );

  return $self->{$JDK_VERSION_CP} ;
}

#------------------------------------------------------------------------------
# txkGetContextType
# input param is a context file
# output is the value of s_contexttype
# this is not an instance method.
#------------------------------------------------------------------------------

sub txkGetContextType
{
    my $self = $ARG[0];

    my $args = $ARG[1];

    TXK::Util->isValidArgs( { args => $args, reqd => ["$APPS_CTX_FILE_KEY"] });

    my $context = TXK::XML->new();

    $context->loadDocument ( { file => $args->{$APPS_CTX_FILE_KEY} } );

    return $context->getOAVar( 's_contexttype' );
}

#------------------------------------------------------------------------------
# txkGetContextVar;
#------------------------------------------------------------------------------
sub txkGetContextVar
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( defined $args  and defined $self->{$APPS_CTX_FILE} )
    {
      if ( !defined ( $self->{$CONTEXT_XMLOBJ} ))
	{
	  $self->{$CONTEXT_XMLOBJ} = TXK::XML->new();
	  $self->{$CONTEXT_XMLOBJ}->loadDocument ( { file => $self->{$APPS_CTX_FILE} } );
	}

      return $self->{$CONTEXT_XMLOBJ}->getOAVar( $args->{'oavar'} );
    }
}


#------------------------------------------------------------------------------
## txkGetJDKVersion()
#------------------------------------------------------------------------------

sub txkGetJDKVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $installed_jdk_version;

  if ( ( ! ( $self->txkIsWebNode() )) &&
       ( ! ( $self->txkIsConcNode() )) &&
       ( ! ( $self->txkIsAdminNode() )) )
    {
      return JDK_NOT_REQUIRED;
    }

  if ( defined($args->{'NODE_TEST'}) && ($args->{'NODE_TEST'} eq 'conc')  )
    {
      if (!  ( $self->txkIsConcNode()) )
	{
	  return JDK_NOT_SPECIFIED_NODE_TYPE;
	}
      if ( defined($self->{$JDK_VERSION_CP}) )
	{
	  return $self->{$JDK_VERSION_CP};
	}
      else
	{
	  return $self->_getJDKVersionCP();
	}
    }

  if ( defined($args->{'NODE_TEST'}) && ($args->{'NODE_TEST'} eq 'web') )
    {
      if (!  ( $self->txkIsWebNode()) )
	{
	  return JDK_NOT_SPECIFIED_NODE_TYPE;
	}
      if ( defined($self->{$JDK_VERSION_WEB}) )
	{
	  return $self->{$JDK_VERSION_WEB};
	}
      else
	{
	  return $self->_getJDKVersionWeb();
	}
    }

  if ( defined($args->{'NODE_TEST'}) && ($args->{'NODE_TEST'} eq 'admin')  )
    {

      # We have to get the jdk version used by AD which is 
      # present on all node types and is used for admin purposes on all nodes

      if ( defined($self->{$JDK_VERSION_ADMIN}) )
	{
	  return $self->{$JDK_VERSION_ADMIN};
	}
      else
	{
	  return $self->_getJDKVersionAdmin();
	}
    }

  if ( defined($args->{'javacmd'}) && ($args->{'javacmd'} ne '')  )
   {
       return $self->_doGetJDKVersion({'javacmd' => $args->{'javacmd'} });
   }

  return JDK_NOT_SPECIFIED_NODE_TYPE;
}


#------------------------------------------------------------------------------
## txkIsConfiguredWithJDK()
#------------------------------------------------------------------------------
sub txkIsConfiguredWithJDK
{
  my $self   = $ARG[0];
  my $args   = $ARG[1];

  my $web_status = TXK::Error::SUCCESS;
  my $conc_status = TXK::Error::SUCCESS;

  my $status_conc = 1;
  if ( $self->txkIsWebNode() )
    {
      $web_status=$self->txkIsConfiguredWithJDKWeb();
    }

  if ( $self->txkIsConcNode() )
    {
      $conc_status=$self->txkIsConfiguredWithJDKCP();
    }

  return ($web_status && $conc_status);
}


#------------------------------------------------------------------------------
## txkIsConfiguredWithJDKWeb()
#------------------------------------------------------------------------------

sub txkIsConfiguredWithJDKWeb
{
  my $self   = $ARG[0];
  my $args   = $ARG[1];

#  $self->txkIsAdConfigEnabled($args);

#  $self->_getNodeInfo();

  if ( $self->txkIsWebNode() )
   {
     my $web_status;
     my $prereq_status = TXK::Error::SUCCESS;
     my $config_status = TXK::Error::SUCCESS;

     my $passed_in_jdk;

     my $JDBC_FILE;

     if ( keys(%$args))

      {
        my $jdkversion_key = JDKVERSION_KEY;

        foreach my $key (keys %$args)
         {
           if ( $key eq $jdkversion_key )
            {
              $passed_in_jdk = $args->{$jdkversion_key};
              last;
            }
         }
      }

     if ( $passed_in_jdk == 0 )
      {
        $passed_in_jdk = DEFAULT_JDKVERSION;
      }

     my $installed_jdk_version = $self->_getJDKVersionWeb();
     my $pattern_1             = $self->{$JDK_TOP_WEB};
     my $pattern_2             = $APPSBORG_FILE;

     #
     # JAVA command not found!!
     #

     if ( $installed_jdk_version eq JAVACMD_NOT_FOUND )
      {
        $self->{$VALIDATE_MESSAGE} .= "\n[FAIL]\t The Java executable shown below could not be located:\n\t $self->{$JAVA_CMD_WEB}\n\n";

        if ( $passed_in_jdk =~ $JDK_13_PATTERN )
         {
           $self->{$VALIDATE_MESSAGE} .= "\t The Web node of this E-Business Suite\n\t Release 11i system is not configured with JDK $passed_in_jdk.\n\t Please see the My Oracle Support Note 130091.1 for more information on\n\t Upgrading to JDK 1.3 with the Oracle E-Business Suite Release 11i.\n\n";

         }
        elsif ( $passed_in_jdk =~ $JDK_14_PATTERN )
         {
           $self->{$VALIDATE_MESSAGE} .= "\n[FAIL]\t The Web node of this E-Business Suite\n\t Release 11i system is not configured with JDK $passed_in_jdk.\n\t Please see the My Oracle Support Note 246105.1 for more information on\n\t Upgrading to J2SE 1.4 with the Oracle E-Business Suite Release 11i.\n\n";

         }

        return TXK::Error::FAIL;

      }
     elsif ( $installed_jdk_version eq JDK_NOT_REQUIRED )
      {
        $self->{$VALIDATE_MESSAGE} .= "\n[WARN]\t It is not necessary to check for an installation of JDK on this node\n\t of the E-Business Suite Release 11i environment, as this node is\n\t neither a Web node nor a Concurrent processing node.\n\n";

        return TXK::Error::WARNING;

      }

     #
     # JAVA cmd found and executed. Now get the version returned by the cmd.
     #

     if ( $installed_jdk_version =~ $JDK_13_PATTERN )
      {
        $JDBC_FILE = $JDBC_FILE_12;
      }
     elsif ( $installed_jdk_version =~ $JDK_14_PATTERN )
      {
        $JDBC_FILE = $JDBC_FILE_14;
      }

     if ( $installed_jdk_version lt $passed_in_jdk )
      {
        if ( $passed_in_jdk =~ $JDK_13_PATTERN )
         {
           $self->{$VALIDATE_MESSAGE} .= "\n[FAIL]\t The version of the currently installed JDK \($installed_jdk_version\)\n\t is lower than $passed_in_jdk.\n\n\t The Web node of this E-Business Suite\n\t Release 11i system is not configured with JDK $passed_in_jdk.\n\t Please see the My Oracle Support Note 130091.1 for more information on\n\t Upgrading to JDK 1.3 with the Oracle E-Business Suite Release 11i.\n\n";

         }
        elsif ( $passed_in_jdk =~ $JDK_14_PATTERN )
         {
           $self->{$VALIDATE_MESSAGE} .= "\n[FAIL]\t The version of the currently installed JDK \($installed_jdk_version\)\n\t is lower than $passed_in_jdk.\n\n\t The Web node of this E-Business Suite\n\t Release 11i system is not configured with JDK $passed_in_jdk.\n\t Please see the My Oracle Support Note 246105.1 for more information on\n\t Upgrading to J2SE 1.4 with the Oracle E-Business Suite Release 11i.\n\n";

         }

        return TXK::Error::FAIL;

      }

     #
     # Verify if the pre-req patches, if any, were already installed.
     #

     if ( keys(%$args))
      {
	my $web_key;
        my $web_prereq_status  = TXK::Error::SUCCESS;

	if ( $installed_jdk_version =~ $JDK_13_PATTERN )
         {
	   $web_key  = WEB_PATCH_LIST_13;

         }
	elsif ( $installed_jdk_version =~ $JDK_14_PATTERN )
	 {
	   $web_key  = WEB_PATCH_LIST_14;
	 }

	if ( exists($args->{$web_key}) )
         {
           $web_prereq_status = $self->_verifyPatch($args->{$web_key});

         }

        if ($web_prereq_status  eq TXK::Error::SUCCESS)
         {
           $prereq_status = TXK::Error::SUCCESS;

         }
        else
         {
           $prereq_status = TXK::Error::WARNING;

         }

      }

     #
     # Version of the installed JAVA is >= the one passed in.
     # Now, determine if the node is fully configured with the installed JAVA.
     #
     # Verify if the Web node is fully configured with the supplied JDK version
     #

     $web_status = $self->_verifyConfigJDKWeb($IS_WEB_NODE);

     #-------------------------------------------------------------------------
     # Wrap up everything and get ready to return to the caller.
     #-------------------------------------------------------------------------

     if ( $web_status    eq TXK::Error::SUCCESS )
      {
	if ( $prereq_status eq TXK::Error::SUCCESS )
	 {
	   $config_status = TXK::Error::SUCCESS;
	 }
	else
	 {
	   $config_status = TXK::Error::WARNING;
	 }

      }
     else
      {
        $config_status = TXK::Error::FAIL;

      }

     if ( $config_status ne TXK::Error::SUCCESS )
      {
	my $FLAG;

	if ( $web_status  eq TXK::Error::SUCCESS )
	 {
	   $FLAG = "WARN";
	 }
	else
	 {
	   $FLAG = "FAIL";
	 }

	if ( $installed_jdk_version =~ $JDK_13_PATTERN )
         {
           $self->{$VALIDATE_MESSAGE} .= "\n[$FLAG]\t The Web node of this E-Business Suite\n\t Release 11i system does not appear already configured\n\t or ready to be configured with JDK $installed_jdk_version.\n\t Please see the My Oracle Support Note 130091.1 for more information on\n\t Upgrading to JDK 1.3 with the Oracle E-Business Suite Release 11i.\n\n";

         }
        elsif ( $installed_jdk_version =~ $JDK_14_PATTERN )
         {
           $self->{$VALIDATE_MESSAGE} .= "\n[$FLAG]\t The Web node of this E-Business Suite\n\t Release 11i system does not appear already configured\n\t or ready to be configured with JDK $installed_jdk_version.\n\t Please see the My Oracle Support Note 246105.1 or more information on\n\t Upgrading to J2SE 1.4 with the Oracle E-Business Suite Release 11i.\n\n";

         }
      }
     else
      {
        $self->{$VALIDATE_MESSAGE} .= "\n[PASS]\t The Web node of this E-Business Suite\n\t Release 11i system appears configured with JDK $installed_jdk_version.\n\n";

      }

     return $config_status;

   }
  else
   {
     $self->{$VALIDATE_MESSAGE} .= "\n[WARN]\t It is not necessary to verify the JDK version if this\n\t is not a Web node.\n";

     return TXK::Error::WARNING;
   }

  return TXK::Error::SUCCESS;
}


#------------------------------------------------------------------------------
## txkIsConfiguredWithJDKCP()
#------------------------------------------------------------------------------

sub txkIsConfiguredWithJDKCP
{
  my $self   = $ARG[0];
  my $args   = $ARG[1];

#  $self->txkIsAdConfigEnabled($args);

#  $self->_getNodeInfo();

  if ( $self->txkIsConcNode() )
   {
     my $conc_status;
     my $prereq_status = TXK::Error::SUCCESS;
     my $config_status = TXK::Error::SUCCESS;

     my $passed_in_jdk;

     my $JDBC_FILE;

     if ( keys(%$args))

      {
        my $jdkversion_key = JDKVERSION_KEY;

        foreach my $key (keys %$args)
         {
           if ( $key eq $jdkversion_key )
            {
              $passed_in_jdk = $args->{$jdkversion_key};
              last;
            }
         }
      }

     if ( $passed_in_jdk == 0 )
      {
        $passed_in_jdk = DEFAULT_JDKVERSION;
      }

     my $installed_jdk_version = $self->_getJDKVersionCP();
     my $pattern_1             = $self->{$JDK_TOP_CP};
     my $pattern_2             = $APPSBORG_FILE;

     #
     # JAVA command not found!!
     #

     if ( $installed_jdk_version eq JAVACMD_NOT_FOUND )
      {
        $self->{$VALIDATE_MESSAGE} .= "\n[FAIL]\t The Java executable shown below could not be located:\n\t $self->{$JAVA_CMD_CP}\n\n";

        if ( $passed_in_jdk =~ $JDK_13_PATTERN )
         {
           $self->{$VALIDATE_MESSAGE} .= "\t The Concurrent processing node of this E-Business Suite\n\t Release 11i system is not configured with JDK $passed_in_jdk.\n\t Please see the My Oracle Support Note 130091.1 for more information on\n\t Upgrading to JDK 1.3 with the Oracle E-Business Suite Release 11i.\n\n";

         }
        elsif ( $passed_in_jdk =~ $JDK_14_PATTERN )
         {
           $self->{$VALIDATE_MESSAGE} .= "\n[FAIL]\t The Concurrent processing node of this E-Business Suite\n\t Release 11i system is not configured with JDK $passed_in_jdk.\n\t Please see the My Oracle Support Note 246105.1 for more information on\n\t Upgrading to J2SE 1.4 with the Oracle E-Business Suite Release 11i.\n\n";

         }

        return TXK::Error::FAIL;

      }
     elsif ( $installed_jdk_version eq JDK_NOT_REQUIRED )
      {
        $self->{$VALIDATE_MESSAGE} .= "\n[WARN]\t It is not necessary to check for an installation of JDK on this node\n\t of the E-Business Suite Release 11i environment, as this node is\n\t neither a Web node nor a Concurrent processing node.\n\n";

        return TXK::Error::WARNING;

      }

     #
     # JAVA cmd found and executed. Now get the version returned by the cmd.
     #

     if ( $installed_jdk_version =~ $JDK_13_PATTERN )
      {
        $JDBC_FILE = $JDBC_FILE_12;
      }
     elsif ( $installed_jdk_version =~ $JDK_14_PATTERN )
      {
        $JDBC_FILE = $JDBC_FILE_14;
      }

     if ( $installed_jdk_version lt $passed_in_jdk )
      {
        if ( $passed_in_jdk =~ $JDK_13_PATTERN )
         {
           $self->{$VALIDATE_MESSAGE} .= "\n[FAIL]\t The version of the currently installed JDK \($installed_jdk_version\)\n\t is lower than $passed_in_jdk.\n\n\t The Concurrent processing node of this E-Business Suite\n\t Release 11i system is not configured with JDK $passed_in_jdk.\n\t Please see the My Oracle Support Note 130091.1 for more information on\n\t Upgrading to JDK 1.3 with the Oracle E-Business Suite Release 11i.\n\n";

         }
        elsif ( $passed_in_jdk =~ $JDK_14_PATTERN )
         {
           $self->{$VALIDATE_MESSAGE} .= "\n[FAIL]\t The version of the currently installed JDK \($installed_jdk_version\)\n\t is lower than $passed_in_jdk.\n\n\t The Concurrent processing node of this E-Business Suite\n\t Release 11i system is not configured with JDK $passed_in_jdk.\n\t Please see the My Oracle Support Note 246105.1 for more information on\n\t Upgrading to J2SE 1.4 with the Oracle E-Business Suite Release 11i.\n\n";

         }

        return TXK::Error::FAIL;

      }

     #
     # Verify if the pre-req patches, if any, were already installed.
     #

     if ( keys(%$args))
      {
	my $conc_key;
        my $conc_prereq_status = TXK::Error::SUCCESS;

	if ( $installed_jdk_version =~ $JDK_13_PATTERN )
         {
	   $conc_key = CONC_PATCH_LIST_13;
         }
	elsif ( $installed_jdk_version =~ $JDK_14_PATTERN )
	 {
	   $conc_key = CONC_PATCH_LIST_14;
	 }

	if ( exists($args->{$conc_key}) )
         {
           $conc_prereq_status = $self->_verifyPatch($args->{$conc_key});
         }

        if ($conc_prereq_status eq TXK::Error::SUCCESS)
	  {
           $prereq_status = TXK::Error::SUCCESS;
         }
        else
         {
           $prereq_status = TXK::Error::WARNING;

         }
      }

     #
     # Verify if the Concurrent processing node is configured with the
     # supplied JDK version
     #

     $conc_status = $self->_verifyConfigJDKCP($IS_CONC_NODE);

     #-------------------------------------------------------------------------
     # Wrap up everything and get ready to return to the caller.
     #-------------------------------------------------------------------------

     if ( $conc_status   eq TXK::Error::SUCCESS )
      {
	if ( $prereq_status eq TXK::Error::SUCCESS )
	 {
	   $config_status = TXK::Error::SUCCESS;
	 }
	else
	 {
	   $config_status = TXK::Error::WARNING;
	 }

      }
     else
      {
        $config_status = TXK::Error::FAIL;
      }

     if ( $config_status ne TXK::Error::SUCCESS )
      {
	my $FLAG;

	if ( $conc_status eq TXK::Error::SUCCESS )
	 {
	   $FLAG = "WARN";
	 }
	else
	 {
	   $FLAG = "FAIL";
	 }

	if ( $installed_jdk_version =~ $JDK_13_PATTERN )
         {
           $self->{$VALIDATE_MESSAGE} .= "\n[$FLAG]\t The Concurrent processing node of this E-Business Suite\n\t Release 11i system does not appear already configured\n\t or ready to be configured with JDK $installed_jdk_version.\n\t Please see the My Oracle Support Note 130091.1 for more information on\n\t Upgrading to JDK 1.3 with the Oracle E-Business Suite Release 11i.\n\n";

         }
        elsif ( $installed_jdk_version =~ $JDK_14_PATTERN )
         {
           $self->{$VALIDATE_MESSAGE} .= "\n[$FLAG]\t The Concurrent processing node of this E-Business Suite\n\t Release 11i system does not appear already configured\n\t or ready to be configured with JDK $installed_jdk_version.\n\t Please see the My Oracle Support Note 246105.1 or more information on\n\t Upgrading to J2SE 1.4 with the Oracle E-Business Suite Release 11i.\n\n";

         }
      }
     else
      {
        $self->{$VALIDATE_MESSAGE} .= "\n[PASS]\t The Concurrent processing node of this E-Business Suite\n\t Release 11i system appears configured with JDK $installed_jdk_version.\n\n";

      }

     return $config_status;

   }
  else
   {
     $self->{$VALIDATE_MESSAGE} .= "\n[WARN]\t It is not necessary to verify the JDK version if this\n\t is not the Web or Concurrent processing node.\n";

     return TXK::Error::WARNING;
   }

  return TXK::Error::SUCCESS;
}


#------------------------------------------------------------------------------
## _getAppsCtxLocation()
#------------------------------------------------------------------------------

sub _getAppsCtxLocation
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $context_file = NULL;
  my $oracle_sid   = NULL;
  my $dirName      = NULL;
  my $ctxfile_1    = NULL;
  my $ctxfile_2    = NULL;
  my $findStatus   = NULL;

  if ( ( ref( $args ) eq "HASH" ) && ( keys %{$args} ) )
   {
     my $APPS_CTX_FILE_KEY = APPS_CTX_FILE_KEY;

     if ( ( exists  ( $args->{$APPS_CTX_FILE_KEY} )) &&
          ( defined ( $args->{$APPS_CTX_FILE_KEY} )) &&
          ( $args->{$APPS_CTX_FILE_KEY} ne NULL    )
        )
      {
	#
	# this means that the context file is specified.
	#

        $context_file = $args->{$APPS_CTX_FILE_KEY};

        my $ctx_status = $self->_isFileFound($context_file);

        if ( $ctx_status eq TXK::Util::FALSE )
         {
           $self->{$ADCONFIG_ENABLED} = TXK::Util::FALSE;

           $self->{$VALIDATE_MESSAGE} .= "\n[NOTE]\t It appears ADCONFIG is not enabled in this E-Business Release 11i\n\t environment, as the supplied Applications context file shown\n\t below cannot be located:\n\t $context_file\n";

           return TXK::Error::FAIL;

         }
        else
         {
           $self->{$APPS_CTX_FILE} = TXK::OSD->trDirPathToBase($context_file);

           return TXK::Error::SUCCESS;

         }
      }
   }


  #
  # If we come to this point, no context file is specified.
  # Will have to figure out by using some default settings:
  #
  # 1. First try to get it from the setting of CONTEXT_FILE
  # 2. Else, base it on <CONTEXTNAME>.xml and check for its existence.
  #

  $context_file = TXK::OSD->getEnvVar({ name => "CONTEXT_FILE"});

  if ( $context_file ne NULL )
   {

     $findStatus = $self->_isFileFound($context_file);

     if ( $findStatus eq TXK::Util::TRUE )
      {
        #
        # If we found it, return immediately.
        #

        $self->{$APPS_CTX_FILE} = TXK::OSD->trDirPathToBase($context_file);

        return TXK::Error::SUCCESS;

      }
   }

  #
  # If we come here, CONTEXT_FILE is not defined.
  # Must base on the <CONTEXTNAME>.xml
  #

  $oracle_sid = $self->{$TWO_TASK};

  #
  # try getting the APPL_TOP from the env var since there is no context file specified
  #
  #my $appl_top = $self->txkGetApplTop($args);
 
  my $appl_top = TXK::OSD->getEnvVar({ name => "APPL_TOP"});

  $dirName = $appl_top . "/admin/";

  $ctxfile_1 = $dirName . $oracle_sid . "_" . $self->{$HOST_NAME} . ".xml";
  $ctxfile_2 = $dirName . $oracle_sid . ".xml";

  #--------------------------------------------------------------------------
  # Our preferred format is <TWO_TASK>_<hostname>.xml
  # If found it, return immediately.
  #--------------------------------------------------------------------------

  $findStatus = $self->_isFileFound($ctxfile_1);

  if ( $findStatus eq TXK::Util::TRUE )
   {
     $self->{$APPS_CTX_FILE} = TXK::OSD->trDirPathToBase($ctxfile_1);

     return TXK::Error::SUCCESS;
   }

  #-------------------------------------------------------------------------
  # Our preferred format is <TWO_TASK>_<hostname>.xml
  # But if it doesn't exist, we try <TWO_TASK>.xml
  #-------------------------------------------------------------------------

  $findStatus = $self->_isFileFound($ctxfile_2);

  if ( $findStatus eq TXK::Util::TRUE )
   {
     $self->{$APPS_CTX_FILE} = TXK::OSD->trDirPathToBase($ctxfile_2);

     return TXK::Error::SUCCESS;
   }

  if ( $findStatus eq TXK::Util::FALSE )
   {
     $self->{$VALIDATE_MESSAGE} .= "\n[NOTE]\t It appears ADCONFIG is not enabled in this E-Business Release 11i\n\t environment, as none of the Applications context files shown\n\t below can be located:\n\t $context_file\n\t $ctxfile_1\n\t $ctxfile_2\n";

     return TXK::Error::FAIL;

   }

}

#-----------------------------------------------------------------------------
## txkIsDBContext
## public non-instnace specific method
#------------------------------------------------------------------------------

sub txkIsDBContext
{
  my $self = $ARG[0];

  my $args = $ARG[1];

  TXK::Util->isValidArgs( { args => $args, reqd => ["$APPS_CTX_FILE_KEY"] });

  my $context = $self->txkGetContextType( { $APPS_CTX_FILE_KEY => $args->{$APPS_CTX_FILE_KEY} } );

  return TXK::Util::TRUE if ( $context =~ m/^Database Context$/ );

  return TXK::Util::FALSE;
}

#-----------------------------------------------------------------------------
## _txkIsDBContext
## private instance specific method
#------------------------------------------------------------------------------

sub _txkIsDBContext
{
  my $self = $ARG[0];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  my $APPS_CTX_FILE_KEY = APPS_CTX_FILE_KEY;

  unless ( defined $self->{$CONTEXT_TYPE} )
   {
       $self->{$CONTEXT_TYPE} = $self->txkGetContextType ( { $APPS_CTX_FILE_KEY => $self->{$APPS_CTX_FILE} } );
   }

 if ( $self->{$APPS_CTX_FILE} ne "" )
  {
      return $self->txkIsDBContext( { $APPS_CTX_FILE_KEY => $self->{$APPS_CTX_FILE} } );
  }
 
  return TXK::Util::FALSE;
}

#------------------------------------------------------------------------------
## _isAdConfigEnabled()
#------------------------------------------------------------------------------

sub _isAdConfigEnabled
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $loc_status = $self->_getAppsCtxLocation($args);

  if ( $loc_status eq TXK::Error::FAIL )
   {
     #
     # Error accessing the context file. Consider that as non-adconfig enabled
     # and return immediately.
     #

     $self->{$ADCONFIG_ENABLED} = TXK::Util::FALSE;

     return $self->{$ADCONFIG_ENABLED};

   }

  my $ad_top  = TXK::OSD->getEnvVar({ name => "AD_TOP"});

  my $o_home  = TXK::OSD->getEnvVar({ name => "ORACLE_HOME"});

  my $adxmlfile = ADXMLCTX_FILE;

  my $adctx = $ad_top . "/admin/template/" . "$adxmlfile";

  if ( $self->_txkIsDBContext() )
   {
       $adxmlfile = ADXMLCTX_FILE_DB;

       $adctx = $o_home . "/appsutil/template/" . "$adxmlfile";
   }

  #
  # 1. Get the version of the template first
  # 2. If successful, get the version of the context file
  # 3. If successful, compare the two versions
  # 3. If equal, it is adconfig enabled
  #
  #    Other than that, it is not adconfig enabled.
  #

  my $adctxtmpl = TXK::OSD->trDirPathToBase($adctx);

  my $ctxtmpl_version = $self->_getCtxVersion($adctxtmpl);

  if ( $ctxtmpl_version eq NULL )
   {
     $self->{$ADCONFIG_ENABLED} = TXK::Util::FALSE;

     $self->{$VALIDATE_MESSAGE} .= "\n[NOTE]\t It appears ADCONFIG is not enabled in this E-Business Release 11i\n\t environment, as the version of the Applications Context template\n\t shown  below cannot be determined:\n\t $adctxtmpl.\n";

     return $self->{$ADCONFIG_ENABLED};

   }
  else
   {
     $self->{$CTXTMPL_VERSION} = $ctxtmpl_version;
   }

  my $apps_ctx_version = $self->_getCtxVersion($self->{$APPS_CTX_FILE});

  ##---------------------------------------------------------------------
  # If we are to be strict about verifying the AutoConfig then we need to
  # compare the versions. Otherwise if we want to just find out whether 
  # this is AutoConfig enabled, then this should be true at this point.
  ##---------------------------------------------------------------------

  $self->{$APPS_CTX_VERSION} = $apps_ctx_version;

  $self->{$ADCONFIG_ENABLED} = TXK::Util::TRUE;

  ##---------------------------------------------------------------------------
  # Now that we have the version of adxmlctx.tmp and the location of the Apps
  # context file. Let's compare the versions.
  # If they are equal, the environment is definitely adconfig enabled.
  ##---------------------------------------------------------------------------

  if ( $ctxtmpl_version ne $apps_ctx_version )
   {
     $self->{$ADCONTEXT_UPTODATE} = TXK::Util::FALSE;

     my $apps_ctx = $self->{$APPS_CTX_FILE};

     $self->{$VALIDATE_MESSAGE} .= "\n[NOTE]\t It appears ADCONFIG is not enabled in this E-Business Release 11i\n\t environment, as the versions of the Applications Context template\n\t and the Applications Context file shown below are not identical:\n\n\t $adctxtmpl\n\t Version: $ctxtmpl_version\n\n\t $apps_ctx\n\t Version: $apps_ctx_version\n";

   }
  else
   {
     $self->{$APPS_CTX_VERSION} = $apps_ctx_version;

     $self->{$ADCONTEXT_UPTODATE} = TXK::Util::TRUE;

     my $apps_ctx = $self->{$APPS_CTX_FILE};

     $self->{$VALIDATE_MESSAGE} .= "\n[NOTE]\t It appears ADCONFIG is enabled in this E-Business Release 11i\n\t environment, as the versions of the Applications Context template\n\t and the Applications Context file shown below are identical:\n\n\t $adctxtmpl\n\t Version: $ctxtmpl_version\n\n\t $apps_ctx\n\t Version: $apps_ctx_version\n";

   }

  return $self->{$ADCONFIG_ENABLED};
}

#----------------------------------------------------------------------------
## getAppsTempDir
## non instance specific. returns the temp dir to be used by Apps both in
## in Apps and DB tier.
#----------------------------------------------------------------------------

sub getAppsTempDir
{
 my $self = $ARG[0];

 my $args = $ARG[1];

 TXK::Util->isValidArgs({args=>$args,reqd=> ["$APPS_CTX_FILE_KEY"]} );

 my $tempDir;

 if ( TXK::Techstack->txkIsDBContext( { $APPS_CTX_FILE_KEY => $args->{$APPS_CTX_FILE_KEY} } ) )
  {
    my $xml = TXK::XML->new();

    $xml->loadDocument( { file => $args->{$APPS_CTX_FILE_KEY} } );

    $tempDir = TXK::OSD->trDirPathToBase( $xml->getOAVar( 's_temp' ) . "/TXK" );
  }
 else
  {
    $tempDir = TXK::OSD->trDirPathToBase(TXK::OSD->getEnvVar({name => 'APPLTMP'})."/TXK" );
  }
   
 # create the tempDir if this does not exist

 my $fsys = TXK::FileSys->new();

 $fsys->access ( { dirName   => $tempDir ,
		   type      => TXK::FileSys::DIRECTORY,
		   checkMode => TXK::FileSys::WRITE_ACCESS } )
 or (
 
 $fsys->create ( { dirName   => $tempDir ,
		   type      => TXK::FileSys::DIRECTORY } ) );

 return $tempDir;
}

#------------------------------------------------------------------------------
## txkIsContextUpToDate()
#------------------------------------------------------------------------------

sub txkIsContextUpToDate
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( !defined ($self->{$ADCONTEXT_UPTODATE})  )
   {
     $self->_isAdConfigEnabled($args);       
   }

  return $self->{$ADCONTEXT_UPTODATE};       
}

#------------------------------------------------------------------------------
## txkIsAdConfigEnabled()
#------------------------------------------------------------------------------

sub txkIsAdConfigEnabled
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( !defined($self->{$ADCONFIG_ENABLED}) )
   {
     $self->_isAdConfigEnabled($args);
   }

  return $self->{$ADCONFIG_ENABLED};
}

#------------------------------------------------------------------------------
## _getNodeInfo()
#------------------------------------------------------------------------------

sub _getNodeInfo
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $isDBNode;
  my $isAdminNode;
  my $isWebNode;
  my $isFormsNode;
  my $isConcNode;

  if ( $self->txkIsAdConfigEnabled($args) eq TXK::Util::TRUE )
   {
     $isDBNode    = $self->txkGetContextVar({ oavar => 's_isDB' });

     $isAdminNode = $self->txkGetContextVar({ oavar => 's_isAdmin' });

     $isWebNode   = $self->txkGetContextVar({ oavar => 's_isWeb' });

     $isFormsNode = $self->txkGetContextVar({ oavar => 's_isForms' });

     $isConcNode  = $self->txkGetContextVar({ oavar => 's_isConc' });
   }
  else
   {
     my $apps_user;
     if ( defined( $self->{$APPS_USER} ) )
      {
	$apps_user = $self->{$APPS_USER};
      }
     else
      {
	$apps_user = APPS_DEFAULT_USER;
      }

     my $apps_password;

     if ( defined( $self->{$APPS_PASSWORD} ) )
      {
	$apps_password = $self->{$APPS_PASSWORD};
      }
     else
      {
	$apps_password = APPS_DEFAULT_PWD;
      }

     my $appltmp  = $self->txkGetApplTmp($args);

     my $log_f    = $appltmp . "/" . "node_info.txt";

     my $logfile  = TXK::OSD->trDirPathToBase($log_f);

     #my $spath  = TXK::OSD->getEnvVar({ name => "FND_TOP"}) . "/patch/115/sql/";

     my $spath  = $self->txkGetContextVar({ oavar => 's_fndtop' }) . "/patch/115/sql/";

     my $s_path = TXK::OSD->trDirPathToBase($spath);

     my $sql    = $s_path . "txkGetNodeInfo.sql";

     my $node   = $self->{$HOST_NAME};

     my $sid    = $self->{$TWO_TASK};

     my $sqlctx = TXK::SQLPLUS->new();

     my $fsys   = TXK::FileSys->new();

     $fsys->access ({
		      fileName	=> $sql,
                      type	=> TXK::FileSys::FILE,
		      checkMode	=> TXK::FileSys::READ_ACCESS
		   })
     or TXK::Error->stop("Check File(s) Error", $fsys->getError());

     $sqlctx->setConnectInfo({
				user     => "$apps_user",
				password => "$apps_password",
				two_task => "$sid"
			    });

     $sqlctx->clearCommand();

     $sqlctx->addCommand("spool " . "$logfile");

     $sqlctx->addCommand("@" . "$sql " . "$node");

     $sqlctx->execute({ showOutput =>TXK::Util::FALSE, });

     my $io = TXK::IO->new();

     $io->open({ fileName => $logfile });

     my $io_ref  = $io->getFileHandle();
     my @io_data = <$io_ref>;

     $io->close();

     unlink($logfile) or return TXK::Error::FAIL;

     foreach (@io_data)
      {

        if ( /\s*SUPPORT_CP\s*=\s*TRUE/ )
         {
           $isConcNode = "yes";
         }

        if ( /\s*SUPPORT_FORMS\s*=\s*TRUE/ )
         {
           $isFormsNode = "yes";
         }

        if ( /\s*SUPPORT_WEB\s*=\s*TRUE/ )
         {
           $isWebNode = "yes";
         }

        if ( /\s*SUPPORT_ADMIN\s*=\s*TRUE/ )
         {
           $isAdminNode = "yes";
         }
      }
   }

  $isDBNode    =~ tr /a-z/A-Z/;
  $isAdminNode =~ tr /a-z/A-Z/;
  $isWebNode   =~ tr /a-z/A-Z/;
  $isFormsNode =~ tr /a-z/A-Z/;
  $isConcNode  =~ tr /a-z/A-Z/;


  if ( $isDBNode eq "YES" )
   {
     $self->{$IS_DB_NODE} = TXK::Util::TRUE;
   }
  else
   {
     $self->{$IS_DB_NODE} = TXK::Util::FALSE;
   }

  if ( $isAdminNode eq "YES" )
   {
     $self->{$IS_ADMIN_NODE} = TXK::Util::TRUE;
   }
  else
   {
     $self->{$IS_ADMIN_NODE} = TXK::Util::FALSE;
   }

  if ( $isWebNode eq "YES" )
   {
     $self->{$IS_WEB_NODE} = TXK::Util::TRUE;
   }
  else
   {
     $self->{$IS_WEB_NODE} = TXK::Util::FALSE;
   }

  if ( $isFormsNode eq "YES" )
   {
     $self->{$IS_FORMS_NODE} = TXK::Util::TRUE;
   }
  else
   {
     $self->{$IS_FORMS_NODE} = TXK::Util::FALSE;
   }

  if ( $isConcNode eq "YES" )
   {
     $self->{$IS_CONC_NODE} = TXK::Util::TRUE;
   }
  else
   {
     $self->{$IS_CONC_NODE} = TXK::Util::FALSE;
   }

}

#------------------------------------------------------------------------------
## txkIsWebNode()
#------------------------------------------------------------------------------

sub txkIsWebNode
{
  my $self = $ARG[0];

  if ( !defined ( $self->{$IS_WEB_NODE} ))
   {
     $self->_getNodeInfo();

   }

  return $self->{$IS_WEB_NODE};

}

#------------------------------------------------------------------------------
## txkIsFormsNode()
#------------------------------------------------------------------------------

sub txkIsFormsNode
{
  my $self = $ARG[0];

  if ( !defined ( $self->{$IS_FORMS_NODE} ))
   {
     $self->_getNodeInfo();

   }

  return $self->{$IS_FORMS_NODE};

}

#------------------------------------------------------------------------------
## txkIsAdminNode()
#------------------------------------------------------------------------------

sub txkIsAdminNode
{
  my $self = $ARG[0];

  if ( !defined ( $self->{$IS_ADMIN_NODE} ))
   {
     $self->_getNodeInfo();

   }

  return $self->{$IS_ADMIN_NODE};

}

#------------------------------------------------------------------------------
## txkIsDBNode()
#------------------------------------------------------------------------------

sub txkIsDBNode
{
  my $self = $ARG[0];

  if ( !defined ( $self->{$IS_DB_NODE} ))
   {
     $self->_getNodeInfo();

   }

  return $self->{$IS_DB_NODE};

}

#------------------------------------------------------------------------------
## txkIsConcNode()
#------------------------------------------------------------------------------

sub txkIsConcNode
{
  my $self = $ARG[0];

  if ( !defined ( $self->{$IS_CONC_NODE} ))
   {
     $self->_getNodeInfo();

   }

  return $self->{$IS_CONC_NODE};

}

#------------------------------------------------------------------------------
## txkIsCPJDKVersion131
#------------------------------------------------------------------------------

sub txkIsCPJDKVersion131
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $installed_jdk_version = $self->_getJDKVersionCP();

  #
  # JAVA command not found!!
  #

  if ( $installed_jdk_version eq JAVACMD_NOT_FOUND )
   {  
       return TXK::Error::FAIL;
   }

  #
  # JAVA cmd found and executed. Now get the version returned by the cmd.
  #

  if ( $installed_jdk_version =~ $JDK_13_PATTERN )
   {
       return TXK::Error::SUCCESS;
   }

  return TXK::Error::FAIL;
}

#------------------------------------------------------------------------------
## txkIsCPJDKVersion14
#------------------------------------------------------------------------------

sub txkIsCPJDKVersion14
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $installed_jdk_version = $self->_getJDKVersionCP();

  #
  # JAVA command not found!!
  #

  if ( $installed_jdk_version eq JAVACMD_NOT_FOUND )
   {
       return TXK::Error::FAIL;
   }

  #
  # JAVA cmd found and executed. Now get the version returned by the cmd.
  #

  if ( $installed_jdk_version =~ $JDK_14_PATTERN )
   {
       return TXK::Error::SUCCESS;
   }

  return TXK::Error::FAIL;
}

#------------------------------------------------------------------------------
## txkGetAppsCtxLocation()
#------------------------------------------------------------------------------

sub txkGetAppsCtxLocation
{
  my $self = $ARG[0];

  return $self->{$APPS_CTX_FILE};
}


#------------------------------------------------------------------------------
## _verifyConfigJDKWeb()
#------------------------------------------------------------------------------

sub _verifyConfigJDKWeb
{
  my $self = $ARG[0];
  my $tier = $ARG[1];

  my $ret_status = TXK::Error::SUCCESS;

  my $web_ret_status  = TXK::Error::SUCCESS;

  my $JDBC_FILE;

  my $CHARSETS_FILE;

  #
  # Verify if the Web node is configured with JDK version x
  #

  if ( $tier eq $IS_WEB_NODE )
   {
     my $web_home = $self->{$IAS_ORACLE_HOME};
     my $jserv_propPath;
     my $jserv_propFile;
     my $pattern_1;
     my $webHeaderPrinted = TXK::Util::FALSE;

     my $installed_jdk = $self->_getJDKVersionWeb();

     if ( $installed_jdk =~ $JDK_13_PATTERN )
      {
        $JDBC_FILE = $JDBC_FILE_12;
      }
     elsif ( $installed_jdk =~ $JDK_14_PATTERN )
      {
        $JDBC_FILE = $JDBC_FILE_14;
      }

     #-------------------------------------------------------------------------
     # 1. Determine if the appropriate JDBC zip file is in the CLASSPATH
     #    as defined in the jserv.properties.
     #-------------------------------------------------------------------------

     if ( TXK::OSD->isNT() )
      {
        $pattern_1 = "${JSERV_PATTERN_1}.*\\${JDBC_FILE}";
      }
     else
      {
        $pattern_1 = "${JSERV_PATTERN_1}.*\/${JDBC_FILE}";
      }

     my $jserv_propPath_f = $web_home . "/Apache/Jserv/etc/";

     $jserv_propPath = TXK::OSD->trDirPathToBase($jserv_propPath_f);

     $jserv_propFile = $jserv_propPath . "jserv.properties";

     my $io = TXK::IO->new();

     $io->open({ fileName => $jserv_propFile });

     my $io_ref = $io->getFileHandle();

     my @io_data = <$io_ref>;

     $io->close();

     if ( my @ret = grep(/$pattern_1/, @io_data ) )
      {
	$self->txkGetJavaTop;

        $web_ret_status = ($web_ret_status && TXK::Error::SUCCESS);

        $webHeaderPrinted = TXK::Util::TRUE;

        $self->{$VALIDATE_MESSAGE} .= "\n[PASS]\t The setting of the CLASSPATH environment in\n\t $jserv_propFile\n\t contains the following value:\n";
	foreach my $retline (@ret) {
	  $self->{$VALIDATE_MESSAGE} .= "\n\t ".$retline;
	}
      }
     else
      {
        $web_ret_status = ($web_ret_status && TXK::Error::FAIL);

        my $dt = TXK::OSD->trDirPathToBase($JDBC_FILE);

        $webHeaderPrinted = TXK::Util::TRUE;

        $self->{$VALIDATE_MESSAGE} .= "\n[FAIL]\t The setting of the CLASSPATH environment in\n\t $jserv_propFile\n\t does not contain the following value:\n\n\t wrapper.classpath=$self->{$JAVA_TOP}$dt\n";

      }

     #-------------------------------------------------------------------------
     # 2. Determine if the appropriate jar files are in the CLASSPATH
     #    as defined in the jserv.properties.
     #-------------------------------------------------------------------------

     my $pattern_2;
     my $jdk_top = $self->{$JDK_TOP_WEB};

     if ( TXK::OSD->isNT() )
      {
        $jdk_top =~ s/\\/\\\\/g;

      }

     $jdk_top = qr/$jdk_top/;

     if ( $installed_jdk =~ $JDK_13_PATTERN )
      {
        $CHARSETS_FILE = $JSERV_I18N_JAR;
      }
     elsif ( $installed_jdk =~ $JDK_14_PATTERN )
      {
        $CHARSETS_FILE = $JSERV_CHARSETS_JAR;
      }

     #
     # Determine if CLASSPATH contains dt.jar
     #

     $pattern_2 = "${JSERV_PATTERN_1}.*${jdk_top}.*${JSERV_DT_JAR}";

     if ( my @ret = grep(/$pattern_2/, @io_data ) )
      {
        $web_ret_status = ($web_ret_status && TXK::Error::SUCCESS);

        $self->{$VALIDATE_MESSAGE} .= "\n[PASS]\t The setting of the CLASSPATH environment in\n\t $jserv_propFile\n\t contains the following value:\n";
	foreach my $retline (@ret) {
	  $self->{$VALIDATE_MESSAGE} .= "\n\t ".$retline;
	}

      }
     else
      {
        $web_ret_status = ($web_ret_status && TXK::Error::FAIL);

        my $dt = TXK::OSD->trDirPathToBase("/lib/dt.jar");

        $self->{$VALIDATE_MESSAGE} .= "\n[FAIL]\t The setting of the CLASSPATH environment in\n\t $jserv_propFile\n\t does not contain the following value:\n\n\t wrapper.classpath=$self->{$JDK_TOP_WEB}$dt\n";

      }

     #
     # Determine if CLASSPATH contains tools.jar
     #

     $pattern_2 = "${JSERV_PATTERN_1}.*${jdk_top}.*${JSERV_TOOLS_JAR}";

     if ( my @ret = grep(/$pattern_2/, @io_data ) )
      {
        $web_ret_status = ($web_ret_status && TXK::Error::SUCCESS);

        $self->{$VALIDATE_MESSAGE} .= "\n[PASS]\t The setting of the CLASSPATH environment in\n\t $jserv_propFile\n\t contains the following value:\n";
	foreach my $retline (@ret) {
	  $self->{$VALIDATE_MESSAGE} .= "\n\t ".$retline;
	}

      }
     else
      {
        $web_ret_status = ($web_ret_status && TXK::Error::FAIL);

        my $tools_jar = TXK::OSD->trDirPathToBase("/lib/tools.jar");

        $self->{$VALIDATE_MESSAGE} .= "\n[FAIL]\t The setting of the CLASSPATH environment in\n\t $jserv_propFile\n\t does not contain the following value:\n\n\t wrapper.classpath=$self->{$JDK_TOP_WEB}$tools_jar\n";

      }

     #
     # Determine if CLASSPATH contains rt.jar
     #

     $pattern_2 = "${JSERV_PATTERN_1}.*${jdk_top}.*${JSERV_RT_JAR}";

     if ( my @ret = grep(/$pattern_2/, @io_data ) )
      {
        $web_ret_status = ($web_ret_status && TXK::Error::SUCCESS);

        $self->{$VALIDATE_MESSAGE} .= "\n[PASS]\t The setting of the CLASSPATH environment in\n\t $jserv_propFile\n\t contains the following value:\n";
	foreach my $retline (@ret) {
	  $self->{$VALIDATE_MESSAGE} .= "\n\t ".$retline;
	}

      }
     else
      {
        $web_ret_status = ($web_ret_status && TXK::Error::FAIL);

        my $rt = TXK::OSD->trDirPathToBase("/jre/lib/rt.jar");

        $self->{$VALIDATE_MESSAGE} .= "\n[FAIL]\t The setting of the CLASSPATH environment in\n\t $jserv_propFile\n\t does not contain the following value:\n\n\t wrapper.classpath=$self->{$JDK_TOP_WEB}$rt\n";

      }

     #
     # Determine if CLASSPATH contains charsets.jar or i18n.jar
     #

     $pattern_2 = "${JSERV_PATTERN_1}.*${jdk_top}.*${CHARSETS_FILE}";

     if ( my @ret = grep(/$pattern_2/, @io_data ) )
      {
        $web_ret_status = ($web_ret_status && TXK::Error::SUCCESS);

        $self->{$VALIDATE_MESSAGE} .= "\n[PASS]\t The setting of the CLASSPATH environment in\n\t $jserv_propFile\n\t contains the following value:\n";
	foreach my $retline (@ret) {
	  $self->{$VALIDATE_MESSAGE} .= "\n\t ".$retline;
	}

      }
     else
      {
        $web_ret_status = ($web_ret_status && TXK::Error::FAIL);

        my $charset;

        if ( $installed_jdk =~ $JDK_13_PATTERN )
         {
           $charset = TXK::OSD->trDirPathToBase("/jre/lib/i18n.jar");

         }
        elsif ( $installed_jdk =~ $JDK_14_PATTERN )
         {
	   $charset = TXK::OSD->trDirPathToBase("/jre/lib/charsets.jar");

         }

         $self->{$VALIDATE_MESSAGE} .= "\n[FAIL]\t The setting of the CLASSPATH environment in\n\t $jserv_propFile\n\t does not contain the following value:\n\n\t wrapper.classpath=$self->{$JDK_TOP_WEB}$charset\n";

      }
   }

  return $web_ret_status;

}



#------------------------------------------------------------------------------
## _verifyConfigJDKCP()
#------------------------------------------------------------------------------

sub _verifyConfigJDKCP
{
  my $self = $ARG[0];
  my $tier = $ARG[1];

  my $ret_status = TXK::Error::SUCCESS;

  my $web_ret_status  = TXK::Error::SUCCESS;
  my $conc_ret_status = TXK::Error::SUCCESS;

  my $JDBC_FILE;

  my $CHARSETS_FILE;

  if ( $tier eq $IS_CONC_NODE )
   {
     #
     #    Verify if the Conc node is configured with JDK version x
     #

     my $af_jvaprg     = TXK::OSD->getEnvVar({ name => "AFJVAPRG"});
     my $af_jre_top    = TXK::OSD->getEnvVar({ name => "AF_JRE_TOP"});
     my $af_classpath  = TXK::OSD->getEnvVar({ name => "AF_CLASSPATH"});

     my $pattern_1 = $self->{$JDK_TOP_CP};
     my $pattern_2 = $APPSBORG_FILE;

     my $ConcHeaderPrinted = TXK::Util::FALSE;

     if ( TXK::OSD->isNT() )
      {
        $pattern_1    =~ s/\\/\\\\/g;
        $pattern_2    =~ s/\\/\\\\/g;

      }

     if ( $af_jvaprg =~ qr/${pattern_1}/ )
      {
        $self->{$VALIDATE_MESSAGE} .= "\n[PASS]\t The current setting of AFJVAPRG is:\n\n\t $self->{$JDK_TOP_CP}\n";

        $conc_ret_status = ($conc_ret_status && TXK::Error::SUCCESS);

      }
     else
      {
        $self->{$VALIDATE_MESSAGE} .= "\n[FAIL]\t The current setting of AFJVAPRG is not:\n\n\t $self->{$JDK_TOP_CP}\n";

        $conc_ret_status = ($conc_ret_status && TXK::Error::FAIL);

      }

     if ( $af_jre_top =~ qr/${pattern_1}/ )
      {
        $self->{$VALIDATE_MESSAGE} .= "\n[PASS]\t The current setting of AF_JRE_TOP is:\n\n\t $self->{$JDK_TOP_CP}\n";

        $conc_ret_status = ($conc_ret_status && TXK::Error::SUCCESS);

      }
     else
      {
        $self->{$VALIDATE_MESSAGE} .= "\n[FAIL]\t The current setting of AF_JRE_TOP is not:\n\n\t $self->{$JDK_TOP_CP}\n";

        $conc_ret_status = ($conc_ret_status && TXK::Error::FAIL);

      }

     if ( $af_classpath =~ qr/${pattern_2}/ )
      {
        $self->{$VALIDATE_MESSAGE} .= "\n[PASS]\t The current setting of AF_CLASSPATH contains\n\t the following value:\n\n\t $pattern_2\n";

        $conc_ret_status = ($conc_ret_status && TXK::Error::SUCCESS);

      }
     else
      {
        $self->{$VALIDATE_MESSAGE} .= "\n[FAIL]\t The current setting of AF_CLASSPATH does not contain\n\t the following value:\n\n\t $pattern_2\n";

        $conc_ret_status = ($conc_ret_status && TXK::Error::FAIL);

      }

     my $installed_jdk_version = $self->_getJDKVersionCP();

     #
     # JAVA command not found!!
     #

     if ( $installed_jdk_version eq JAVACMD_NOT_FOUND )
      {
           return TXK::Error::FAIL;
      }

     #
     # JAVA cmd found and executed. Now get the version returned by the cmd.
     #

     if ( $installed_jdk_version =~ $JDK_13_PATTERN )
      {

        # Check for i18n.jar in AF_CLASSPATH

         if (  $af_classpath !~ $JSERV_I18N_JAR    ) 
          {
	      $conc_ret_status = ($conc_ret_status && TXK::Error::FAIL) ;    
	  }
         else
          {
	      $conc_ret_status = ($conc_ret_status && TXK::Error::SUCCESS);
	  }
      }
     elsif ( $installed_jdk_version =~ $JDK_14_PATTERN )
      {
        # Check for charsets.jar in AF_CLASSPATH

         if (  $af_classpath !~ $JSERV_CHARSETS_JAR    ) 
          {
	      $conc_ret_status = ($conc_ret_status && TXK::Error::FAIL) ;    
	  }
         else
          {
	      $conc_ret_status = ($conc_ret_status && TXK::Error::SUCCESS);
	  }
      }
    
     #------------------------------------------------------------------
     # check for dt.jar, tools.jar , rt.jar for both JDK 1.3 and JDK 1.4
     #------------------------------------------------------------------

     if ( ( $af_classpath !~ $JSERV_DT_JAR    ) or
          ( $af_classpath !~ $JSERV_TOOLS_JAR ) or
          ( $af_classpath !~ $JSERV_RT_JAR    ) )
      {
          $conc_ret_status = ($conc_ret_status && TXK::Error::FAIL);
      }
    else
     {
          $conc_ret_status = ($conc_ret_status && TXK::Error::SUCCESS);
     }

   }

  return $conc_ret_status;

}


#------------------------------------------------------------------------------
# _getCtxVersion
#------------------------------------------------------------------------------

sub _getCtxVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $fsys = TXK::FileSys->new();

  my @identData;
  my $rc      = NULL;

  if ( $args eq NULL )
   {
     return NULL;
   }

  $fsys->abortOnError ( { enable => TXK::Util::FALSE } );

  $rc = $fsys->ident({
                       fileName => $args,
                       identData => \@identData,
                    });

  if ( ( $rc eq TXK::Error::FAIL ) || ( $rc eq TXK::FileSys::NO_IDENT ) )
   {
     return NULL;
   }
  elsif ( $rc eq TXK::FileSys::MULTIPLE_IDENTS )
   {
     my $header;

     foreach $header (@identData)
      {
        $rc = $header->{TXK::FileSys::IDENT_VERSION};
      }

     return $rc;
   }
  else
   {
     return $rc;
   }

}

#------------------------------------------------------------------------------
## _isFileFound()
#------------------------------------------------------------------------------

sub _isFileFound
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $io;

  my $f        = NULL;
  my $status   = TXK::Util::FALSE;

  if ( $args eq NULL )
   {
     return TXK::Util::FALSE;
   }

  $f = TXK::OSD->trDirPathToBase($args);

  $io = TXK::IO->new();

  $io->abortOnError ({ enable => TXK::Util::FALSE });

  $status = $io->open ({
			 fileName	=> $f,
			 mode		=> TXK::IO::RDWR
		      });

  $io->abortOnError ({ enable => TXK::Util::TRUE });

  if ( $status eq TXK::Error::SUCCESS )
   {
     $io->close();
   }

  if ( $status eq TXK::Error::SUCCESS )
   {
     return TXK::Util::TRUE;
   }
  else
   {
     return TXK::Util::FALSE;
   }

}

#------------------------------------------------------------------------------
## txkPrintStatus()
#------------------------------------------------------------------------------

sub txkPrintStatus
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( $args eq $VALIDATE_MESSAGE )
   {
     print "$self->{$VALIDATE_MESSAGE}";
   }

}

#------------------------------------------------------------------------------
## _isPatchApplied()
#------------------------------------------------------------------------------

sub _verifyPatch
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $patch_status  = TXK::Error::SUCCESS;

  foreach my $patch ( @{$args} )
   {
     if ( $self->_isPatchApplied($patch) )
      {
	$patch_status = ($patch_status && TXK::Error::SUCCESS);

        $self->{$VALIDATE_MESSAGE} .= "\n[PASS]\t It appears that patch $patch was already applied.\n";

      }
     else
      {
        $patch_status = ($patch_status && TXK::Error::WARNING);

        $self->{$VALIDATE_MESSAGE} .= "\n[WARN]\t It appears that patch $patch has not been applied.\n";

      }

   }

  return $patch_status;

}

#------------------------------------------------------------------------------
## _isPatchApplied()
#------------------------------------------------------------------------------

sub _isPatchApplied
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $patch_applied = TXK::Util::FALSE;
  my $apps_user;
  if ( defined( $self->{$APPS_USER} ) )
    {
      $apps_user = $self->{$APPS_USER};
    }
  else
    {
      $apps_user = APPS_DEFAULT_USER;
    }

  my $apps_password;

  if ( defined( $self->{$APPS_PASSWORD} ) )
   {
     $apps_password = $self->{$APPS_PASSWORD};
   }
  else
   {
     $apps_password = APPS_DEFAULT_PWD;
   }

  my $appltmp  = $self->txkGetApplTmp($args);

  my $log_f    = $appltmp . "/" . "patch_info.txt";

  my $logfile  = TXK::OSD->trDirPathToBase($log_f);

  my $spath  = TXK::OSD->getEnvVar({ name => "FND_TOP"}) . "/patch/115/sql/";

  my $s_path = TXK::OSD->trDirPathToBase($spath);

  my $sql    = $s_path . "txkVerifyPatch.sql";

  my $node   = $self->{$HOST_NAME};

  my $sid    = $self->{$TWO_TASK};

  my $sqlctx = TXK::SQLPLUS->new();

  my $fsys   = TXK::FileSys->new();

  $fsys->access ({
                   fileName  => $sql,
                   type      => TXK::FileSys::FILE,
                   checkMode => TXK::FileSys::READ_ACCESS
                })
  or TXK::Error->stop("Check File(s) Error", $fsys->getError());

  $sqlctx->setConnectInfo({
                             user     => "$apps_user",
                             password => "$apps_password",
                             two_task => "$sid"
                         });

  $sqlctx->clearCommand();

  $sqlctx->addCommand("spool " . "$logfile");

  $sqlctx->addCommand("@" . "$sql " . "$args " . "$node");

  $sqlctx->execute({ showOutput =>TXK::Util::FALSE, });

  my $io = TXK::IO->new();

  $io->open({ fileName => $logfile });

  my $io_ref  = $io->getFileHandle();
  my @io_data = <$io_ref>;

  $io->close();

  unlink($logfile);

  if ( grep(/$args/, @io_data ) )
   {
     $patch_applied = TXK::Util::TRUE;

   }
  else
   {
     $patch_applied = TXK::Util::FALSE;

   }

  return $patch_applied;

}

#------------------------------------------------------------------------------
## _getAppsVersion()
#------------------------------------------------------------------------------

sub _getAppsVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $apps_user;
  if ( defined( $self->{$APPS_USER} ) )
    {
      $apps_user = $self->{$APPS_USER};
    }
  else
    {
      $apps_user = APPS_DEFAULT_USER;
    }

  my $apps_password;
  my $apps_version;
  my $pattern = qr/11\.5\.(\d)+/;

  if ( defined( $self->{$APPS_PASSWORD} ) )
   {
     $apps_password = $self->{$APPS_PASSWORD};
   }
  else
   {
     $apps_password = APPS_DEFAULT_PWD;
   }

  my $appltmp  = $self->txkGetApplTmp($args);

  my $log_f    = $appltmp . "/" . "apps_ver.txt";

  my $logfile   = TXK::OSD->trDirPathToBase($log_f);

  my $sid    = $self->{$TWO_TASK};

  my $sqlctx = TXK::SQLPLUS->new();

  $sqlctx->setConnectInfo({
                             user     => "$apps_user",
                             password => "$apps_password",
                             two_task => "$sid"
                         });

  $sqlctx->clearCommand();

  $sqlctx->addCommand("set heading off");
  $sqlctx->addCommand("set echo off");
  $sqlctx->addCommand("set verify off");

  $sqlctx->addCommand("spool " . "$logfile");

  $sqlctx->addCommand("select RELEASE_NAME from fnd_product_groups");

  $sqlctx->execute({ showOutput =>TXK::Util::FALSE, });

  my $io = TXK::IO->new();

  $io->open({ fileName => $logfile });

  my $io_ref  = $io->getFileHandle();
  my @io_data = <$io_ref>;

  $io->close();

  unlink($logfile);

  foreach (@io_data)
   {
     chomp;

     if ( /$pattern/ )
      {
        $apps_version = $_;

      }
   }

  return $apps_version;
}

#------------------------------------------------------------------------------
## txkGetAppsVersion()
#------------------------------------------------------------------------------

sub txkGetAppsVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( !defined($self->{$APPS_VERSION}) )
   {
     $self->{$APPS_VERSION} = $self->_getAppsVersion();

   }

  return $self->{$APPS_VERSION};

}

#------------------------------------------------------------------------------
# Destructor
#------------------------------------------------------------------------------

sub DESTROY
{
}

#------------------------------------------------------------------------------
# txkGetAppsCtxFile
#------------------------------------------------------------------------------

sub txkGetAppsCtxFile
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( !defined($self->{$ADCONFIG_ENABLED}) )
   {
     $self->txkIsAdConfigEnabled($args);

   }

  return $self->{$APPS_CTX_FILE};
}

#------------------------------------------------------------------------------
# txkIsFormsLsnrServlet
#------------------------------------------------------------------------------

sub txkIsFormsLsnrServlet
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( !defined($self->{$IS_FORMS_LSNR_SERVLET}) )
   {
     $self->_getFormsLsnrServlet($args);

   }

  return $self->{$IS_FORMS_LSNR_SERVLET};

}

#------------------------------------------------------------------------------
# _getFormsLsnrServlet
#------------------------------------------------------------------------------

sub _getFormsLsnrServlet
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( ! ( $self->txkIsAdConfigEnabled($args) ))
   {
     print "$AUTOCONFIG_MESSAGE\n";

     TXK::Error->stop("");

   }
  else
   {
     if ( ( $self->txkIsWebNode()   eq TXK::Util::TRUE ) &&
          ( $self->txkIsFormsNode() eq TXK::Util::TRUE )
        )
      {

        my $servlet_serverurl = $self->txkGetContextVar({ oavar => 's_forms_servlet_serverurl' });
        my $servlet_comment   = $self->txkGetContextVar({ oavar => 's_forms_servlet_comment' });

        if ( ( $servlet_serverurl =~ $SERVLET_SERVERURL ) &&
             ( $servlet_comment   =~ $SERVLET_COMMENT   )
           )
         {
           $self->{$IS_FORMS_LSNR_SERVLET} = TXK::Util::TRUE;

           $self->{$FORMS_CONFIGURATION} = TXK_FORMS_LSNR_SERVLET;
         }
        else
         {
	   $self->{$IS_FORMS_LSNR_SERVLET} = TXK::Util::FALSE;

         }
      }
   }

}

#------------------------------------------------------------------------------
# txkIsSSLConfigured
#------------------------------------------------------------------------------

sub txkIsSSLConfigured
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( !defined($self->{$IS_SSL}) )
   {
     $self->_getSSLConfig($args);

   }

  return $self->{$IS_SSL};

}

#------------------------------------------------------------------------------
# _getSSLConfig
#------------------------------------------------------------------------------

sub _getSSLConfig
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( ! ( $self->txkIsAdConfigEnabled($args) ))
   {
     print "$AUTOCONFIG_MESSAGE\n";

     TXK::Error->stop("");

   }

  my $webssl_port           = $self->txkGetContextVar({ oavar => 's_webssl_port' });
  my $active_webport        = $self->txkGetContextVar({ oavar => 's_active_webport' });
  my $login_page            = $self->txkGetContextVar({ oavar => 's_login_page' });
  my $url_protocol          = $self->txkGetContextVar({ oavar => 's_url_protocol' });
  my $local_url_protocol    = $self->txkGetContextVar({ oavar => 's_local_url_protocol' });
  my $webentryurl_protocol  = $self->txkGetContextVar({ oavar => 's_webentryurlprotocol' });
  my $web_ssl_keyfile       = $self->txkGetContextVar({ oavar => 's_web_ssl_keyfile' });
  my $web_ssl_certfile      = $self->txkGetContextVar({ oavar => 's_web_ssl_certfile' });
  my $web_ssl_certchainfile = $self->txkGetContextVar({ oavar => 's_web_ssl_certchainfile' });

  my $fsys_a = TXK::FileSys->new();

  my $fsys_b = TXK::FileSys->new();

  my $fsys_c = TXK::FileSys->new();

  my $status_a = $fsys_a->access({
                                   fileName   => $web_ssl_keyfile,
                                   type       => TXK::FileSys::FILE,
                                   checkMode  => TXK::FileSys::READ_ACCESS
                                });

  my $status_b = $fsys_b->access({
                                   fileName   => $web_ssl_certfile,
                                   type       => TXK::FileSys::FILE,
                                   checkMode  => TXK::FileSys::READ_ACCESS
                                });

  my $status_c = $fsys_c->access({
                                   fileName   => $web_ssl_certchainfile,
                                   type       => TXK::FileSys::FILE,
                                   checkMode  => TXK::FileSys::READ_ACCESS
                                });

  if ( ( $url_protocol         ne TXK_SSL_DEFAULT_PROTOCOL ) ||
       ( $local_url_protocol   ne TXK_SSL_DEFAULT_PROTOCOL ) ||
       ( $webentryurl_protocol ne TXK_SSL_DEFAULT_PROTOCOL ) ||
       ( $status_a eq TXK::Error::FAIL ) ||
       ( $status_b eq TXK::Error::FAIL ) ||
       ( $status_c eq TXK::Error::FAIL )
     )
   {
     $self->{$IS_SSL} = TXK::Util::FALSE;

   }
  else
   {
     $self->{$IS_SSL} = TXK::Util::TRUE;

     $self->{$WEBSSL_PORT} = $webssl_port;

     $self->{$WEB_SSL_KEYFILE} = $web_ssl_keyfile;

     $self->{$WEB_SSL_CERTFILE} = $web_ssl_certfile;

     $self->{$WEB_SSL_CERTCHAINFILE} = $web_ssl_certchainfile;

     $self->{$URL_PROTOCOL} = $url_protocol;

     $self->{$LOCAL_URL_PROTOCOL} = $local_url_protocol;

     $self->{$WEB_ENTRY_URL_PROTOCOL} = $webentryurl_protocol;

     $self->{$LOGIN_PAGE} = $login_page;

     $self->{$ACTIVE_WEBPORT} = $active_webport;

   }

}

#------------------------------------------------------------------------------
# txkPrintMsg
#------------------------------------------------------------------------------

sub txkPrintMsg
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  $self->{$PRINTMSG} = _setPrintMsg($args) if ( defined $args );

}

#------------------------------------------------------------------------------
# _setPrintMsg
#------------------------------------------------------------------------------

sub _setPrintMsg
{

  my $self  = $ARG[0];
  my $args  = $ARG[1];

  my $status;

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID,
                         args=>$args});

  $status = $args->{'enable'} if exists $args->{'enable'};

  $status = "true" unless ( $status eq "false" || $status eq "0" );
  $status = "0" if ( $status eq "false" );

  return $status;

}

#------------------------------------------------------------------------------
# txkIsPrintMsgOn
#------------------------------------------------------------------------------

sub txkIsPrintMsgOn
{
  my $self = shift;

  return $self->{$PRINTMSG};
}

#
# doit
#

#------------------------------------------------------------------------------
# txkSetError
#------------------------------------------------------------------------------

sub txkSetError
{
  my $self = $ARG[0];;
  my $args = $ARG[1];;

  my $ERROR_STATUS  = "error_status";
  my $ERROR_MESSAGE = "error_message";

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  TXK::Util->isValidArgs({args=>$args,reqd=>["$ERROR_STATUS"]});

  foreach my $key ( keys %$args )
   {
     if ( $key eq $ERROR_STATUS )
      {
        $self->{$CURRENT_ERROR} = $args->{$key};

      }
     elsif ( $key eq $ERROR_MESSAGE )
      {
        $self->{$CURRENT_ERROR_MSG} = $args->{$key};

      }
   }
}

#------------------------------------------------------------------------------
# txkGetError
#------------------------------------------------------------------------------

sub txkGetError
{
  my $self = $ARG[0];;
  my $args = $ARG[1];;

  my $ERROR_STATUS  = "error_status";
  my $ERROR_MESSAGE = "error_message";

  return $self->{$CURRENT_ERROR};

}

#------------------------------------------------------------------------------
# txkGetErrorMsg
#------------------------------------------------------------------------------

sub txkGetErrorMsg
{
  my $self = $ARG[0];;
  my $args = $ARG[1];;

  my $ERROR_STATUS  = "error_status";
  my $ERROR_MESSAGE = "error_message";

  return $self->{$CURRENT_ERROR_MSG};

}

#------------------------------------------------------------------------------
# txkPrintErrorMsg
#------------------------------------------------------------------------------

sub txkPrintErrorMsg
{
  my $self = $ARG[0];;
  my $args = $ARG[1];;

  my $ERROR_STATUS  = "error_status";
  my $ERROR_MESSAGE = "error_message";

  my $errorMsg = $self->txkGetErrorMsg();

  print "TXK_MSG:$errorMsg\n";

  TXK::Error->printMsg($errorMsg);

}

#------------------------------------------------------------------------------
# txkGetApplTop
#------------------------------------------------------------------------------

sub txkGetApplTop
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( !defined ( $self->{$APPL_TOP} ))
   {
     $self->_getApplTop($args);

   }

  return $self->{$APPL_TOP};

}

#-------------------------------------
# _getApplTop
#-------------------------------------

sub _getApplTop
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( ! ( $self->txkIsAdConfigEnabled($args) ))
   {
     #
     # Need to rely on environmental variable if not adconfig enabled.
     #

     $self->{$APPL_TOP} = TXK::OSD->getEnvVar({name => "APPL_TOP"});

   }
  else
   {
     $self->{$APPL_TOP} = $self->txkGetContextVar({ oavar => 's_at' });
   }
}

#------------------------------------------------------------------------------
# txkGetApplTmp
#------------------------------------------------------------------------------

sub txkGetApplTmp
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( !defined ( $self->{$APPLTMP} ))
   {
     $self->_getApplTmp($args);

   }

  return $self->{$APPLTMP};

}

#-------------------------------------
# _getApplTmp
#-------------------------------------

sub _getApplTmp
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( ! ( $self->txkIsAdConfigEnabled($args) ))
   {
     #
     # Need to rely on environmental variable if not adconfig enabled.
     #

     $self->{$APPLTMP} = TXK::OSD->getEnvVar({name => "APPLTMP"});

   }
  else
   {
     $self->{$APPLTMP} = $self->txkGetContextVar({ oavar => 's_appltmp' });
   }
}

#------------------------------------------------------------------------------
# txkGetJavaTop
#------------------------------------------------------------------------------

sub txkGetJavaTop
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( !defined ( $self->{$JAVA_TOP} ))
   {
     $self->_getJavaTop($args);

   }

  return $self->{$JAVA_TOP};

}

#-------------------------------------
# _getJavaTop
#-------------------------------------

sub _getJavaTop
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( ! ( $self->txkIsAdConfigEnabled($args) ))
   {
     #
     # Do nothing for now!
     #

     $self->txkSetError({
                          error_status  => TXK::Error::FAIL,
                          error_message => "Autoconfig is not enabled.",
                       });

     return TXK::Error::FAIL;

   }
  elsif ( ( ! ( $self->txkIsWebNode($args)  )) &&
          ( ! ( $self->txkIsConcNode($args) )) 
        )
   {
     #
     # Do nothing for now!
     #

     $self->txkSetError({
                          error_status   => TXK::Error::FAIL,
                          error_message => "Not a Web or Concurrent processing node.",
                       });

     return TXK::Error::WARNING;

   }
  else
   {
     $self->{$JAVA_TOP} = $self->txkGetContextVar({ oavar => 's_javatop' });
   }
}

#------------------------------------------------------------------------------
# txkGetToolsOracleHome
#------------------------------------------------------------------------------

sub txkGetToolsOracleHome
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( !defined ( $self->{$TOOLS_ORACLE_HOME} ))
   {
     $self->_getToolsOracleHome($args);

   }

  return $self->{$TOOLS_ORACLE_HOME};

}

#-------------------------------------
# _getToolsOracleHome
#-------------------------------------

sub _getToolsOracleHome
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( ! ( $self->txkIsAdConfigEnabled($args) ))
   {
     #
     # Do nothing for now!
     #

     $self->txkSetError({
                          error_status   => TXK::Error::FAIL,
                          error_message => "Autoconfig is not enabled.",
                       });

     return TXK::Error::FAIL;

   }
  else
   {
     $self->{$TOOLS_ORACLE_HOME} = $self->txkGetContextVar({ oavar => 's_tools_oh' });
   }
}


#------------------------------------------------------------------------------
# txkGetToolsOracleHomeVersion
#------------------------------------------------------------------------------

sub txkGetToolsOracleHomeVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( !defined ( $self->{$TOOLS_VERSION} ))
   {
     if ( TXK::OSD->isNT () )
      {
	  $self->_getToolsOracleHomeVersionNT($args);
      }
     else
      {
	  $self->_getToolsOracleHomeVersionUnix($args);
      }
   }

  return $self->{$TOOLS_VERSION};
}

#------------------------------------------------------------------------------
# _getToolsOracleHomeVersionNT
#------------------------------------------------------------------------------

sub _getToolsOracleHomeVersionNT
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $fsys = TXK::FileSys->new();

  my $io   = TXK::IO->new();

  my $ntrgs = TXK::OSD->trDirPathToBase ( $self->txkGetToolsOracleHome($args) . "/orainst/nt.rgs" );

  unless( $fsys->access ( { fileName  => $ntrgs,
			    checkMode => TXK::FileSys::READ_ACCESS } ) )
   {
     $self->txkSetError({
      	                error_status  => TXK::Error::FAIL,
       			error_message => "$ntrgs file does not exist.",
       		      });
     return TXK::Error::FAIL;
   }

  $io->open ( { fileName => $ntrgs,
	        mode     => TXK::IO::READ } );

  my $ioref = $io->getFileHandle();

  my @contents = <$ioref>;


  @contents = grep ( /34 w32rsf80/, @contents );
       
  $contents[0] =~ s/\s+/ /g;

  my @lines = split ( / /, $contents[0] );

  $self->{$TOOLS_VERSION} = $lines[5];

  $self->{$TOOLS_VERSION} =~ s/(^"|"$)//g;
       
  $io->close();

  return TXK::Error::SUCCESS;
}


#------------------------------------------------------------------------------
# _getToolsOracleHomeVersionUnix
#------------------------------------------------------------------------------

sub _getToolsOracleHomeVersionUnix
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $oracle_home = $self->txkGetToolsOracleHome($args);

  #
  # Construct the Forms Compiler path & executable
  #

  my $f60gen = $oracle_home . "/bin/" . "f60gen";

  my $appltmp  = $self->txkGetApplTmp($args);
  my $log_f    = $appltmp . "/" . "forms_ver.txt";
  my $logfile  = TXK::OSD->trDirPathToBase($log_f);

  my $process  = TXK::Process->new();
  $process->abortOnError( { enable => TXK::Util::FALSE } );
  my $run_status = $process->run({
				  command    => "$f60gen help=y"	,
				  stdout     => $logfile            ,
				 });

  if ( $run_status eq TXK::Error::FAIL )
    {
      $self->txkSetError({
			  error_status   => TXK::Error::FAIL,
			  error_message => "$f60gen not found",
			 });

      return TXK::Error::FAIL;
    }

  my $io = TXK::IO->new();
  $io->open({ fileName => $logfile });
  my $io_ref = $io->getFileHandle();
  my @io_data = <$io_ref>;
  $io->close();
  unlink($logfile);

  my $oh_version = "0";

  my @dbvers = grep(/$PLSQL_PATTERN/,@io_data);

  foreach ( @dbvers )
    {
      chomp;
      s/$PLSQL_PATTERN//;
      $self->{$TOOLS_VERSION} = $1;
    }

  if ($self->{$TOOLS_VERSION})
    {
      return $self->{$TOOLS_VERSION};
    }
  else
    {
      $self->txkSetError({
                          error_status   => TXK::Error::FAIL,
                          error_message => "Cannot determine PL/SQL version from f60gen.",
			 });

      return TXK::Error::FAIL;
    }
}


#-------------------------------------
# txkCheck8063Backport
#-------------------------------------
sub txkCheck8063Backport
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( !defined ( $self->{$HAS_8063_BACKPORT} ))
   {
     $self->_check8063Backport($args);
   }

  return $self->{$HAS_8063_BACKPORT};
}

#-------------------------------------
# txkGet8063VersionOnWindows
#-------------------------------------
sub txkGet8063VersionOnWindows
 {
  my $self = $ARG[0];
  my $args = $ARG[1];

  # method works only for Windows

  return TXK::Error::FAIL unless ( TXK::OSD->isWindows() );

  if ( !defined ( $self->{$WINDOWS_8063_VERSION} ))
   {
     $self->{$WINDOWS_8063_VERSION} = $self->_getRDBMSValueFromRGS($args);
   }

  return $self->{$WINDOWS_8063_VERSION};
 }


#------------------------------------------------------------------
# txkCheckXMLParser904 
#
# this checks whether the XML parser 904 is present in the $OA_JAVA/appsborg2.zip
#------------------------------------------------------------------

sub txkCheckXMLParser904
{
  my $self = $ARG[0];

  my $process = TXK::Process->new();    

  my $filesys = TXK::FileSys->new();

  my $appsborg2  = TXK::OSD->trDirPathToBase( $self->txkGetJavaTop() . "/appsborg2.zip" );

  $filesys->access ( { 'fileName' => $appsborg2,
		       'type'     => TXK::FileSys::FILE,
		       'checkMode'=> TXK::FileSys::READ_ACCESS } )
      or ( TXK::Error->printMsg( $filesys->getError() ) and  return TXK::Error::FAIL );

#  my $command    = TXK::OSD->trDirPathToBase( TXK::OSD->getEnvVar({ name => "AF_JRE_TOP"}) . "/bin/jar" );

  my $command    = TXK::OSD->trDirPathToBase($self->txkGetContextVar({ oavar => 's_jdktop' }) . "/bin/jar" );

  $command .= '.exe' if ( TXK::OSD->isNT() );

  my $tempfile   = TXK::OSD->trDirPathToBase( $self->txkGetApplTmp() ."/appsborg2.txt" );

  $process->run ( { 'command'    => { 'text' => $command,
				      'type' => TXK::Process::TRANSLATE_PATH },
		    'arg1'       => "-tvf",
		    'arg2'       => { 'text' => $appsborg2,
				      'type' => TXK::Process::TRANSLATE_PATH },
		    'stdout'     => $tempfile,
		    'showOutput' => TXK::Util::FALSE,
		    'showCommand'=> TXK::Util::TRUE } );

  $filesys->access ( { 'fileName' => $tempfile,
		       'type'     => TXK::FileSys::FILE,
		       'checkMode'=> TXK::FileSys::READ_ACCESS } )
      or ( TXK::Error->printMsg( $filesys->getError() ) and  return TXK::Error::FAIL );
 
  my $io = TXK::IO->new();

  $io->open( { 'fileName' => $tempfile,
	       'mode'     => TXK::IO::READ } );

  my $io_ref = $io->getFileHandle();

  my @contents = <$io_ref>;

  my $retVal = TXK::Util::FALSE;

  $retVal = TXK::Util::TRUE if ( grep(/$XMLPARSER904_PATTERN/,@contents) );
  
  $io->close();

  $filesys->rmfile ( {'fileName' => $tempfile } )
      or TXK::Error->printMsg( $filesys->getError() );

  return $retVal;
}

#-------------------------------------
# _check8063Backport
#-------------------------------------
sub _check8063Backport
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $status = 0;

  return TXK::Error::FAIL if ( TXK::OSD->isNT() );

  my $tempdir    = $self->txkGetApplTmp($args);
  my $oracle_home = $self->txkGetToolsOracleHome($args);
  my $sharedLib = TXK::OSD->trDirPathToBase($oracle_home . "/lib/" . "libclient.a");

  my $fileToTest = "kpuex.o";

  # Extract file using ar

  TXK::OSD->chdir($tempdir);
  my $process = TXK::Process->new();
  $process->run ( { command     => "ar",
		    arg1        => "-x",
		    arg2        => $sharedLib,
		    arg3        => $fileToTest,
		    showCommand => TXK::Util::FALSE,
		    showOutput  => TXK::Util::FALSE } );

  # List of checksums for file to test on each Unix platform. (We look in
  # Oracle Installer file for Windows.)

  my $CHECKSUM_8063 = { "filename"   => $fileToTest,
			"Linux"      => [ 3806454614  ] ,
			"Solaris"    => [ 2465321145 ] ,
			"UNIX_Alpha" => [ 228499987  ] ,
			"IBM_AIX"    => [ 3625318333 ] ,
			"HP_UX"      => [ 3600301261, 3493193040 ]
		      };

  $status = $self->_validateChecksum($CHECKSUM_8063);
  
  my $fsys = TXK::FileSys->new();
  
  $fsys->rmfile ( { fileName => $fileToTest } ) or TXK::Error->printMsg( $fsys->getError()); 

  if ($status)
    {
      $self->{$VALIDATE_MESSAGE} .= "\n[PASS]\t Backport 1227566 has been applied to your 8.0.6 ORACLE_HOME.\n";
      $self->{$HAS_8063_BACKPORT}=TXK::Error::SUCCESS;
    }
  else
    {
      if ( TXK::OSD->isNT() )
	{
	  $self->{$VALIDATE_MESSAGE} .= "\n[FAIL]\t Checksum of file kpuex.o indicates backport 1227566 has not\n\t been applied. 11.5.9 and later maintenance packs require\n\t you to apply the 8.0.6.3 backport for bug 1227566 to your\n\t 8.0.6 ORACLE_HOME.\n";
	}
      else
	{
	  $self->{$VALIDATE_MESSAGE} .= "\n[FAIL]\t RDBMS version in nt.rgs indicates backport 2845564 has not\n\t been applied. 11.5.9 and later maintenance packs require\n\t you to apply the 8.0.6.3 backport for bug 2845564 to your\n\t 8.0.6 ORACLE_HOME.\n";
	}

      $self->{$HAS_8063_BACKPORT}=TXK::Error::FAIL;
    }

  return $self->{$HAS_8063_BACKPORT};
}

#-------------------------------------
# _getValueFromRGS
# only works for Windows for 806 oracle home/orainst/nt.rgs 
#-------------------------------------

sub _getRDBMSValueFromRGS
 {
  my $self = $ARG[0];
  my $args = $ARG[1];

  return unless ( TXK::OSD->isWindows() );

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  my $oracle_home = $self->txkGetToolsOracleHome();
  my $rgsfile = TXK::OSD->trDirPathToBase($oracle_home . "/orainst/" . "nt.rgs");

  my $io = TXK::IO->new();
  $io->open({ fileName => $rgsfile });
  my $io_ref = $io->getFileHandle();
  my @io_data = <$io_ref>;
  $io->close();

  my $ntrdbmsver = "0";
  my @rgsdata = grep(/$DBMS_PATTERN_1/,@io_data);

  foreach ( @rgsdata )
    {
      chomp;
      s/$DBMS_PATTERN_1//;
      $ntrdbmsver = $1;
    }   

  return $ntrdbmsver;
 }

#-------------------------------------
# _validateChecksum (Unix only)
#-------------------------------------
sub _validateChecksum
{
  my $self = shift;
  my $cksumHash = shift;

  my $tempdir    = $self->txkGetApplTmp();

  my $outputFile = TXK::OSD->trDirPathToBase( $tempdir."/cksum.out");

  my $process = TXK::Process->new();
  $process->run ( { command     => "cksum",
		    arg1        => { text => $cksumHash->{'filename'},
				     type => TXK::Process::TRANSLATE_PATH },
		    stdout      => $outputFile,
		    showCommand => TXK::Util::FALSE,
		    showOutput  => TXK::Util::FALSE } );

  my $io = TXK::IO->new();
  $io->open ( { fileName => $outputFile,
		mode     => TXK::IO::READ } );
  my $fileHandle = $io->getFileHandle() ;
  my @fileContents = <$fileHandle>;
  $io->close();
  unlink($outputFile);

  my $line;
  my $platform=TXK::OSD->getAutoConfigName();
  foreach $line ( @fileContents )
    {
      chomp($line);

      my $cksum;
      ($cksum,,) = split(' ',$line );
      foreach (@{$cksumHash->{$platform}})
	{
	  if ( m/^$cksum$/ )
	    {
	      return TXK::Error::SUCCESS;
	    }
	}
    }

  return TXK::Error::FAIL;
}

#-----------------------------------------------------------------------------
# txkCheckiASExpiration
#-----------------------------------------------------------------------------

sub txkCheckiASExpiration
{
  my $self = $ARG[0];

  my ( $expYr, $expMon , $expDay );

  my ( $curYr, $curMon , $curDay );

  my $techstack = $self->txkGetTechstackVersion();

  if ( $techstack eq  $TECHSTACK_1022_VER )
  {
      # no expiration as of 07/04/2004

      return $techstack;
  }
 elsif ( $techstack eq  $TECHSTACK_1021_VER )
  {
      # expires on 01/31/2005

      $expYr  = "2005";
      $expMon = "0";   # for January
      $expDay = "31";

  }
 elsif ( $techstack eq  $TECHSTACK_1000_VER )
  {
      # expires on 10/31/2005

      $expYr  = "2004";
      $expMon = "9";   # for October
      $expDay = "31";
  }
 else
  {
      return $techstack;
  }

  ( $curDay , $curMon , $curYr ) = (localtime)[3..5];

  $curYr  += 1900;

  if ( $curYr > $expYr )
   {
       return -1 ; # expired
   }
  elsif ( ( $curYr == $expYr ) and ( $curMon > $expMon ) )
   {
       return -1 ; # expired
   }
  elsif ( ( $curYr == $expYr ) and ( $curMon == $expMon ) and ( $curDay > $expDay ) )
  {
      return -1;
  }

  return $techstack;
}

#------------------------------------------------------------------------------
# txkGetiASOracleHome
#------------------------------------------------------------------------------

sub txkGetIASOracleHome
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( !defined ( $self->{$IAS_ORACLE_HOME} ))
   {
     $self->_getIASOracleHome($args);

   }

  return $self->{$IAS_ORACLE_HOME};

}

#-------------------------------------
# _getIASOracleHome
#-------------------------------------

sub _getIASOracleHome
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  #----------------------------------------------------------------------------
  # If environmental variable is not set, get it from the Apps Context file.
  #----------------------------------------------------------------------------

  if ( ! ( $self->txkIsAdConfigEnabled($args) ))
   {
     #
     # Do nothing for now!
     #

     $self->txkSetError({
                          error_status  => TXK::Error::FAIL,
                          error_message => "Autoconfig is not enabled.",
                       });

     return TXK::Error::FAIL;

   }
  elsif ( ! ( $self->txkIsWebNode($args) ))
   {
     #
     # Do nothing for now!
     #

     $self->txkSetError({
                          error_status  => TXK::Error::WARNING,
                          error_message => "Not a Web node.",
                       });

     return TXK::Error::WARNING;

   }
  else
   {
     $self->{$IAS_ORACLE_HOME} = $self->txkGetContextVar({ oavar => 's_weboh_oh' });
   }
}

#------------------------------------------------------------------------------
# txkGetACTemplateRollupBugNo
# returns the rollup bugno.
#------------------------------------------------------------------------------


sub txkGetACTemplateRollupBugNo
 {
  my $self = $ARG[0];
  my $args = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  if (defined($self->{$AC_TEMP_ROLLUP_BUG}))
   {
     return $self->{$AC_TEMP_ROLLUP_BUG};
   }

  unless (defined($self->{$AC_TEMP_RUP_ALL_TXT}))
   {
     $self->_getACTemplateRollupFullText();
   }

  $self->{$AC_TEMP_ROLLUP_BUG} = $self->{$AC_TEMP_RUP_ALL_TXT};

  if ( $self->{$AC_TEMP_ROLLUP_BUG} =~ m/(Patch\#\s*\d*)/i )
   {
       $self->{$AC_TEMP_ROLLUP_BUG} = $1;
       $self->{$AC_TEMP_ROLLUP_BUG} =~ s/Patch\#//g;
   }
  else
   {
       $self->{$AC_TEMP_ROLLUP_BUG} = 'UNKNOWN_BUG_NO';     
   }

  return  $self->{$AC_TEMP_ROLLUP_BUG};

 }

#------------------------------------------------------------------------------
# txkGetACTemplateRollupBugtext
# returns the bug tex for the Rollup bug no.
#------------------------------------------------------------------------------


sub txkGetACTemplateRollupBugtext
 {
  my $self = $ARG[0];
  my $args = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  if (defined($self->{$AC_TEMP_ROLLUP_TXT}))
   {
     return $self->{$AC_TEMP_ROLLUP_TXT};
   }

  unless (defined($self->{$AC_TEMP_RUP_ALL_TXT}))
   {
     $self->_getACTemplateRollupFullText();
   }

  $self->{$AC_TEMP_ROLLUP_TXT} = $self->{$AC_TEMP_RUP_ALL_TXT};

  if ( $self->{$AC_TEMP_ROLLUP_TXT} =~ m/(-)/ )
   {
       $self->{$AC_TEMP_ROLLUP_TXT} = $';
   }
  else
   {
       $self->{$AC_TEMP_ROLLUP_TXT} = 'UNKNOWN_TEXT';     
   }

  return  $self->{$AC_TEMP_ROLLUP_TXT};

 }

#------------------------------------------------------------------------------
# txkGetACTemplateRollupVer
# returns 'Rollup E' , 'Rollup F' etc
#------------------------------------------------------------------------------

sub txkGetACTemplateRollupVer
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  if (defined($self->{$AC_TEMP_ROLLUP_VER}))
   {
     return $self->{$AC_TEMP_ROLLUP_VER};
   }

  unless (defined($self->{$AC_TEMP_RUP_ALL_TXT}))
   {
     $self->_getACTemplateRollupFullText();
   }

  $self->{$AC_TEMP_ROLLUP_VER} = $self->{$AC_TEMP_RUP_ALL_TXT};

  if ( $self->{$AC_TEMP_ROLLUP_VER} =~ m/(Rollup\s*Patch\s*[A-Za-z])/i )
   {
       $self->{$AC_TEMP_ROLLUP_VER} = $1;
       $self->{$AC_TEMP_ROLLUP_VER} =~ s/Patch//g;
   }
  else
   {
       $self->{$AC_TEMP_ROLLUP_VER} = 'ROLLUP_VERSION_NOT_DETERMINED';     
   }

  return  $self->{$AC_TEMP_ROLLUP_VER};

}

#------------------------------------------------------------------------------
# txkGetFndTemplateDriverVersion
# returns the fndtmpl.drv version
#------------------------------------------------------------------------------

sub txkGetFndTemplateDriverVersion
 {
  my $self = $ARG[0];
  my $args = $ARG[1];
  my @identData;

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  if (defined ($self->{$FND_TMPL_DRV}) )
   {
     return $self->{$FND_TMPL_DRV};
   }

  my $fsys = TXK::FileSys->new();
	
  $fsys->abortOnError ( { enable => TXK::Util::FALSE } );
  
  if ( !defined ( $self->{$FND_TOP} ))
    {
      $self->{$FND_TOP} = TXK::OSD->getEnvVar({ name => "FND_TOP"});
    }

  my $tmpl_driver = TXK::OSD->trDirPathToBase($self->{$FND_TOP}."/admin/driver/fndtmpl.drv");

	
  my $rc = $fsys->ident({
			 fileName => $tmpl_driver,
			 identData => \@identData,
			});

  if ( ( $rc eq TXK::Error::FAIL ) || ( $rc eq TXK::FileSys::NO_IDENT ) )
   {
       $self->{$FND_TMPL_DRV} = NULL;
   }

  $self->{$FND_TMPL_DRV} = $rc;

  return $self->{$FND_TMPL_DRV} ;
 }

#------------------------------------------------------------------------------
# _getACTemplateRollupFullText
# returns the full text for the rollup version
#------------------------------------------------------------------------------

sub _getACTemplateRollupFullText
 {
  my $self = $ARG[0];
  my $args = $ARG[1];
  my @identData;

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  if (defined($self->{$AC_TEMP_RUP_ALL_TXT}))
   {
     return $self->{$AC_TEMP_RUP_ALL_TXT};
   }

  my $tmpl_driver_version = $self->txkGetFndTemplateDriverVersion();

  my $rollup_xml_file = $self->{$FND_TOP}."/html/txkRollupVer.xml";
  
  # Fix for bug 4720383 - Load txkRollupVer.xml only if it exists
  if (-e $rollup_xml_file)
  {
    my $xmlobj = TXK::XML->new();
    $xmlobj->loadDocument( { file => TXK::OSD->trDirPathToBase($rollup_xml_file) });
    $self->{$AC_TEMP_RUP_ALL_TXT} = $xmlobj->getOAVar("$tmpl_driver_version");
  }
  else
  {
    print "DEBUG: $rollup_xml_file does not exist.\n";
  }

  if (! $self->{$AC_TEMP_RUP_ALL_TXT})
   {
     $self->{$AC_TEMP_RUP_ALL_TXT}=-1;
   }
  return $self->{$AC_TEMP_RUP_ALL_TXT};

 }

#-------------------------------------
# txkGetTechstackVersion
#-------------------------------------

sub txkGetTechstackVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( !defined ( $self->{$TECHSTACK_VERSION} ) )
   {
     $self->_getTechstackVersion($args);

   }

  return $self->{$TECHSTACK_VERSION};

}

#-------------------------------------
# _getTechstackVersion
#-------------------------------------

sub _getTechstackVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( ! ( $self->txkIsAdConfigEnabled($args) ))
   {
     #
     # Do nothing for now!
     #

     $self->txkSetError({
                          error_status  => TXK::Error::FAIL,
                          error_message => "Autoconfig is not enabled.",
                       });

     return TXK::Error::FAIL;

   }
  elsif ( ! ( $self->txkIsWebNode($args) ))
   {
     #
     # Do nothing for now!
     #

     $self->txkSetError({
                          error_status  => TXK::Error::WARNING,
                          error_message => "Not a Web node",
                       });

     return TXK::Error::WARNING;

   }
  else
   {
     my $ias = $self->txkGetContextVar({ oavar => 's_techstack' });

     if ( $ias =~ $TECHSTACK_1022 )
      {
	$self->{$TECHSTACK_VERSION} = $TECHSTACK_1022_VER;

      }
     elsif ( $ias =~ $TECHSTACK_1000 )
      {
	$self->{$TECHSTACK_VERSION} = $TECHSTACK_1000_VER;

      }
     elsif ( $ias =~ $TECHSTACK_1021 )
      {
	$self->{$TECHSTACK_VERSION} = $TECHSTACK_1021_VER;

      }
   }
}

#------------------------------------------------------
# txkGetOracleServerVerFromInv : gets the orale.server patch set from iAS orale home inventory
#------------------------------------------------------


sub txkGetOracleServerVerFromInv
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  return $self->_getOracleServerVerFromInv();

}

sub _getOracleServerVerFromInv
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  #---------------------------------------------------------------------------
  # this is needed now only for NT.
  # we will be getting  the version from a string like this
  #
  #
  #      <PATCHSET NAME="oracle.server" VER="8.1.7.4.1" BUILD_NUMBER="0" RELEASE="Production" INV_LOC="PatchSets/oracle.server/8.1.7.4.1/1/" XML_INV_LOC="PatchSets21/oracle.server/8.1.7.4.1/" ACT_INST_VER="1.7.0.0.0" DEINST_VER="1.7.0.0.0" INSTALL_TIME="2004.Aug.23 16:48:45 PDT">
  #
  #
  #----------------------------------------------------------------------------

  return TXK::Error::FAIL if ( TXK::OSD->isUNIX() );

  my $ias_oracle_home = $self->txkGetIASOracleHome($args);

  my $inv_file = TXK::OSD->trDirPathToBase($ias_oracle_home. "/inventory/ContentsXML/comps.xml");

  my $io   = TXK::IO->new();
  $io->open({fileName => $inv_file});
  my $io_ref = $io->getFileHandle();
  my @io_data = <$io_ref>;
  $io->close();
  
  my @version_string= grep (/PATCHSET NAME="oracle.server"/,@io_data);

  unless ( scalar(@version_string) > 0 )
   {
       #--------------------------------------------------
       # if there is no patch set applied look for COMP NAME
       #
       #       <COMP NAME="oracle.server" VER="8.1.7.0.0" BUILD_NUMBER="0" REP_VER="0.0.0.0.0" RELEASE="Production" INV_LOC="Components/oracle.server/8.1.7.0.0/1/" LANGS="en" XML_INV_LOC="Components21/oracle.server/8.1.7.0.0/" ACT_INST_VER="2.1.0.10.0" DEINST_VER="2.1.0.6.0" INSTALL_TIME="2002.Nov.08 12:56:01 PST" INST_LOC="E:\ab\apps\p25ora\iAS\oracle.server">
       #
       #---------------------------------------------------

       @version_string= grep (/COMP NAME="oracle.server"/,@io_data);
   }

    @version_string = split(/\s+/,$version_string[0]);

    my $version;

    foreach $version ( @version_string )
     {
	 if ( $version =~ /^VER=/ )
	  {
	      $version =~ s/(VER="|")//g;

	      return $version;
	  }
     }

  return TXK::Error::FAIL;
}



#-------------------------------------
# txkGetRSFVersionfor iAS Oracle Home
#-------------------------------------

sub txkGetRSFVersionForIASOracleHome
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  return $self->_getRSFVersionForIASOracleHome();
}

sub _getRSFVersionForIASOracleHome
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  #-----------------------------------------------------------------------
  # true only on UNIX platforms, since on NT the inventory is not updated
  #------------------------------------------------------------------------

  return TXK::Error::FAIL if ( TXK::OSD->isNT() );

  my $ias_oracle_home = $self->txkGetIASOracleHome($args);

  my $inv_file = TXK::OSD->trDirPathToBase($ias_oracle_home. "/inventory/ContentsXML/comps.xml");

  my $io   = TXK::IO->new();
  $io->open({fileName => $inv_file});
  my $io_ref = $io->getFileHandle();
  my @io_data = <$io_ref>;
  $io->close();
  
  my @version_string= grep (/oracle.rsf.clntsh_rsf/,@io_data);
  
  my $search_string = "PS_PATCH NAME";
  
  unless ( scalar ( grep (/$search_string/,@version_string) ) > 0 )
   {
       $search_string = "COMP NAME";
   }

    my @version_temp_string= grep(/$search_string/,@version_string);

    @version_temp_string = split(/\s+/,$version_temp_string[0]);

    my $version;

    foreach $version ( @version_temp_string )
     {
	 if ( $version =~ /^VER=/ )
	  {
	      $version =~ s/(VER="|")//g;

	      return $version;
	  }
     }

  return TXK::Error::FAIL;
}

#-------------------------------------
# txkGetJinitVersion
#-------------------------------------

sub txkGetJinitVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( !defined ( $self->{$JINIT_VERSION} ) )
   {
     $self->_getJinitVersion($args);

   }

  return $self->{$JINIT_VERSION};

}

#-------------------------------------
# _getJinitVersion
#-------------------------------------

sub _getJinitVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( ! ( $self->txkIsAdConfigEnabled($args) ))
   {
     #
     # Do nothing for now!
     #

     $self->txkSetError({
                          error_status  => TXK::Error::FAIL,
                          error_message => "Autoconfig is not enabled.",
                       });

     return TXK::Error::FAIL;

   }
  elsif ( ! ( $self->txkIsWebNode($args) ))
   {
     #
     # Do nothing for now!
     #

     $self->txkSetError({
                          error_status  => TXK::Error::WARNING,
                          error_message => "Not a Web node.",
                       });

     return TXK::Error::WARNING;

   }
  else
   {
     $self->{$JINIT_VERSION} = $self->txkGetContextVar({ oavar => 's_jinit_ver_dot' });
   }
}

#-------------------------------------
# txkGetDiscoVersion
#-------------------------------------

sub txkGetDiscoVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( !defined ( $self->{$DISCO_VERSION} ) )
   {
     $self->_getDiscoVersion($args);

   }

  return $self->{$DISCO_VERSION};

}

#-------------------------------------
# _getDiscoVersion
#-------------------------------------

sub _getDiscoVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( ! ( $self->txkIsAdConfigEnabled($args) ))
   {
     #
     # Do nothing for now!
     #

     $self->txkSetError({
                          error_status  => TXK::Error::FAIL,
                          error_message => "Autoconfig is not enabled.",
                       });

     return TXK::Error::FAIL;

   }
  elsif ( ! ( $self->txkIsWebNode($args) ))
   {
     #
     # Do nothing for now!
     #

     $self->txkSetError({
                          error_status  => TXK::Error::WARNING,
                          error_message => "Not a Web node",
                       });

     return TXK::Error::WARNING;

   }
  else
   {
     $self->{$DISCO_VERSION} = $self->txkGetContextVar({ oavar => 's_disco_ver_comma' });
     $self->{$DISCO_VERSION} =~ s/,/\./g;

   }
}

#-------------------------------------
# txkGetJDKTop
#-------------------------------------

sub txkGetJDKTop
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( ($args->{'NODE_TEST'} eq 'conc') ||
       ( $self->txkIsConcNode() &&
	 (! $self->txkIsWebNode()) ) )
    {
      if ( !defined ( $self->{$JDK_TOP_CP} ) )
	{
	  $self->_getJDKTopCP($args);
	}
      else
	{
	  return $self->{$JDK_TOP_CP};
	}
    }
  else
    {
      if ( !defined ( $self->{$JDK_TOP_WEB} ) )
	{
	  $self->_getJDKTopWeb($args);
	}
      else
	{
	  return $self->{$JDK_TOP_WEB};
	}
    }
}


#-------------------------------------
# _getJDKTopWeb
#-------------------------------------

sub _getJDKTopWeb
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( ! ( $self->txkIsAdConfigEnabled($args) ))
   {
     #
     # @@ Do nothing for now!
     #

     $self->txkSetError({
                          error_status  => TXK::Error::FAIL,
                          error_message => "Autoconfig is not enabled.",
                       });

     return TXK::Error::FAIL;

   }
  elsif ( ! $self->txkIsWebNode($args) )
   {
     #
     # @@ Do nothing for now!
     #

     $self->txkSetError({
                          error_status  => TXK::Error::WARNING,
                          error_message => "Not a Web or node.",
                       });

     return TXK::Error::WARNING;

   }
  else
   {
     $self->{$JDK_TOP_WEB} = $self->txkGetContextVar({ oavar => 's_jdktop' });
   }
}

#-------------------------------------
# _getJDKTopCP
#-------------------------------------

sub _getJDKTopCP
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( ! ( $self->txkIsAdConfigEnabled($args) ))
   {
     #
     # @@ Do nothing for now!
     #

     $self->txkSetError({
                          error_status  => TXK::Error::FAIL,
                          error_message => "Autoconfig is not enabled.",
                       });

     return TXK::Error::FAIL;

   }
  elsif ( ( ! ( $self->txkIsWebNode($args) )) &&
          ( ! ( $self->txkIsConcNode($args) ))
        )
   {
     #
     # @@ Do nothing for now!
     #

     $self->txkSetError({
                          error_status  => TXK::Error::WARNING,
                          error_message => "Not a Concurrent Processing node.",
                       });

     return TXK::Error::WARNING;

   }
  else
   {
     $self->{$JDK_TOP_CP} = $self->txkGetContextVar({ oavar => 's_afjretop' });
   }
}


#-------------------------------------
# txkGetOHSTop
#-------------------------------------

sub txkGetOHSTop
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( !defined ( $self->{$OHS_TOP} ) )
   {
     $self->_getOHSTop($args);

   }

  return $self->{$OHS_TOP};

}

#-------------------------------------
# _getOHSTop
#-------------------------------------

sub _getOHSTop
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( ! ( $self->txkIsAdConfigEnabled($args) ))
   {
     #
     # @@ Do nothing for now!
     #

     $self->txkSetError({
                          error_status  => TXK::Error::FAIL,
                          error_message => "Autoconfig is not enabled.",
                       });

     return TXK::Error::FAIL;

   }
  elsif ( ! ( $self->txkIsWebNode($args) ))
   {
     #
     # @@ Do nothing for now!
     #

     $self->txkSetError({
                          error_status  => TXK::Error::WARNING,
                          error_message => "Not a Web node",
                       });

     return TXK::Error::WARNING;

   }
  else
   {
     $self->{$OHS_TOP} = $self->txkGetContextVar({ oavar => 's_tp' });
   }
}

#-------------------------------------
# txkGetOHSVersion
#-------------------------------------

sub txkGetOHSVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( !defined ( $self->{$OHS_VERSION} ) )
   {
     $self->_getOHSVersion($args);

   }

  return $self->{$OHS_VERSION};

}

#-------------------------------------
# _getOHSVersion
#-------------------------------------

sub _getOHSVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( ! ( $self->txkIsAdConfigEnabled($args) ))
   {
     #
     # @@ Do nothing for now!
     #

     $self->txkSetError({
                          error_status  => TXK::Error::FAIL,
                          error_message => "Autoconfig is not enabled.",
                       });

     return TXK::Error::FAIL;

   }
  elsif ( ! ( $self->txkIsWebNode($args) ))
   {
     #
     # @@ Do nothing for now!
     #

     $self->txkSetError({
                          error_status  => TXK::Error::WARNING,
                          error_message => "Not a Web node.",
                       });

     return TXK::Error::WARNING;

   }
  else
   {
     my $s_tp = $self->txkGetOHSTop($args);

     my $ohs_version = "0";

     my $httpd_exe;

     my $httpd_path;

     if ( TXK::OSD->isNT() )
      {
	$httpd_exe = "Apache.exe";

        $httpd_path = $s_tp . "/Apache/Apache/" . $httpd_exe;

      }
     else
      {
	$httpd_exe = "httpd";

	$httpd_path = $s_tp . "/Apache/Apache/bin/" . $httpd_exe;

      }

     my $httpd_cmd = TXK::OSD->trDirPathToBase($httpd_path);

     my $appltmp  = $self->txkGetApplTmp($args);

     my $log_f    = $appltmp . "/" . "ohs_ver.txt";

     my $logfile   = TXK::OSD->trDirPathToBase($log_f);

     my $process   = TXK::Process->new();

     $process->abortOnError( { enable => TXK::Util::FALSE } );

     my $run_status = $process->run({
                                      command    => "$httpd_cmd",
	                              arg1       => "-version",   
                                      stdout     => $logfile 
                                   });

     #---------------------------------------------------------------
     # On NT the return code for the above command is 256 so
     # it is returning a FAIL . We should check if the stdout file
     # exists then the process is successful otherwise it is a FAIL 
     #---------------------------------------------------------------

     #if ( $run_status eq TXK::Error::FAIL )
     # {
     #   print "\n[FAIL]\t $httpd_cmd not found!!!\n\n";

     #   TXK::Error->stop("");

     # }

     my $fsys = TXK::FileSys->new();

     $fsys->access( { fileName => $logfile,
	              checkMode=> TXK::FileSys::READ_ACCESS,
	              type     => TXK::FileSys::FILE } )
        or ( TXK::Error->stop( $process->getError() ));

     my $io = TXK::IO->new();

     $io->open({ fileName => $logfile });

     my $io_ref = $io->getFileHandle();
     my @io_data = <$io_ref>;

     $io->close();

     unlink($logfile);

     unless ( grep(/$OHS_PATTERN_1/,@io_data) )
      {
        return $ohs_version;

      }

     my @ohsobj = grep(/$OHS_PATTERN_1/,@io_data);

     foreach ( @ohsobj )
      {
        chomp;
        s/$OHS_PATTERN_1//;
	s/$OHS_PATTERN_2//;

        $self->{$OHS_VERSION} = $_;
      }

     return $self->{$OHS_VERSION};

   }
}

#-------------------------------------
# txkGetIASpatchset
#-------------------------------------

sub txkGetIASpatchset
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( !defined ( $self->{$IAS_PATCHSET} ) )
   {
     $self->_getIASpatchset($args);

   }

  return $self->{$IAS_PATCHSET};

}

#-------------------------------------
# _getIASpatchset
#-------------------------------------

sub _getIASpatchset
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( ! ( $self->txkIsAdConfigEnabled($args) ))
   {
     #
     # @@ Do nothing for now!
     #

     $self->txkSetError({
                          error_status  => TXK::Error::FAIL,
                          error_message => "Autoconfig is not enabled.",
                       });

     return TXK::Error::FAIL;

   }
  elsif ( ! ( $self->txkIsWebNode($args) ))
   {
     #
     # @@ Do nothing for now!
     #

     $self->txkSetError({
                          error_status  => TXK::Error::WARNING,
                          error_message => "Not a Web node.",
                       });

     return TXK::Error::WARNING;

   }
  else
   {
     if ( $self->txkGetTechstackVersion($args) =~ $TECHSTACK_1022 )
      {
        my $s_tp = $self->txkGetIASOracleHome($args);

        my $ias_ps = "0";

        my $httpd_exe;

        my $httpd_path;

        if ( TXK::OSD->isNT() )
         {
   	$httpd_exe = "Apache.exe";

           $httpd_path = $s_tp . "/Apache/Apache/" . $httpd_exe;

         }
        else
         {
   	$httpd_exe = "httpd";

   	$httpd_path = $s_tp . "/Apache/Apache/bin/" . $httpd_exe;

         }

        my $httpd_cmd = TXK::OSD->trDirPathToBase($httpd_path);

        my $appltmp  = $self->txkGetApplTmp($args);

        my $log_f     = $appltmp . "/" . "ohs_ver.txt";

        my $logfile   = TXK::OSD->trDirPathToBase($log_f);

        my $process   = TXK::Process->new();

        $process->abortOnError( { enable => TXK::Util::FALSE } );

        my $run_status = $process->run({
                                         command    => "$httpd_cmd -version ",
                                         stdout     => $logfile             ,
                                      });


     #---------------------------------------------------------------
     # On NT the return code for the above command is 256 so
     # it is returning a FAIL . We should check if the stdout file
     # exists then the process is successful otherwise it is a FAIL 
     #---------------------------------------------------------------

     #if ( $run_status eq TXK::Error::FAIL )
     # {
     #   print "\n[FAIL]\t $httpd_cmd not found!!!\n\n";

     #   TXK::Error->stop("");

     # }

     my $fsys = TXK::FileSys->new();

     $fsys->access( { fileName => $logfile,
	              checkMode=> TXK::FileSys::READ_ACCESS,
	              type     => TXK::FileSys::FILE } )
        or ( TXK::Error->stop( $process->getError() ) );

        my $io = TXK::IO->new();

        $io->open({ fileName => $logfile });

        my $io_ref = $io->getFileHandle();
        my @io_data = <$io_ref>;

        $io->close();

        unlink($logfile);

        unless ( grep(/$OHS_PATTERN_3/,@io_data) )
         {
	     $self->{$IAS_PATCHSET} = $self->_getIASPatchSetFromInv();

             #return $ias_ps;
             return $self->{$IAS_PATCHSET};
         }

        my @ohsobj = grep(/$OHS_PATTERN_3/,@io_data);

        foreach ( @ohsobj )
         {
           chomp;
           s/$OHS_PATTERN_3//;
   	s/$OHS_PATTERN_4//;

           $self->{$IAS_PATCHSET} = $_;
         }

        return $self->{$IAS_PATCHSET};
      }
     else
      {
	#
	# So it is not iAS 1.0.2.2.2. Not necessary to check for patchset.
	#
      }
   }
}

#-------------------------------------
# _getIASPatchSetFromInv()
#-------------------------------------

sub _getIASPatchSetFromInv
{
    my $self = $ARG[0];

    my $invFile = TXK::OSD->trDirPathToBase( $self->txkGetIASOracleHome()."/inventory/ContentsXML/comps.xml");

    my $fsys = TXK::FileSys->new();

    my $process = TXK::Process->new();

    $fsys->access ( { fileName => $invFile,
		      type     => TXK::FileSys::FILE,
		      checkMode=> TXK::FileSys::READ_ACCESS })
	or TXK::Error->printMsg($fsys->getError());

    my $io = TXK::IO->new();

    $io->open( {fileName => $invFile,
		mode     => TXK::IO::READ });

    my $fileRef = $io->getFileHandle();

    my @contents=<$fileRef>;

    my @patchSetLines = grep ( /PS_PATCH NAME/, @contents );

    my @invLine  = grep ( /$OHS_PATTERN_5/, @patchSetLines );

    $invLine[0] =~ s/$OHS_PATTERN_6//g;

    $invLine[0] =~ s/(^"|"$)//g;

    $invLine[0] =~ s/\s*//g;

    $io->close();

    return $invLine[0];
}

#-------------------------------------
# txkGetOAVersionIdent
#-------------------------------------
sub txkGetOAVersionIdent
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( !defined ( $self->{$OA_VERSION_IDENT} ) )
   {
     $self->_getOAVersionIdent($args);

   }

  return $self->{$OA_VERSION_IDENT};

}

#-------------------------------------
# txkCheckForSymLinks
#-------------------------------------

sub txkCheckForSymLinks
{
    my $self = $ARG[0];

    TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});    

    return TXK::Util::FALSE if ( TXK::OSD->isNT() );

    return (scalar (keys %{$self->{$SYM_LINK_REPOS}} ) ) if ( defined $self->{$SYM_LINK_REPOS} );

    $self->{$SYM_LINK_REPOS} = {};

    my $args = $ARG[1];

    my ( $ctxVar, $dir );

    foreach $ctxVar ( @$CONTEXT_VARS_TO_CHECK_FOR_SYM_LINK )
     {
	 $dir = $self->txkGetContextVar({oavar=>$ctxVar});

	 next unless ( defined $dir );

	 # both of these methods populate $self->{$SYM_LINK_REPOS} hash

	 $self->_countSymLinksInPath($dir);

	 $self->_countSymLinks($dir);
     }

    return (scalar (keys %{$self->{$SYM_LINK_REPOS}} ) );
}

#------------------------------------
#
#-------------------------------------

sub txkCheckOptionsSymLink
{
    my $self = $ARG[0];

    TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});    

    my $noOfSymLinks = $self->txkCheckForSymLinks();

    #-------------------------------------------------------------------------
    # see if the %s_options_symlinks% contains value of "Options -FollowSymLinks"
    # if the value is indeed "Options FollowSymLinks" then return a negative number.
    #-------------------------------------------------------------------------

    my $options_symlink = $self->txkGetContextVar({oavar=>'s_options_symlinks'});

    $options_symlink =~ s/(^\s|\s$)//g;

    if ( $noOfSymLinks == 0 ) 
     {
	 return TXK::Error::SUCCESS;
     }

    if ( $options_symlink !~ m/^Options\s+-FollowSymLinks$/i ) 
     {
	 return TXK::Error::WARNING ;
     }

   return $TXK::Error::FAIL;
}

#-------------------------------------
# _countSymLinksInPath
#-------------------------------------

sub _countSymLinksInPath
{
    my $self = $ARG[0];

    my $dir = $ARG[1];

    my @baseDir = split( '/' , $dir );

    my $fsys = TXK::FileSys->new();

    my $index = 1; # since zeroth element is blank always

    if ( scalar(@baseDir) > 0 )
     {
	 my $dirToTest =  "/".$baseDir[$index] ;
	 
	 while ( $index++ < scalar(@baseDir) )
	  {
	      if ( $fsys->isSymLink({fileName=>$dirToTest}) )
	       {
		   unless ( exists $self->{$SYM_LINK_REPOS}->{$dirToTest} )
		    {
			$self->{$SYM_LINK_REPOS}->{$dirToTest} = "Dir_Link";

			print ( " $dirToTest is a symbolic link\n");
		    }
	       }
		   
	      $dirToTest .= "/".$baseDir[$index];
	  }
     }			   

    return  TXK::Error::SUCCESS;
}

#-------------------------------------
# _countSymLinks
#-------------------------------------

sub _countSymLinks
{
    my $self = $ARG[0];

    my $dir = $ARG[1];

    my $fsys = TXK::FileSys->new();

    if ( $fsys->isDirectory({dirName=>$dir}) )
     {
	 my @fileContents = $fsys->getDirList({dirName=>$dir});

	 my $file;
	 
	 foreach $file ( @fileContents )
	  {
	      $file = $dir ."/". $file;

	      $self->_countSymLinks($file) if ( $fsys->isDirectory({dirName=>$file}) );
	      
	      if ( $fsys->isSymLink({fileName=>$file}) )
	       {
		   unless ( exists $self->{$SYM_LINK_REPOS}->{$file} )
		    {
			$self->{$SYM_LINK_REPOS}->{$file} = "File_Link";

			print ( " $file is a symbolic link\n");
		    }
	       }
	  }
     }

    return TXK::Error::SUCCESS;
}


#-------------------------------------
# _getOAVersionIdent
#-------------------------------------
sub _getOAVersionIdent
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( ! ( $self->txkIsAdConfigEnabled($args) ))
   {
     #
     # @@ Do nothing for now!
     #

     $self->txkSetError({
                          error_status  => TXK::Error::FAIL,
                          error_message => "Autoconfig is not enabled.",
                       });

     return TXK::Error::FAIL;


   }
  if ( ! ( $self->txkIsWebNode($args) ))
   {
     #
     # @@ Do nothing for now!
     #

     $self->txkSetError({
                          error_status  => TXK::Error::WARNING,
                          error_message => "Not a Web node.",
                       });

     return TXK::Error::WARNING;

   }

  my $s_fndtop = $self->txkGetContextVar({ oavar => 's_fndtop' });

  my $oa_path = $s_fndtop . "/html/" . "OA.jsp";
  my $oa_jsp = TXK::OSD->trDirPathToBase($oa_path);

  my $fsys = TXK::FileSys->new();
  my @identData;
  my $rc = "0";
  $fsys->abortOnError ( { enable => TXK::Util::FALSE } );

  $rc = $fsys->ident({
		      fileName => $oa_jsp,
		      identData => \@identData,
		     });

  if ( ( $rc eq TXK::Error::FAIL ) || ( $rc eq TXK::FileSys::NO_IDENT ) )
    {
      print "\n[FAIL]\t Unable to determine the version of:\n";
      print "\n\t $oa_jsp\n\n";
      TXK::Error->stop("");
    }

  elsif ( $rc eq TXK::FileSys::MULTIPLE_IDENTS )
    {
      my $header;

      foreach $header (@identData)
	{
	  $rc = $header->{TXK::FileSys::IDENT_VERSION};
	}
      }

  $self->{$OA_VERSION_IDENT} = $rc;
  return $self->{$OA_VERSION_IDENT};
}


#-------------------------------------
# txkGetOAVersion
#-------------------------------------

sub txkGetOAVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( !defined ( $self->{$OA_VERSION_IDENT} ) )
   {
     $self->_getOAVersionIdent($args);

   }

  if ( !defined ( $self->{$OA_VERSION} ) )
   {
     $self->_getOAVersion($args);

   }

  return $self->{$OA_VERSION};

}

#-------------------------------------
# _getOAVersion
#-------------------------------------

sub _getOAVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( ! ( $self->txkIsAdConfigEnabled($args) ))
   {
     #
     # @@ Do nothing for now!
     #

     $self->txkSetError({
                          error_status  => TXK::Error::FAIL,
                          error_message => "Autoconfig is not enabled.",
                       });

     return TXK::Error::FAIL;

   }

  if ( ! ( $self->txkIsWebNode($args) ))
   {
     #
     # @@ Do nothing for now!
     #

     $self->txkSetError({
                          error_status  => TXK::Error::WARNING,
                          error_message => "Not a Web node.",
                       });

     return TXK::Error::WARNING;

   }

  my $ident = $self->{$OA_VERSION_IDENT};

  if ( $ident =~ $OAJSP_26 )
    {
      $self->{$OA_VERSION} = $OAFWK_552;
      $self->{$OA_VERSION} =~ s/\s*//;
    }
  elsif ( $ident =~ $OAJSP_27 )
    {
      $self->{$OA_VERSION} = $OAFWK_560;
      $self->{$OA_VERSION} =~ s/\s*//;
    }
  elsif ( $ident =~ $OAJSP_36 )
    {
      $self->{$OA_VERSION} = $OAFWK_570;
      $self->{$OA_VERSION} =~ s/\s*//;
    }
  elsif ( $self->_txkGetOAVersionFromJava() )  # sets the $self->{$OA_VERSION}
   {
       return $self->{$OA_VERSION};
   }
  else
    {
      $self->txkSetError({
                          error_status   => TXK::Error::WARNING,
                          error_message => "Cannot determine OA Framework version from OA.jsp version $ident."
			 });
      $self->{$OA_VERSION} = "N/A";
      return TXK::Error::WARNING;
    }

  return $self->{$OA_VERSION};
}

#-------------------------------------
# _txkGetOAVersionFromJava
#-------------------------------------

sub _txkGetOAVersionFromJava
{
    my $self = $ARG[0];

    my $fsys = TXK::FileSys->new();

    my $proc = TXK::Process->new();

    my $java = TXK::OSD->trDirPathToBase( $self->txkGetContextVar({ oavar => 's_jdktop' })  . "/bin/java" );

    my $verfile = TXK::OSD->trDirPathToBase( TXK::OSD->getEnvVar ( { 'name' => 'APPLTMP' } ) . "fwkver.txt" );

    my $class = "oracle.apps.fnd.txk.TXKGetFWKVersion";

    $java .= ".exe" if ( TXK::OSD->isNT() );

    $fsys->access( { fileName => $java,
		     type     => TXK::FileSys::FILE,
		     checkMode=> TXK::FileSys::READ_ACCESS } )
	or TXK::Error->stop ($fsys->getError() );

    $proc->run ( { command     => $java,
		      arg1        => $class,
		      showCommand => TXK::Util::TRUE,
		      showOutput  => TXK::Util::TRUE,
		      stdout      => $verfile } );

    my $io = TXK::IO->new();

    $io->open ( { fileName => $verfile,
		  mode     => TXK::IO::READ } );

    my $ioref = $io->getFileHandle();

    my @contents = <$ioref>;

    $io->close();
   
    $fsys->rmfile( { fileName => $verfile } );

    if ( ( scalar(@contents) > 0 ) and ( $contents[0] =~ /^\d/ ) )
     {
	 $self->{$OA_VERSION} = $contents[0] ;

	 return TXK::Util::TRUE;
     }
    
    return TXK::Util::FALSE ;
}

#-------------------------------------
# txkGetOjspVersion
#-------------------------------------

sub txkGetOjspVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( !defined ( $self->{$OJSP_VERSION} ) )
   {
     $self->_getOjspVersion($args);

   }

  return $self->{$OJSP_VERSION};

}

#-------------------------------------
# _getOjspVersion
#-------------------------------------

sub _getOjspVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( ! ( $self->txkIsAdConfigEnabled($args) ))
   {
     #
     # @@ Do nothing for now!
     #

     $self->txkSetError({
                          error_status  => TXK::Error::FAIL,
                          error_message => "Autoconfig is not enabled.",
                       });

     return TXK::Error::FAIL;

   }
  elsif ( ! ( $self->txkIsWebNode($args) ))
   {
     #
     # @@ Do nothing for now!
     #

     $self->txkSetError({
                          error_status  => TXK::Error::WARNING,
                          error_message => "Not a Web node.",
                       });

     return TXK::Error::WARNING;

   }
  else
   {

     my $get_ojsp_cmd;

     my $ojsp_classpath;

     my $ias_oh = $self->txkGetIASOracleHome($args);

     #-------------------------------------------------------------------------
     # Check first for IAS_ORACLE_HOME/jsp/lib/ojsp.jar
     #-------------------------------------------------------------------------

     my $ojsp_lib_1  = $ias_oh . "/jsp/lib/";

     my $ojsp_jar_1  = $ojsp_lib_1 . "ojsp.jar";

     my $ojsp_path_1 = TXK::OSD->trDirPathToBase($ojsp_jar_1);

     #-------------------------------------------------------------------------
     # Then for IAS_ORACLE_HOME/jsp/lib/ojsp.jar if the former does not exist.
     #-------------------------------------------------------------------------

     my $ojsp_lib_2  = $ias_oh . "/Apache/Ojsp/lib/";

     my $ojsp_jar_2  = $ojsp_lib_2 . "ojsp.jar";

     my $ojsp_path_2 = TXK::OSD->trDirPathToBase($ojsp_jar_2);

     #-------------------------------------------------------------------------
     # Regardless of the result from above, path to ojsputil.jar is same
     #-------------------------------------------------------------------------

     my $ojsp_util     = $ias_oh . "/jsp/lib/" . "ojsputil.jar";

     my $ojsp_utl_path = TXK::OSD->trDirPathToBase($ojsp_util);


     #-------------------------------------------------------------------------
     # Extract s_jdktop & s_afjvaprg from the Apps context file
     #-------------------------------------------------------------------------

     my $s_jdktop = $self->txkGetContextVar({ oavar => 's_jdktop' });

     my $s_afjvaprg   = $self->txkGetContextVar({ oavar => 's_afjvaprg' });

     #-------------------------------------------------------------------------
     # Get LD_LIBRARY_PATH & CLASSPATH since we need them later on
     #-------------------------------------------------------------------------

     my $classpath = TXK::OSD->getEnvVar({ name => "CLASSPATH" });

     my $ld_lib_path_orig = TXK::OSD->getEnvVar({ name => "LD_LIBRARY_PATH" });

     #-------------------------------------------------------------------------
     # Now define LD_LIBRARY_PATH
     #-------------------------------------------------------------------------

     my $ld_lib_path_tmp = $ias_oh . "/lib" . ":" . $s_jdktop . "/lib" ;

     my $ld_lib_path = TXK::OSD->trDirPathToBase($ld_lib_path_tmp);

     TXK::OSD->setEnvVar({
			   name   => "LD_LIBRARY_PATH",
			   value  => $ld_lib_path,
			});


     if ( $self->_isFileFound($ojsp_path_1) )
      {
	$ojsp_classpath = $ojsp_path_1 .
	  TXK::OSD->getPathSeparator() . $ojsp_utl_path . TXK::OSD->getPathSeparator() . "$classpath";

      }
     elsif ( $self->_isFileFound($ojsp_path_2) )
      {
	$ojsp_classpath = $ojsp_path_2 .
	  TXK::OSD->getPathSeparator() . $ojsp_utl_path . TXK::OSD->getPathSeparator() . "$classpath";

      }
     else
      {
        print "\n[FAIL]\t Cannot locate OA.jsp!!!\n\n";

        return;

      }
     

     #-------------------------------------------------------------------------
     # Now construct the JRE command to get the version
     #-------------------------------------------------------------------------

     my $get_ojsp = $s_afjvaprg . " -classpath " . $ojsp_classpath .
			          " oracle.jsp.tools.Jspc " .
				  " -version" ;

     $get_ojsp_cmd = TXK::OSD->trDirPathToBase($get_ojsp);

     my $appltmp  = $self->txkGetApplTmp($args);

     my $log_f     = $appltmp . "/" . "ojsp_ver.txt";

     my $logfile   = TXK::OSD->trDirPathToBase($log_f);

     my $process   = TXK::Process->new();

     $process->abortOnError( { enable => TXK::Util::FALSE } );

     my $run_status = $process->run({
                                      command    => $get_ojsp_cmd,
                                      stdout     => $logfile             ,
				      showCommand=> TXK::Util::TRUE      ,
				      showOutput => TXK::Util::TRUE
                                   });

     if ( $run_status eq TXK::Error::FAIL )
      {
        print "\n[FAIL]\t $get_ojsp_cmd not found!!!\n\n";

        TXK::Error->stop("");

      }

     my $io = TXK::IO->new();

     $io->open({ fileName => $logfile });

     my $io_ref = $io->getFileHandle();
     my @io_data = <$io_ref>;

     $io->close();

     unlink($logfile);

     my $ojsp_version = "000";

     unless ( grep(/$OJSP_PATTERN_1/,@io_data) )
      {
        $self->{$OJSP_VERSION} = $ojsp_version;

        return $self->{$OJSP_VERSION};

      }

     my @ojspobj = grep(/$OJSP_PATTERN_1/,@io_data);

     foreach ( @ojspobj )
      {
        chomp;
        s/$OJSP_PATTERN_1//;

        $self->{$OJSP_VERSION} = $_;
      }

     #
     # Now reset LD_LIBRARY_PATH to what it was.
     #

     TXK::OSD->setEnvVar({
                           name   => "LD_LIBRARY_PATH",
                           value  => $ld_lib_path_orig,
                        });

     return $self->{$OJSP_VERSION};
   }
}

#-------------------------------------
# txkGetFormsVersion
#-------------------------------------

sub txkGetFormsVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( !defined ( $self->{$FORMS_VERSION} ) )
   {
     $self->_getFormsVersion($args);

   }

  return $self->{$FORMS_VERSION};

}

#-------------------------------------
# _getFormsVersion
#-------------------------------------

sub _getFormsVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( ! ( $self->txkIsAdConfigEnabled($args) ))
   {
     #
     # @@ Do nothing for now!
     #

     $self->txkSetError({
                          error_status  => TXK::Error::FAIL,
                          error_message => "Autoconfig is not enabled.",
                       });

     return TXK::Error::FAIL;

   }
  elsif ( ! ( $self->txkIsFormsNode($args) ))
   {
     #
     # @@ Do nothing for now!
     #

     $self->txkSetError({
                          error_status  => TXK::Error::WARNING,
                          error_message => "Not a Forms node.",
                       });

     return TXK::Error::WARNING;

   }
  else
   {
     my $status = TXK::Error::SUCCESS;

     if (TXK::OSD->isNT() )
      {
	$status = $self->_getFormsVersionNT($args);

      }
     else
      {
	$status = $self->_getFormsVersionUnix($args);

      }

     if ( $status eq TXK::Error::FAIL )
      {
	$self->{$FORMS_VERSION} = undef;

	$self->txkPrintErrorMsg();
      }
     else
      {

      }

   }

}

#-------------------------------------
# _getFormsVersionUnix
#-------------------------------------

sub _getFormsVersionUnix
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( ! ( $self->txkIsAdConfigEnabled($args) ))
   {
     #
     # @@ Do nothing for now!
     #

     $self->txkSetError({
                          error_status   => TXK::Error::FAIL,
                          error_message => "Autoconfig is not enabled.",
                       });

     return TXK::Error::FAIL;

   }
  elsif ( ! ( $self->txkIsFormsNode($args) ))
   {
     #
     # @@ Do nothing for now!
     #

     $self->txkSetError({
                          error_status   => TXK::Error::WARNING,
                          error_message => "Not a Forms node",
                       });

     return TXK::Error::WARNING;

   }
  else
   {
     my $oracle_term = TXK_ORACLE_TERM;

     #
     # Set ORACLE_TERM
     #

     TXK::OSD->setEnvVar({
			   name   => "ORACLE_TERM",
			   value  => $oracle_term,
			});

     my $oracle_home = $self->txkGetToolsOracleHome($args);

     #
     # Construct the Forms Compiler path & executable
     #

     my $f60gen = $oracle_home . "/bin/" . "f60gen";

     my $appltmp  = $self->txkGetApplTmp($args);

     my $log_f     = $appltmp . "/" . "forms_ver.txt";

     my $logfile   = TXK::OSD->trDirPathToBase($log_f);

     my $process   = TXK::Process->new();

     $process->abortOnError( { enable => TXK::Util::FALSE } );

     my $run_status = $process->run({
                                      command    => "$f60gen help=y"	,
                                      stdout     => $logfile            ,
                                   });

     if ( $run_status eq TXK::Error::FAIL )
      {
TXK::Error->stop("ERROR FROM f60gen:", $process->getError());
        $self->txkSetError({
			     error_status   => TXK::Error::FAIL,
			     error_message => "$f60gen not found"." ".$process->getError(),
			  });

        return TXK::Error::FAIL;

      }

     my $io = TXK::IO->new();

     $io->open({ fileName => $logfile });

     my $io_ref = $io->getFileHandle();
     my @io_data = <$io_ref>;

     $io->close();

     unlink($logfile);

     my $forms_version = "0";

     unless ( grep(/$FORMS_PATTERN_1/,@io_data) )
      {
        return $forms_version;

      }

     my @frmsobj = grep(/$FORMS_PATTERN_1/,@io_data);

     foreach ( @frmsobj )
      {
        chomp;
        s/$FORMS_PATTERN_1//;
        s/$FORMS_PATTERN_2//;

        $self->{$FORMS_VERSION} = $_;
      }

     return TXK::Error::SUCCESS;

   }
}

#-------------------------------------
# _getFormsVersionNT
#-------------------------------------

sub _getFormsVersionNT
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( ! ( $self->txkIsAdConfigEnabled($args) ))
   {
     #
     # Do nothing for now!
     #

     $self->txkSetError({
                          error_status  => TXK::Error::FAIL,
                          error_message => "Autoconfig is not enabled.",
                       });

     return TXK::Error::FAIL;

   }
  elsif ( ! ( $self->txkIsFormsNode($args) ))
   {
     #
     # Do nothing for now!
     #

     $self->txkSetError({
                          error_status  => TXK::Error::WARNING,
                          error_message => "Not a Forms node.",
                       });

     return TXK::Error::WARNING;

   }
  else
   {
       my $fsys = TXK::FileSys->new();

       my $io   = TXK::IO->new();

       my $ntrgs = TXK::OSD->trDirPathToBase ( $self->txkGetToolsOracleHome($args) . "/orainst/nt.rgs" );

       unless( $fsys->access ( { fileName  => $ntrgs,
		  	         checkMode => TXK::FileSys::READ_ACCESS } ) )
        {
	    $self->txkSetError({
		                error_status  => TXK::Error::FAIL,
				error_message => "$ntrgs file does not exist.",
			      });
	    return TXK::Error::FAIL;
	}

       $io->open ( { fileName => $ntrgs,
		     mode     => TXK::IO::READ } );

       my $ioref = $io->getFileHandle();

       my @contents = <$ioref>;


       @contents = grep ( /854 w32forms60    tools/, @contents );
       
       $contents[0] =~ s/\s+/ /g;

       my @lines = split ( / /, $contents[0] );

       $self->{$FORMS_VERSION} = $lines[5];

       $self->{$FORMS_VERSION} =~ s/(^"|"$)//g;
       
       $io->close();

       return TXK::Error::SUCCESS;
   }

}

#-------------------------------------
# txkGetDev6iPsVersion
#-------------------------------------

sub txkGetEWTVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  if ( !defined ( $self->{$ORACLE_EWT_VERSION} ) )
   {
     $self->{$ORACLE_EWT_VERSION} = $self->_getEWTVersion($args);
   }

  return $self->{$ORACLE_EWT_VERSION};
}

#-------------------------------------
# _getEWTVersion
#-------------------------------------

sub _getEWTVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  # EWT version is needed only on Forms Node

  return TXK::Error::FAIL unless ($self->{$IS_FORMS_NODE});

  my $fnd_ewt_jar = TXK::OSD->trDirPathToBase( $self->txkGetContextVar({ oavar => 's_javatop' }).
					       "/oracle/apps/fnd/jar/fndewt.jar" );

  my $fsys = TXK::FileSys->new();
  $fsys->access({ fileName => $fnd_ewt_jar,
		  checkMode=> TXK::FileSys::READ_ACCESS,
		  type     => TXK::FileSys::FILE} )
    or TXK::Error->stop($fsys->getError());

  my $jar_exe = TXK::OSD->trDirPathToBase ( $self->txkGetContextVar({ oavar => 's_jdktop' }).
					    "/bin/jar" );

  $jar_exe .= ".exe" if ( TXK::OSD->isWindows() );

  $fsys->access({ fileName => $jar_exe,
		  checkMode=> TXK::FileSys::READ_ACCESS,
		  type     => TXK::FileSys::FILE} )
    or TXK::Error->stop($fsys->getError());  

  my $file_list = TXK::OSD->trDirPathToBase( $self->txkGetContextVar({ oavar => 's_applrgf' }).
					     "/fndewt_jar.lst" );

  $fsys->access({ fileName => $file_list,
		  checkMode=> TXK::FileSys::CREATE_ACCESS,
		  type     => TXK::FileSys::FILE} )
    or TXK::Error->stop($fsys->getError());  

  my $proc = TXK::Process->new();

  $proc->run( { command     =>  $jar_exe,
		arg1        =>  "-tvf",
		arg2        =>  "$fnd_ewt_jar",
		showCommand =>  TXK::Util::TRUE,
		showOutput  =>  TXK::Util::TRUE,
		stdout      =>  $file_list } );
  
  my $io = TXK::IO->new();
  $io->open ( { mode     => TXK::IO::READ,
		fileName => $file_list } );

  my $file_handle = $io->getFileHandle();
  my @file_contents= <$file_handle>;
  $io->close();
  $fsys->rmfile({fileName => $file_list});

  my $line;

  foreach $line ( @file_contents )
   {
     #-----------------------------------------------------
     # get the version from a line like
     #
     # 1 Tue Sep 09 10:22:52 PDT 2003 oracle/ewt/3_4_24_2
     #-----------------------------------------------------

     if ( $line =~ m#oracle\/ewt\/(\d+_)# )
       {
	 return $+.$'; # get everything matched in parenthesis i.e. "3_" plus rest of the string "4_24_2" 
       }
   }

  return TXK::Error::FAIL;
}

#-------------------------------------
# txkGetDev6iPsVersion
#-------------------------------------

sub txkGetDev6iPsVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( !defined ( $self->{$DEV6I_PS_VERSION} ) )
   {
     $self->_getDev6iPsVersion($args);

   }

  return $self->{$DEV6I_PS_VERSION};

}

#-------------------------------------
# _getDev6iPsVersion
#-------------------------------------

sub _getDev6iPsVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( ! ( $self->txkIsAdConfigEnabled($args) ))
   {
     #
     # @@ Do nothing for now!
     #

     $self->txkSetError({
                          error_status   => TXK::Error::FAIL,
                          error_message => "Autoconfig is not enabled.",
                       });

     return TXK::Error::FAIL;

   }
  elsif ( ! ( $self->txkIsFormsNode($args) ))
   {
     #
     # @@ Do nothing for now!
     #

     $self->txkSetError({
                          error_status   => TXK::Error::WARNING,
                          error_message => "Not a Forms node",
                       });

     return TXK::Error::WARNING;

   }
  else
   {
     my $forms_version = $self->txkGetFormsVersion($args);

     if ( $forms_version =~ $FORMS_DEV6I_PS9 )
      {
	$self->{$DEV6I_PS_VERSION} = "9";

      }
     elsif ( $forms_version =~ $FORMS_DEV6I_PS10 )
      {
        $self->{$DEV6I_PS_VERSION} = "10";

      }
     elsif ( $forms_version =~ $FORMS_DEV6I_PS11 )
      {
        $self->{$DEV6I_PS_VERSION} = "11";

      }
     elsif ( $forms_version =~ $FORMS_DEV6I_PS12 )
      {
	$self->{$DEV6I_PS_VERSION} = "12";

      }
     else
      {
	$self->txkSetError({
			    error_status   => TXK::Error::WARNING,
			    error_message => "Cannot determine Forms Version from $forms_version."
			   });
	return TXK::Error::WARNING;
      }
   }
  return $self->{$DEV6I_PS_VERSION};
}

#-------------------------------------
# txkIsFormsPS15
#-------------------------------------

sub txkIsFormsPS15
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  my $forms_version  = $self->txkGetFormsVersion($args);

  return TXK::Util::TRUE if ( $forms_version =~ m/^6\.0\.8\.24/ );

  return TXK::Util::FALSE;
}

#-------------------------------------
# txkCheckFormsOneOff3339123
#-------------------------------------

sub txkCheckFormsOneOff
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  TXK::Util->isValidArgs( { args => $args, reqd => ["oneoff"] });

  if ( $args->{'oneoff'} eq 3339123 )
   {
       return $self->_checkFormsOneOff3339123();
   }

  return TXK::Util::FALSE;
}

#-------------------------------------
# _checkFormsOneOff3339123
#-------------------------------------

sub _checkFormsOneOff3339123
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  return TXK::Util::FALSE if ( TXK::OSD->isNT() );

  my $FORMS_3339123_HASH = { 'ifgc.o'   => { 'archive'  => 'libifcw.a', 
					      'checksum'=> { Linux      => '875291771',
							     UNIX_Alpha => '3029440677',
							     Solaris    => '2992266109',
							     IBM_AIX    => '2689470890',
							     HP_UX      => '3679553708' } } ,
			    'ifzmsk.o' => { 'archive' => 'libiffw.a',
					    'checksum'=> {   Linux =>      '1955765053',
							     UNIX_Alpha => '258486057',
							     Solaris    => '1291981273',
							     IBM_AIX    => '4202138253',
							     HP_UX      => '3761638088' } } ,

			    'iifrg.o'  => { 'archive' => 'libiifw.a',
					    'checksum'=> {   Linux =>      '3741178978',
							     UNIX_Alpha => '1698228651',
							     Solaris    => '3118887440',
							     IBM_AIX    => '103175620',
							     HP_UX      => '349225038' } } ,

			    'iwit.o'   => { 'archive' => 'libiwfw.a',
					    'checksum'=> {   Linux =>      '3527666751',
							     UNIX_Alpha => '3054719010',
							     Solaris    => '1142845317',
							     IBM_AIX    => '733790178',
							     HP_UX      => '2614416091' } } 
			   };

  my $tempdir = $self->getAppsTempDir({ $APPS_CTX_FILE_KEY => $self->{$APPS_CTX_FILE} } );

  my $oracle_home   = $self->txkGetToolsOracleHome();
  
  TXK::OSD->chdir($tempdir);  
  
  my $file;
  
  my $proc = TXK::Process->new();

  my $fsys = TXK::FileSys->new();

  my $platform = TXK::OSD->getAutoConfigName();

  foreach $file ( keys %{$FORMS_3339123_HASH} )
   {
       $proc->run ( { command     => 'ar',
		      showCommand => TXK::Util::TRUE,
		      arg1        => 'x',
		      arg2        => TXK::OSD->trDirPathToBase( $oracle_home."/lib/$FORMS_3339123_HASH->{$file}->{'archive'}" ),
		      arg3        => $file,
		   } );

       my $chksumHash = { "filename" => $file,
			  $platform  => [ $FORMS_3339123_HASH->{$file}->{'checksum'}->{$platform} ] };

       my $status = $self->_validateChecksum($chksumHash) ;

       $fsys->rmfile( {fileName => $file } ); 

       return TXK::Util::FALSE unless ( $status );
		      
   }

  return TXK::Util::TRUE;
}

#-------------------------------------
# txkCheckDisplay
#-------------------------------------
sub txkCheckDisplay
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  if (TXK::OSD->isNT() )
    {
      return TXK::Error::SUCCESS;
    }

  if ( !defined ( $self->{$IS_DISPLAY} ))
   {
     $self->_checkDisplay($args);
   }

  return $self->{$IS_DISPLAY};
}


#-------------------------------------
# _checkDisplay
#-------------------------------------
sub _checkDisplay
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( TXK::OSD->isNT() )
    {
      $self->{$IS_DISPLAY} = TXK::Util::TRUE;
      return TXK::Error::SUCCESS;
    }

  my $display;

  if ( ! ( $self->txkIsAdConfigEnabled($args) ))
   {
     $display = TXK::OSD->getEnvVar({name => "DISPLAY"});
   }
  else
   {
     $display = $self->txkGetContextVar({ oavar => 's_display' });
   }

  if (! $display)
    {
      $self->{$VALIDATE_MESSAGE} .= "\n[FAIL]\t The DISPLAY variable is not set. 11.5.9 and later maintenance\n\t packs require you to configure your X Display Server. For more\n\t information please see My Oracle Support Note 181244.1 titled\n\t \"Configuring an X Display Server for Applications on Unix\n\t Platforms\".\n";
      $self->{$IS_DISPLAY} = DISPLAY_NOT_SET;
      return TXK::Error::FAIL;
    }

  my $process   = TXK::Process->new();

  $process->abortOnError( { enable => TXK::Util::FALSE } );

  if (! $process->which({ command=> "xdpyinfo" }) )
    {
      $self->{$VALIDATE_MESSAGE} .= "\n[WARN]\t The xdpyinfo executable is not in the SYSTEM path.\n";
      $self->{$IS_DISPLAY} = XDPYINFO_NOT_FOUND;
      return TXK::Error::WARNING;
    }

  my $tempdir    = $self->txkGetApplTmp($args);
  my $outputFile = TXK::OSD->trDirPathToBase( $tempdir."/xdpyinfo.out");

  my $xdpytest = $process->run({ command => "xdpyinfo",
				 stdout      => $outputFile,
				 showCommand => TXK::Util::FALSE,
				 showOutput => TXK::Util::FALSE
			       });

  unlink($outputFile);

  if ($xdpytest)
    {
      $self->{$VALIDATE_MESSAGE} .= "\n[PASS]\t The DISPLAY variable is set to $display\n\t and xdpyinfo indicates an X Display Server is running.\n";

      $self->{$IS_DISPLAY} = TXK::Util::TRUE;
      return TXK::Error::SUCCESS;
    }
  else
    {
      $self->{$VALIDATE_MESSAGE} .= "\n[WARN]\t The DISPLAY variable is set to $display, but an\n\t active X Display Server could not be found. For more\n\t information please see My Oracle Support Note 181244.1 titled\n\t \"Configuring an X Display Server for Applications on Unix\n\t Platforms\".\n";
      $self->{$IS_DISPLAY} = NO_ACTIVE_DISPLAY;
      return TXK::Error::WARNING;
    }

}


#-------------------------------------
# txkCheckOProcAndWebPort
#-------------------------------------
sub txkCheckOProcAndWebPort
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  my $oprocmgr_port = $self->txkGetContextVar({ oavar => 's_oprocmgr_port' });
  my $web_port      = $self->txkGetContextVar({ oavar => 's_webport' });

  if ($oprocmgr_port == $web_port)
    {
      $self->{$VALIDATE_MESSAGE} .= "\n[FAIL]\t The oprocmgr port and HTTP port values are both set to $web_port.\n\t  Update the \"OPROC Manager Port\" context value in your\n\t  Applications Context File to an unused port number and \n\t  ensure that this is different from the web port. For\n\t  more information, please see My Oracle Support Note 165195.1.\n";
      return TXK::Error::FAIL;
    }
  else
    {
      $self->{$VALIDATE_MESSAGE} .= "\n[PASS]\t The oprocmgr port and HTTP port values are set to $oprocmgr_port and $web_port.\n";
      return TXK::Error::SUCCESS;
    }
}

## -------------------------------------
## txkGetOracleHome : get Oracle Home from db context file
## -------------------------------------

sub txkGetDBOracleHome
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  if ( !defined ( $self->{$DB_HOME}) )
   {
     $self->_getDBOracleHome($args);
   }

  return $self->{$DB_HOME};
}


## -------------------------------------
## _getDBOracleHome
## -------------------------------------

sub _getDBOracleHome
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  $self->{$DB_HOME} = $self->txkGetContextVar({ oavar => 's_db_oh' });
}

## -------------------------------------
## txkGetJDBCDriverVersion
## -------------------------------------

sub txkGetJDBCDriverVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  if ( !defined ( $self->{$JDBC_DRIVER_VERSION}) )
   {
     $self->_getJDBCDriverVersion($args);
   }

  return $self->{$JDBC_DRIVER_VERSION};
}

## -------------------------------------
## txkGetJDBCDriverFile
## -------------------------------------

sub txkGetJDBCDriverFile
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  my $appl_top = $self->txkGetApplTop();

  my $adjborg2_txt = TXK::OSD->trDirPathToBase( $appl_top ."/admin/adjborg2.txt" );

  $adjborg2_txt = TXK::OSD->trDirPathToBase( $appl_top ."/admin/adjborg2_nt.txt" )
      if ( TXK::OSD->isNT() );

  my $io = TXK::IO->new();

  $io->open ( { fileName => $adjborg2_txt,
		mode     => TXK::IO::READ 
	    } );
  my $io_file_handle = $io->getFileHandle();
  my @io_data = <$io_file_handle>;
  $io->close();  

  my @jdbc_file = grep ( /jdbc/ , @io_data );
  $jdbc_file[0] =~ s/\n//g;

  # Fix for bug 4720383 - replacing "$FND_TOP" with its value - this is 
  # required for older apps e.g. 11.5.8 where adjborg2.txt contained
  # references to environment variables instead of full path.
  my $fnd_top = TXK::OSD->getEnvVar( { name => 'FND_TOP' } );
  $jdbc_file[0] =~ s/\$FND_TOP/$fnd_top/g;  

  return $jdbc_file[0] if ( scalar(@jdbc_file) > 0 );

  return TXK::Error::FAIL;
}

## -------------------------------------
## _getJDBCDriverVersion
## -------------------------------------

sub txkGetJDBCDriverFileVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  my $jdbc_file = $self->txkGetJDBCDriverFile();

  my $jdbc_file_version;

  return TXK::Error::FAIL if ( $jdbc_file eq TXK::Error::FAIL );

  my ($jdk12, $jdk14) = TXK::Util::FALSE;

  my ($jdbc12,$jdbc14)= TXK::Util::FALSE;

  $jdbc12 = TXK::Util::TRUE if ( $jdbc_file =~ m/jdbc12\.zip$/ );

  $jdbc14 = TXK::Util::TRUE if ( $jdbc_file =~ m/jdbc14\.zip$/ );

  my $jdbc_file_to_check = "jdbc12.zip";

  $jdbc_file_to_check = "jdbc14.zip" if ( $jdbc14 ) ;

  my $fsys = TXK::FileSys->new();
  my $rc;
  my @identData;

  $rc = $fsys->ident({ fileName  => "$jdbc_file",
                       identData => \@identData,
                    });

  if ( $rc eq TXK::FileSys::NO_IDENT )
   {  
       print "No Ident found in $jdbc_file\n";

       return TXK::Error::FAIL;
   }
  elsif ( $rc eq TXK::FileSys::MULTIPLE_IDENTS )
   {
       my $header;
       
       foreach $header (@identData)
         {
	     if ( $header->{TXK::FileSys::IDENT_NAME} =~ m/$jdbc_file_to_check/ )
	      {
		  $jdbc_file_version = $header->{TXK::FileSys::IDENT_VERSION} ;
	      }
         }
   }

  return $jdbc_file_version;
}

## -------------------------------------
## _getJDBCDriverVersion
## -------------------------------------

sub _getJDBCDriverVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $jdbc_file = $self->txkGetJDBCDriverFile();

  my $jdbc_file_version = $self->txkGetJDBCDriverFileVersion();

  my ($jdk12, $jdk14) = TXK::Util::FALSE;

  my ($jdbc12,$jdbc14)= TXK::Util::FALSE;

  $jdbc12 = TXK::Util::TRUE if ( $jdbc_file =~ m/jdbc12\.zip$/ );

  $jdbc14 = TXK::Util::TRUE if ( $jdbc_file =~ m/jdbc14\.zip$/ );

  my $jdbc_driver_mapping; 

  if ( $jdbc12 )
   {
       $jdbc_driver_mapping = $JDBC_DRIVER_MAPPING_12;

       $self->{$JDBC_DRIVER_VERSION} = "9.2.0.4" ;
   }
  elsif ( $jdbc14 )
   {
       $jdbc_driver_mapping = $JDBC_DRIVER_MAPPING_14;

       $self->{$JDBC_DRIVER_VERSION} = "9.2.0.5 or higher" ;
   }
  else
   {
       return TXK::Error::FAIL;
   }

  my $driver_version ;

  foreach $driver_version ( keys %{$jdbc_driver_mapping} )
   {
       $self->{$JDBC_DRIVER_VERSION} = $driver_version 
	      if ( scalar ( grep ( /$jdbc_file_version/ , @{$jdbc_driver_mapping->{$driver_version}} ) ) > 0 );
   }
}

## -------------------------------------
## txkGetModPLSQLVersion
## -------------------------------------

sub txkGetModPLSQLVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  if ( !defined ( $self->{$MODPLSQL_VERSION}) )
   {
     $self->_getModPLSQLVersion($args);
   }

  return $self->{$MODPLSQL_VERSION};
}

## -------------------------------------
## _getModPLSQLVersion
## -------------------------------------

sub _getModPLSQLVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  return $self->_getModPLSQLVersionNT($args) if ( TXK::OSD->isNT() );

  #-------------------------------------------------------------------------------------
  # we can not depend upon the "select owa_util.get_verion from dual" since some patch
  # set e.g. 3.0.9.8.5c do not update the version.
  #
  # for unix, we do a "strings modplsql.so" and then grep for 3.0.9
  # which returns the following
  # [user4@ap667wgs bin]$ strings  modplsql.so | grep 3.0.9
  # 3.0.9.8.4
  # 3.0.9.8.4
  # mod_plsql/3.0.9.8.4
  # then we get the version from the last line
  #-----------------------------------------------------------------------------------

  my $ias_oracle_home = $self->txkGetIASOracleHome();

  my $plsql_so = TXK::OSD->trDirPathToBase($ias_oracle_home."/Apache/modplsql/bin/modplsql.so");

  my $strings_cmd = "strings";

  my $proc = TXK::Process->new();

  my $outFile = TXK::OSD->trDirPathToBase( $self->getAppsTempDir({ $APPS_CTX_FILE_KEY => $self->{$APPS_CTX_FILE} } ) . 
					    "/plsql.dmp" );

  $proc->run( { command      => "$strings_cmd",
		showCommand  => TXK::Util::FALSE,
		arg1         => "$plsql_so",
		stdout       => $outFile,
		showOutput   => TXK::Util::FALSE } );

  my $fsys = TXK::FileSys->new();

  my $io = TXK::IO->new();

  $io->open ( { fileName => $outFile,
		mode     => TXK::IO::READ 
	    } );
  my $io_file_handle = $io->getFileHandle();
  my @io_data = <$io_file_handle>;
  $io->close();

  $fsys->rmfile ( { fileName => $outFile } );

  my $modplsql_version;
  my @version =  grep ( /mod_plsql\/3.0.9/, @io_data ) ;
  ($_,$modplsql_version) = split( /\//, $version[0] );

  chomp($modplsql_version);
  $modplsql_version =~ s/\s*//g;

  if ( $modplsql_version ne "" )
   {
       $self->{$MODPLSQL_VERSION} = $modplsql_version ;

       return TXK::Util::TRUE;
   }
  elsif ( TXK::OSD->getAutoConfigName() =~ /^IBM_AIX$/  )
   {
       #------------------------------------------------
       # on certain platforms like AIX, there won't be version for 30985d
       # we need to check the checksum for them
       #------------------------------------------------

       my $checksum_hash = { 'filename'=> $plsql_so,
			     'IBM_AIX' => $AIX_MODPLSQL_CHECKSUM_30985D 
			     } ;

       if ( $self->_validateChecksum($checksum_hash) )
        {
	    $self->{$MODPLSQL_VERSION} = '3.0.9.8.5d' ;

	    return TXK::Util::TRUE;
	}
   }

  $self->{$MODPLSQL_VERSION} = "UNKNOWN_VERSION ($modplsql_version)" ;

  return;
}

## -------------------------------------
## _getModPLSQLVersionNT
## -------------------------------------

sub _getModPLSQLVersionNT
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $ias_oracle_home = $self->txkGetIASOracleHome();

  my $plsql_dll = TXK::OSD->trDirPathToBase($ias_oracle_home."/Apache/modplsql/bin/modplsql.dll");

  my $dumpbin = TXK::OSD->trDirPathToBase( $self->txkGetContextVar( { oavar => 's_MSDEVdir'} ) . '\bin\dumpbin.exe');

  my $proc = TXK::Process->new();

  my $dmpFile = TXK::OSD->trDirPathToBase( $self->getAppsTempDir({ $APPS_CTX_FILE_KEY => $self->{$APPS_CTX_FILE} } ) . 
					    "/plsql.dmp" );

  my $outFile = TXK::OSD->trDirPathToBase( $self->getAppsTempDir({ $APPS_CTX_FILE_KEY => $self->{$APPS_CTX_FILE} } ) . 
					    "/plsql.out" );

  $proc->run( { command      => "$dumpbin",
		showCommand  => TXK::Util::TRUE,
		arg1         => '/all /out:'."$dmpFile",
		arg2         => $plsql_dll,
		stdout       => $outFile,
		showOutput   => TXK::Util::TRUE } );

  my $fsys = TXK::FileSys->new();

  my $io = TXK::IO->new();

  $io->open ( { fileName => $dmpFile,
		mode     => TXK::IO::READ 
	    } );
  my $io_file_handle = $io->getFileHandle();
  my @io_data = <$io_file_handle>;
  $io->close();

  $fsys->rmfile ( { fileName => $outFile } );
  $fsys->rmfile ( { fileName => $dmpFile } );

  #----------------------------------------------------------------------------------
  # we can not depend upon the "select owa_util.get_verion from dual" since some patch
  # set e.g. 3.0.9.8.5c do not update the version.
  # 
  # for Windows, we use the "dumpbin.exe" to dump the "modplsql.dll"
  # using "dumpbin /all /out:<tempfile> modplsql.dll"
  # the dump file will contain a line like this.
  #-----------------------------------------------------------------------------
  # 1003B550: 33 2E 30 2E 39 2E 38 2E 34 00 00 00 54 68 65 73  3.0.9.8.4...Thes
  #-----------------------------------------------------------------------------
  # or 
  #-----------------------------------------------------------------------------
  # 10031110: 6C 2F 33 2E 30 2E 39 2E 38 2E 35 64 00 00 00 00  l/3.0.9.8.5d....
  #-----------------------------------------------------------------------------
  # we extract the version from this line
  #-----------------------------------------------------------------------------------

  my $modplsql_version;

  my @version  = grep ( /3.0.9/, @io_data ) ;
  my @version1 = grep ( /\.\.\./, @version ) ;
  my @version2 = split ( / / , $version1[0]);
  my @version3 = grep (/3.0.9/,@version2);
  ($modplsql_version,$_) = split ( /\.\.\./, $version3[0]);

  $modplsql_version =~ s/^(\w|\W)*3\.0/3\.0/g;

  $self->{$MODPLSQL_VERSION} = $modplsql_version ;
}

##-------------------------------------------------
## applySSLFix
##-------------------------------------------------

sub detectAndApplySSLFix
{
    my $self = $ARG[0];

    my $args = $ARG[1];

    TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

    my $scriptName = TXK::OSD->trDirPathToBase( TXK::OSD->getEnvVar( { 'name' => 'FND_TOP' } ). "/patch/115/bin/txkInstSSLFix.pl" );

    my $outFile = TXK::OSD->trDirPathToBase( TXK::OSD->getEnvVar( { 'name' => 'APPLTMP' } )."/sslfix.out" );

    my $fsys = TXK::FileSys->new();

    $fsys->access ( { 'fileName'  => $scriptName,
		       'type'     => TXK::FileSys::FILE,
		       'checkMode'=> TXK::FileSys::READ_ACCESS } )
	or ( TXK::Error->printMsg("$scriptName does not exist. Not applying SSL Fix") and return TXK::Error::FAIL );

    my $proc = TXK::Process->new();

    my $perl = $proc->which({command=>'perl'});

    my $fsys = TXK::FileSys->new();

    my $io = TXK::IO->new();

    my $procArgs = { command      => "$perl $scriptName",
		     showCommand  => TXK::Util::TRUE,
		     arg1         => "-iashome=".$self->txkGetIASOracleHome(),
		     stdout       => $outFile,
		     showOutput   => TXK::Util::TRUE };

    my $stringToSearch = "PATCH APPLIED SUCCESSFULLY";

    if ( exists $args->{'detect'} and
	 $args->{'detect'} eq 'Yes' )
     {
         my $argNumber = 1;

	 # if there are more than args e.g. detectOJSPVersion then pass them along 

	 my $arg;
	 foreach $arg ( keys %$args )
	   {
	     $procArgs->{"arg".$argNumber++} = "-".$arg."=".$args->{$arg} if ( $arg =~ /^detect/ ) ;
	   }

	 $stringToSearch  = "PATCH HAS NOT BEEN APPLIED";
     }

     $proc->run ( $procArgs );

     $io->open ( { fileName => $outFile,
	           mode     => TXK::IO::READ } );
     my $io_file_handle = $io->getFileHandle();
     my @io_data = <$io_file_handle>;
     $io->close();

     $fsys->rmfile ( { fileName => $outFile } );

    my $warningString = "WARNING: CHECKSUM SHOWS A RECENT PATCH HAS BEEN APPLIED";

     return TXK::Error::WARNING if  ( scalar ( grep ( /$warningString/ ,@io_data) ) > 0 );

     return TXK::Util::TRUE if  ( scalar ( grep ( /$stringToSearch/ ,@io_data) ) > 0 );

     return TXK::Util::FALSE;	      
}

##-------------------------------------------------
## detect8174Patch3966092
## detects the 8174 patch in iAS oracle home
##-------------------------------------------------

sub detect8174Patch3966092
{
    my $self = $ARG[0];

    my $args = $ARG[1];

    return TXK::Util::FALSE if ( TXK::OSD->isNT() ); # since there is no correct way for us to detect this

    my $ias_oracle_home = $self->txkGetIASOracleHome();

    my $outFile = TXK::OSD->trDirPathToBase( TXK::OSD->getEnvVar( { 'name' => 'APPLTMP' } )."/oneoffobjlst.out" );

    my $platform_bug_array =  $PLATFORM_8174_BUGNO_3966092->{TXK::OSD->getAutoConfigName()}->{bugno};

    my $platform_bugno;

    my $fsys = TXK::FileSys->new();

    my $proc = TXK::Process->new();

    my $io = TXK::IO->new();

    foreach $platform_bugno ( @$platform_bug_array )
    {
	my $objects = $PLATFORM_8174_BUGNO_3966092->{TXK::OSD->getAutoConfigName()}->{objects};

	my $procArgs = { command      => "ls",
			 showCommand  => TXK::Util::TRUE,
			 arg1         => "-l $ias_oracle_home/lib/" . '.*pre'.$platform_bugno.'*',
			 arg2         => '| wc -l',
			 stdout       => $outFile,
			 showOutput   => TXK::Util::TRUE };

	$proc->run ( $procArgs );

	$io->open ( { fileName => $outFile,
		      mode     => TXK::IO::READ } );
	my $io_file_handle = $io->getFileHandle();
	my @io_data = <$io_file_handle>;
	$io->close();

	$fsys->rmfile ( { fileName => $outFile } );

	$io_data[0] =~ s/\s*//g;

	return TXK::Util::TRUE if ( ( $io_data[0] =~ m/$objects/ ) or ( $io_data[0] > 1 ) );
    }

	return TXK::Util::FALSE;
}



#------------------------------------------------------------------------------
# txkVerifyWebSSLPort
#------------------------------------------------------------------------------
sub txkVerifyWebSSLPort
{
	
	my $self = $ARG[0];
	
	my $URL_PROTOCOL="https";
	    
	my $urlprotocol = $self->txkGetContextVar({ oavar => 's_url_protocol'});
	       
	if ($urlprotocol eq $URL_PROTOCOL)
	{
		my $webport =  $self->txkGetContextVar({ oavar => 's_webport'});
		
		my $webssl_port = $self->txkGetContextVar({ oavar => 's_webssl_port'});
		

		#if ($urlprotocol eq "https")
		#{
			my $ias_oracle_home = $self->txkGetContextVar({ oavar => 's_weboh_oh'});
		
			

			my $tchkCertFile = $ias_oracle_home."/Apache/Apache/conf/ssl.crt/ca.crt";
			my $chkCertFile = TXK::OSD->trDirPathToBase($tchkCertFile);

			if (-e $chkCertFile)
			{
				 
			}
			else
			{
				
				my $fsyscert = TXK::FileSys->new();
			
				$fsyscert->create( { fileName => $chkCertFile,
						type    => TXK::FileSys::FILE,
				    } );
			}
	
		#}
		if ($webport ne $webssl_port)
		{
			return 1; # Fail condn
		} 
		else
		{
			return 0; #Pass condn
		}
		
	}
	
}


#------------------------------------------------------------------------------
# txkVerifyJvmArguments
#------------------------------------------------------------------------------
sub txkVerifyJvmArguments
{
	#my $self = shift;
	my $self = $ARG[0];
	
	my $clientcheck;
	my $client_opt = 0;
	my $server_opt = 0;
	
	my $jdk_top = $self->txkGetContextVar({ oavar => 's_jdktop' });
	
	my $jvm_options = $self->txkGetContextVar({ oavar => 's_jvm_options'});
	
	my $java_complete_version =  $self->_getJDKVersionWeb();
	
	my($java_version, $java_build) = split("_", $java_complete_version);
	
	my $java_version_num = $java_version;
	$java_version_num =~ s/\.//g;
	
	
	if ($java_build eq "")
	{
		$java_build="00";
	}
	
	if ($java_version_num gt "131")
	{
		$clientcheck="true";
	}
	elsif ($java_version eq "1.3.1" &&  $java_build lt "02")
	{
		$clientcheck="false";
	}
	else
	{
		$clientcheck="true";
	}
	
	if ($jvm_options =~ m/\-client/)
	{
		$client_opt = 1;
	}
	
	if ($jvm_options =~ m/\-server/)
	{
		$server_opt = 1;
	}
	
	if ($clientcheck eq "true"  && $client_opt gt "0")
	{
		return 21; #Warning Condn 1
	}
	if ($clientcheck eq "false" && $server_opt gt "0")
	{
		return 22; #Warning Condn 2
	}
	return 0;
	
}


#------------------------------------------------------------------------------
# _getProduct
#------------------------------------------------------------------------------

sub _getProduct
{
	my $self = $ARG[0];
	my $args = $ARG[1];
	
	my $isDBNode;
	my $isAdminNode;
	my $isWebNode;
	my $isFormsNode;
	my $isConcNode;
	
	
	   
	my $apps_user;
	if ( defined( $self->{$APPS_USER} ) )
	{
		$apps_user = $self->{$APPS_USER};
	}
	else
	{
		$apps_user = APPS_DEFAULT_USER;
	}
	
	my $apps_password;
	if ( defined( $self->{$APPS_PASSWORD} ) )
	{
		$apps_password = $self->{$APPS_PASSWORD};
	}
	else
	{
		$apps_password = APPS_DEFAULT_PWD;
	}
	
	my $appltmp  = $self->txkGetApplTmp($args);
	my $log_f    = $appltmp . "/" . "node_info.txt";
	my $logfile  = TXK::OSD->trDirPathToBase($log_f);
	my $spath  = TXK::OSD->getEnvVar({ name => "FND_TOP"}) . "/patch/115/sql/";
	my $s_path = TXK::OSD->trDirPathToBase($spath);
	my $sql    = $s_path . "txkVerifyProduct.sql";
	
	my $arg = "IES";
	my $sid    = $self->{$TWO_TASK};
	my $sqlctx = TXK::SQLPLUS->new();
	my $fsys   = TXK::FileSys->new();
	
	my $retVal;
	$fsys->access ({
	fileName=> $sql,
	type=> TXK::FileSys::FILE,
	checkMode=> TXK::FileSys::READ_ACCESS
	})
	or TXK::Error->stop("Check File(s) Error", $fsys->getError());
	$sqlctx->setConnectInfo({
	user     => "$apps_user",
	password => "$apps_password",
	two_task => "$sid"
	});
	$sqlctx->clearCommand();
	$sqlctx->addCommand("spool " . "$logfile");
	
	$sqlctx->addCommand("@" . "$sql " . "$arg");
	$sqlctx->execute({ showOutput =>TXK::Util::FALSE, });
	my $io = TXK::IO->new();
	$io->open({ fileName => $logfile });
	
	my $io_ref  = $io->getFileHandle();
	my @io_data = <$io_ref>;
	$io->close();
	unlink($logfile) or return TXK::Error::FAIL;
	foreach (@io_data)
	{
		
		if( m/TRUE/g)
		{
			$retVal = TXK::Util::TRUE;
		}
		if( m/FALSE/g)
		{
			$retVal = TXK::Util::FALSE;
		}
	}
	
	return $retVal;
}


#------------------------------------------------------------------------------
# _getHostForProfile
#------------------------------------------------------------------------------

sub _getHostForProfile
{
	my $self = shift;
	my $args = $ARG[1];
	
	
	my $apps_user;
	if ( defined( $self->{$APPS_USER} ) )
	{
		$apps_user = $self->{$APPS_USER};
	}
	else
	{
		$apps_user = APPS_DEFAULT_USER;
	}
	
	my $apps_password;
	if ( defined( $self->{$APPS_PASSWORD} ) )
	{
		$apps_password = $self->{$APPS_PASSWORD};
	}
	else
	{
		$apps_password = APPS_DEFAULT_PWD;
	}
	
	my $appltmp  = $self->txkGetApplTmp($args);
	my $log_f    = $appltmp . "/" . "node_info.txt";
	my $logfile  = TXK::OSD->trDirPathToBase($log_f);
	my $spath  = TXK::OSD->getEnvVar({ name => "FND_TOP"}) . "/patch/115/sql/";
	my $s_path = TXK::OSD->trDirPathToBase($spath);
	my $sql    = $s_path . "txkGetHostForProfile.sql";
	my $profile_option="ICX_FORMS_LAUNCHER";
	my $sid    = $self->{$TWO_TASK};
	my $sqlctx = TXK::SQLPLUS->new();
	my $fsys   = TXK::FileSys->new();
	
	my $retVal;
	$fsys->access ({
	fileName=> $sql,
	type=> TXK::FileSys::FILE,
	checkMode=> TXK::FileSys::READ_ACCESS
	})
	or TXK::Error->stop("Check File(s) Error", $fsys->getError());
	$sqlctx->setConnectInfo({
	user     => "$apps_user",
	password => "$apps_password",
	two_task => "$sid"
	});
	$sqlctx->clearCommand();
	$sqlctx->addCommand("spool " . "$logfile");
	
	$sqlctx->addCommand("@" . "$sql " . "$profile_option");
	$sqlctx->execute({ showOutput =>TXK::Util::FALSE, });
	my $io = TXK::IO->new();
	$io->open({ fileName => $logfile });
	my $io_ref  = $io->getFileHandle();
	my @io_data = <$io_ref>;
	$io->close();
	unlink($logfile) or return TXK::Error::FAIL;


	foreach (@io_data)
	{
		
		if( m/""/g)
		{
			;
		}
		else
		{
			$retVal = $_;
		}
	}
	
	return $retVal;
	
	
}







1;
